"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const child_process_1 = require("child_process");
const os_1 = require("os");
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const tasks_1 = require("../../tasks");
const util_1 = require("../util");
test('minimal case (just a shell command)', () => {
    // GIVEN
    const p = new util_1.TestProject();
    // WHEN
    p.addTask('test', {
        exec: 'echo hello_tasks!',
    });
    // THEN
    expect(executeTask(p, 'test')).toEqual(['hello_tasks!']);
});
test('fails if the step fails', () => {
    // GIVEN
    const p = new util_1.TestProject();
    // WHEN
    p.addTask('test', {
        exec: 'false',
    });
    // THEN
    expect(() => executeTask(p, 'test')).toThrow(/Task \"test\" failed when executing \"false\"/);
});
test('multiple steps', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const t = p.addTask('test');
    // WHEN
    t.exec('echo step1');
    t.exec('echo step2');
    t.exec('echo step3');
    // THEN
    expect(executeTask(p, 'test')).toEqual([
        'step1',
        'step2',
        'step3',
    ]);
});
test('execution stops if a step fails', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const t = p.addTask('test');
    // WHEN
    t.exec('echo step1');
    t.exec('echo step2');
    t.exec('echo step3');
    t.exec('echo failing && false');
    t.exec('echo step4');
    // THEN
    expect(() => executeTask(p, 'test')).toThrow(/Task \"test\" failed when executing \"echo failing && false\"/);
});
describe('condition', () => {
    test('zero exit code means that steps should be executed', () => {
        // GIVEN
        const p = new util_1.TestProject();
        // WHEN
        const t = p.addTask('foo', {
            condition: 'echo evaluating_condition',
        });
        t.exec('echo step1');
        t.exec('echo step2');
        // THEN
        expect(executeTask(p, 'foo')).toEqual([
            'evaluating_condition',
            'step1',
            'step2',
        ]);
    });
    test('non-zero exit code means steps should not be executed', () => {
        // GIVEN
        const p = new util_1.TestProject();
        // WHEN
        const t = p.addTask('foo', {
            condition: 'echo failing_condition && false',
        });
        t.exec('echo step1');
        t.exec('echo step2');
        // THEN
        expect(executeTask(p, 'foo')).toEqual([
            'failing_condition',
        ]);
    });
});
describe('cwd', () => {
    test('default cwd is project root', () => {
        const p = new util_1.TestProject();
        p.addTask('test', { exec: 'echo cwd is $PWD' });
        expect(executeTask(p, 'test')[0].includes(path_1.basename(p.outdir))).toBeTruthy();
    });
    test('if a step changes cwd, it will not affect next steps', () => {
        const p = new util_1.TestProject();
        const task = p.addTask('test');
        task.exec('cd /tmp');
        task.exec('echo $PWD');
        expect(executeTask(p, 'test')[0].includes(path_1.basename(p.outdir))).toBeTruthy();
    });
    test('cwd can be set at the task level', () => {
        const p = new util_1.TestProject();
        const cwd = path_1.join(p.outdir, 'mypwd');
        fs_extra_1.mkdirpSync(cwd);
        const task = p.addTask('test', {
            cwd,
        });
        task.exec('echo step1=$PWD');
        task.exec('echo step2=$PWD');
        for (const line of executeTask(p, 'test')) {
            expect(line.includes('mypwd')).toBeTruthy();
        }
    });
    test('cwd can be set at step level', () => {
        const p = new util_1.TestProject();
        const taskcwd = path_1.join(p.outdir, 'mypwd');
        const stepcwd = path_1.join(p.outdir, 'yourpwd');
        fs_extra_1.mkdirpSync(taskcwd);
        fs_extra_1.mkdirpSync(stepcwd);
        const task = p.addTask('test', { cwd: taskcwd });
        task.exec('echo step1=$PWD');
        task.exec('echo step2=$PWD', { cwd: stepcwd });
        const lines = executeTask(p, 'test');
        expect(lines[0].includes('mypwd')).toBeTruthy();
        expect(lines[1].includes('yourpwd')).toBeTruthy();
    });
    test('fails gracefully if cwd does not exist (task level)', () => {
        const p = new util_1.TestProject();
        p.addTask('test', {
            cwd: path_1.join(p.outdir, 'not-found'),
            exec: 'echo hi',
        });
        expect(() => executeTask(p, 'test')).toThrow(/invalid workdir/);
    });
    test('fails gracefully if cwd does not exist (step level)', () => {
        const p = new util_1.TestProject();
        const task = p.addTask('test');
        task.exec('echo step', { cwd: path_1.join(p.outdir, 'mystep') });
        expect(() => executeTask(p, 'test')).toThrow(/must be an existing directory/);
    });
});
describe('say', () => {
    test('"say" can be used to print an info log during execution', () => {
        const p = new util_1.TestProject();
        const task = p.addTask('say');
        task.say('hello, world');
        p.synth();
        const rt = new tasks_1.TaskRuntime(p.outdir);
        expect(rt.tasks).toStrictEqual([{
                name: 'say',
                steps: [{ say: 'hello, world' }],
            }]);
    });
});
test('builtin tasks are scripts embedded inside projen', () => {
    const p = new util_1.TestProject();
    const task = p.addTask('boom');
    task.builtin('__tests__/tasks/fixtures/builtin-example');
    p.synth();
    const lines = executeTask(p, 'boom');
    expect(lines).toStrictEqual([
        'hello, I am a builtin task',
        'second line',
    ]);
});
test('env is inherited from parent tasks', () => {
    const p = new util_1.TestProject();
    const parent = p.addTask('parent', { env: { E1: 'parent1', E2: 'parent2' } });
    const child = p.addTask('child', { env: { E2: 'child1', E3: 'child2' }, exec: 'echo "child: [$E1,$E2,$E3]"' });
    parent.exec('echo "parent: [$E1,$E2,$E3]"');
    parent.spawn(child);
    const lines = executeTask(p, 'parent');
    expect(lines).toStrictEqual([
        'parent: [parent1,parent2,]',
        'child: [parent1,child1,child2]',
    ]);
});
test('requiredEnv can be used to specify required environment variables', () => {
    const p = new util_1.TestProject();
    p.addTask('my-task', {
        requiredEnv: ['ENV1', 'ENV2', 'ENV3'],
        exec: 'echo "$ENV1 $ENV2 $ENV3"',
    });
    expect(() => executeTask(p, 'my-task')).toThrow(/missing required environment variables: ENV1,ENV2,ENV3/);
    expect(() => executeTask(p, 'my-task', { ENV1: 'env1' })).toThrow(/missing required environment variables: ENV2,ENV3/);
    expect(executeTask(p, 'my-task', { ENV1: 'env1', ENV2: 'env2', ENV3: 'env3' })).toStrictEqual([
        'env1 env2 env3',
    ]);
});
function executeTask(p, taskName, env = {}) {
    p.synth();
    const args = [
        require.resolve('../../cli'),
        taskName,
    ].map(x => `"${x}"`);
    const result = child_process_1.spawnSync(`"${process.execPath}"`, args, { cwd: p.outdir, shell: true, env: { ...process.env, ...env } });
    if (result.status !== 0) {
        throw new Error(`non-zero exit code: ${result.stderr.toString('utf-8')}`);
    }
    return result.stdout.toString('utf-8').trim().split(os_1.EOL);
}
//# sourceMappingURL=data:application/json;base64,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