"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tasks_1 = require("../../tasks");
const util_1 = require("../util");
test('no tasks, no tasks file', () => {
    const p = new util_1.TestProject();
    expect(synthTasksManifest(p)).toBeUndefined();
});
test('empty task', () => {
    const p = new util_1.TestProject();
    // WHEN
    p.addTask('empty');
    // THEN
    expectManifest(p, {
        tasks: {
            empty: {
                name: 'empty',
            },
        },
    });
});
test('remove task', () => {
    const p = new util_1.TestProject();
    // WHEN
    const task = p.addTask('task1');
    p.addTask('task2');
    const removeTask = p.removeTask('task1');
    // THEN
    expect(removeTask).toEqual(task);
    expectManifest(p, {
        tasks: {
            task2: {
                name: 'task2',
            },
        },
    });
});
test('re-add removed task', () => {
    const p = new util_1.TestProject();
    // WHEN
    p.addTask('task1');
    p.addTask('task2');
    const removeTask = p.removeTask('task2');
    p.addTask('task2');
    // THEN
    expect(removeTask).toBeTruthy();
    expectManifest(p, {
        tasks: {
            task1: {
                name: 'task1',
            },
            task2: {
                name: 'task2',
            },
        },
    });
});
test('throw when removing a dependent task', () => {
    const p = new util_1.TestProject();
    // WHEN
    const primary = p.addTask('primary');
    const dependent = p.addTask('dependent');
    primary.spawn(dependent);
    // THEN
    expect(() => p.removeTask('dependent'))
        .toThrowError('Unable to remove task "dependent" because the following tasks depend on it: primary');
});
test('remove already removed task', () => {
    const p = new util_1.TestProject();
    expect(p.removeTask('task1')).toBe(undefined);
});
test('multiple "exec" commands', () => {
    const p = new util_1.TestProject();
    // WHEN
    const task = p.addTask('hello', {
        description: 'hello, world',
        exec: 'echo hello',
        env: {
            FOO: 'bar',
        },
    });
    task.exec('echo world');
    task.exec('echo "with quotes"');
    task.env('BAR', 'baz');
    // THEN
    expectManifest(p, {
        tasks: {
            hello: {
                name: 'hello',
                description: 'hello, world',
                env: {
                    FOO: 'bar',
                    BAR: 'baz',
                },
                steps: [
                    { exec: 'echo hello' },
                    { exec: 'echo world' },
                    { exec: 'echo "with quotes"' },
                ],
            },
        },
    });
});
test('subtasks', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const hello = p.addTask('hello', { exec: 'echo hello' });
    const world = p.addTask('world');
    // WHEN
    world.exec('echo "running hello"');
    world.spawn(hello);
    // THEN
    expectManifest(p, {
        tasks: {
            hello: {
                name: 'hello',
                steps: [{ exec: 'echo hello' }],
            },
            world: {
                name: 'world',
                steps: [
                    { exec: 'echo "running hello"' },
                    { spawn: 'hello' },
                ],
            },
        },
    });
});
test('reset() can be used to reset task steps', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const t0 = p.addTask('your-task');
    const t = p.addTask('my-task');
    t.exec('line1');
    t.spawn(t0);
    t.exec('line2');
    // WHEN
    t.reset('line3');
    t.exec('line4', { cwd: 'foo' });
    // THEN
    expectManifest(p, {
        tasks: {
            'your-task': {
                name: 'your-task',
            },
            'my-task': {
                name: 'my-task',
                steps: [
                    { exec: 'line3' },
                    { cwd: 'foo', exec: 'line4' },
                ],
            },
        },
    });
});
test('prependXXX() can be used to add steps from the top', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const sub = p.addTask('my-sub-task', { exec: 'subexec' });
    const t = p.addTask('my-task');
    t.exec('line1');
    // WHEN
    t.prependExec('line2');
    t.prependSpawn(sub);
    t.prependSay('message');
    // THEN
    expectManifest(p, {
        tasks: {
            'my-sub-task': {
                name: 'my-sub-task',
                steps: [
                    { exec: 'subexec' },
                ],
            },
            'my-task': {
                name: 'my-task',
                steps: [
                    { say: 'message' },
                    { spawn: 'my-sub-task' },
                    { exec: 'line2' },
                    { exec: 'line1' },
                ],
            },
        },
    });
});
test('env() can be used to add environment variables', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const t = p.addTask('my-task', {
        env: {
            INITIAL: '123',
            ENV: '456',
        },
    });
    // WHEN
    t.env('FOO', 'BAR');
    t.env('HELLO', 'world');
    // THEN
    expectManifest(p, {
        tasks: {
            'my-task': {
                name: 'my-task',
                env: {
                    INITIAL: '123',
                    ENV: '456',
                    FOO: 'BAR',
                    HELLO: 'world',
                },
            },
        },
    });
});
test('.steps can be used to list all steps in the current task', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const t0 = p.addTask('your');
    const t = p.addTask('my');
    t.exec('step1');
    t.exec('step2');
    t.exec('step3');
    t.spawn(t0);
    t.exec('step4');
    // WHEN
    const steps = t.steps;
    // THEN
    expect(steps).toStrictEqual([
        { exec: 'step1' },
        { exec: 'step2' },
        { exec: 'step3' },
        { spawn: 'your' },
        { exec: 'step4' },
    ]);
});
test('"condition" can be used to define a command that will determine if a task should be skipped', () => {
    // GIVEN
    const p = new util_1.TestProject();
    p.addTask('foo', {
        condition: 'false',
        exec: 'foo bar',
    });
    // THEN
    expectManifest(p, {
        tasks: {
            foo: {
                name: 'foo',
                condition: 'false',
                steps: [
                    { exec: 'foo bar' },
                ],
            },
        },
    });
});
test('"builtin" can be used to execute builtin commands', () => {
    const p = new util_1.TestProject();
    const task = p.addTask('foo', {
        condition: 'false',
    });
    task.builtin('my/builtin');
    task.builtin('your/builtin');
    // THEN
    expectManifest(p, {
        tasks: {
            foo: {
                name: 'foo',
                condition: 'false',
                steps: [
                    { builtin: 'my/builtin' },
                    { builtin: 'your/builtin' },
                ],
            },
        },
    });
});
test('"requiredEnv" can be used to specify required environment variables', () => {
    const p = new util_1.TestProject();
    p.addTask('foo', {
        requiredEnv: ['MISSING1', 'MISSING2', 'NOT_MISSING'],
    });
    // THEN
    expectManifest(p, {
        tasks: {
            foo: {
                name: 'foo',
                requiredEnv: [
                    'MISSING1', 'MISSING2', 'NOT_MISSING',
                ],
            },
        },
    });
});
function expectManifest(p, toStrictEqual) {
    const manifest = synthTasksManifest(p);
    delete manifest['//'];
    expect(manifest).toStrictEqual(toStrictEqual);
}
function synthTasksManifest(p) {
    return util_1.synthSnapshot(p)[tasks_1.Tasks.MANIFEST_FILE];
    ;
}
//# sourceMappingURL=data:application/json;base64,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