"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const common_1 = require("../common");
const typescript_1 = require("../typescript");
const util_1 = require("./util");
describe('mergeTsconfigOptions', () => {
    test('merging includes', () => {
        const mergedTsconfigOptions = typescript_1.mergeTsconfigOptions([
            {
                include: ['typescript.test.ts'],
                compilerOptions: {},
            },
            {
                include: ['abc'],
                compilerOptions: {},
            },
        ]);
        expect(mergedTsconfigOptions).toEqual(expect.objectContaining({
            include: ['typescript.test.ts', 'abc'],
        }));
    });
    test('merging excludes', () => {
        const mergedTsconfigOptions = typescript_1.mergeTsconfigOptions([
            {
                exclude: ['typescript.test.ts'],
                compilerOptions: {},
            },
            {
                exclude: ['abc'],
                compilerOptions: {},
            },
        ]);
        expect(mergedTsconfigOptions).toEqual(expect.objectContaining({
            exclude: ['typescript.test.ts', 'abc'],
        }));
    });
    test('merging compilerOptions', () => {
        const mergedTsconfigOptions = typescript_1.mergeTsconfigOptions([
            {
                compilerOptions: {
                    esModuleInterop: false,
                },
            },
            {
                compilerOptions: {
                    esModuleInterop: true,
                },
            },
        ]);
        expect(mergedTsconfigOptions).toEqual(expect.objectContaining({
            compilerOptions: {
                esModuleInterop: true,
            },
        }));
    });
});
test('tsconfig prop is propagated to eslint and jest tsconfigs', () => {
    const prj = new typescript_1.TypeScriptProject({
        name: 'test',
        outdir: util_1.mkdtemp(),
        defaultReleaseBranch: 'test',
        tsconfig: {
            include: ['typescript.test.ts'],
            compilerOptions: {
                esModuleInterop: true,
            },
        },
    });
    const out = util_1.synthSnapshot(prj);
    expect(out['tsconfig.json']).toEqual(expect.objectContaining({
        include: expect.arrayContaining([
            `${prj.srcdir}/**/*.ts`,
            'typescript.test.ts',
        ]),
        compilerOptions: expect.objectContaining({
            esModuleInterop: true,
        }),
    }));
    expect(out['tsconfig.eslint.json']).toEqual(expect.objectContaining({
        include: expect.arrayContaining([
            common_1.PROJEN_RC,
            `${prj.srcdir}/**/*.ts`,
            `${prj.testdir}/**/*.ts`,
            'typescript.test.ts',
        ]),
        compilerOptions: expect.objectContaining({
            esModuleInterop: true,
        }),
    }));
    expect(out['tsconfig.jest.json']).toEqual(expect.objectContaining({
        include: expect.arrayContaining([
            common_1.PROJEN_RC,
            `${prj.srcdir}/**/*.ts`,
            `${prj.testdir}/**/*.ts`,
            'typescript.test.ts',
        ]),
        compilerOptions: expect.objectContaining({
            esModuleInterop: true,
        }),
    }));
});
test('sources and compiled output can be collocated', () => {
    var _a;
    const prj = new typescript_1.TypeScriptProject({
        name: 'test',
        outdir: util_1.mkdtemp(),
        defaultReleaseBranch: 'test',
        libdir: 'lib',
        srcdir: 'lib',
    });
    expect((_a = prj.tsconfig) === null || _a === void 0 ? void 0 : _a.exclude).not.toContain('/lib');
    const snapshot = util_1.synthSnapshot(prj)['.gitignore'];
    expect(snapshot).toMatchSnapshot();
});
//# sourceMappingURL=data:application/json;base64,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