"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.sanitizeOutput = exports.directorySnapshot = exports.synthSnapshotWithPost = exports.synthSnapshot = exports.mkdtemp = exports.execProjenCLI = exports.TestProject = void 0;
const os = require("os");
const path = require("path");
const fs = require("fs-extra");
const glob_1 = require("glob");
const __1 = require("..");
const logging = require("../logging");
const util_1 = require("../util");
const PROJEN_CLI = require.resolve('../../bin/projen');
logging.disable(); // no logging during tests
class TestProject extends __1.Project {
    constructor(options = {}) {
        const tmpdir = mkdtemp();
        super({
            name: 'my-project',
            outdir: tmpdir,
            clobber: false,
            logging: {
                level: __1.LogLevel.OFF,
            },
            ...options,
        });
    }
    // override runTaskCommand in tests since the default includes the version
    // number and that will break regresion tests.
    runTaskCommand(task) {
        return `projen ${task.name}`;
    }
    postSynthesize() {
        fs.writeFileSync(path.join(this.outdir, '.postsynth'), '# postsynth');
    }
}
exports.TestProject = TestProject;
function execProjenCLI(workdir, args = []) {
    const command = [
        process.execPath,
        PROJEN_CLI,
        ...args,
    ];
    return util_1.exec(command.map(x => `"${x}"`).join(' '), { cwd: workdir });
}
exports.execProjenCLI = execProjenCLI;
const autoRemove = new Set();
// Hook to automatically remove temporary directories without needing each
// place to actually handle this specifically.
afterAll((done) => {
    // Array.from used to get a copy, so we can safely remove from the set
    for (const dir of Array.from(autoRemove)) {
        try {
            // Note - fs-extra.removeSync is idempotent, so we're safe if the
            // directory has already been cleaned up before we get there!
            fs.removeSync(dir);
        }
        catch (e) {
            done.fail(e);
        }
        autoRemove.delete(dir);
    }
    done();
});
function mkdtemp() {
    const tmpdir = fs.mkdtempSync(path.join(os.tmpdir(), 'projen-test-'));
    autoRemove.add(tmpdir);
    return tmpdir;
}
exports.mkdtemp = mkdtemp;
/**
 * Creates a snapshot of the files generated by a project. Ignores any non-text
 * files so that the snapshots are human readable.
 */
function synthSnapshot(project) {
    // defensive: verify that "outdir" is actually in a temporary directory
    if (!path.resolve(project.outdir).startsWith(os.tmpdir()) && !project.outdir.includes('project-temp-dir')) {
        throw new Error('Trying to capture a snapshot of a project outside of tmpdir, which implies this test might corrupt an existing project');
    }
    const synthed = Symbol.for('synthed');
    if (synthed in project) {
        throw new Error('duplicate synth()');
    }
    project[synthed] = true;
    const ENV_PROJEN_DISABLE_POST = process.env.PROJEN_DISABLE_POST;
    try {
        process.env.PROJEN_DISABLE_POST = 'true';
        project.synth();
        const ignoreExts = ['png', 'ico'];
        return directorySnapshot(project.outdir, { excludeGlobs: ignoreExts.map(ext => `**/*.${ext}`) });
    }
    finally {
        fs.removeSync(project.outdir);
        // values assigned to process.env.XYZ are automatically converted to strings
        if (ENV_PROJEN_DISABLE_POST === undefined) {
            delete process.env.PROJEN_DISABLE_POST;
        }
        else {
            process.env.PROJEN_DISABLE_POST = ENV_PROJEN_DISABLE_POST;
        }
    }
}
exports.synthSnapshot = synthSnapshot;
function synthSnapshotWithPost(project) {
    try {
        project.synth();
        return directorySnapshot(project.outdir);
    }
    finally {
        fs.removeSync(project.outdir);
    }
}
exports.synthSnapshotWithPost = synthSnapshotWithPost;
function directorySnapshot(root, options = {}) {
    var _a;
    const output = {};
    const files = glob_1.glob.sync('**', {
        ignore: (_a = options.excludeGlobs) !== null && _a !== void 0 ? _a : [],
        cwd: root,
        nodir: true,
        dot: true,
    }); // returns relative file paths with POSIX separators
    for (const file of files) {
        const filePath = path.join(root, file);
        let content;
        if (path.extname(filePath) === '.json') {
            content = fs.readJsonSync(filePath);
        }
        else {
            content = fs.readFileSync(filePath, 'utf-8');
        }
        output[file] = content;
    }
    return output;
}
exports.directorySnapshot = directorySnapshot;
/**
 * Removes any non-deterministic aspects from the synthesized output.
 * @param dir The output directory.
 */
function sanitizeOutput(dir) {
    const filepath = path.join(dir, 'package.json');
    const pkg = fs.readJsonSync(filepath);
    const prev = pkg.devDependencies.projen;
    if (!prev) {
        throw new Error(`expecting "${filepath}" to include a devDependency on "projen"`);
    }
    // replace the current projen version with 999.999.999 for deterministic output.
    // this will preserve any semantic version requirements (e.g. "^", "~", etc).
    pkg.devDependencies.projen = prev.replace(/\d+\.\d+\.\d+/, '999.999.999');
    fs.writeJsonSync(filepath, pkg);
    // we will also patch deps.json so that all projen deps will be set to 999.999.999
    const depsPath = path.join(dir, '.projen', 'deps.json');
    const deps = fs.readJsonSync(depsPath);
    for (const dep of deps.dependencies) {
        if (dep.name === 'projen' && dep.version) {
            dep.version = dep.version.replace(/\d+\.\d+\.\d+/, '999.999.999');
        }
    }
    fs.chmodSync(depsPath, '777');
    fs.writeJsonSync(depsPath, deps);
}
exports.sanitizeOutput = sanitizeOutput;
//# sourceMappingURL=data:application/json;base64,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