"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const os = require("os");
const path = require("path");
const vm = require("vm");
const fs = require("fs-extra");
const yargs = require("yargs");
const inventory = require("../../inventory");
const render_options_1 = require("../../javascript/render-options");
const logging = require("../../logging");
const option_hints_1 = require("../../option-hints");
const util_1 = require("../../util");
const macros_1 = require("../macros");
class Command {
    constructor() {
        this.command = 'new [PROJECT-TYPE-NAME] [OPTIONS]';
        this.describe = 'Creates a new projen project';
    }
    builder(args) {
        var _a;
        args.positional('PROJECT-TYPE-NAME', { describe: 'optional only when --from is used and there is a single project type in the external module', type: 'string' });
        args.option('synth', { type: 'boolean', default: true, desc: 'Synthesize after creating .projenrc.js' });
        args.option('comments', { type: 'boolean', default: true, desc: 'Include commented out options in .projenrc.js (use --no-comments to disable)' });
        args.option('from', { type: 'string', alias: 'f', desc: 'External jsii npm module to create project from. Supports any package spec supported by yarn (such as "my-pack@^2.0")' });
        args.option('git', { type: 'boolean', default: true, desc: 'Run `git init` and create an initial commit (use --no-git to disable)' });
        args.example('projen new awscdk-app-ts', 'Creates a new project of built-in type "awscdk-app-ts"');
        args.example('projen new --from projen-vue@^2', 'Creates a new project from an external module "projen-vue" with the specified version');
        for (const type of inventory.discover()) {
            args.command(type.pjid, (_a = type.docs) !== null && _a !== void 0 ? _a : '', {
                builder: cargs => {
                    var _a, _b, _c;
                    cargs.showHelpOnFail(false);
                    for (const option of (_a = type.options) !== null && _a !== void 0 ? _a : []) {
                        if (option.type !== 'string' && option.type !== 'number' && option.type !== 'boolean' && option.kind !== 'enum') {
                            continue; // we only support primitive and enum fields as command line options
                        }
                        let desc = [(_c = (_b = option.docs) === null || _b === void 0 ? void 0 : _b.replace(/\ *\.$/, '')) !== null && _c !== void 0 ? _c : ''];
                        const required = !option.optional;
                        let defaultValue;
                        if (option.default && option.default !== 'undefined') {
                            if (!required) {
                                // if the field is not required, just describe the default but don't actually assign a value
                                desc.push(`[default: ${option.default.replace(/^\ *-/, '').replace(/\.$/, '').trim()}]`);
                            }
                            else {
                                // if the field is required and we have a @default, then assign
                                // the value here so it appears in `--help`
                                defaultValue = renderDefault(process.cwd(), option.default);
                            }
                        }
                        const argType = option.kind === 'enum' ? 'string' : option.type;
                        cargs.option(option.switch, {
                            group: required ? 'Required:' : 'Optional:',
                            type: argType,
                            description: desc.join(' '),
                            default: defaultValue,
                            required,
                        });
                    }
                    return cargs;
                },
                handler: argv => newProject(process.cwd(), type, argv),
            });
        }
        return args;
    }
    async handler(args) {
        // handle --from which means we want to first install a jsii module and then
        // create a project defined within this module.
        if (args.from) {
            return newProjectFromModule(process.cwd(), args.from, args);
        }
        // project type is defined but was not matched by yargs, so print the list of supported types
        if (args.projectTypeName) {
            console.log(`Invalid project type ${args.projectTypeName}. Supported types:`);
            for (const pjid of inventory.discover().map(x => x.pjid)) {
                console.log(`  ${pjid}`);
            }
            return;
        }
        // Handles the use case that nothing was specified since PROJECT-TYPE is now an optional positional parameter
        yargs.showHelp();
    }
}
/**
 * Creates a new project with defaults.
 *
 * This function creates the project type in-process (with in VM) and calls
 * `.synth()` on it (if `options.synth` is not `false`).
 *
 * At the moment, it also generates a `.projenrc.js` file with the same code
 * that was just executed. In the future, this will also be done by the project
 * type, so we can easily support multiple languages of projenrc.
 */
function createProject(opts) {
    // Default project resolution location
    let mod = '../../index';
    // External projects need to load the module from the modules directory
    if (opts.type.moduleName !== 'projen') {
        try {
            mod = path.dirname(require.resolve(path.join(opts.type.moduleName, 'package.json'), { paths: [process.cwd()] }));
        }
        catch (err) {
            throw new Error(`External project module '${opts.type.moduleName}' could not be resolved.`);
        }
    }
    // pass the FQN of the project type to the project initializer so it can
    // generate the projenrc file.
    const { renderedOptions } = render_options_1.renderJavaScriptOptions({
        bootstrap: true,
        comments: opts.comments,
        type: opts.type,
        args: opts.params,
    });
    const newProjectCode = `const project = new ${opts.type.typename}(${renderedOptions});`;
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const module = require(mod);
    const ctx = vm.createContext(module);
    process.env.PROJEN_DISABLE_POST = (!opts.post).toString();
    vm.runInContext([
        newProjectCode,
        opts.synth ? 'project.synth();' : '',
    ].join('\n'), ctx);
}
/**
 * Given a value from "@default", processes macros and returns a stringified
 * (quoted) result.
 *
 * @returns a javascript primitive (could be a string, number or boolean)
 */
function renderDefault(cwd, value) {
    var _a;
    return (_a = macros_1.tryProcessMacro(cwd, value)) !== null && _a !== void 0 ? _a : JSON.parse(value);
}
/**
 * Converts yargs command line switches to project type props.
 * @param type Project type
 * @param argv Command line switches
 */
function commandLineToProps(cwd, type, argv) {
    var _a;
    const props = {};
    // initialize props with default values
    for (const prop of type.options) {
        if (prop.default && prop.default !== 'undefined' && !prop.optional) {
            props[prop.name] = renderDefault(cwd, prop.default);
        }
    }
    for (const [arg, value] of Object.entries(argv)) {
        for (const prop of type.options) {
            if (prop.switch === arg) {
                let curr = props;
                const queue = [...prop.path];
                while (true) {
                    const p = queue.shift();
                    if (!p) {
                        break;
                    }
                    if (queue.length === 0) {
                        curr[p] = value;
                    }
                    else {
                        curr[p] = (_a = curr[p]) !== null && _a !== void 0 ? _a : {};
                        curr = curr[p];
                    }
                }
            }
        }
    }
    return props;
}
/**
 * Generates a new project from an external module.
 *
 * @param spec The name of the external module to load
 * @param args Command line arguments (incl. project type)
 */
async function newProjectFromModule(baseDir, spec, args) {
    var _a;
    const installCommand = `yarn add --modules-folder=${baseDir}/node_modules --silent --no-lockfile --dev`;
    if (args.projenVersion) {
        util_1.exec(`${installCommand} projen@${args.projenVersion}`, { cwd: baseDir });
    }
    else {
        // do not overwrite existing installation
        util_1.exec(`yarn list --depth=0 --pattern projen || ${installCommand} projen`, { cwd: baseDir });
    }
    const specDependencyInfo = yarnAdd(baseDir, spec);
    // Remove optional semver information from spec to retrieve the module name
    const moduleName = spec.replace(/\@([0-9]+)\.([0-9]+)\.([0-9]+)(?:-([0-9A-Za-z-]+(?:\.[0-9A-Za-z-]+)*))?(?:\+[0-9A-Za-z-]+)?$/, '');
    // Find the just installed package and discover the rest recursively from this package folder
    const moduleDir = path.dirname(require.resolve(`${moduleName}/.jsii`, {
        paths: [
            baseDir,
        ],
    }));
    // Only leave projects from the main (requested) package
    const projects = inventory
        .discover(moduleDir)
        .filter(x => x.moduleName === moduleName); // Only list project types from the requested 'from' module
    if (projects.length < 1) {
        throw new Error(`No projects found after installing ${spec}. The module must export at least one class which extends projen.Project`);
    }
    const requested = args.projectTypeName;
    const types = projects.map(p => p.pjid);
    // if user did not specify a project type but the module has more than one, we need them to tell us which one...
    if (!requested && projects.length > 1) {
        throw new Error(`Multiple projects found after installing ${spec}: ${types.join(',')}. Please specify a project name.\nExample: npx projen new --from ${spec} ${types[0]}`);
    }
    // if user did not specify a type (and we know we have only one), the select it. otherwise, search by pjid.
    const type = !requested ? projects[0] : projects.find(p => p.pjid === requested);
    if (!type) {
        throw new Error(`Project type ${requested} not found. Found ${types.join(',')}`);
    }
    for (const option of (_a = type.options) !== null && _a !== void 0 ? _a : []) {
        if (option.type !== 'string' && option.type !== 'number' && option.type !== 'boolean') {
            continue; // we don't support non-primitive fields as command line options
        }
        if (args[option.name] !== undefined) {
            continue; // do not overwrite passed arguments
        }
        if (option.default && option.default !== 'undefined') {
            if (!option.optional) {
                const defaultValue = renderDefault(baseDir, option.default);
                args[option.name] = defaultValue;
                args[option.switch] = defaultValue;
            }
        }
    }
    // include a dev dependency for the external module
    await newProject(baseDir, type, args, {
        devDeps: [specDependencyInfo],
    });
}
/**
 * Generates a new project.
 * @param type Project type
 * @param args Command line arguments
 * @param additionalProps Additional parameters to include in .projenrc.js
 */
async function newProject(baseDir, type, args, additionalProps) {
    // convert command line arguments to project props using type information
    const props = commandLineToProps(baseDir, type, args);
    // merge in additional props if specified
    for (const [k, v] of Object.entries(additionalProps !== null && additionalProps !== void 0 ? additionalProps : {})) {
        props[k] = v;
    }
    createProject({
        dir: baseDir,
        type,
        params: props,
        comments: args.comments ? option_hints_1.NewProjectOptionHints.FEATURED : option_hints_1.NewProjectOptionHints.NONE,
        synth: args.synth,
        post: args.post,
    });
    if (args.git) {
        const git = (cmd) => util_1.exec(`git ${cmd}`, { cwd: baseDir });
        git('init');
        git('add .');
        git('commit --allow-empty -m "chore: project created with projen"');
        git('branch -M main');
    }
}
/**
 * Installs the npm module (through `yarn add`) to node_modules under `projectDir`.
 * @param spec The npm package spec (e.g. foo@^1.2)
 * @returns String info for the project devDeps (e.g. foo@^1.2 or foo@/var/folders/8k/qcw0ls5pv_ph0000gn/T/projen-RYurCw/pkg.tgz)
 */
function yarnAdd(baseDir, spec) {
    const packageJsonPath = path.join(baseDir, 'package.json');
    const packageJsonExisted = fs.existsSync(packageJsonPath);
    let dependencyInfo = spec;
    // workaround: yarn fails to extract tgz if it contains '@' in the name, so we
    // create a temp copy called pkg.tgz and install from there.
    // see: https://github.com/yarnpkg/yarn/issues/6339
    if (spec.endsWith('.tgz') && spec.includes('@')) {
        // if user passes in a file spec then we have to specify the project name and the package location
        // (e.g foo@/var/folders/8k/qcw0ls5pv_ph0000gn/T/projen-RYurCw/pkg.tgz)
        const moduleName = spec.split('/').slice(-1)[0].trim().split('@')[0].trim(); // Example: ./cdk-project/dist/js/cdk-project@1.0.0.jsii.tgz
        const packageDir = fs.mkdtempSync(path.join(os.tmpdir(), 'projen-'));
        const copy = path.join(packageDir, 'pkg.tgz');
        fs.copyFileSync(spec, copy);
        spec = copy;
        dependencyInfo = `${moduleName}@${spec}`;
    }
    logging.info(`installing external module ${spec}...`);
    util_1.exec(`yarn add --modules-folder=${baseDir}/node_modules --silent --no-lockfile --dev ${spec}`, { cwd: baseDir });
    // if package.json did not exist before calling yarn add, we should remove it
    // so we can start off clean.
    if (!packageJsonExisted) {
        fs.removeSync(packageJsonPath);
    }
    return dependencyInfo;
}
module.exports = new Command();
//# sourceMappingURL=data:application/json;base64,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