"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Eslint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const common_1 = require("./common");
const component_1 = require("./component");
const json_1 = require("./json");
/**
 * @experimental
 */
class Eslint extends component_1.Component {
    /**
     * @experimental
     */
    constructor(project, options) {
        var _b, _c, _d, _e, _f, _g;
        super(project);
        project.addDevDeps('eslint', '@typescript-eslint/eslint-plugin', '@typescript-eslint/parser', 'eslint-import-resolver-node', 'eslint-import-resolver-typescript', 'eslint-plugin-import', 'json-schema');
        if (options.prettier) {
            project.addDevDeps('prettier', 'eslint-plugin-prettier', 'eslint-config-prettier');
        }
        const devdirs = (_b = options.devdirs) !== null && _b !== void 0 ? _b : [];
        const dirs = [...options.dirs, ...devdirs];
        const fileExtensions = (_c = options.fileExtensions) !== null && _c !== void 0 ? _c : ['.ts'];
        this._allowDevDeps = (devdirs !== null && devdirs !== void 0 ? devdirs : []).map(dir => `**/${dir}/**`);
        const lintProjenRc = (_d = options.lintProjenRc) !== null && _d !== void 0 ? _d : true;
        const eslint = project.addTask('eslint', {
            description: 'Runs eslint against the codebase',
            exec: [
                'eslint',
                `--ext ${fileExtensions.join(',')}`,
                '--fix',
                '--no-error-on-unmatched-pattern',
                ...dirs,
                ...lintProjenRc ? [common_1.PROJEN_RC] : [],
            ].join(' '),
        });
        project.testTask.spawn(eslint);
        // exclude some files
        (_e = project.npmignore) === null || _e === void 0 ? void 0 : _e.exclude('/.eslintrc.json');
        const formattingRules = options.prettier ? {
            'prettier/prettier': ['error'],
        } : {
            // see https://github.com/typescript-eslint/typescript-eslint/blob/master/packages/eslint-plugin/docs/rules/indent.md
            'indent': ['off'],
            '@typescript-eslint/indent': ['error', 2],
            // Style
            'quotes': ['error', 'single', { avoidEscape: true }],
            'comma-dangle': ['error', 'always-multiline'],
            'comma-spacing': ['error', { before: false, after: true }],
            'no-multi-spaces': ['error', { ignoreEOLComments: false }],
            'array-bracket-spacing': ['error', 'never'],
            'array-bracket-newline': ['error', 'consistent'],
            'object-curly-spacing': ['error', 'always'],
            'object-curly-newline': ['error', { multiline: true, consistent: true }],
            'object-property-newline': ['error', { allowAllPropertiesOnSameLine: true }],
            'keyword-spacing': ['error'],
            'brace-style': ['error', '1tbs', { allowSingleLine: true }],
            'space-before-blocks': ['error'],
            'curly': ['error', 'multi-line', 'consistent'],
            '@typescript-eslint/member-delimiter-style': ['error'],
            // Require semicolons
            'semi': ['error', 'always'],
            // Max line lengths
            'max-len': ['error', {
                    code: 150,
                    ignoreUrls: true,
                    ignoreStrings: true,
                    ignoreTemplateLiterals: true,
                    ignoreComments: true,
                    ignoreRegExpLiterals: true,
                }],
            // Don't unnecessarily quote properties
            'quote-props': ['error', 'consistent-as-needed'],
        };
        this.rules = {
            ...formattingRules,
            // Require use of the `import { foo } from 'bar';` form instead of `import foo = require('bar');`
            '@typescript-eslint/no-require-imports': ['error'],
            // Require all imported dependencies are actually declared in package.json
            'import/no-extraneous-dependencies': [
                'error',
                {
                    // Only allow importing devDependencies from "devdirs".
                    devDependencies: () => this.renderDevDepsAllowList(),
                    optionalDependencies: false,
                    peerDependencies: true,
                },
            ],
            // Require all imported libraries actually resolve (!!required for import/no-extraneous-dependencies to work!!)
            'import/no-unresolved': ['error'],
            // Require an ordering on all imports
            'import/order': ['warn', {
                    groups: ['builtin', 'external'],
                    alphabetize: { order: 'asc', caseInsensitive: true },
                }],
            // Cannot import from the same module twice
            'no-duplicate-imports': ['error'],
            // Cannot shadow names
            'no-shadow': ['off'],
            '@typescript-eslint/no-shadow': ['error'],
            // Required spacing in property declarations (copied from TSLint, defaults are good)
            'key-spacing': ['error'],
            // No multiple empty lines
            'no-multiple-empty-lines': ['error'],
            // One of the easiest mistakes to make
            '@typescript-eslint/no-floating-promises': ['error'],
            // Make sure that inside try/catch blocks, promises are 'return await'ed
            // (must disable the base rule as it can report incorrect errors)
            'no-return-await': ['off'],
            '@typescript-eslint/return-await': ['error'],
            // Useless diff results
            'no-trailing-spaces': ['error'],
            // Must use foo.bar instead of foo['bar'] if possible
            'dot-notation': ['error'],
            // Are you sure | is not a typo for || ?
            'no-bitwise': ['error'],
            // Member ordering
            '@typescript-eslint/member-ordering': ['error', {
                    default: [
                        'public-static-field',
                        'public-static-method',
                        'protected-static-field',
                        'protected-static-method',
                        'private-static-field',
                        'private-static-method',
                        'field',
                        // Constructors
                        'constructor',
                        // Methods
                        'method',
                    ],
                }],
        };
        // Overrides for .projenrc.js
        this.overrides = [
            {
                files: [common_1.PROJEN_RC],
                rules: {
                    '@typescript-eslint/no-require-imports': 'off',
                    'import/no-extraneous-dependencies': 'off',
                },
            },
        ];
        this.ignorePatterns = (_f = options.ignorePatterns) !== null && _f !== void 0 ? _f : [
            '*.js',
            `!${common_1.PROJEN_RC}`,
            '*.d.ts',
            'node_modules/',
            '*.generated.ts',
            'coverage',
        ];
        const tsconfig = (_g = options.tsconfigPath) !== null && _g !== void 0 ? _g : './tsconfig.json';
        const plugins = [
            '@typescript-eslint',
            'import',
            ...(options.prettier ? ['prettier'] : []),
        ];
        const extendsConf = [
            'plugin:import/typescript',
            ...(options.prettier ? [
                'prettier',
                'plugin:prettier/recommended',
            ] : []),
        ];
        this.config = {
            env: {
                jest: true,
                node: true,
            },
            root: true,
            plugins,
            parser: '@typescript-eslint/parser',
            parserOptions: {
                ecmaVersion: 2018,
                sourceType: 'module',
                project: tsconfig,
            },
            extends: extendsConf,
            settings: {
                'import/parsers': {
                    '@typescript-eslint/parser': ['.ts', '.tsx'],
                },
                'import/resolver': {
                    node: {},
                    typescript: {
                        project: tsconfig,
                    },
                },
            },
            ignorePatterns: this.ignorePatterns,
            rules: this.rules,
            overrides: this.overrides,
        };
        new json_1.JsonFile(project, '.eslintrc.json', { obj: this.config, marker: false });
    }
    /**
     * (experimental) Add an eslint rule.
     *
     * @experimental
     */
    addRules(rules) {
        for (const [k, v] of Object.entries(rules)) {
            this.rules[k] = v;
        }
    }
    /**
     * (experimental) Add an eslint override.
     *
     * @experimental
     */
    addOverride(override) {
        this.overrides.push(override);
    }
    /**
     * (experimental) Do not lint these files.
     *
     * @experimental
     */
    addIgnorePattern(pattern) {
        this.ignorePatterns.push(pattern);
    }
    /**
     * (experimental) Add a glob file pattern which allows importing dev dependencies.
     *
     * @param pattern glob pattern.
     * @experimental
     */
    allowDevDeps(pattern) {
        this._allowDevDeps.push(pattern);
    }
    renderDevDepsAllowList() {
        return this._allowDevDeps;
    }
}
exports.Eslint = Eslint;
_a = JSII_RTTI_SYMBOL_1;
Eslint[_a] = { fqn: "projen.Eslint", version: "0.27.22" };
//# sourceMappingURL=data:application/json;base64,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