"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.readJsiiManifest = exports.resolveProjectType = exports.discover = void 0;
const path = require("path");
const fs = require("fs-extra");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const decamelize = require('decamelize');
const PROJEN_MODULE_ROOT = path.join(__dirname, '..');
const PROJECT_BASE_FQN = 'projen.Project';
/**
 * Returns a list of project types exported the modules defined in `moduleDirs`.
 * This list will always also include the built-in projen project types.
 * Modules without a .jsii manifest are skipped.
 *
 * @param moduleDirs A list of npm module directories
 */
function discover(...moduleDirs) {
    const jsii = discoverJsiiTypes(...moduleDirs);
    const result = new Array();
    for (const fqn of Object.keys(jsii)) {
        const p = toProjectType(jsii, fqn);
        if (!p) {
            continue;
        }
        result.push(p);
    }
    return result.sort((r1, r2) => r1.pjid.localeCompare(r2.pjid));
}
exports.discover = discover;
/**
 * Resolve all jsii types from @modulesDirs.
 * When a jsii module is found it will recusively list the types from the dependant module as well
 *
 * @param moduleDirs
 * @returns
 */
function discoverJsiiTypes(...moduleDirs) {
    const jsii = {};
    const discoverJsii = (dir) => {
        const jsiiFile = path.join(dir, '.jsii');
        if (!fs.existsSync(jsiiFile)) {
            return;
        } // no jsii manifest
        const manifest = fs.readJsonSync(jsiiFile);
        for (const [fqn, type] of Object.entries(manifest.types)) {
            jsii[fqn] = {
                ...type,
            };
        }
        // Also search recursively in nested project dependencies. If the requested module is an external module
        // this will also end-up in the projen module and add the projen types
        if (manifest.dependencies) {
            for (const dependency of Object.keys(manifest.dependencies)) {
                const nestedDependencyFolder = path.dirname(require.resolve(`${dependency}/package.json`, {
                    paths: [dir],
                }));
                if (fs.existsSync(nestedDependencyFolder)) {
                    discoverJsii(nestedDependencyFolder);
                }
            }
        }
    };
    // read all .jsii manifests from all requested modules and merge
    // them all into a single map of fqn->type.
    for (const dir of [...moduleDirs, PROJEN_MODULE_ROOT]) {
        discoverJsii(dir);
        // Read from scoped packages
        if (dir.includes('@') && fs.lstatSync(dir).isDirectory()) {
            const childDirs = fs.readdirSync(dir).map(file => path.join(dir, file));
            for (const child of childDirs) {
                discoverJsii(child);
            }
        }
    }
    return jsii;
}
function resolveProjectType(projectFqn) {
    let [moduleName] = projectFqn.split('.');
    if (moduleName === 'projen') {
        moduleName = PROJEN_MODULE_ROOT;
    }
    // try picking the manifest. We only need the base folder but this is directly a nice check if we request from a valid jsii package
    const jsiiManifestFile = require.resolve(`${moduleName}/.jsii`, {
        paths: [
            process.cwd(),
        ],
    });
    const moduleFolder = path.dirname(jsiiManifestFile);
    // Read all jsii types that can be loaded from this project type
    const jsii = discoverJsiiTypes(moduleFolder);
    return toProjectType(jsii, projectFqn);
}
exports.resolveProjectType = resolveProjectType;
function toProjectType(jsii, fqn) {
    var _a, _b, _c, _d;
    if (!isProjectType(jsii, fqn)) {
        return undefined;
    }
    const typeinfo = jsii[fqn];
    // projen.web.ReactProject -> web.ReactProject
    const typename = fqn.substring(fqn.indexOf('.') + 1);
    const docsurl = `https://github.com/projen/projen/blob/master/API.md#projen-${typename.toLocaleLowerCase().replace(/\./g, '-')}`;
    let pjid = (_c = (_b = (_a = typeinfo.docs) === null || _a === void 0 ? void 0 : _a.custom) === null || _b === void 0 ? void 0 : _b.pjid) !== null && _c !== void 0 ? _c : decamelize(typename).replace(/_project$/, '');
    return {
        moduleName: typeinfo.assembly,
        typename,
        pjid,
        fqn,
        options: discoverOptions(jsii, fqn).sort((o1, o2) => o1.name.localeCompare(o2.name)),
        docs: (_d = typeinfo.docs) === null || _d === void 0 ? void 0 : _d.summary,
        docsurl,
    };
}
function readJsiiManifest(jsiiFqn) {
    let [moduleName] = jsiiFqn.split('.');
    if (moduleName === 'projen') {
        moduleName = PROJEN_MODULE_ROOT;
    }
    const jsiiManifestFile = require.resolve(`${moduleName}/.jsii`);
    return fs.readJsonSync(jsiiManifestFile);
}
exports.readJsiiManifest = readJsiiManifest;
function discoverOptions(jsii, fqn) {
    var _a, _b, _c, _d;
    const options = {};
    const params = (_c = (_b = (_a = jsii[fqn]) === null || _a === void 0 ? void 0 : _a.initializer) === null || _b === void 0 ? void 0 : _b.parameters) !== null && _c !== void 0 ? _c : [];
    const optionsParam = params[0];
    const optionsTypeFqn = (_d = optionsParam === null || optionsParam === void 0 ? void 0 : optionsParam.type) === null || _d === void 0 ? void 0 : _d.fqn;
    if (params.length > 1 || (params.length === 1 && (optionsParam === null || optionsParam === void 0 ? void 0 : optionsParam.name) !== 'options')) {
        throw new Error(`constructor for project ${fqn} must have a single "options" argument of a struct type. got ${JSON.stringify(params)}`);
    }
    addOptions(optionsTypeFqn);
    const opts = Object.values(options);
    return opts.sort((a, b) => a.switch.localeCompare(b.switch));
    function addOptions(ofqn, basePath = [], optional = false) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
        if (!ofqn) {
            return;
        }
        const struct = jsii[ofqn];
        if (!struct) {
            throw new Error(`unable to find options type ${ofqn} for project ${fqn}`);
        }
        for (const prop of (_a = struct.properties) !== null && _a !== void 0 ? _a : []) {
            const propPath = [...basePath, prop.name];
            // protect against double-booking
            if (prop.name in options) {
                throw new Error(`duplicate option "${prop.name}" in ${fqn} (already declared in ${options[prop.name].parent})`);
            }
            let typeName;
            let jsiiKind;
            if ((_b = prop.type) === null || _b === void 0 ? void 0 : _b.primitive) {
                typeName = (_c = prop.type) === null || _c === void 0 ? void 0 : _c.primitive; // e.g. 'string', 'boolean', 'number'
            }
            else if ((_d = prop.type) === null || _d === void 0 ? void 0 : _d.fqn) {
                typeName = (_e = prop.type) === null || _e === void 0 ? void 0 : _e.fqn.split('.').pop(); // projen.NodeProjectOptions -> NodeProjectOptions
                jsiiKind = jsii[(_f = prop.type) === null || _f === void 0 ? void 0 : _f.fqn].kind; // e.g. 'class', 'interface', 'enum'
            }
            else { // any other types such as collection types
                typeName = 'unknown';
            }
            const isOptional = optional || prop.optional;
            let defaultValue = (_g = prop.docs) === null || _g === void 0 ? void 0 : _g.default;
            if (defaultValue === 'undefined') {
                defaultValue = undefined;
            }
            // if this is a mandatory option and we have a default value, it has to be JSON-parsable to the correct type
            if (!isOptional && defaultValue) {
                if (!((_h = prop.type) === null || _h === void 0 ? void 0 : _h.primitive)) {
                    throw new Error(`required option "${prop.name}" with a @default must use primitive types (string, number or boolean). type found is: ${typeName}`);
                }
                checkDefaultIsParsable(prop.name, defaultValue, (_j = prop.type) === null || _j === void 0 ? void 0 : _j.primitive);
            }
            options[prop.name] = filterUndefined({
                path: propPath,
                parent: struct.name,
                name: prop.name,
                fqn: (_k = prop.type) === null || _k === void 0 ? void 0 : _k.fqn,
                docs: prop.docs.summary,
                type: typeName,
                kind: jsiiKind,
                switch: propPath.map(p => decamelize(p).replace(/_/g, '-')).join('-'),
                default: defaultValue,
                optional: isOptional,
                featured: ((_m = (_l = prop.docs) === null || _l === void 0 ? void 0 : _l.custom) === null || _m === void 0 ? void 0 : _m.featured) === 'true',
                deprecated: prop.docs.stability === 'deprecated' ? true : undefined,
            });
        }
        for (const ifc of (_o = struct.interfaces) !== null && _o !== void 0 ? _o : []) {
            addOptions(ifc);
        }
    }
}
function filterUndefined(obj) {
    const ret = {};
    for (const [k, v] of Object.entries(obj)) {
        if (v !== undefined) {
            ret[k] = v;
        }
    }
    return ret;
}
function isProjectType(jsii, fqn) {
    var _a;
    const type = jsii[fqn];
    if (type.kind !== 'class') {
        return false;
    }
    if (type.abstract) {
        return false;
    }
    if ((_a = type.docs) === null || _a === void 0 ? void 0 : _a.deprecated) {
        return false;
    }
    let curr = type;
    while (true) {
        if (curr.fqn === PROJECT_BASE_FQN) {
            return true;
        }
        if (!curr.base) {
            return false;
        }
        curr = jsii[curr.base];
        if (!curr) {
            return false;
        }
    }
}
function checkDefaultIsParsable(prop, value, type) {
    // macros are pass-through
    if (value.startsWith('$')) {
        return;
    }
    try {
        const parsed = JSON.parse(value);
        if (typeof (parsed) !== type) {
            throw new Error(`cannot parse @default value for mandatory option ${prop} as a ${type}: ${parsed}`);
        }
    }
    catch (e) {
        throw new Error(`unable to JSON.parse() value "${value}" specified as @default for mandatory option "${prop}": ${e.message}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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