"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Projenrc = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const common_1 = require("../common");
const component_1 = require("../component");
const deps_1 = require("../deps");
const inventory_1 = require("../inventory");
const project_1 = require("../project");
/**
 * (experimental) Allows writing projenrc files in java.
 *
 * This will install `org.projen/projen` as a Maven dependency and will add a
 * `synth` task which will compile & execute `main()` from
 * `src/main/java/projenrc.java`.
 *
 * @experimental
 */
class Projenrc extends component_1.Component {
    /**
     * @experimental
     */
    constructor(project, pom, options = {}) {
        var _b, _c, _d;
        super(project);
        const projenVersion = (_b = options.projenVersion) !== null && _b !== void 0 ? _b : common_1.PROJEN_VERSION;
        this.className = (_c = options.className) !== null && _c !== void 0 ? _c : 'projenrc';
        this.testScope = (_d = options.testScope) !== null && _d !== void 0 ? _d : true;
        const depType = this.testScope ? deps_1.DependencyType.TEST : deps_1.DependencyType.RUNTIME;
        const execOpts = this.testScope ? ' -Dexec.classpathScope="test"' : '';
        const compileGoal = this.testScope ? 'compiler:testCompile' : 'compiler:compile';
        project.deps.addDependency(`com.github.eladb/projen@${projenVersion}`, depType);
        pom.addPlugin('org.codehaus.mojo/exec-maven-plugin@3.0.0');
        // set up the "default" task which is the task executed when `projen` is executed for this project.
        const defaultTask = project.addTask(project_1.Project.DEFAULT_TASK, { description: 'Synthesize the project' });
        defaultTask.exec(`mvn ${compileGoal} --quiet`);
        defaultTask.exec(`mvn exec:java --quiet -Dexec.mainClass=${this.className}${execOpts}`);
        // if this is a new project, generate a skeleton for projenrc.java
        this.generateProjenrc();
    }
    generateProjenrc() {
        var _b, _c, _d;
        const bootstrap = this.project.newProject;
        if (!bootstrap) {
            return;
        }
        const jsiiFqn = bootstrap.fqn;
        const jsiiManifest = inventory_1.readJsiiManifest(jsiiFqn);
        const jsiiType = jsiiManifest.types[jsiiFqn];
        const javaTarget = jsiiManifest.targets.java;
        const optionsTypeFqn = (_d = (_c = (_b = jsiiType.initializer) === null || _b === void 0 ? void 0 : _b.parameters) === null || _c === void 0 ? void 0 : _c[0].type) === null || _d === void 0 ? void 0 : _d.fqn;
        if (!optionsTypeFqn) {
            this.project.logger.warn('cannot determine jsii type for project options');
            return;
        }
        const jsiiOptionsType = jsiiManifest.types[optionsTypeFqn];
        if (!jsiiOptionsType) {
            this.project.logger.warn(`cannot find jsii type for project options: ${optionsTypeFqn}`);
            return;
        }
        const dir = this.testScope ? 'src/test/java' : 'src/main/java';
        const split = this.className.split('.');
        let javaClass, javaPackage;
        if (split.length === 1) {
            javaClass = split[0];
            javaPackage = [];
        }
        else {
            javaPackage = split.slice(0, split.length - 2);
            javaClass = split[split.length - 1];
        }
        const javaFile = path_1.join(this.project.outdir, dir, ...javaPackage, javaClass + '.java');
        // skip if file exists
        if (fs_extra_1.existsSync(javaFile)) {
            return;
        }
        const lines = new Array();
        let indent = 0;
        const emit = (line = '') => lines.push(' '.repeat(indent * 4) + line);
        const openBlock = (line = '') => { emit(line + ' {'); indent++; };
        const closeBlock = () => { indent--; emit('}'); };
        const optionFqns = {};
        for (const option of bootstrap.type.options) {
            if (option.fqn) {
                optionFqns[option.name] = toJavaFullTypeName(jsiiManifest.types[option.fqn]);
            }
        }
        if (javaPackage.length > 0) {
            emit(`package ${javaPackage.join('.')};`);
            emit();
        }
        const { renderedOptions, imports } = renderJavaOptions(2, jsiiOptionsType.name, optionFqns, bootstrap.args);
        emit(`import ${javaTarget.package}.${toJavaFullTypeName(jsiiType)};`);
        emit(`import ${javaTarget.package}.${toJavaFullTypeName(jsiiOptionsType)};`);
        for (const optionTypeName of imports) {
            emit(`import ${javaTarget.package}.${optionTypeName};`);
        }
        emit();
        openBlock(`public class ${javaClass}`);
        openBlock('public static void main(String[] args)');
        emit(`${jsiiType.name} project = new ${jsiiType.name}(${renderedOptions});`);
        emit('project.synth();');
        closeBlock();
        closeBlock();
        fs_extra_1.mkdirpSync(path_1.dirname(javaFile));
        fs_extra_1.writeFileSync(javaFile, lines.join('\n'));
        this.project.logger.info(`Project definition file was created at ${javaFile}`);
    }
}
exports.Projenrc = Projenrc;
_a = JSII_RTTI_SYMBOL_1;
Projenrc[_a] = { fqn: "projen.java.Projenrc", version: "0.27.22" };
function renderJavaOptions(indent, optionsTypeName, optionFqns, initOptions) {
    const imports = new Set();
    if (!initOptions || Object.keys(initOptions).length === 0) {
        return { renderedOptions: '', imports }; // no options
    }
    const lines = [`${optionsTypeName}.builder()`];
    for (const [name, value] of Object.entries(initOptions)) {
        const { javaValue, importName } = toJavaValue(value, name, optionFqns);
        if (importName)
            imports.add(importName);
        lines.push(`.${toJavaProperty(name)}(${javaValue})`);
    }
    lines.push('.build()');
    const renderedOptions = lines.join(`\n${' '.repeat((indent + 1) * 4)}`);
    return { renderedOptions, imports };
}
function toJavaProperty(prop) {
    return prop;
}
function toJavaValue(value, name, optionFqns) {
    if (typeof value === 'string' && optionFqns[name] !== undefined) {
        const parts = optionFqns[name].split('.');
        const base = parts[parts.length - 1];
        const choice = String(value).toUpperCase().replace(/-/g, '_');
        return { javaValue: `${base}.${choice}`, importName: optionFqns[name] };
    }
    else {
        return { javaValue: JSON.stringify(value) };
    }
}
function toJavaFullTypeName(jsiiType) {
    return [jsiiType.namespace, jsiiType.name].filter(x => x).join('.');
}
//# sourceMappingURL=data:application/json;base64,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