"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NpmAccess = exports.NodePackageManager = exports.NodePackage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const url_1 = require("url");
const fs_extra_1 = require("fs-extra");
const semver = require("semver");
const _resolve_1 = require("./_resolve");
const component_1 = require("./component");
const deps_1 = require("./deps");
const json_1 = require("./json");
const util_1 = require("./util");
const UNLICENSED = 'UNLICENSED';
const DEFAULT_NPM_REGISTRY_URL = 'https://registry.npmjs.org/';
const DEFAULT_NPM_TAG = 'latest';
const DEFAULT_NPM_TOKEN_SECRET = 'NPM_TOKEN';
/**
 * (experimental) Represents the npm `package.json` file.
 *
 * @experimental
 */
class NodePackage extends component_1.Component {
    /**
     * @experimental
     */
    constructor(project, options = {}) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p;
        super(project);
        this.keywords = new Set();
        this.bin = {};
        this.engines = {};
        this.packageName = (_b = options.packageName) !== null && _b !== void 0 ? _b : project.name;
        this.projenCommand = (_c = options.projenCommand) !== null && _c !== void 0 ? _c : 'npx projen';
        this.peerDependencyOptions = (_d = options.peerDependencyOptions) !== null && _d !== void 0 ? _d : {};
        this.allowLibraryDependencies = (_e = options.allowLibraryDependencies) !== null && _e !== void 0 ? _e : true;
        this.packageManager = (_f = options.packageManager) !== null && _f !== void 0 ? _f : NodePackageManager.YARN;
        this.entrypoint = (_g = options.entrypoint) !== null && _g !== void 0 ? _g : 'lib/index.js';
        if (this.packageManager === NodePackageManager.YARN) {
            project.root.annotateGenerated('/yarn.lock');
        }
        const { npmDistTag, npmAccess, npmRegistry, npmRegistryUrl, npmTokenSecret } = this.parseNpmOptions(options);
        this.npmDistTag = npmDistTag;
        this.npmAccess = npmAccess;
        this.npmRegistry = npmRegistry;
        this.npmRegistryUrl = npmRegistryUrl;
        this.npmTokenSecret = npmTokenSecret;
        this.processDeps(options);
        const prev = (_h = this.readPackageJson()) !== null && _h !== void 0 ? _h : {};
        // empty objects are here to preserve order for backwards compatibility
        this.manifest = {
            name: this.packageName,
            description: options.description,
            repository: !options.repository ? undefined : {
                type: 'git',
                url: options.repository,
                directory: options.repositoryDirectory,
            },
            bin: () => this.renderBin(),
            scripts: () => this.renderScripts(),
            author: this.renderAuthor(options),
            devDependencies: {},
            peerDependencies: {},
            dependencies: {},
            bundledDependencies: [],
            keywords: () => this.renderKeywords(),
            engines: () => this.renderEngines(),
            main: this.entrypoint !== '' ? this.entrypoint : undefined,
            license: () => { var _b; return (_b = this.license) !== null && _b !== void 0 ? _b : UNLICENSED; },
            homepage: options.homepage,
            publishConfig: () => this.renderPublishConfig(),
            // in release CI builds we bump the version before we run "build" so we want
            // to preserve the version number. otherwise, we always set it to 0.0.0
            version: this.determineVersion(prev === null || prev === void 0 ? void 0 : prev.version),
        };
        // override any scripts from options (if specified)
        for (const [cmdname, shell] of Object.entries((_j = options.scripts) !== null && _j !== void 0 ? _j : {})) {
            project.addTask(cmdname, { exec: shell });
        }
        this.file = new json_1.JsonFile(this.project, 'package.json', {
            obj: this.manifest,
            readonly: false,
        });
        this.addKeywords(...(_k = options.keywords) !== null && _k !== void 0 ? _k : []);
        this.addBin((_l = options.bin) !== null && _l !== void 0 ? _l : {});
        // automatically add all executable files under "bin"
        if ((_m = options.autoDetectBin) !== null && _m !== void 0 ? _m : true) {
            this.autoDiscoverBinaries();
        }
        // node version
        this.minNodeVersion = options.minNodeVersion;
        this.maxNodeVersion = options.maxNodeVersion;
        this.addNodeEngine();
        // license
        if ((_o = options.licensed) !== null && _o !== void 0 ? _o : true) {
            this.license = (_p = options.license) !== null && _p !== void 0 ? _p : 'Apache-2.0';
        }
    }
    /**
     * (experimental) Defines normal dependencies.
     *
     * @param deps Names modules to install.
     * @experimental
     */
    addDeps(...deps) {
        for (const dep of deps) {
            this.project.deps.addDependency(dep, deps_1.DependencyType.RUNTIME);
        }
    }
    /**
     * (experimental) Defines development/test dependencies.
     *
     * @param deps Names modules to install.
     * @experimental
     */
    addDevDeps(...deps) {
        for (const dep of deps) {
            this.project.deps.addDependency(dep, deps_1.DependencyType.BUILD);
        }
    }
    /**
     * (experimental) Defines peer dependencies.
     *
     * When adding peer dependencies, a devDependency will also be added on the
     * pinned version of the declared peer. This will ensure that you are testing
     * your code against the minimum version required from your consumers.
     *
     * @param deps Names modules to install.
     * @experimental
     */
    addPeerDeps(...deps) {
        if (Object.keys(deps).length && !this.allowLibraryDependencies) {
            throw new Error(`cannot add peer dependencies to an APP project: ${Object.keys(deps).join(',')}`);
        }
        for (const dep of deps) {
            this.project.deps.addDependency(dep, deps_1.DependencyType.PEER);
        }
    }
    /**
     * (experimental) Defines bundled dependencies.
     *
     * Bundled dependencies will be added as normal dependencies as well as to the
     * `bundledDependencies` section of your `package.json`.
     *
     * @param deps Names modules to install.
     * @experimental
     */
    addBundledDeps(...deps) {
        if (deps.length && !this.allowLibraryDependencies) {
            throw new Error(`cannot add bundled dependencies to an APP project: ${deps.join(',')}`);
        }
        for (const dep of deps) {
            this.project.deps.addDependency(dep, deps_1.DependencyType.BUNDLED);
        }
    }
    /**
     * (experimental) Adds an `engines` requirement to your package.
     *
     * @param engine The engine (e.g. `node`).
     * @param version The semantic version requirement (e.g. `^10`).
     * @experimental
     */
    addEngine(engine, version) {
        this.engines[engine] = version;
    }
    /**
     * (experimental) Adds keywords to package.json (deduplicated).
     *
     * @param keywords The keywords to add.
     * @experimental
     */
    addKeywords(...keywords) {
        for (const k of keywords) {
            this.keywords.add(k);
        }
    }
    /**
     * @experimental
     */
    addBin(bins) {
        for (const [k, v] of Object.entries(bins)) {
            this.bin[k] = v;
        }
    }
    /**
     * (experimental) Override the contents of an npm package.json script.
     *
     * @param name The script name.
     * @param command The command to execute.
     * @experimental
     */
    setScript(name, command) {
        this.file.addOverride(`scripts.${name}`, command);
    }
    /**
     * (experimental) Removes the npm script (always successful).
     *
     * @param name The name of the script.
     * @experimental
     */
    removeScript(name) {
        this.file.addDeletionOverride(`scripts.${name}`);
    }
    /**
     * (deprecated) Indicates if a script by the given name is defined.
     *
     * @param name The name of the script.
     * @deprecated Use `project.tasks.tryFind(name)`
     */
    hasScript(name) {
        return this.project.tasks.tryFind(name) !== undefined;
    }
    /**
     * (experimental) Directly set fields in `package.json`.
     *
     * @param name field name.
     * @param value field value.
     * @experimental
     * @escape true
     */
    addField(name, value) {
        this.manifest[name] = value;
    }
    /**
     * (experimental) Sets the package version.
     *
     * @param version Package version.
     * @experimental
     */
    addVersion(version) {
        this.manifest.version = version;
    }
    /**
     * (experimental) Returns the command to execute in order to install all dependencies (always frozen).
     *
     * @experimental
     */
    get installCommand() {
        return this.renderInstallCommand(true);
    }
    /**
     * (experimental) Renders `yarn install` or `npm install` with lockfile update (not frozen).
     *
     * @experimental
     */
    get installAndUpdateLockfileCommand() {
        return this.renderInstallCommand(false);
    }
    /**
     * (experimental) Render a package manager specific command to upgrade all requested dependencies.
     *
     * @experimental
     */
    renderUpgradePackagesCommand(exclude, include) {
        const project = this.project;
        function upgradePackages(command) {
            return () => {
                if (exclude.length === 0 && !include) {
                    // request to upgrade all packages
                    // separated for asthetic reasons.
                    return command;
                }
                // filter by exclude and include.
                return `${command} ${project.deps.all
                    .map(d => d.name)
                    .filter(d => include ? include.includes(d) : true)
                    .filter(d => !exclude.includes(d))
                    .join(' ')}`;
            };
        }
        let lazy = undefined;
        switch (this.packageManager) {
            case NodePackageManager.YARN:
                lazy = upgradePackages('yarn upgrade');
                break;
            case NodePackageManager.NPM:
                lazy = upgradePackages('npm update');
                break;
            case NodePackageManager.PNPM:
                lazy = upgradePackages('pnpm update');
                break;
            default:
                throw new Error(`unexpected package manager ${this.packageManager}`);
        }
        // return a lazy function so that dependencies include ones that were
        // added post project instantiation (i.e using project.addDeps)
        return lazy;
    }
    // ---------------------------------------------------------------------------------------
    /**
     * (experimental) Called before synthesis.
     *
     * @experimental
     */
    preSynthesize() {
        super.preSynthesize();
        this._renderedDeps = this.renderDependencies();
    }
    /**
     * (experimental) Called after synthesis.
     *
     * Order is *not* guaranteed.
     *
     * @experimental
     */
    postSynthesize() {
        super.postSynthesize();
        const outdir = this.project.outdir;
        util_1.exec(this.renderInstallCommand(this.isAutomatedBuild), { cwd: outdir });
        this.resolveDepsAndWritePackageJson(outdir);
    }
    /**
     * Returns `true` if we are running within a CI build.
     */
    get isAutomatedBuild() {
        return util_1.isTruthy(process.env.CI);
    }
    determineVersion(currVersion) {
        if (!this.isReleaseBuild) {
            return '0.0.0';
        }
        return currVersion !== null && currVersion !== void 0 ? currVersion : '0.0.0';
    }
    /**
     * Returns `true` if this is a CI release build.
     */
    get isReleaseBuild() {
        return util_1.isTruthy(process.env.RELEASE);
    }
    // -------------------------------------------------------------------------------------------
    parseNpmOptions(options) {
        var _b, _c, _d;
        let npmRegistryUrl = options.npmRegistryUrl;
        if (options.npmRegistry) {
            if (npmRegistryUrl) {
                throw new Error('cannot use the deprecated "npmRegistry" together with "npmRegistryUrl". please use "npmRegistryUrl" instead.');
            }
            npmRegistryUrl = `https://${options.npmRegistry}`;
        }
        const npmr = url_1.parse(npmRegistryUrl !== null && npmRegistryUrl !== void 0 ? npmRegistryUrl : DEFAULT_NPM_REGISTRY_URL);
        if (!npmr || !npmr.hostname || !npmr.href) {
            throw new Error(`unable to determine npm registry host from url ${npmRegistryUrl}. Is this really a URL?`);
        }
        const npmAccess = (_b = options.npmAccess) !== null && _b !== void 0 ? _b : defaultNpmAccess(this.packageName);
        if (!isScoped(this.packageName) && npmAccess === NpmAccess.RESTRICTED) {
            throw new Error(`"npmAccess" cannot be RESTRICTED for non-scoped npm package "${this.packageName}"`);
        }
        return {
            npmDistTag: (_c = options.npmDistTag) !== null && _c !== void 0 ? _c : DEFAULT_NPM_TAG,
            npmAccess,
            npmRegistry: npmr.hostname,
            npmRegistryUrl: npmr.href,
            npmTokenSecret: (_d = options.npmTokenSecret) !== null && _d !== void 0 ? _d : DEFAULT_NPM_TOKEN_SECRET,
        };
    }
    addNodeEngine() {
        if (!this.minNodeVersion && !this.maxNodeVersion) {
            return;
        }
        let nodeVersion = '';
        if (this.minNodeVersion) {
            nodeVersion += `>= ${this.minNodeVersion}`;
        }
        if (this.maxNodeVersion) {
            nodeVersion += ` <= ${this.maxNodeVersion}`;
        }
        this.addEngine('node', nodeVersion);
    }
    renderInstallCommand(frozen) {
        switch (this.packageManager) {
            case NodePackageManager.YARN:
                return [
                    'yarn install',
                    '--check-files',
                    ...frozen ? ['--frozen-lockfile'] : [],
                ].join(' ');
            case NodePackageManager.NPM:
                return frozen
                    ? 'npm ci'
                    : 'npm install';
            case NodePackageManager.PNPM:
                return [
                    'pnpm i',
                    ...frozen ? ['--frozen-lockfile'] : [],
                ].join(' ');
            default:
                throw new Error(`unexpected package manager ${this.packageManager}`);
        }
    }
    processDeps(options) {
        var _b, _c, _d, _e;
        this.addDeps(...(_b = options.deps) !== null && _b !== void 0 ? _b : []);
        this.addDevDeps(...(_c = options.devDeps) !== null && _c !== void 0 ? _c : []);
        this.addPeerDeps(...(_d = options.peerDeps) !== null && _d !== void 0 ? _d : []);
        this.addBundledDeps(...(_e = options.bundledDeps) !== null && _e !== void 0 ? _e : []);
    }
    renderDependencies() {
        var _b, _c, _d;
        const devDependencies = {};
        const peerDependencies = {};
        const dependencies = {};
        const bundledDependencies = new Array();
        // synthetic dependencies: add a pinned build dependency to ensure we are
        // testing against the minimum requirement of the peer.
        const pinned = (_b = this.peerDependencyOptions.pinnedDevDependency) !== null && _b !== void 0 ? _b : true;
        if (pinned) {
            for (const dep of this.project.deps.all.filter(d => d.type === deps_1.DependencyType.PEER)) {
                let req = dep.name;
                if (dep.version) {
                    const ver = (_c = semver.minVersion(dep.version)) === null || _c === void 0 ? void 0 : _c.version;
                    if (!ver) {
                        throw new Error(`unable to determine minimum semver for peer dependency ${dep.name}@${dep.version}`);
                    }
                    req += '@' + ver;
                }
                this.addDevDeps(req);
            }
        }
        for (const dep of this.project.deps.all) {
            const version = (_d = dep.version) !== null && _d !== void 0 ? _d : '*';
            switch (dep.type) {
                case deps_1.DependencyType.BUNDLED:
                    bundledDependencies.push(dep.name);
                    if (this.project.deps.all.find(d => d.name === dep.name && d.type === deps_1.DependencyType.PEER)) {
                        throw new Error(`unable to bundle "${dep.name}". it cannot appear as a peer dependency`);
                    }
                    // also add as a runtime dependency
                    dependencies[dep.name] = version;
                    break;
                case deps_1.DependencyType.PEER:
                    peerDependencies[dep.name] = version;
                    break;
                case deps_1.DependencyType.RUNTIME:
                    dependencies[dep.name] = version;
                    break;
                case deps_1.DependencyType.TEST:
                case deps_1.DependencyType.DEVENV:
                case deps_1.DependencyType.BUILD:
                    devDependencies[dep.name] = version;
                    break;
            }
        }
        // update the manifest we are about to save into `package.json`
        this.manifest.devDependencies = devDependencies;
        this.manifest.peerDependencies = peerDependencies;
        this.manifest.dependencies = dependencies;
        this.manifest.bundledDependencies = bundledDependencies;
        // nothing further to do if package.json file does not exist
        const pkg = this.readPackageJson();
        if (!pkg) {
            return { devDependencies, peerDependencies, dependencies };
        }
        const readDeps = (user, current = {}) => {
            for (const [name, userVersion] of Object.entries(user)) {
                const currentVersion = current[name];
                // respect user version if it's not '*' or if current version is undefined
                if (userVersion !== '*' || !currentVersion || currentVersion === '*') {
                    continue;
                }
                // memoize current version in memory so it is preserved when saving
                user[name] = currentVersion;
            }
            // report removals
            for (const name of Object.keys(current !== null && current !== void 0 ? current : {})) {
                if (!user[name]) {
                    this.project.logger.verbose(`${name}: removed`);
                }
            }
        };
        readDeps(devDependencies, pkg.devDependencies);
        readDeps(dependencies, pkg.dependencies);
        readDeps(peerDependencies, pkg.peerDependencies);
        return { devDependencies, dependencies, peerDependencies };
    }
    resolveDepsAndWritePackageJson(outdir) {
        const root = path_1.join(outdir, 'package.json');
        const pkg = fs_extra_1.readJsonSync(root);
        const resolveDeps = (current, user) => {
            const result = {};
            current = current !== null && current !== void 0 ? current : {};
            user = user !== null && user !== void 0 ? user : {};
            for (const [name, currentDefinition] of Object.entries(user)) {
                // find actual version from node_modules
                let desiredVersion = currentDefinition;
                if (currentDefinition === '*') {
                    try {
                        const modulePath = require.resolve(`${name}/package.json`, { paths: [outdir] });
                        const module = fs_extra_1.readJsonSync(modulePath);
                        desiredVersion = `^${module.version}`;
                    }
                    catch (e) { }
                    if (!desiredVersion) {
                        this.project.logger.warn(`unable to resolve version for ${name} from installed modules`);
                        continue;
                    }
                }
                if (currentDefinition !== desiredVersion) {
                    this.project.logger.verbose(`${name}: ${currentDefinition} => ${desiredVersion}`);
                }
                result[name] = desiredVersion;
            }
            // print removed packages
            for (const name of Object.keys(current)) {
                if (!result[name]) {
                    this.project.logger.verbose(`${name} removed`);
                }
            }
            return util_1.sorted(result);
        };
        const rendered = this._renderedDeps;
        if (!rendered) {
            throw new Error('assertion failed');
        }
        pkg.dependencies = resolveDeps(pkg.dependencies, rendered.dependencies);
        pkg.devDependencies = resolveDeps(pkg.devDependencies, rendered.devDependencies);
        pkg.peerDependencies = resolveDeps(pkg.peerDependencies, rendered.peerDependencies);
        util_1.writeFile(root, JSON.stringify(pkg, undefined, 2));
    }
    renderPublishConfig() {
        // omit values if they are the same as the npm defaults
        return _resolve_1.resolve({
            registry: this.npmRegistryUrl !== DEFAULT_NPM_REGISTRY_URL ? this.npmRegistryUrl : undefined,
            tag: this.npmDistTag !== DEFAULT_NPM_TAG ? this.npmDistTag : undefined,
            access: this.npmAccess !== defaultNpmAccess(this.packageName) ? this.npmAccess : undefined,
        }, { omitEmpty: true });
    }
    renderKeywords() {
        const kwds = Array.from(this.keywords);
        return util_1.sorted(kwds.sort());
    }
    renderEngines() {
        return util_1.sorted(this.engines);
    }
    autoDiscoverBinaries() {
        const binrel = 'bin';
        const bindir = path_1.join(this.project.outdir, binrel);
        if (fs_extra_1.existsSync(bindir)) {
            for (const file of fs_extra_1.readdirSync(bindir)) {
                try {
                    fs_extra_1.accessSync(path_1.join(bindir, file), fs_extra_1.constants.X_OK);
                    this.bin[file] = path_1.join(binrel, file).replace(/\\/g, '/');
                }
                catch (e) {
                    // not executable, skip
                }
            }
        }
    }
    renderAuthor(options) {
        var _b;
        let author;
        if (options.authorName) {
            author = {
                name: options.authorName,
                email: options.authorEmail,
                url: options.authorUrl,
                organization: (_b = options.authorOrganization) !== null && _b !== void 0 ? _b : false,
            };
        }
        else {
            if (options.authorEmail || options.authorUrl || options.authorOrganization !== undefined) {
                throw new Error('"authorName" is required if specifying "authorEmail" or "authorUrl"');
            }
        }
        return author;
    }
    renderBin() {
        return util_1.sorted(this.bin);
    }
    renderScripts() {
        const result = {};
        for (const task of this.project.tasks.all) {
            result[task.name] = this.npmScriptForTask(task);
        }
        return result;
    }
    npmScriptForTask(task) {
        return `${this.projenCommand} ${task.name}`;
    }
    readPackageJson() {
        const file = path_1.join(this.project.outdir, 'package.json');
        if (!fs_extra_1.existsSync(file)) {
            return undefined;
        }
        return fs_extra_1.readJsonSync(file);
    }
}
exports.NodePackage = NodePackage;
_a = JSII_RTTI_SYMBOL_1;
NodePackage[_a] = { fqn: "projen.NodePackage", version: "0.27.22" };
/**
 * (experimental) The node package manager to use.
 *
 * @experimental
 */
var NodePackageManager;
(function (NodePackageManager) {
    NodePackageManager["YARN"] = "yarn";
    NodePackageManager["NPM"] = "npm";
    NodePackageManager["PNPM"] = "pnpm";
})(NodePackageManager = exports.NodePackageManager || (exports.NodePackageManager = {}));
/**
 * (experimental) Npm package access level.
 *
 * @experimental
 */
var NpmAccess;
(function (NpmAccess) {
    NpmAccess["PUBLIC"] = "public";
    NpmAccess["RESTRICTED"] = "restricted";
})(NpmAccess = exports.NpmAccess || (exports.NpmAccess = {}));
/**
 * Determines if an npm package is "scoped" (i.e. it starts with "xxx@").
 */
function isScoped(packageName) {
    return packageName.includes('@');
}
function defaultNpmAccess(packageName) {
    return isScoped(packageName) ? NpmAccess.RESTRICTED : NpmAccess.PUBLIC;
}
//# sourceMappingURL=data:application/json;base64,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