"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProjectType = exports.Project = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const cleanup_1 = require("./cleanup");
const clobber_1 = require("./clobber");
const common_1 = require("./common");
const deps_1 = require("./deps");
const file_1 = require("./file");
const gitattributes_1 = require("./git/gitattributes");
const github_1 = require("./github");
const stale_1 = require("./github/stale");
const gitpod_1 = require("./gitpod");
const ignore_file_1 = require("./ignore-file");
const render_options_1 = require("./javascript/render-options");
const json_1 = require("./json");
const logger_1 = require("./logger");
const object_file_1 = require("./object-file");
const readme_1 = require("./readme");
const tasks_1 = require("./tasks/tasks");
const util_1 = require("./util");
const vscode_1 = require("./vscode");
/**
 * (experimental) Base project.
 *
 * @experimental
 */
class Project {
    /**
     * @experimental
     */
    constructor(options) {
        var _b, _c, _d, _e, _f, _g, _h;
        this._components = new Array();
        this.subprojects = new Array();
        this.tips = new Array();
        this.newProject = render_options_1.resolveNewProject(options);
        this.name = options.name;
        this.parent = options.parent;
        this.excludeFromCleanup = [];
        this.projectType = (_b = options.projectType) !== null && _b !== void 0 ? _b : ProjectType.UNKNOWN;
        if (this.parent && options.outdir && path.isAbsolute(options.outdir)) {
            throw new Error('"outdir" must be a relative path');
        }
        let outdir;
        if (options.parent) {
            if (!options.outdir) {
                throw new Error('"outdir" must be specified for subprojects');
            }
            outdir = path.join(options.parent.outdir, options.outdir);
        }
        else {
            outdir = (_c = options.outdir) !== null && _c !== void 0 ? _c : '.';
        }
        this.outdir = path.resolve(outdir);
        this.root = this.parent ? this.parent.root : this;
        // must happen after this.outdir, this.parent and this.root are initialized
        (_d = this.parent) === null || _d === void 0 ? void 0 : _d._addSubProject(this);
        // ------------------------------------------------------------------------
        this.gitattributes = new gitattributes_1.GitAttributesFile(this);
        this.annotateGenerated('/.projen/**'); // contents  of the .projen/ directory are generated by projen
        this.annotateGenerated(`/${this.gitattributes.path}`); // the .gitattributes file itself is generated
        this.gitignore = new ignore_file_1.IgnoreFile(this, '.gitignore');
        this.gitignore.exclude('node_modules/'); // created by running `npx projen`
        this.gitignore.include(`/${this.gitattributes.path}`);
        // oh no: tasks depends on gitignore so it has to be initialized after
        // smells like dep injectionn but god forbid.
        this.tasks = new tasks_1.Tasks(this);
        this.deps = new deps_1.Dependencies(this);
        this.logger = new logger_1.Logger(this, options.logging);
        // we only allow these global services to be used in root projects
        const github = (_e = options.github) !== null && _e !== void 0 ? _e : (this.parent ? false : true);
        this.github = github ? new github_1.GitHub(this, {
            mergify: options.mergify,
            ...options.githubOptions,
        }) : undefined;
        const vscode = (_f = options.vscode) !== null && _f !== void 0 ? _f : (this.parent ? false : true);
        this.vscode = vscode ? new vscode_1.VsCode(this) : undefined;
        this.gitpod = options.gitpod ? new gitpod_1.Gitpod(this) : undefined;
        this.devContainer = options.devContainer ? new vscode_1.DevContainer(this) : undefined;
        if ((_g = options.clobber) !== null && _g !== void 0 ? _g : true) {
            new clobber_1.Clobber(this);
        }
        new readme_1.SampleReadme(this, options.readme);
        if (options.autoApproveOptions && this.github) {
            this.autoApprove = new github_1.AutoApprove(this.github, options.autoApproveOptions);
        }
        const stale = (_h = options.stale) !== null && _h !== void 0 ? _h : true;
        if (stale && this.github) {
            new stale_1.Stale(this.github, options.staleOptions);
        }
    }
    /**
     * (experimental) Returns all the components within this project.
     *
     * @experimental
     */
    get components() {
        return [...this._components];
    }
    /**
     * (experimental) All files in this project.
     *
     * @experimental
     */
    get files() {
        const isFile = (c) => c instanceof file_1.FileBase;
        return this._components.filter(isFile).sort((f1, f2) => f1.path.localeCompare(f2.path));
    }
    /**
     * (experimental) Adds a new task to this project.
     *
     * This will fail if the project already has
     * a task with this name.
     *
     * @param name The task name to add.
     * @param props Task properties.
     * @experimental
     */
    addTask(name, props = {}) {
        return this.tasks.addTask(name, props);
    }
    /**
     * (experimental) Removes a task from a project.
     *
     * @param name The name of the task to remove.
     * @returns The `Task` that was removed, otherwise `undefined`.
     * @experimental
     */
    removeTask(name) {
        return this.tasks.removeTask(name);
    }
    /**
     * (experimental) Finds a file at the specified relative path within this project and all its subprojects.
     *
     * @param filePath The file path.
     * @returns a `FileBase` or undefined if there is no file in that path
     * @experimental
     */
    tryFindFile(filePath) {
        const absolute = path.isAbsolute(filePath) ? filePath : path.resolve(this.outdir, filePath);
        for (const file of this.files) {
            if (absolute === file.absolutePath) {
                return file;
            }
        }
        for (const child of this.subprojects) {
            const file = child.tryFindFile(absolute);
            if (file) {
                return file;
            }
        }
        return undefined;
    }
    /**
     * (deprecated) Finds a json file by name.
     *
     * @param filePath The file path.
     * @deprecated use `tryFindObjectFile`
     */
    tryFindJsonFile(filePath) {
        const file = this.tryFindObjectFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof json_1.JsonFile)) {
            throw new Error(`found file ${filePath} but it is not a JsonFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * (experimental) Finds an object file (like JsonFile, YamlFile, etc.) by name.
     *
     * @param filePath The file path.
     * @experimental
     */
    tryFindObjectFile(filePath) {
        const file = this.tryFindFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof object_file_1.ObjectFile)) {
            throw new Error(`found file ${filePath} but it is not a ObjectFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * (deprecated) Prints a "tip" message during synthesis.
     *
     * @param message The message.
     * @deprecated - use `project.logger.info(message)` to show messages during synthesis
     */
    addTip(message) {
        this.tips.push(message);
    }
    /**
     * (experimental) Exclude the matching files from pre-synth cleanup.
     *
     * Can be used when, for example, some
     * source files include the projen marker and we don't want them to be erased during synth.
     *
     * @param globs The glob patterns to match.
     * @experimental
     */
    addExcludeFromCleanup(...globs) {
        this.excludeFromCleanup.push(...globs);
    }
    /**
     * (experimental) Returns the shell command to execute in order to run a task.
     *
     * By default, this is `npx projen@<version> <task>`
     *
     * @param task The task for which the command is required.
     * @experimental
     */
    runTaskCommand(task) {
        return `npx projen@${common_1.PROJEN_VERSION} ${task.name}`;
    }
    /**
     * (experimental) Exclude these files from the bundled package.
     *
     * Implemented by project types based on the
     * packaging mechanism. For example, `NodeProject` delegates this to `.npmignore`.
     *
     * @param _pattern The glob pattern to exclude.
     * @experimental
     */
    addPackageIgnore(_pattern) {
        // nothing to do at the abstract level
    }
    /**
     * (experimental) Adds a .gitignore pattern.
     *
     * @param pattern The glob pattern to ignore.
     * @experimental
     */
    addGitIgnore(pattern) {
        this.gitignore.addPatterns(pattern);
    }
    /**
     * (experimental) Marks the provided file(s) as being generated.
     *
     * This is achieved using the
     * github-linguist attributes. Generated files do not count against the
     * repository statistics and language breakdown.
     *
     * @param glob the glob pattern to match (could be a file path).
     * @see https://github.com/github/linguist/blob/master/docs/overrides.md
     * @experimental
     */
    annotateGenerated(glob) {
        this.gitattributes.addAttributes(glob, 'linguist-generated');
    }
    /**
     * (experimental) Synthesize all project files into `outdir`.
     *
     * 1. Call "this.preSynthesize()"
     * 2. Delete all generated files
     * 3. Synthesize all sub-projects
     * 4. Synthesize all components of this project
     * 5. Call "postSynthesize()" for all components of this project
     * 6. Call "this.postSynthesize()"
     *
     * @experimental
     */
    synth() {
        const outdir = this.outdir;
        this.logger.info('Synthesizing project...');
        this.preSynthesize();
        for (const comp of this._components) {
            comp.preSynthesize();
        }
        // we exclude all subproject directories to ensure that when subproject.synth()
        // gets called below after cleanup(), subproject generated files are left intact
        for (const subproject of this.subprojects) {
            this.addExcludeFromCleanup(subproject.outdir + '/**');
        }
        // delete all generated files before we start synthesizing new ones
        cleanup_1.cleanup(outdir, this.excludeFromCleanup);
        for (const subproject of this.subprojects) {
            subproject.synth();
        }
        for (const comp of this._components) {
            comp.synthesize();
        }
        if (!util_1.isTruthy(process.env.PROJEN_DISABLE_POST)) {
            for (const comp of this._components) {
                comp.postSynthesize();
            }
            // project-level hook
            this.postSynthesize();
        }
        this.logger.info('Synthesis complete');
    }
    /**
     * (experimental) Called before all components are synthesized.
     *
     * @experimental
     */
    preSynthesize() { }
    /**
     * (experimental) Called after all components are synthesized.
     *
     * Order is *not* guaranteed.
     *
     * @experimental
     */
    postSynthesize() { }
    /**
     * Adds a component to the project.
     * @internal
     */
    _addComponent(component) {
        this._components.push(component);
    }
    /**
     * Adds a sub-project to this project.
     *
     * This is automatically called when a new project is created with `parent`
     * pointing to this project, so there is no real need to call this manually.
     *
     * @param sub-project The child project to add.
     * @internal
     */
    _addSubProject(subproject) {
        if (subproject.parent !== this) {
            throw new Error('"parent" of child project must be this project');
        }
        // check that `outdir` is exclusive
        for (const p of this.subprojects) {
            if (path.resolve(p.outdir) === path.resolve(subproject.outdir)) {
                throw new Error(`there is already a sub-project with "outdir": ${subproject.outdir}`);
            }
        }
        this.subprojects.push(subproject);
    }
}
exports.Project = Project;
_a = JSII_RTTI_SYMBOL_1;
Project[_a] = { fqn: "projen.Project", version: "0.27.22" };
/**
 * (experimental) The name of the default task (the task executed when `projen` is run without arguments).
 *
 * Normally
 * this task should synthesize the project files.
 *
 * @experimental
 */
Project.DEFAULT_TASK = 'default';
/**
 * (experimental) Which type of project this is.
 *
 * @experimental
 */
var ProjectType;
(function (ProjectType) {
    ProjectType["UNKNOWN"] = "unknown";
    ProjectType["LIB"] = "lib";
    ProjectType["APP"] = "app";
})(ProjectType = exports.ProjectType || (exports.ProjectType = {}));
//# sourceMappingURL=data:application/json;base64,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