"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.bump = void 0;
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const logging = require("../logging");
const util_1 = require("../util");
/**
 * Resolves the latest version from git tags and uses `standard-version` to bump
 * to the next version based on commits.
 *
 * This expects `standard-version` to be installed in the path.
 *
 * @param cwd working directory (git repository)
 * @param options options
 */
async function bump(cwd, options) {
    const versionFile = path_1.join(cwd, options.versionFile);
    const prerelease = options.prerelease;
    const major = options.majorVersion;
    const changelogFile = path_1.join(cwd, options.changelog);
    const bumpFile = path_1.join(cwd, options.bumpFile);
    await fs_extra_1.mkdirp(path_1.dirname(bumpFile));
    await fs_extra_1.mkdirp(path_1.dirname(changelogFile));
    // filter only tags for this major version if specified (start  with "vNN.").
    const prefix = major ? `v${major}.*` : 'v*';
    const listGitTags = [
        'git',
        '-c "versionsort.suffix=-"',
        'tag',
        '--sort="-version:refname"',
        '--list',
        `"${prefix}"`,
    ].join(' ');
    const stdout = util_1.execCapture(listGitTags, { cwd }).toString('utf8');
    let tags = stdout === null || stdout === void 0 ? void 0 : stdout.split('\n');
    // if "pre" is set, filter versions that end with "-PRE.ddd".
    if (prerelease) {
        tags = tags.filter(x => new RegExp(`-${prerelease}\.[0-9]+$`).test(x));
    }
    tags = tags.filter(x => x);
    // if a pre-release tag is used, then add it to the initial version
    let firstRelease = false;
    let latest;
    if (tags.length > 0) {
        latest = tags[0];
    }
    else {
        const initial = `v${major !== null && major !== void 0 ? major : 0}.0.0`;
        latest = prerelease ? `${initial}-${prerelease}.0` : initial;
        firstRelease = true;
    }
    // remove "v" prefix (if exists)
    if (latest.startsWith('v')) {
        latest = latest.substr(1);
    }
    const content = await tryReadVersionFile(versionFile);
    // update version
    content.version = latest;
    logging.info(`Update ${versionFile} to latest resolved version: ${latest}`);
    await fs_extra_1.writeFile(versionFile, JSON.stringify(content, undefined, 2));
    // create a standard-version configuration file
    const rcfile = path_1.join(cwd, '.versionrc.json');
    await fs_extra_1.writeFile(rcfile, JSON.stringify({
        packageFiles: [{
                filename: versionFile,
                type: 'json',
            }],
        bumpFiles: [{
                filename: versionFile,
                type: 'json',
            }],
        commitAll: false,
        infile: changelogFile,
        prerelease: prerelease,
        header: '',
        skip: {
            commit: true,
            tag: true,
        },
    }, undefined, 2));
    const cmd = ['npx', 'standard-version@^9'];
    if (firstRelease) {
        cmd.push('--first-release');
    }
    util_1.exec(cmd.join(' '), { cwd });
    await fs_extra_1.remove(rcfile);
    const newVersion = (await tryReadVersionFile(versionFile)).version;
    if (!newVersion) {
        throw new Error(`bump failed: ${versionFile} does not have a new version`);
    }
    // if MAJOR is defined, ensure that the new version is within the same major version
    if (major) {
        if (!newVersion.startsWith(`${major}.`)) {
            throw new Error(`bump failed: this branch is configured to only publish v${major} releases - bump resulted in ${newVersion}`);
        }
    }
    await fs_extra_1.writeFile(bumpFile, newVersion);
}
exports.bump = bump;
async function tryReadVersionFile(versionFile) {
    if (!(await fs_extra_1.pathExists(versionFile))) {
        return {};
    }
    return JSON.parse(await fs_extra_1.readFile(versionFile, 'utf8'));
}
//# sourceMappingURL=data:application/json;base64,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