"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Release = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const github_1 = require("../github");
const workflows_model_1 = require("../github/workflows-model");
const version_1 = require("../version");
const publisher_1 = require("./publisher");
const BUILD_JOBID = 'release';
const GIT_REMOTE_STEPID = 'git_remote';
const LATEST_COMMIT_OUTPUT = 'latest_commit';
/**
 * (experimental) Manages releases (currently through GitHub workflows).
 *
 * By default, no branches are released. To add branches, call `addBranch()`.
 *
 * @experimental
 */
class Release extends component_1.Component {
    /**
     * @experimental
     */
    constructor(project, options) {
        var _b, _c, _d, _e, _f, _g, _h, _j;
        super(project);
        this.branches = new Array();
        this.jobs = {};
        if (Array.isArray(options.releaseBranches)) {
            throw new Error('"releaseBranches" is no longer an array. See type annotations');
        }
        this.buildTask = options.task;
        this.preBuildSteps = (_b = options.releaseWorkflowSetupSteps) !== null && _b !== void 0 ? _b : [];
        this.postBuildSteps = (_c = options.postBuildSteps) !== null && _c !== void 0 ? _c : [];
        this.antitamper = (_d = options.antitamper) !== null && _d !== void 0 ? _d : true;
        this.artifactsDirectory = (_e = options.artifactsDirectory) !== null && _e !== void 0 ? _e : 'dist';
        this.versionFile = options.versionFile;
        this.releaseSchedule = options.releaseSchedule;
        this.releaseEveryCommit = (_f = options.releaseEveryCommit) !== null && _f !== void 0 ? _f : true;
        this.containerImage = options.workflowContainerImage;
        this.version = new version_1.Version(project, {
            versionInputFile: this.versionFile,
            artifactsDirectory: this.artifactsDirectory,
        });
        this.publisher = new publisher_1.Publisher(project, {
            artifactName: this.artifactsDirectory,
            condition: `needs.${BUILD_JOBID}.outputs.${LATEST_COMMIT_OUTPUT} == github.sha`,
            buildJobId: BUILD_JOBID,
            jsiiReleaseVersion: options.jsiiReleaseVersion,
        });
        const githubRelease = (_g = options.githubRelease) !== null && _g !== void 0 ? _g : true;
        if (githubRelease) {
            this.publisher.publishToGitHubReleases({
                changelogFile: this.version.changelogFileName,
                versionFile: this.version.versionFileName,
            });
        }
        // add the default branch
        this.defaultBranch = {
            name: options.branch,
            prerelease: options.prerelease,
            majorVersion: options.majorVersion,
            workflowName: (_h = options.releaseWorkflowName) !== null && _h !== void 0 ? _h : 'release',
        };
        this.branches.push(this.defaultBranch);
        for (const [name, opts] of Object.entries((_j = options.releaseBranches) !== null && _j !== void 0 ? _j : {})) {
            this.addBranch(name, opts);
        }
    }
    /**
     * (experimental) Adds a release branch.
     *
     * It is a git branch from which releases are published. If a project has more than one release
     * branch, we require that `majorVersion` is also specified for the primary branch in order to
     * ensure branches always release the correct version.
     *
     * @param branch The branch to monitor (e.g. `main`, `v2.x`).
     * @param options Branch definition.
     * @experimental
     */
    addBranch(branch, options) {
        if (this.branches.find(b => b.name === branch)) {
            throw new Error(`The release branch ${branch} is already defined`);
        }
        // if we add a branch, we require that the default branch will also define a
        // major version.
        if (this.defaultBranch.majorVersion === undefined) {
            throw new Error('you must specify "majorVersion" for the default branch when adding multiple release branches');
        }
        this.branches.push({
            name: branch,
            ...options,
        });
    }
    /**
     * (experimental) Adds jobs to all release workflows.
     *
     * @param jobs The jobs to add (name => job).
     * @experimental
     */
    addJobs(jobs) {
        for (const [name, job] of Object.entries(jobs)) {
            this.jobs[name] = job;
        }
    }
    // render a workflow per branch and all the jobs to it
    /**
     * (experimental) Called before synthesis.
     *
     * @experimental
     */
    preSynthesize() {
        for (const branch of this.branches) {
            const workflow = this.createWorkflow(branch);
            if (workflow) {
                workflow.addJobs(this.publisher.render());
                workflow.addJobs(this.jobs);
            }
        }
    }
    /**
     * @returns a workflow or `undefined` if github integration is disabled.
     */
    createWorkflow(branch) {
        var _b;
        const workflowName = (_b = branch.workflowName) !== null && _b !== void 0 ? _b : `release-${branch.name}`;
        // to avoid race conditions between two commits trying to release the same
        // version, we check if the head sha is identical to the remote sha. if
        // not, we will skip the release and just finish the build.
        const noNewCommits = `\${{ steps.${GIT_REMOTE_STEPID}.outputs.${LATEST_COMMIT_OUTPUT} == github.sha }}`;
        // The arrays are being cloned to avoid accumulating values from previous branches
        const preBuildSteps = [...this.preBuildSteps];
        const env = {
            RELEASE: 'true',
        };
        if (branch.majorVersion !== undefined) {
            env.MAJOR = branch.majorVersion.toString();
        }
        if (branch.prerelease) {
            env.PRERELEASE = branch.prerelease;
        }
        // the "release" task prepares a release but does not publish anything. the
        // output of the release task is: `dist`, `.version.txt`, and
        // `.changelog.md`. this is what publish tasks expect.
        // if this is the release for "main" or "master", just call it "release".
        // otherwise, "release:BRANCH"
        const releaseTaskName = (branch.name === 'main' || branch.name === 'master') ? 'release' : `release:${branch.name}`;
        const releaseTask = this.project.addTask(releaseTaskName, {
            description: `Prepare a release from "${branch.name}" branch`,
            env,
        });
        releaseTask.exec(`rm -fr ${this.artifactsDirectory}`);
        releaseTask.spawn(this.version.bumpTask);
        releaseTask.spawn(this.buildTask);
        releaseTask.spawn(this.version.unbumpTask);
        // anti-tamper check (fails if there were changes to committed files)
        // this will identify any non-committed files generated during build (e.g. test snapshots)
        if (this.antitamper) {
            releaseTask.exec('git diff --ignore-space-at-eol --exit-code');
        }
        const postBuildSteps = [...this.postBuildSteps];
        // check if new commits were pushed to the repo while we were building.
        // if new commits have been pushed, we will cancel this release
        postBuildSteps.push({
            name: 'Check for new commits',
            id: GIT_REMOTE_STEPID,
            run: `echo ::set-output name=${LATEST_COMMIT_OUTPUT}::"$(git ls-remote origin -h \${{ github.ref }} | cut -f1)"`,
        });
        postBuildSteps.push({
            name: 'Upload artifact',
            if: noNewCommits,
            uses: 'actions/upload-artifact@v2.1.1',
            with: {
                name: this.artifactsDirectory,
                path: this.artifactsDirectory,
            },
        });
        if (this.project.github) {
            return new github_1.TaskWorkflow(this.project.github, {
                name: workflowName,
                jobId: BUILD_JOBID,
                outputs: {
                    latest_commit: {
                        stepId: GIT_REMOTE_STEPID,
                        outputName: LATEST_COMMIT_OUTPUT,
                    },
                },
                triggers: {
                    schedule: this.releaseSchedule ? [{ cron: this.releaseSchedule }] : undefined,
                    push: (this.releaseEveryCommit) ? { branches: [branch.name] } : undefined,
                },
                container: this.containerImage ? { image: this.containerImage } : undefined,
                env: {
                    CI: 'true',
                },
                permissions: {
                    contents: workflows_model_1.JobPermission.WRITE,
                },
                checkoutWith: {
                    // we must use 'fetch-depth=0' in order to fetch all tags
                    // otherwise tags are not checked out
                    'fetch-depth': 0,
                },
                preBuildSteps,
                task: releaseTask,
                postBuildSteps,
            });
        }
        else {
            return undefined;
        }
    }
}
exports.Release = Release;
_a = JSII_RTTI_SYMBOL_1;
Release[_a] = { fqn: "projen.release.Release", version: "0.27.22" };
//# sourceMappingURL=data:application/json;base64,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