"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TaskRuntime = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const child_process_1 = require("child_process");
const fs_1 = require("fs");
const os_1 = require("os");
const path_1 = require("path");
const util_1 = require("util");
const chalk = require("chalk");
const logging = require("../logging");
const tasks_1 = require("./tasks");
const ENV_TRIM_LEN = 20;
/**
 * (experimental) The runtime component of the tasks engine.
 *
 * @experimental
 */
class TaskRuntime {
    /**
     * @experimental
     */
    constructor(workdir) {
        this.workdir = path_1.resolve(workdir);
        const manifestPath = path_1.join(this.workdir, tasks_1.Tasks.MANIFEST_FILE);
        this.manifest = fs_1.existsSync(manifestPath)
            ? JSON.parse(fs_1.readFileSync(manifestPath, 'utf-8'))
            : { tasks: {} };
    }
    /**
     * (experimental) The tasks in this project.
     *
     * @experimental
     */
    get tasks() {
        var _b;
        return Object.values((_b = this.manifest.tasks) !== null && _b !== void 0 ? _b : {});
    }
    /**
     * (experimental) Find a task by name, or `undefined` if not found.
     *
     * @experimental
     */
    tryFindTask(name) {
        if (!this.manifest.tasks) {
            return undefined;
        }
        return this.manifest.tasks[name];
    }
    /**
     * (experimental) Runs the task.
     *
     * @param name The task name.
     * @experimental
     */
    runTask(name, parents = []) {
        const task = this.tryFindTask(name);
        if (!task) {
            throw new Error(`cannot find command ${task}`);
        }
        new RunTask(this, task, parents);
    }
}
exports.TaskRuntime = TaskRuntime;
_a = JSII_RTTI_SYMBOL_1;
TaskRuntime[_a] = { fqn: "projen.tasks.TaskRuntime", version: "0.27.22" };
class RunTask {
    constructor(runtime, task, parents = []) {
        var _b, _c, _d;
        this.runtime = runtime;
        this.task = task;
        this.env = {};
        this.workdir = (_b = task.cwd) !== null && _b !== void 0 ? _b : this.runtime.workdir;
        this.parents = parents;
        this.env = this.resolveEnvironment(parents);
        const envlogs = [];
        for (const [k, v] of Object.entries(this.env)) {
            const vv = v !== null && v !== void 0 ? v : '';
            const trimmed = vv.length > ENV_TRIM_LEN ? vv.substr(0, ENV_TRIM_LEN) + '...' : vv;
            envlogs.push(`${k}=${trimmed}`);
        }
        if (envlogs.length) {
            this.log(chalk.gray(`${chalk.underline('env')}: ${envlogs.join(' ')}`));
        }
        // evaluate condition
        if (!this.evalCondition(task)) {
            this.log('condition exited with non-zero - skipping');
            return;
        }
        if (!task.steps || task.steps.length === 0) {
            logging.verbose(this.fmtLog('No actions have been specified for this task.'));
            return;
        }
        // verify we required environment variables are defined
        const merged = { ...process.env, ...this.env };
        const missing = new Array();
        for (const name of (_c = task.requiredEnv) !== null && _c !== void 0 ? _c : []) {
            if ((!(name in merged))) {
                missing.push(name);
            }
        }
        if (missing.length > 0) {
            throw new Error(`missing required environment variables: ${missing.join(',')}`);
        }
        for (const step of task.steps) {
            if (step.say) {
                logging.info(this.fmtLog(step.say));
            }
            if (step.spawn) {
                this.runtime.runTask(step.spawn, [...this.parents, this.task.name]);
            }
            const execs = step.exec ? [step.exec] : [];
            if (step.builtin) {
                execs.push(this.renderBuiltin(step.builtin));
            }
            for (const exec of execs) {
                let command = '';
                let hasError = false;
                const cmd = exec.split(' ')[0];
                if (os_1.platform() == 'win32' && ['mkdir', 'mv', 'rm'].includes(cmd)) {
                    command = `shx ${exec}`;
                }
                else {
                    command = exec;
                }
                const cwd = step.cwd;
                try {
                    const result = this.shell({
                        command,
                        cwd,
                    });
                    hasError = result.status !== 0;
                }
                catch (e) {
                    // This is the error 'shx' will throw
                    if ((_d = e === null || e === void 0 ? void 0 : e.message) === null || _d === void 0 ? void 0 : _d.startsWith('non-zero exit code:')) {
                        hasError = true;
                    }
                    throw e;
                }
                if (hasError) {
                    throw new Error(`Task "${this.fullname}" failed when executing "${command}" (cwd: ${path_1.resolve(cwd !== null && cwd !== void 0 ? cwd : this.workdir)})`);
                }
            }
        }
    }
    /**
     * Determines if a task should be executed based on "condition".
     *
     * @returns true if the task should be executed or false if the condition
     * evaluates to false (exits with non-zero), indicating that the task should
     * be skipped.
     */
    evalCondition(task) {
        // no condition, carry on
        if (!task.condition) {
            return true;
        }
        this.log(chalk.gray(`${chalk.underline('condition')}: ${task.condition}`));
        const result = this.shell({
            command: task.condition,
            logprefix: 'condition: ',
            quiet: true,
        });
        if (result.status === 0) {
            return true;
        }
        else {
            return false;
        }
    }
    /**
     * Renders the runtime environment for a task. Namely, it supports this syntax
     * `$(xx)` for allowing environment to be evaluated by executing a shell
     * command and obtaining its result.
     */
    resolveEnvironment(parents) {
        var _b, _c, _d, _e, _f, _g;
        let env = (_b = this.runtime.manifest.env) !== null && _b !== void 0 ? _b : {};
        // add env from all parent tasks one by one
        for (const parent of parents) {
            env = {
                ...env,
                ...(_d = (_c = this.runtime.tryFindTask(parent)) === null || _c === void 0 ? void 0 : _c.env) !== null && _d !== void 0 ? _d : {},
            };
        }
        // apply the task's environment last
        env = {
            ...env,
            ...(_e = this.task.env) !== null && _e !== void 0 ? _e : {},
        };
        const output = {};
        for (const [key, value] of Object.entries(env !== null && env !== void 0 ? env : {})) {
            if (value.startsWith('$(') && value.endsWith(')')) {
                const query = value.substring(2, value.length - 1);
                const result = this.shellEval({ command: query });
                if (result.status !== 0) {
                    const error = result.error
                        ? result.error.stack
                        : (_g = (_f = result.stderr) === null || _f === void 0 ? void 0 : _f.toString()) !== null && _g !== void 0 ? _g : 'unknown error';
                    throw new Error(`unable to evaluate environment variable ${key}=${value}: ${error}`);
                }
                output[key] = result.stdout.toString('utf-8').trim();
            }
            else {
                output[key] = value;
            }
        }
        return output;
    }
    /**
     * Returns the "full name" of the task which includes all it's parent task names concatenated by chevrons.
     */
    get fullname() {
        return [...this.parents, this.task.name].join(' » ');
    }
    log(...args) {
        logging.verbose(this.fmtLog(...args));
    }
    fmtLog(...args) {
        return util_1.format(`${chalk.underline(this.fullname)} |`, ...args);
    }
    shell(options) {
        var _b, _c;
        const quiet = (_b = options.quiet) !== null && _b !== void 0 ? _b : false;
        if (!quiet) {
            const log = new Array();
            if (options.logprefix) {
                log.push(options.logprefix);
            }
            log.push(options.command);
            if (options.cwd) {
                log.push(`(cwd: ${options.cwd})`);
            }
            this.log(log.join(' '));
        }
        const cwd = (_c = options.cwd) !== null && _c !== void 0 ? _c : this.workdir;
        if (!fs_1.existsSync(cwd) || !fs_1.statSync(cwd).isDirectory()) {
            throw new Error(`invalid workdir (cwd): ${cwd} must be an existing directory`);
        }
        return child_process_1.spawnSync(options.command, {
            ...options,
            cwd,
            shell: true,
            stdio: 'inherit',
            env: {
                ...process.env,
                ...this.env,
            },
            ...options.spawnOptions,
        });
    }
    shellEval(options) {
        return this.shell({
            quiet: true,
            ...options,
            spawnOptions: {
                stdio: ['inherit', 'pipe', 'inherit'],
            },
        });
    }
    renderBuiltin(builtin) {
        const moduleRoot = path_1.dirname(require.resolve('../../package.json'));
        const program = require.resolve(path_1.join(moduleRoot, 'lib', `${builtin}.task.js`));
        return `${process.execPath} ${program}`;
    }
}
//# sourceMappingURL=data:application/json;base64,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