# Grimoire SSG

Grimoire is a minimalistic Static Site Generator (SSG) designed to simplify the process of 
creating static websites. With Grimoire, most use cases can be addressed without the need 
for programming knowledge — simply modify YAML files to generate your site.

## Features

- **YAML Configuration**: Define your content and structure using simple YAML files.
- **Template Rendering**: Utilize Jinja2 templates for dynamic content generation.
- **Markdown Support**: Write content in Markdown, which is automatically converted to HTML.
- **Tagging System**: Organize your content with tags for easy referencing in templates.
- **File Inclusion**: Include other YAML files to create a modular content structure.
- **Plugin System**: Extend the functionality with modules that can be added at runtime. 

## Getting Started

### Installation & Usage

To get started with Grimoire, you can directly install it using pip:

```bash
pip install grimoire-ssg
```

To generate your static site, run the Grimoire command with your input YAML files. 
You can specify an output directory using the `-o` or `--output` flag.

```bash
python -m grimoiressg -o output_directory one_or_more_input_files.yml
```

### Alternative Installation

Alternatively, you can clone the repository and install the 
required dependencies with [Poetry](https://python-poetry.org/):

```bash
git clone https://github.com/sigmasternchen/grimoire-ssg.git
cd grimoire-ssg
poetry install
```

You can then run the program directly using Poetry:

```bash
poetry run python -m grimoiressg -o output_directory one_or_more_input_files.yml
```

### Example YAML File

Here is an example of a YAML file that defines a content structure:

```yaml
# (optional) Included files will also be considered for generation.
# If this attribute is missing or empty, no other files will be included.
include:
  - pages/*.yml
  - blog/*.yml

# (optional) List of tags for this file.
# These can be used in templates later to reference this content.
# If this attribute is missing or empty, this file will not be accessible 
# via any tags.
tags:
  - page

# (optional) The file that should be generated from this .yml file.
# If this attribute is missing, no output file will be generated.
output: index.html

# (optional) Path to the template for this .yml file.
# If this attribute is missing, no output will be generated.
# It's also possible to just use `template` without `output`. In that case
# the rendered template can still be accessed by other templates.
template: ../templates/homepage.html

# (optional) The markdown content for this output file.
# If this attribute is missing, the markdown content can not be 
# referenced by the template.
markdown: |
  # Hello, World!

# All other defined attributes are not interpreted by the program, but 
# can still be referenced by a template.
# The following are some examples:
Date: 2025-01-06
Author: Sigma
```

### Template Example

Grimoire uses Jinja2 templates for rendering. Below is an example of a template that 
extends a layout and includes dynamic content:

```jinja
{% extends template_dir + "/layout.templ.html" %}
{% block title %}Homepage{% endblock %}
{% block content %}
    {{ current.markdown_compiled | safe }}

    <h2>My latest blog articles:</h2>
    <ul>
    {% for entry in tags["blog"] %}
        <li>
            <a href="{{ entry.output }}">
                {{ entry.title }}
            </a> ({{ entry.date }})
        </li>
    {% endfor %}
    </ul>
{% endblock %}
```

### Template Parameters

The following parameters are available in your templates:

- `current`: The current content file being rendered.
- `all`: A list of all content files.
- `tags`: A dictionary of tags with corresponding content files.
- `template_dir`: The absolute path to the parent directory of the current template.

The content file objects in the template contain all fields from the corresponding YAML file. 
Additionally, the following fields are defined:
- `filename` is the absolute filename of the yml file.
- `relative_filename` is the filename of the yml file relative to the working directory.
- `markdown_compiled` is the compiled markdown content in HTML form. In combination with the `safe` filter in Jinja2 the markdown content can be output.
- `rendered` is the rendered template of that file. This can be useful for including other pages in a template.


### Output Structure

The output files will be generated in the specified output directory, with paths defined in the `output` 
attribute of your YAML files.

## Advanced Features

### Custom Plugins

The program supports the addition of custom plugins at runtime. To utilize this, create a Python module 
that modifies the list of available modules:

```Python
from grimoiressg.modules import available_modules
from grimoiressg.utils import logger


def test(data, context):
    logger.info("This is test module.")


available_modules["test"] = test

```

You then need a config file that loads, and enables this module. Please note that you need to specify 
all `enabled_modules` to be used - not just the additional one.

```yaml
load_modules:
  - external_module_test

enabled_modules:
  - tags       # built-in module for tagging
  - markdown   # built-in module for markdown support
  - templating # built-in module for templating
  - test       # our custom module; the name is the 
               # key in the `available_modules` dict above
```

## Contributing

Contributions are welcome! If you have suggestions or improvements, feel free to open an issue or submit a pull request.

## License

This project is licensed under the BSD-2-Clause License. See the [LICENSE](LICENSE) file for details.

## Acknowledgments

- [Jinja2](https://jinja.palletsprojects.com/) for the templating engine.
- [Markdown](https://python-markdown.github.io/) for content formatting.
- [PyYAML](https://pyyaml.org/) for YAML parsing.

---

For more information, please refer to the documentation or the source code.