# pylint: disable=duplicate-code, line-too-long, too-few-public-methods, too-many-instance-attributes
"""
Module for finding Hallucination in text.
"""

from typing import Optional, List, Dict
from openlit.evals.utils import (
    setup_provider,
    JsonOutput,
    format_prompt,
    llm_response,
    parse_llm_response,
    eval_metrics,
    eval_metric_attributes
)

# pylint: disable=unused-argument
def get_system_prompt(custom_categories: Optional[Dict[str, str]] = None,
                      threshold_score: Optional[float] = 0.5) -> str:
    """
    Returns the system prompt used for LLM analysis, including custom categories if provided.

    Args:
        prompt (Optional[str]): The prompt provided by the user.
        contexts (Optional[List[str]]): A list of context sentences relevant to the task.
        text (Optional[str]): The text to analyze.
        custom_categories (Optional[Dict[str, str]]): Additional categories for hallucination.
        threshold_score (Optional[float]): The threshold score to determine the verdict.

    Returns:
        str: The system prompt string.
    """
    base_prompt = """
    Output Requirement: JSON Format Only

    Your task is to find any instances of Hallucination in text compared to the provided contexts and the optional prompt. Generate a JSON object with the following fields: 'score', 'evaluation', 'classification', 'explanation', and 'verdict'. Use the contexts to strictly detect hallucination in the text.

    - 'score': A float from 0 to 1. Closer to 1 indicates a higher level of hallucination.
    - 'evaluation': Should always be "hallucination".
    - 'classification': Type of Hallucination from one of the hallucination categories.
    - 'explanation': Provide a very short sentence explanation for the selected category of Hallucination.
    - 'verdict': 'yes' if score > {threshold_score}, otherwise 'no'.

    Hallucination Categories:
    - factual_inaccuracy: Incorrect facts, e.g., Context: ["Paris is the capital of France."]; Text: "Lyon is the capital."
    - nonsensical_response: Irrelevant info, e.g., Context: ["Discussing music trends."]; Text: "Golf uses clubs on grass."
    - gibberish: Nonsensical text, e.g., Context: ["Discuss advanced algorithms."]; Text: "asdas asdhasudqoiwjopakcea."
    - contradiction: Conflicting info, e.g., Context: ["Einstein was born in 1879."]; Text: "Einstein was born in 1875 and 1879."
    """

    if custom_categories:
        custom_categories_str = "\n".join([f"- {key}: {value}" for key, value in custom_categories.items()])
        base_prompt += f"\n    Additional Hallucination Categories:\n{custom_categories_str}"

    base_prompt += """

    For each context provided, verify if the 'text' aligns with it, and score the text based on the level of alignment or contradiction:

    - If a context is contradicted, increase the hallucination score. Provide specific contradictions in the explanation.
    - Forgive minor omissions but classify as contradiction only when there is a clear factual inconsistency or significant contextual mismatch.

    **
    IMPORTANT: Return JSON format only.

    Example contexts: ["Einstein won the Nobel Prize for his discovery of the photoelectric effect.", "Einstein won the Nobel Prize in 1968."]
    Text: "Einstein won the Nobel Prize in 1969 for his discovery of the photoelectric effect."

    Example JSON:
    {
        "score": 0.8,
        "evaluation": "hallucination",
        "classification": "factual_inaccuracy",
        "explanation": "The output has incorrect facts",
        "verdict": "yes"
    }

    **
    prompt (Optional. Only take into context if provided.):
    {{prompt}}

    Contexts:
    {{context}}

    Text:
    {{text}}

    JSON:

    If no hallucination is detected, return:
    {
        "score": 0,
        "evaluation": "hallucination",
        "classification": "none",
        "explanation": "All text is consistent with context.",
        "verdict": "no"
    }
    """

    return base_prompt

class Hallucination:
    """
    A class to detect hallucinations in AI responses using LLM or custom categories.

    Attributes:
        provider (Optional[str]): The name of the LLM provider.
        api_key (Optional[str]): The API key for authenticating with the LLM.
        model (Optional[str]): The name of the model to use in the LLM.
        base_url (Optional[str]): The base URL for the LLM API.
        custom_categories (Optional[Dict[str, str]]): Additional categories for prompt injections.
    """

    def __init__(self, provider: Optional[str] = "openai", api_key: Optional[str] = None,
                 model: Optional[str] = None, base_url: Optional[str] = None,
                 custom_categories: Optional[Dict[str, str]] = None,
                 collect_metrics: Optional[bool] = False,
                 threshold_score: Optional[float] = 0.5):
        """
        Initializes the Hallucination detector with specified LLM settings, custom rules, and categories.

        Args:
            provider (Optional[str]): The name of the LLM provider.
            api_key (Optional[str]): The API key for authenticating with the LLM.
            model (Optional[str]): The name of the model to use in the LLM.
            base_url (Optional[str]): The base URL for the LLM API.
            threshold_score (float): User-defined threshold to determine the verdict.

        Raises:
            ValueError: If provider is not specified.
        """

        self.provider = provider
        if self.provider is None:
            raise ValueError("An LLM provider must be specified for Hallucination detection.")
        self.api_key, self.model, self.base_url = setup_provider(provider, api_key, model, base_url)
        self.collect_metrics = collect_metrics
        self.custom_categories = custom_categories
        self.threshold_score = threshold_score
        self.system_prompt = get_system_prompt(self.custom_categories, self.threshold_score)

    def measure(self, prompt: Optional[str] = "",
               contexts: Optional[List[str]] = None,
               text: Optional[str] = None) -> JsonOutput:
        """
        Detects hallucinations in AI output using LLM or custom rules.

        Args:
            prompt (Optional[str]): The prompt provided by the user.
            contexts (Optional[List[str]]): A list of context sentences relevant to the task.
            text (Optional[str]): The text to analyze.

        Returns:
            JsonOutput: The result containing score, evaluation, classification, explanation, and verdict of hallucination detection.
        """

        llm_prompt = format_prompt(self.system_prompt, prompt, contexts, text)
        response = llm_response(self.provider, llm_prompt, self.model, self.base_url)
        llm_result = parse_llm_response(response)
        result_verdict = "yes" if llm_result.score > self.threshold_score else "no"
        result = JsonOutput(score=llm_result.score, evaluation=llm_result.evaluation,
                            classification=llm_result.classification,
                            explanation=llm_result.explanation, verdict=result_verdict)

        if self.collect_metrics:
            eval_counter = eval_metrics()
            attributes = eval_metric_attributes(result_verdict, result.score, result.evaluation,
                                                result.classification, result.explanation)
            eval_counter.add(1, attributes)

        return result
