"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateAvailabilityZone = exports.validateRegion = exports.SUPPORTED_AVAILABILITY_ZONES = exports.SUPPORTED_REGIONS = void 0;
/**
 * Regions where Managed Blockchain is supported, for details see
 * https://aws.amazon.com/managed-blockchain/pricing/hyperledger/
*/
exports.SUPPORTED_REGIONS = [
    'ap-northeast-1',
    'ap-northeast-2',
    'ap-southeast-1',
    'eu-west-1',
    'eu-west-2',
    'us-east-1',
];
/**
 * Map of supported regions to their availability zones
 */
exports.SUPPORTED_AVAILABILITY_ZONES = {
    'ap-northeast-1': [
        'ap-northeast-1a',
        'ap-northeast-1b',
        'ap-northeast-1c',
    ],
    'ap-northeast-2': [
        'ap-northeast-2a',
        'ap-northeast-2b',
        'ap-northeast-2c',
        'ap-northeast-2d',
    ],
    'ap-southeast-1': [
        'ap-southeast-1a',
        'ap-southeast-1b',
        'ap-southeast-1c',
    ],
    'eu-west-1': [
        'eu-west-1a',
        'eu-west-1b',
        'eu-west-1c',
    ],
    'eu-west-2': [
        'eu-west-2a',
        'eu-west-2b',
        'eu-west-2c',
    ],
    'us-east-1': [
        'us-east-1a',
        'us-east-1b',
        'us-east-1c',
        'us-east-1d',
        'us-east-1e',
        'us-east-1f',
    ],
};
/**
 * Throw an error if provided region is not in the supported list
 */
function validateRegion(region) {
    if (!exports.SUPPORTED_REGIONS.includes(region)) {
        const regionList = exports.SUPPORTED_REGIONS.join(', ');
        throw new Error(`Managed Blockchain is only available in the following regions: ${regionList}.`);
    }
}
exports.validateRegion = validateRegion;
/**
 * Throw an error if provided availability zone is not in the supported list
 */
function validateAvailabilityZone(region, availabilityZone) {
    const availabililtyZonesForRegion = exports.SUPPORTED_AVAILABILITY_ZONES[region];
    if (typeof availabilityZone === 'undefined' || !availabililtyZonesForRegion.includes(availabilityZone)) {
        const availabilityZoneList = availabililtyZonesForRegion.join(', ');
        throw new Error(`Managed Blockchain in ${region} is only available in the following availability zones: ${availabilityZoneList}.`);
    }
}
exports.validateAvailabilityZone = validateAvailabilityZone;
//# sourceMappingURL=data:application/json;base64,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