#! /usr/bin/env python
# wdecoster
'''
The main purpose of this script is to create plots for nanopore data.
Input data can be given as
-compressed, standard or streamed fastq file
-compressed, standard or streamed fastq file with additional information added by albacore or MinKNOW
-a bam file
-a summary file generated by albacore
'''


from __future__ import division, print_function
import argparse
import sys
import os
from time import time
import logging
from datetime import datetime as dt
import matplotlib
matplotlib.use('Agg')
import matplotlib.pyplot as plt
import seaborn as sns
import pandas as pd
import numpy as np
from multiprocessing import cpu_count
from scipy import stats
import pysam
import nanoget
import nanoplotter
import nanomath
from .version import __version__


def main():
	'''
	Organization function
	-setups logging
	-gets inputdata
	-calls plotting function
	'''
	try:
		args = getArgs()
		initlogs(args)
		args.format = nanoplotter.checkvalidFormat(args.format)
		datadf, lengthprefix, logBool, readlengthsPointer = getInput(args)
		makePlots(datadf, lengthprefix, logBool, readlengthsPointer, args)
		logging.info("Succesfully processed all input.")
	except Exception as e:
		logging.error(e, exc_info=True)
		raise


def getArgs():
	parser = argparse.ArgumentParser(description="Perform diagnostic plotting, QC analysis Nanopore sequencing data and alignments.")
	parser.add_argument("-v", "--version",
						help="Print version and exit.",
						action="version",
						version='NanoPlot {}'.format(__version__))
	parser.add_argument("-t", "--threads",
						help="Set the allowed number of threads to be used by the script",
						default=4,
						type=int)
	parser.add_argument("--maxlength",
						help="Drop reads longer than length specified.",
						type=int)
	parser.add_argument("--drop_outliers",
						help="Drop outlier reads with extreme long length.",
						action="store_true")
	parser.add_argument("--downsample",
						help="Reduce dataset to N reads by random sampling.",
						type=int)
	parser.add_argument("--loglength",
						help="Logarithmic scaling of lengths in plots.",
						action="store_true")
	parser.add_argument("--alength",
						help="Use aligned read lengths rather than sequenced length (bam mode)",
						action="store_true")
	parser.add_argument("-c", "--color",
						help="Specify a color for the plots, must be a valid matplotlib color",
						default="#4CB391")
	parser.add_argument("-o", "--outdir",
						help="Specify directory in which output has to be created.",
						default=".")
	parser.add_argument("-p", "--prefix",
						help="Specify an optional prefix to be used for the output files.",
						default="",
						type=str)
	parser.add_argument("-f", "--format",
						help="Specify the output format of the plots.",
						default="png",
						type=str,
						choices=['eps', 'jpeg', 'jpg', 'pdf', 'pgf', 'png', 'ps', 'raw', 'rgba', 'svg', 'svgz', 'tif', 'tiff'])
	parser.add_argument("--plots",
						help="Specify which bivariate plots have to be made.",
						default=['kde', 'hex', 'dot'],
						type=str,
						nargs='*',
						choices=['kde', 'hex', 'dot'])
	target = parser.add_mutually_exclusive_group(required=True)
	target.add_argument("--fastq",
						help="Data is in default fastq format.")
	target.add_argument("--fastq_rich",
						help="Data is in fastq format generated by albacore or MinKNOW with additional information concerning channel and time.")
	target.add_argument("--summary",
						help="Data is a summary file generated by albacore.")
	target.add_argument("--bam",
						help="Data as a sorted bam file.")
	return parser.parse_args()


def initlogs(args):
	try:
		if not os.path.exists(args.outdir):
			os.makedirs(args.outdir)
		start_time = dt.fromtimestamp(time()).strftime('%Y%m%d_%H%M')
		logging.basicConfig(
			format='%(asctime)s %(message)s',
			filename=os.path.join(args.outdir, args.prefix + "Nanoplot_" + start_time + ".log"),
			level=logging.INFO)
	except IOError:
		sys.exit("ERROR: No writing permission to the output directory.")
	logging.info('Nanoplot {} started with arguments {}'.format(__version__, args))
	logging.info('Python version is: {}'.format(sys.version.replace('\n', ' ')))
	logging.info('Versions of key modules are:')
	for module in [np, sns, pd, pysam, nanoget, matplotlib, nanoplotter, nanomath]:
		logging.info('{}: {}'.format(module, module.__version__))
	logging.info("{} cpu's are available".format(cpu_count()))


def getInput(args):
	'''
	Get input and process accordingly. 	Data can be:
	- a uncompressed, bgzip, bzip2 or gzip compressed fastq file
	- a sorted bam file
	- a sequencing_summary.txt file generated by albacore
	Handle is passed to the proper functions to get DataFrame with metrics
	'''
	if args.fastq:
		datadf = nanoget.processFastqPlain(args.fastq)
	elif args.bam:
		datadf = nanoget.processBam(args.bam, min(cpu_count() - 1, args.threads))
	elif args.fastq_rich:
		datadf = nanoget.processFastq_rich(args.fastq_rich)
	elif args.summary:
		datadf = nanoget.processSummary(args.summary)
	logging.info("Gathered metrics for plotting")
	nanomath.writeStats(datadf, os.path.join(args.outdir, "NanoStats.txt"))
	logging.info("Calculated statistics")
	datadf, lengthprefix, logBool, readlengthsPointer = filterData(datadf, args)
	logging.info("Processed the metrics, optionally filtering.")
	return (datadf, lengthprefix, logBool, readlengthsPointer)


def filterData(datadf, args):
	'''
	Perform filtering on the data based on arguments set on commandline
	- use alighned length or sequenced length (bam mode only)
	- drop outliers
	- drop reads longer than args.maxlength
	- use log10 scaled reads
	- downsample reads to args.downsample
	Return an accurate prefix which is added to plotnames using this filtered data
	'''
	lengthprefix = []
	if args.alength and args.bam:
		readlengthsPointer = "aligned_lengths"
		lengthprefix.append("Aligned_")
		logging.info("Using aligned read lengths for plotting.")
	else:
		readlengthsPointer = "lengths"
		logging.info("Using sequenced read lengths for plotting.")
	if args.drop_outliers:
		datadf=nanomath.removeLengthOutliers(datadf, readlengthsPointer)
		lengthprefix.append("OutliersRemoved_")
		logging.info("Removing length outliers for plotting.")
	if args.maxlength:
		datadf=datadf[datadf[readlengthsPointer] < args.maxlength]
		lengthprefix.append("MaxLength-" + str(args.maxlength) + '_')
		logging.info("Removing reads longer than {}.".format(str(args.maxlength)))
	if args.loglength:
		datadf["log_" + readlengthsPointer] = np.log10(datadf[readlengthsPointer])
		readlengthsPointer = "log_" + readlengthsPointer
		lengthprefix.append("Log_")
		logging.info("Using Log10 scaled read lengths.")
		logBool = True
	else:
		logBool = False
	if args.downsample:
		newNum = min(args.downsample, len(datadf.index))
		lengthprefix.append("Downsampled_")
		logging.info("Downsampling the dataset from {} to {} reads".format(len(datadf.index), newNum))
		datadf = datadf.sample(newNum)
	return(datadf, ''.join(lengthprefix), logBool, readlengthsPointer)


def makePlots(datadf, lengthprefix, logBool, readlengthsPointer, args):
	'''Call plotting functions'''
	color = nanoplotter.checkvalidColor(args.color)
	plotdict = {type: args.plots.count(type) for type in ["kde", "hex", "dot"]}
	nanoplotter.lengthPlots(
		array=datadf[readlengthsPointer],
		name="Read length",
		path=os.path.join(args.outdir, args.prefix + lengthprefix),
		n50=nanomath.getN50(np.sort(datadf["lengths"])),
		color=color,
		format=args.format,
		log=logBool)
	logging.info("Created length plots")
	nanoplotter.scatter(
		x=datadf[readlengthsPointer],
		y=datadf["quals"],
		names=['Read lengths', 'Average read quality'],
		path=os.path.join(args.outdir, args.prefix + lengthprefix + "LengthvsQualityScatterPlot"),
		color=color,
		format=args.format,
		plots=plotdict,
		log=logBool)
	logging.info("Created LengthvsQual plot")
	if args.fastq_rich or args.summary:
		nanoplotter.spatialHeatmap(
			array=datadf["channelIDs"],
			title="Number of reads generated per channel",
			path=os.path.join(args.outdir, args.prefix + "ActivityMap_ReadsPerChannel"),
			color="Greens",
			format=args.format)
		logging.info("Created spatialheatmap for succesfull basecalls.")
		nanoplotter.timePlots(
			df=datadf,
			path=os.path.join(args.outdir, args.prefix),
			color=color,
			format=args.format)
		logging.info("Created timeplots.")
	if args.bam:
		nanoplotter.scatter(
			x=datadf["aligned_lengths"],
			y=datadf["lengths"],
			names=["Aligned read lengths", "Sequenced read length"],
			path=os.path.join(args.outdir, args.prefix + "AlignedReadlengthvsSequencedReadLength"),
			format=args.format,
			plots=plotdict,
			color=color)
		logging.info("Created AlignedLength vs Length plot.")
		nanoplotter.scatter(
			x=datadf["mapQ"],
			y=datadf["quals"],
			names=["Read mapping quality", "Average basecall quality"],
			path=os.path.join(args.outdir, args.prefix + "MappingQualityvsAverageBaseQuality"),
			color=color,
			format=args.format,
			plots=plotdict)
		logging.info("Created MapQvsBaseQ plot.")
		nanoplotter.scatter(
			x=datadf[readlengthsPointer],
			y=datadf["mapQ"],
			names=["Read length", "Read mapping quality"],
			path=os.path.join(args.outdir, args.prefix + lengthprefix + "MappingQualityvsReadLength"),
			color=color,
			format=args.format,
			plots=plotdict,
			log=logBool)
		logging.info("Created MapQvsBaseQ plot.")
		minPID = np.amin(datadf["percentIdentity"])
		nanoplotter.scatter(
			x=datadf["percentIdentity"],
			y=datadf["aligned_quals"],
			names=["Percent identity", "Read quality"],
			path=os.path.join(args.outdir, args.prefix + "PercentIdentityvsAverageBaseQuality"),
			color=color,
			format=args.format,
			plots=plotdict,
			stat=stats.pearsonr,
			minvalx=minPID)
		logging.info("Created Percent ID Base quality plot.")
		nanoplotter.scatter(
			x=datadf[readlengthsPointer],
			y=datadf["percentIdentity"],
			names=["Aligned read length", "Percent identity"],
			path=os.path.join(args.outdir, args.prefix + "PercentIdentityvsAlignedReadLength"),
			color=color,
			format=args.format,
			plots=plotdict,
			stat=stats.pearsonr,
			log=logBool,
			minvaly=minPID)
		logging.info("Created Percent ID vs Length plot")


if __name__ == "__main__":
	main()
