import os
import logging
from typing import Any, Dict

try:  # pragma: no cover - dependencia opcional
    import yaml
except ModuleNotFoundError:  # pragma: no cover - entornos sin PyYAML
    yaml = None  # type: ignore[assignment]

try:
    import tomllib  # Python ≥3.11
except ModuleNotFoundError:  # pragma: no cover
    import tomli as tomllib

logger = logging.getLogger(__name__)

MODULE_MAP_PATH = os.environ.get(
    'COBRA_MODULE_MAP',
    os.path.abspath(
        os.path.join(os.path.dirname(__file__), '..', '..', '..', '..', 'cobra.mod')
    ),
)

COBRA_TOML_PATH = os.environ.get(
    'COBRA_TOML',
    os.path.abspath(
        os.path.join(os.path.dirname(__file__), '..', '..', '..', '..', 'cobra.toml')
    ),
)

_cache = None
_toml_cache = None

def get_map() -> Dict[str, Any]:
    """Carga el mapa YAML de módulos soportados."""
    global _cache
    if _cache is None:
        if yaml is None:
            logger.debug(
                "PyYAML no está instalado; se devuelve un mapa de módulos vacío.",
            )
            _cache = {}
            return _cache
        try:
            if os.path.exists(MODULE_MAP_PATH):
                with open(MODULE_MAP_PATH, 'r', encoding='utf-8') as f:
                    data = yaml.safe_load(f) or {}
            else:
                data = {}
            _cache = data
        except (yaml.YAMLError, OSError) as e:  # type: ignore[attr-defined]
            logger.error(f"Error al cargar el archivo de mapeo: {e}")
            return {}
    return _cache


def get_toml_map():
    """Devuelve la configuración del archivo ``cobra.toml``."""
    global _toml_cache
    if _toml_cache is None:
        try:
            if os.path.exists(COBRA_TOML_PATH):
                with open(COBRA_TOML_PATH, 'rb') as f:
                    data = tomllib.load(f) or {}
            else:
                data = {}
            _toml_cache = data
        except (tomllib.TOMLDecodeError, OSError) as e:
            logger.error(f"Error al cargar cobra.toml: {e}")
            _toml_cache = {}
    return _toml_cache


def get_mapped_path(module: str, backend: str) -> str:
    """Return the path for *module* mapped for the given *backend*.

    If no mapping exists, the original module path is returned.
    """
    mapa = get_toml_map()
    return mapa.get(module, {}).get(backend, module)
