"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const AWS = require("aws-sdk");
const calculate_ses_smtp_password_1 = require("./calculate-ses-smtp-password");
const credentials_1 = require("./credentials");
/**
 * The onEvent handler is invoked whenever a resource lifecycle event for a credentials custom-resource occurs
 *
 * @see https://docs.aws.amazon.com/cdk/api/v1/docs/custom-resources-readme.html#handling-lifecycle-events-onevent
 */
async function handler(event) {
    const requestType = event.RequestType;
    console.log(`Request of type ${requestType} received`);
    if (requestType == "Create") {
        // Create access key
        const username = event.ResourceProperties.UserName;
        const secretId = event.ResourceProperties.SecretId;
        const region = process.env.AWS_DEFAULT_REGION;
        const iam = new AWS.IAM();
        const secretsManager = new AWS.SecretsManager();
        const accessKey = await iam
            .createAccessKey({
            UserName: username,
        })
            .promise();
        const accessKeyId = accessKey.AccessKey.AccessKeyId;
        const secretAccessKey = accessKey.AccessKey.SecretAccessKey;
        // Create ses smtp credentials
        const smtpPassword = calculate_ses_smtp_password_1.calculateSesSmtpPassword(secretAccessKey, region);
        // TODO: Update in SecretsManager
        await secretsManager
            .putSecretValue({
            SecretId: secretId,
            SecretString: JSON.stringify({
                [credentials_1.Credentials.USERNAME]: accessKeyId,
                [credentials_1.Credentials.PASSWORD]: smtpPassword,
            }),
        })
            .promise();
        return {
            ...event,
            PhysicalResourceId: accessKeyId,
            Data: {
                ...event.ResourceProperties,
                AccessKeyId: accessKeyId,
            },
        };
    }
    if (requestType == "Update") {
        console.log("Access key has already been provisioned, no operation to be performed");
    }
    if (requestType == "Delete") {
        console.log("No operation required, deletion of this resource is assumed to occur in conjunction with deletion of an IAM User and a SecretsManager Secret");
    }
    return {
        ...event,
        Data: {
            ...event.ResourceProperties,
        },
    };
}
exports.handler = handler;
//# sourceMappingURL=data:application/json;base64,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