"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SesSmtpCredentials = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const secretsmanager = require("aws-cdk-lib/aws-secretsmanager");
const constructs_1 = require("constructs");
const credentials_provider_1 = require("./provider/credentials-provider");
const ses_smtp_regions_1 = require("./provider/ses-smtp-regions");
/**
 * This construct creates an access key for the given user and stores the generated SMTP credentials inside a secret.
 *
 * Attaches an inline policy to the user allowing to send emails
 *
 * ```typescript
 * const user = User.fromUserName("ses-user-example");
 * const credentials = new SesSmtpCredentials(this, 'SmtpCredentials', {
 *     user: user,
 * });
 * // smtpCredentials.secret contains json value {username: "<the generated access key id>", password: "<the calculated ses smtp password>"}
 * ```
 */
class SesSmtpCredentials extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const region = aws_cdk_lib_1.Stack.of(this).region;
        if (!ses_smtp_regions_1.SMTP_REGIONS.includes(region)) {
            aws_cdk_lib_1.Annotations.of(this).addWarning(`AWS SES Smtp Endpoint is not available in region ${region}\n see https://docs.aws.amazon.com/general/latest/gr/ses.html`);
        }
        const user = props.user ??
            new iam.User(this, "User", {
                userName: props.userName,
            });
        const policy = new iam.Policy(this, "Policy", {
            statements: [
                new iam.PolicyStatement({
                    // https://github.com/awsdocs/amazon-ses-developer-guide/blob/master/doc-source/sending-authorization-policies.md
                    effect: iam.Effect.ALLOW,
                    actions: ["ses:SendRawEmail"],
                    resources: ["*"],
                }),
            ],
        });
        user.attachInlinePolicy(policy);
        this.secret =
            props.secret ||
                new secretsmanager.Secret(this, "Secret", {
                    description: `SES Smtp Credentials (username, password) for ${user.userName}`,
                });
        const provider = credentials_provider_1.CredentialsProvider.getOrCreate(this);
        provider.grant(user, this.secret);
        const customResource = new aws_cdk_lib_1.CustomResource(this, "Lambda", {
            serviceToken: provider.serviceToken,
            properties: {
                UserName: user.userName,
                SecretId: this.secret.secretArn,
            },
        });
        customResource.node.addDependency(user);
        customResource.node.addDependency(this.secret);
    }
}
exports.SesSmtpCredentials = SesSmtpCredentials;
_a = JSII_RTTI_SYMBOL_1;
SesSmtpCredentials[_a] = { fqn: "@pepperize/cdk-ses-smtp-credentials.SesSmtpCredentials", version: "0.3.185" };
//# sourceMappingURL=data:application/json;base64,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