// Copyright 2025- European Centre for Medium-Range Weather Forecasts (ECMWF)

// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at

//     http://www.apache.org/licenses/LICENSE-2.0

// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// In applying this licence, ECMWF does not waive the privileges and immunities
// granted to it by virtue of its status as an intergovernmental organisation nor
// does it submit to any jurisdiction.

// Copyright 2025- Niall Oswald and Kenneth Martin and Jo Wayne Tan

use geo::algorithm::Area;
use geo::geometry::{LineString, Triangle};
use geo::{Coord, CoordFloat, CoordNum, Point};

// geo::Triangle sorts the vertices so they are always oriented ccw
// this is not acceptable for our use case
#[derive(Copy, Clone, Hash, Eq, PartialEq)]
pub struct OrdTriangle<T: CoordNum>(Coord<T>, Coord<T>, Coord<T>);

impl<T: CoordNum> OrdTriangle<T> {
    pub fn new(v1: Coord<T>, v2: Coord<T>, v3: Coord<T>) -> Self {
        Self(v1, v2, v3)
    }
}

impl<T: CoordNum> From<OrdTriangle<T>> for Triangle<T> {
    fn from(from: OrdTriangle<T>) -> Self {
        Triangle::new(from.0, from.1, from.2)
    }
}

impl<T: CoordFloat> Area<T> for OrdTriangle<T> {
    fn signed_area(&self) -> T {
        Point::from(self.0).cross_prod(self.1.into(), self.2.into()) / (T::one() + T::one())
    }

    fn unsigned_area(&self) -> T {
        self.signed_area().abs()
    }
}

pub trait OrdTriangles<T: CoordNum> {
    fn ord_triangles(&'_ self) -> impl ExactSizeIterator<Item = OrdTriangle<T>> + '_;
}

impl<T: CoordNum> OrdTriangles<T> for LineString<T> {
    fn ord_triangles(&'_ self) -> impl ExactSizeIterator<Item = OrdTriangle<T>> + '_ {
        self.0.windows(3).map(|w| {
            let &[x, y, z] = w else { unreachable!() };
            OrdTriangle::new(x, y, z)
        })
    }
}
