import unittest
from math import pi

import numpy as np

from wisdem.ccblade.Polar import Polar, blend


class TestBlend(unittest.TestCase):
    def setUp(self):
        alpha = [
            -3.04,
            -2.03,
            -1.01,
            0.01,
            1.03,
            2.05,
            3.07,
            4.09,
            5.11,
            6.13,
            7.14,
            8.16,
            9.17,
            10.18,
            11.18,
            12.19,
            13.18,
            14.18,
            15.18,
            16.17,
            17.14,
            18.06,
            19.06,
            20.07,
            25,
        ]
        cl = [
            -0.071,
            0.044,
            0.144,
            0.241,
            0.338,
            0.435,
            0.535,
            0.632,
            0.728,
            0.813,
            0.883,
            0.946,
            1.001,
            1.054,
            1.056,
            1.095,
            1.138,
            1.114,
            1.073,
            1.008,
            0.95,
            0.902,
            0.795,
            0.797,
            0.8,
        ]
        cd = [
            0.0122,
            0.0106,
            0.0114,
            0.0134,
            0.0136,
            0.014,
            0.0147,
            0.0156,
            0.0162,
            0.0173,
            0.0191,
            0.0215,
            0.0248,
            0.0339,
            0.0544,
            0.0452,
            0.0445,
            0.067,
            0.0748,
            0.1028,
            0.1473,
            0.2819,
            0.2819,
            0.2819,
            0.3,
        ]
        cm = [
            -0.0044,
            -0.0051,
            0.0018,
            -0.0216,
            -0.0282,
            -0.0346,
            -0.0405,
            -0.0455,
            -0.0507,
            -0.0404,
            -0.0321,
            -0.0281,
            -0.0284,
            -0.0322,
            -0.0361,
            -0.0363,
            -0.0393,
            -0.0398,
            -0.0983,
            -0.1242,
            -0.1155,
            -0.1068,
            -0.0981,
            -0.0894,
            -0.0807,
        ]
        Re = 1

        self.polar1 = Polar(Re, alpha, cl, cd, cm)

        alpha = [
            -3.04,
            -2.03,
            -1.01,
            0.01,
            1.03,
            2.05,
            3.07,
            4.09,
            5.11,
            6.13,
            7.14,
            8.16,
            9.17,
            10.18,
            11.18,
            12.19,
            13.18,
            14.18,
            15.189,
            16.17,
            17.14,
            18.06,
            19.06,
            20.07,
            21.08,
            22.09,
            23.1,
            25,
        ]
        cl = [
            -0.0852,
            0.0528,
            0.1728,
            0.2892,
            0.4056,
            0.522,
            0.642,
            0.7584,
            0.8736,
            0.9756,
            1.0596,
            1.1352,
            1.2012,
            1.2648,
            1.2672,
            1.314,
            1.3656,
            1.3368,
            1.2876,
            1.2096,
            1.14,
            1.0824,
            0.954,
            0.9564,
            1,
            1.2,
            1.4,
            1.6,
        ]
        cd = [
            0.01464,
            0.01272,
            0.01368,
            0.01608,
            0.01632,
            0.0168,
            0.01764,
            0.01872,
            0.01944,
            0.02076,
            0.02292,
            0.0258,
            0.02976,
            0.04068,
            0.06528,
            0.05424,
            0.0534,
            0.0804,
            0.08976,
            0.12336,
            0.17676,
            0.33828,
            0.33828,
            0.33828,
            0.35,
            0.4,
            0.45,
            0.5,
        ]
        cm = [
            -0.0037,
            -0.0044,
            -0.0051,
            0.0018,
            -0.0216,
            -0.0282,
            -0.0346,
            -0.0405,
            -0.0455,
            -0.0507,
            -0.0404,
            -0.0321,
            -0.0281,
            -0.0284,
            -0.0322,
            -0.0361,
            -0.0363,
            -0.0393,
            -0.0398,
            -0.0983,
            -0.1242,
            -0.1155,
            -0.1068,
            -0.0981,
            -0.0894,
            -0.0807,
            -0.072,
            -0.0633,
        ]

        self.polar2 = Polar(Re, alpha, cl, cd, cm)

    def test_blend1(self):
        polar3 = blend(self.polar1, self.polar2, 0.5)

        alpha_blend = [
            -3.04,
            -2.03,
            -1.01,
            0.01,
            1.03,
            2.05,
            3.07,
            4.09,
            5.11,
            6.13,
            7.14,
            8.16,
            9.17,
            10.18,
            11.18,
            12.19,
            13.18,
            14.18,
            15.18,
            16.17,
            17.14,
            18.06,
            19.06,
            20.07,
            25,
        ]
        cl_blend = [
            -0.078,
            0.048,
            0.158,
            0.265,
            0.372,
            0.479,
            0.589,
            0.695,
            0.801,
            0.894,
            0.971,
            1.041,
            1.101,
            1.159,
            1.162,
            1.205,
            1.252,
            1.225,
            1.181,
            1.109,
            1.045,
            0.992,
            0.875,
            0.877,
            1.200,
        ]
        cd_blend = [
            0.0134,
            0.0117,
            0.0125,
            0.0147,
            0.0150,
            0.0154,
            0.0162,
            0.0172,
            0.0178,
            0.0190,
            0.0210,
            0.0237,
            0.0273,
            0.0373,
            0.0598,
            0.0497,
            0.0490,
            0.0737,
            0.0822,
            0.1131,
            0.1620,
            0.3101,
            0.3101,
            0.3101,
            0.4000,
        ]
        cm_blend = [
            -0.00405,
            -0.00475,
            -0.00165,
            -0.0099,
            -0.0249,
            -0.0314,
            -0.03755,
            -0.043,
            -0.0481,
            -0.04555,
            -0.03625,
            -0.0301,
            -0.02825,
            -0.0303,
            -0.03415,
            -0.0362,
            -0.0378,
            -0.03955,
            -0.06905,
            -0.11125,
            -0.11985,
            -0.11115,
            -0.10245,
            -0.09375,
            -0.072,
        ]

        # re-interpolate b/c angles of attack are different
        cl3 = np.interp(alpha_blend, polar3.alpha, polar3.cl)
        cd3 = np.interp(alpha_blend, polar3.alpha, polar3.cd)
        cm3 = np.interp(alpha_blend, polar3.alpha, polar3.cm)

        # should be within 1e-3
        np.testing.assert_allclose(cl3, cl_blend, atol=1e-3)
        np.testing.assert_allclose(cd3, cd_blend, atol=1e-3)
        np.testing.assert_allclose(cm3, cm_blend, atol=1e-3)

    def test_blend2(self):
        polar3 = blend(self.polar1, self.polar2, 0.7)

        alpha_blend = [
            -3.04,
            -2.03,
            -1.01,
            0.01,
            1.03,
            2.05,
            3.07,
            4.09,
            5.11,
            6.13,
            7.14,
            8.16,
            9.17,
            10.18,
            11.18,
            12.19,
            13.18,
            14.18,
            15.18,
            16.17,
            17.14,
            18.06,
            19.06,
            20.07,
            25,
        ]
        cl_blend = [
            -0.081,
            0.050,
            0.164,
            0.275,
            0.385,
            0.496,
            0.610,
            0.720,
            0.830,
            0.927,
            1.007,
            1.078,
            1.141,
            1.202,
            1.204,
            1.248,
            1.297,
            1.270,
            1.224,
            1.149,
            1.083,
            1.028,
            0.906,
            0.909,
            1.360,
        ]
        cd_blend = [
            0.0139,
            0.0121,
            0.0130,
            0.0153,
            0.0155,
            0.0160,
            0.0168,
            0.0178,
            0.0185,
            0.0197,
            0.0218,
            0.0245,
            0.0283,
            0.0386,
            0.0620,
            0.0515,
            0.0507,
            0.0764,
            0.0852,
            0.1172,
            0.1679,
            0.3214,
            0.3214,
            0.3214,
            0.4400,
        ]
        cm_blend = [
            -0.00391,
            -0.00461,
            -0.00303,
            -0.00522,
            -0.02358,
            -0.03012,
            -0.03637,
            -0.042,
            -0.04706,
            -0.04761,
            -0.03791,
            -0.0309,
            -0.02819,
            -0.02954,
            -0.03337,
            -0.03616,
            -0.0372,
            -0.03945,
            -0.057347,
            -0.10607,
            -0.12159,
            -0.11289,
            -0.10419,
            -0.09549,
            -0.06852,
        ]

        # re-interpolate b/c angles of attack are different
        cl3 = np.interp(alpha_blend, polar3.alpha, polar3.cl)
        cd3 = np.interp(alpha_blend, polar3.alpha, polar3.cd)
        cm3 = np.interp(alpha_blend, polar3.alpha, polar3.cm)

        # should be within 1e-3
        np.testing.assert_allclose(cl3, cl_blend, atol=1e-3)
        np.testing.assert_allclose(cd3, cd_blend, atol=1e-3)
        np.testing.assert_allclose(cm3, cm_blend, atol=1e-3)

    def test_blend3(self):
        polar3 = blend(self.polar1, self.polar2, 0.2)

        alpha_blend = [
            -3.04,
            -2.03,
            -1.01,
            0.01,
            1.03,
            2.05,
            3.07,
            4.09,
            5.11,
            6.13,
            7.14,
            8.16,
            9.17,
            10.18,
            11.18,
            12.19,
            13.18,
            14.18,
            15.18,
            16.17,
            17.14,
            18.06,
            19.06,
            20.07,
            25,
        ]
        cl_blend = [
            -0.074,
            0.046,
            0.150,
            0.251,
            0.352,
            0.452,
            0.556,
            0.657,
            0.757,
            0.846,
            0.918,
            0.984,
            1.041,
            1.096,
            1.098,
            1.139,
            1.184,
            1.159,
            1.116,
            1.048,
            0.988,
            0.938,
            0.827,
            0.829,
            0.960,
        ]
        cd_blend = [
            0.0127,
            0.0110,
            0.0119,
            0.0139,
            0.0141,
            0.0146,
            0.0153,
            0.0162,
            0.0168,
            0.0180,
            0.0199,
            0.0224,
            0.0258,
            0.0353,
            0.0566,
            0.0470,
            0.0463,
            0.0697,
            0.0778,
            0.1069,
            0.1532,
            0.2932,
            0.2932,
            0.2932,
            0.3400,
        ]
        cm_blend = [
            -0.00426,
            -0.00496,
            0.00042,
            -0.01692,
            -0.02688,
            -0.03332,
            -0.03932,
            -0.0445,
            -0.04966,
            -0.04246,
            -0.03376,
            -0.0289,
            -0.02834,
            -0.03144,
            -0.03532,
            -0.03626,
            -0.0387,
            -0.0397,
            -0.0866,
            -0.11902,
            -0.11724,
            -0.10854,
            -0.09984,
            -0.09114,
            -0.07722,
        ]

        # re-interpolate b/c angles of attack are different
        cl3 = np.interp(alpha_blend, polar3.alpha, polar3.cl)
        cd3 = np.interp(alpha_blend, polar3.alpha, polar3.cd)
        cm3 = np.interp(alpha_blend, polar3.alpha, polar3.cm)

        # should be within 1e-3
        np.testing.assert_allclose(cl3, cl_blend, atol=1e-3)
        np.testing.assert_allclose(cd3, cd_blend, atol=1e-3)
        np.testing.assert_allclose(cm3, cm_blend, atol=1e-3)


class Test3DStall(unittest.TestCase):
    def setUp(self):
        alpha = [
            -9.000,
            -8.000,
            -7.000,
            -6.000,
            -5.000,
            -4.000,
            -3.000,
            -2.000,
            -1.000,
            0.000,
            1.000,
            2.000,
            3.000,
            4.000,
            5.000,
            6.000,
            7.000,
            8.000,
            9.000,
            10.000,
            11.000,
            12.000,
            13.000,
            14.000,
            15.000,
            16.000,
            17.000,
            18.000,
            19.000,
            20.000,
            30.000,
            40.000,
            50.000,
        ]
        cl = [
            -0.802,
            -0.721,
            -0.611,
            -0.506,
            -0.408,
            -0.313,
            -0.220,
            -0.133,
            -0.060,
            0.036,
            0.227,
            0.342,
            0.436,
            0.556,
            0.692,
            0.715,
            0.761,
            0.830,
            0.893,
            0.954,
            1.013,
            1.042,
            1.061,
            1.083,
            1.078,
            0.882,
            0.811,
            0.793,
            0.793,
            0.798,
            0.772,
            0.757,
            0.700,
        ]
        cd = [
            0.027,
            0.025,
            0.024,
            0.023,
            0.022,
            0.022,
            0.023,
            0.025,
            0.027,
            0.028,
            0.024,
            0.019,
            0.017,
            0.015,
            0.017,
            0.019,
            0.021,
            0.024,
            0.027,
            0.031,
            0.037,
            0.046,
            0.058,
            0.074,
            0.088,
            0.101,
            0.114,
            0.128,
            0.142,
            0.155,
            0.321,
            0.525,
            0.742,
        ]
        cm = [
            -0.0037,
            -0.0044,
            -0.0051,
            0.0018,
            -0.0216,
            -0.0282,
            -0.0346,
            -0.0405,
            -0.0455,
            -0.0507,
            -0.0404,
            -0.0321,
            -0.0281,
            -0.0284,
            -0.0322,
            -0.0361,
            -0.0363,
            -0.0393,
            -0.0398,
            -0.0983,
            -0.1242,
            -0.1155,
            -0.1068,
            -0.0981,
            -0.0894,
            -0.0807,
            -0.072,
            -0.0633,
            -0.054,
            -0.045,
            -0.036,
            -0.22,
            -0.13,
        ]
        cm_zeros = np.zeros(len(cm))
        Re = 1

        self.polar = Polar(Re, alpha, cl, cd, cm)
        self.polar2 = Polar(Re, alpha, cl, cd, cm_zeros)

    def test_stall1(self):
        R = 2.4
        r = 0.25 * R
        chord = 0.18
        Omega = 200 * pi / 30
        Uinf = 10.0
        tsr = Omega * R / Uinf

        newpolar = self.polar.correction3D(
            r / R, chord / r, tsr, alpha_max_corr=30, alpha_linear_min=-4, alpha_linear_max=4
        )

        cl_3d = [
            -0.84628298,
            -0.75228154,
            -0.64170322,
            -0.53398298,
            -0.43026406,
            -0.32825998,
            -0.22739914,
            -0.12996799,
            -0.04053948,
            0.06203622,
            0.21891545,
            0.33235184,
            0.4337843,
            0.55007878,
            0.67551912,
            0.73636683,
            0.81036171,
            0.89750377,
            0.98121612,
            1.06378525,
            1.14521114,
            1.20948854,
            1.26804979,
            1.32832588,
            1.328,
            1.132,
            1.061,
            1.043,
            1.043,
            1.048,
            0.9595,
            0.8195,
            0.7,
        ]
        cd_3d = [
            0.027,
            0.025,
            0.024,
            0.023,
            0.022,
            0.022,
            0.023,
            0.025,
            0.027,
            0.028,
            0.024,
            0.019,
            0.017,
            0.015,
            0.017,
            0.019,
            0.021,
            0.024,
            0.027,
            0.031,
            0.037,
            0.046,
            0.058,
            0.074,
            0.088,
            0.101,
            0.114,
            0.128,
            0.142,
            0.155,
            0.321,
            0.525,
            0.742,
        ]
        # test equality
        np.testing.assert_allclose(newpolar.cl, cl_3d, atol=1e-3, rtol=1e-3)
        np.testing.assert_allclose(newpolar.cd, cd_3d, atol=1e-3, rtol=1e-3)

    def test_stall2(self):
        R = 2.4
        r = 0.75 * R
        chord = 0.28
        Omega = 200 * pi / 30
        Uinf = 14.0
        tsr = Omega * R / Uinf

        newpolar = self.polar.correction3D(
            r / R, chord / r, tsr, alpha_max_corr=30, alpha_linear_min=-4, alpha_linear_max=4
        )

        cl_3d = [
            -0.81312305,
            -0.72885733,
            -0.61871207,
            -0.5130288,
            -0.41359231,
            -0.31683302,
            -0.22185852,
            -0.13223842,
            -0.05511188,
            0.04253981,
            0.22496931,
            0.33957657,
            0.43544346,
            0.5545127,
            0.68786031,
            0.72036695,
            0.77339873,
            0.84695567,
            0.91515823,
            0.98157599,
            1.04620895,
            1.08406997,
            1.113007,
            1.14462124,
            1.15214072,
            0.98921218,
            0.93783339,
            0.9337517,
            0.94573318,
            0.96217664,
            0.9595,
            0.8195,
            0.7,
        ]
        cd_3d = [
            0.027,
            0.025,
            0.024,
            0.023,
            0.022,
            0.022,
            0.023,
            0.025,
            0.027,
            0.028,
            0.024,
            0.019,
            0.017,
            0.015,
            0.017,
            0.019,
            0.021,
            0.024,
            0.027,
            0.031,
            0.037,
            0.046,
            0.058,
            0.074,
            0.088,
            0.101,
            0.114,
            0.128,
            0.142,
            0.155,
            0.321,
            0.525,
            0.742,
        ]

        # test equality
        np.testing.assert_allclose(newpolar.cl, cl_3d, atol=1e-3)
        np.testing.assert_allclose(newpolar.cd, cd_3d, atol=1e-3)

    def test_stall3(self):
        R = 5.0
        r = 0.5 * R
        chord = 0.5
        Omega = 100 * pi / 30
        Uinf = 10.0
        tsr = Omega * R / Uinf

        newpolar = self.polar.correction3D(
            r / R, chord / r, tsr, alpha_max_corr=30, alpha_linear_min=-4, alpha_linear_max=4
        )

        cl_3d = [
            -0.82374342,
            -0.73635957,
            -0.62607561,
            -0.51973994,
            -0.41893189,
            -0.32049281,
            -0.22363306,
            -0.13151125,
            -0.05044467,
            0.04878406,
            0.2230304,
            0.33726265,
            0.43491207,
            0.55309262,
            0.68390771,
            0.72549134,
            0.78523713,
            0.86314507,
            0.93631506,
            1.00790573,
            1.07791708,
            1.12423867,
            1.16266366,
            1.20345763,
            1.22293081,
            1.09157913,
            1.05893482,
            1.043,
            1.043,
            1.048,
            0.9595,
            0.8195,
            0.7,
        ]
        cd_3d = [
            0.027,
            0.025,
            0.024,
            0.023,
            0.022,
            0.022,
            0.023,
            0.025,
            0.027,
            0.028,
            0.024,
            0.019,
            0.017,
            0.015,
            0.017,
            0.019,
            0.021,
            0.024,
            0.027,
            0.031,
            0.037,
            0.046,
            0.058,
            0.074,
            0.088,
            0.101,
            0.114,
            0.128,
            0.142,
            0.155,
            0.321,
            0.525,
            0.742,
        ]

        # test equality
        np.testing.assert_allclose(newpolar.cl, cl_3d, atol=1e-3)
        np.testing.assert_allclose(newpolar.cd, cd_3d, atol=1e-3)

    def test_stall4_cm(self):
        R = 5.0
        r = 0.5 * R
        chord = 0.5
        Omega = 100 * pi / 30
        Uinf = 10.0
        tsr = Omega * R / Uinf

        newpolar = self.polar2.correction3D(
            r / R, chord / r, tsr, alpha_max_corr=30, alpha_linear_min=-4, alpha_linear_max=4
        )

        cl_3d = [
            -0.82374342,
            -0.73635957,
            -0.62607561,
            -0.51973994,
            -0.41893189,
            -0.32049281,
            -0.22363306,
            -0.13151125,
            -0.05044467,
            0.04878406,
            0.2230304,
            0.33726265,
            0.43491207,
            0.55309262,
            0.68390771,
            0.72549134,
            0.78523713,
            0.86314507,
            0.93631506,
            1.00790573,
            1.07791708,
            1.12423867,
            1.16266366,
            1.20345763,
            1.22293081,
            1.09157913,
            1.05893482,
            1.043,
            1.043,
            1.048,
            0.9595,
            0.8195,
            0.7,
        ]
        cd_3d = [
            0.027,
            0.025,
            0.024,
            0.023,
            0.022,
            0.022,
            0.023,
            0.025,
            0.027,
            0.028,
            0.024,
            0.019,
            0.017,
            0.015,
            0.017,
            0.019,
            0.021,
            0.024,
            0.027,
            0.031,
            0.037,
            0.046,
            0.058,
            0.074,
            0.088,
            0.101,
            0.114,
            0.128,
            0.142,
            0.155,
            0.321,
            0.525,
            0.742,
        ]
        cm_zeros = np.zeros(len(cd_3d))

        # test equality
        np.testing.assert_allclose(newpolar.cl, cl_3d, atol=1e-3)
        np.testing.assert_allclose(newpolar.cd, cd_3d, atol=1e-3)
        np.testing.assert_allclose(newpolar.cm, cm_zeros, atol=1e-3)


class TestExtrap(unittest.TestCase):
    def setUp(self):
        alpha = [
            -10.1,
            -8.2,
            -6.1,
            -4.1,
            -2.1,
            0.1,
            2,
            4.1,
            6.2,
            8.1,
            10.2,
            11.3,
            12.1,
            13.2,
            14.2,
            15.3,
            16.3,
            17.1,
            18.1,
            19.1,
            20.1,
        ]
        cl = [
            -0.6300,
            -0.5600,
            -0.6400,
            -0.4200,
            -0.2100,
            0.0500,
            0.3000,
            0.5400,
            0.7900,
            0.9000,
            0.9300,
            0.9200,
            0.9500,
            0.9900,
            1.0100,
            1.0200,
            1.0000,
            0.9400,
            0.8500,
            0.7000,
            0.6600,
        ]
        cd = [
            0.0390,
            0.0233,
            0.0131,
            0.0134,
            0.0119,
            0.0122,
            0.0116,
            0.0144,
            0.0146,
            0.0162,
            0.0274,
            0.0303,
            0.0369,
            0.0509,
            0.0648,
            0.0776,
            0.0917,
            0.0994,
            0.2306,
            0.3142,
            0.3186,
        ]
        cm = [
            -0.0044,
            -0.0051,
            0.0018,
            -0.0216,
            -0.0282,
            -0.0346,
            -0.0405,
            -0.0455,
            -0.0507,
            -0.0404,
            -0.0321,
            -0.0281,
            -0.0284,
            -0.0322,
            -0.0361,
            -0.0363,
            -0.0393,
            -0.0398,
            -0.0983,
            -0.1242,
            -0.1155,
        ]
        cm_zeros = np.zeros(len(cm))
        Re = 1
        self.polar = Polar(Re, alpha, cl, cd, cm)
        self.polar2 = Polar(Re, alpha, cl, cd, cm_zeros)

    def test_extrap1(self):
        cdmax = 1.29
        newpolar = self.polar.extrapolate(cdmax=cdmax)

        alpha_extrap = [
            -180,
            -170,
            -160,
            -150,
            -140,
            -130,
            -120,
            -110,
            -100,
            -90,
            -80,
            -70,
            -60,
            -50,
            -40,
            -30,
            -20,
            -10.1,
            -8.2,
            -6.1,
            -4.1,
            -2.1,
            0.1,
            2,
            4.1,
            6.2,
            8.1,
            10.2,
            11.3,
            12.1,
            13.2,
            14.2,
            15.3,
            16.3,
            17.1,
            18.1,
            19.1,
            20.1,
            30,
            40,
            50,
            60,
            70,
            80,
            90,
            100,
            110,
            120,
            130,
            140,
            150,
            160,
            170,
            180,
        ]
        cl_extrap = [
            0.0000,
            0.2299,
            0.4597,
            0.4907,
            0.5053,
            0.4805,
            0.4102,
            0.2985,
            0.1565,
            0.0000,
            -0.1565,
            -0.2985,
            -0.4102,
            -0.4805,
            -0.5053,
            -0.4907,
            -0.4637,
            -0.6300,
            -0.5600,
            -0.6400,
            -0.4200,
            -0.2100,
            0.0500,
            0.3000,
            0.5400,
            0.7900,
            0.9000,
            0.9300,
            0.9200,
            0.9500,
            0.9900,
            1.0100,
            1.0200,
            1.0000,
            0.9400,
            0.8500,
            0.7000,
            0.6600,
            0.7010,
            0.7219,
            0.6864,
            0.5860,
            0.4264,
            0.2235,
            0.0000,
            -0.1565,
            -0.2985,
            -0.4102,
            -0.4805,
            -0.5053,
            -0.4907,
            -0.4597,
            -0.2299,
            0.0000,
        ]
        cd_extrap = [
            0.1770,
            0.2132,
            0.3173,
            0.4758,
            0.6686,
            0.8708,
            1.0560,
            1.1996,
            1.2818,
            1.2900,
            1.2818,
            1.1996,
            1.0560,
            0.8708,
            0.6686,
            0.4758,
            0.3158,
            0.0390,
            0.0233,
            0.0131,
            0.0134,
            0.0119,
            0.0122,
            0.0116,
            0.0144,
            0.0146,
            0.0162,
            0.0274,
            0.0303,
            0.0369,
            0.0509,
            0.0648,
            0.0776,
            0.0917,
            0.0994,
            0.2306,
            0.3142,
            0.3186,
            0.4758,
            0.6686,
            0.8708,
            1.0560,
            1.1996,
            1.2818,
            1.2900,
            1.2818,
            1.1996,
            1.0560,
            0.8708,
            0.6686,
            0.4758,
            0.3173,
            0.2132,
            0.1770,
        ]
        cm_extrap = [
            0.0000,
            0.4000,
            0.2431,
            0.2568,
            0.2865,
            0.3185,
            0.3458,
            0.3632,
            0.3672,
            0.3559,
            0.3443,
            0.3182,
            0.2808,
            0.2362,
            0.1886,
            0.1414,
            0.0942,
            -0.0044,
            -0.0051,
            0.0018,
            -0.0216,
            -0.0282,
            -0.0346,
            -0.0405,
            -0.0455,
            -0.0507,
            -0.0404,
            -0.0321,
            -0.0281,
            -0.0284,
            -0.0322,
            -0.0361,
            -0.0363,
            -0.0393,
            -0.0398,
            -0.0983,
            -0.1242,
            -0.1155,
            -0.1710,
            -0.2202,
            -0.2637,
            -0.3002,
            -0.3284,
            -0.3471,
            -0.3559,
            -0.3672,
            -0.3632,
            -0.3458,
            -0.3185,
            -0.2865,
            -0.2568,
            -0.2431,
            -0.5000,
            0.0000,
        ]

        # re-interpolate b/c angles of attack are different
        cl = np.interp(alpha_extrap, newpolar.alpha, newpolar.cl)
        cd = np.interp(alpha_extrap, newpolar.alpha, newpolar.cd)
        cm = np.interp(alpha_extrap, newpolar.alpha, newpolar.cm)

        # test equality
        np.testing.assert_allclose(cl, cl_extrap, atol=1.5e-4)
        np.testing.assert_allclose(cd, cd_extrap, atol=1.5e-4)
        np.testing.assert_allclose(cm, cm_extrap, atol=5e-3)

    def test_extrap2(self):
        cdmax = 1.0
        newpolar = self.polar.extrapolate(cdmax=cdmax)

        alpha_extrap = [
            -180,
            -170,
            -160,
            -150,
            -140,
            -130,
            -120,
            -110,
            -100,
            -90,
            -80,
            -70,
            -60,
            -50,
            -40,
            -30,
            -20,
            -10.1,
            -8.2,
            -6.1,
            -4.1,
            -2.1,
            0.1,
            2,
            4.1,
            6.2,
            8.1,
            10.2,
            11.3,
            12.1,
            13.2,
            14.2,
            15.3,
            16.3,
            17.1,
            18.1,
            19.1,
            20.1,
            30,
            40,
            50,
            60,
            70,
            80,
            90,
            100,
            110,
            120,
            130,
            140,
            150,
            160,
            170,
            180,
        ]
        cl_extrap = [
            0.0000,
            0.2299,
            0.4597,
            0.4411,
            0.4287,
            0.3943,
            0.3297,
            0.2364,
            0.1225,
            0.0000,
            -0.1225,
            -0.2364,
            -0.3297,
            -0.3943,
            -0.4287,
            -0.4411,
            -0.4637,
            -0.6300,
            -0.5600,
            -0.6400,
            -0.4200,
            -0.2100,
            0.0500,
            0.3000,
            0.5400,
            0.7900,
            0.9000,
            0.9300,
            0.9200,
            0.9500,
            0.9900,
            1.0100,
            1.0200,
            1.0000,
            0.9400,
            0.8500,
            0.7000,
            0.6600,
            0.6302,
            0.6124,
            0.5633,
            0.4710,
            0.3378,
            0.1750,
            0.0000,
            -0.1225,
            -0.2364,
            -0.3297,
            -0.3943,
            -0.4287,
            -0.4411,
            -0.4597,
            -0.2299,
            0.0000,
        ]
        cd_extrap = [
            0.2135,
            0.2404,
            0.3176,
            0.4349,
            0.5767,
            0.7241,
            0.8568,
            0.9560,
            1.0069,
            1.0000,
            1.0069,
            0.9560,
            0.8568,
            0.7241,
            0.5767,
            0.4349,
            0.3158,
            0.0390,
            0.0233,
            0.0131,
            0.0134,
            0.0119,
            0.0122,
            0.0116,
            0.0144,
            0.0146,
            0.0162,
            0.0274,
            0.0303,
            0.0369,
            0.0509,
            0.0648,
            0.0776,
            0.0917,
            0.0994,
            0.2306,
            0.3142,
            0.3186,
            0.4349,
            0.5767,
            0.7241,
            0.8568,
            0.9560,
            1.0069,
            1.0000,
            1.0069,
            0.9560,
            0.8568,
            0.7241,
            0.5767,
            0.4349,
            0.3176,
            0.2404,
            0.2135,
        ]
        cm_extrap = [
            0.0000,
            0.4000,
            0.2432,
            0.2354,
            0.2500,
            0.2695,
            0.2864,
            0.2961,
            0.2956,
            0.2834,
            0.2776,
            0.2603,
            0.2337,
            0.2013,
            0.1663,
            0.1310,
            0.0942,
            -0.0044,
            -0.0051,
            0.0018,
            -0.0216,
            -0.0282,
            -0.0346,
            -0.0405,
            -0.0455,
            -0.0507,
            -0.0404,
            -0.0321,
            -0.0281,
            -0.0284,
            -0.0322,
            -0.0361,
            -0.0363,
            -0.0393,
            -0.0398,
            -0.0983,
            -0.1242,
            -0.1155,
            -0.1577,
            -0.1930,
            -0.2239,
            -0.2494,
            -0.2683,
            -0.2798,
            -0.2834,
            -0.2956,
            -0.2961,
            -0.2864,
            -0.2695,
            -0.2500,
            -0.2354,
            -0.2432,
            -0.5000,
            0.0000,
        ]

        # re-interpolate b/c angles of attack are different
        cl = np.interp(alpha_extrap, newpolar.alpha, newpolar.cl)
        cd = np.interp(alpha_extrap, newpolar.alpha, newpolar.cd)
        cm = np.interp(alpha_extrap, newpolar.alpha, newpolar.cm)

        # test equality
        np.testing.assert_allclose(cl, cl_extrap, atol=1.5e-4)
        np.testing.assert_allclose(cd, cd_extrap, atol=1.5e-4)
        np.testing.assert_allclose(cm, cm_extrap, atol=5e-3)

    def test_extrap3(self):
        cdmax = 1.5
        newpolar = self.polar.extrapolate(cdmax)

        alpha_extrap = [
            -180,
            -170,
            -160,
            -150,
            -140,
            -130,
            -120,
            -110,
            -100,
            -90,
            -80,
            -70,
            -60,
            -50,
            -40,
            -30,
            -20,
            -10.1,
            -8.2,
            -6.1,
            -4.1,
            -2.1,
            0.1,
            2,
            4.1,
            6.2,
            8.1,
            10.2,
            11.3,
            12.1,
            13.2,
            14.2,
            15.3,
            16.3,
            17.1,
            18.1,
            19.1,
            20.1,
            30,
            40,
            50,
            60,
            70,
            80,
            90,
            100,
            110,
            120,
            130,
            140,
            150,
            160,
            170,
            180,
        ]
        cl_extrap = [
            0.0000,
            0.2299,
            0.4597,
            0.5266,
            0.5608,
            0.5429,
            0.4685,
            0.3434,
            0.1810,
            0.0000,
            -0.1810,
            -0.3434,
            -0.4685,
            -0.5429,
            -0.5608,
            -0.5266,
            -0.4637,
            -0.6300,
            -0.5600,
            -0.6400,
            -0.4200,
            -0.2100,
            0.0500,
            0.3000,
            0.5400,
            0.7900,
            0.9000,
            0.9300,
            0.9200,
            0.9500,
            0.9900,
            1.0100,
            1.0200,
            1.0000,
            0.9400,
            0.8500,
            0.7000,
            0.6600,
            0.7523,
            0.8012,
            0.7756,
            0.6693,
            0.4906,
            0.2586,
            0.0000,
            -0.1810,
            -0.3434,
            -0.4685,
            -0.5429,
            -0.5608,
            -0.5266,
            -0.4597,
            -0.2299,
            0.0000,
        ]
        cd_extrap = [
            0.1506,
            0.1936,
            0.3170,
            0.5054,
            0.7351,
            0.9771,
            1.2003,
            1.3760,
            1.4809,
            1.5000,
            1.4809,
            1.3760,
            1.2003,
            0.9771,
            0.7351,
            0.5054,
            0.3158,
            0.0390,
            0.0233,
            0.0131,
            0.0134,
            0.0119,
            0.0122,
            0.0116,
            0.0144,
            0.0146,
            0.0162,
            0.0274,
            0.0303,
            0.0369,
            0.0509,
            0.0648,
            0.0776,
            0.0917,
            0.0994,
            0.2306,
            0.3142,
            0.3186,
            0.5054,
            0.7351,
            0.9771,
            1.2003,
            1.3760,
            1.4809,
            1.5000,
            1.4809,
            1.3760,
            1.2003,
            0.9771,
            0.7351,
            0.5054,
            0.3170,
            0.1936,
            0.1506,
        ]
        cm_extrap = [
            0.0000,
            0.4000,
            0.2431,
            0.2723,
            0.3130,
            0.3540,
            0.3888,
            0.4118,
            0.4190,
            0.4084,
            0.3926,
            0.3602,
            0.3148,
            0.2614,
            0.2049,
            0.1488,
            0.0942,
            -0.0044,
            -0.0051,
            0.0018,
            -0.0216,
            -0.0282,
            -0.0346,
            -0.0405,
            -0.0455,
            -0.0507,
            -0.0404,
            -0.0321,
            -0.0281,
            -0.0284,
            -0.0322,
            -0.0361,
            -0.0363,
            -0.0393,
            -0.0398,
            -0.0983,
            -0.1242,
            -0.1155,
            -0.1807,
            -0.2399,
            -0.2925,
            -0.3370,
            -0.3719,
            -0.3959,
            -0.4084,
            -0.4190,
            -0.4118,
            -0.3888,
            -0.3540,
            -0.3130,
            -0.2723,
            -0.2431,
            -0.5000,
            0.0000,
        ]

        # re-interpolate b/c angles of attack are different
        cl = np.interp(alpha_extrap, newpolar.alpha, newpolar.cl)
        cd = np.interp(alpha_extrap, newpolar.alpha, newpolar.cd)
        cm = np.interp(alpha_extrap, newpolar.alpha, newpolar.cm)

        # test equality
        np.testing.assert_allclose(cl, cl_extrap, atol=1.5e-4)
        np.testing.assert_allclose(cd, cd_extrap, atol=1.5e-4)
        np.testing.assert_allclose(cm, cm_extrap, atol=5e-3)


class TestMisc(unittest.TestCase):
    def setUp(self):
        alpha = [
            -10.1,
            -8.2,
            -6.1,
            -4.1,
            -2.1,
            0.1,
            2,
            4.1,
            6.2,
            8.1,
            10.2,
            11.3,
            12.1,
            13.2,
            14.2,
            15.3,
            16.3,
            17.1,
            18.1,
            19.1,
            20.1,
        ]
        cl = [
            -0.6300,
            -0.5600,
            -0.6400,
            -0.4200,
            -0.2100,
            0.0500,
            0.3000,
            0.5400,
            0.7900,
            0.9000,
            0.9300,
            0.9200,
            0.9500,
            0.9900,
            1.0100,
            1.0200,
            1.0000,
            0.9400,
            0.8500,
            0.7000,
            0.6600,
        ]
        cd = [
            0.0390,
            0.0233,
            0.0131,
            0.0134,
            0.0119,
            0.0122,
            0.0116,
            0.0144,
            0.0146,
            0.0162,
            0.0274,
            0.0303,
            0.0369,
            0.0509,
            0.0648,
            0.0776,
            0.0917,
            0.0994,
            0.2306,
            0.3142,
            0.3186,
        ]
        cm = [
            -0.0044,
            -0.0051,
            0.0018,
            -0.0216,
            -0.0282,
            -0.0346,
            -0.0405,
            -0.0455,
            -0.0507,
            -0.0404,
            -0.0321,
            -0.0281,
            -0.0284,
            -0.0322,
            -0.0361,
            -0.0363,
            -0.0393,
            -0.0398,
            -0.0983,
            -0.1242,
            -0.1155,
        ]
        cm_zeros = np.zeros(len(cm))
        Re = 1
        self.polar = Polar(Re, alpha, cl, cd, cm)
        self.polar2 = Polar(Re, alpha, cl, cd, cm_zeros)

    def test_unsteady(self):
        alpha0, alpha1, alpha2, cnSlope, cn1, cn2, cd0, cm0 = self.polar.unsteadyParams()

        np.testing.assert_allclose(alpha0, -0.32307692307692304)
        np.testing.assert_allclose(alpha1, 9.260783831245934)
        np.testing.assert_allclose(alpha2, -6.779334979177289)
        np.testing.assert_allclose(cnSlope, 6.4380618436681765)
        np.testing.assert_allclose(cn1, 0.9201540372961516)
        np.testing.assert_allclose(cn2, -0.6377683435797556)
        np.testing.assert_allclose(cd0, 0.012142307692307694)
        np.testing.assert_allclose(cm0, -0.03336923076923077)

    def test_fully_separated(self):
        cl_fs, f_st = self.polar.cl_fully_separated()

        cl_fs_ref = np.array(
            [
                -0.63,
                -0.42017185,
                -0.35815607,
                -0.23440711,
                -0.11213462,
                0.02669872,
                0.15,
                0.2815297,
                0.41432191,
                0.51685242,
                0.60852946,
                0.6464375,
                0.68202361,
                0.7299095,
                0.76769179,
                0.8037866,
                0.82370687,
                0.81723832,
                0.78926905,
                0.69419819,
                0.65999953,
            ]
        )
        f_st_ref = np.array(
            [
                0.00000000e00,
                2.34199688e-01,
                7.26644559e-01,
                7.32580663e-01,
                8.34063987e-01,
                8.34063987e-01,
                1.00000000e00,
                8.92315821e-01,
                8.77625013e-01,
                6.71133852e-01,
                4.28392660e-01,
                3.20122429e-01,
                2.90558283e-01,
                2.55881726e-01,
                2.18728235e-01,
                1.78134763e-01,
                1.33254382e-01,
                8.56818538e-02,
                3.81986876e-02,
                3.19820908e-03,
                2.39632149e-07,
            ]
        )

        np.testing.assert_allclose(cl_fs, cl_fs_ref)
        np.testing.assert_allclose(f_st, f_st_ref)

    def test_cl_max(self):
        cl_max, alpha_cl_max = self.polar.cl_max()

        np.testing.assert_allclose(cl_max, 1.02)
        np.testing.assert_allclose(alpha_cl_max, 15.3)

    def test_linear_region(self):
        alpha_linear_region, cl_linear_region, slope, alpha0 = self.polar.linear_region()

        np.testing.assert_allclose(alpha_linear_region, np.array([-3.36474623, 5.96375]))
        np.testing.assert_allclose(cl_linear_region, np.array([-0.39279835, 0.811875]))
        np.testing.assert_allclose(slope, 0.1291390728476821)
        np.testing.assert_allclose(alpha0, -0.32307692307692304)


def suite():
    suite = unittest.TestSuite()
    suite.addTest(unittest.makeSuite(TestBlend))
    suite.addTest(unittest.makeSuite(Test3DStall))
    suite.addTest(unittest.makeSuite(TestExtrap))
    suite.addTest(unittest.makeSuite(TestMisc))
    return suite


if __name__ == "__main__":
    result = unittest.TextTestRunner().run(suite())

    if result.wasSuccessful():
        exit(0)
    else:
        exit(1)
