"""
Tests for skyborn.ROF module.

This module tests the climate attribution analysis functionality,
including attribution models, data preprocessing, trend analysis, and utility functions.
"""

import pytest
import numpy as np
import pandas as pd
from unittest.mock import Mock, patch, MagicMock
from scipy import stats
import warnings

from skyborn.ROF import (
    AttributionModel,
    PreProcess,
    calculate_trend,
    calculate_uncertainty,
    all_trends,
    speco,
    chi2_test,
    project_vectors,
    unproject_vectors,
    SSM,
    get_nruns,
    Cm_estimate,
    Cv_estimate,
)


class TestAttributionModel:
    """Test the AttributionModel class."""

    @pytest.fixture
    def sample_data(self):
        """Create sample data for attribution testing."""
        np.random.seed(42)
        nt = 50  # time steps
        nf = 2  # number of forcings

        # Create synthetic observation data
        y = np.random.randn(nt) * 0.5 + np.linspace(0, 2, nt)

        # Create synthetic forcing data
        X = np.zeros((nt, nf))
        X[:, 0] = np.random.randn(nt) * 0.3 + np.linspace(0, 1.5, nt)  # GHG forcing
        X[:, 1] = np.random.randn(
            # Natural forcing
            nt
        ) * 0.2 + 0.2 * np.sin(np.linspace(0, 4 * np.pi, nt))

        return y, X

    @pytest.fixture
    def sample_matrices(self):
        """Create sample covariance matrices and projection matrix."""
        nt = 50
        nf = 2

        # Create symmetric positive definite covariance matrix
        np.random.seed(42)
        A = np.random.randn(nt - 1, nt - 1)
        Cf = np.dot(A, A.T) + 0.1 * np.eye(nt - 1)

        # Create projection matrix
        Proj = np.eye(nf)

        # Create control simulation data
        Z2 = np.random.randn(nt - 1, 100)

        return Cf, Proj, Z2

    def test_attribution_model_creation(self, sample_data):
        """Test AttributionModel initialization."""
        y, X = sample_data

        model = AttributionModel(X, y)

        assert model.nt == len(y)
        assert model.num_forcings == X.shape[1]
        assert model.n_reduced == len(y) - 1
        np.testing.assert_array_equal(model.y, y)
        np.testing.assert_array_equal(model.X, X)

    def test_ols_method(self, sample_data, sample_matrices):
        """Test OLS attribution method."""
        y, X = sample_data
        Cf, Proj, Z2 = sample_matrices

        model = AttributionModel(X, y)

        # Suppress print statements during testing
        with patch("builtins.print"):
            result = model.ols(Cf, Proj, Z2, cons_test="AT99")

        # Check return structure
        assert isinstance(result, dict)
        assert "beta_hat" in result
        assert "beta_hat_inf" in result
        assert "beta_hat_sup" in result

        # Check array shapes
        assert len(result["beta_hat"]) == X.shape[1]
        assert len(result["beta_hat_inf"]) == X.shape[1]
        assert len(result["beta_hat_sup"]) == X.shape[1]

        # Check confidence intervals are reasonable
        assert np.all(result["beta_hat_inf"] <= result["beta_hat"])
        assert np.all(result["beta_hat"] <= result["beta_hat_sup"])

    def test_ols_invalid_consistency_test(self, sample_data, sample_matrices):
        """Test OLS with invalid consistency test parameter."""
        y, X = sample_data
        Cf, Proj, Z2 = sample_matrices

        model = AttributionModel(X, y)

        # This should work with AT99 (only implemented option)
        with patch("builtins.print"):
            result = model.ols(Cf, Proj, Z2, cons_test="AT99")
        assert isinstance(result, dict)

    def test_ribes_method(self, sample_data):
        """Test Ribes attribution method."""
        y, X = sample_data
        nt, nf = X.shape

        # Create covariance matrices
        np.random.seed(42)

        # Create Cy (covariance for observations)
        A_y = np.random.randn(nt, nt)
        Cy = np.dot(A_y, A_y.T) + 0.1 * np.eye(nt)

        # Create Cxi (covariance for each forcing)
        Cxi = np.zeros((nf, nt, nt))
        for i in range(nf):
            A_i = np.random.randn(nt, nt)
            Cxi[i] = np.dot(A_i, A_i.T) + 0.1 * np.eye(nt)

        model = AttributionModel(X, y)

        # Suppress print statements during testing
        with patch("builtins.print"):
            y_star_hat, Xi_star_hat, Cy_star_hat, Cxi_star_hat = model.ribes(Cxi, Cy)

        # Check output shapes
        assert y_star_hat.shape == y.shape
        assert Xi_star_hat.shape == X.shape
        assert Cy_star_hat.shape == Cy.shape
        assert Cxi_star_hat.shape == Cxi.shape

        # Check that covariance matrices are positive definite
        assert np.all(np.linalg.eigvals(Cy_star_hat) > -1e-10)
        for i in range(nf):
            assert np.all(np.linalg.eigvals(Cxi_star_hat[i]) > -1e-10)

    def test_ribes_method_different_forcing_numbers(self):
        """Test Ribes method with different numbers of forcings."""
        np.random.seed(42)

        for nf in [1, 3, 5]:
            nt = 30
            y = np.random.randn(nt)
            X = np.random.randn(nt, nf)

            # Create covariance matrices
            A_y = np.random.randn(nt, nt)
            Cy = np.dot(A_y, A_y.T) + 0.1 * np.eye(nt)

            Cxi = np.zeros((nf, nt, nt))
            for i in range(nf):
                A_i = np.random.randn(nt, nt)
                Cxi[i] = np.dot(A_i, A_i.T) + 0.1 * np.eye(nt)

            model = AttributionModel(X, y)

            with patch("builtins.print"):
                y_star_hat, Xi_star_hat, Cy_star_hat, Cxi_star_hat = model.ribes(
                    Cxi, Cy
                )

            assert y_star_hat.shape == (nt,)
            assert Xi_star_hat.shape == (nt, nf)
            assert Cxi_star_hat.shape == (nf, nt, nt)


class TestPreProcess:
    """Test the PreProcess class."""

    @pytest.fixture
    def sample_preprocessing_data(self):
        """Create sample data for preprocessing tests."""
        np.random.seed(42)
        nt = 40
        nf = 2
        nz = 200  # number of control simulations

        y = np.random.randn(nt) + np.linspace(0, 1, nt)
        X = np.random.randn(nt, nf)
        Z = np.random.randn(nt, nz)

        return y, X, Z

    def test_preprocess_creation(self, sample_preprocessing_data):
        """Test PreProcess initialization."""
        y, X, Z = sample_preprocessing_data

        preprocess = PreProcess(y, X, Z)

        assert preprocess.nt == len(y)
        np.testing.assert_array_equal(preprocess.y, y)
        np.testing.assert_array_equal(preprocess.X, X)
        np.testing.assert_array_equal(preprocess.Z, Z)

    def test_extract_Z2_regular(self, sample_preprocessing_data):
        """Test extract_Z2 method with regular sampling."""
        y, X, Z = sample_preprocessing_data
        preprocess = PreProcess(y, X, Z)

        Z1, Z2 = preprocess.extract_Z2(method="regular", frac=0.5)

        # Check that Z1 and Z2 have correct shapes
        assert Z1.shape[0] == Z.shape[0]  # same number of time steps
        assert Z2.shape[0] == Z.shape[0]
        # total columns preserved
        assert Z1.shape[1] + Z2.shape[1] == Z.shape[1]

        # Check that we have approximately half in each
        assert abs(Z2.shape[1] / Z.shape[1] - 0.5) < 0.1

    def test_extract_Z2_different_fractions(self, sample_preprocessing_data):
        """Test extract_Z2 with different fractions."""
        y, X, Z = sample_preprocessing_data
        preprocess = PreProcess(y, X, Z)

        fractions = [0.2, 0.3, 0.7]

        for frac in fractions:
            Z1, Z2 = preprocess.extract_Z2(method="regular", frac=frac)

            # Check that total is preserved
            assert Z1.shape[1] + Z2.shape[1] == Z.shape[1]

            # Check approximate fraction (allowing for rounding)
            actual_frac = Z2.shape[1] / Z.shape[1]
            assert abs(actual_frac - frac) < 0.2  # Allow some tolerance

    def test_extract_Z2_invalid_method(self, sample_preprocessing_data):
        """Test extract_Z2 with invalid method."""
        y, X, Z = sample_preprocessing_data
        preprocess = PreProcess(y, X, Z)

        with pytest.raises(NotImplementedError):
            preprocess.extract_Z2(method="invalid")

    def test_proj_fullrank(self, sample_preprocessing_data):
        """Test proj_fullrank method."""
        y, X, Z = sample_preprocessing_data
        preprocess = PreProcess(y, X, Z)

        Z1, Z2 = preprocess.extract_Z2(method="regular", frac=0.5)
        yc, Xc, Z1c, Z2c = preprocess.proj_fullrank(Z1, Z2)

        # Check output shapes (should be reduced by 1 in first dimension)
        assert yc.shape == (y.shape[0] - 1,)
        assert Xc.shape == (X.shape[0] - 1, X.shape[1])
        assert Z1c.shape == (Z1.shape[0] - 1, Z1.shape[1])
        assert Z2c.shape == (Z2.shape[0] - 1, Z2.shape[1])

    def test_creg_ledoit_method(self, sample_preprocessing_data):
        """Test creg method with Ledoit-Wolf regularization."""
        y, X, Z = sample_preprocessing_data
        preprocess = PreProcess(y, X, Z)

        Z1, Z2 = preprocess.extract_Z2(method="regular", frac=0.5)
        yc, Xc, Z1c, Z2c = preprocess.proj_fullrank(Z1, Z2)

        Cr = preprocess.creg(Z1c.T, method="ledoit")

        # Check output is square matrix with correct dimensions
        expected_size = Z1c.shape[0]
        assert Cr.shape == (expected_size, expected_size)

        # Check that matrix is symmetric
        np.testing.assert_allclose(Cr, Cr.T, rtol=1e-10)

        # Check that matrix is positive definite
        eigenvals = np.linalg.eigvals(Cr)
        assert np.all(eigenvals > -1e-10)

    def test_creg_specified_method(self, sample_preprocessing_data):
        """Test creg method with specified parameters."""
        y, X, Z = sample_preprocessing_data
        preprocess = PreProcess(y, X, Z)

        Z1, Z2 = preprocess.extract_Z2(method="regular", frac=0.5)
        yc, Xc, Z1c, Z2c = preprocess.proj_fullrank(Z1, Z2)

        alpha1, alpha2 = 0.1, 0.9
        Cr = preprocess.creg(Z1c.T, method="specified", alpha1=alpha1, alpha2=alpha2)

        # Check output shape
        expected_size = Z1c.shape[0]
        assert Cr.shape == (expected_size, expected_size)

        # Check that matrix is symmetric and positive definite
        np.testing.assert_allclose(Cr, Cr.T, rtol=1e-10)
        eigenvals = np.linalg.eigvals(Cr)
        assert np.all(eigenvals > -1e-10)

    def test_creg_invalid_method(self, sample_preprocessing_data):
        """Test creg method with invalid method."""
        y, X, Z = sample_preprocessing_data
        preprocess = PreProcess(y, X, Z)

        Z1, Z2 = preprocess.extract_Z2(method="regular", frac=0.5)
        yc, Xc, Z1c, Z2c = preprocess.proj_fullrank(Z1, Z2)

        with pytest.raises(NotImplementedError):
            preprocess.creg(Z1c.T, method="invalid")


class TestTrendAnalysis:
    """Test trend analysis functions."""

    @pytest.fixture
    def sample_trend_data(self):
        """Create sample data for trend analysis."""
        np.random.seed(42)
        nt = 30

        # Create data with known trend
        trend_value = 0.05
        y = np.random.randn(nt - 1) * 0.2 + trend_value * np.arange(nt - 1)

        # Create covariance matrix
        A = np.random.randn(nt - 1, nt - 1)
        Cy = np.dot(A, A.T) * 0.01 + 0.001 * np.eye(nt - 1)

        return y, Cy, trend_value

    def test_calculate_trend(self, sample_trend_data):
        """Test calculate_trend function."""
        y, Cy, expected_trend = sample_trend_data

        calculated_trend = calculate_trend(y)

        # Check that calculated trend is a scalar
        assert np.isscalar(calculated_trend)

        # Check that trend is approximately correct (within reasonable range)
        assert abs(calculated_trend - expected_trend) < 0.1

    def test_calculate_trend_zero_trend(self):
        """Test calculate_trend with zero trend data."""
        np.random.seed(42)
        y = np.random.randn(29) * 0.1  # No trend, just noise

        calculated_trend = calculate_trend(y)

        # Should be close to zero
        assert abs(calculated_trend) < 0.1

    def test_calculate_uncertainty(self, sample_trend_data):
        """Test calculate_uncertainty function."""
        y, Cy, expected_trend = sample_trend_data

        # Use fewer samples for testing speed
        uncertainty = calculate_uncertainty(y, Cy, alpha=0.05, nsamples=100)

        # Check output shape
        assert uncertainty.shape == (2,)

        # Check that confidence interval makes sense
        trend_min, trend_max = uncertainty
        assert trend_min < trend_max

        # Check that the interval contains reasonable values
        calculated_trend = calculate_trend(y)
        assert (
            trend_min <= calculated_trend <= trend_max
            or abs(calculated_trend - trend_min) < 0.2
        )

    def test_calculate_uncertainty_different_alpha(self, sample_trend_data):
        """Test calculate_uncertainty with different significance levels."""
        y, Cy, expected_trend = sample_trend_data

        # Test with different alpha values
        alphas = [0.01, 0.05, 0.1]

        intervals = []
        for alpha in alphas:
            uncertainty = calculate_uncertainty(y, Cy, alpha=alpha, nsamples=100)
            intervals.append(uncertainty[1] - uncertainty[0])  # interval width

        # Smaller alpha should give wider intervals
        assert intervals[0] > intervals[1] > intervals[2]

    @patch("skyborn.ROF.trend.calculate_trend")
    @patch("skyborn.ROF.trend.calculate_uncertainty")
    def test_all_trends(self, mock_uncertainty, mock_trend):
        """Test all_trends function."""
        # Mock the functions to return predictable values
        mock_trend.side_effect = [0.05, 0.03, 0.02]  # obs, forcing1, forcing2
        mock_uncertainty.side_effect = [
            np.array([0.02, 0.08]),  # obs uncertainty
            np.array([0.01, 0.05]),  # forcing1 uncertainty
            np.array([0.00, 0.04]),  # forcing2 uncertainty
        ]

        # Create sample data
        np.random.seed(42)
        nt = 30
        nf = 2

        y_star_hat = np.random.randn(nt - 1)
        Xi_star_hat = np.random.randn(nt - 1, nf)
        Cy_star_hat = np.eye(nt - 1) * 0.01
        Cxi_star_hat = np.array([np.eye(nt - 1) * 0.01 for _ in range(nf)])

        # Suppress print statements
        with patch("builtins.print"):
            df = all_trends(y_star_hat, Xi_star_hat, Cy_star_hat, Cxi_star_hat)

        # Check output is DataFrame
        assert isinstance(df, pd.DataFrame)

        # Check DataFrame structure
        expected_columns = ["forcing", "trend", "trend_min", "trend_max"]
        assert list(df.columns) == expected_columns

        # Check number of rows (1 observation + nf forcings)
        assert len(df) == 1 + nf

        # Check that trends and uncertainties were called correctly
        assert mock_trend.call_count == 3
        assert mock_uncertainty.call_count == 3


class TestUtilityFunctions:
    """Test utility functions."""

    def test_speco_symmetric_matrix(self):
        """Test speco function with symmetric matrix."""
        # Create a symmetric positive definite matrix
        np.random.seed(42)
        A = np.random.randn(5, 5)
        C = np.dot(A, A.T)

        eigenvectors, eigenvalues_diag = speco(C)

        # Check output shapes
        assert eigenvectors.shape == C.shape
        assert eigenvalues_diag.shape == C.shape

        # Check that eigenvalues are in descending order
        eigenvals = np.diag(eigenvalues_diag)
        assert np.all(eigenvals[:-1] >= eigenvals[1:])

        # Check that eigenvectors are orthogonal
        np.testing.assert_allclose(
            np.dot(eigenvectors.T, eigenvectors), np.eye(C.shape[0]), rtol=1e-10
        )

    def test_speco_identity_matrix(self):
        """Test speco with identity matrix."""
        I = np.eye(4)
        eigenvectors, eigenvalues_diag = speco(I)

        # For identity matrix, eigenvalues should all be 1
        eigenvals = np.diag(eigenvalues_diag)
        np.testing.assert_allclose(eigenvals, np.ones(4), rtol=1e-10)

    def test_speco_non_symmetric_matrix(self):
        """Test speco with non-symmetric matrix raises error."""
        # Create a non-symmetric matrix
        C = np.array([[1, 2], [3, 4]])

        with pytest.raises(ValueError, match="Matrix is not symmetric"):
            speco(C)

    def test_chi2_test(self):
        """Test chi2_test function."""
        # Test with known values
        d_cons = 5.0
        df = 3

        p_value = chi2_test(d_cons, df)

        # Check that p-value is between 0 and 1
        assert 0 <= p_value <= 1

        # Check against scipy implementation
        expected = 1 - stats.chi2.cdf(d_cons, df=df)
        np.testing.assert_allclose(p_value, expected, rtol=1e-10)

    def test_chi2_test_edge_cases(self):
        """Test chi2_test with edge cases."""
        # Test with d_cons = 0
        p_value = chi2_test(0.0, 5)
        assert abs(p_value - 1.0) < 1e-10

        # Test with large d_cons
        p_value = chi2_test(100.0, 5)
        assert p_value < 1e-10

    def test_project_vectors(self):
        """Test project_vectors function."""
        np.random.seed(42)
        nt = 20
        nf = 3

        X = np.random.randn(nt, nf)
        projected_X = project_vectors(nt, X)

        # Check output shape
        assert projected_X.shape == (nt - 1, nf)

        # Check that projection removes mean (projected data should have zero mean)
        # Note: Due to the nature of the projection, this is approximately true
        means = np.mean(projected_X, axis=0)
        assert np.all(np.abs(means) < 1e-10)

    def test_unproject_vectors(self):
        """Test unproject_vectors function."""
        np.random.seed(42)
        nt = 15
        nf = 2

        # Create projected data
        Xc = np.random.randn(nt - 1, nf)

        unprojected_X = unproject_vectors(nt, Xc)

        # Check output shape
        assert unprojected_X.shape == (nt, nf)

    def test_project_unproject_roundtrip(self):
        """Test that project and unproject are approximately inverse operations."""
        np.random.seed(42)
        nt = 10
        nf = 2

        # Start with zero-mean data (since projection removes mean)
        X_original = np.random.randn(nt, nf)
        X_original = X_original - np.mean(X_original, axis=0)

        # Project and then unproject
        X_projected = project_vectors(nt, X_original)
        X_recovered = unproject_vectors(nt, X_projected)

        # Should be approximately equal (up to mean removal)
        X_recovered_centered = X_recovered - np.mean(X_recovered, axis=0)
        np.testing.assert_allclose(X_original, X_recovered_centered, rtol=1e-10)

    def test_SSM_function(self):
        """Test SSM function."""
        np.random.seed(42)
        nt = 20
        n_members = 10

        # Create sample experiment dictionary
        X_dict = {
            "exp1": np.random.randn(n_members, nt),
            "exp2": np.random.randn(n_members, nt),
            "exp3": np.random.randn(n_members, nt),
        }

        # Create multi-model mean
        X_mm = np.random.randn(nt)

        result = SSM(X_dict, X_mm)

        # Check output shape
        assert result.shape == (nt - 1, nt - 1)

        # Check that result is diagonal matrix
        off_diagonal = result - np.diag(np.diag(result))
        np.testing.assert_allclose(
            off_diagonal, np.zeros_like(off_diagonal), atol=1e-10
        )

        # Check that diagonal values are non-negative
        assert np.all(np.diag(result) >= 0)

    def test_get_nruns(self):
        """Test get_nruns function."""
        # Create sample experiment dictionary
        X_dict = {
            "exp1": np.random.randn(5, 20),  # 5 members
            "exp2": np.random.randn(8, 20),  # 8 members
            "exp3": np.random.randn(3, 20),  # 3 members
        }

        nruns = get_nruns(X_dict)

        # Check output
        expected = np.array([5, 8, 3])
        np.testing.assert_array_equal(nruns, expected)

    def test_Cm_estimate(self):
        """Test Cm_estimate function."""
        np.random.seed(42)
        nt = 15
        n_members = 5

        # Create sample data
        X_dict = {
            "exp1": np.random.randn(n_members, nt),
            "exp2": np.random.randn(n_members, nt),
        }
        X_mm = np.random.randn(nt)

        # Create covariance matrix
        A = np.random.randn(nt - 1, nt - 1)
        Cv = np.dot(A, A.T) * 0.01 + 0.001 * np.eye(nt - 1)

        Cm_pos_hat = Cm_estimate(X_dict, Cv, X_mm)

        # Check output shape
        assert Cm_pos_hat.shape == (nt - 1, nt - 1)

        # Check that result is symmetric
        np.testing.assert_allclose(Cm_pos_hat, Cm_pos_hat.T, rtol=1e-10)

        # Check that result is positive semi-definite
        eigenvals = np.linalg.eigvals(Cm_pos_hat)
        assert np.all(eigenvals >= -1e-10)

    def test_Cv_estimate(self):
        """Test Cv_estimate function."""
        np.random.seed(42)
        nt = 12
        n_members = 4

        # Create sample data
        X_dict = {
            "exp1": np.random.randn(n_members, nt),
            # Different number of members
            "exp2": np.random.randn(n_members + 2, nt),
            "exp3": np.random.randn(n_members - 1, nt),
        }

        # Create covariance matrix
        A = np.random.randn(nt - 1, nt - 1)
        Cv = np.dot(A, A.T) * 0.01 + 0.001 * np.eye(nt - 1)

        Cv_estimate_result = Cv_estimate(X_dict, Cv)

        # Check output shape
        assert Cv_estimate_result.shape == Cv.shape

        # Check that result is symmetric
        np.testing.assert_allclose(Cv_estimate_result, Cv_estimate_result.T, rtol=1e-10)

        # Check that result is positive semi-definite
        eigenvals = np.linalg.eigvals(Cv_estimate_result)
        assert np.all(eigenvals >= -1e-10)


class TestEdgeCasesAndErrorHandling:
    """Test edge cases and error handling."""

    def test_attribution_model_empty_data(self):
        """Test AttributionModel with edge case data."""
        # Test with minimal data
        y = np.array([1.0, 2.0])
        X = np.array([[1.0], [2.0]])

        model = AttributionModel(X, y)
        assert model.nt == 2
        assert model.num_forcings == 1
        assert model.n_reduced == 1

    def test_singular_covariance_matrix(self):
        """Test behavior with singular covariance matrices."""
        np.random.seed(42)
        nt = 10

        # Create singular covariance matrix
        Cf = np.zeros((nt - 1, nt - 1))
        Cf[0, 0] = 1.0  # Only one non-zero eigenvalue

        y = np.random.randn(nt)
        X = np.random.randn(nt, 2)
        Proj = np.eye(2)
        Z2 = np.random.randn(nt - 1, 50)

        model = AttributionModel(X, y)

        # This should handle singular matrix (might use pseudoinverse internally)
        with patch("builtins.print"):
            with warnings.catch_warnings():
                warnings.simplefilter("ignore")
                try:
                    result = model.ols(Cf, Proj, Z2)
                    # If it succeeds, check basic structure
                    assert isinstance(result, dict)
                except np.linalg.LinAlgError:
                    # If it fails due to singular matrix, that's expected
                    pass

    def test_small_sample_sizes(self):
        """Test functions with very small sample sizes."""
        # Test with minimal data
        y = np.array([1.0])

        # This should work for trend calculation
        trend = calculate_trend(y)
        assert np.isscalar(trend)

    def test_numerical_stability(self):
        """Test numerical stability with extreme values."""
        np.random.seed(42)

        # Test with very small values
        small_matrix = np.eye(3) * 1e-12
        eigenvecs, eigenvals = speco(small_matrix)
        assert eigenvecs.shape == (3, 3)
        assert eigenvals.shape == (3, 3)

        # Test chi2_test with extreme values
        p_val = chi2_test(1e-10, 5)
        assert 0 <= p_val <= 1

        p_val = chi2_test(1e10, 5)
        assert 0 <= p_val <= 1


class TestIntegrationTests:
    """Integration tests for complete ROF workflow."""

    def test_complete_ols_workflow(self):
        """Test complete OLS workflow from preprocessing to attribution."""
        np.random.seed(42)
        nt = 30
        nf = 2
        nz = 100

        # Create synthetic data
        y = np.random.randn(nt) + np.linspace(0, 1, nt)
        X = np.random.randn(nt, nf)
        Z = np.random.randn(nt, nz)

        # Preprocessing
        preprocess = PreProcess(y, X, Z)
        Z1, Z2 = preprocess.extract_Z2(method="regular", frac=0.5)
        yc, Xc, Z1c, Z2c = preprocess.proj_fullrank(Z1, Z2)
        Cf = preprocess.creg(Z1c.T, method="ledoit")

        # Attribution
        model = AttributionModel(Xc.T, yc)
        Proj = np.eye(nf)

        with patch("builtins.print"):
            result = model.ols(Cf, Proj, Z2c)

        # Check that we get reasonable results
        assert isinstance(result, dict)
        assert len(result["beta_hat"]) == nf

    def test_complete_ribes_workflow(self):
        """Test complete Ribes workflow."""
        np.random.seed(42)
        nt = 25
        nf = 2

        # Create synthetic data
        y = np.random.randn(nt)
        X = np.random.randn(nt, nf)

        # Create covariance matrices
        A_y = np.random.randn(nt, nt)
        Cy = np.dot(A_y, A_y.T) * 0.01 + 0.001 * np.eye(nt)

        Cxi = np.zeros((nf, nt, nt))
        for i in range(nf):
            A_i = np.random.randn(nt, nt)
            Cxi[i] = np.dot(A_i, A_i.T) * 0.01 + 0.001 * np.eye(nt)

        # Run Ribes analysis
        model = AttributionModel(X, y)

        with patch("builtins.print"):
            y_star_hat, Xi_star_hat, Cy_star_hat, Cxi_star_hat = model.ribes(Cxi, Cy)

        # Calculate trends
        with patch("builtins.print"):
            df = all_trends(
                y_star_hat[:-1],
                Xi_star_hat[:-1],
                Cy_star_hat[:-1, :-1],
                Cxi_star_hat[:, :-1, :-1],
            )

        # Check that we get a complete result
        assert isinstance(df, pd.DataFrame)
        assert len(df) == 1 + nf

    def test_multimodel_analysis_workflow(self):
        """Test workflow with multiple models."""
        np.random.seed(42)
        nt = 20

        # Create multi-model ensemble data
        X_dict = {
            "model1": np.random.randn(5, nt),
            "model2": np.random.randn(7, nt),
            "model3": np.random.randn(4, nt),
        }
        X_mm = np.random.randn(nt)

        # Create covariance matrix
        A = np.random.randn(nt - 1, nt - 1)
        Cv = np.dot(A, A.T) * 0.01 + 0.001 * np.eye(nt - 1)

        # Run multi-model analysis
        nruns = get_nruns(X_dict)
        ssm_result = SSM(X_dict, X_mm)
        Cm_result = Cm_estimate(X_dict, Cv, X_mm)
        Cv_result = Cv_estimate(X_dict, Cv)

        # Check all results
        assert len(nruns) == 3
        assert ssm_result.shape == (nt - 1, nt - 1)
        assert Cm_result.shape == (nt - 1, nt - 1)
        assert Cv_result.shape == (nt - 1, nt - 1)


if __name__ == "__main__":
    pytest.main([__file__])
