


import os
import sys
import asyncio
import inspect
import traceback
import datetime
from typing import Any, Callable, Dict, List, Optional, Pattern, Tuple, Union, AsyncGenerator
from dataclasses import dataclass
from aioquic.asyncio.server import serve
from aioquic.asyncio.protocol import QuicConnectionProtocol
from aioquic.quic.configuration import QuicConfiguration
from aioquic.quic.events import QuicEvent, StreamDataReceived
from typing import Any, AsyncGenerator, Union


from gnobjects.net.objects import GNRequest, GNResponse, FileObject, CORSObject, TemplateObject
from gnobjects.net.fastcommands import AllGNFastCommands, GNFastCommand


from ._func_params_validation import register_schema_by_key, validate_params_by_key
from ._cors_resolver import resolve_cors

from ._routes import Route, _compile_path, _ensure_async, _convert_value



try:
    if not sys.platform.startswith("win"):
        import uvloop # type: ignore
        asyncio.set_event_loop_policy(uvloop.EventLoopPolicy())
except ImportError:
    print("uvloop не установлен")





import logging

logger = logging.getLogger("GNServer")
logger.setLevel(logging.DEBUG)

console = logging.StreamHandler()
console.setLevel(logging.DEBUG)
console.setFormatter(logging.Formatter("[GNServer] %(name)s: %(levelname)s: %(message)s"))
logger.addHandler(console)


def guess_type(filename: str) -> str:
    """
    Возвращает актуальный MIME-тип по расширению файла.
    Только современные и часто используемые типы.
    """
    ext = filename.lower().rsplit('.', 1)[-1] if '.' in filename else ''

    mime_map = {
        # 🔹 Текст и данные
        "txt": "text/plain",
        "html": "text/html",
        "css": "text/css",
        "csv": "text/csv",
        "xml": "application/xml",
        "json": "application/json",
        "js": "application/javascript",

        # 🔹 Изображения (актуальные для веба)
        "jpg": "image/jpeg",
        "jpeg": "image/jpeg",
        "png": "image/png",
        "gif": "image/gif",
        "webp": "image/webp",
        "svg": "image/svg+xml",
        "avif": "image/avif",
        "ico": "image/x-icon",

        # 🔹 Видео (современные форматы)
        "mp4": "video/mp4",
        "webm": "video/webm",

        # 🔹 Аудио (современные форматы)
        "mp3": "audio/mpeg",
        "ogg": "audio/ogg",
        "oga": "audio/ogg",
        "m4a": "audio/mp4",
        "flac": "audio/flac",

        # 🔹 Архивы
        "zip": "application/zip",
        "gz": "application/gzip",
        "tar": "application/x-tar",
        "7z": "application/x-7z-compressed",
        "rar": "application/vnd.rar",

        # 🔹 Документы (актуальные офисные)
        "pdf": "application/pdf",
        "docx": "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
        "xlsx": "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
        "pptx": "application/vnd.openxmlformats-officedocument.presentationml.presentation",

        # 🔹 Шрифты
        "woff": "font/woff",
        "woff2": "font/woff2",
        "ttf": "font/ttf",
        "otf": "font/otf",
    }

    return mime_map.get(ext, "application/octet-stream")








class App:
    def __init__(self):
        self._routes: List[Route] = []
        self._cors: Optional[CORSObject] = None
        self._events: Dict[str, List[Dict[str, Union[Any, Callable]]]] = {}

        self.domain: str = None # type: ignore

    def route(self, method: str, path: str, cors: Optional[CORSObject] = None):
        if path == '/':
            path = ''
        def decorator(fn: Callable[..., Any]):
            regex, param_types = _compile_path(path)
            self._routes.append(
                Route(
                    method.upper(),
                    path,
                    regex,
                    param_types,
                    _ensure_async(fn),
                    fn.__name__,
                    cors
                )
            )
            register_schema_by_key(fn)
            return fn
        return decorator

    def get(self, path: str, *, cors: Optional[CORSObject] = None):
        return self.route("GET", path, cors)

    def post(self, path: str, *, cors: Optional[CORSObject] = None):
        return self.route("POST", path, cors)

    def put(self, path: str, *, cors: Optional[CORSObject] = None):
        return self.route("PUT", path, cors)

    def delete(self, path: str, *, cors: Optional[CORSObject] = None):
        return self.route("DELETE", path, cors)

    
    def setRouteCors(self, cors: Optional[CORSObject] = None):
        self._cors = cors


    def addEventListener(self, name: str):
        def decorator(fn: Callable[[Optional[dict]], Any]):
            events = self._events.get(name, [])
            events.append({
                'func': fn,
                'async': inspect.iscoroutinefunction(fn),
                'parameters': inspect.signature(fn).parameters
                })
            self._events[name] = events
            
            return fn
        return decorator
    
    async def dispatchEvent(self, name: str, payload: Optional[str] = None) -> None:
        data_list = self._events.get(name, None)
        if data_list:
            for data in data_list:
                func: Callable = data['func']
                is_async = data['async']

                if not is_async:
                    if payload in data['parameters']: # type: ignore
                        func(payload=payload)
                    else:
                        func()
                else:
                    if payload in data['parameters']: # type: ignore
                        await func(payload=payload)
                    else:
                        await func()

        
    
    


    async def dispatchRequest(
        self, request: GNRequest
    ) -> Union[GNResponse, AsyncGenerator[GNResponse, None]]:
        path    = request.url.path
        method  = request.method.upper()
        cand    = {path, path.rstrip("/") or "/", f"{path}/"}
        allowed = set()

        for r in self._routes:
            m = next((r.regex.fullmatch(p) for p in cand if r.regex.fullmatch(p)), None)
            if not m:
                continue

            allowed.add(r.method)
            if r.method != method:
                continue

            if r.cors is not None and r.cors.allow_origins is not None:
                if request._origin is None:
                    return GNResponse("gn:backend:801", {'error': 'Cors error. Route has cors but request has no origin url.'})
                if not resolve_cors(request._origin, r.cors.allow_origins):
                    return GNResponse("gn:backend:802", {'error': 'Cors error: origin'})
                if r.cors.allow_methods is not None and request.method not in r.cors.allow_methods and '*' not in r.cors.allow_methods:
                    return GNResponse("gn:backend:803", {'error': 'Cors error: method'})

            sig = inspect.signature(r.handler)
            def _ann(name: str):
                param = sig.parameters.get(name)
                return param.annotation if param else inspect._empty

            kw: dict[str, Any] = {
                name: _convert_value(val, _ann(name), r.param_types.get(name, str))
                for name, val in m.groupdict().items()
            }

            for qn, qvals in request.url.params.items():
                if qn in kw:
                    continue
                raw = qvals if len(qvals) > 1 else qvals[0]
                kw[qn] = _convert_value(raw, _ann(qn), str)

            
            params = set(sig.parameters.keys())
            kw = {k: v for k, v in kw.items() if k in params}

            
            rv = validate_params_by_key(kw, r.handler)
            if rv is not None:
                raise AllGNFastCommands.UnprocessableEntity({'dev_error': rv, 'user_error': f'Server request error {self.domain}'})

            if "request" in sig.parameters:
                kw["request"] = request

            if inspect.isasyncgenfunction(r.handler):
                return r.handler(**kw)

            result = await r.handler(**kw)
            if isinstance(result, GNResponse):
                if r.cors is None:
                    if result._cors is None:
                        result._cors = self._cors
                else:
                    result._cors = r.cors

                if result._cors is not None and result._cors != r.cors and result._cors.allow_origins is not None:
                    if request._origin is None:
                        print(result._cors.allow_origins)
                        return GNResponse("gn:backend:801", {'error': 'Cors error. Route has cors but request has no origin url. [2]'})
                    if not resolve_cors(request._origin, result._cors.allow_origins):
                        return GNResponse("gn:backend:802", {'error': 'Cors error: origin'})
                    if result._cors.allow_methods is not None and request.method not in result._cors.allow_methods and '*' not in result._cors.allow_methods:
                        return GNResponse("gn:backend:803", {'error': 'Cors error: method'})
                return result
            else:
                raise TypeError(
                    f"{r.handler.__name__} returned {type(result)}; GNResponse expected"
                )

        if allowed:
            raise AllGNFastCommands.MethodNotAllowed()
        raise AllGNFastCommands.NotFound()


    def fastFile(self, path: str, file_path: str, cors: Optional[CORSObject] = None, template: Optional[TemplateObject] = None, payload: Optional[dict] = None):
        @self.get(path)
        async def r_static():
            nonlocal file_path
            if file_path.endswith('/'):
                file_path = file_path[:-1]
                
            if not os.path.isfile(file_path):
                raise AllGNFastCommands.NotFound()

            fileObject = FileObject(file_path, template)
            return GNResponse('ok', payload=payload, files=fileObject, cors=cors)


    def static(self, path: str, dir_path: str, cors: Optional[CORSObject] = None, template: Optional[TemplateObject] = None, payload: Optional[dict] = None):
        @self.get(f"{path}/{{_path:path}}")
        async def r_static(_path: str):
            file_path = os.path.join(dir_path, _path)
            
            if file_path.endswith('/'):
                file_path = file_path[:-1]
                
            if not os.path.isfile(file_path):
                raise AllGNFastCommands.NotFound()
            
            fileObject = FileObject(file_path, template)
            return GNResponse('ok', payload=payload, files=fileObject, cors=cors)




    def _init_sys_routes(self):
        @self.post('/!gn-vm-host/ping', cors=CORSObject())
        async def r_ping(request: GNRequest):
            if request.client.ip != '127.0.0.1':
                raise AllGNFastCommands.Forbidden()
            return GNResponse('ok', {'time': datetime.datetime.now(datetime.UTC).isoformat()})



    class _ServerProto(QuicConnectionProtocol):
        def __init__(self, *a, api: "App", **kw):
            super().__init__(*a, **kw)
            self._api = api
            self._buffer: Dict[int, bytearray] = {}
            self._streams: Dict[int, Tuple[asyncio.Queue[Optional[GNRequest]], bool]] = {}

        def quic_event_received(self, event: QuicEvent):
            if isinstance(event, StreamDataReceived):
                buf = self._buffer.setdefault(event.stream_id, bytearray())
                buf.extend(event.data)

                # пока не знаем, это стрим или нет

                if len(buf) < 8: # не дошел даже frame пакета
                    logger.debug(f'Пакет отклонен: {buf} < 8. Не доставлен фрейм')
                    return
                
                    
                # получаем длинну пакета
                mode, stream, lenght = GNRequest.type(buf)

                if mode not in (1, 2): # не наш пакет
                    logger.debug(f'Пакет отклонен: mode пакета {mode}. Разрешен 1, 2')
                    return
                
                if not stream: # если не стрим, то ждем конец quic стрима и запускаем обработку ответа
                    if event.end_stream:
                        request = GNRequest.deserialize(buf, mode)
                        request.stream_id = event.stream_id  # type: ignore
                        asyncio.create_task(self._handle_request(request, mode))
                        logger.debug(f'Отправлена задача разрешения пакета {request} route -> {request.route}')

                        self._buffer.pop(event.stream_id, None)
                    return
                
                # если стрим, то смотрим сколько пришло данных
                if len(buf) < lenght: # если пакет не весь пришел, пропускаем
                    return

                # первый в буфере пакет пришел полностью
        
                # берем пакет
                data = buf[:lenght]

                # удаляем его из буфера
                del buf[:lenght]

                # формируем запрос
                request = GNRequest.deserialize(data, mode)

                logger.debug(request, f'event.stream_id -> {event.stream_id}')

                request.stream_id = event.stream_id  # type: ignore

                queue, inapi = self._streams.setdefault(event.stream_id, (asyncio.Queue(), False))

                if request.method == 'gn:end-stream':
                    if event.stream_id in self._streams:
                        _ = self._streams.get(event.stream_id)
                        if _ is not None:
                            queue, inapi = _
                            if inapi:
                                queue.put_nowait(None)
                                self._buffer.pop(event.stream_id)
                                self._streams.pop(event.stream_id)
                                logger.debug(f'Закрываем стрим [{event.stream_id}]')
                                return

                queue.put_nowait(request)

                # отдаем очередь в интерфейс
                if not inapi:
                    self._streams[event.stream_id] = (queue, True)

                    async def w():
                        while True:
                            chunk = await queue.get()
                            if chunk is None:
                                break
                            yield chunk

                    request._stream = w  # type: ignore
                    asyncio.create_task(self._handle_request(request, mode))

        async def _handle_request(self, request: GNRequest, mode: int):

            request.client._data['remote_addr'] = self._quic._network_paths[0].addr

            try:
                response = await self._api.dispatchRequest(request)

                if inspect.isasyncgen(response):
                    async for chunk in response:  # type: ignore[misc]
                        chunk._stream = True
                        await self.sendResponse(request, chunk, mode, False)
                        
                    resp = GNResponse('gn:end-stream')
                    resp._stream = True

                    await self.sendResponse(request, resp, mode)
                    return

                if not isinstance(response, GNResponse):
                    await self.sendResponse(request, AllGNFastCommands.InternalServerError(), mode)
                    return

                await self.sendResponse(request, response, mode)
                logger.debug(f'Отправлен на сервер ответ -> {response.command} {response.payload if response.payload and len(str(response.payload)) < 200 else ''}')
            except Exception as e:
                if isinstance(e, GNFastCommand):
                    await self.sendResponse(request, e, mode)
                else:
                    logger.error('GNServer: error\n'  + traceback.format_exc())

                    await self.sendResponse(request, AllGNFastCommands.InternalServerError(), mode)
            

        
        async def sendResponse(self, request: GNRequest, response: GNResponse, mode: int, end_stream: bool = True):
            await response.assembly()


            self._quic.send_stream_data(request.stream_id, response.serialize(mode), end_stream=end_stream) # type: ignore
            self.transmit()

    def run(
        self,
        domain: str,
        port: int,
        cert_path: str,
        key_path: str,
        *,
        host: str = '0.0.0.0',
        idle_timeout: float = 20.0,
        wait: bool = True,
        run: Optional[Callable] = None
    ):
        """
        # Запустить сервер

        Запускает сервер в главном процессе asyncio.run()
        """

        self.domain = domain


        self._init_sys_routes()

        cfg = QuicConfiguration(
            alpn_protocols=["gn:backend"], is_client=False, idle_timeout=idle_timeout
        )
        cfg.load_cert_chain(cert_path, key_path) # type: ignore

        async def _main():
            
            await self.dispatchEvent('start')

            await serve(
                host,
                port,
                configuration=cfg,
                create_protocol=lambda *a, **kw: App._ServerProto(*a, api=self, **kw),
                retry=False,
            )
            
            if run is not None:
                await run()


            if wait:
                await asyncio.Event().wait()

        asyncio.run(_main())


    def runByVMHost(self):
        """
        # Запусить через VM-host

        Заупскает сервер через процесс vm-host
        """
        argv = sys.argv[1:]
        command = argv[0]
        if command == 'gn:vm-host:start':
            domain = argv[1]
            port = int(argv[2])
            cert_path = argv[3]
            key_path = argv[4]
            host = argv[5]

            self.run(
                domain=domain,
                port=port,
                cert_path=cert_path,
                key_path=key_path,
                host=host
            )
