import functools
import inspect
from typing import Any, Callable


def api_method(name: str, opts: bool = False) -> Callable:
    """
    Decorator for CoCalcAPI methods.
    Converts arguments (excluding self) into a dict, removes None values,
    and calls parent.call(name, [args_dict]).

    - name: of the api call
    - opts: if given,  
    """

    def decorator(func: Callable) -> Callable:
        sig = inspect.signature(func)

        @functools.wraps(func)
        def wrapper(self, *args, **kwargs) -> Any:
            # Bind args/kwargs to parameter names
            bound = sig.bind(self, *args, **kwargs)
            bound.apply_defaults()
            args_dict = {
                k: v
                for k, v in bound.arguments.items()
                if k != "self" and v is not None
            }
            if opts:
                args_dict = {'opts': args_dict}
            return self._parent.call(name, [args_dict])

        return wrapper

    return decorator
