#!/usr/bin/env bash
set -euo pipefail

# shellcheck disable=SC2155
SOURCE_ROOT="${VIBEGO_PACKAGE_ROOT:-$(cd "$(dirname "${BASH_SOURCE[0]}")/.." && pwd)}"

resolve_config_root() {
  local raw=""
  if [[ -n "${MASTER_CONFIG_ROOT:-}" ]]; then
    raw="$MASTER_CONFIG_ROOT"
  elif [[ -n "${VIBEGO_CONFIG_DIR:-}" ]]; then
    raw="$VIBEGO_CONFIG_DIR"
  elif [[ -n "${XDG_CONFIG_HOME:-}" ]]; then
    raw="${XDG_CONFIG_HOME%/}/vibego"
  else
    raw="$HOME/.config/vibego"
  fi
  if [[ "$raw" == ~* ]]; then
    printf '%s' "${raw/#\~/$HOME}"
  else
    printf '%s' "$raw"
  fi
}

CONFIG_ROOT="$(resolve_config_root)"
RUNTIME_ROOT="${VIBEGO_RUNTIME_ROOT:-$CONFIG_ROOT/runtime}"
VENV_DIR="$RUNTIME_ROOT/.venv"
MODELS_DIR="$SOURCE_ROOT/scripts/models"
LOG_ROOT="${LOG_ROOT:-$CONFIG_ROOT/logs}"
DEFAULT_LOG_FILE="$LOG_ROOT/run_bot.log"
LOG_FILE="${LOG_FILE:-$DEFAULT_LOG_FILE}"
MODEL_DEFAULT="${MODEL_DEFAULT:-codex}"
PROJECT_DEFAULT="${PROJECT_NAME:-}"

usage() {
  cat <<USAGE
usage:${0##*/} [--model name] [--project name] [--foreground] [--no-stop]
  --model        Start the specified model (codex|claudecode|gemini), default: $MODEL_DEFAULT
  --project      Specify project alias for logging/Conversation distinction
  --foreground   Run in foreground(for debugging), defaultBackstage
  --no-stop      Not executed before startup stop_bot.sh(defaultOld instances will be stopped first)
USAGE
}

MODEL="$MODEL_DEFAULT"
PROJECT_OVERRIDE="$PROJECT_DEFAULT"
FOREGROUND=0
NO_STOP=0

while [[ $# -gt 0 ]]; do
  case "$1" in
    --model)
      MODEL="$2"; shift 2 ;;
    --project)
      PROJECT_OVERRIDE="$2"; shift 2 ;;
    --foreground)
      FOREGROUND=1; shift ;;
    --no-stop)
      NO_STOP=1; shift ;;
    -h|--help)
      usage; exit 0 ;;
    *)
      echo "unknown parameters: $1" >&2
      usage
      exit 1 ;;
  esac
done

MODEL_SCRIPT="$MODELS_DIR/$MODEL.sh"
if [[ ! -f "$MODEL_SCRIPT" ]]; then
  echo "[run-bot] Unsupported model: $MODEL" >&2
  exit 1
fi

# Load public functions + Model configuration
# shellcheck disable=SC1090
source "$MODELS_DIR/common.sh"
# Allow model scripts to reuse environment variables
MODEL_WORKDIR="${MODEL_WORKDIR:-}" 
# shellcheck disable=SC1090
source "$MODEL_SCRIPT"
model_configure

if [[ -z "$PROJECT_OVERRIDE" ]]; then
  PROJECT_NAME="$(project_slug_from_workdir "$MODEL_WORKDIR")"
else
  PROJECT_NAME="$(sanitize_slug "$PROJECT_OVERRIDE")"
fi
LOG_DIR="$(log_dir_for "$MODEL" "$PROJECT_NAME")"
MODEL_LOG="$LOG_DIR/model.log"
RUN_LOG="$LOG_DIR/run_bot.log"
POINTER_FILE="$LOG_DIR/${MODEL_POINTER_BASENAME:-current_session.txt}"
LOCK_FILE="${SESSION_LOCK_FILE:-$LOG_DIR/${MODEL_POINTER_LOCK_BASENAME:-session_lock.json}}"
TMUX_SESSION="$(tmux_session_for "$PROJECT_NAME")"

expand_model_workdir() {
  local path="$1"
  if [[ "$path" == ~* ]]; then
    printf '%s' "${path/#\~/$HOME}"
  else
    printf '%s' "$path"
  fi
}

MODEL_WORKDIR="$(expand_model_workdir "$MODEL_WORKDIR")"

if [[ -z "$MODEL_WORKDIR" ]]; then
  echo "[run-bot] Configuration is missing MODEL_WORKDIR, Please check config/projects.json" >&2
  exit 1
fi

if [[ ! -d "$MODEL_WORKDIR" ]]; then
  echo "[run-bot] Working directory does not exist: $MODEL_WORKDIR" >&2
  exit 1
fi

if ! command -v tmux >/dev/null 2>&1; then
  echo "[run-bot] tmux not detected, pass 'brew install tmux' Install" >&2
  exit 1
fi

if [[ -n "$MODEL_CMD" ]]; then
  IFS=' ' read -r MODEL_CMD_BIN _ <<<"$MODEL_CMD"
  if [[ -n "$MODEL_CMD_BIN" ]] && ! command -v "$MODEL_CMD_BIN" >/dev/null 2>&1; then
    echo "[run-bot] Model command not detected: $MODEL_CMD_BIN" >&2
    exit 1
  fi
fi

ensure_dir "$LOG_DIR"

if (( FOREGROUND == 0 )); then
  mkdir -p "$LOG_DIR"
  CMD=("$0" --model "$MODEL" --project "$PROJECT_NAME" --foreground)
  (( NO_STOP )) && CMD+=(--no-stop)
  nohup "${CMD[@]}" >>"$RUN_LOG" 2>&1 &
  echo "[run-bot] Start in background (model=$MODEL project=$PROJECT_NAME) log: $RUN_LOG"
  exit 0
fi

if (( NO_STOP == 0 )); then
  "$SOURCE_ROOT/scripts/stop_bot.sh" --model "$MODEL" --project "$PROJECT_NAME" >/dev/null 2>&1 || true
fi

mkdir -p "$RUNTIME_ROOT"

if [[ ! -d "$VENV_DIR" ]]; then
  python3 -m venv "$VENV_DIR"
fi
# shellcheck disable=SC1091
source "$VENV_DIR/bin/activate"
REQUIREMENTS_PATH="${VIBEGO_REQUIREMENTS_PATH:-$SOURCE_ROOT/scripts/requirements.txt}"
if [[ ! -f "$REQUIREMENTS_PATH" ]]; then
  echo "[run-bot] Dependency list not found: $REQUIREMENTS_PATH" >&2
  exit 1
fi
if [[ ! -f "$VENV_DIR/.requirements.installed" ]] || [[ "$REQUIREMENTS_PATH" -nt "$VENV_DIR/.requirements.installed" ]]; then
  pip install -r "$REQUIREMENTS_PATH" >/dev/null
  touch "$VENV_DIR/.requirements.installed"
fi

export LOG_FILE="$LOG_FILE"
export ACTIVE_MODEL="$MODEL"
export MODEL_NAME="$MODEL"
export MODEL_WORKDIR
export MODEL_CMD
export MODEL_SESSION_ROOT
export MODEL_SESSION_GLOB
export SESSION_POINTER_FILE="$POINTER_FILE"
export CODEX_SESSION_FILE_PATH="$POINTER_FILE"
export SESSION_LOCK_FILE="$LOCK_FILE"
export SESSION_LOCK_FILE_PATH="$LOCK_FILE"
export TMUX_SESSION="$TMUX_SESSION"
export TMUX_LOG="$MODEL_LOG"
export PROJECT_NAME="$PROJECT_NAME"
export LOG_DIR="$LOG_DIR"
export ROOT_DIR="$SOURCE_ROOT"
if [[ -n "${CLAUDE_CODE_DISABLE_FILE_CHECKPOINTING:-}" ]]; then
  export CLAUDE_CODE_DISABLE_FILE_CHECKPOINTING
fi

"$SOURCE_ROOT/scripts/start_tmux_codex.sh" --kill >/dev/null

echo $$ > "$LOG_DIR/bot.pid"
exec python "$SOURCE_ROOT/bot.py"
