/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
import { aws_ec2 as ec2, aws_fsx } from 'aws-cdk-lib';
import { ISecret } from 'aws-cdk-lib/aws-secretsmanager';
import { Construct } from 'constructs';
import { DomainWindowsNode } from '../skylight-compute';
/**
 * The properties for the PersistentStorage class.
 */
export interface IFSxWindowsProps {
    /**
     * The Filesystem size in GB
     *
     * @default - 200.
     */
    fileSystemSize?: number;
    /**
     * The Filesystem throughput in MBps
     *
     * @default - 128.
     */
    throughputMbps?: number;
    /**
     * Choosing Single-AZ or Multi-AZ file system deployment
     * See: https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html
     * @default - true.
     */
    multiAZ?: boolean;
    /**
     * Deploy the Amazon FSx file system in private subnet or public subnet
     * See: https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html
     * @default - true.
     */
    fileSystemInPrivateSubnet?: boolean;
    /**
     * The VPC to use, must have private subnets.
     */
    vpc: ec2.IVpc;
    directoryId: string;
    ssmParameters?: IFSxWindowsParameters;
}
export interface IFSxWindowsParameters {
    /**
     * The name of the parameter to save the FSxEndpoint DNS Endpoint
     * @default - 'FSxEndpoint-DNS'.
     */
    dnsEndpoint?: string;
    /**
     * The SSM namespace to read/write parameters to
     * @default - 'cdk-skylight'.
     */
    namespace?: string;
}
/**
* A FSxWindows represents an integration pattern of Amazon FSx and Managed AD in a specific VPC.

* The Construct creates Amazon FSx for Windows
* The construct also creates (optionally) t3.nano machine that is part of the domain that can be used to run admin-tasks (such as createFolder)
*
* The createFolder() method creates an SMB Folder in the FSx filesystem, using the domain admin user.
* Please note: When calling createFolder() API, a Lambda will be created to start the worker machine (Using AWS-SDK),
* then each command will be scheduled with State Manager, and the instance will be shut down after complete .
 *
 */
export declare class FSxWindows extends Construct {
    readonly ssmParameters: IFSxWindowsParameters;
    readonly fsxObject: aws_fsx.CfnFileSystem;
    readonly props: IFSxWindowsProps;
    constructor(scope: Construct, id: string, props: IFSxWindowsProps);
    createWorker(domainName: string, domainPassword: ISecret): DomainWindowsNode;
    createFolder(worker: DomainWindowsNode, folderName: string, secretName: ISecret): void;
}
