import * as cdk from 'aws-cdk-lib';
import * as ec2 from 'aws-cdk-lib/aws-ec2';
import * as ecs from 'aws-cdk-lib/aws-ecs';
import * as elbv2 from 'aws-cdk-lib/aws-elasticloadbalancingv2';
import * as iam from 'aws-cdk-lib/aws-iam';
import * as ssm from 'aws-cdk-lib/aws-ssm';
import * as secretsmanager from 'aws-cdk-lib/aws-secretsmanager';
import * as acm from 'aws-cdk-lib/aws-certificatemanager';
import { Construct } from 'constructs';
import { EcsCodeDeploy } from 'must-cdk';

/**
 * Enhanced ECS CodeDeploy Stack demonstrating new features:
 * - Environment variables support
 * - Secrets management
 * - Custom container names
 * - Additional container configuration options
 * - Container access after creation
 * 
 * IMPORTANT SUBNET CONFIGURATION LESSON:
 * This example demonstrates the critical distinction between:
 * - taskSubnets: Where your ECS containers run (usually PRIVATE subnets)
 * - loadBalancerSubnets: Where your ALB runs (usually PUBLIC subnets for internet-facing)
 * 
 * Common mistakes to avoid:
 * 1. Using single AZ for ALB (reduces availability)
 * 2. Running containers in PUBLIC subnets (security risk)
 * 3. Confusing ALB subnets with container subnets
 * 
 * Migration from raw CDK? Check these patterns:
 * - vpc_subnets=ec2.SubnetSelection(availability_zones=["us-east-1a"]) → single AZ (bad for ALB)
 * - taskSubnets: { availabilityZones: ["us-east-1a"] } → containers in specific AZ (OK)
 * - loadBalancerSubnets: {} → ALB in all AZs (good for availability)
 */
export class EnhancedEcsCodeDeployStack extends cdk.Stack {
  constructor(scope: Construct, id: string, props?: cdk.StackProps) {
    super(scope, id, props);

    // Create VPC for ECS cluster
    const vpc = new ec2.Vpc(this, 'EcsVpc', {
      maxAzs: 3,
      natGateways: 2,
      subnetConfiguration: [
        {
          cidrMask: 24,
          name: 'Public',
          subnetType: ec2.SubnetType.PUBLIC,
        },
        {
          cidrMask: 24,
          name: 'Private',
          subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS,
        }
      ]
    });

    // Create ECS cluster
    const cluster = new ecs.Cluster(this, 'EcsCluster', {
      vpc,
      clusterName: 'enhanced-cluster',
      containerInsights: true
    });

    // Create security group
    const securityGroup = new ec2.SecurityGroup(this, 'EcsSecurityGroup', {
      vpc,
      description: 'Security group for ECS service',
      allowAllOutbound: true
    });

    securityGroup.addIngressRule(
      ec2.Peer.anyIpv4(),
      ec2.Port.tcp(80),
      'Allow HTTP traffic from ALB'
    );

    securityGroup.addIngressRule(
      ec2.Peer.anyIpv4(),
      ec2.Port.tcp(443),
      'Allow HTTPS traffic from ALB'
    );

    // Create SSL certificate (replace with your domain)
    const certificate = new acm.Certificate(this, 'Certificate', {
      domainName: 'example.com',
      validation: acm.CertificateValidation.fromDns(),
    });

    // Create secrets for sensitive data
    const dbSecret = new secretsmanager.Secret(this, 'DatabaseSecret', {
      description: 'Database connection credentials',
      generateSecretString: {
        secretStringTemplate: JSON.stringify({ username: 'admin' }),
        generateStringKey: 'password',
        excludeCharacters: '"@/\\'
      }
    });

    // Create SSM parameters for configuration
    const apiKeyParam = new ssm.StringParameter(this, 'ApiKeyParam', {
      parameterName: '/myapp/api-key',
      stringValue: 'your-api-key-here',
      description: 'API key for external service'
    });

    // Enhanced ECS service using native CDK interfaces
    const ecsService = new EcsCodeDeploy(this, 'EnhancedWebApiService', {
      vpc,
      cluster,
      serviceName: 'enhanced-web-api',
      securityGroups: [securityGroup],
      
      // IMPORTANT: Subnet configuration - these control DIFFERENT resources!
      taskSubnets: {
        // ECS containers run in PRIVATE subnets for security
        subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS
      },
      loadBalancerSubnets: {
        // ALB runs in PUBLIC subnets for internet access
        // Using all AZs for high availability (NOT single AZ!)
        subnetType: ec2.SubnetType.PUBLIC
      },
      
      // Use native CDK task definition props
      taskDefinition: {
        cpu: 1024,
        memoryLimitMiB: 2048,
      },
      
      // Use native CDK service props  
      service: {
        desiredCount: 2,
        assignPublicIp: false,
        healthCheckGracePeriod: cdk.Duration.seconds(300),
      },
      
      // Use native CDK load balancer props
      loadBalancer: {
        internetFacing: true,
        // NOTE: loadBalancer.vpcSubnets is DEPRECATED - use top-level loadBalancerSubnets instead!
        // vpcSubnets: { subnetType: ec2.SubnetType.PUBLIC }, // DON'T use this
      },
      
      // Production listener configuration  
      productionListener: {
        port: 443,
        protocol: elbv2.ApplicationProtocol.HTTPS,
        certificates: [certificate],
      },
      
      // Test listener configuration
      testListener: {
        port: 8080,
        protocol: elbv2.ApplicationProtocol.HTTP,
      },
      
      // Production target group configuration
      productionTargetGroup: {
        healthCheck: {
          path: '/health',
          interval: cdk.Duration.seconds(30),
          timeout: cdk.Duration.seconds(10),
          healthyThresholdCount: 2,
          unhealthyThresholdCount: 5,
          healthyHttpCodes: '200'
        },
        stickinessCookieDuration: cdk.Duration.days(1),
      },
      
      // Test target group configuration
      testTargetGroup: {
        healthCheck: {
          path: '/health',
          interval: cdk.Duration.seconds(30),
          timeout: cdk.Duration.seconds(10),
          healthyThresholdCount: 2,
          unhealthyThresholdCount: 5,
          healthyHttpCodes: '200'
        }
      },
      
      // Enhanced container configuration using native CDK interfaces
      containers: [
        {
          name: 'web-api-container',
          options: {
            image: ecs.ContainerImage.fromRegistry('nginx:latest'),
            portMappings: [{ containerPort: 80 }],
            memoryReservationMiB: 512,
            cpu: 512,
            essential: true,
            
            // All native CDK container options available
            environment: {
              NODE_ENV: 'production',
              PORT: '80',
              LOG_LEVEL: 'info',
              APP_VERSION: '1.0.0',
              REGION: cdk.Stack.of(this).region,
              DB_TABLE_NAME: `${cdk.Stack.of(this).stackName}-users`,
              CACHE_TTL: '3600'
            },
            
            secrets: {
              DB_PASSWORD: ecs.Secret.fromSecretsManager(dbSecret, 'password'),
              DB_USERNAME: ecs.Secret.fromSecretsManager(dbSecret, 'username'),
              API_KEY: ecs.Secret.fromSsmParameter(apiKeyParam)
            },
            
            workingDirectory: '/app',
            user: 'nginx',
            entryPoint: ['/docker-entrypoint.sh'],
            command: ['nginx', '-g', 'daemon off;'],
            
            healthCheck: {
              command: [
                'CMD-SHELL',
                'curl -f http://localhost:80/health || exit 1'
              ],
              interval: cdk.Duration.seconds(30),
              timeout: cdk.Duration.seconds(5),
              retries: 3,
              startPeriod: cdk.Duration.seconds(60)
            }
          }
        },
        
        {
          name: 'log-router',
          options: {
            image: ecs.ContainerImage.fromRegistry('fluent/fluent-bit:latest'),
            portMappings: [{ containerPort: 24224 }],
            memoryReservationMiB: 256,
            cpu: 256,
            essential: false,
            
            environment: {
              FLB_LOG_LEVEL: 'info',
              AWS_REGION: cdk.Stack.of(this).region
            },
            
            workingDirectory: '/fluent-bit',
            command: ['/fluent-bit/bin/fluent-bit', '--config=/fluent-bit/etc/fluent-bit.conf']
          }
        }
      ],

      // Tags for resource management
      // Note: Tags also support multiple formats via TAGS environment variable:
      // 1. Key-value pairs: TAGS=Product=Mufin,Owner=Platform,Environment=production
      // 2. JSON string: TAGS='{"Product":"Mufin","Owner":"Platform","Environment":"production"}'
      // 3. JSON object: TAGS={"Product":"Mufin","Owner":"Platform","Environment":"production"}
      // Environment tags take precedence over these user-provided tags
      tags: {
        Environment: 'production',
        Application: 'web-api',
        Team: 'platform',
        CostCenter: 'engineering'
      }
    });

    // Demonstrate accessing containers after creation (new feature!)
    const webApiContainer = ecsService.containers[0];
    const logRouterContainer = ecsService.containers[1];

    // Grant additional permissions to the web API container's task role
    ecsService.taskRole.addToPolicy(new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        's3:GetObject',
        's3:PutObject'
      ],
      resources: ['arn:aws:s3:::my-app-bucket/*']
    }));

    // Output important values
    new cdk.CfnOutput(this, 'LoadBalancerDnsName', {
      value: ecsService.loadBalancerDnsName,
      description: 'Load Balancer DNS Name'
    });

    new cdk.CfnOutput(this, 'ServiceArn', {
      value: ecsService.serviceArn,
      description: 'ECS Service ARN'
    });

    new cdk.CfnOutput(this, 'WebApiContainerName', {
      value: webApiContainer.containerName,
      description: 'Web API Container Name'
    });

    new cdk.CfnOutput(this, 'LogRouterContainerName', {
      value: logRouterContainer.containerName,
      description: 'Log Router Container Name'
    });

    new cdk.CfnOutput(this, 'ProductionTargetGroupArn', {
      value: ecsService.productionTargetGroup.targetGroupArn,
      description: 'Production Target Group ARN'
    });

    new cdk.CfnOutput(this, 'TestTargetGroupArn', {
      value: ecsService.testTargetGroup.targetGroupArn,
      description: 'Test Target Group ARN'
    });
  }
}

// Example usage in app.ts:
// const app = new cdk.App();
// new EnhancedEcsCodeDeployStack(app, 'EnhancedEcsCodeDeployStack', {
//   env: {
//     account: process.env.CDK_DEFAULT_ACCOUNT,
//     region: process.env.CDK_DEFAULT_REGION,
//   },
// });
