"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const assert_1 = require("@aws-cdk/assert");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const sns = require("@aws-cdk/aws-sns");
const cdk = require("@aws-cdk/core");
const chatbot = require("../lib");
describe('SlackChannelConfiguration', () => {
    let stack;
    beforeEach(() => {
        stack = new cdk.Stack();
    });
    test('created with minimal properties creates a new IAM Role', () => {
        new chatbot.SlackChannelConfiguration(stack, 'MySlackChannel', {
            slackWorkspaceId: 'ABC123',
            slackChannelId: 'DEF456',
            slackChannelConfigurationName: 'Test',
        });
        expect(stack).toHaveResourceLike('AWS::Chatbot::SlackChannelConfiguration', {
            ConfigurationName: 'Test',
            IamRoleArn: {
                'Fn::GetAtt': [
                    'MySlackChannelConfigurationRole1D3F23AE',
                    'Arn',
                ],
            },
            SlackChannelId: 'DEF456',
            SlackWorkspaceId: 'ABC123',
        });
        expect(stack).toHaveResourceLike('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: 'sts:AssumeRole',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'chatbot.amazonaws.com',
                        },
                    },
                ],
                Version: '2012-10-17',
            },
        });
    });
    test('created and pass loggingLevel parameter [LoggingLevel.ERROR], it should be set [ERROR] logging level in Cloudformation', () => {
        new chatbot.SlackChannelConfiguration(stack, 'MySlackChannel', {
            slackWorkspaceId: 'ABC123',
            slackChannelId: 'DEF456',
            slackChannelConfigurationName: 'Test',
            loggingLevel: chatbot.LoggingLevel.ERROR,
        });
        expect(stack).toHaveResourceLike('AWS::Chatbot::SlackChannelConfiguration', {
            ConfigurationName: 'Test',
            IamRoleArn: {
                'Fn::GetAtt': [
                    'MySlackChannelConfigurationRole1D3F23AE',
                    'Arn',
                ],
            },
            SlackChannelId: 'DEF456',
            SlackWorkspaceId: 'ABC123',
            LoggingLevel: 'ERROR',
        });
    });
    test('created with new sns topic', () => {
        const topic = new sns.Topic(stack, 'MyTopic');
        new chatbot.SlackChannelConfiguration(stack, 'MySlackChannel', {
            slackWorkspaceId: 'ABC123',
            slackChannelId: 'DEF456',
            slackChannelConfigurationName: 'Test',
            notificationTopics: [topic],
        });
        expect(stack).toHaveResourceLike('AWS::Chatbot::SlackChannelConfiguration', {
            ConfigurationName: 'Test',
            IamRoleArn: {
                'Fn::GetAtt': [
                    'MySlackChannelConfigurationRole1D3F23AE',
                    'Arn',
                ],
            },
            SlackChannelId: 'DEF456',
            SlackWorkspaceId: 'ABC123',
            SnsTopicArns: [
                {
                    Ref: 'MyTopic86869434',
                },
            ],
        });
    });
    test('created with existing role', () => {
        const role = iam.Role.fromRoleArn(stack, 'Role', 'arn:aws:iam:::role/test-role');
        new chatbot.SlackChannelConfiguration(stack, 'MySlackChannel', {
            slackWorkspaceId: 'ABC123',
            slackChannelId: 'DEF456',
            slackChannelConfigurationName: 'Test',
            role: role,
        });
        expect(stack).toCountResources('AWS::IAM::Role', 0);
    });
    test('created with new role and extra iam policies', () => {
        const slackChannel = new chatbot.SlackChannelConfiguration(stack, 'MySlackChannel', {
            slackWorkspaceId: 'ABC123',
            slackChannelId: 'DEF456',
            slackChannelConfigurationName: 'Test',
        });
        slackChannel.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                's3:GetObject',
            ],
            resources: ['arn:aws:s3:::abc/xyz/123.txt'],
        }));
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 's3:GetObject',
                        Effect: 'Allow',
                        Resource: 'arn:aws:s3:::abc/xyz/123.txt',
                    },
                ],
                Version: '2012-10-17',
            },
        });
    });
    test('specifying log retention', () => {
        new chatbot.SlackChannelConfiguration(stack, 'MySlackChannel', {
            slackWorkspaceId: 'ABC123',
            slackChannelId: 'DEF456',
            slackChannelConfigurationName: 'ConfigurationName',
            logRetention: logs.RetentionDays.ONE_MONTH,
        });
        expect(stack).toHaveResourceLike('Custom::LogRetention', {
            LogGroupName: '/aws/chatbot/ConfigurationName',
            RetentionInDays: 30,
            LogGroupRegion: 'us-east-1',
        });
    });
    test('getting configuration metric', () => {
        const slackChannel = new chatbot.SlackChannelConfiguration(stack, 'MySlackChannel', {
            slackWorkspaceId: 'ABC123',
            slackChannelId: 'DEF456',
            slackChannelConfigurationName: 'ConfigurationName',
            logRetention: logs.RetentionDays.ONE_MONTH,
        });
        const metric = slackChannel.metric('MetricName');
        new cloudwatch.Alarm(stack, 'Alarm', {
            evaluationPeriods: 1,
            threshold: 0,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            metric: metric,
        });
        expect(metric).toEqual(new cloudwatch.Metric({
            namespace: 'AWS/Chatbot',
            region: 'us-east-1',
            dimensions: {
                ConfigurationName: 'ConfigurationName',
            },
            metricName: 'MetricName',
        }));
        expect(stack).toHaveResourceLike('AWS::CloudWatch::Alarm', {
            Namespace: 'AWS/Chatbot',
            MetricName: 'MetricName',
            Dimensions: [
                {
                    Name: 'ConfigurationName',
                    Value: 'ConfigurationName',
                },
            ],
            ComparisonOperator: 'GreaterThanThreshold',
            EvaluationPeriods: 1,
            Threshold: 0,
        });
    });
    test('getting all configurations metric', () => {
        const metric = chatbot.SlackChannelConfiguration.metricAll('MetricName');
        new cloudwatch.Alarm(stack, 'Alarm', {
            evaluationPeriods: 1,
            threshold: 0,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            metric: metric,
        });
        expect(metric).toEqual(new cloudwatch.Metric({
            namespace: 'AWS/Chatbot',
            region: 'us-east-1',
            metricName: 'MetricName',
        }));
        expect(stack).toHaveResourceLike('AWS::CloudWatch::Alarm', {
            Namespace: 'AWS/Chatbot',
            MetricName: 'MetricName',
            Dimensions: assert_1.ABSENT,
            ComparisonOperator: 'GreaterThanThreshold',
            EvaluationPeriods: 1,
            Threshold: 0,
        });
    });
    test('added a iam policy to a from slack channel configuration ARN will nothing to do', () => {
        const imported = chatbot.SlackChannelConfiguration.fromSlackChannelConfigurationArn(stack, 'MySlackChannel', 'arn:aws:chatbot::1234567890:chat-configuration/slack-channel/my-slack');
        imported.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                's3:GetObject',
            ],
            resources: ['arn:aws:s3:::abc/xyz/123.txt'],
        }));
        expect(stack).toCountResources('AWS::IAM::Role', 0);
        expect(stack).toCountResources('AWS::IAM::Policy', 0);
    });
    test('should throw error if ARN invalid', () => {
        expect(() => chatbot.SlackChannelConfiguration.fromSlackChannelConfigurationArn(stack, 'MySlackChannel', 'arn:aws:chatbot::1234567890:chat-configuration/my-slack')).toThrow(/The ARN of a Slack integration must be in the form: arn:aws:chatbot:{region}:{account}:chat-configuration\/slack-channel\/{slackChannelName}/);
    });
    test('from slack channel configuration ARN', () => {
        const imported = chatbot.SlackChannelConfiguration.fromSlackChannelConfigurationArn(stack, 'MySlackChannel', 'arn:aws:chatbot::1234567890:chat-configuration/slack-channel/my-slack');
        expect(imported.slackChannelConfigurationName).toEqual('my-slack');
        expect(imported.slackChannelConfigurationArn).toEqual('arn:aws:chatbot::1234567890:chat-configuration/slack-channel/my-slack');
    });
});
//# sourceMappingURL=data:application/json;base64,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