Non-official test suite inherited from https://github.com/Hecate2/neo-ruler/ , in Python.

Features comply with https://github.com/Hecate2/neo-rpc-server-with-session/ .

A crude JavaScript version is available at https://github.com/Hecate2/neo-fairy-gate/blob/master/src/libs/NeoFairyClient.jsx .

#### Tutorial for testing

**Python >= 3.8 required!** Some steps in this tutorial is to help you understand the details about how Fairy works. In actual combat, you can read the source codes of `FairyClient` and enjoy many automatic conveniences that Fairy offers. 

##### Extremely fast but close-to-base version:

Visit [test_nftloan.py](test_nftloan.py) as a sample of usage. The tested contract can be found at https://github.com/Hecate2/NFTLoan . Contract `AnyUpdateShortSafe` is an old-fashioned contract for testing, deployed on testnet T4, with source codes at https://github.com/Hecate2/AnyUpdate/ . You can skip using `AnyUpdate` by calling the RPC method `virtualdeploy`. 

##### Step 1: Run a neo-cli with Fairy plugin!

Head to https://github.com/Hecate2/neo-fairy-test/ to prepare it. You do not really have to wait for the blocks to be completely synchronized. The plugin is an HTTP server that will help you interact with Neo.

##### Step 2: Using your client, prepare your server snapshot

Place a json file of neo wallet (assumed to be `testnet.json` with password `1`) beside `neo-cli.exe`, and call your Fairy server with the following Python codes:

```python
from neo_fairy_client.rpc import FairyClient
from neo_fairy_client.utils import Hash160Str
target_url = 'http://127.0.0.1:16868'
wallet_address = 'Nb2CHYY5wTh2ac58mTue5S3wpG6bQv5hSY'
wallet_scripthash = Hash160Str.from_address(wallet_address)
wallet_path = 'testnet.json'
wallet_password = '1'
client = FairyClient(fairy_session='Hello world! Your first contact with Fairy!',
            wallet_address_or_scripthash=wallet_address,
            wallet_path=wallet_path, wallet_password=wallet_password,
            auto_preparation=True)
```

Here `auto_preparation=True` tries to delete the old snapshot on the Fairy server named `Hello world! Your first contact with Fairy!`, and creates a new snapshot of the same name based on the current Neo system snapshot, then opens the wallet on Fairy server and automatically sets you NEO and GAS balance both to 100 (*10^8). 

If you are planning to run a public Fairy server, you need to open the Fairy wallet so that users do not have to open it through RPC. I am also planning to remove wallet objects in Fairy service. 

#### Step 2.1: Mint billions of NEO and transfer them!

(Of course these are just fairy NEO in the memory of your imaginiation)

```python
from neo_fairy_client.utils import neo
client.set_neo_balance(1_000_000_000)
print(f"Your NEO balance: {client.invokefunction_of_any_contract(neo.hash, 'balanceOf', [wallet_scripthash])}")
client.invokefunction_of_any_contract(neo.hash, 'transfer', [wallet_scripthash, Hash160Str.zero(), 1_000_000_000, None])
print(f"NEO balance of zero address: {client.invokefunction_of_any_contract(neo.hash, 'balanceOf', [Hash160Str.zero()])}")
```

```
Hello world! Your first contact with Fairy!::balanceOf[0xb1983fa2479a0c8e2beae032d2df564b5451b7a5] relay=None [{'account': '0xb1983fa2479a0c8e2beae032d2df564b5451b7a5', 'scopes': 'CalledByEntry', 'allowedcontracts': [], 'allowedgroups': [], 'rules': []}]
Your NEO balance: 1000000000
Hello world! Your first contact with Fairy!::transfer[0xb1983fa2479a0c8e2beae032d2df564b5451b7a5, 0x0000000000000000000000000000000000000000, 1000000000, None] relay=None [{'account': '0xb1983fa2479a0c8e2beae032d2df564b5451b7a5', 'scopes': 'CalledByEntry', 'allowedcontracts': [], 'allowedgroups': [], 'rules': []}]
Hello world! Your first contact with Fairy!::balanceOf[0x0000000000000000000000000000000000000000] relay=None [{'account': '0xb1983fa2479a0c8e2beae032d2df564b5451b7a5', 'scopes': 'CalledByEntry', 'allowedcontracts': [], 'allowedgroups': [], 'rules': []}]
NEO balance of zero address: 1000000000
```

##### Step 2.2: I just want to interact with the real mainnet and testnet...

**DO NOT** set the `fairy_session` string for you `FairyClient`, or set it to `None`. Fairy will play real transactions without fairy session. Set `function_default_relay=True` in `FairyClient` or `relay=True` in a single `invokefunction` to automatically relay the transaction. 

**BE CAREFUL: ** By default, Fairy does interact with the real blockchain and relay transactions. **Do not use a wallet with real assets when you just want a test!**

Sometimes you may want to actually relay something after fairy tests. In such cases, set `confirm_relay_to_blockchain=True` in `FairyClient` to prevent automatic relaying as the final safety belt. 

##### Step 3: Deploy your contract virtually

Get the tested contracts in my example through these repos:

https://github.com/Hecate2/AnyUpdate

https://github.com/Hecate2/NFTLoan/

and place them properly.

```python
nef_file, manifest = client.get_nef_and_manifest_from_path('../NFTLoan/NFTLoan/bin/sc/NFTFlashLoan.nef')
test_nopht_d_hash = client.virutal_deploy_from_path('../NFTLoan/NophtD/bin/sc/TestNophtD.nef')
anyupdate_short_safe_hash = client.virutal_deploy_from_path('../AnyUpdate/AnyUpdateShortSafe.nef')
```

`client.virutal_deploy_from_path` deploys the `.nef` and `.manifest.json` to the snapshot of your Fairy session. The snapshot is similar to a fork of the current blockchain, named by your session string. You can now access to the deployed contract through your snapshot, but not through the actual blockchain. 

In our case, we deployed `AnyUpdate` to be updated to any other contract, and `test_nopht_d` as a divisible NFT to be operated. Though you can continue deploying `NFTLoan` by yourself, we are now going to call `AnyUpdate` to perform all the actions the same as `NFTLoan`.

##### Step 4: Call your contracts!

By design, `NFTLoan` initializes its token ID to be 1. However, this is not performed by `AnyUpdate`. Therefore, we first ask `AnyUpdate` to prepare the storage environment:

```python
client.invokefunction('putStorage', params=[0x02, 1])
```

Here we did not explicitly indicate the address of the called contract. This is because `client.virutal_deploy_from_path` has set `client.contract_scripthash` to be the address of the just deployed contract in the previous step.

Also notice that we should always put some string in `client.fairy_session`. If `fairy_session` is set to `None`, the client will (by my design) directly interact with the real blockchain, and write real transactions. 

##### Step 5: The storage written by Fairy is always valid in the same snapshot

```python
import json
manifest_dict = json.loads(manifest)
manifest_dict['name'] = 'AnyUpdateShortSafe'
manifest = json.dumps(manifest_dict, separators=(',', ':'))
print(
    client.invokefunction('anyUpdate', params=
    [nef_file, manifest, 'registerRental',
     [wallet_scripthash, test_nopht_d_hash, 68, 1, 5, 7, True]
    ]
    )
)
```

Extremely complex, huh? Not really. 

In the first 4 lines we are changing the contract name of `NFTLoan` in its manifest. This is because the contract cannot change its name in an update. Then we are just calling `AnyUpdate` to update itself becoming our `NFTLoan`, and execute the method `registerRental`. 

And **happily we will see a lot of red alerts** ending with:

```
[0x5c1068339fae89eb1a743909d0213e1d99dc5dc9] AnyUpdateShortSafe: Transfer failed
```

**W**hiskey **T**ango **F**oxtrot? Well, by reading the codes, we can assume that we have forgotten to add proper witnesses (in other words, signatures) to our call (we are not going to explain how to make the assumption for now). But how to add signatures?

##### Step 6: Adding signatures to your call

Signatures are important elements in Neo blockchain to check whether the operation is really allowed by stakeholders. In smart contracts, you should always check the witness of the token holder before transferring his/her tokens to someone else. 

```python
import json
manifest_dict = json.loads(manifest)
manifest_dict['name'] = 'AnyUpdateShortSafe'
manifest = json.dumps(manifest_dict, separators=(',', ':'))
from neo_fairy_client.utils import Signer, WitnessScope
signer = Signer(wallet_scripthash, scopes=WitnessScope.Global)  # watch this!
print(
    client.invokefunction('anyUpdate', params=
    [nef_file, manifest, 'registerRental',
     [wallet_scripthash, test_nopht_d_hash, 68, 1, 5, 7, True]
    ], 
    signers=signer)  # watch this!
)
```

For testing purposes, you can just use `WitnessScope.Global` to allow any contract the transfer the assets of `wallet_scripthash` freely. A good news is that Fairy does not actually check if your signatures are really signed by the wallet owner. You can use any scripthash in `Signer` **(does not have to be the scripthash of the wallet)**, and Fairy will always recognize it to be a valid signature. 

If everything goes well, your Fairy client should print:

```
Hello world! Your first contact with Fairy!::putStorage[2, 1] relay=True [{'account': '0xb1983fa2479a0c8e2beae032d2df564b5451b7a5', 'scopes': 'CalledByEntry', 'allowedcontracts': [], 'allowedgroups': [], 'rules': []}]
Hello world! Your first contact with Fairy!::anyUpdate[b'NEF3Neo.Compiler.CSharp 3.1.0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05\xc0\xef9\xce\xe0\xe4\xe9%\xc6\xc2\xa0jy\xe1D\r\xd8o\xce\xac\x0bdeserialize\x01\x00\x01\x0f\xfd\xa3\xfaCF\xeaS*%\x8f\xc4\x97\xdd\xad\xdbd7\xc9\xfd\xff\x0bgetContract\x01\x00\x01\x0f\x1b\xf5u\xab\x11\x89h\x84\x13a\n5\xa1(\x86\xcd\xe0\xb6lr\x06sha256\x01\x00\x01\x0f\xc0\xef9\xce\xe0\xe4\xe9%\xc6\xc2\xa0jy\xe1D\r\xd8o\xce\xac\tserialize\x01\x00\x01\x0f\xc0\xef9\xce\xe0\xe4\xe9%\xc6\xc2\xa0jy\xe1D\r\xd8o\xce\xac\x04itoa\x01\x00\x01\x0f\x00\x00\xfdF!W\x00\x01\x0c\x08NEPHRENT@W\x00\x01x4\x03@W\x00\x01x4\x03@W\x00\x01x4\x03@W\x00\x01@W\x00\x01\x00d@\x0c\x01\x00\xdb0A\x9b\xf6g\xceA\x92]\xe81J\xd8&\x04E\x10\xdb!@J\xd8&\x04E\x10\xdb!@A\x92]\xe81@A\x9b\xf6g\xce@W\x01\x01xph\x0b\x97&\x07\x11\xdb "\rxJ\xd9(P\xca\x00\x14\xb3\xab\xaa&%\x0c The argument "owner" is invalid.:A\x9b\xf6g\xce\x11\x11\x88N\x10Q\xd0P\x12\xc0phxK\x11\xceP\x8bP\x10\xceA\x92]\xe81J\xd8&\x04E\x10\xdb!"\x02@J\xd9(P\xca\x00\x14\xb3\xab@\x11\x88N\x10Q\xd0P\x12\xc0@K\x11\xceP\x8bP\x10\xceA\x92]\xe81@W\x03\x01A\x9b\xf6g\xcep\x0c\x01\x00\xdb0qihA\x92]\xe81J\xd8&\x04E\x10\xdb!rjx\x9eJrEjihA\xe6?\x18\x84@A\xe6?\x18\x84@W\x02\x02A\x9b\xf6g\xce\x11\x11\x88N\x10Q\xd0P\x12\xc0phxK\x11\xceP\x8bP\x10\xceA\x92]\xe81J\xd8&\x04E\x10\xdb!qiy\x9eJqEi\x10\xb5&\x07\x10\xdb ".i\x10\xb3&\x13hxK\x11\xceP\x8bP\x10\xceA/X\xc5\xed"\x13hxi\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84\x11\xdb "\x02@K\x11\xceP\x8bP\x10\xceA/X\xc5\xed@\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84@\x0c\x01\x00\xdb0A\x9b\xf6g\xceA\x92]\xe81J\xd8&\x04E\x10\xdb!@W\x00\x01x\xca\x00@\xb7&\x18\x0c\x13tokenId.Length > 64:\x13x\xca\xdb(x\x8b\xdb(A\x9b\xf6g\xce\x15\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a"\x02@\xca@N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@\xdb(@W\x01\x01xph\x0b\x97&\x07\x11\xdb "\rxJ\xd9(P\xca\x00\x14\xb3\xab\xaa&%\x0c The argument "owner" is invalid.:A\x9b\xf6g\xce\x11\x11\x88N\x10Q\xd0P\x12\xc0phxK\x11\xceP\x8bP\x10\xceA\x92]\xe81J\xd8&\x04E\x10\xdb!"\x02@W\x00\x02xJ\xd9(P\xca\x00\x14\xb3\xab\xaa&$\x0c\x1fThe argument "owner" is invalid:y\xca\x00@\xb7&\x18\x0c\x13tokenId.Length > 64:A\x9b\xf6g\xce\x14\x11\x88N\x10Q\xd0P\x12\xc0xy\x8b\xdb(K\x11\xceP\x8bP\x10\xceA\x92]\xe81J\xd8&\x04E\x10\xdb!"\x02@K\x11\xceP\x8bP\x10\xceA\x92]\xe81@W\x03\x02A\x9b\xf6g\xce\x13\x11\x88N\x10Q\xd0P\x12\xc0phyK\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00q\xc8ri\x10\xceJ\x0c\x04namejS\xd0Ej"\x02@7\x00\x00@\xc8@\xd0@W\x01\x00A\x9b\xf6g\xce\x13\x11\x88N\x10Q\xd0P\x12\xc0p\x13hJ\x11\xceP\x10\xceA\xdf0\xb8\x9a"\x02@J\x11\xceP\x10\xceA\xdf0\xb8\x9a@W\x01\x01xph\x0b\x97&\x07\x11\xdb "\rxJ\xd9(P\xca\x00\x14\xb3\xab\xaa&$\x0c\x1fThe argument "owner" is invalid:A\x9b\xf6g\xce\x14\x11\x88N\x10Q\xd0P\x12\xc0p\x13xhN\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a"\x02@W\x01\x05xA\xf8\'\xec\x8c\xaa&\n\x10\xdb #\x83\x00\x00\x00yph\x0b\x97&\x07\x11\xdb "\ryJ\xd9(P\xca\x00\x14\xb3\xab\xaa&"\x0c\x1dThe argument "to" is invalid.:z\x10\xb5&\x0f\x0c\namount < 0:xy\x98$\x07\x10\xdb "\x08z\x9b{x4,$\x07\x10\xdb "\x07z{y4 &\r|{yx5!\x01\x00\x00"\x07\x10\xdb "\x07\x11\xdb "\x02@A\xf8\'\xec\x8c@W\x07\x03A\x9b\xf6g\xce\x11\x11\x88N\x10Q\xd0P\x12\xc0phxK\x11\xceP\x8bP\x10\xceA\x92]\xe81J\xd8&\x04E\x10\xdb!qiz\x9eJqEi\x10\xb5&\n\x10\xdb #\xcf\x00\x00\x00A\x9b\xf6g\xce\x14\x11\x88N\x10Q\xd0P\x12\xc0rA\x9b\xf6g\xce\x15\x11\x88N\x10Q\xd0P\x12\xc0sxy\x8b\xdb(ty\xca\xdb(y\x8b\xdb(x\x8b\xdb(ujlK\x11\xceP\x8bP\x10\xceA\x92]\xe81J\xd8&\x04E\x10\xdb!z\x9evn\x10\xb5&\x07\x10\xdb "ui\x10\xb3&\x13hxK\x11\xceP\x8bP\x10\xceA/X\xc5\xed"\x13hxi\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84n\x10\xb7&&jln\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84kmn\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84" jlK\x11\xceP\x8bP\x10\xceA/X\xc5\xedkmK\x11\xceP\x8bP\x10\xceA/X\xc5\xed\x11\xdb "\x02@W\x01\x04\xc2Jx\xcfJy\xcfJ\x11\xcfJz\xcf\x0c\x08TransferA\x95\x01oayph\x0b\x97\xaa$\x07\x10\xdb "\x0by7\x01\x00ph\x0b\x97\xaa& {z\x11x\x14\xc0\x1f\x0c\x0eonNEP11PaymentyAb}[RE@7\x01\x00@Ab}[R@W\x05\x00A\x9b\xf6g\xcep\x0c\x01\x02\xdb0qihA\x92]\xe81rjJ\xd8&\x04E\x10\xdb!\x11\x9eihA\xe6?\x18\x84A\xdb\xfe\xa8tsjtl\x0b\x97\xaa&\x08kj\x9eJsEk7\x02\x00"\x02@A\xdb\xfe\xa8t@7\x02\x00@W\x01\x05y\x10\xb6&\x15\x0c\x10mint amount <= 0:A\x9b\xf6g\xce\x13\x11\x88N\x10Q\xd0P\x12\xc0p{7\x03\x00JzhS\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84Eyzx5\x04\xfe\xff\xffEy4\x1f|zx\x0b5\x00\xff\xff\xff@7\x03\x00@\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84@W\x03\x01A\x9b\xf6g\xcep\x0c\x01\x00\xdb0qihA\x92]\xe81J\xd8&\x04E\x10\xdb!rjx\x9eJrEjihA\xe6?\x18\x84@W\x00\x03y\x10\xb6&\x15\x0c\x10burn amount <= 0:y\x9bzx5\x91\xfd\xff\xffEy\x9b4\xab\x0bz\x0bx5\x8c\xfe\xff\xff@W\x07\x03A\x9b\xf6g\xce\x11\x11\x88N\x10Q\xd0P\x12\xc0phxK\x11\xceP\x8bP\x10\xceA\x92]\xe81J\xd8&\x04E\x10\xdb!qiz\x9eJqEi\x10\xb5&\n\x10\xdb #\xcf\x00\x00\x00A\x9b\xf6g\xce\x14\x11\x88N\x10Q\xd0P\x12\xc0rA\x9b\xf6g\xce\x15\x11\x88N\x10Q\xd0P\x12\xc0sxy\x8b\xdb(ty\xca\xdb(y\x8b\xdb(x\x8b\xdb(ujlK\x11\xceP\x8bP\x10\xceA\x92]\xe81J\xd8&\x04E\x10\xdb!z\x9evn\x10\xb5&\x07\x10\xdb "ui\x10\xb3&\x13hxK\x11\xceP\x8bP\x10\xceA/X\xc5\xed"\x13hxi\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84n\x10\xb7&&jln\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84kmn\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84" jlK\x11\xceP\x8bP\x10\xceA/X\xc5\xedkmK\x11\xceP\x8bP\x10\xceA/X\xc5\xed\x11\xdb "\x02@W\x03\x01A\x9b\xf6g\xcep\x0c\x01\x00\xdb0qihA\x92]\xe81J\xd8&\x04E\x10\xdb!rjx\x9eJrEjihA\xe6?\x18\x84@W\x01\x04\xc2Jx\xcfJy\xcfJ\x11\xcfJz\xcf\x0c\x08TransferA\x95\x01oayph\x0b\x97\xaa$\x07\x10\xdb "\x0by7\x01\x00ph\x0b\x97\xaa& {z\x11x\x14\xc0\x1f\x0c\x0eonNEP11PaymentyAb}[RE@W\x00\x02y&\x04"\x12\x11\x0c\x01\x02\xdb0A\x9b\xf6g\xceA\xe6?\x18\x84@W\x00\x04\x0c-Do not send NFTs directly into this contract!{\x0c\x08NEPHRENT\x974\x03@W\x00\x02x&\x04"\x0cyA\xcf\xe7G\x96\x10\xdb 9@A\xcf\xe7G\x96@9@W\x00\x03\x0c/Do not send tokens directly into this contract!z\x0c\x08NEPHRENT\x974\xa5@W\x00\x01\xc2\x11\x0c\x08decimalsxAb}[R@W\x00\x01xA\xdb\xfe\xa8t5%\xf9\xff\xff@\x10A\x9b\xf6g\xce\x00r\x11\x88N\x10Q\xd0P\x12\xc0J\x11\xceP\x10\xceA\xdf0\xb8\x9a@W\x00\x01\x12xA\x9b\xf6g\xce\x00r\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x02\x12xy\xca\xdb(\x8b\xdb(y\x8b\xdb(A\x9b\xf6g\xce\x00r\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x03A\x9b\xf6g\xce\x00r\x11\x88N\x10Q\xd0P\x12\xc0xy\xca\xdb(\x8b\xdb(y\x8b\xdb(z\x8b\xdb(K\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00@\x10A\x9b\xf6g\xce\x00o\x11\x88N\x10Q\xd0P\x12\xc0J\x11\xceP\x10\xceA\xdf0\xb8\x9a@W\x00\x01\x12xA\x9b\xf6g\xce\x00o\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x02\x12xy\x8b\xdb(A\x9b\xf6g\xce\x00o\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x03A\x9b\xf6g\xce\x00o\x11\x88N\x10Q\xd0P\x12\xc0xy\x8b\xdb(z\x8b\xdb(K\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00@\x12A\x9b\xf6g\xce\x00d\x11\x88N\x10Q\xd0P\x12\xc0J\x11\xceP\x10\xceA\xdf0\xb8\x9a@W\x00\x01\x12xA\x9b\xf6g\xce\x00d\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x02\x12xy\xca\xdb(\x8b\xdb(y\x8b\xdb(A\x9b\xf6g\xce\x00d\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x03\x12xy\xca\xdb(\x8b\xdb(y\x8b\xdb(z\x8b\xdb(A\x9b\xf6g\xce\x00d\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x04A\x9b\xf6g\xce\x00d\x11\x88N\x10Q\xd0P\x12\xc0xy\xca\xdb(\x8b\xdb(y\x8b\xdb(z\x8b\xdb({\xdb(\x8b\xdb(K\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00@\x12A\x9b\xf6g\xce\x00t\x11\x88N\x10Q\xd0P\x12\xc0J\x11\xceP\x10\xceA\xdf0\xb8\x9a@W\x00\x01\x12xA\x9b\xf6g\xce\x00t\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x02\x12xy\xca\xdb(\x8b\xdb(y\x8b\xdb(A\x9b\xf6g\xce\x00t\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x03\x12xy\xca\xdb(\x8b\xdb(y\x8b\xdb(z\x8b\xdb(A\x9b\xf6g\xce\x00t\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x04A\x9b\xf6g\xce\x00t\x11\x88N\x10Q\xd0P\x12\xc0xy\xca\xdb(\x8b\xdb(y\x8b\xdb(z\x8b\xdb({\xdb(\x8b\xdb(K\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00@W\x00\x01\x10xA\x9b\xf6g\xce\x00i\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x01A\x9b\xf6g\xce\x00i\x11\x88N\x10Q\xd0P\x12\xc0xK\x11\xceP\x8bP\x10\xceA\x92]\xe81@W\x00\x01\x10xA\x9b\xf6g\xce\x00e\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x02\x10xy\x8b\xdb(A\x9b\xf6g\xce\x00e\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x02A\x9b\xf6g\xce\x00e\x11\x88N\x10Q\xd0P\x12\xc0xy\x8b\xdb(K\x11\xceP\x8bP\x10\xceA\x92]\xe81@W\x00\x02xy\xb7&\x05x"\x03y@W\x02\x05A\xdb\xfe\xa8tpxh\x98$\x07\x10\xdb "\x05yh\x98&\n\x10\xdb #\x84\x00\x00\x00xA\xf8\'\xec\x8c\xaa&\x07\x10\xdb "syqi\x0b\x97&\x07\x11\xdb "\ryJ\xd9(P\xca\x00\x14\xb3\xab\xaa&"\x0c\x1dThe argument "to" is invalid.:z\x10\xb5&\x0f\x0c\namount < 0:xy\x98&\x15z\x9b{x5h\xf9\xff\xffEz{y5_\xf9\xff\xffE|{yx5\x8c\xfa\xff\xff\x11\xdb "\x02@W\x03\x00A\x9b\xf6g\xcep\x0c\x01\x02\xdb0qihA\x92]\xe81r\x0c\x0bToo long idj\xca\x01\xfd\x00\xb55\x07\xfb\xff\xffjJ\xd8&\x04E\x10\xdb!\x11\x9eihA\xe6?\x18\x84j"\x02@W\x06\x04\x0c\nNo witnessxA\xf8\'\xec\x8c5\xd7\xfa\xff\xffA\x9b\xf6g\xceph\x00r\x11\x88N\x10Q\xd0P\x12\xc0qyz\xca\xdb(\x8b\xdb(z\x8b\xdb(x\x8b\xdb(rijK\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00s\x0c\x12No token at rentalk\x10\xce\x10\xb75\x82\xfa\xff\xff{Jk\x11Q\xd0Ek7\x03\x00tijl\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84h\x00o\x11\x88N\x10Q\xd0P\x12\xc0umxy\x8b\xdb(z\x8b\xdb(l\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84@\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84@W\x05\x03A\x9b\xf6g\xceph\x00e\x11\x88N\x10Q\xd0P\x12\xc0qxy\x8b\xdb(rijK\x11\xceP\x8bP\x10\xceA\x92]\xe81sktl\x0b\x97&75\xc6\xfe\xff\xffJsEijk\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84h\x00i\x11\x88N\x10Q\xd0P\x12\xc0kj\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84\x0c\x08NEPHRENT\x0b\x11\xc0J46J\x10j\xd0kzA\xdb\xfe\xa8t5\xf4\xf6\xff\xff\xc2Jx\xcfJx\xcfJk\xcfJz\xcf\x0c\x0cTokenCreatedA\x95\x01oak"\x02@W\x00\x01@W\x03\x04A\x9b\xf6g\xceph\x00e\x11\x88N\x10Q\xd0P\x12\xc0qyz\x8b\xdb(r\x0c\x08NEPHRENT\x0b\x11\xc0J4\xd2J\x10j\xd0x{A\xdb\xfe\xa8t5\x90\xf6\xff\xff\xc2Jy\xcfJy\xcfJx\xcfJ{\xcf\x0c\x0cTokenCreatedA\x95\x01oax"\x02@W\x04\x07z}\xca\xdb(\x8b\xdb(}\x8b\xdb({\x8b\xdb(pxhK\x11\xceP\x8bP\x10\xceA\x92]\xe81qi\x0b\x98&\x1ei7\x00\x00JrEj\x10KK\xce|\x9eJTS\xd0E~Jj\x11Q\xd0E"\t~|\x12\xc0JrEj7\x03\x00sxhk\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84y{z\x8b\xdb(}\x8b\xdb(k\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84\xc2J{\xcfJz\xcfJ}\xcfJ~\xcf\x0c\x08PriceSetA\x95\x01oaj\x10\xce"\x02@W\x03\x06z}\xca\xdb(\x8b\xdb(}\x8b\xdb({\x8b\xdb(pxhK\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00qi\x10KK\xce|\x9eJTS\xd0Ei7\x03\x00rxhj\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84y{z\x8b\xdb(}\x8b\xdb(j\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84i\x10\xce"\x02@W\x04\x07\x0c\x13tokenId.Length > 64{\xca\x00@\xb65\xf8\xf7\xff\xff\x0c9Cannot register rental for tokens issued by this contractyA\xdb\xfe\xa8t\x985\xb1\xf7\xff\xff~&?\x0c\x0fTransfer failed\x0c\x08NEPHRENT{zA\xdb\xfe\xa8tx\x15\xc0\x1f\x0c\x08transferyAb}[R5s\xf7\xff\xff"?\x0c\x0fTransfer failed\x0c\x08NEPHRENT{A\xdb\xfe\xa8t\x13\xc0\x1f\x0c\x08transferyAb}[R58\xf7\xff\xff\x11J\x82EA\x9b\xf6g\xceph\x00r\x11\x88N\x10Q\xd0P\x12\xc0qh\x00o\x11\x88N\x10Q\xd0P\x12\xc0r|{zxyji5\xee\xfd\xff\xffEz{y5\xd5\xfc\xff\xffs}kzxA\xdb\xfe\xa8tji5\xd4\xfd\xff\xff"\x02@W\x00\x03zyx5\xdc\xfa\xff\xff4\x05"\x02@W\x00\x02\x0c Failed to find the token to burnx\x0c\x14\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x985\x99\xf6\xff\xffxyA\xdb\xfe\xa8t5s\xf4\xff\xffx"\x02@\x0c\x14\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@W\x03\x06z}\xca\xdb(\x8b\xdb(}\x8b\xdb({\x8b\xdb(pxhK\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00qi\x10KK\xce|\x9fJTS\xd0Ei\x10\xceJ\x84E\x0c\x1fNo enough token to unregister: z\x8b\xdb({\x8b\xdb(|7\x04\x00\x8b\xdb(}\x8b\xdb(|\x10\xb85\xfd\xf5\xff\xff|\x10\xb7&3i7\x03\x00rxhj\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84y{z\x8b\xdb(}\x8b\xdb(j\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84"(xhK\x11\xceP\x8bP\x10\xceA/X\xc5\xedy{z\x8b\xdb(}\x8b\xdb(K\x11\xceP\x8bP\x10\xceA/X\xc5\xed|"\x02@W\x04\x05\x0c\nNo witnessxA\xf8\'\xec\x8c5\x83\xf5\xff\xff\x0c;Cannot unregister rental for tokens issued by this contractyA\xdb\xfe\xa8t\x985:\xf5\xff\xffA\x9b\xf6g\xceph\x00r\x11\x88N\x10Q\xd0P\x12\xc0qh\x00o\x11\x88N\x10Q\xd0P\x12\xc0r|&V{zxyji5\x9f\xfe\xff\xffEz{y5\x1d\xfe\xff\xffJsE\x0c\x0fTransfer failed\x0c\x08NEPHRENT{zxA\xdb\xfe\xa8t\x15\xc0\x1f\x0c\x08transferyAb}[R5\xc5\xf4\xff\xff"R\x11J\x82E{zxyji5G\xfe\xff\xffEz{y5\xc5\xfd\xff\xffJsE\x0c\x0fTransfer failed\x0c\x08NEPHRENT{x\x13\xc0\x1f\x0c\x08transferyAb}[R5s\xf4\xff\xff\xc2Jy\xcfJ{\xcfJz\xcf\x0c\x0eTokenWithdrawnA\x95\x01oakzxA\xdb\xfe\xa8tji5\xd8\xfd\xff\xff"\x02@W\x00\x02\x00dxy\xa0\x01\xe8\x03\xa15S\xf8\xff\xff@W\x00\x01x\x01\x88\x13\xa0\x01\x10\'\xa1@W\x00\x02yx4\xdc\x01\x88\x13\xa0\x01\x10\'\xa1@W\x00\x06}A\x9b\xf6g\xce\x00e\x11\x88N\x10Q\xd0P\x12\xc0{|\x8b\xdb(K\x11\xceP\x8bP\x10\xceA\x92]\xe81zyx4\x05"\x02@W\x0b\x05A\xdb\xfe\xa8tpA\xb7\xc3\x88\x03q\x0c\x14Too long borrow time|\x02\x00\x98\x7f3\xb65\xb4\xf3\xff\xffA\x9b\xf6g\xcerj\x00r\x11\x88N\x10Q\xd0P\x12\xc0sh{\xca\xdb(\x8b\xdb({\x8b\xdb(x\x8b\xdb(tklK\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00u|m\x11\xce57\xff\xff\xffvn5B\xff\xff\xffw\x07\x0c\x11Failed to pay GAS\x0c\x08NEPHRENTnxy\x14\xc0\x1f\x0c\x08transfer\x0c\x14\xcfv\xe2\x8b\xd0\x06,JG\x8e\xe3Ua\x01\x13\x19\xf3\xcf\xa4\xd2Ab}[R5\x1e\xf3\xff\xff\x0c\x18Failed to pay collateral\x0c\x08NEPHRENTo\x07hy\x14\xc0\x1f\x0c\x08transfer\x0c\x14\xcfv\xe2\x8b\xd0\x06,JG\x8e\xe3Ua\x01\x13\x19\xf3\xcf\xa4\xd2Ab}[R5\xc9\xf2\xff\xffm\x10KK\xcez\x9fJTS\xd0E\x0c\x17No enough token to lendm\x10\xce\x10\xb85\x9a\xf2\xff\xffm\x10\xce\x10\xb7&Cm7\x03\x00Jw\x08Eklo\x08\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84j\x00o\x11\x88N\x10Q\xd0P\x12\xc0xh\x8b\xdb({\x8b\xdb(o\x08\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84"3klK\x11\xceP\x8bP\x10\xceA/X\xc5\xedj\x00o\x11\x88N\x10Q\xd0P\x12\xc0xh\x8b\xdb({\x8b\xdb(K\x11\xceP\x8bP\x10\xceA/X\xc5\xedj\x00d\x11\x88N\x10Q\xd0P\x12\xc0w\tx{\xca\xdb(\x8b\xdb({\x8b\xdb(y\x8b\xdb(i\xdb(\x8b\xdb(JtE\x0c%Cannot borrow twice in a single blocko\tlK\x11\xceP\x8bP\x10\xceA\x92]\xe81w\no\n\x0b\x975\xb8\xf1\xff\xff\x11i|\x9eo\x07z\x14\xc0w\no\n7\x03\x00Jw\x08Eo\tlo\x08\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84j\x00t\x11\x88N\x10Q\xd0P\x12\xc0y{\xca\xdb(\x8b\xdb({\x8b\xdb(x\x8b\xdb(i\xdb(\x8b\xdb(o\x08\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84\x0c\x08NEPHRENT{zyA\xdb\xfe\xa8t5w\xf5\xff\xff9\xc2Jx\xcfJA\xdb\xfe\xa8t\xcfJ{\xcfJy\xcfJi\xcfJz\xcfJn\xcfJo\x07\xcfJo\n\x12\xce\xcf\x0c\x0bTokenRentedA\x95\x01oai"\x02@A\xb7\xc3\x88\x03@\x0c\x14\xcfv\xe2\x8b\xd0\x06,JG\x8e\xe3Ua\x01\x13\x19\xf3\xcf\xa4\xd2@W\x06\x04\x0c\x16No witness from renterxA\xf8\'\xec\x8c5\xca\xf0\xff\xffA\x9b\xf6g\xceph\x00d\x11\x88N\x10Q\xd0P\x12\xc0qxy\xca\xdb(\x8b\xdb(y\x8b\xdb(z\x8b\xdb({\x8b\xdb(rijK\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00s\x10Jk\x13Q\xd0Ek7\x03\x00tlJjiS\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84Eh\x00t\x11\x88N\x10Q\xd0P\x12\xc0ulJzy\xca\xdb(\x8b\xdb(y\x8b\xdb(x\x8b\xdb({\x8b\xdb(mS\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84E\xc2Jx\xcfJy\xcfJz\xcfJ{J\xd8&\x04E\x10\xdb!\xcfJk\x10\xce\xcfJk\x11\xce\xcfJk\x12\xce\xcf\x0c\x0cRentalClosedA\x95\x01oa@W\x06\x04\x0c\x16No witness from renterxA\xf8\'\xec\x8c5\xdd\xef\xff\xffA\x9b\xf6g\xceph\x00d\x11\x88N\x10Q\xd0P\x12\xc0qxy\xca\xdb(\x8b\xdb(y\x8b\xdb(z\x8b\xdb({\x8b\xdb(rijK\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00s\x11Jk\x13Q\xd0Ek7\x03\x00tlJjiS\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84Eh\x00t\x11\x88N\x10Q\xd0P\x12\xc0ulJzy\xca\xdb(\x8b\xdb(y\x8b\xdb(x\x8b\xdb({\x8b\xdb(mS\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84E\xc2Jx\xcfJy\xcfJz\xcfJ{J\xd8&\x04E\x10\xdb!\xcfJk\x10\xce\xcfJk\x11\xce\xcfJk\x12\xce\xcf\x0c\x0cRentalOpenedA\x95\x01oa@W\x01\x07A\x9b\xf6g\xce\x00e\x11\x88N\x10Q\xd0P\x12\xc0z{\x8b\xdb(K\x11\xceP\x8bP\x10\xceA\x92]\xe81p~}|{zhyx4\x03@W\x07\x08A\x9b\xf6g\xceph\x00t\x11\x88N\x10Q\xd0P\x12\xc0qyz\xca\xdb(\x8b\xdb(z\x8b\xdb(x\x8b\xdb(}\x8b\xdb(rijK\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00sA\xb7\xc3\x88\x03k\x12\xce\xb6&=\x0c.Rental not expired. Need signature from tenantyA\xf8\'\xec\x8c5a\xee\xff\xffijK\x11\xceP\x8bP\x10\xceA/X\xc5\xedh\x00d\x11\x88N\x10Q\xd0P\x12\xc0xz\xca\xdb(\x8b\xdb(z\x8b\xdb(y\x8b\xdb(}\x8b\xdb(K\x11\xceP\x8bP\x10\xceA/X\xc5\xedk\x10\xcetzly5\xff\xeb\xff\xffk\x13\xce\x10\x97\'\xce\x00\x00\x00h\x00r\x11\x88N\x10Q\xd0P\x12\xc0uh\x00o\x11\x88N\x10Q\xd0P\x12\xc0v\x7f\x07&K|lx{nm5y\xf7\xff\xffE\x0c\x0fTransfer failed\x0c\x08NEPHRENT|lxA\xdb\xfe\xa8t\x15\xc0\x1f\x0c\x08transfer{Ab}[R5\xaa\xed\xff\xff"C|lx{nm50\xf7\xff\xffE\x0c\x0fTransfer failed\x0c\x08NEPHRENT|x\x13\xc0\x1f\x0c\x08transfer{Ab}[R5g\xed\xff\xff\xc2J{\xcfJ|\xcfJl\xcf\x0c\x0eTokenWithdrawnA\x95\x01oa"6k\x10\xce|{z5\xc1\xf3\xff\xffEzk\x10\xcexA\xdb\xfe\xa8th\x00o\x11\x88N\x10Q\xd0P\x12\xc0h\x00r\x11\x88N\x10Q\xd0P\x12\xc05\x96\xf4\xff\xffE\xc2Jx\xcfJz\xcfJy\xcfJ}J\xd8&\x04E\x10\xdb!\xcfJk\x10\xce\xcfJk\x11\xce\xcfJk\x12\xce\xcf\x0c\rRentalRevokedA\x95\x01oa\x0c\x08NEPHRENTk\x11\xce~A\xdb\xfe\xa8t\x14\xc0\x1f\x0c\x08transfer\x0c\x14\xcfv\xe2\x8b\xd0\x06,JG\x8e\xe3Ua\x01\x13\x19\xf3\xcf\xa4\xd2Ab}[R9@W\x01\x06A\x9b\xf6g\xce\x00i\x11\x88N\x10Q\xd0P\x12\xc0zK\x11\xceP\x8bP\x10\xceA\x92]\xe81\xdb0p}|{hJ\xcaP\x00\x14QK\x9f\x8c\xdb(h\x00\x14P\x10QK\x9f\x8c\xdb(J\xd8$\tJ\xca\x00\x14(\x03:zyx5l\xfd\xff\xff@\xdb0@\xdb(@\xdb(J\xd8$\tJ\xca\x00\x14(\x03:@W\t\x08A\x9b\xf6g\xceph\x00r\x11\x88N\x10Q\xd0P\x12\xc0q{\x0c\x14\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x97\'9\x01\x00\x00\x12yz\xca\xdb(\x8b\xdb(z\x8b\xdb(iN\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9as\x10tkA\x9c\x08\xed\x9c$\x07\x10\xdb "\x05l|\xb5\'\xe1\x00\x00\x00kA\xf3T\xbf\x1dw\x07o\x07\x10\xceJ\xd8$\tJ\xca\x00\x14(\x03:J\x83Eo\x07\x11\xce7\x00\x00w\x08o\x08\x10\xceJvE|l\x9fJuEnm\xb7&\x06mJvEln\x9eJtE\x00dno\x08\x11\xce\xa05\xa7\xef\xff\xffJrE\x0c\x13GAS transfer failed\x0c\x08NEPHRENTj{x\x14\xc0\x1f\x0c\x08transfer\x0c\x14\xcfv\xe2\x8b\xd0\x06,JG\x8e\xe3Ua\x01\x13\x19\xf3\xcf\xa4\xd2Ab}[R53\xeb\xff\xff\xc2J{\xcfJy\xcfJz\xcfJx\xcfJA\xb7\xc3\x88\x03\xcfJo\x08\x10\xce\xcfJj\xcfJ\x10\xcfJA\xb7\xc3\x88\x03\xcf\x0c\x0bTokenRentedA\x95\x01oa#\x14\xff\xff\xff\x0c\x16No enough NFTs to rentl|\x975\xd5\xea\xff\xff#\xe1\x00\x00\x00iyz\xca\xdb(\x8b\xdb(z\x8b\xdb({\x8b\xdb(K\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00s\x0c\x16No enough NFTs to rentk\x10\xce|\xb85\x8c\xea\xff\xff\x00dk\x10\xcek\x11\xce\xa05\xa0\xee\xff\xffJrE\x0c\x13GAS transfer failed\x0c\x08NEPHRENTj{x\x14\xc0\x1f\x0c\x08transfer\x0c\x14\xcfv\xe2\x8b\xd0\x06,JG\x8e\xe3Ua\x01\x13\x19\xf3\xcf\xa4\xd2Ab}[R5,\xea\xff\xff\xc2J{\xcfJy\xcfJz\xcfJx\xcfJA\xb7\xc3\x88\x03\xcfJk\x10\xce\xcfJj\xcfJ\x10\xcfJA\xb7\xc3\x88\x03\xcf\x0c\x0bTokenRentedA\x95\x01oa\x0c\x13NFT transfer failed\x0c\x08NEPHRENTz|xA\xdb\xfe\xa8t\x15\xc0\x1f\x0c\x08transferyAb}[R5\xb5\xe9\xff\xff\x7f\x07\x1f~}Ab}[Rs\x0c\x12NFT payback failed\x0c\x08NEPHRENTz|A\xdb\xfe\xa8tx\x15\xc0\x1f\x0c\x08transferyAb}[R5l\xe9\xff\xffk"\x02@A\x9c\x08\xed\x9c@A\xf3T\xbf\x1d@W\x05\x07A\x9b\xf6g\xce\x00r\x11\x88N\x10Q\xd0P\x12\xc0p\x12yz\xca\xdb(\x8b\xdb(z\x8b\xdb(hN\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9aq\x0c\x15Failed to find renteriA\x9c\x08\xed\x9c5\n\xe9\xff\xffiA\xf3T\xbf\x1d7\x00\x00r\x00dj\x11\xce5\x18\xed\xff\xffs\x0c\x13GAS transfer failed\x0c\x08NEPHRENTk{x\x14\xc0\x1f\x0c\x08transfer\x0c\x14\xcfv\xe2\x8b\xd0\x06,JG\x8e\xe3Ua\x01\x13\x19\xf3\xcf\xa4\xd2Ab}[R5\xa6\xe8\xff\xff\xc2J{\xcfJy\xcfJz\xcfJx\xcfJA\xb7\xc3\x88\x03\xcfJj\x10\xce\xcfJj\x10\xcek\xa0\xcfJ\x10\xcfJA\xb7\xc3\x88\x03\xcf\x0c\x0bTokenRentedA\x95\x01oa\x0c\x13NFT transfer failed\x0c\x08NEPHRENTzx\x13\xc0\x1f\x0c\x08transferyAb}[R51\xe8\xff\xff~\x1f}|Ab}[Rt\x0c\x12NFT payback failed\x0c\x08NEPHRENTzA\xdb\xfe\xa8t\x13\xc0\x1f\x0c\x08transferyAb}[R5\xeb\xe7\xff\xffl"\x02@V\x02\n\x0b\xe2\xff\xff\n*\xdf\xff\xff\n\xfe\xde\xff\xff\x13\xc0`\n\xf9\xe1\xff\xff\n\x18\xdf\xff\xff\x0b\x13\xc0a@\xc2JX\xcfJ5\xf0\xde\xff\xff#\xdd\xde\xff\xff\xc2JY\xcfJ5\xe8\xde\xff\xff#\xf5\xde\xff\xff\xc2JY\xcfJ5\xd9\xde\xff\xff#\xc2\xe1\xff\xff\x90z\xe1\xc6', '{"name":"AnyUpdateShortSafe","groups":[],"features":{},"supportedstandards":["NEP-11"],"abi":{"methods":[{"name":"symbol","offset":8473,"safe":false,"returntype":"String","parameters":[]},{"name":"decimals","offset":8488,"safe":false,"returntype":"Integer","parameters":[]},{"name":"totalSupply","offset":414,"safe":true,"returntype":"Integer","parameters":[]},{"name":"ownerOf","offset":438,"safe":true,"returntype":"InteropInterface","parameters":[{"name":"tokenId","type":"ByteArray"}]},{"name":"balanceOf","offset":529,"safe":true,"returntype":"Integer","parameters":[{"name":"owner","type":"Hash160"}]},{"name":"balanceOf","offset":634,"safe":true,"returntype":"Integer","parameters":[{"name":"owner","type":"Hash160"},{"name":"tokenId","type":"ByteArray"}]},{"name":"properties","offset":8503,"safe":true,"returntype":"Map","parameters":[{"name":"tokenId","type":"ByteArray"}]},{"name":"tokens","offset":837,"safe":true,"returntype":"InteropInterface","parameters":[]},{"name":"tokensOf","offset":884,"safe":true,"returntype":"InteropInterface","parameters":[{"name":"owner","type":"Hash160"}]},{"name":"_deploy","offset":2174,"safe":false,"returntype":"Void","parameters":[{"name":"data","type":"Any"},{"name":"update","type":"Boolean"}]},{"name":"onNEP11Payment","offset":2199,"safe":false,"returntype":"Void","parameters":[{"name":"from","type":"Hash160"},{"name":"amount","type":"Integer"},{"name":"tokenId","type":"ByteArray"},{"name":"data","type":"Any"}]},{"name":"onNEP17Payment","offset":2291,"safe":false,"returntype":"Void","parameters":[{"name":"from","type":"Hash160"},{"name":"amount","type":"Integer"},{"name":"data","type":"Any"}]},{"name":"getDecimals","offset":2358,"safe":false,"returntype":"Integer","parameters":[{"name":"externalTokenContract","type":"Hash160"}]},{"name":"balanceOfRentalToken","offset":2380,"safe":false,"returntype":"Integer","parameters":[{"name":"internalTokenId","type":"ByteArray"}]},{"name":"listRegisteredRentalByToken","offset":2395,"safe":false,"returntype":"InteropInterface","parameters":[]},{"name":"listRegisteredRentalByToken","offset":2424,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"externalTokenContract","type":"Hash160"}]},{"name":"listRegisteredRentalByToken","offset":2459,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"externalTokenContract","type":"Hash160"},{"name":"externalTokenId","type":"ByteArray"}]},{"name":"getRegisteredRentalByToken","offset":2505,"safe":false,"returntype":"Array","parameters":[{"name":"externalTokenContract","type":"Hash160"},{"name":"externalTokenId","type":"ByteArray"},{"name":"renter","type":"Hash160"}]},{"name":"listRegisteredRentalByRenter","offset":2557,"safe":false,"returntype":"InteropInterface","parameters":[]},{"name":"listRegisteredRentalByRenter","offset":2586,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"renter","type":"Hash160"}]},{"name":"listRegisteredRentalByRenter","offset":2621,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"renter","type":"Hash160"},{"name":"externalTokenContract","type":"Hash160"}]},{"name":"getRegisteredRentalByRenter","offset":2660,"safe":false,"returntype":"Array","parameters":[{"name":"renter","type":"Hash160"},{"name":"externalTokenContract","type":"Hash160"},{"name":"externalTokenId","type":"ByteArray"}]},{"name":"listRentalDeadlineByRenter","offset":2705,"safe":false,"returntype":"InteropInterface","parameters":[]},{"name":"listRentalDeadlineByRenter","offset":2734,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"renter","type":"Hash160"}]},{"name":"listRentalDeadlineByRenter","offset":2769,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"renter","type":"Hash160"},{"name":"internalTokenId","type":"ByteArray"}]},{"name":"listRentalDeadlineByRenter","offset":2815,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"renter","type":"Hash160"},{"name":"internalTokenId","type":"ByteArray"},{"name":"tenant","type":"Hash160"}]},{"name":"getRentalDeadlineByRenter","offset":2865,"safe":false,"returntype":"Array","parameters":[{"name":"renter","type":"Hash160"},{"name":"internalTokenId","type":"ByteArray"},{"name":"tenant","type":"Hash160"},{"name":"startTime","type":"Integer"}]},{"name":"listRentalDeadlineByTenant","offset":2923,"safe":false,"returntype":"InteropInterface","parameters":[]},{"name":"listRentalDeadlineByTenant","offset":2952,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"tenant","type":"Hash160"}]},{"name":"listRentalDeadlineByTenant","offset":2987,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"tenant","type":"Hash160"},{"name":"internalTokenId","type":"ByteArray"}]},{"name":"listRentalDeadlineByTenant","offset":3033,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"tenant","type":"Hash160"},{"name":"internalTokenId","type":"ByteArray"},{"name":"renter","type":"Hash160"}]},{"name":"getRentalDeadlineByTenant","offset":3083,"safe":false,"returntype":"Array","parameters":[{"name":"tenant","type":"Hash160"},{"name":"internalTokenId","type":"ByteArray"},{"name":"renter","type":"Hash160"},{"name":"startTime","type":"Integer"}]},{"name":"listExternalTokenInfo","offset":3141,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"prefix","type":"ByteArray"}]},{"name":"getExternalTokenInfo","offset":3176,"safe":false,"returntype":"ByteArray","parameters":[{"name":"internalTokenId","type":"ByteArray"}]},{"name":"listInternalTokenId","offset":3210,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"prefix","type":"ByteArray"}]},{"name":"listInternalTokenId","offset":3245,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"externalTokenContract","type":"Hash160"},{"name":"prefix","type":"ByteArray"}]},{"name":"getInternalTokenId","offset":3284,"safe":false,"returntype":"ByteArray","parameters":[{"name":"externalTokenContract","type":"Hash160"},{"name":"externalTokenId","type":"ByteArray"}]},{"name":"max","offset":3322,"safe":false,"returntype":"Integer","parameters":[{"name":"v1","type":"Integer"},{"name":"v2","type":"Integer"}]},{"name":"transfer","offset":3335,"safe":false,"returntype":"Boolean","parameters":[{"name":"from","type":"Hash160"},{"name":"to","type":"Hash160"},{"name":"amount","type":"Integer"},{"name":"tokenId","type":"ByteArray"},{"name":"data","type":"Any"}]},{"name":"setRentalPrice","offset":3564,"safe":false,"returntype":"Void","parameters":[{"name":"renter","type":"Hash160"},{"name":"tokenContract","type":"Hash160"},{"name":"tokenId","type":"ByteArray"},{"name":"price","type":"Integer"}]},{"name":"registerRental","offset":4291,"safe":false,"returntype":"Integer","parameters":[{"name":"renter","type":"Hash160"},{"name":"externalTokenContract","type":"Hash160"},{"name":"amountForRent","type":"Integer"},{"name":"externalTokenId","type":"ByteArray"},{"name":"flashLoanPrice","type":"Integer"},{"name":"ordinaryLoanPrice","type":"Integer"},{"name":"isDivisible","type":"Boolean"}]},{"name":"unregisterRental","offset":4928,"safe":false,"returntype":"Integer","parameters":[{"name":"renter","type":"Hash160"},{"name":"externalTokenContract","type":"Hash160"},{"name":"amountToUnregister","type":"Integer"},{"name":"externalTokenId","type":"ByteArray"},{"name":"isDivisible","type":"Boolean"}]},{"name":"getTotalPrice","offset":5275,"safe":false,"returntype":"Integer","parameters":[{"name":"pricePerSecond","type":"Integer"},{"name":"borrowTimeMilliseconds","type":"Integer"}]},{"name":"getCollateralAmount","offset":5293,"safe":false,"returntype":"Integer","parameters":[{"name":"totalPrice","type":"Integer"}]},{"name":"getCollateralAmount","offset":5306,"safe":false,"returntype":"Integer","parameters":[{"name":"pricePerSecond","type":"Integer"},{"name":"borrowTimeMilliseconds","type":"Integer"}]},{"name":"borrow","offset":5322,"safe":false,"returntype":"Integer","parameters":[{"name":"renter","type":"Hash160"},{"name":"tenant","type":"Hash160"},{"name":"amount","type":"Integer"},{"name":"externalTokenContract","type":"Hash160"},{"name":"externalTokenId","type":"ByteArray"},{"name":"borrowTimeMilliseconds","type":"Integer"}]},{"name":"borrow","offset":5368,"safe":false,"returntype":"Integer","parameters":[{"name":"renter","type":"Hash160"},{"name":"tenant","type":"Hash160"},{"name":"amount","type":"Integer"},{"name":"internalTokenId","type":"ByteArray"},{"name":"borrowTimeMilliseconds","type":"Integer"}]},{"name":"closeNextRental","offset":6125,"safe":false,"returntype":"Void","parameters":[{"name":"renter","type":"Hash160"},{"name":"internalTokenId","type":"ByteArray"},{"name":"tenant","type":"Hash160"},{"name":"startTime","type":"ByteArray"}]},{"name":"openNextRental","offset":6362,"safe":false,"returntype":"Void","parameters":[{"name":"renter","type":"Hash160"},{"name":"internalTokenId","type":"ByteArray"},{"name":"tenant","type":"Hash160"},{"name":"startTime","type":"ByteArray"}]},{"name":"payback","offset":6599,"safe":false,"returntype":"Void","parameters":[{"name":"renter","type":"Hash160"},{"name":"tenant","type":"Hash160"},{"name":"externalTokenContract","type":"Hash160"},{"name":"externalTokenId","type":"ByteArray"},{"name":"startTime","type":"ByteArray"},{"name":"collateralReceiver","type":"Hash160"},{"name":"isDivisible","type":"Boolean"}]},{"name":"payback","offset":7232,"safe":false,"returntype":"Void","parameters":[{"name":"renter","type":"Hash160"},{"name":"tenant","type":"Hash160"},{"name":"internalTokenId","type":"ByteArray"},{"name":"startTime","type":"ByteArray"},{"name":"collateralReceiver","type":"Hash160"},{"name":"isDivisible","type":"Boolean"}]},{"name":"flashBorrowDivisible","offset":7334,"safe":false,"returntype":"Any","parameters":[{"name":"tenant","type":"Hash160"},{"name":"token","type":"Hash160"},{"name":"tokenId","type":"ByteArray"},{"name":"renter","type":"Hash160"},{"name":"neededAmount","type":"Integer"},{"name":"renterCalledContract","type":"Hash160"},{"name":"renterCalledMethod","type":"String"},{"name":"arguments","type":"Array"}]},{"name":"flashBorrowNonDivisible","offset":8065,"safe":false,"returntype":"Any","parameters":[{"name":"tenant","type":"Hash160"},{"name":"token","type":"Hash160"},{"name":"tokenId","type":"ByteArray"},{"name":"renter","type":"Hash160"},{"name":"calledContract","type":"Hash160"},{"name":"calledMethod","type":"String"},{"name":"arguments","type":"Array"}]},{"name":"_initialize","offset":8438,"safe":false,"returntype":"Void","parameters":[]}],"events":[{"name":"Transfer","parameters":[{"name":"from","type":"Hash160"},{"name":"to","type":"Hash160"},{"name":"amount","type":"Integer"},{"name":"tokenId","type":"ByteArray"}]},{"name":"TokenCreated","parameters":[{"name":"arg1","type":"Hash160"},{"name":"arg2","type":"ByteArray"},{"name":"arg3","type":"ByteArray"},{"name":"arg4","type":"Integer"}]},{"name":"PriceSet","parameters":[{"name":"arg1","type":"Hash160"},{"name":"arg2","type":"Hash160"},{"name":"arg3","type":"ByteArray"},{"name":"arg4","type":"Integer"}]},{"name":"TokenRented","parameters":[{"name":"arg1","type":"Hash160"},{"name":"arg2","type":"Hash160"},{"name":"arg3","type":"ByteArray"},{"name":"arg4","type":"Hash160"},{"name":"arg5","type":"Integer"},{"name":"arg6","type":"Integer"},{"name":"arg7","type":"Integer"},{"name":"arg8","type":"Integer"},{"name":"arg9","type":"Integer"}]},{"name":"RentalClosed","parameters":[{"name":"arg1","type":"Hash160"},{"name":"arg2","type":"ByteArray"},{"name":"arg3","type":"Hash160"},{"name":"arg4","type":"Integer"},{"name":"arg5","type":"Integer"},{"name":"arg6","type":"Integer"},{"name":"arg7","type":"Integer"}]},{"name":"RentalOpened","parameters":[{"name":"arg1","type":"Hash160"},{"name":"arg2","type":"ByteArray"},{"name":"arg3","type":"Hash160"},{"name":"arg4","type":"Integer"},{"name":"arg5","type":"Integer"},{"name":"arg6","type":"Integer"},{"name":"arg7","type":"Integer"}]},{"name":"RentalRevoked","parameters":[{"name":"arg1","type":"Hash160"},{"name":"arg2","type":"ByteArray"},{"name":"arg3","type":"Hash160"},{"name":"arg4","type":"Integer"},{"name":"arg5","type":"Integer"},{"name":"arg6","type":"Integer"},{"name":"arg7","type":"Integer"}]},{"name":"TokenWithdrawn","parameters":[{"name":"arg1","type":"Hash160"},{"name":"arg2","type":"ByteArray"},{"name":"arg3","type":"Integer"}]}]},"permissions":[{"contract":"*","methods":"*"}],"trusts":[],"extra":{"Author":"Hecate2","Email":"developer@neo.org","Description":"NFTFlashLoan"}}', 'registerRental', [0xb1983fa2479a0c8e2beae032d2df564b5451b7a5, 0x2704f0fa22877b2e0a53afe858b8db971f348b91, 68, 1, 5, 7, True]] relay=True [{'account': '0xb1983fa2479a0c8e2beae032d2df564b5451b7a5', 'scopes': 'Global', 'allowedcontracts': [], 'allowedgroups': [], 'rules': []}]
68
```

##### Step 7: Cloning snapshots

By cloning snapshots, you are "forking the blockchain" from your old snapshot again. The written transactions in the old snapshot will be remembered in the new snapshot. 

```python
client.copy_snapshot('Hello world! Your first contact with Fairy!', 'Cloned snapshot')
client.fairy_session = 'Cloned snapshot'  # selecting the new snapshot
```

Now just select a snapshot continue to invoke more methods! Everything happening in the cloned snapshot will affect neither the real blockchain nor the old snapshot. 

##### Last step: Understanding the errors and fixing the bugs

We are not going to continue with the cloned snapshots, but explain the red error information given by Fairy. Head to [tutorial.py](tutorial.py) and comment out the line mentioned in Step 4:

```python
# client.invokefunction('putStorage', params=[0x02, 1])
```

And run the whole tutorial. You'll see confusing errors like this:

```
Hello world! Your first contact with Fairy!::anyUpdate[b'NEF3Neo.Compiler.CSharp 3.1.0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05\xc0\xef9\xce\xe0\xe4\xe9%\xc6\xc2\xa0jy\xe1D\r\xd8o\xce\xac\x0bdeserialize\x01\x00\x01\x0f\xfd\xa3\xfaCF\xeaS*%\x8f\xc4\x97\xdd\xad\xdbd7\xc9\xfd\xff\x0bgetContract\x01\x00\x01\x0f\x1b\xf5u\xab\x11\x89h\x84\x13a\n5\xa1(\x86\xcd\xe0\xb6lr\x06sha256\x01\x00\x01\x0f\xc0\xef9\xce\xe0\xe4\xe9%\xc6\xc2\xa0jy\xe1D\r\xd8o\xce\xac\tserialize\x01\x00\x01\x0f\xc0\xef9\xce\xe0\xe4\xe9%\xc6\xc2\xa0jy\xe1D\r\xd8o\xce\xac\x04itoa\x01\x00\x01\x0f\x00\x00\xfdF!W\x00\x01\x0c\x08NEPHRENT@W\x00\x01x4\x03@W\x00\x01x4\x03@W\x00\x01x4\x03@W\x00\x01@W\x00\x01\x00d@\x0c\x01\x00\xdb0A\x9b\xf6g\xceA\x92]\xe81J\xd8&\x04E\x10\xdb!@J\xd8&\x04E\x10\xdb!@A\x92]\xe81@A\x9b\xf6g\xce@W\x01\x01xph\x0b\x97&\x07\x11\xdb "\rxJ\xd9(P\xca\x00\x14\xb3\xab\xaa&%\x0c The argument "owner" is invalid.:A\x9b\xf6g\xce\x11\x11\x88N\x10Q\xd0P\x12\xc0phxK\x11\xceP\x8bP\x10\xceA\x92]\xe81J\xd8&\x04E\x10\xdb!"\x02@J\xd9(P\xca\x00\x14\xb3\xab@\x11\x88N\x10Q\xd0P\x12\xc0@K\x11\xceP\x8bP\x10\xceA\x92]\xe81@W\x03\x01A\x9b\xf6g\xcep\x0c\x01\x00\xdb0qihA\x92]\xe81J\xd8&\x04E\x10\xdb!rjx\x9eJrEjihA\xe6?\x18\x84@A\xe6?\x18\x84@W\x02\x02A\x9b\xf6g\xce\x11\x11\x88N\x10Q\xd0P\x12\xc0phxK\x11\xceP\x8bP\x10\xceA\x92]\xe81J\xd8&\x04E\x10\xdb!qiy\x9eJqEi\x10\xb5&\x07\x10\xdb ".i\x10\xb3&\x13hxK\x11\xceP\x8bP\x10\xceA/X\xc5\xed"\x13hxi\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84\x11\xdb "\x02@K\x11\xceP\x8bP\x10\xceA/X\xc5\xed@\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84@\x0c\x01\x00\xdb0A\x9b\xf6g\xceA\x92]\xe81J\xd8&\x04E\x10\xdb!@W\x00\x01x\xca\x00@\xb7&\x18\x0c\x13tokenId.Length > 64:\x13x\xca\xdb(x\x8b\xdb(A\x9b\xf6g\xce\x15\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a"\x02@\xca@N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@\xdb(@W\x01\x01xph\x0b\x97&\x07\x11\xdb "\rxJ\xd9(P\xca\x00\x14\xb3\xab\xaa&%\x0c The argument "owner" is invalid.:A\x9b\xf6g\xce\x11\x11\x88N\x10Q\xd0P\x12\xc0phxK\x11\xceP\x8bP\x10\xceA\x92]\xe81J\xd8&\x04E\x10\xdb!"\x02@W\x00\x02xJ\xd9(P\xca\x00\x14\xb3\xab\xaa&$\x0c\x1fThe argument "owner" is invalid:y\xca\x00@\xb7&\x18\x0c\x13tokenId.Length > 64:A\x9b\xf6g\xce\x14\x11\x88N\x10Q\xd0P\x12\xc0xy\x8b\xdb(K\x11\xceP\x8bP\x10\xceA\x92]\xe81J\xd8&\x04E\x10\xdb!"\x02@K\x11\xceP\x8bP\x10\xceA\x92]\xe81@W\x03\x02A\x9b\xf6g\xce\x13\x11\x88N\x10Q\xd0P\x12\xc0phyK\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00q\xc8ri\x10\xceJ\x0c\x04namejS\xd0Ej"\x02@7\x00\x00@\xc8@\xd0@W\x01\x00A\x9b\xf6g\xce\x13\x11\x88N\x10Q\xd0P\x12\xc0p\x13hJ\x11\xceP\x10\xceA\xdf0\xb8\x9a"\x02@J\x11\xceP\x10\xceA\xdf0\xb8\x9a@W\x01\x01xph\x0b\x97&\x07\x11\xdb "\rxJ\xd9(P\xca\x00\x14\xb3\xab\xaa&$\x0c\x1fThe argument "owner" is invalid:A\x9b\xf6g\xce\x14\x11\x88N\x10Q\xd0P\x12\xc0p\x13xhN\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a"\x02@W\x01\x05xA\xf8\'\xec\x8c\xaa&\n\x10\xdb #\x83\x00\x00\x00yph\x0b\x97&\x07\x11\xdb "\ryJ\xd9(P\xca\x00\x14\xb3\xab\xaa&"\x0c\x1dThe argument "to" is invalid.:z\x10\xb5&\x0f\x0c\namount < 0:xy\x98$\x07\x10\xdb "\x08z\x9b{x4,$\x07\x10\xdb "\x07z{y4 &\r|{yx5!\x01\x00\x00"\x07\x10\xdb "\x07\x11\xdb "\x02@A\xf8\'\xec\x8c@W\x07\x03A\x9b\xf6g\xce\x11\x11\x88N\x10Q\xd0P\x12\xc0phxK\x11\xceP\x8bP\x10\xceA\x92]\xe81J\xd8&\x04E\x10\xdb!qiz\x9eJqEi\x10\xb5&\n\x10\xdb #\xcf\x00\x00\x00A\x9b\xf6g\xce\x14\x11\x88N\x10Q\xd0P\x12\xc0rA\x9b\xf6g\xce\x15\x11\x88N\x10Q\xd0P\x12\xc0sxy\x8b\xdb(ty\xca\xdb(y\x8b\xdb(x\x8b\xdb(ujlK\x11\xceP\x8bP\x10\xceA\x92]\xe81J\xd8&\x04E\x10\xdb!z\x9evn\x10\xb5&\x07\x10\xdb "ui\x10\xb3&\x13hxK\x11\xceP\x8bP\x10\xceA/X\xc5\xed"\x13hxi\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84n\x10\xb7&&jln\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84kmn\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84" jlK\x11\xceP\x8bP\x10\xceA/X\xc5\xedkmK\x11\xceP\x8bP\x10\xceA/X\xc5\xed\x11\xdb "\x02@W\x01\x04\xc2Jx\xcfJy\xcfJ\x11\xcfJz\xcf\x0c\x08TransferA\x95\x01oayph\x0b\x97\xaa$\x07\x10\xdb "\x0by7\x01\x00ph\x0b\x97\xaa& {z\x11x\x14\xc0\x1f\x0c\x0eonNEP11PaymentyAb}[RE@7\x01\x00@Ab}[R@W\x05\x00A\x9b\xf6g\xcep\x0c\x01\x02\xdb0qihA\x92]\xe81rjJ\xd8&\x04E\x10\xdb!\x11\x9eihA\xe6?\x18\x84A\xdb\xfe\xa8tsjtl\x0b\x97\xaa&\x08kj\x9eJsEk7\x02\x00"\x02@A\xdb\xfe\xa8t@7\x02\x00@W\x01\x05y\x10\xb6&\x15\x0c\x10mint amount <= 0:A\x9b\xf6g\xce\x13\x11\x88N\x10Q\xd0P\x12\xc0p{7\x03\x00JzhS\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84Eyzx5\x04\xfe\xff\xffEy4\x1f|zx\x0b5\x00\xff\xff\xff@7\x03\x00@\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84@W\x03\x01A\x9b\xf6g\xcep\x0c\x01\x00\xdb0qihA\x92]\xe81J\xd8&\x04E\x10\xdb!rjx\x9eJrEjihA\xe6?\x18\x84@W\x00\x03y\x10\xb6&\x15\x0c\x10burn amount <= 0:y\x9bzx5\x91\xfd\xff\xffEy\x9b4\xab\x0bz\x0bx5\x8c\xfe\xff\xff@W\x07\x03A\x9b\xf6g\xce\x11\x11\x88N\x10Q\xd0P\x12\xc0phxK\x11\xceP\x8bP\x10\xceA\x92]\xe81J\xd8&\x04E\x10\xdb!qiz\x9eJqEi\x10\xb5&\n\x10\xdb #\xcf\x00\x00\x00A\x9b\xf6g\xce\x14\x11\x88N\x10Q\xd0P\x12\xc0rA\x9b\xf6g\xce\x15\x11\x88N\x10Q\xd0P\x12\xc0sxy\x8b\xdb(ty\xca\xdb(y\x8b\xdb(x\x8b\xdb(ujlK\x11\xceP\x8bP\x10\xceA\x92]\xe81J\xd8&\x04E\x10\xdb!z\x9evn\x10\xb5&\x07\x10\xdb "ui\x10\xb3&\x13hxK\x11\xceP\x8bP\x10\xceA/X\xc5\xed"\x13hxi\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84n\x10\xb7&&jln\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84kmn\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84" jlK\x11\xceP\x8bP\x10\xceA/X\xc5\xedkmK\x11\xceP\x8bP\x10\xceA/X\xc5\xed\x11\xdb "\x02@W\x03\x01A\x9b\xf6g\xcep\x0c\x01\x00\xdb0qihA\x92]\xe81J\xd8&\x04E\x10\xdb!rjx\x9eJrEjihA\xe6?\x18\x84@W\x01\x04\xc2Jx\xcfJy\xcfJ\x11\xcfJz\xcf\x0c\x08TransferA\x95\x01oayph\x0b\x97\xaa$\x07\x10\xdb "\x0by7\x01\x00ph\x0b\x97\xaa& {z\x11x\x14\xc0\x1f\x0c\x0eonNEP11PaymentyAb}[RE@W\x00\x02y&\x04"\x12\x11\x0c\x01\x02\xdb0A\x9b\xf6g\xceA\xe6?\x18\x84@W\x00\x04\x0c-Do not send NFTs directly into this contract!{\x0c\x08NEPHRENT\x974\x03@W\x00\x02x&\x04"\x0cyA\xcf\xe7G\x96\x10\xdb 9@A\xcf\xe7G\x96@9@W\x00\x03\x0c/Do not send tokens directly into this contract!z\x0c\x08NEPHRENT\x974\xa5@W\x00\x01\xc2\x11\x0c\x08decimalsxAb}[R@W\x00\x01xA\xdb\xfe\xa8t5%\xf9\xff\xff@\x10A\x9b\xf6g\xce\x00r\x11\x88N\x10Q\xd0P\x12\xc0J\x11\xceP\x10\xceA\xdf0\xb8\x9a@W\x00\x01\x12xA\x9b\xf6g\xce\x00r\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x02\x12xy\xca\xdb(\x8b\xdb(y\x8b\xdb(A\x9b\xf6g\xce\x00r\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x03A\x9b\xf6g\xce\x00r\x11\x88N\x10Q\xd0P\x12\xc0xy\xca\xdb(\x8b\xdb(y\x8b\xdb(z\x8b\xdb(K\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00@\x10A\x9b\xf6g\xce\x00o\x11\x88N\x10Q\xd0P\x12\xc0J\x11\xceP\x10\xceA\xdf0\xb8\x9a@W\x00\x01\x12xA\x9b\xf6g\xce\x00o\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x02\x12xy\x8b\xdb(A\x9b\xf6g\xce\x00o\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x03A\x9b\xf6g\xce\x00o\x11\x88N\x10Q\xd0P\x12\xc0xy\x8b\xdb(z\x8b\xdb(K\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00@\x12A\x9b\xf6g\xce\x00d\x11\x88N\x10Q\xd0P\x12\xc0J\x11\xceP\x10\xceA\xdf0\xb8\x9a@W\x00\x01\x12xA\x9b\xf6g\xce\x00d\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x02\x12xy\xca\xdb(\x8b\xdb(y\x8b\xdb(A\x9b\xf6g\xce\x00d\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x03\x12xy\xca\xdb(\x8b\xdb(y\x8b\xdb(z\x8b\xdb(A\x9b\xf6g\xce\x00d\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x04A\x9b\xf6g\xce\x00d\x11\x88N\x10Q\xd0P\x12\xc0xy\xca\xdb(\x8b\xdb(y\x8b\xdb(z\x8b\xdb({\xdb(\x8b\xdb(K\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00@\x12A\x9b\xf6g\xce\x00t\x11\x88N\x10Q\xd0P\x12\xc0J\x11\xceP\x10\xceA\xdf0\xb8\x9a@W\x00\x01\x12xA\x9b\xf6g\xce\x00t\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x02\x12xy\xca\xdb(\x8b\xdb(y\x8b\xdb(A\x9b\xf6g\xce\x00t\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x03\x12xy\xca\xdb(\x8b\xdb(y\x8b\xdb(z\x8b\xdb(A\x9b\xf6g\xce\x00t\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x04A\x9b\xf6g\xce\x00t\x11\x88N\x10Q\xd0P\x12\xc0xy\xca\xdb(\x8b\xdb(y\x8b\xdb(z\x8b\xdb({\xdb(\x8b\xdb(K\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00@W\x00\x01\x10xA\x9b\xf6g\xce\x00i\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x01A\x9b\xf6g\xce\x00i\x11\x88N\x10Q\xd0P\x12\xc0xK\x11\xceP\x8bP\x10\xceA\x92]\xe81@W\x00\x01\x10xA\x9b\xf6g\xce\x00e\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x02\x10xy\x8b\xdb(A\x9b\xf6g\xce\x00e\x11\x88N\x10Q\xd0P\x12\xc0N\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9a@W\x00\x02A\x9b\xf6g\xce\x00e\x11\x88N\x10Q\xd0P\x12\xc0xy\x8b\xdb(K\x11\xceP\x8bP\x10\xceA\x92]\xe81@W\x00\x02xy\xb7&\x05x"\x03y@W\x02\x05A\xdb\xfe\xa8tpxh\x98$\x07\x10\xdb "\x05yh\x98&\n\x10\xdb #\x84\x00\x00\x00xA\xf8\'\xec\x8c\xaa&\x07\x10\xdb "syqi\x0b\x97&\x07\x11\xdb "\ryJ\xd9(P\xca\x00\x14\xb3\xab\xaa&"\x0c\x1dThe argument "to" is invalid.:z\x10\xb5&\x0f\x0c\namount < 0:xy\x98&\x15z\x9b{x5h\xf9\xff\xffEz{y5_\xf9\xff\xffE|{yx5\x8c\xfa\xff\xff\x11\xdb "\x02@W\x03\x00A\x9b\xf6g\xcep\x0c\x01\x02\xdb0qihA\x92]\xe81r\x0c\x0bToo long idj\xca\x01\xfd\x00\xb55\x07\xfb\xff\xffjJ\xd8&\x04E\x10\xdb!\x11\x9eihA\xe6?\x18\x84j"\x02@W\x06\x04\x0c\nNo witnessxA\xf8\'\xec\x8c5\xd7\xfa\xff\xffA\x9b\xf6g\xceph\x00r\x11\x88N\x10Q\xd0P\x12\xc0qyz\xca\xdb(\x8b\xdb(z\x8b\xdb(x\x8b\xdb(rijK\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00s\x0c\x12No token at rentalk\x10\xce\x10\xb75\x82\xfa\xff\xff{Jk\x11Q\xd0Ek7\x03\x00tijl\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84h\x00o\x11\x88N\x10Q\xd0P\x12\xc0umxy\x8b\xdb(z\x8b\xdb(l\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84@\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84@W\x05\x03A\x9b\xf6g\xceph\x00e\x11\x88N\x10Q\xd0P\x12\xc0qxy\x8b\xdb(rijK\x11\xceP\x8bP\x10\xceA\x92]\xe81sktl\x0b\x97&75\xc6\xfe\xff\xffJsEijk\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84h\x00i\x11\x88N\x10Q\xd0P\x12\xc0kj\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84\x0c\x08NEPHRENT\x0b\x11\xc0J46J\x10j\xd0kzA\xdb\xfe\xa8t5\xf4\xf6\xff\xff\xc2Jx\xcfJx\xcfJk\xcfJz\xcf\x0c\x0cTokenCreatedA\x95\x01oak"\x02@W\x00\x01@W\x03\x04A\x9b\xf6g\xceph\x00e\x11\x88N\x10Q\xd0P\x12\xc0qyz\x8b\xdb(r\x0c\x08NEPHRENT\x0b\x11\xc0J4\xd2J\x10j\xd0x{A\xdb\xfe\xa8t5\x90\xf6\xff\xff\xc2Jy\xcfJy\xcfJx\xcfJ{\xcf\x0c\x0cTokenCreatedA\x95\x01oax"\x02@W\x04\x07z}\xca\xdb(\x8b\xdb(}\x8b\xdb({\x8b\xdb(pxhK\x11\xceP\x8bP\x10\xceA\x92]\xe81qi\x0b\x98&\x1ei7\x00\x00JrEj\x10KK\xce|\x9eJTS\xd0E~Jj\x11Q\xd0E"\t~|\x12\xc0JrEj7\x03\x00sxhk\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84y{z\x8b\xdb(}\x8b\xdb(k\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84\xc2J{\xcfJz\xcfJ}\xcfJ~\xcf\x0c\x08PriceSetA\x95\x01oaj\x10\xce"\x02@W\x03\x06z}\xca\xdb(\x8b\xdb(}\x8b\xdb({\x8b\xdb(pxhK\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00qi\x10KK\xce|\x9eJTS\xd0Ei7\x03\x00rxhj\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84y{z\x8b\xdb(}\x8b\xdb(j\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84i\x10\xce"\x02@W\x04\x07\x0c\x13tokenId.Length > 64{\xca\x00@\xb65\xf8\xf7\xff\xff\x0c9Cannot register rental for tokens issued by this contractyA\xdb\xfe\xa8t\x985\xb1\xf7\xff\xff~&?\x0c\x0fTransfer failed\x0c\x08NEPHRENT{zA\xdb\xfe\xa8tx\x15\xc0\x1f\x0c\x08transferyAb}[R5s\xf7\xff\xff"?\x0c\x0fTransfer failed\x0c\x08NEPHRENT{A\xdb\xfe\xa8t\x13\xc0\x1f\x0c\x08transferyAb}[R58\xf7\xff\xff\x11J\x82EA\x9b\xf6g\xceph\x00r\x11\x88N\x10Q\xd0P\x12\xc0qh\x00o\x11\x88N\x10Q\xd0P\x12\xc0r|{zxyji5\xee\xfd\xff\xffEz{y5\xd5\xfc\xff\xffs}kzxA\xdb\xfe\xa8tji5\xd4\xfd\xff\xff"\x02@W\x00\x03zyx5\xdc\xfa\xff\xff4\x05"\x02@W\x00\x02\x0c Failed to find the token to burnx\x0c\x14\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x985\x99\xf6\xff\xffxyA\xdb\xfe\xa8t5s\xf4\xff\xffx"\x02@\x0c\x14\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@W\x03\x06z}\xca\xdb(\x8b\xdb(}\x8b\xdb({\x8b\xdb(pxhK\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00qi\x10KK\xce|\x9fJTS\xd0Ei\x10\xceJ\x84E\x0c\x1fNo enough token to unregister: z\x8b\xdb({\x8b\xdb(|7\x04\x00\x8b\xdb(}\x8b\xdb(|\x10\xb85\xfd\xf5\xff\xff|\x10\xb7&3i7\x03\x00rxhj\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84y{z\x8b\xdb(}\x8b\xdb(j\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84"(xhK\x11\xceP\x8bP\x10\xceA/X\xc5\xedy{z\x8b\xdb(}\x8b\xdb(K\x11\xceP\x8bP\x10\xceA/X\xc5\xed|"\x02@W\x04\x05\x0c\nNo witnessxA\xf8\'\xec\x8c5\x83\xf5\xff\xff\x0c;Cannot unregister rental for tokens issued by this contractyA\xdb\xfe\xa8t\x985:\xf5\xff\xffA\x9b\xf6g\xceph\x00r\x11\x88N\x10Q\xd0P\x12\xc0qh\x00o\x11\x88N\x10Q\xd0P\x12\xc0r|&V{zxyji5\x9f\xfe\xff\xffEz{y5\x1d\xfe\xff\xffJsE\x0c\x0fTransfer failed\x0c\x08NEPHRENT{zxA\xdb\xfe\xa8t\x15\xc0\x1f\x0c\x08transferyAb}[R5\xc5\xf4\xff\xff"R\x11J\x82E{zxyji5G\xfe\xff\xffEz{y5\xc5\xfd\xff\xffJsE\x0c\x0fTransfer failed\x0c\x08NEPHRENT{x\x13\xc0\x1f\x0c\x08transferyAb}[R5s\xf4\xff\xff\xc2Jy\xcfJ{\xcfJz\xcf\x0c\x0eTokenWithdrawnA\x95\x01oakzxA\xdb\xfe\xa8tji5\xd8\xfd\xff\xff"\x02@W\x00\x02\x00dxy\xa0\x01\xe8\x03\xa15S\xf8\xff\xff@W\x00\x01x\x01\x88\x13\xa0\x01\x10\'\xa1@W\x00\x02yx4\xdc\x01\x88\x13\xa0\x01\x10\'\xa1@W\x00\x06}A\x9b\xf6g\xce\x00e\x11\x88N\x10Q\xd0P\x12\xc0{|\x8b\xdb(K\x11\xceP\x8bP\x10\xceA\x92]\xe81zyx4\x05"\x02@W\x0b\x05A\xdb\xfe\xa8tpA\xb7\xc3\x88\x03q\x0c\x14Too long borrow time|\x02\x00\x98\x7f3\xb65\xb4\xf3\xff\xffA\x9b\xf6g\xcerj\x00r\x11\x88N\x10Q\xd0P\x12\xc0sh{\xca\xdb(\x8b\xdb({\x8b\xdb(x\x8b\xdb(tklK\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00u|m\x11\xce57\xff\xff\xffvn5B\xff\xff\xffw\x07\x0c\x11Failed to pay GAS\x0c\x08NEPHRENTnxy\x14\xc0\x1f\x0c\x08transfer\x0c\x14\xcfv\xe2\x8b\xd0\x06,JG\x8e\xe3Ua\x01\x13\x19\xf3\xcf\xa4\xd2Ab}[R5\x1e\xf3\xff\xff\x0c\x18Failed to pay collateral\x0c\x08NEPHRENTo\x07hy\x14\xc0\x1f\x0c\x08transfer\x0c\x14\xcfv\xe2\x8b\xd0\x06,JG\x8e\xe3Ua\x01\x13\x19\xf3\xcf\xa4\xd2Ab}[R5\xc9\xf2\xff\xffm\x10KK\xcez\x9fJTS\xd0E\x0c\x17No enough token to lendm\x10\xce\x10\xb85\x9a\xf2\xff\xffm\x10\xce\x10\xb7&Cm7\x03\x00Jw\x08Eklo\x08\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84j\x00o\x11\x88N\x10Q\xd0P\x12\xc0xh\x8b\xdb({\x8b\xdb(o\x08\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84"3klK\x11\xceP\x8bP\x10\xceA/X\xc5\xedj\x00o\x11\x88N\x10Q\xd0P\x12\xc0xh\x8b\xdb({\x8b\xdb(K\x11\xceP\x8bP\x10\xceA/X\xc5\xedj\x00d\x11\x88N\x10Q\xd0P\x12\xc0w\tx{\xca\xdb(\x8b\xdb({\x8b\xdb(y\x8b\xdb(i\xdb(\x8b\xdb(JtE\x0c%Cannot borrow twice in a single blocko\tlK\x11\xceP\x8bP\x10\xceA\x92]\xe81w\no\n\x0b\x975\xb8\xf1\xff\xff\x11i|\x9eo\x07z\x14\xc0w\no\n7\x03\x00Jw\x08Eo\tlo\x08\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84j\x00t\x11\x88N\x10Q\xd0P\x12\xc0y{\xca\xdb(\x8b\xdb({\x8b\xdb(x\x8b\xdb(i\xdb(\x8b\xdb(o\x08\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84\x0c\x08NEPHRENT{zyA\xdb\xfe\xa8t5w\xf5\xff\xff9\xc2Jx\xcfJA\xdb\xfe\xa8t\xcfJ{\xcfJy\xcfJi\xcfJz\xcfJn\xcfJo\x07\xcfJo\n\x12\xce\xcf\x0c\x0bTokenRentedA\x95\x01oai"\x02@A\xb7\xc3\x88\x03@\x0c\x14\xcfv\xe2\x8b\xd0\x06,JG\x8e\xe3Ua\x01\x13\x19\xf3\xcf\xa4\xd2@W\x06\x04\x0c\x16No witness from renterxA\xf8\'\xec\x8c5\xca\xf0\xff\xffA\x9b\xf6g\xceph\x00d\x11\x88N\x10Q\xd0P\x12\xc0qxy\xca\xdb(\x8b\xdb(y\x8b\xdb(z\x8b\xdb({\x8b\xdb(rijK\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00s\x10Jk\x13Q\xd0Ek7\x03\x00tlJjiS\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84Eh\x00t\x11\x88N\x10Q\xd0P\x12\xc0ulJzy\xca\xdb(\x8b\xdb(y\x8b\xdb(x\x8b\xdb({\x8b\xdb(mS\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84E\xc2Jx\xcfJy\xcfJz\xcfJ{J\xd8&\x04E\x10\xdb!\xcfJk\x10\xce\xcfJk\x11\xce\xcfJk\x12\xce\xcf\x0c\x0cRentalClosedA\x95\x01oa@W\x06\x04\x0c\x16No witness from renterxA\xf8\'\xec\x8c5\xdd\xef\xff\xffA\x9b\xf6g\xceph\x00d\x11\x88N\x10Q\xd0P\x12\xc0qxy\xca\xdb(\x8b\xdb(y\x8b\xdb(z\x8b\xdb({\x8b\xdb(rijK\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00s\x11Jk\x13Q\xd0Ek7\x03\x00tlJjiS\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84Eh\x00t\x11\x88N\x10Q\xd0P\x12\xc0ulJzy\xca\xdb(\x8b\xdb(y\x8b\xdb(x\x8b\xdb({\x8b\xdb(mS\x12M\x11\xceQ\x8bQ\x10\xceA\xe6?\x18\x84E\xc2Jx\xcfJy\xcfJz\xcfJ{J\xd8&\x04E\x10\xdb!\xcfJk\x10\xce\xcfJk\x11\xce\xcfJk\x12\xce\xcf\x0c\x0cRentalOpenedA\x95\x01oa@W\x01\x07A\x9b\xf6g\xce\x00e\x11\x88N\x10Q\xd0P\x12\xc0z{\x8b\xdb(K\x11\xceP\x8bP\x10\xceA\x92]\xe81p~}|{zhyx4\x03@W\x07\x08A\x9b\xf6g\xceph\x00t\x11\x88N\x10Q\xd0P\x12\xc0qyz\xca\xdb(\x8b\xdb(z\x8b\xdb(x\x8b\xdb(}\x8b\xdb(rijK\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00sA\xb7\xc3\x88\x03k\x12\xce\xb6&=\x0c.Rental not expired. Need signature from tenantyA\xf8\'\xec\x8c5a\xee\xff\xffijK\x11\xceP\x8bP\x10\xceA/X\xc5\xedh\x00d\x11\x88N\x10Q\xd0P\x12\xc0xz\xca\xdb(\x8b\xdb(z\x8b\xdb(y\x8b\xdb(}\x8b\xdb(K\x11\xceP\x8bP\x10\xceA/X\xc5\xedk\x10\xcetzly5\xff\xeb\xff\xffk\x13\xce\x10\x97\'\xce\x00\x00\x00h\x00r\x11\x88N\x10Q\xd0P\x12\xc0uh\x00o\x11\x88N\x10Q\xd0P\x12\xc0v\x7f\x07&K|lx{nm5y\xf7\xff\xffE\x0c\x0fTransfer failed\x0c\x08NEPHRENT|lxA\xdb\xfe\xa8t\x15\xc0\x1f\x0c\x08transfer{Ab}[R5\xaa\xed\xff\xff"C|lx{nm50\xf7\xff\xffE\x0c\x0fTransfer failed\x0c\x08NEPHRENT|x\x13\xc0\x1f\x0c\x08transfer{Ab}[R5g\xed\xff\xff\xc2J{\xcfJ|\xcfJl\xcf\x0c\x0eTokenWithdrawnA\x95\x01oa"6k\x10\xce|{z5\xc1\xf3\xff\xffEzk\x10\xcexA\xdb\xfe\xa8th\x00o\x11\x88N\x10Q\xd0P\x12\xc0h\x00r\x11\x88N\x10Q\xd0P\x12\xc05\x96\xf4\xff\xffE\xc2Jx\xcfJz\xcfJy\xcfJ}J\xd8&\x04E\x10\xdb!\xcfJk\x10\xce\xcfJk\x11\xce\xcfJk\x12\xce\xcf\x0c\rRentalRevokedA\x95\x01oa\x0c\x08NEPHRENTk\x11\xce~A\xdb\xfe\xa8t\x14\xc0\x1f\x0c\x08transfer\x0c\x14\xcfv\xe2\x8b\xd0\x06,JG\x8e\xe3Ua\x01\x13\x19\xf3\xcf\xa4\xd2Ab}[R9@W\x01\x06A\x9b\xf6g\xce\x00i\x11\x88N\x10Q\xd0P\x12\xc0zK\x11\xceP\x8bP\x10\xceA\x92]\xe81\xdb0p}|{hJ\xcaP\x00\x14QK\x9f\x8c\xdb(h\x00\x14P\x10QK\x9f\x8c\xdb(J\xd8$\tJ\xca\x00\x14(\x03:zyx5l\xfd\xff\xff@\xdb0@\xdb(@\xdb(J\xd8$\tJ\xca\x00\x14(\x03:@W\t\x08A\x9b\xf6g\xceph\x00r\x11\x88N\x10Q\xd0P\x12\xc0q{\x0c\x14\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x97\'9\x01\x00\x00\x12yz\xca\xdb(\x8b\xdb(z\x8b\xdb(iN\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9as\x10tkA\x9c\x08\xed\x9c$\x07\x10\xdb "\x05l|\xb5\'\xe1\x00\x00\x00kA\xf3T\xbf\x1dw\x07o\x07\x10\xceJ\xd8$\tJ\xca\x00\x14(\x03:J\x83Eo\x07\x11\xce7\x00\x00w\x08o\x08\x10\xceJvE|l\x9fJuEnm\xb7&\x06mJvEln\x9eJtE\x00dno\x08\x11\xce\xa05\xa7\xef\xff\xffJrE\x0c\x13GAS transfer failed\x0c\x08NEPHRENTj{x\x14\xc0\x1f\x0c\x08transfer\x0c\x14\xcfv\xe2\x8b\xd0\x06,JG\x8e\xe3Ua\x01\x13\x19\xf3\xcf\xa4\xd2Ab}[R53\xeb\xff\xff\xc2J{\xcfJy\xcfJz\xcfJx\xcfJA\xb7\xc3\x88\x03\xcfJo\x08\x10\xce\xcfJj\xcfJ\x10\xcfJA\xb7\xc3\x88\x03\xcf\x0c\x0bTokenRentedA\x95\x01oa#\x14\xff\xff\xff\x0c\x16No enough NFTs to rentl|\x975\xd5\xea\xff\xff#\xe1\x00\x00\x00iyz\xca\xdb(\x8b\xdb(z\x8b\xdb({\x8b\xdb(K\x11\xceP\x8bP\x10\xceA\x92]\xe817\x00\x00s\x0c\x16No enough NFTs to rentk\x10\xce|\xb85\x8c\xea\xff\xff\x00dk\x10\xcek\x11\xce\xa05\xa0\xee\xff\xffJrE\x0c\x13GAS transfer failed\x0c\x08NEPHRENTj{x\x14\xc0\x1f\x0c\x08transfer\x0c\x14\xcfv\xe2\x8b\xd0\x06,JG\x8e\xe3Ua\x01\x13\x19\xf3\xcf\xa4\xd2Ab}[R5,\xea\xff\xff\xc2J{\xcfJy\xcfJz\xcfJx\xcfJA\xb7\xc3\x88\x03\xcfJk\x10\xce\xcfJj\xcfJ\x10\xcfJA\xb7\xc3\x88\x03\xcf\x0c\x0bTokenRentedA\x95\x01oa\x0c\x13NFT transfer failed\x0c\x08NEPHRENTz|xA\xdb\xfe\xa8t\x15\xc0\x1f\x0c\x08transferyAb}[R5\xb5\xe9\xff\xff\x7f\x07\x1f~}Ab}[Rs\x0c\x12NFT payback failed\x0c\x08NEPHRENTz|A\xdb\xfe\xa8tx\x15\xc0\x1f\x0c\x08transferyAb}[R5l\xe9\xff\xffk"\x02@A\x9c\x08\xed\x9c@A\xf3T\xbf\x1d@W\x05\x07A\x9b\xf6g\xce\x00r\x11\x88N\x10Q\xd0P\x12\xc0p\x12yz\xca\xdb(\x8b\xdb(z\x8b\xdb(hN\x11\xceP\x8bP\x10\xceA\xdf0\xb8\x9aq\x0c\x15Failed to find renteriA\x9c\x08\xed\x9c5\n\xe9\xff\xffiA\xf3T\xbf\x1d7\x00\x00r\x00dj\x11\xce5\x18\xed\xff\xffs\x0c\x13GAS transfer failed\x0c\x08NEPHRENTk{x\x14\xc0\x1f\x0c\x08transfer\x0c\x14\xcfv\xe2\x8b\xd0\x06,JG\x8e\xe3Ua\x01\x13\x19\xf3\xcf\xa4\xd2Ab}[R5\xa6\xe8\xff\xff\xc2J{\xcfJy\xcfJz\xcfJx\xcfJA\xb7\xc3\x88\x03\xcfJj\x10\xce\xcfJj\x10\xcek\xa0\xcfJ\x10\xcfJA\xb7\xc3\x88\x03\xcf\x0c\x0bTokenRentedA\x95\x01oa\x0c\x13NFT transfer failed\x0c\x08NEPHRENTzx\x13\xc0\x1f\x0c\x08transferyAb}[R51\xe8\xff\xff~\x1f}|Ab}[Rt\x0c\x12NFT payback failed\x0c\x08NEPHRENTzA\xdb\xfe\xa8t\x13\xc0\x1f\x0c\x08transferyAb}[R5\xeb\xe7\xff\xffl"\x02@V\x02\n\x0b\xe2\xff\xff\n*\xdf\xff\xff\n\xfe\xde\xff\xff\x13\xc0`\n\xf9\xe1\xff\xff\n\x18\xdf\xff\xff\x0b\x13\xc0a@\xc2JX\xcfJ5\xf0\xde\xff\xff#\xdd\xde\xff\xff\xc2JY\xcfJ5\xe8\xde\xff\xff#\xf5\xde\xff\xff\xc2JY\xcfJ5\xd9\xde\xff\xff#\xc2\xe1\xff\xff\x90z\xe1\xc6', '{"name":"AnyUpdateShortSafe","groups":[],"features":{},"supportedstandards":["NEP-11"],"abi":{"methods":[{"name":"symbol","offset":8473,"safe":false,"returntype":"String","parameters":[]},{"name":"decimals","offset":8488,"safe":false,"returntype":"Integer","parameters":[]},{"name":"totalSupply","offset":414,"safe":true,"returntype":"Integer","parameters":[]},{"name":"ownerOf","offset":438,"safe":true,"returntype":"InteropInterface","parameters":[{"name":"tokenId","type":"ByteArray"}]},{"name":"balanceOf","offset":529,"safe":true,"returntype":"Integer","parameters":[{"name":"owner","type":"Hash160"}]},{"name":"balanceOf","offset":634,"safe":true,"returntype":"Integer","parameters":[{"name":"owner","type":"Hash160"},{"name":"tokenId","type":"ByteArray"}]},{"name":"properties","offset":8503,"safe":true,"returntype":"Map","parameters":[{"name":"tokenId","type":"ByteArray"}]},{"name":"tokens","offset":837,"safe":true,"returntype":"InteropInterface","parameters":[]},{"name":"tokensOf","offset":884,"safe":true,"returntype":"InteropInterface","parameters":[{"name":"owner","type":"Hash160"}]},{"name":"_deploy","offset":2174,"safe":false,"returntype":"Void","parameters":[{"name":"data","type":"Any"},{"name":"update","type":"Boolean"}]},{"name":"onNEP11Payment","offset":2199,"safe":false,"returntype":"Void","parameters":[{"name":"from","type":"Hash160"},{"name":"amount","type":"Integer"},{"name":"tokenId","type":"ByteArray"},{"name":"data","type":"Any"}]},{"name":"onNEP17Payment","offset":2291,"safe":false,"returntype":"Void","parameters":[{"name":"from","type":"Hash160"},{"name":"amount","type":"Integer"},{"name":"data","type":"Any"}]},{"name":"getDecimals","offset":2358,"safe":false,"returntype":"Integer","parameters":[{"name":"externalTokenContract","type":"Hash160"}]},{"name":"balanceOfRentalToken","offset":2380,"safe":false,"returntype":"Integer","parameters":[{"name":"internalTokenId","type":"ByteArray"}]},{"name":"listRegisteredRentalByToken","offset":2395,"safe":false,"returntype":"InteropInterface","parameters":[]},{"name":"listRegisteredRentalByToken","offset":2424,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"externalTokenContract","type":"Hash160"}]},{"name":"listRegisteredRentalByToken","offset":2459,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"externalTokenContract","type":"Hash160"},{"name":"externalTokenId","type":"ByteArray"}]},{"name":"getRegisteredRentalByToken","offset":2505,"safe":false,"returntype":"Array","parameters":[{"name":"externalTokenContract","type":"Hash160"},{"name":"externalTokenId","type":"ByteArray"},{"name":"renter","type":"Hash160"}]},{"name":"listRegisteredRentalByRenter","offset":2557,"safe":false,"returntype":"InteropInterface","parameters":[]},{"name":"listRegisteredRentalByRenter","offset":2586,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"renter","type":"Hash160"}]},{"name":"listRegisteredRentalByRenter","offset":2621,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"renter","type":"Hash160"},{"name":"externalTokenContract","type":"Hash160"}]},{"name":"getRegisteredRentalByRenter","offset":2660,"safe":false,"returntype":"Array","parameters":[{"name":"renter","type":"Hash160"},{"name":"externalTokenContract","type":"Hash160"},{"name":"externalTokenId","type":"ByteArray"}]},{"name":"listRentalDeadlineByRenter","offset":2705,"safe":false,"returntype":"InteropInterface","parameters":[]},{"name":"listRentalDeadlineByRenter","offset":2734,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"renter","type":"Hash160"}]},{"name":"listRentalDeadlineByRenter","offset":2769,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"renter","type":"Hash160"},{"name":"internalTokenId","type":"ByteArray"}]},{"name":"listRentalDeadlineByRenter","offset":2815,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"renter","type":"Hash160"},{"name":"internalTokenId","type":"ByteArray"},{"name":"tenant","type":"Hash160"}]},{"name":"getRentalDeadlineByRenter","offset":2865,"safe":false,"returntype":"Array","parameters":[{"name":"renter","type":"Hash160"},{"name":"internalTokenId","type":"ByteArray"},{"name":"tenant","type":"Hash160"},{"name":"startTime","type":"Integer"}]},{"name":"listRentalDeadlineByTenant","offset":2923,"safe":false,"returntype":"InteropInterface","parameters":[]},{"name":"listRentalDeadlineByTenant","offset":2952,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"tenant","type":"Hash160"}]},{"name":"listRentalDeadlineByTenant","offset":2987,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"tenant","type":"Hash160"},{"name":"internalTokenId","type":"ByteArray"}]},{"name":"listRentalDeadlineByTenant","offset":3033,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"tenant","type":"Hash160"},{"name":"internalTokenId","type":"ByteArray"},{"name":"renter","type":"Hash160"}]},{"name":"getRentalDeadlineByTenant","offset":3083,"safe":false,"returntype":"Array","parameters":[{"name":"tenant","type":"Hash160"},{"name":"internalTokenId","type":"ByteArray"},{"name":"renter","type":"Hash160"},{"name":"startTime","type":"Integer"}]},{"name":"listExternalTokenInfo","offset":3141,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"prefix","type":"ByteArray"}]},{"name":"getExternalTokenInfo","offset":3176,"safe":false,"returntype":"ByteArray","parameters":[{"name":"internalTokenId","type":"ByteArray"}]},{"name":"listInternalTokenId","offset":3210,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"prefix","type":"ByteArray"}]},{"name":"listInternalTokenId","offset":3245,"safe":false,"returntype":"InteropInterface","parameters":[{"name":"externalTokenContract","type":"Hash160"},{"name":"prefix","type":"ByteArray"}]},{"name":"getInternalTokenId","offset":3284,"safe":false,"returntype":"ByteArray","parameters":[{"name":"externalTokenContract","type":"Hash160"},{"name":"externalTokenId","type":"ByteArray"}]},{"name":"max","offset":3322,"safe":false,"returntype":"Integer","parameters":[{"name":"v1","type":"Integer"},{"name":"v2","type":"Integer"}]},{"name":"transfer","offset":3335,"safe":false,"returntype":"Boolean","parameters":[{"name":"from","type":"Hash160"},{"name":"to","type":"Hash160"},{"name":"amount","type":"Integer"},{"name":"tokenId","type":"ByteArray"},{"name":"data","type":"Any"}]},{"name":"setRentalPrice","offset":3564,"safe":false,"returntype":"Void","parameters":[{"name":"renter","type":"Hash160"},{"name":"tokenContract","type":"Hash160"},{"name":"tokenId","type":"ByteArray"},{"name":"price","type":"Integer"}]},{"name":"registerRental","offset":4291,"safe":false,"returntype":"Integer","parameters":[{"name":"renter","type":"Hash160"},{"name":"externalTokenContract","type":"Hash160"},{"name":"amountForRent","type":"Integer"},{"name":"externalTokenId","type":"ByteArray"},{"name":"flashLoanPrice","type":"Integer"},{"name":"ordinaryLoanPrice","type":"Integer"},{"name":"isDivisible","type":"Boolean"}]},{"name":"unregisterRental","offset":4928,"safe":false,"returntype":"Integer","parameters":[{"name":"renter","type":"Hash160"},{"name":"externalTokenContract","type":"Hash160"},{"name":"amountToUnregister","type":"Integer"},{"name":"externalTokenId","type":"ByteArray"},{"name":"isDivisible","type":"Boolean"}]},{"name":"getTotalPrice","offset":5275,"safe":false,"returntype":"Integer","parameters":[{"name":"pricePerSecond","type":"Integer"},{"name":"borrowTimeMilliseconds","type":"Integer"}]},{"name":"getCollateralAmount","offset":5293,"safe":false,"returntype":"Integer","parameters":[{"name":"totalPrice","type":"Integer"}]},{"name":"getCollateralAmount","offset":5306,"safe":false,"returntype":"Integer","parameters":[{"name":"pricePerSecond","type":"Integer"},{"name":"borrowTimeMilliseconds","type":"Integer"}]},{"name":"borrow","offset":5322,"safe":false,"returntype":"Integer","parameters":[{"name":"renter","type":"Hash160"},{"name":"tenant","type":"Hash160"},{"name":"amount","type":"Integer"},{"name":"externalTokenContract","type":"Hash160"},{"name":"externalTokenId","type":"ByteArray"},{"name":"borrowTimeMilliseconds","type":"Integer"}]},{"name":"borrow","offset":5368,"safe":false,"returntype":"Integer","parameters":[{"name":"renter","type":"Hash160"},{"name":"tenant","type":"Hash160"},{"name":"amount","type":"Integer"},{"name":"internalTokenId","type":"ByteArray"},{"name":"borrowTimeMilliseconds","type":"Integer"}]},{"name":"closeNextRental","offset":6125,"safe":false,"returntype":"Void","parameters":[{"name":"renter","type":"Hash160"},{"name":"internalTokenId","type":"ByteArray"},{"name":"tenant","type":"Hash160"},{"name":"startTime","type":"ByteArray"}]},{"name":"openNextRental","offset":6362,"safe":false,"returntype":"Void","parameters":[{"name":"renter","type":"Hash160"},{"name":"internalTokenId","type":"ByteArray"},{"name":"tenant","type":"Hash160"},{"name":"startTime","type":"ByteArray"}]},{"name":"payback","offset":6599,"safe":false,"returntype":"Void","parameters":[{"name":"renter","type":"Hash160"},{"name":"tenant","type":"Hash160"},{"name":"externalTokenContract","type":"Hash160"},{"name":"externalTokenId","type":"ByteArray"},{"name":"startTime","type":"ByteArray"},{"name":"collateralReceiver","type":"Hash160"},{"name":"isDivisible","type":"Boolean"}]},{"name":"payback","offset":7232,"safe":false,"returntype":"Void","parameters":[{"name":"renter","type":"Hash160"},{"name":"tenant","type":"Hash160"},{"name":"internalTokenId","type":"ByteArray"},{"name":"startTime","type":"ByteArray"},{"name":"collateralReceiver","type":"Hash160"},{"name":"isDivisible","type":"Boolean"}]},{"name":"flashBorrowDivisible","offset":7334,"safe":false,"returntype":"Any","parameters":[{"name":"tenant","type":"Hash160"},{"name":"token","type":"Hash160"},{"name":"tokenId","type":"ByteArray"},{"name":"renter","type":"Hash160"},{"name":"neededAmount","type":"Integer"},{"name":"renterCalledContract","type":"Hash160"},{"name":"renterCalledMethod","type":"String"},{"name":"arguments","type":"Array"}]},{"name":"flashBorrowNonDivisible","offset":8065,"safe":false,"returntype":"Any","parameters":[{"name":"tenant","type":"Hash160"},{"name":"token","type":"Hash160"},{"name":"tokenId","type":"ByteArray"},{"name":"renter","type":"Hash160"},{"name":"calledContract","type":"Hash160"},{"name":"calledMethod","type":"String"},{"name":"arguments","type":"Array"}]},{"name":"_initialize","offset":8438,"safe":false,"returntype":"Void","parameters":[]}],"events":[{"name":"Transfer","parameters":[{"name":"from","type":"Hash160"},{"name":"to","type":"Hash160"},{"name":"amount","type":"Integer"},{"name":"tokenId","type":"ByteArray"}]},{"name":"TokenCreated","parameters":[{"name":"arg1","type":"Hash160"},{"name":"arg2","type":"ByteArray"},{"name":"arg3","type":"ByteArray"},{"name":"arg4","type":"Integer"}]},{"name":"PriceSet","parameters":[{"name":"arg1","type":"Hash160"},{"name":"arg2","type":"Hash160"},{"name":"arg3","type":"ByteArray"},{"name":"arg4","type":"Integer"}]},{"name":"TokenRented","parameters":[{"name":"arg1","type":"Hash160"},{"name":"arg2","type":"Hash160"},{"name":"arg3","type":"ByteArray"},{"name":"arg4","type":"Hash160"},{"name":"arg5","type":"Integer"},{"name":"arg6","type":"Integer"},{"name":"arg7","type":"Integer"},{"name":"arg8","type":"Integer"},{"name":"arg9","type":"Integer"}]},{"name":"RentalClosed","parameters":[{"name":"arg1","type":"Hash160"},{"name":"arg2","type":"ByteArray"},{"name":"arg3","type":"Hash160"},{"name":"arg4","type":"Integer"},{"name":"arg5","type":"Integer"},{"name":"arg6","type":"Integer"},{"name":"arg7","type":"Integer"}]},{"name":"RentalOpened","parameters":[{"name":"arg1","type":"Hash160"},{"name":"arg2","type":"ByteArray"},{"name":"arg3","type":"Hash160"},{"name":"arg4","type":"Integer"},{"name":"arg5","type":"Integer"},{"name":"arg6","type":"Integer"},{"name":"arg7","type":"Integer"}]},{"name":"RentalRevoked","parameters":[{"name":"arg1","type":"Hash160"},{"name":"arg2","type":"ByteArray"},{"name":"arg3","type":"Hash160"},{"name":"arg4","type":"Integer"},{"name":"arg5","type":"Integer"},{"name":"arg6","type":"Integer"},{"name":"arg7","type":"Integer"}]},{"name":"TokenWithdrawn","parameters":[{"name":"arg1","type":"Hash160"},{"name":"arg2","type":"ByteArray"},{"name":"arg3","type":"Integer"}]}]},"permissions":[{"contract":"*","methods":"*"}],"trusts":[],"extra":{"Author":"Hecate2","Email":"developer@neo.org","Description":"NFTFlashLoan"}}', 'registerRental', [0xb1983fa2479a0c8e2beae032d2df564b5451b7a5, 0x2704f0fa22877b2e0a53afe858b8db971f348b91, 68, 1, 5, 7, True]] relay=True [{'account': '0xb1983fa2479a0c8e2beae032d2df564b5451b7a5', 'scopes': 'Global', 'allowedcontracts': [], 'allowedgroups': [], 'rules': []}]
{"jsonrpc":"2.0","method":"invokefunctionwithsession","params":["Hello world! Your first contact with Fairy!",true,"0x5c1068339fae89eb1a743909d0213e1d99dc5dc9","anyUpdate",[{"type":"ByteArray","value":"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"},{"type":"String","value":"{\"name\":\"AnyUpdateShortSafe\",\"groups\":[],\"features\":{},\"supportedstandards\":[\"NEP-11\"],\"abi\":{\"methods\":[{\"name\":\"symbol\",\"offset\":8473,\"safe\":false,\"returntype\":\"String\",\"parameters\":[]},{\"name\":\"decimals\",\"offset\":8488,\"safe\":false,\"returntype\":\"Integer\",\"parameters\":[]},{\"name\":\"totalSupply\",\"offset\":414,\"safe\":true,\"returntype\":\"Integer\",\"parameters\":[]},{\"name\":\"ownerOf\",\"offset\":438,\"safe\":true,\"returntype\":\"InteropInterface\",\"parameters\":[{\"name\":\"tokenId\",\"type\":\"ByteArray\"}]},{\"name\":\"balanceOf\",\"offset\":529,\"safe\":true,\"returntype\":\"Integer\",\"parameters\":[{\"name\":\"owner\",\"type\":\"Hash160\"}]},{\"name\":\"balanceOf\",\"offset\":634,\"safe\":true,\"returntype\":\"Integer\",\"parameters\":[{\"name\":\"owner\",\"type\":\"Hash160\"},{\"name\":\"tokenId\",\"type\":\"ByteArray\"}]},{\"name\":\"properties\",\"offset\":8503,\"safe\":true,\"returntype\":\"Map\",\"parameters\":[{\"name\":\"tokenId\",\"type\":\"ByteArray\"}]},{\"name\":\"tokens\",\"offset\":837,\"safe\":true,\"returntype\":\"InteropInterface\",\"parameters\":[]},{\"name\":\"tokensOf\",\"offset\":884,\"safe\":true,\"returntype\":\"InteropInterface\",\"parameters\":[{\"name\":\"owner\",\"type\":\"Hash160\"}]},{\"name\":\"_deploy\",\"offset\":2174,\"safe\":false,\"returntype\":\"Void\",\"parameters\":[{\"name\":\"data\",\"type\":\"Any\"},{\"name\":\"update\",\"type\":\"Boolean\"}]},{\"name\":\"onNEP11Payment\",\"offset\":2199,\"safe\":false,\"returntype\":\"Void\",\"parameters\":[{\"name\":\"from\",\"type\":\"Hash160\"},{\"name\":\"amount\",\"type\":\"Integer\"},{\"name\":\"tokenId\",\"type\":\"ByteArray\"},{\"name\":\"data\",\"type\":\"Any\"}]},{\"name\":\"onNEP17Payment\",\"offset\":2291,\"safe\":false,\"returntype\":\"Void\",\"parameters\":[{\"name\":\"from\",\"type\":\"Hash160\"},{\"name\":\"amount\",\"type\":\"Integer\"},{\"name\":\"data\",\"type\":\"Any\"}]},{\"name\":\"getDecimals\",\"offset\":2358,\"safe\":false,\"returntype\":\"Integer\",\"parameters\":[{\"name\":\"externalTokenContract\",\"type\":\"Hash160\"}]},{\"name\":\"balanceOfRentalToken\",\"offset\":2380,\"safe\":false,\"returntype\":\"Integer\",\"parameters\":[{\"name\":\"internalTokenId\",\"type\":\"ByteArray\"}]},{\"name\":\"listRegisteredRentalByToken\",\"offset\":2395,\"safe\":false,\"returntype\":\"InteropInterface\",\"parameters\":[]},{\"name\":\"listRegisteredRentalByToken\",\"offset\":2424,\"safe\":false,\"returntype\":\"InteropInterface\",\"parameters\":[{\"name\":\"externalTokenContract\",\"type\":\"Hash160\"}]},{\"name\":\"listRegisteredRentalByToken\",\"offset\":2459,\"safe\":false,\"returntype\":\"InteropInterface\",\"parameters\":[{\"name\":\"externalTokenContract\",\"type\":\"Hash160\"},{\"name\":\"externalTokenId\",\"type\":\"ByteArray\"}]},{\"name\":\"getRegisteredRentalByToken\",\"offset\":2505,\"safe\":false,\"returntype\":\"Array\",\"parameters\":[{\"name\":\"externalTokenContract\",\"type\":\"Hash160\"},{\"name\":\"externalTokenId\",\"type\":\"ByteArray\"},{\"name\":\"renter\",\"type\":\"Hash160\"}]},{\"name\":\"listRegisteredRentalByRenter\",\"offset\":2557,\"safe\":false,\"returntype\":\"InteropInterface\",\"parameters\":[]},{\"name\":\"listRegisteredRentalByRenter\",\"offset\":2586,\"safe\":false,\"returntype\":\"InteropInterface\",\"parameters\":[{\"name\":\"renter\",\"type\":\"Hash160\"}]},{\"name\":\"listRegisteredRentalByRenter\",\"offset\":2621,\"safe\":false,\"returntype\":\"InteropInterface\",\"parameters\":[{\"name\":\"renter\",\"type\":\"Hash160\"},{\"name\":\"externalTokenContract\",\"type\":\"Hash160\"}]},{\"name\":\"getRegisteredRentalByRenter\",\"offset\":2660,\"safe\":false,\"returntype\":\"Array\",\"parameters\":[{\"name\":\"renter\",\"type\":\"Hash160\"},{\"name\":\"externalTokenContract\",\"type\":\"Hash160\"},{\"name\":\"externalTokenId\",\"type\":\"ByteArray\"}]},{\"name\":\"listRentalDeadlineByRenter\",\"offset\":2705,\"safe\":false,\"returntype\":\"InteropInterface\",\"parameters\":[]},{\"name\":\"listRentalDeadlineByRenter\",\"offset\":2734,\"safe\":false,\"returntype\":\"InteropInterface\",\"parameters\":[{\"name\":\"renter\",\"type\":\"Hash160\"}]},{\"name\":\"listRentalDeadlineByRenter\",\"offset\":2769,\"safe\":false,\"returntype\":\"InteropInterface\",\"parameters\":[{\"name\":\"renter\",\"type\":\"Hash160\"},{\"name\":\"internalTokenId\",\"type\":\"ByteArray\"}]},{\"name\":\"listRentalDeadlineByRenter\",\"offset\":2815,\"safe\":false,\"returntype\":\"InteropInterface\",\"parameters\":[{\"name\":\"renter\",\"type\":\"Hash160\"},{\"name\":\"internalTokenId\",\"type\":\"ByteArray\"},{\"name\":\"tenant\",\"type\":\"Hash160\"}]},{\"name\":\"getRentalDeadlineByRenter\",\"offset\":2865,\"safe\":false,\"returntype\":\"Array\",\"parameters\":[{\"name\":\"renter\",\"type\":\"Hash160\"},{\"name\":\"internalTokenId\",\"type\":\"ByteArray\"},{\"name\":\"tenant\",\"type\":\"Hash160\"},{\"name\":\"startTime\",\"type\":\"Integer\"}]},{\"name\":\"listRentalDeadlineByTenant\",\"offset\":2923,\"safe\":false,\"returntype\":\"InteropInterface\",\"parameters\":[]},{\"name\":\"listRentalDeadlineByTenant\",\"offset\":2952,\"safe\":false,\"returntype\":\"InteropInterface\",\"parameters\":[{\"name\":\"tenant\",\"type\":\"Hash160\"}]},{\"name\":\"listRentalDeadlineByTenant\",\"offset\":2987,\"safe\":false,\"returntype\":\"InteropInterface\",\"parameters\":[{\"name\":\"tenant\",\"type\":\"Hash160\"},{\"name\":\"internalTokenId\",\"type\":\"ByteArray\"}]},{\"name\":\"listRentalDeadlineByTenant\",\"offset\":3033,\"safe\":false,\"returntype\":\"InteropInterface\",\"parameters\":[{\"name\":\"tenant\",\"type\":\"Hash160\"},{\"name\":\"internalTokenId\",\"type\":\"ByteArray\"},{\"name\":\"renter\",\"type\":\"Hash160\"}]},{\"name\":\"getRentalDeadlineByTenant\",\"offset\":3083,\"safe\":false,\"returntype\":\"Array\",\"parameters\":[{\"name\":\"tenant\",\"type\":\"Hash160\"},{\"name\":\"internalTokenId\",\"type\":\"ByteArray\"},{\"name\":\"renter\",\"type\":\"Hash160\"},{\"name\":\"startTime\",\"type\":\"Integer\"}]},{\"name\":\"listExternalTokenInfo\",\"offset\":3141,\"safe\":false,\"returntype\":\"InteropInterface\",\"parameters\":[{\"name\":\"prefix\",\"type\":\"ByteArray\"}]},{\"name\":\"getExternalTokenInfo\",\"offset\":3176,\"safe\":false,\"returntype\":\"ByteArray\",\"parameters\":[{\"name\":\"internalTokenId\",\"type\":\"ByteArray\"}]},{\"name\":\"listInternalTokenId\",\"offset\":3210,\"safe\":false,\"returntype\":\"InteropInterface\",\"parameters\":[{\"name\":\"prefix\",\"type\":\"ByteArray\"}]},{\"name\":\"listInternalTokenId\",\"offset\":3245,\"safe\":false,\"returntype\":\"InteropInterface\",\"parameters\":[{\"name\":\"externalTokenContract\",\"type\":\"Hash160\"},{\"name\":\"prefix\",\"type\":\"ByteArray\"}]},{\"name\":\"getInternalTokenId\",\"offset\":3284,\"safe\":false,\"returntype\":\"ByteArray\",\"parameters\":[{\"name\":\"externalTokenContract\",\"type\":\"Hash160\"},{\"name\":\"externalTokenId\",\"type\":\"ByteArray\"}]},{\"name\":\"max\",\"offset\":3322,\"safe\":false,\"returntype\":\"Integer\",\"parameters\":[{\"name\":\"v1\",\"type\":\"Integer\"},{\"name\":\"v2\",\"type\":\"Integer\"}]},{\"name\":\"transfer\",\"offset\":3335,\"safe\":false,\"returntype\":\"Boolean\",\"parameters\":[{\"name\":\"from\",\"type\":\"Hash160\"},{\"name\":\"to\",\"type\":\"Hash160\"},{\"name\":\"amount\",\"type\":\"Integer\"},{\"name\":\"tokenId\",\"type\":\"ByteArray\"},{\"name\":\"data\",\"type\":\"Any\"}]},{\"name\":\"setRentalPrice\",\"offset\":3564,\"safe\":false,\"returntype\":\"Void\",\"parameters\":[{\"name\":\"renter\",\"type\":\"Hash160\"},{\"name\":\"tokenContract\",\"type\":\"Hash160\"},{\"name\":\"tokenId\",\"type\":\"ByteArray\"},{\"name\":\"price\",\"type\":\"Integer\"}]},{\"name\":\"registerRental\",\"offset\":4291,\"safe\":false,\"returntype\":\"Integer\",\"parameters\":[{\"name\":\"renter\",\"type\":\"Hash160\"},{\"name\":\"externalTokenContract\",\"type\":\"Hash160\"},{\"name\":\"amountForRent\",\"type\":\"Integer\"},{\"name\":\"externalTokenId\",\"type\":\"ByteArray\"},{\"name\":\"flashLoanPrice\",\"type\":\"Integer\"},{\"name\":\"ordinaryLoanPrice\",\"type\":\"Integer\"},{\"name\":\"isDivisible\",\"type\":\"Boolean\"}]},{\"name\":\"unregisterRental\",\"offset\":4928,\"safe\":false,\"returntype\":\"Integer\",\"parameters\":[{\"name\":\"renter\",\"type\":\"Hash160\"},{\"name\":\"externalTokenContract\",\"type\":\"Hash160\"},{\"name\":\"amountToUnregister\",\"type\":\"Integer\"},{\"name\":\"externalTokenId\",\"type\":\"ByteArray\"},{\"name\":\"isDivisible\",\"type\":\"Boolean\"}]},{\"name\":\"getTotalPrice\",\"offset\":5275,\"safe\":false,\"returntype\":\"Integer\",\"parameters\":[{\"name\":\"pricePerSecond\",\"type\":\"Integer\"},{\"name\":\"borrowTimeMilliseconds\",\"type\":\"Integer\"}]},{\"name\":\"getCollateralAmount\",\"offset\":5293,\"safe\":false,\"returntype\":\"Integer\",\"parameters\":[{\"name\":\"totalPrice\",\"type\":\"Integer\"}]},{\"name\":\"getCollateralAmount\",\"offset\":5306,\"safe\":false,\"returntype\":\"Integer\",\"parameters\":[{\"name\":\"pricePerSecond\",\"type\":\"Integer\"},{\"name\":\"borrowTimeMilliseconds\",\"type\":\"Integer\"}]},{\"name\":\"borrow\",\"offset\":5322,\"safe\":false,\"returntype\":\"Integer\",\"parameters\":[{\"name\":\"renter\",\"type\":\"Hash160\"},{\"name\":\"tenant\",\"type\":\"Hash160\"},{\"name\":\"amount\",\"type\":\"Integer\"},{\"name\":\"externalTokenContract\",\"type\":\"Hash160\"},{\"name\":\"externalTokenId\",\"type\":\"ByteArray\"},{\"name\":\"borrowTimeMilliseconds\",\"type\":\"Integer\"}]},{\"name\":\"borrow\",\"offset\":5368,\"safe\":false,\"returntype\":\"Integer\",\"parameters\":[{\"name\":\"renter\",\"type\":\"Hash160\"},{\"name\":\"tenant\",\"type\":\"Hash160\"},{\"name\":\"amount\",\"type\":\"Integer\"},{\"name\":\"internalTokenId\",\"type\":\"ByteArray\"},{\"name\":\"borrowTimeMilliseconds\",\"type\":\"Integer\"}]},{\"name\":\"closeNextRental\",\"offset\":6125,\"safe\":false,\"returntype\":\"Void\",\"parameters\":[{\"name\":\"renter\",\"type\":\"Hash160\"},{\"name\":\"internalTokenId\",\"type\":\"ByteArray\"},{\"name\":\"tenant\",\"type\":\"Hash160\"},{\"name\":\"startTime\",\"type\":\"ByteArray\"}]},{\"name\":\"openNextRental\",\"offset\":6362,\"safe\":false,\"returntype\":\"Void\",\"parameters\":[{\"name\":\"renter\",\"type\":\"Hash160\"},{\"name\":\"internalTokenId\",\"type\":\"ByteArray\"},{\"name\":\"tenant\",\"type\":\"Hash160\"},{\"name\":\"startTime\",\"type\":\"ByteArray\"}]},{\"name\":\"payback\",\"offset\":6599,\"safe\":false,\"returntype\":\"Void\",\"parameters\":[{\"name\":\"renter\",\"type\":\"Hash160\"},{\"name\":\"tenant\",\"type\":\"Hash160\"},{\"name\":\"externalTokenContract\",\"type\":\"Hash160\"},{\"name\":\"externalTokenId\",\"type\":\"ByteArray\"},{\"name\":\"startTime\",\"type\":\"ByteArray\"},{\"name\":\"collateralReceiver\",\"type\":\"Hash160\"},{\"name\":\"isDivisible\",\"type\":\"Boolean\"}]},{\"name\":\"payback\",\"offset\":7232,\"safe\":false,\"returntype\":\"Void\",\"parameters\":[{\"name\":\"renter\",\"type\":\"Hash160\"},{\"name\":\"tenant\",\"type\":\"Hash160\"},{\"name\":\"internalTokenId\",\"type\":\"ByteArray\"},{\"name\":\"startTime\",\"type\":\"ByteArray\"},{\"name\":\"collateralReceiver\",\"type\":\"Hash160\"},{\"name\":\"isDivisible\",\"type\":\"Boolean\"}]},{\"name\":\"flashBorrowDivisible\",\"offset\":7334,\"safe\":false,\"returntype\":\"Any\",\"parameters\":[{\"name\":\"tenant\",\"type\":\"Hash160\"},{\"name\":\"token\",\"type\":\"Hash160\"},{\"name\":\"tokenId\",\"type\":\"ByteArray\"},{\"name\":\"renter\",\"type\":\"Hash160\"},{\"name\":\"neededAmount\",\"type\":\"Integer\"},{\"name\":\"renterCalledContract\",\"type\":\"Hash160\"},{\"name\":\"renterCalledMethod\",\"type\":\"String\"},{\"name\":\"arguments\",\"type\":\"Array\"}]},{\"name\":\"flashBorrowNonDivisible\",\"offset\":8065,\"safe\":false,\"returntype\":\"Any\",\"parameters\":[{\"name\":\"tenant\",\"type\":\"Hash160\"},{\"name\":\"token\",\"type\":\"Hash160\"},{\"name\":\"tokenId\",\"type\":\"ByteArray\"},{\"name\":\"renter\",\"type\":\"Hash160\"},{\"name\":\"calledContract\",\"type\":\"Hash160\"},{\"name\":\"calledMethod\",\"type\":\"String\"},{\"name\":\"arguments\",\"type\":\"Array\"}]},{\"name\":\"_initialize\",\"offset\":8438,\"safe\":false,\"returntype\":\"Void\",\"parameters\":[]}],\"events\":[{\"name\":\"Transfer\",\"parameters\":[{\"name\":\"from\",\"type\":\"Hash160\"},{\"name\":\"to\",\"type\":\"Hash160\"},{\"name\":\"amount\",\"type\":\"Integer\"},{\"name\":\"tokenId\",\"type\":\"ByteArray\"}]},{\"name\":\"TokenCreated\",\"parameters\":[{\"name\":\"arg1\",\"type\":\"Hash160\"},{\"name\":\"arg2\",\"type\":\"ByteArray\"},{\"name\":\"arg3\",\"type\":\"ByteArray\"},{\"name\":\"arg4\",\"type\":\"Integer\"}]},{\"name\":\"PriceSet\",\"parameters\":[{\"name\":\"arg1\",\"type\":\"Hash160\"},{\"name\":\"arg2\",\"type\":\"Hash160\"},{\"name\":\"arg3\",\"type\":\"ByteArray\"},{\"name\":\"arg4\",\"type\":\"Integer\"}]},{\"name\":\"TokenRented\",\"parameters\":[{\"name\":\"arg1\",\"type\":\"Hash160\"},{\"name\":\"arg2\",\"type\":\"Hash160\"},{\"name\":\"arg3\",\"type\":\"ByteArray\"},{\"name\":\"arg4\",\"type\":\"Hash160\"},{\"name\":\"arg5\",\"type\":\"Integer\"},{\"name\":\"arg6\",\"type\":\"Integer\"},{\"name\":\"arg7\",\"type\":\"Integer\"},{\"name\":\"arg8\",\"type\":\"Integer\"},{\"name\":\"arg9\",\"type\":\"Integer\"}]},{\"name\":\"RentalClosed\",\"parameters\":[{\"name\":\"arg1\",\"type\":\"Hash160\"},{\"name\":\"arg2\",\"type\":\"ByteArray\"},{\"name\":\"arg3\",\"type\":\"Hash160\"},{\"name\":\"arg4\",\"type\":\"Integer\"},{\"name\":\"arg5\",\"type\":\"Integer\"},{\"name\":\"arg6\",\"type\":\"Integer\"},{\"name\":\"arg7\",\"type\":\"Integer\"}]},{\"name\":\"RentalOpened\",\"parameters\":[{\"name\":\"arg1\",\"type\":\"Hash160\"},{\"name\":\"arg2\",\"type\":\"ByteArray\"},{\"name\":\"arg3\",\"type\":\"Hash160\"},{\"name\":\"arg4\",\"type\":\"Integer\"},{\"name\":\"arg5\",\"type\":\"Integer\"},{\"name\":\"arg6\",\"type\":\"Integer\"},{\"name\":\"arg7\",\"type\":\"Integer\"}]},{\"name\":\"RentalRevoked\",\"parameters\":[{\"name\":\"arg1\",\"type\":\"Hash160\"},{\"name\":\"arg2\",\"type\":\"ByteArray\"},{\"name\":\"arg3\",\"type\":\"Hash160\"},{\"name\":\"arg4\",\"type\":\"Integer\"},{\"name\":\"arg5\",\"type\":\"Integer\"},{\"name\":\"arg6\",\"type\":\"Integer\"},{\"name\":\"arg7\",\"type\":\"Integer\"}]},{\"name\":\"TokenWithdrawn\",\"parameters\":[{\"name\":\"arg1\",\"type\":\"Hash160\"},{\"name\":\"arg2\",\"type\":\"ByteArray\"},{\"name\":\"arg3\",\"type\":\"Integer\"}]}]},\"permissions\":[{\"contract\":\"*\",\"methods\":\"*\"}],\"trusts\":[],\"extra\":{\"Author\":\"Hecate2\",\"Email\":\"developer@neo.org\",\"Description\":\"NFTFlashLoan\"}}"},{"type":"String","value":"registerRental"},{"type":"Array","value":[{"type":"Hash160","value":"0xb1983fa2479a0c8e2beae032d2df564b5451b7a5"},{"type":"Hash160","value":"0x2704f0fa22877b2e0a53afe858b8db971f348b91"},{"type":"Integer","value":"68"},{"type":"Integer","value":"1"},{"type":"Integer","value":"5"},{"type":"Integer","value":"7"},{"type":"Boolean","value":true}]}],[{"account":"0xb1983fa2479a0c8e2beae032d2df564b5451b7a5","scopes":"Global","allowedcontracts":[],"allowedgroups":[],"rules":[]}]],"id":1}
{'jsonrpc': '2.0', 'id': 1, 'result': {'script': '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', 'state': 'FAULT', 'gasconsumed': '218412865', 'exception': '   at Neo.VM.ExecutionEngine.ExecuteInstruction(Instruction instruction) in C:\\Users\\RhantolkYtriHistoria\\NEO\\neo-vm\\src\\Neo.VM\\ExecutionEngine.cs:line 1143\r\n   at Neo.VM.ExecutionEngine.ExecuteNext() in C:\\Users\\RhantolkYtriHistoria\\NEO\\neo-vm\\src\\Neo.VM\\ExecutionEngine.cs:line 1454\nInvalid type for SIZE: Any', 'traceback': '   at Neo.VM.ExecutionEngine.ExecuteInstruction(Instruction instruction) in C:\\Users\\RhantolkYtriHistoria\\NEO\\neo-vm\\src\\Neo.VM\\ExecutionEngine.cs:line 1143\r\n   at Neo.VM.ExecutionEngine.ExecuteNext() in C:\\Users\\RhantolkYtriHistoria\\NEO\\neo-vm\\src\\Neo.VM\\ExecutionEngine.cs:line 1454\nInvalid type for SIZE: Any\r\nCallingScriptHash=0x5c1068339fae89eb1a743909d0213e1d99dc5dc9[AnyUpdateShortSafe]\r\nCurrentScriptHash=0x5c1068339fae89eb1a743909d0213e1d99dc5dc9[AnyUpdateShortSafe]\r\nEntryScriptHash=0xb493e9d3b67262ba1f35dfc85dbfe6464c83c092\r\n   at Neo.VM.ExecutionEngine.ExecuteInstruction(Instruction instruction) in C:\\Users\\RhantolkYtriHistoria\\NEO\\neo-vm\\src\\Neo.VM\\ExecutionEngine.cs:line 1143\r\n   at Neo.VM.ExecutionEngine.ExecuteNext() in C:\\Users\\RhantolkYtriHistoria\\NEO\\neo-vm\\src\\Neo.VM\\ExecutionEngine.cs:line 1454\r\nInstructionPointer=3532, OpCode SIZE, Script Length=8518\r\nInstructionPointer=3814, OpCode DUP, Script Length=8518\r\nInstructionPointer=4574, OpCode STLOC3, Script Length=8518\r\nInstructionPointer=502, OpCode STLOC2, Script Length=3194\r\nInstructionPointer=21384, OpCode , Script Length=21384', 'stack': []}}
Traceback (most recent call last):
  File "C:/Users/RhantolkYtriHistoria/NEO/neo-test-client/tutorial.py", line 26, in <module>
    client.invokefunction('anyUpdate', params=
  File "C:\Users\RhantolkYtriHistoria\NEO\neo-test-client\neo_fairy_client\rpc\fairy_client.py", line 478, in invokefunction
    return self.invokefunction_of_any_contract(self.contract_scripthash, operation, params,
  File "C:\Users\RhantolkYtriHistoria\NEO\neo-test-client\neo_fairy_client\rpc\fairy_client.py", line 465, in invokefunction_of_any_contract
    result = self.meta_rpc_method(
  File "C:\Users\RhantolkYtriHistoria\NEO\neo-test-client\neo_fairy_client\rpc\fairy_client.py", line 224, in meta_rpc_method
    raise ValueError(result_result['traceback'])
ValueError:    at Neo.VM.ExecutionEngine.ExecuteInstruction(Instruction instruction) in C:\Users\RhantolkYtriHistoria\NEO\neo-vm\src\Neo.VM\ExecutionEngine.cs:line 1143
   at Neo.VM.ExecutionEngine.ExecuteNext() in C:\Users\RhantolkYtriHistoria\NEO\neo-vm\src\Neo.VM\ExecutionEngine.cs:line 1454
Invalid type for SIZE: Any
CallingScriptHash=0x5c1068339fae89eb1a743909d0213e1d99dc5dc9[AnyUpdateShortSafe]
CurrentScriptHash=0x5c1068339fae89eb1a743909d0213e1d99dc5dc9[AnyUpdateShortSafe]
EntryScriptHash=0xb493e9d3b67262ba1f35dfc85dbfe6464c83c092
   at Neo.VM.ExecutionEngine.ExecuteInstruction(Instruction instruction) in C:\Users\RhantolkYtriHistoria\NEO\neo-vm\src\Neo.VM\ExecutionEngine.cs:line 1143
   at Neo.VM.ExecutionEngine.ExecuteNext() in C:\Users\RhantolkYtriHistoria\NEO\neo-vm\src\Neo.VM\ExecutionEngine.cs:line 1454
InstructionPointer=3532, OpCode SIZE, Script Length=8518
InstructionPointer=3814, OpCode DUP, Script Length=8518
InstructionPointer=4574, OpCode STLOC3, Script Length=8518
InstructionPointer=502, OpCode STLOC2, Script Length=3194
InstructionPointer=21384, OpCode , Script Length=21384
```

Now pay attention to the `InstructionPointer` stacks at last, especially the first line of the `InstructionPointer`s. By reading `NFTFlashLoan.nef.txt` (available in NFTLoan repository releases) created by `Dumpnef`, you'll get the following information near `InstructionPointer=3532`:

```
# Code NFTLoan.cs line 141: "ExecutionEngine.Assert(id.Length < 0xFD, "Too long id");"
3518 PUSHDATA1 54-6F-6F-20-6C-6F-6E-67-20-69-64 # as text: "Too long id"
3531 LDLOC2
3532 SIZE
3533 PUSHINT16 FD-00 # 253
3536 LT
3537 CALL_L 07-FB-FF-FF # pos: 2264, offset: -1273
```

And you can immediately locate the problem, finding that `id` is actually `null` and the operation `id.Length` is invalid. 

#### Tutorial for debugging

Still feeling difficult to locate bugs in testing? Just debug the contract with step-in, step-out and step-over. Prepare your debugging storage environment automatically with your test codes, set breakpoints on either source code lines or InstructionPointers of assembly codes, and watch all the values of variables based on their names! 

##### Step 0: Set debug info

Well... Thanks to the `auto_set_debug_info=True` option in `client.virutal_deploy_from_path`, Fairy has automatically registered the debug info of `AnyUpdate` and `TestNophtD` for you when you deployed them. But Fairy does not recognize the source codes of `NFTLoan` because we did not actually deploy it. Now we are going to set debug info manually. 

```python
with open('../NFTLoan/NFTLoan/bin/sc/NFTFlashLoan.nefdbgnfo', 'rb') as f:
    nefdbgnfo = f.read()
with open('../NFTLoan/NFTLoan/bin/sc/NFTFlashLoan.nef.txt', 'r') as f:
    dumpnef = f.read()
client.virutal_deploy_from_path('../NFTLoan/NFTLoan/bin/sc/NFTFlashLoan.nef', auto_set_debug_info=False)  # client.contract_scripthash is set
client.set_debug_info(nefdbgnfo, dumpnef)  # the debug info is by default registered for client.contract_scripthash
```

##### Step 1: Call a method in debug mode

Your debugging runtime storage environment is always inherited from a test session. It is recommended to build the debugging environment automatically with testing codes.

```python
print(breakpoint := client.debug_function_with_session(  # do not invokefunction in debugging!
    'registerRental',
    params=[wallet_scripthash, test_nopht_d_hash, 68, 1, 5, 7, True],
    signers=None,  # Watch this! I wrote this on purpose
))
```

With `signers=None` your Fairy client uses `CalledByEntry` signature, which is actually insufficient in our case. You should get the following to be printed

```
Cloned snapshot::debugfunction registerRental
RpcBreakpoint VMState.FAULT ExecutionEngine.cs line 33 instructionPointer 2281: Assert(false);
```

which directly leads you to the source code! Now you can easily figure out the signature problems.

If no fault occurs, you should get VMState.HALT in `breakpoint`.

Note that **all debugging executions write nothing to the snapshot!**

##### Step 2: Set breakpoints; step-in, step-out, step-over, and watch variable values

Head to [test_debug.py](test_debug.py) in this repo to learn these operations!