import * as cdk from 'aws-cdk-lib';
import { RestApiToLambda } from 'must-cdk';

export class RestApiStack extends cdk.Stack {
  constructor(scope: cdk.App, id: string, props?: cdk.StackProps) {
    super(scope, id, props);

    new RestApiToLambda(this, 'UserApi', {
      apiName: 'user-service',
      description: 'User management API',
      lambdaFunction: {
        runtime: 'nodejs18.x',
        handler: 'index.handler',
        code: 'lambda/user-service',
        environment: {
          TABLE_NAME: 'users-table',
          REGION: 'ap-northeast-2',
        },
        memorySize: 256,
        timeout: 30,
      },
      customDomain: {
        domainName: 'api.example.com',
        certificateArn: 'arn:aws:acm:ap-northeast-2:123456789012:certificate/xxx',
      },
      cors: {
        allowOrigins: ['https://example.com'],
        allowMethods: ['GET', 'POST', 'PUT', 'DELETE'],
        allowHeaders: ['Content-Type', 'Authorization'],
      },
      authorizer: {
        type: 'JWT',
        jwtConfiguration: {
          issuer: 'https://cognito-idp.ap-northeast-2.amazonaws.com/xxx',
          audience: ['client-id'],
        },
      },
      apiKey: {
        name: 'client-api-key',
        description: 'API key for client application',
      },
      usagePlan: {
        name: 'standard-plan',
        throttle: {
          rateLimit: 100,
          burstLimit: 200,
        },
        quota: {
          limit: 10000,
          period: 'MONTH',
        },
      },
    });
  }
}

const app = new cdk.App();
new RestApiStack(app, 'RestApiStack', {
  env: {
    account: process.env.CDK_DEFAULT_ACCOUNT,
    region: 'ap-northeast-2',
  },
});
