"""" Reader file for Thermo Fisher Qubit Flex Parser"""

import numpy as np
import pandas as pd

from allotropy.exceptions import AllotropeConversionError
from allotropy.named_file_contents import NamedFileContents
from allotropy.parsers.lines_reader import CsvReader
from allotropy.parsers.utils.pandas import read_excel


class ThermoFisherQubitFlexReader:
    SUPPORTED_EXTENSIONS = "csv,xlsx"

    @classmethod
    def read(cls, named_file_contents: NamedFileContents) -> pd.DataFrame:
        """
        Reads the content of the provided named file and returns it as a pandas DataFrame.

        Parameters:
        named_file_contents (NamedFileContents): The named file contents to read.
            It includes the original file name, file contents, and encoding.

        Returns:
        pd.DataFrame: The content of the file as a pandas DataFrame.

        Raises:
        AllotropeConversionError: If the file format is not supported.
        """
        if named_file_contents.extension == "csv":
            csv_reader = CsvReader.create(named_file_contents)
            # Skip lines until we find the "Run ID" header
            csv_reader.drop_until(r"Test Date")
            # Read the CSV block starting from the header
            df = csv_reader.pop_csv_block_as_df(index_col=False, header=0)
            if df is None:
                msg = "Unable to find 'Run ID' header in CSV file"
                raise AllotropeConversionError(msg)
        else:
            df = read_excel(named_file_contents.contents)
        df = df.replace(np.nan, None)
        return df
