import asyncio
import logging
import os
import subprocess
import sys
import warnings
from pathlib import Path

import click

# Config file locations in priority order
CONFIG_PATHS = [
    Path(".aiautocommit"),  # $PWD/.aiautocommit
    Path(os.environ.get("XDG_CONFIG_HOME", "~/.config")).expanduser()
    / "aiautocommit",  # XDG config dir
    Path(os.environ.get("AIAUTOCOMMIT_CONFIG", "")),  # Custom config path
]

COMMIT_PROMPT_FILE = "commit_prompt.txt"
EXCLUSIONS_FILE = "exclusions.txt"
COMMIT_SUFFIX_FILE = "commit_suffix.txt"

# https://platform.openai.com/docs/models
MODEL_NAME = os.environ.get("AIAUTOCOMMIT_MODEL", "gpt-4o-mini")

COMMIT_PROMPT = """
Generate a commit message from the `git diff` output below using these rules:

* No more than 50 character subject line
* Subject line (after conventional commit syntax) should be a concise description that completes the sentence "If applied, this commit will...", but **do not include this sentence** in the commit message
* Write in imperative mood
* Only lines removed or added should be considered
* Use conventional commit format
  * Use `docs` instead of `feat` ONLY if documentation or code comments are the ONLY changes
  * Only use `refactor` for changes that do not change behavior and simply refactor code
  * Use `style` when updating linting or formatting or configuration for linting or formatting
* Only include extended commit message when the diff is large (hundreds of lines added or removed)
  * Use the extended commit (body) to explain what and why vs. how
  * Use markdown bullets to describe changes
  * Explain the problem that this commit is solving. Focus on why this change was made as opposed to how (the code explains that).
  * Explain any side effects or other unintuitive consequences of this change.
* Some hints on newer file types of programming tools:
  * `Justfile` is similar to a Makefile and should be considered part of the build system
* If the diff output below is small, do not include an extended commit message
* Do not wrap output in a codeblock
* Write why a change was made and avoid general statements like:
  * "Improved comments and structured logic for clarity..."
  * "Separated logic from the original function..."
  * "Refactored X into Y..."
  * "Introduced new function..."
  * "Enhances clarity and ease of use..."
  * "add new file to the project..."
* Don't mention verbose details like:
  * What variable is changed "feat: update prompt text in DIFF_PROMPT variable"
* If docs and functionality are modified, focus on the functionality
* If there is not enough information to generate a summary, return an empty string

---
"""

# trailers are a native git feature that can be used to add metadata to a commit
# https://git-scm.com/docs/git-interpret-trailers
# let's indicate that this message was generated by aiautocommit
COMMIT_SUFFIX = """

Generated-by: aiautocommit
"""

# TODO should we ignore files without an extension? can we detect binary files?
EXCLUDED_FILES = [
    "Gemfile.lock",
    "uv.lock",
    "poetry.lock",
    "package-lock.json",
    "yarn.lock",
    "pnpm-lock.yaml",
    "package-lock.json",
    "yarn.lock",
    "pnpm-lock.yaml",
    "pnpm-lock.yaml",
    "composer.lock",
    "cargo.lock",
    "mix.lock",
    "Pipfile.lock",
    "pdm.lock",
    "flake.lock",
    "bun.lockb",
]

# characters, not tokens
PROMPT_CUTOFF = 10_000

logging.basicConfig(
    level=os.environ.get("LOG_LEVEL", "INFO").upper(),
    **(
        {"filename": os.environ.get("AIAUTOCOMMIT_LOG_PATH")}
        if os.environ.get("AIAUTOCOMMIT_LOG_PATH")
        else {"stream": sys.stderr}
    ),
)

# this is called within py dev environments. Unless it looks like we are explicitly debugging aiautocommit, we force a
# more silent operation. Checking for AIAUTOCOMMIT_LOG_PATH is not a perfect heuristic, but it works for now.
if not os.environ.get("AIAUTOCOMMIT_LOG_PATH"):
    # Suppress ResourceWarnings
    warnings.filterwarnings("ignore", category=ResourceWarning)

    # Disable asyncio debug logging
    logging.getLogger("asyncio").setLevel(logging.ERROR)

    # Optional: Disable httpx logging if desired
    logging.getLogger("httpx").setLevel(logging.WARNING)

# allow a unique API key to be set for OpenAI, for tracking/costing
if os.environ.get("AIAUTOCOMMIT_OPENAI_API_KEY"):
    os.environ["OPENAI_API_KEY"] = os.environ["AIAUTOCOMMIT_OPENAI_API_KEY"]


def configure_prompts(config_dir=None):
    global DIFF_PROMPT, COMMIT_MSG_PROMPT, EXCLUDED_FILES, CONFIG_PATHS

    if config_dir:
        CONFIG_PATHS.insert(0, Path(config_dir))

    # Find first existing config dir
    config_dir = next((path for path in CONFIG_PATHS if path and path.exists()), None)

    if not config_dir:
        logging.debug("No config directory found")
        return

    logging.debug(f"Found config directory at {config_dir}")

    # Load commit prompt
    commit_file = config_dir / COMMIT_PROMPT_FILE
    if commit_file.exists():
        logging.debug("Loading custom commit prompt from commit.txt")
        COMMIT_MSG_PROMPT = commit_file.read_text().strip()

    # Load exclusions
    exclusions_file = config_dir / EXCLUSIONS_FILE
    if exclusions_file.exists():
        logging.debug("Loading custom exclusions from exclusions.txt")
        EXCLUDED_FILES = [
            line.strip()
            for line in exclusions_file.read_text().splitlines()
            if line.strip()
        ]

    # Load commit suffix
    commit_suffix_file = config_dir / COMMIT_SUFFIX_FILE
    if commit_suffix_file.exists():
        logging.debug("Loading custom commit suffix from commit_suffix.txt")
        global COMMIT_SUFFIX
        COMMIT_SUFFIX = commit_suffix_file.read_text().strip()


def get_diff(ignore_whitespace=True):
    arguments = [
        "git",
        "--no-pager",
        "diff",
        "--staged",
    ]
    if ignore_whitespace:
        arguments += [
            "--ignore-space-change",
            "--ignore-blank-lines",
        ]

    for file in EXCLUDED_FILES:
        arguments += [f":(exclude){file}"]

    diff_process = subprocess.run(arguments, capture_output=True, text=True)
    diff_process.check_returncode()
    normalized_diff = diff_process.stdout.strip()

    logging.debug(f"Discovered Diff:\n{normalized_diff}")

    return normalized_diff


def asyncopenai() -> "AsyncOpenAI":
    # waiting to import so the initial load is quick (so --help displays more quickly)
    from openai import AsyncOpenAI

    if not hasattr(asyncopenai, "_instance"):
        # Using a class-level attribute to store the singleton instance
        setattr(asyncopenai, "_instance", AsyncOpenAI())
    return getattr(asyncopenai, "_instance")


async def complete(prompt):
    aclient = asyncopenai()
    completion_resp = await aclient.chat.completions.create(
        model=MODEL_NAME,
        messages=[{"role": "user", "content": prompt[: PROMPT_CUTOFF + 100]}],
        # TODO this seems awfully small?
        # max_completion_tokens=128,
    )

    completion = completion_resp.choices[0].message.content.strip()
    return completion


SINGLE_PROMPT = True


async def generate_commit_message(diff):
    if not diff:
        logging.debug("No commit message generated")
        return ""

    return (await complete(COMMIT_PROMPT + "\n\n" + diff)) + COMMIT_SUFFIX


def git_commit(message):
    # will ignore message if diff is empty
    return subprocess.run(["git", "commit", "--message", message, "--edit"]).returncode


def is_reversion():
    # Check if we're in the middle of a git revert
    if (Path(".git") / "REVERT_HEAD").exists():
        return True

    if (Path(".git") / "MERGE_MSG").exists():
        return True

    return False


@click.group(invoke_without_command=True)
def main():
    """
    Generate a commit message for staged files and commit them.
    Git will prompt you to edit the generated commit message.
    """
    ctx = click.get_current_context()
    if ctx.invoked_subcommand is None:
        ctx.invoke(commit)


@main.command()
@click.option(
    "-p",
    "--print-message",
    is_flag=True,
    default=False,
    help="print commit msg to stdout instead of performing commit",
)
@click.option(
    "-o",
    "--output-file",
    type=click.Path(writable=True),
    help="write commit message to specified file",
)
@click.option(
    "--config-dir",
    type=click.Path(exists=True, file_okay=False, dir_okay=True),
    help="specify custom config directory",
)
def commit(print_message, output_file, config_dir):
    """
    Generate commit message from git diff.
    """

    if is_reversion():
        return 0

    configure_prompts(config_dir)

    try:
        if not get_diff(ignore_whitespace=False):
            click.echo(
                "No changes staged. Use `git add` to stage files before invoking gpt-commit.",
                err=True,
            )
            return 1

        # run async so when multiple chunks exist we can get summaries concurrently
        commit_message = asyncio.run(generate_commit_message(get_diff()))
    except UnicodeDecodeError:
        click.echo("aiautocommit does not support binary files", err=True)

        commit_message = (
            # TODO use heredoc
            "# gpt-commit does not support binary files. "
            "Please enter a commit message manually or unstage any binary files."
        )

    if output_file:
        if commit_message:
            Path(output_file).write_text(commit_message)
            return 0
        return 1
    elif print_message:
        click.echo(commit_message)
        return 0
    else:
        return git_commit(commit_message)


@main.command()
@click.option(
    "--overwrite",
    is_flag=True,
    help="Overwrite existing pre-commit hook if it exists",
)
def install_pre_commit(overwrite):
    """Install pre-commit script into git hooks directory"""
    git_result = subprocess.run(
        ["git", "rev-parse", "--git-dir"],
        capture_output=True,
        text=True,
    )
    git_result.check_returncode()

    git_dir = git_result.stdout.strip()

    target_hooks_dir = Path(git_dir) / "hooks"
    target_hooks_dir.mkdir(exist_ok=True)

    commit_msg_git_hook_name = "prepare-commit-msg"
    pre_commit = target_hooks_dir / commit_msg_git_hook_name
    pre_commit_script = Path(__file__).parent / commit_msg_git_hook_name

    if not pre_commit.exists() or overwrite:
        pre_commit.write_text(pre_commit_script.read_text())
        pre_commit.chmod(0o755)
        click.echo("Installed pre-commit hook")
    else:
        click.echo(
            "pre-commit hook already exists. Here's the contents we would have written:"
        )
        click.echo(pre_commit_script.read_text())


@main.command()
def dump_prompts():
    """Dump default prompts into .aiautocommit directory for easy customization"""
    config_dir = Path(".aiautocommit")
    config_dir.mkdir(exist_ok=True)

    commit_prompt = config_dir / COMMIT_PROMPT_FILE
    exclusions = config_dir / EXCLUSIONS_FILE
    commit_suffix = config_dir / COMMIT_SUFFIX_FILE

    if not commit_prompt.exists():
        commit_prompt.write_text(COMMIT_MSG_PROMPT)
    if not exclusions.exists():
        exclusions.write_text("\n".join(EXCLUDED_FILES))
    if not commit_suffix.exists():
        commit_suffix.write_text(COMMIT_SUFFIX)

    click.echo(
        f"""Dumped default prompts to .aiautocommit directory:

- {COMMIT_PROMPT_FILE}: Template for generating commit messages
- {EXCLUSIONS_FILE}: List of file patterns to exclude from processing
- {COMMIT_SUFFIX}: Text appended to the end of every commit message
"""
    )
