# Copyright 2025 David Trimm
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
import numpy as np
import networkx as nx
from scipy.sparse.csgraph import dijkstra
from scipy.sparse import csr_matrix
from sklearn.decomposition import PCA

__name__ = 'mds_at_scale'

#
# Implementation of Landmark MDS
#
# V. de Silva and J. Tenenbaum. Global versus local methods in nonlinear dimensionality reduction. 
# In Proc. NIPS, pages 721–728, 2003.
#
# Generated by Claude Sonnet 4.5 (2025-10-23)
#
# Note that this implementation does not perform the minmax operation
# (instead, it selects landmarks randomly)
# ... fixed by copying the pivotmds implementation (but now it performs it twice like pivotmds does)
#
class LandmarkMDSLayout(object):
    def __init__(self, g, num_landmarks=None, dimensions=2, landmarks=None, landmark_pos=None):
        """
        Landmark Multidimensional Scaling (L-MDS) algorithm for graph embedding.
        
        Parameters:
        -----------
        g : networkx.Graph or scipy.sparse matrix or numpy.ndarray
            Input graph. Can be a NetworkX graph, sparse matrix, or dense adjacency matrix.
        num_landmarks : int, optional
            Number of landmark nodes to select. If None, uses sqrt(n) where n is number of nodes.
        dimensions : int, default=2
            Number of dimensions for the embedding.
        landmarks : list, optional
            List of landmark nodes. If provided, uses these landmarks instead of selecting randomly.
            These will be converted into indices by the class.
        landmark_pos : dict, optional
            If specified, provides coordinates for the landmarks. If not provided, then the algorithm will use MDS to compute them.
            The keys will be in the original node ids... so also will be converted via the class into numpy indices.
        """
        
        # Convert graph to adjacency matrix if needed
        if isinstance(g, nx.Graph):
            n = g.number_of_nodes()
            # Create adjacency matrix with edge weights
            adj_matrix = nx.to_scipy_sparse_array(g, weight='weight', format='csr')
        elif isinstance(g, csr_matrix):
            adj_matrix = g
            n = adj_matrix.shape[0]
        elif isinstance(g, np.ndarray):
            adj_matrix = csr_matrix(g)
            n = adj_matrix.shape[0]
        else:
            raise ValueError("Graph must be NetworkX graph, scipy sparse matrix, or numpy array")

        # If the landmarks weren't specified by the user, determine them via the MaxMin strategy
        if landmarks is None and landmark_pos is None:
            # Set number of landmarks if not specified
            if num_landmarks is None:
                num_landmarks = max(int(np.sqrt(n)), dimensions + 1)
            
            num_landmarks = min(num_landmarks, n)
            
            #
            # vvv -- copied from the pivotmds implementation below --- vvv
            #
            # Step 1: Select pivots using MaxMin strategy
            landmarks = []
            
            # Select first pivot randomly
            first_landmark = np.random.randint(0, n)
            landmarks.append(first_landmark)
            
            # Initialize minimum distances to first pivot
            min_distances = dijkstra(adj_matrix, directed=False, indices=first_landmark)
            min_distances[np.isinf(min_distances)] = 0
            
            # Select remaining pivots
            for _ in range(num_landmarks - 1):
                # Select node with maximum minimum distance to existing pivots
                next_landmark = np.argmax(min_distances)
                landmarks.append(next_landmark)
                
                # Update minimum distances
                new_distances = dijkstra(adj_matrix, directed=False, indices=next_landmark)
                new_distances[np.isinf(new_distances)] = 0
                min_distances = np.minimum(min_distances, new_distances)
            
            landmarks = np.array(landmarks)
            #
            # ^^^ --- copied from the pivotmds implementation below --- ^^^
            #

        else:
            # Pull them from the positional keys if the landmarks parameter is None
            if landmarks is None: landmarks = list(landmark_pos.keys())
            # Need to convert them to numpy array indices
            num_landmarks   = len(landmarks)
            _as_set_        = set(landmarks)
            _nodes_as_list_ = list(g.nodes())
            landmarks       = np.array([_nodes_as_list_.index(landmark) for landmark in _as_set_])
            # Ensure that landmark positions were provided ... if not, set it to None            
            if landmark_pos is not None:
                all_positions_filled = True
                for landmark in landmarks:
                    if _nodes_as_list_[landmark] not in landmark_pos:
                        all_positions_filled = False
                        break
                if not all_positions_filled: landmark_pos = None

        # Create node mapping for NetworkX graphs
        node_mapping = None
        if isinstance(g, nx.Graph): node_mapping = list(g.nodes())

        if landmark_pos is None:
            # Step 2: Compute shortest path distances from all nodes to landmarks
            # Using Dijkstra's algorithm from each landmark
            distances = np.zeros((n, num_landmarks))
            
            for i, landmark in enumerate(landmarks):
                dist = dijkstra(adj_matrix, directed=False, indices=landmark)
                distances[:, i] = dist
            
            # Handle infinite distances (disconnected components)
            max_finite_dist = np.max(distances[np.isfinite(distances)])
            distances[np.isinf(distances)] = 2 * max_finite_dist
            
            # Step 3: Apply classical MDS on the distance matrix
            # Center the squared distance matrix
            D_squared = distances ** 2
            n_samples = D_squared.shape[0]
            n_landmarks = D_squared.shape[1]
            
            # Centering matrix
            landmark_mean = D_squared.mean(axis=0)
            overall_mean = D_squared.mean()
            
            # Double centering
            B = -0.5 * (D_squared - landmark_mean - D_squared.mean(axis=1, keepdims=True) + overall_mean)
            
            # Step 4: Compute eigendecomposition and extract coordinates
            # Use PCA for efficiency (equivalent to eigendecomposition)
            pca = PCA(n_components=dimensions)
            coords = pca.fit_transform(B)
                        
            # Get landmark coordinates
            landmark_coords = coords[landmarks]
        else:
            # Use landmark positions provided by user
            # ... note that the algorithm assumes the positions are within some range (guessing, centered around origin)
            coords = np.array([landmark_pos[_nodes_as_list_[landmark]] for landmark in landmarks])
            # Get landmark coordinates
            landmark_coords = coords

        # Refine non-landmark positions using weighted least squares
        # (triangulation based on distances to landmarks)
        if isinstance(g, nx.Graph):
            adj_matrix = nx.to_scipy_sparse_array(g, weight='weight', format='csr')
        elif isinstance(g, csr_matrix):
            adj_matrix = g
        else:
            adj_matrix = csr_matrix(g)
        
        n = adj_matrix.shape[0]
        
        # Compute distances to landmarks again
        distances = np.zeros((n, len(landmarks)))
        for i, landmark in enumerate(landmarks):
            dist = dijkstra(adj_matrix, directed=False, indices=landmark)
            distances[:, i] = dist
        
        max_finite_dist = np.max(distances[np.isfinite(distances)])
        distances[np.isinf(distances)] = 2 * max_finite_dist

        self.coords, self.landmarks, self.node_mapping = coords, landmarks, node_mapping
    
    def results(self):
        _pos_ = {}
        for i in range(len(self.coords)): _pos_[self.node_mapping[i]] = self.coords[i]
        return _pos_

#
# Implementation of PivotMDS
#
# U. Brandes and C. Pich. Eigensolver methods for progressive multidimensional scaling of large data. 
# In Proceedings 14th Symposium on Graph Drawing (GD), pages 42–53, 2006.
#
# Generated by Claude Sonnet 4.5 (2025-10-23)
#
# Note that this implementation repeats the dijkstra algorithm (instead of re-using the pivot selection results)
#
class PivotMDSLayout(object):
    def __init__(self, g, num_pivots=None, dimensions=2):
        """
        Pivot MDS with MaxMin pivot selection strategy.
        
        Instead of random selection, pivots are chosen iteratively to maximize
        the minimum distance to already selected pivots, providing better coverage.
        
        Parameters:
        -----------
        g : networkx.Graph or scipy.sparse matrix or numpy.ndarray
            Input graph.
        num_pivots : int, optional
            Number of pivot nodes to select.
        dimensions : int, default=2
            Number of dimensions for the final embedding.
        
        Returns:
        --------
        coords : numpy.ndarray
            Coordinates of all nodes in the embedded space.
        pivots : numpy.ndarray
            Indices of selected pivot nodes.
        node_mapping : list or None
            List mapping array indices to original node identifiers.
        """
        
        # Convert graph to adjacency matrix if needed
        if isinstance(g, nx.Graph):
            n = g.number_of_nodes()
            node_mapping = list(g.nodes())
            adj_matrix = nx.to_scipy_sparse_array(g, weight='weight', format='csr')
        elif isinstance(g, csr_matrix):
            adj_matrix = g
            n = adj_matrix.shape[0]
            node_mapping = None
        elif isinstance(g, np.ndarray):
            adj_matrix = csr_matrix(g)
            n = adj_matrix.shape[0]
            node_mapping = None
        else:
            raise ValueError("Graph must be NetworkX graph, scipy sparse matrix, or numpy array")
        
        # Set number of pivots if not specified
        if num_pivots is None:
            num_pivots = max(int(np.sqrt(n)), dimensions + 1)
        
        num_pivots = min(num_pivots, n)
        
        # Step 1: Select pivots using MaxMin strategy
        pivots = []
        
        # Select first pivot randomly
        first_pivot = np.random.randint(0, n)
        pivots.append(first_pivot)
        
        # Initialize minimum distances to first pivot
        min_distances = dijkstra(adj_matrix, directed=False, indices=first_pivot)
        min_distances[np.isinf(min_distances)] = 0
        
        # Select remaining pivots
        for _ in range(num_pivots - 1):
            # Select node with maximum minimum distance to existing pivots
            next_pivot = np.argmax(min_distances)
            pivots.append(next_pivot)
            
            # Update minimum distances
            new_distances = dijkstra(adj_matrix, directed=False, indices=next_pivot)
            new_distances[np.isinf(new_distances)] = 0
            min_distances = np.minimum(min_distances, new_distances)
        
        pivots = np.array(pivots)
        
        # Step 2: Compute distances from all nodes to selected pivots
        distances = np.zeros((n, num_pivots))
        
        for i, pivot in enumerate(pivots):
            dist = dijkstra(adj_matrix, directed=False, indices=pivot)
            distances[:, i] = dist
        
        # Handle infinite distances
        max_finite_dist = np.max(distances[np.isfinite(distances)])
        distances[np.isinf(distances)] = 2 * max_finite_dist
        
        # Step 3: Center and reduce dimensions
        col_means = distances.mean(axis=0)
        distances_centered = distances - col_means
        
        # Step 4: SVD for dimensionality reduction
        if num_pivots > dimensions:
            U, S, Vt = np.linalg.svd(distances_centered, full_matrices=False)
            coords = U[:, :dimensions] * S[:dimensions]
        else:
            coords = distances_centered
        
        self.coords, self.pivots, self.node_mapping = coords, pivots, node_mapping

    def results(self):
        _pos_ = {}
        for i in range(len(self.coords)): _pos_[self.node_mapping[i]] = self.coords[i]
        return _pos_
