import urllib
from typing import Dict


class _PoolManagerWrapper:
    """
    Wrapper around urllib3.PoolManager that implements request() which
    is the only method used by the autogenerated API code

    It can be used to replace tiledb.cloud.rest_api.rest.RESTClientObject.pool_manager
    at runtime in order to cache HTTP redirects and work around
    https://github.com/urllib3/urllib3/issues/2475
    """

    def __init__(self, pool_manager):
        self._pool = pool_manager
        # The cache key is the URL without the query string, while
        # the cache value is the FQDN (netloc) of the redirect location
        # For example:
        # {'https://api.tiledb.com/v1/user': 'us-east-1.aws.api.tiledb.com'}
        self._redirect_cache: Dict[str, str] = {}

    def request(self, method, url, **kwargs):
        kwargs["retries"] = self._pool.connection_pool_kw.get("retries")

        parsed_url = urllib.parse.urlparse(url)
        cacheable_url = urllib.parse.urljoin(
            f"{parsed_url.scheme}://{parsed_url.netloc}",
            parsed_url.path,
        )

        cached_netloc = self._redirect_cache.get(cacheable_url)
        if cached_netloc:
            url = parsed_url._replace(netloc=cached_netloc).geturl()

        resp = self._pool.request(method, url, **kwargs)

        for retry in reversed(resp.retries.history):
            if retry.redirect_location:
                redirect_netloc = urllib.parse.urlparse(retry.redirect_location).netloc
                self._redirect_cache[cacheable_url] = redirect_netloc

                break

        return resp
