"""
Исключения для XMLRiver Pro API
"""

from typing import Optional


class XMLRiverError(Exception):
    """Базовое исключение для XMLRiver"""

    def __init__(self, code: int, message: str):
        self.code = code
        self.message = message
        super().__init__(f"[{code}] {message}")


class AuthenticationError(XMLRiverError):
    """Ошибка аутентификации"""


class RateLimitError(XMLRiverError):
    """Превышен лимит запросов"""


class NoResultsError(XMLRiverError):
    """Нет результатов поиска"""


class NetworkError(XMLRiverError):
    """Ошибка сети"""


class ValidationError(XMLRiverError):
    """Ошибка валидации параметров"""


class InsufficientFundsError(XMLRiverError):
    """Недостаточно средств на счету"""


class ServiceUnavailableError(XMLRiverError):
    """Сервис недоступен"""


class APIError(XMLRiverError):
    """Ошибка API"""


# Коды ошибок XMLRiver API (официальная документация)
ERROR_CODES = {
    2: "Задан пустой поисковый запрос (в элементе query передано пустое значение)",
    15: "Для заданного поискового запроса отсутствуют результаты поиска",
    20: "Внутренняя ошибка. Обратитесь в службу поддержки",
    21: "Внутренняя ошибка. Обратитесь в службу поддержки",
    22: "Внутренняя ошибка. Обратитесь в службу поддержки",
    23: "Внутренняя ошибка. Обратитесь в службу поддержки",
    24: "Внутренняя ошибка. Обратитесь в службу поддержки",
    31: "Пользователь не зарегистрирован на сервисе",
    42: (
        "Ключ, выданный при регистрации, содержит ошибку. "
        "Необходимо проверить корректность адреса"
    ),
    45: ("С вашего IP сбор запрещён. " "Проверьте настройки доступа по IP в кабинете"),
    101: "Сервис сбора данных на обновлении. Попробуйте чуть позже",
    102: (
        "Неверный параметр groupby! Возможные значения для Google: "
        "10, 20, 30, 50, 100. Для Яндекс: 10"
    ),
    103: (
        "Неверный параметр lr! Может возникнуть при передаче "
        "неверного значения параметра"
    ),
    104: (
        "Неверный параметр loc! Может возникнуть при передаче "
        "неверного значения параметра"
    ),
    105: (
        "Неверный параметр country! Может возникнуть при передаче "
        "неверного значения параметра"
    ),
    106: (
        "Неверный параметр domain! Может возникнуть при передаче "
        "неверного значения параметра"
    ),
    107: (
        "По Яндексу возможное значение ТОПа: 10! Если в запросе передан "
        "параметр groupby и он не равен возможным значениям"
    ),
    108: (
        "Не передан параметр zoom или coords для Google карт. "
        "Параметры zoom и coords – обязательны для поиска заведений по картам Google"
    ),
    110: (
        "Заняты все доступные вам каналы для сбора данных. Попробуйте позже. "
        "Вами превышено максимально допустимое количество потоков"
    ),
    111: (
        "Нет свободных каналов для сбора данных. Попробуйте позже. "
        "По техническим причинам сейчас нет больше свободных каналов"
    ),
    115: (
        "Ваше ПО слишком часто отправляет большее количество параллельных "
        "запросов, чем вам доступно. Доступ откроется через N секунд"
    ),
    120: "Недопустимые символы или операторы в запросе",
    121: (
        "Bad request id. Возвращается при сборе в отложенном режиме "
        "если переданный req_id у нас не найден"
    ),
    200: (
        "На вашем счету закончились деньги. "
        "Для дальнейшей работы пополните свой счет"
    ),
    201: (
        "Ваше ПО не забирает ответы по запросам! Прием запросов приостановлен "
        "на 20 минут для предотвращения пустой траты денег"
    ),
    202: (
        "Запрос ещё не обработан. Ошибка может возникать при работе с "
        "Яндекс Search API. Ответ от Яндекса ещё не готов"
    ),
    500: (
        "Ошибка сети. Повторите попытку. До 20% ответов с этой ошибкой – "
        "нормальное поведение сервиса"
    ),
    999: "Неизвестная ошибка",
}


def get_error_message(code: int) -> str:
    """Получить сообщение об ошибке по коду"""
    return ERROR_CODES.get(code, ERROR_CODES[999])


def raise_xmlriver_error(code: int, message: Optional[str] = None) -> None:
    """Вызвать соответствующее исключение по коду ошибки"""
    if message is None:
        message = get_error_message(code)

    # Нет результатов поиска
    if code == 15:
        raise NoResultsError(code, message)

    # Ошибки аутентификации и доступа
    if code in [31, 42, 45]:
        raise AuthenticationError(code, message)

    # Ошибки лимитов и потоков
    if code in [110, 111, 115]:
        raise RateLimitError(code, message)

    # Ошибки сети (временные, требуют повтора)
    if code in [500, 202]:
        raise NetworkError(code, message)

    # Недостаточно средств
    if code == 200:
        raise InsufficientFundsError(code, message)

    # Сервис недоступен
    if code in [101, 201]:
        raise ServiceUnavailableError(code, message)

    # Ошибки валидации параметров
    if code in [2, 102, 103, 104, 105, 106, 107, 108, 120, 121]:
        raise ValidationError(code, message)

    # Остальные ошибки API
    raise APIError(code, message)
