"""supporting functions"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../../nbs/utils/convert.ipynb.

# %% auto 0
__all__ = ['convert_json_to_obj', 'convert_enum_member_to_name', 'convert_str_to_snake_case', 'convert_str_remove_accents',
           'convert_str_keep_alphanumeric', 'convert_str_remove_emojis', 'convert_str_file_name', 'convert_str_to_date',
           'convert_datetime_to_str', 'sort_dictionary']

# %% ../../nbs/utils/convert.ipynb 3
from enum import Enum
import json
from typing import Union
import re
import unicodedata

import datetime as dt
from dateutil.parser import parse as dtu_parse

# %% ../../nbs/utils/convert.ipynb 5
def convert_json_to_obj(json_str: Union[str, dict]) -> dict:
    """
    receives a json string and converts to a dict
    """
    if isinstance(json_str, str):
        return json.loads(json_str)
    return json_str

# %% ../../nbs/utils/convert.ipynb 6
def convert_enum_member_to_name(enum_member):
    return enum_member.name if isinstance(enum_member, Enum) else enum_member

# %% ../../nbs/utils/convert.ipynb 7
def convert_str_to_snake_case(text_str):
    """converts 'snake_case_str' to 'snakeCaseStr'"""

    return text_str.replace(" ", "_").lower()

# %% ../../nbs/utils/convert.ipynb 8
def convert_str_remove_accents(text_str: str) -> str:
    return "".join(
        c
        for c in unicodedata.normalize("NFD", text_str)
        if unicodedata.category(c) != "Mn"
    )

# %% ../../nbs/utils/convert.ipynb 10
def convert_str_keep_alphanumeric(text_str) -> str:
    pattern = r"[^0-9a-zA-Z_\-\s]+"

    return re.sub(pattern, "", text_str)

# %% ../../nbs/utils/convert.ipynb 11
def convert_str_remove_emojis(text_str: str) -> str:
    emoj = re.compile(
        "["
        "\U0001f600-\U0001f64f"  # emoticons
        "\U0001f300-\U0001f5ff"  # symbols & pictographs
        "\U0001f680-\U0001f6ff"  # transport & map symbols
        "\U0001f1e0-\U0001f1ff"  # flags (iOS)
        "\U00002500-\U00002bef"  # chinese char
        "\U00002702-\U000027b0"
        "\U00002702-\U000027b0"
        "\U000024c2-\U0001f251"
        "\U0001f926-\U0001f937"
        "\U00010000-\U0010ffff"
        "\u2640-\u2642"
        "\u2600-\u2b55"
        "\u200d"
        "\u23cf"
        "\u23e9"
        "\u231a"
        "\ufe0f"  # dingbats
        "\u3030"
        "]+",
        re.UNICODE,
    )
    return re.sub(emoj, "", text_str)

# %% ../../nbs/utils/convert.ipynb 13
def convert_str_file_name(text_str: str) -> str:
    """convert strings to clean file name or url"""

    return convert_str_keep_alphanumeric(
        convert_str_to_snake_case(convert_str_remove_accents(text_str))
    )

# %% ../../nbs/utils/convert.ipynb 15
def convert_str_to_date(datefield: str) -> dt.datetime:
    """converts string date to datetime object"""
    return dtu_parse(datefield) if datefield else None

# %% ../../nbs/utils/convert.ipynb 17
def convert_datetime_to_str(date_date: dt.datetime, format_str="%Y-%m-%d %H:%M") -> str:
    return date_date.strftime(format_str) if date_date else None

# %% ../../nbs/utils/convert.ipynb 18
def sort_dictionary(my_dict):
    my_keys = list(my_dict.keys())
    my_keys.sort()
    return {key: my_dict[key] for key in my_keys}
