"""Edge deployment strategy for AI models (Ollama, LFM, local models)."""

import json
from uuid import UUID

import asyncio
import httpx
import typing as t
from pydantic import SecretStr

from acb.adapters import AdapterCapability, AdapterMetadata, AdapterStatus
from acb.depends import depends

from ._base import (
    AIBase,
    AIBaseSettings,
    AIRequest,
    AIResponse,
    DeploymentStrategy,
    ModelCapability,
    ModelInfo,
    ModelProvider,
    StreamingResponse,
    validate_request,
)

MODULE_ID = UUID("0197ff44-8c12-7f30-af61-2d41c6c89a74")
MODULE_STATUS = AdapterStatus.STABLE

MODULE_METADATA = AdapterMetadata(
    module_id=MODULE_ID,
    name="Edge AI",
    category="ai",
    provider="edge",
    version="1.0.0",
    acb_min_version="0.19.0",
    author="lesleslie <les@wedgwoodwebworks.com>",
    created_date="2025-01-15",
    last_modified="2025-01-15",
    status=MODULE_STATUS,
    capabilities=[
        AdapterCapability.ASYNC_OPERATIONS,
        AdapterCapability.STREAMING,
        AdapterCapability.EDGE_INFERENCE,
        AdapterCapability.MODEL_CACHING,
        AdapterCapability.MODEL_QUANTIZATION,
        AdapterCapability.COLD_START_OPTIMIZATION,
        AdapterCapability.TEXT_GENERATION,
        AdapterCapability.MULTIMODAL_PROCESSING,
        AdapterCapability.PROMPT_TEMPLATING,
        AdapterCapability.FALLBACK_MECHANISMS,
    ],
    required_packages=[
        "ollama>=0.1.0",
        "onnx>=1.14.0",
        "onnxruntime>=1.16.0",
        "transformers>=4.30.0",
        "torch>=2.0.0",
        "httpx[http2]>=0.28.0",
    ],
    description="Edge AI adapter for Ollama, Liquid AI LFM, and local model inference with memory optimization",
    settings_class="EdgeAISettings",
    config_example={
        "provider": "ollama",
        "ollama_host": "http://localhost:11434",
        "default_model": "llama2",
        "memory_budget_mb": 512,
        "enable_quantization": True,
        "cold_start_optimization": True,
    },
)


class EdgeAISettings(AIBaseSettings):
    """Settings for edge AI deployment."""

    # Provider selection for edge deployment
    provider: ModelProvider = ModelProvider.OLLAMA

    # Ollama settings
    ollama_host: str = "http://localhost:11434"
    ollama_timeout: float = 120.0

    # Liquid AI LFM settings
    liquid_ai_endpoint: str | None = None
    liquid_ai_api_key: SecretStr | None = None
    lfm_model_path: str | None = None

    # Local model settings
    local_model_path: str | None = None
    model_cache_dir: str = "/tmp/acb_ai_models"  # nosec B108

    # Performance optimization
    memory_budget_mb: int = 1024
    enable_quantization: bool = True
    quantization_bits: int = 8
    enable_gpu: bool = True
    max_concurrent_requests: int = 4

    # Cold start optimization
    cold_start_optimization: bool = True
    model_preload: bool = True
    keep_alive_minutes: int = 30

    # LFM-specific settings
    lfm_adaptive_weights: bool = True
    lfm_precision: str = "fp16"  # fp32, fp16, int8
    lfm_deployment_target: str = "edge"  # edge, mobile, server

    # Edge-specific limits
    max_context_length: int = 4096
    max_tokens_per_request: int = 512


class EdgeAI(AIBase):
    """Edge AI adapter implementation for local inference."""

    def __init__(self, **kwargs: t.Any) -> None:
        super().__init__(**kwargs)
        self._settings = EdgeAISettings(**kwargs)
        self._http_client: httpx.AsyncClient | None = None
        self._model_loaded: bool = False
        self._model_info_cache: dict[str, ModelInfo] = {}

    @property
    def settings(self) -> EdgeAISettings:
        """Get adapter settings with correct type."""
        if self._settings is None:
            msg = "Settings not initialized"
            raise RuntimeError(msg)
        return self._settings  # type: ignore[return-value]

    async def _create_client(self) -> t.Any:
        """Create edge AI client based on provider."""
        if self.settings.provider == ModelProvider.OLLAMA:
            return await self._create_ollama_client()
        if self.settings.provider == ModelProvider.LIQUID_AI:
            return await self._create_liquid_ai_client()
        if self.settings.provider == ModelProvider.LOCAL:
            return await self._create_local_client()
        msg = f"Unsupported edge provider: {self.settings.provider}"
        raise ValueError(msg)

    async def _create_ollama_client(self) -> httpx.AsyncClient:
        """Create Ollama HTTP client."""
        if self._http_client is None:
            self._http_client = httpx.AsyncClient(
                base_url=self.settings.ollama_host,
                timeout=self.settings.ollama_timeout,
                limits=httpx.Limits(
                    max_connections=self.settings.max_concurrent_requests,
                    max_keepalive_connections=2,
                ),
            )
            self.register_resource(self._http_client)

        # Test connectivity and preload model if enabled
        if self.settings.model_preload:
            await self._preload_model()

        return self._http_client

    def _get_lfm_model_id(self) -> str:
        """Get HuggingFace model ID for LFM2."""
        lfm_models = {
            "lfm2-350m": "liquid-ai/lfm2-350m",
            "lfm2-700m": "liquid-ai/lfm2-700m",
            "lfm2-1.2b": "liquid-ai/lfm2-1.2b",
        }

        model_name = self.settings.default_model
        if model_name not in lfm_models:
            model_name = "lfm2-350m"  # Default to smallest for edge

        return lfm_models[model_name]

    async def _create_liquid_ai_client(self) -> t.Any:
        """Create Liquid AI LFM client using HuggingFace transformers."""
        try:
            from transformers import AutoModelForCausalLM, AutoTokenizer
        except ImportError:
            msg = (
                "transformers package required for Liquid AI LFM provider. "
                "Install with: uv add transformers"
            )
            raise ImportError(
                msg,
            )

        model_id = self._get_lfm_model_id()

        class LiquidAIClient:
            """Real LFM2 client using HuggingFace transformers."""

            def __init__(self, model_id: str, settings: EdgeAISettings) -> None:
                self.model_id = model_id
                self.settings = settings
                self._model: t.Any = None
                self._tokenizer: t.Any = None
                self._models_loaded: dict[str, str] = {}

            async def load_model(self, model_name: str, **config: t.Any) -> str:
                """Load LFM2 model from HuggingFace."""
                if model_name in self._models_loaded:
                    return self._models_loaded[model_name]

                # Run model loading in executor to avoid blocking
                loop = asyncio.get_event_loop()

                def load() -> tuple[t.Any, t.Any]:
                    # Determine quantization settings
                    load_kwargs: dict[str, t.Any] = {
                        "low_cpu_mem_usage": True,
                        "trust_remote_code": True,
                    }

                    if self.settings.enable_quantization:
                        if self.settings.quantization_bits == 8:
                            load_kwargs["load_in_8bit"] = True
                        elif self.settings.quantization_bits == 4:
                            load_kwargs["load_in_4bit"] = True

                    # Load tokenizer
                    tokenizer = AutoTokenizer.from_pretrained(  # type: ignore[no-untyped-call]
                        self.model_id,
                        trust_remote_code=True,
                        revision="main",  # nosec B615
                    )

                    # Load model with edge optimizations
                    model = AutoModelForCausalLM.from_pretrained(
                        self.model_id,
                        revision="main",  # nosec B615
                        **load_kwargs,
                    )

                    return (model, tokenizer)

                result = await loop.run_in_executor(None, load)
                self._model, self._tokenizer = result
                self._models_loaded[model_name] = model_name
                return model_name

            async def generate(
                self,
                model_id: str,
                prompt: str,
                **kwargs: t.Any,
            ) -> t.Any:
                """Generate text using LFM2 model."""
                if self._model is None or self._tokenizer is None:
                    msg = "Model not loaded. Call load_model first."
                    raise RuntimeError(msg)

                # Run inference in executor
                loop = asyncio.get_event_loop()
                start_time = loop.time()

                def inference() -> tuple[str, int]:
                    inputs = self._tokenizer(prompt, return_tensors="pt")
                    max_new_tokens = kwargs.get("max_tokens", 512)
                    temperature = kwargs.get("temperature", 0.7)

                    outputs = self._model.generate(
                        **inputs,
                        max_new_tokens=max_new_tokens,
                        temperature=temperature,
                        do_sample=True,
                        pad_token_id=self._tokenizer.eos_token_id,
                    )

                    generated_text = self._tokenizer.decode(
                        outputs[0],
                        skip_special_tokens=True,
                    )

                    # Remove input prompt from output
                    if generated_text.startswith(prompt):
                        generated_text = generated_text[len(prompt) :].strip()

                    return (generated_text, len(outputs[0]))

                text, tokens_used = await loop.run_in_executor(None, inference)
                latency_ms = int((loop.time() - start_time) * 1000)

                # Create response object
                class Response:
                    def __init__(self, text: str, tokens: int, latency: int) -> None:
                        self.text = text
                        self.tokens_used = tokens
                        self.latency_ms = latency
                        self.memory_usage_mb = 256  # LFM2 optimized memory

                return Response(text, tokens_used, latency_ms)

        client = LiquidAIClient(model_id, self.settings)

        # Preload default model if enabled
        if self.settings.model_preload:
            await client.load_model(self.settings.default_model)

        return client

    async def _create_local_client(self) -> t.Any:
        """Create local model client using transformers."""
        try:
            from transformers import pipeline
        except ImportError:
            msg = "transformers package required for local provider"
            raise ImportError(msg)

        # Create text generation pipeline with optimization
        device = 0 if self.settings.enable_gpu else -1

        model_path = self.settings.local_model_path or "microsoft/DialoGPT-small"

        model_kwargs: dict[str, t.Any] = {
            "torch_dtype": "auto",
            "low_cpu_mem_usage": True,
        }

        if self.settings.enable_quantization:
            # Add quantization for memory efficiency
            model_kwargs["load_in_8bit"] = True

        return pipeline(
            task="text-generation",
            model=model_path,
            device=device,
            model_kwargs=model_kwargs,
        )

    async def _check_and_pull_ollama_model(self, client: t.Any) -> bool:
        """Check if Ollama model exists and pull if needed. Returns success status."""
        response = await client.get("/api/tags")
        if response.status_code != 200:
            return False

        models = response.json().get("models", [])
        model_names = [m["name"] for m in models]

        if self.settings.default_model in model_names:
            return True

        # Pull model if missing
        self._log_info(f"Pulling model: {self.settings.default_model}")
        pull_response = await client.post(
            "/api/pull",
            json={"name": self.settings.default_model},
            timeout=600.0,  # Model pulling can take time
        )

        if pull_response.status_code != 200:
            self._log_error(f"Failed to pull model: {pull_response.text}")
            return False

        return True

    async def _load_ollama_model_into_memory(self, client: t.Any) -> None:
        """Load Ollama model into memory with keep-alive."""
        await client.post(
            "/api/generate",
            json={
                "model": self.settings.default_model,
                "prompt": "Hello",
                "stream": False,
                "keep_alive": f"{self.settings.keep_alive_minutes}m",
            },
        )
        self._model_loaded = True
        self._log_info(f"Model preloaded: {self.settings.default_model}")

    async def _preload_model(self) -> None:
        """Preload model for faster inference."""
        if self._model_loaded:
            return

        if self.settings.provider != ModelProvider.OLLAMA:
            return

        if self._http_client is None:
            self._log_error("HTTP client not initialized")
            return

        try:
            # Check and pull model if needed
            if await self._check_and_pull_ollama_model(self._http_client):
                # Load model into memory
                await self._load_ollama_model_into_memory(self._http_client)

        except Exception as e:
            self._log_warning(f"Model preload failed: {e}")

    async def _generate_text(self, request: AIRequest) -> AIResponse:
        """Generate text using edge provider."""
        await validate_request(request)
        client = await self._ensure_client()

        # Apply edge-specific limits
        request.max_tokens = min(
            request.max_tokens,
            self.settings.max_tokens_per_request,
        )

        start_time = asyncio.get_event_loop().time()

        try:
            if self.settings.provider == ModelProvider.OLLAMA:
                response = await self._ollama_generate(client, request)
            elif self.settings.provider == ModelProvider.LIQUID_AI:
                response = await self._liquid_ai_generate(client, request)
            elif self.settings.provider == ModelProvider.LOCAL:
                response = await self._local_generate(client, request)
            else:
                msg = f"Unsupported provider: {self.settings.provider}"
                raise ValueError(msg)

            latency_ms = int((asyncio.get_event_loop().time() - start_time) * 1000)
            response.latency_ms = latency_ms

            return response

        except Exception as e:
            self._log_exception(f"Edge text generation failed: {e}")
            raise

    async def _generate_text_stream(self, request: AIRequest) -> StreamingResponse:
        """Generate streaming text response on edge."""
        await validate_request(request)
        client = await self._ensure_client()

        if self.settings.provider == ModelProvider.OLLAMA:
            generator = self._ollama_stream(client, request)
        elif self.settings.provider == ModelProvider.LIQUID_AI:
            generator = self._liquid_ai_stream(client, request)
        else:
            msg = f"Streaming not supported for provider: {self.settings.provider}"
            raise ValueError(
                msg,
            )

        return StreamingResponse(generator)

    async def _ollama_generate(
        self,
        client: httpx.AsyncClient,
        request: AIRequest,
    ) -> AIResponse:
        """Generate text using Ollama."""
        prompt = (
            request.prompt if isinstance(request.prompt, str) else str(request.prompt)
        )

        if request.system_prompt:
            prompt = f"System: {request.system_prompt}\n\nUser: {prompt}"

        payload = {
            "model": request.model or self.settings.default_model,
            "prompt": prompt,
            "stream": False,
            "options": {
                "num_predict": request.max_tokens,
                "temperature": request.temperature,
            },
            "keep_alive": f"{self.settings.keep_alive_minutes}m",
        }

        response = await client.post("/api/generate", json=payload)
        response.raise_for_status()

        data = response.json()

        return AIResponse(
            content=data.get("response", ""),
            model=data.get("model", request.model or self.settings.default_model),
            provider=ModelProvider.OLLAMA,
            strategy=DeploymentStrategy.EDGE,
            tokens_used=data.get("eval_count", 0) + data.get("prompt_eval_count", 0),
            finish_reason="stop" if data.get("done") else "length",
        )

    async def _liquid_ai_generate(
        self,
        client: t.Any,
        request: AIRequest,
    ) -> AIResponse:
        """Generate text using Liquid AI LFM."""
        prompt = (
            request.prompt if isinstance(request.prompt, str) else str(request.prompt)
        )
        model_name = request.model or self.settings.default_model

        # Load model if not already loaded
        if model_name not in client._models_loaded:
            await client.load_model(
                model_name,
                deployment_target=self.settings.lfm_deployment_target,
                precision=self.settings.lfm_precision,
                adaptive_weights=self.settings.lfm_adaptive_weights,
                memory_budget_mb=self.settings.memory_budget_mb,
            )

        model_id = client._models_loaded[model_name]

        response = await client.generate(
            model_id=model_id,
            prompt=prompt,
            max_tokens=request.max_tokens,
            temperature=request.temperature,
        )

        return AIResponse(
            content=response.text,
            model=model_name,
            provider=ModelProvider.LIQUID_AI,
            strategy=DeploymentStrategy.EDGE,
            tokens_used=response.tokens_used,
            latency_ms=response.latency_ms,
        )

    async def _local_generate(self, client: t.Any, request: AIRequest) -> AIResponse:
        """Generate text using local transformers model."""
        prompt = (
            request.prompt if isinstance(request.prompt, str) else str(request.prompt)
        )

        # Run inference in thread pool to avoid blocking
        loop = asyncio.get_event_loop()

        def generate() -> list[dict[str, t.Any]]:
            return client(  # type: ignore[no-any-return]
                prompt,
                max_length=len(prompt.split()) + request.max_tokens,
                temperature=request.temperature,
                do_sample=True,
                pad_token_id=client.tokenizer.eos_token_id,
            )

        results = await loop.run_in_executor(None, generate)

        # Extract generated text (remove input prompt)
        generated_text: str = results[0]["generated_text"]
        if generated_text.startswith(prompt):
            generated_text = generated_text[len(prompt) :].strip()

        return AIResponse(
            content=generated_text,
            model=request.model or "local",
            provider=ModelProvider.LOCAL,
            strategy=DeploymentStrategy.EDGE,
            tokens_used=len(generated_text.split()),  # Rough estimate
        )

    def _build_ollama_prompt(self, request: AIRequest) -> str:
        """Build Ollama prompt with optional system prompt."""
        prompt = (
            request.prompt if isinstance(request.prompt, str) else str(request.prompt)
        )

        if request.system_prompt:
            prompt = f"System: {request.system_prompt}\n\nUser: {prompt}"

        return prompt

    def _build_ollama_payload(
        self, request: AIRequest, prompt: str
    ) -> dict[str, t.Any]:
        """Build Ollama API payload."""
        return {
            "model": request.model or self.settings.default_model,
            "prompt": prompt,
            "stream": True,
            "options": {
                "num_predict": request.max_tokens,
                "temperature": request.temperature,
            },
            "keep_alive": f"{self.settings.keep_alive_minutes}m",
        }

    def _process_ollama_stream_line(self, line: str) -> tuple[str | None, bool]:
        """Process a single Ollama streaming line. Returns (text, is_done)."""
        if not line.strip():
            return None, False

        try:
            data = json.loads(line)
            text = data.get("response")
            is_done = data.get("done", False)
            return text, is_done
        except json.JSONDecodeError:
            return None, False

    async def _ollama_stream(
        self,
        client: httpx.AsyncClient,
        request: AIRequest,
    ) -> t.AsyncGenerator[str]:
        """Stream text generation from Ollama."""
        prompt = self._build_ollama_prompt(request)
        payload = self._build_ollama_payload(request, prompt)

        async with client.stream("POST", "/api/generate", json=payload) as response:
            response.raise_for_status()

            async for line in response.aiter_lines():
                text, is_done = self._process_ollama_stream_line(line)
                if text:
                    yield text
                if is_done:
                    break

    async def _liquid_ai_stream(
        self,
        client: t.Any,
        request: AIRequest,
    ) -> t.AsyncGenerator[str]:
        """Stream text generation from Liquid AI LFM."""
        # Mock streaming implementation for LFM
        response = await self._liquid_ai_generate(client, request)

        # Simulate streaming by chunking the response
        content = response.content
        chunk_size = 10  # Characters per chunk

        for i in range(0, len(content), chunk_size):
            chunk = content[i : i + chunk_size]
            yield chunk
            await asyncio.sleep(0.01)  # Simulate streaming delay

    async def _get_available_models(self) -> list[ModelInfo]:
        """Get available models for edge deployment."""
        if self.settings.provider == ModelProvider.OLLAMA:
            return await self._get_ollama_models()
        if self.settings.provider == ModelProvider.LIQUID_AI:
            return await self._get_liquid_ai_models()
        if self.settings.provider == ModelProvider.LOCAL:
            return await self._get_local_models()
        return []

    async def _get_ollama_models(self) -> list[ModelInfo]:
        """Get available Ollama models."""
        try:
            client = await self._ensure_client()
            response = await client.get("/api/tags")
            response.raise_for_status()

            data = response.json()
            models = []

            for model_data in data.get("models", []):
                name = model_data["name"]
                size_mb = model_data.get("size", 0) // (1024 * 1024)

                models.append(
                    ModelInfo(
                        name=name,
                        provider=ModelProvider.OLLAMA,
                        capabilities=[
                            ModelCapability.TEXT_GENERATION,
                            ModelCapability.CHAT_COMPLETION,
                        ],
                        context_length=self.settings.max_context_length,
                        deployment_strategies=[DeploymentStrategy.EDGE],
                        memory_footprint_mb=size_mb,
                        latency_p95_ms=200,  # Typical edge latency
                        supports_streaming=True,
                    ),
                )

            return models

        except Exception as e:
            self._log_exception(f"Failed to get Ollama models: {e}")
            return []

    async def _get_liquid_ai_models(self) -> list[ModelInfo]:
        """Get available Liquid AI LFM models."""
        return [
            ModelInfo(
                name="lfm-7b",
                provider=ModelProvider.LIQUID_AI,
                capabilities=[
                    ModelCapability.TEXT_GENERATION,
                    ModelCapability.CHAT_COMPLETION,
                ],
                context_length=8192,
                deployment_strategies=[DeploymentStrategy.EDGE],
                memory_footprint_mb=256,  # 70% less than traditional 7B models
                latency_p95_ms=45,  # 3x faster than traditional models
                supports_streaming=True,
            ),
            ModelInfo(
                name="lfm2",
                provider=ModelProvider.LIQUID_AI,
                capabilities=[
                    ModelCapability.TEXT_GENERATION,
                    ModelCapability.CHAT_COMPLETION,
                ],
                context_length=16384,
                deployment_strategies=[DeploymentStrategy.EDGE],
                memory_footprint_mb=512,
                latency_p95_ms=60,
                supports_streaming=True,
            ),
            ModelInfo(
                name="lfm2-vl",
                provider=ModelProvider.LIQUID_AI,
                capabilities=[
                    ModelCapability.TEXT_GENERATION,
                    ModelCapability.CHAT_COMPLETION,
                    ModelCapability.VISION,
                ],
                context_length=8192,
                deployment_strategies=[DeploymentStrategy.EDGE],
                memory_footprint_mb=768,
                latency_p95_ms=80,
                supports_streaming=True,
            ),
        ]

    async def _get_local_models(self) -> list[ModelInfo]:
        """Get available local models."""
        return [
            ModelInfo(
                name="microsoft/DialoGPT-small",
                provider=ModelProvider.LOCAL,
                capabilities=[ModelCapability.TEXT_GENERATION],
                context_length=1024,
                deployment_strategies=[DeploymentStrategy.EDGE],
                memory_footprint_mb=400,
                latency_p95_ms=150,
                supports_streaming=False,
            ),
        ]

    async def optimize_for_edge(self, model_name: str) -> dict[str, t.Any]:
        """Optimize model for edge deployment."""
        optimizations = {
            "quantization_applied": self.settings.enable_quantization,
            "precision": self.settings.lfm_precision
            if self.settings.provider == ModelProvider.LIQUID_AI
            else "fp32",
            "memory_budget_mb": self.settings.memory_budget_mb,
            "cold_start_optimized": self.settings.cold_start_optimization,
        }

        if self.settings.provider == ModelProvider.LIQUID_AI:
            optimizations.update(
                {
                    "adaptive_weights": self.settings.lfm_adaptive_weights,
                    "deployment_target": self.settings.lfm_deployment_target,
                    "memory_reduction_percent": 70,  # LFM advantage
                    "latency_improvement_percent": 200,  # 3x faster
                },
            )

        return optimizations

    async def get_memory_usage(self) -> dict[str, t.Any]:
        """Get current memory usage stats."""
        try:
            import psutil

            process = psutil.Process()
            memory_info = process.memory_info()

            return {
                "rss_mb": memory_info.rss // (1024 * 1024),
                "vms_mb": memory_info.vms // (1024 * 1024),
                "budget_mb": self.settings.memory_budget_mb,
                "usage_percent": (memory_info.rss // (1024 * 1024))
                / self.settings.memory_budget_mb
                * 100,
                "model_loaded": self._model_loaded,
            }
        except ImportError:
            return {"error": "psutil not available for memory monitoring"}


# Alias for backward compatibility and convention
Ai = EdgeAI
AiSettings = EdgeAISettings

depends.set(Ai, "edge")
