import * as cdk8s from 'cdk8s';
import { Construct } from 'constructs';
import { ResourceProps, Resource, IResource } from './base';
/**
 * Properties for initialization of `ConfigMap`.
 *
 * @stability stable
 */
export interface ConfigMapProps extends ResourceProps {
    /**
     * BinaryData contains the binary data.
     *
     * Each key must consist of alphanumeric characters, '-', '_' or '.'.
     * BinaryData can contain byte sequences that are not in the UTF-8 range. The
     * keys stored in BinaryData must not overlap with the ones in the Data field,
     * this is enforced during validation process. Using this field will require
     * 1.10+ apiserver and kubelet.
     *
     * You can also add binary data using `configMap.addBinaryData()`.
     *
     * @stability stable
     */
    readonly binaryData?: {
        [key: string]: string;
    };
    /**
     * Data contains the configuration data.
     *
     * Each key must consist of alphanumeric characters, '-', '_' or '.'. Values
     * with non-UTF-8 byte sequences must use the BinaryData field. The keys
     * stored in Data must not overlap with the keys in the BinaryData field, this
     * is enforced during validation process.
     *
     * You can also add data using `configMap.addData()`.
     *
     * @stability stable
     */
    readonly data?: {
        [key: string]: string;
    };
}
/**
 * Represents a config map.
 *
 * @stability stable
 */
export interface IConfigMap extends IResource {
}
/**
 * ConfigMap holds configuration data for pods to consume.
 *
 * @stability stable
 */
export declare class ConfigMap extends Resource implements IConfigMap {
    /**
     * Represents a ConfigMap created elsewhere.
     *
     * @param name The name of the config map to import.
     * @stability stable
     */
    static fromConfigMapName(name: string): IConfigMap;
    /**
     * The underlying cdk8s API object.
     *
     * @see base.Resource.apiObject
     * @stability stable
     */
    protected readonly apiObject: cdk8s.ApiObject;
    private readonly _binaryData;
    private readonly _data;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props?: ConfigMapProps);
    /**
     * Adds a data entry to the config map.
     *
     * @param key The key.
     * @param value The value.
     * @stability stable
     * @throws if there is either a `data` or `binaryData` entry with the same key
     */
    addData(key: string, value: string): void;
    /**
     * The data associated with this config map.
     *
     * Returns an copy. To add data records, use `addData()` or `addBinaryData()`.
     *
     * @stability stable
     */
    get data(): Record<string, string>;
    /**
     * Adds a binary data entry to the config map.
     *
     * BinaryData can contain byte
     * sequences that are not in the UTF-8 range.
     *
     * @param key The key.
     * @param value The value.
     * @stability stable
     * @throws if there is either a `data` or `binaryData` entry with the same key
     */
    addBinaryData(key: string, value: string): void;
    /**
     * The binary data associated with this config map.
     *
     * Returns a copy. To add data records, use `addBinaryData()` or `addData()`.
     *
     * @stability stable
     */
    get binaryData(): Record<string, string>;
    /**
     * Adds a file to the ConfigMap.
     *
     * @param localFile The path to the local file.
     * @param key The ConfigMap key (default to the file name).
     * @stability stable
     */
    addFile(localFile: string, key?: string): void;
    /**
     * Adds a directory to the ConfigMap.
     *
     * @param localDir A path to a local directory.
     * @param options Options.
     * @stability stable
     */
    addDirectory(localDir: string, options?: AddDirectoryOptions): void;
    private verifyKeyAvailable;
    private synthesizeData;
    private synthesizeBinaryData;
}
/**
 * Options for `configmap.addDirectory()`.
 *
 * @stability stable
 */
export interface AddDirectoryOptions {
    /**
     * A prefix to add to all keys in the config map.
     *
     * @default ""
     * @stability stable
     */
    readonly keyPrefix?: string;
    /**
     * Glob patterns to exclude when adding files.
     *
     * @default - include all files
     * @stability stable
     */
    readonly exclude?: string[];
}
