import { ApiObject, ApiObjectMetadataDefinition, Duration } from 'cdk8s';
import { Construct } from 'constructs';
import { Resource, ResourceProps } from './base';
import { Container, ContainerProps } from './container';
import * as k8s from './imports/k8s';
import { RestartPolicy, PodTemplateProps, IPodTemplate } from './pod';
import { IServiceAccount } from './service-account';
import { Volume } from './volume';
/**
 * Properties for initialization of `Job`.
 *
 * @stability stable
 */
export interface JobProps extends ResourceProps, PodTemplateProps {
    /**
     * Specifies the duration the job may be active before the system tries to terminate it.
     *
     * @default - If unset, then there is no deadline.
     * @stability stable
     */
    readonly activeDeadline?: Duration;
    /**
     * Specifies the number of retries before marking this job failed.
     *
     * @default - If not set, system defaults to 6.
     * @stability stable
     */
    readonly backoffLimit?: number;
    /**
     * Limits the lifetime of a Job that has finished execution (either Complete or Failed).
     *
     * If this field is set, after the Job finishes, it is eligible to
     * be automatically deleted. When the Job is being deleted, its lifecycle
     * guarantees (e.g. finalizers) will be honored. If this field is set to zero,
     * the Job becomes eligible to be deleted immediately after it finishes. This
     * field is alpha-level and is only honored by servers that enable the
     * `TTLAfterFinished` feature.
     *
     * @default - If this field is unset, the Job won't be automatically deleted.
     * @stability stable
     */
    readonly ttlAfterFinished?: Duration;
}
/**
 * A Job creates one or more Pods and ensures that a specified number of them successfully terminate.
 *
 * As pods successfully complete,
 * the Job tracks the successful completions. When a specified number of successful completions is reached, the task (ie, Job) is complete.
 * Deleting a Job will clean up the Pods it created. A simple case is to create one Job object in order to reliably run one Pod to completion.
 * The Job object will start a new Pod if the first Pod fails or is deleted (for example due to a node hardware failure or a node reboot).
 * You can also use a Job to run multiple Pods in parallel.
 *
 * @stability stable
 */
export declare class Job extends Resource implements IPodTemplate {
    /**
     * Duration before job is terminated.
     *
     * If undefined, there is no deadline.
     *
     * @stability stable
     */
    readonly activeDeadline?: Duration;
    /**
     * Number of retries before marking failed.
     *
     * @stability stable
     */
    readonly backoffLimit?: number;
    /**
     * TTL before the job is deleted after it is finished.
     *
     * @stability stable
     */
    readonly ttlAfterFinished?: Duration;
    /**
     * The underlying cdk8s API object.
     *
     * @see base.Resource.apiObject
     * @stability stable
     */
    protected readonly apiObject: ApiObject;
    private readonly _podTemplate;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props?: JobProps);
    /**
     * Provides read/write access to the underlying pod metadata of the resource.
     *
     * @stability stable
     */
    get podMetadata(): ApiObjectMetadataDefinition;
    /**
     * The containers belonging to the pod.
     *
     * Use `addContainer` to add containers.
     *
     * @stability stable
     */
    get containers(): Container[];
    /**
     * The volumes associated with this pod.
     *
     * Use `addVolume` to add volumes.
     *
     * @stability stable
     */
    get volumes(): Volume[];
    /**
     * Restart policy for all containers within the pod.
     *
     * @stability stable
     */
    get restartPolicy(): RestartPolicy | undefined;
    /**
     * The service account used to run this pod.
     *
     * @stability stable
     */
    get serviceAccount(): IServiceAccount | undefined;
    /**
     * Add a container to the pod.
     *
     * @stability stable
     */
    addContainer(container: ContainerProps): Container;
    /**
     * Add a volume to the pod.
     *
     * @stability stable
     */
    addVolume(volume: Volume): void;
    /**
     * @internal
     */
    _toKube(): k8s.JobSpec;
}
