"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Protocol = exports.Service = exports.ServiceType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s = require("cdk8s");
const base_1 = require("./base");
const k8s = require("./imports/k8s");
const ingress_v1beta1_1 = require("./ingress-v1beta1");
/**
 * For some parts of your application (for example, frontends) you may want to expose a Service onto an external IP address, that's outside of your cluster.
 *
 * Kubernetes ServiceTypes allow you to specify what kind of Service you want.
 * The default is ClusterIP.
 *
 * @stability stable
 */
var ServiceType;
(function (ServiceType) {
    ServiceType["CLUSTER_IP"] = "ClusterIP";
    ServiceType["NODE_PORT"] = "NodePort";
    ServiceType["LOAD_BALANCER"] = "LoadBalancer";
    ServiceType["EXTERNAL_NAME"] = "ExternalName";
})(ServiceType = exports.ServiceType || (exports.ServiceType = {}));
/**
 * An abstract way to expose an application running on a set of Pods as a network service.
 *
 * With Kubernetes you don't need to modify your application to use an unfamiliar service discovery mechanism.
 * Kubernetes gives Pods their own IP addresses and a single DNS name for a set of Pods, and can load-balance across them.
 *
 * For example, consider a stateless image-processing backend which is running with 3 replicas. Those replicas are fungible—frontends do not care which backend they use.
 * While the actual Pods that compose the backend set may change, the frontend clients should not need to be aware of that,
 * nor should they need to keep track of the set of backends themselves.
 * The Service abstraction enables this decoupling.
 *
 * If you're able to use Kubernetes APIs for service discovery in your application, you can query the API server for Endpoints,
 * that get updated whenever the set of Pods in a Service changes. For non-native applications, Kubernetes offers ways to place a network port
 * or load balancer in between your application and the backend Pods.
 *
 * @stability stable
 */
class Service extends base_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _b, _c, _d;
        super(scope, id);
        this.apiObject = new k8s.KubeService(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s.Lazy.any({ produce: () => this._toKube() }),
        });
        this.clusterIP = props.clusterIP;
        this.externalName = props.externalName;
        if (props.externalName !== undefined) {
            this.type = ServiceType.EXTERNAL_NAME;
        }
        else {
            this.type = (_b = props.type) !== null && _b !== void 0 ? _b : ServiceType.CLUSTER_IP;
        }
        this._externalIPs = (_c = props.externalIPs) !== null && _c !== void 0 ? _c : [];
        this._ports = [];
        this._selector = {};
        this._loadBalancerSourceRanges = props.loadBalancerSourceRanges;
        for (const portAndOptions of (_d = props.ports) !== null && _d !== void 0 ? _d : []) {
            this.serve(portAndOptions.port, portAndOptions);
        }
    }
    /**
     * Expose a service via an ingress using the specified path.
     *
     * @param path The path to expose the service under.
     * @param options Additional options.
     * @returns The `Ingress` resource that was used.
     * @stability stable
     */
    exposeViaIngress(path, options = {}) {
        var _b;
        const ingress = (_b = options.ingress) !== null && _b !== void 0 ? _b : new ingress_v1beta1_1.IngressV1Beta1(this, 'Ingress');
        ingress.addRule(path, ingress_v1beta1_1.IngressV1Beta1Backend.fromService(this));
        return ingress;
    }
    /**
     * Returns the labels which are used to select pods for this service.
     *
     * @stability stable
     */
    get selector() {
        return this._selector;
    }
    /**
     * Ports for this service.
     *
     * Use `serve()` to expose additional service ports.
     *
     * @stability stable
     */
    get ports() {
        return [...this._ports];
    }
    /**
     * Associate a deployment to this service.
     *
     * If not targetPort is specific in the portOptions, then requests will be routed
     * to the port exposed by the first container in the deployment's pods.
     * The deployment's `labelSelector` will be used to select pods.
     *
     * @param deployment The deployment to expose.
     * @param options Optional settings for the port.
     * @stability stable
     */
    addDeployment(deployment, options = {}) {
        var _b, _c;
        const containers = deployment.containers;
        if (containers.length === 0) {
            throw new Error('Cannot expose a deployment without containers');
        }
        // just a PoC, we assume the first container is the main one.
        // TODO: figure out what the correct thing to do here.
        const container = containers[0];
        const port = (_b = options.port) !== null && _b !== void 0 ? _b : container.port;
        const targetPort = (_c = options.targetPort) !== null && _c !== void 0 ? _c : containers[0].port;
        if (!port) {
            throw new Error('Cannot determine port. Either pass `port` in options or configure a port on the first container of the deployment');
        }
        const selector = Object.entries(deployment.labelSelector);
        if (selector.length === 0) {
            throw new Error('deployment does not have a label selector');
        }
        if (Object.keys(this.selector).length > 0) {
            throw new Error('a selector is already defined for this service. cannot add a deployment');
        }
        for (const [k, v] of selector) {
            this.addSelector(k, v);
        }
        this.serve(port, { ...options, targetPort });
    }
    /**
     * Services defined using this spec will select pods according the provided label.
     *
     * @param label The label key.
     * @param value The label value.
     * @stability stable
     */
    addSelector(label, value) {
        this._selector[label] = value;
    }
    /**
     * Configure a port the service will bind to.
     *
     * This method can be called multiple times.
     *
     * @param port The port definition.
     * @stability stable
     */
    serve(port, options = {}) {
        this._ports.push({ ...options, port });
    }
    /**
     * @internal
     */
    _toKube() {
        if (this._ports.length === 0 && this.type !== ServiceType.EXTERNAL_NAME) {
            throw new Error('A service must be configured with a port');
        }
        if (this.type === ServiceType.EXTERNAL_NAME && this.externalName === undefined) {
            throw new Error('A service with type EXTERNAL_NAME requires an externalName prop');
        }
        const ports = [];
        for (const port of this._ports) {
            ports.push({
                name: port.name,
                port: port.port,
                targetPort: port.targetPort ? k8s.IntOrString.fromNumber(port.targetPort) : undefined,
                nodePort: port.nodePort,
                protocol: port.protocol,
            });
        }
        return this.type !== ServiceType.EXTERNAL_NAME ? {
            clusterIp: this.clusterIP,
            externalIPs: this._externalIPs,
            externalName: this.externalName,
            type: this.type,
            selector: this._selector,
            ports: ports,
            loadBalancerSourceRanges: this._loadBalancerSourceRanges,
        } : {
            type: this.type,
            externalName: this.externalName,
        };
    }
}
exports.Service = Service;
_a = JSII_RTTI_SYMBOL_1;
Service[_a] = { fqn: "cdk8s-plus-21.Service", version: "1.0.0-beta.54" };
/**
 * @stability stable
 */
var Protocol;
(function (Protocol) {
    Protocol["TCP"] = "TCP";
    Protocol["UDP"] = "UDP";
    Protocol["SCTP"] = "SCTP";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
//# sourceMappingURL=data:application/json;base64,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