# Solana/Account Model/Regular Accounts

The most commonly used type of account on the Solana network is the regular account. These accounts are typically generated by wallet applications and are used to store and manage SOL tokens. You can browse a list of popular Solana wallets on [this page](https://solana.com/solana-wallets) and choose one that suits your preferences and that you trust.

That said, here are a few recommendations that might help you navigate the blockchain space more safely:

0. Always download wallet applications from their official websites, and never trust search engines blindly. Around 2022, there was a surge of phishing websites mimicking legitimate wallets. These fake sites distributed compromised wallet apps and even paid for search engine ads to appear above the official sites in search results.
0. Choose wallets developed by teams with a solid reputation. The team should have a clean track record and clear origins.
0. Prefer wallets with a large and active user base.

## Creating a Regular Account

Creating this type of account is actually quite simple, all you need to do is generate a private key. The public key derived from that private key becomes your wallet address, which others can use to send you funds. The private key, on the other hand, acts as the password to control your wallet; only you should know it, and no one else can use your wallet without it.

Using `pxsol`, you can create a new account with private key `0x02` as follows:

```py
import pxsol

bob = pxsol.wallet.Wallet(pxsol.core.PriKey.int_decode(0x02))
ret = pxsol.rpc.get_account_info(bob.pubkey.base58(), {})
print(ret) # None
```

The newly created wallet account is not yet recorded on the Solana network. As a result, when the above code tries to query the account using the [`get_account_info`](https://solana.com/docs/rpc/http/getaccountinfo) method, the returned result will be `None`.

Transferring some SOL to this account will "activate" it.

```py
import pxsol

ada = pxsol.wallet.Wallet(pxsol.core.PriKey.int_decode(0x01))
bob = pxsol.wallet.Wallet(pxsol.core.PriKey.int_decode(0x02))

# Transfer 1 sol from ada to bob.
ada.sol_transfer(bob.pubkey, 1 * pxsol.denomination.sol)

ret = pxsol.rpc.get_account_info(bob.pubkey.base58(), {})
print(ret)
# {
#     "data": ["", "base64"],
#     "executable": false,
#     "lamports": 1000000000,
#     "owner": "11111111111111111111111111111111",
#     "rentEpoch": 18446744073709551615,
#     "space": 0,
# }
```

As you can see, Bob's account information is now recorded on the Solana network. The account owner is `1111111...`, which refers to the Solana system program.

## Destroying a Regular Account

If you no longer need a regular account or simply want to clean it up, you can destroy it. Destroying a regular account is straightforward. Suppose we want to destroy the `bob` account we created earlier, just transfer all the SOL in the account to another account. Once the balance reaches zero, the account will be automatically removed.

The transfer command looks like this:

```py
import pxsol

ada = pxsol.wallet.Wallet(pxsol.core.PriKey.int_decode(0x01))
bob = pxsol.wallet.Wallet(pxsol.core.PriKey.int_decode(0x02))

# Transfer all lamports from bob to ada.
bob.sol_transfer(ada.pubkey, bob.sol_balance() - 5000)

ret = pxsol.rpc.get_account_info(bob.pubkey.base58(), {})
print(ret)
# None
```

When Bob constructs the transaction, he must reserve 5,000 lamports for the transaction fee and transfer the remaining SOL to Ada. Once the transfer is complete, Bob's account will be immediately destroyed, and you will no longer be able to find Bob's account information on the Solana network.

You can easily create and destroy regular wallet accounts on Solana. Generally, creating wallet accounts is a common operation, while destroying accounts usually happens when you no longer use a particular account or want to clean up your list of accounts.
