# Solana/Private Key, Public Key and Address/Public Key

In the Solana blockchain, a public key is a unique identifier generated by cryptographic algorithms, usually paired with a private key. **The private key is held by the user and must be kept strictly confidential**, while the public key can be publicly shared to represent the user's identity or account address. Unlike Bitcoin or Ethereum addresses, Solana public keys are themselves account identifiers, not hashed into derived addresses.

## Public Key

Solana uses the ed25519 elliptic curve encryption algorithm, which is a high-security and high-performance signing algorithm. Each public key is a 32-byte string, typically displayed in base58 encoding format for users. For example, a typical Solana public key looks like this:

```text
6ASf5EcmmEHTgDJ4X4ZT5vT6iHVJBXPg5AN5YoTCpGWt
```

Q: Find the public key corresponding to the Solana address `6ASf5EcmmEHTgDJ4X4ZT5vT6iHVJBXPg5AN5YoTCpGWt`. The public key is displayed in hexadecimal format.

A: Solana's address is its own public key, so you can decode it by converting the address string to base58. To solve this problem, we first install the `pxsol` library using the command `pip install pxsol`. This is a human-friendly Solana Python library that provides base58 functionality. You can use the library's base58 features to decode the base58 string.

```py
import pxsol

pubkey = pxsol.base58.decode('6ASf5EcmmEHTgDJ4X4ZT5vT6iHVJBXPg5AN5YoTCpGWt')
print(pubkey.hex())  # 4cb5abf6ad79fbf5abbccafcc269d85cd2651ed4b885b5869f241aedf0a5ba29
```

## Public Key Generation

Solana's public key and private key pair are generated using the ed25519 algorithm. The generation process is roughly as follows:

* Private key generation: Generate a random 32-byte array.
* Derived private key: Use the private key and ed25519-related algorithms to generate a 64-byte derived private key.
* Public key derivation: Derive the corresponding 32-byte public key from the derived private key through mathematical operations.

If you have already read previous sections, you should be familiar with each step. If not, don't worry, because `pxsol` has already implemented everything for you!

Q: Given a base58-encoded private key `11111111111111111111111111111112`, find its corresponding public key.

A:

```py
import pxsol

prikey = pxsol.core.PriKey.base58_decode('11111111111111111111111111111112')
pubkey = prikey.pubkey()
print(pubkey)  # 6ASf5EcmmEHTgDJ4X4ZT5vT6iHVJBXPg5AN5YoTCpGWt
```

## Public Key Purpose

Solana's public key has multiple uses in the network, including:

**Account identification**

In Solana, each account is uniquely identified by a public key. Accounts can store sol (Solana's native cryptocurrency), tokens, and program data. Whether it's an ordinary user account or a smart contract (program account), its address is represented by its public key. As such, users can share their public keys with others to receive funds.

> Note: Be careful when inputting public keys to avoid losing funds due to errors.

**Transaction verification**

When users initiate transactions (such as transfers or calling programs), they need to use the corresponding private key to sign the transaction. The network verifies the signature using the public key, ensuring that the transaction is issued by a legitimate account holder.

**Permission management**

Solana accounts support permission separation. For example, one public key can be designated as "read-only", while another has "signing" permissions. This design is particularly useful in decentralized applications and team account management.

**Program interaction**

In Solana, smart contracts (programs) also have their own public keys. Users interact with programs by specifying the program's public key and calling its functions. The public key serves as a location and validation factor.
