# Uncountable Python SDK

![PyPI - Python Version](https://img.shields.io/pypi/pyversions/UncountablePythonSDK)


The Uncountable Python SDK is a python package that provides a wrapper around the Uncountable REST API.

Using this SDK provides the following advantages:

- In-editor parameter/type safety
- Automatic parsing of response data.
- Reduced code boilerplate
- Helper methods

## Getting Started
The first step in any integration is to create a [Client](uncountable.core.client.Client) object. The client provides access to all available SDK methods, and includes built-in request authentication & error propagation.

### Creating a Client
Create a client using one of the supported authentication mechanisms. API credentials can be generated by a member of Uncountable staff.

::::{tab-set}
:::{tab-item} Basic Auth
```{literalinclude} ../examples/basic_auth.py
```
:::

:::{tab-item} OAuth
```{literalinclude} ../examples/oauth.py
```
:::
::::

The provided code examples assume that a Client has been created and stored in the `client` variable


### Basic Usage

:::{dropdown} List Ingredient Names & IDs
```{code-block} python
from uncountable.types import entity_t, id_source_t

client.list_id_source(
    spec=id_source_t.IdSourceSpecEntity(entity_type=entity_t.EntityType.INGREDIENT),
    search_label="",
)
```
Example Response:
```code
Data(
    results=[
        IdName(id=1, name='Filler'), 
        IdName(id=2, name='Calcium Oxide 2'), 
        IdName(id=3, name='Carbon Black'),
    ]
)
```
:::

:::{dropdown} Create an Experiment
```{code-block} python
client.create_recipe(material_family_id=1, workflow_id=1, name="Example Recipe")
```
Example Response:
```code
Data(result_id=52271)
```
:::

:::{dropdown} Upload a file
```{code-block} python
from uncountable.core.file_upload import MediaFileUpload

client.upload_files(file_uploads=[MediaFileUpload(path="/path/to/local/example_file.pdf")])
```
Example Response:
```code
[
    UploadedFile(name='example_file.pdf', file_id=718)
]
```
:::

[More examples](integration_examples/index)

## Errors
Client methods will raise Exceptions when the API returns codes in the `3xx`, `4xx` or `5xx` ranges. Ensure all method calls are wrapped in Exception handling logic.

## Pagination
Many of the Uncountable APIs require pagination to fetch more than 100 results at once. The following code snippet implements pagination to fetch the Names & IDs of all Projects:
:::{dropdown} Pagination Example
```{code-block} python
from uncountable.types import entity_t, id_source_t
from uncountable.types.api.id_source.list_id_source import IdName

def fetch_all_projects(client: Client) -> list[IdName]:
    projects: list[IdName] = []
    while True:
        response = client.list_id_source(
            spec=IdSourceSpecEntity(entity_type=entity_t.EntityType.PROJECT),
            search_label="",
            offset=len(projects),
        )
        projects.extend(response.results)
        if len(response.results) < 100:
            return projects
```
:::


```{toctree}
:hidden:
Overview <self>
Available SDK Methods <api/uncountable/core/client/Client>
integration_examples/index
SDK Reference <api/uncountable/index>
```
