"""Handles interactions with AWS APIs.

Not to be handled directly.
"""

from __future__ import annotations

import logging
import os
from datetime import datetime, timezone
from functools import cached_property
from typing import TYPE_CHECKING, ClassVar

import boto3
import botocore
from botocore.exceptions import ClientError

from ssmbak.typing import Preview, Version

if TYPE_CHECKING:
    from mypy_boto3_s3 import S3Client
    from mypy_boto3_s3.service_resource import S3ServiceResource
    from mypy_boto3_s3.type_defs import GetObjectOutputTypeDef
    from mypy_boto3_ssm import SSMClient
    from mypy_boto3_ssm.type_defs import PutParameterRequestTypeDef


logger = logging.getLogger(__name__)


class Resource:
    """Parent to actions.Path.

    Interface between what's in SSM now and corresponding s3 backups
    from the Lambda function. The region for SSM params and bucket
    access need to be the same.

    Attributes:
      region: The AWS region for params and bucket access.
      bucketname: The same bucket that the lambda writes to.
      _CALLS: class attribute strictly for testing efficiency of AWS calls
    """

    _CALLS: ClassVar[dict[str, int]] = {"tags": 0, "versions": 0, "version_objects": 0}

    def __init__(self, region, bucketname):
        self.region = region
        self.bucketname = bucketname

    @classmethod
    def clear_call_cache(cls) -> None:
        """Reset call counts between tests."""
        cls._CALLS = {"tags": 0, "versions": 0, "version_objects": 0}

    @classmethod
    def get_calls(cls) -> dict[str, int]:
        """Access call counts from tests."""
        return cls._CALLS

    @cached_property
    def s3(self) -> S3Client:
        """boto3 s3 client. There should only be one."""
        return boto3.client(
            "s3", endpoint_url=os.getenv("AWS_ENDPOINT"), region_name=self.region
        )

    @cached_property
    def s3res(self) -> S3ServiceResource:
        """boto3 s3 resource for backup contents. There should only be one."""
        return boto3.resource(
            "s3", endpoint_url=os.getenv("AWS_ENDPOINT"), region_name=self.region
        )

    @cached_property
    def ssm(self) -> SSMClient:
        """boto3 ssm client. There should only be one."""
        return boto3.client(
            "ssm", endpoint_url=os.getenv("AWS_ENDPOINT"), region_name=self.region
        )

    def _tagtime(self, version: Version) -> datetime:
        """Extracts datetime from the backup version.

        Corresponds to the time of the event and not necessary when it
        was backed-up, e.g. if there was some failures that held event
        processing up in a queue.

        In the case of deleted versions, no tags can be written so we
        just use the time of backup (LastModified).

        Arguments:
          version: dict of s3 version with processed tagset.
        """
        try:
            ssmbak_time = datetime.fromtimestamp(int(version["tagset"]["ssmbakTime"]))
            tagtime = ssmbak_time.astimezone(timezone.utc)
        except KeyError:
            tagtime = version["LastModified"]
        return tagtime

    def _get_tagset(self, name: str, versionid: str) -> dict[str, str]:
        """Get the tagset from S3 for the object version, using time
        of original event not backup.

        Arguments:
          name: name of the s3 object/ssm param in question
          versionid: s3 object versionid

        Returns:
        {
            "ssmbakTime": "1659560971",
            "ssmbakType": "SecureString",
            "ssmbakDescription": "fancy description", --OPTIONAL
        }
        """
        try:
            logger.debug("actually getting tagset for %s %s", name, versionid)
            Resource._CALLS["tags"] += 1
            tagset = self.s3.get_object_tagging(
                Bucket=self.bucketname, Key=name, VersionId=versionid
            )["TagSet"]
            nice_tagset = {x["Key"]: x["Value"] for x in tagset}
        except ClientError as e:
            if e.response["Error"]["Code"] in ["MethodNotAllowed"]:
                nice_tagset = {}
            else:
                raise e
        return nice_tagset

    def _make_ssm_kwargs(self, param: Preview) -> PutParameterRequestTypeDef:
        """Preps the kwargs for boto3 client ssm.put_parameter().

        Arguments:
          param: dict as generated by preview

        Returns:
        {
            "Name": "/testyssmbak/AP66LQ",
            "Value": "RUL38Y",
            "Type": "SecureString",
            "Overwrite": True,
            "Description": "fancy description",
        }
        """
        kwargs: PutParameterRequestTypeDef = {
            "Name": param["Name"],
            "Value": param["Value"],
            "Type": param["Type"],
            "Overwrite": True,
        }
        if "Description" in param:
            kwargs["Description"] = param["Description"]
        return kwargs

    def _restore_preview(self, param: Preview) -> None:
        """Sets the ssm param to the desired state.

        Arguments:
          param: dict as generated by preview
        """
        if "Deleted" in param and param["Deleted"]:
            self.ssm.delete_parameter(Name=param["Name"])
        else:
            ssm_kwargs = self._make_ssm_kwargs(param)
            self.ssm.put_parameter(**ssm_kwargs)

    def _ssm_del_multi(self, names: list) -> None:
        """Delete SSM Params efficiently"""
        batch_size = 10
        chunks = [names[x : x + batch_size] for x in range(0, len(names), batch_size)]
        for chunk in chunks:
            logger.debug("deleting %s", chunk)
            self.ssm.delete_parameters(Names=chunk)

    def _ssmgetpath(self, path: str, recurse=False) -> dict[str, Version]:
        """Gets params currently in place.

        Needed to determine what to delete when _getting_versions.

        Arguments:
          path: can correspond to just one key
          recurse: A boolean to operate on all paths/keys under path/

        Returns:
          The same format used everywhere, keyed by param name.

          {
              "/testyssmbak/82P11M": {
                  "Name": "/testyssmbak/82P11M",
                  "Type": "String",
                  "Value": "ES9IT7",
                  "Version": 2,
                  "LastModifiedDate": datetime.datetime(
                      2024, 6, 9, 9, 37, 34, 202000, tzinfo=tzlocal()
                  ),
                  "ARN": "arn:aws:ssm:us-west-2:000000000000:parameter/testyssmbak/82P11M",
                  "DataType": "text",
              },
          }
        """
        paginator = self.ssm.get_paginator("get_parameters_by_path")
        paginated = paginator.paginate(
            Path=path, Recursive=recurse, WithDecryption=True
        )
        result = paginated.build_full_result()
        keyed_params = {}
        if result["Parameters"]:
            params = result["Parameters"]
        elif not path.endswith("/"):
            try:
                param = self.ssm.get_parameter(Name=path, WithDecryption=True)[
                    "Parameter"
                ]
                params = [param]
            except KeyError:  # nothing found
                params = []
        else:
            params = []
        for name in {x["Name"] for x in params}:
            keyed_params[name] = [x for x in params if x["Name"] == name][0]
        return keyed_params

    def _get_object_versions(self, key: str) -> botocore.paginate.PageIterator:
        """Get a versions iterator from AWS

        Arguments:
          key: a single s3 key

        They come a thousand at a time.
        """
        logger.debug("actually getting versions for %s", key)
        paginator = self.s3.get_paginator("list_object_versions")
        # they come back most recent (LastModified) first
        Resource._CALLS["versions"] += 1
        return paginator.paginate(Bucket=self.bucketname, Prefix=key)

    def _collect_all_candidate_versions(
        self,
        key: str,
        recurse: bool,
        paginated: botocore.paginate.PageIterator,
    ) -> list[Version]:
        """Collects all candidate versions from paginated S3 response.

        Extracts both DeleteMarkers and regular Versions, applies key filtering
        based on recurse flag, but does NOT deduplicate or filter by time.
        This allows subsequent per-key timeline analysis.

        Arguments:
          key: a single s3 key or path
          recurse: operate on all paths/keys under key/
          paginated: paginated response from _get_object_versions()

        Returns:
          Flat list of all candidate versions (with "Deleted" flag added to DeleteMarkers)
        """
        all_versions: list[Version] = []

        for param_page in paginated:
            page_versions = []

            # Extract both DeleteMarkers and Versions, then merge by LastModified
            delete_markers = []
            if "DeleteMarkers" in param_page:
                for deleted_version in param_page["DeleteMarkers"]:
                    # Normalize DeleteMarkers to match Version structure
                    deleted_version["Deleted"] = True
                    deleted_version["Size"] = 0
                    deleted_version["ETag"] = ""
                    deleted_version["StorageClass"] = "STANDARD"
                    delete_markers.append(deleted_version)
            else:
                logger.debug("no delete markers")

            versions = []
            if "Versions" in param_page:
                versions = param_page["Versions"]
            else:
                logger.debug("no versions")

            # Merge DeleteMarkers and Versions, preserving S3's LastModified ordering
            # Both arrays are already sorted by LastModified (most recent first)
            # Merge them while maintaining that order
            i, j = 0, 0
            while i < len(delete_markers) and j < len(versions):
                if delete_markers[i]["LastModified"] >= versions[j]["LastModified"]:
                    page_versions.append(delete_markers[i])
                    i += 1
                else:
                    page_versions.append(versions[j])
                    j += 1
            # Append remaining
            page_versions.extend(delete_markers[i:])
            page_versions.extend(versions[j:])

            # Apply key filtering logic for non-recursive or specific key queries
            if not recurse or not key.endswith("/"):
                # Check if exact key exists in results so far
                all_keys = [x["Key"] for x in page_versions + all_versions]
                if key in all_keys:
                    # Filter to only the exact key
                    page_versions = [x for x in page_versions if x["Key"] == key]
                else:
                    # Filter to keys at same depth (same number of slashes)
                    n = key.count("/")
                    page_versions = [
                        x for x in page_versions if x["Key"].count("/") == n
                    ]

            all_versions.extend(page_versions)

        return all_versions

    def _get_versions(
        self, key: str, checktime: datetime, recurse: bool = False
    ) -> dict[str, Version]:
        """Efficiently looks for the version most recently backed-up before checktime.

        The objects come from AWS a thousand at a time, but only with
        modified times corresponding to when they were backed-up (LastModified) and
        not when the original event was reported. Typically it's less
        than a minute, but in the case of an outage it might be
        longer. To be safe, we check the event times, encoded in s3
        tags by the Lambda (ssmbakTime).

        Arguments:
          key: a single s3 key or path
          checktime: the point in time for which to retrieve relative latest version
          recurse: operate on all paths/keys under key/

        Returns:
          The same keyed versions as everywhere.

          {
              "/testyssmbak/88JCRX": {
                  "ETag": '"9d2f3ea8da7b4feba87aeb4da1fcb5e0"',
                  "Size": 6,
                  "StorageClass": "STANDARD",
                  "Key": "/testyssmbak/88JCRX",
                  "VersionId": "vuyAs6cfwwSbMUi4o8O1qA",
                  "IsLatest": True,
                  "LastModified": datetime.datetime(
                      2024, 6, 9, 16, 45, 4, tzinfo=tzutc()
                  ),
                  "Owner": {
                      "DisplayName": "webfile",
                      "ID": "75aaa08ebf849d0f8e7faeebf76c078efc7c6caea54ba06a",
                  },
                  "tagset": {"ssmbakTime": "1717951504", "ssmbakType": "SecureString"},
              },
          }
        """
        # Step 1: Get paginated versions from S3
        paginated = self._get_object_versions(key)

        # Step 2: Collect all candidate versions preserving order
        # (DeleteMarkers before Versions, as in original code)
        all_versions = self._collect_all_candidate_versions(key, recurse, paginated)

        # Step 3: Select version with latest event time for each key
        # Cannot rely on LastModified order - must check event times (ssmbakTime tags)
        # to find the version with the latest event time before checktime
        result = {}
        candidates: dict[str, list] = {}

        for version in all_versions:
            param_key = version["Key"]

            # Fetch tags and check time
            version["tagset"] = self._get_tagset(param_key, version["VersionId"])
            tagtime = self._tagtime(version)
            # ssmbakTime is truncated to seconds (losing microseconds) when stored
            # So compare at second-level precision to be fair
            # Use < to mean "event happened in an earlier second"
            if tagtime.replace(microsecond=0) < checktime.replace(microsecond=0):
                # Collect all versions that pass the time filter
                if param_key not in candidates:
                    candidates[param_key] = []
                candidates[param_key].append((tagtime, version))

        # Select version with latest event time for each key
        for param_key, versions in candidates.items():
            # Sort by tagtime descending to get latest event time first
            versions.sort(key=lambda x: x[0], reverse=True)
            result[param_key] = versions[0][1]

        return result

    def _get_version_body(self, name: str, versionid: str) -> str:
        """Uses s3 object resource to get the contents of the version.

        Should only be run after all last versions are got.

        Arguments:
          name: single s3 key
          versionid: s3 object versionid

        Returns:
          String of the backed-up ssm paramter's value.
        """
        Resource._CALLS["version_objects"] += 1
        logger.debug("actually getting contents for %s", name)
        version = self.s3res.ObjectVersion(self.bucketname, name, versionid)
        try:
            res = version.get()  # boto3 issue #832
            body = self._get_contents(res)
        except ClientError as e:
            # NoSuchVersion to accommodate localstack
            if e.response["Error"]["Code"] in ["MethodNotAllowed", "NoSuchVersion"]:
                body = ""
            else:
                raise e
        return body

    def _get_contents(self, version: GetObjectOutputTypeDef) -> str:
        """Reads and decodes the s3 object's body."""
        stuff = ""
        try:
            stuff = version["Body"].read().decode("utf-8").strip()
        except ClientError as e:
            if e.response["Error"]["Code"] == "MethodNotAllowed":
                # it was deleted
                pass
        return stuff
