import * as codebuild from '@aws-cdk/aws-codebuild';
import * as iam from '@aws-cdk/aws-iam';
import { IResource, Resource, SecretValue } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { BasicAuth } from './basic-auth';
import { Branch, BranchOptions } from './branch';
import { Domain, DomainOptions } from './domain';
/**
 * An Amplify Console application
 */
export interface IApp extends IResource {
    /**
     * The application id
     *
     * @attribute
     */
    readonly appId: string;
}
/**
 * Configuration for the source code provider
 */
export interface SourceCodeProviderConfig {
    /**
     * The repository for the application. Must use the `HTTPS` protocol.
     *
     * For example, `https://github.com/aws/aws-cdk`.
     */
    readonly repository: string;
    /**
     * OAuth token for 3rd party source control system for an Amplify App, used
     * to create webhook and read-only deploy key. OAuth token is not stored.
     *
     * Either `accessToken` or `oauthToken` must be specified if `repository`
     * is specified.
     *
     * @default - do not use a token
     */
    readonly oauthToken?: SecretValue;
    /**
     * Personal Access token for 3rd party source control system for an Amplify
     * App, used to create webhook and read-only deploy key. Token is not stored.
     *
     * Either `accessToken` or `oauthToken` must be specified if `repository`
     * is sepcified.
     *
     * @default - do not use a token
     */
    readonly accessToken?: SecretValue;
}
/**
 * A source code provider
 */
export interface ISourceCodeProvider {
    /**
     * Binds the source code provider to an app
     *
     * @param app The app [disable-awslint:ref-via-interface]
     */
    bind(app: App): SourceCodeProviderConfig;
}
/**
 * Properties for an App
 */
export interface AppProps {
    /**
     * The name for the application
     *
     * @default - a CDK generated name
     */
    readonly appName?: string;
    /**
     * The source code provider for this application
     *
     * @default - not connected to a source code provider
     */
    readonly sourceCodeProvider?: ISourceCodeProvider;
    /**
     * The auto branch creation configuration. Use this to automatically create
     * branches that match a certain pattern.
     *
     * @default - no auto branch creation
     */
    readonly autoBranchCreation?: AutoBranchCreation;
    /**
     * Automatically disconnect a branch in the Amplify Console when you delete a
     * branch from your Git repository.
     *
     * @default false
     */
    readonly autoBranchDeletion?: boolean;
    /**
     * The Basic Auth configuration. Use this to set password protection at an
     * app level to all your branches.
     *
     * @default - no password protection
     */
    readonly basicAuth?: BasicAuth;
    /**
     * BuildSpec for the application. Alternatively, add a `amplify.yml`
     * file to the repository.
     *
     * @see https://docs.aws.amazon.com/amplify/latest/userguide/build-settings.html
     *
     * @default - no build spec
     */
    readonly buildSpec?: codebuild.BuildSpec;
    /**
     * The custom HTTP response headers for an Amplify app.
     *
     * @see https://docs.aws.amazon.com/amplify/latest/userguide/custom-headers.html
     *
     * @default - no custom response headers
     */
    readonly customResponseHeaders?: CustomResponseHeader[];
    /**
     * Custom rewrite/redirect rules for the application
     *
     * @default - no custom rewrite/redirect rules
     */
    readonly customRules?: CustomRule[];
    /**
     * A description for the application
     *
     * @default - no description
     */
    readonly description?: string;
    /**
     * Environment variables for the application.
     *
     * All environment variables that you add are encrypted to prevent rogue
     * access so you can use them to store secret information.
     *
     * @default - no environment variables
     */
    readonly environmentVariables?: {
        [name: string]: string;
    };
    /**
     * The IAM service role to associate with the application. The App
     * implements IGrantable.
     *
     * @default - a new role is created
     */
    readonly role?: iam.IRole;
}
/**
 * An Amplify Console application
 */
export declare class App extends Resource implements IApp, iam.IGrantable {
    /**
     * Import an existing application
     */
    static fromAppId(scope: Construct, id: string, appId: string): IApp;
    readonly appId: string;
    /**
     * The name of the application
     *
     * @attribute
     */
    readonly appName: string;
    /**
     * The ARN of the application
     *
     * @attribute
     */
    readonly arn: string;
    /**
     * The default domain of the application
     *
     * @attribute
     */
    readonly defaultDomain: string;
    /**
     * The principal to grant permissions to
     */
    readonly grantPrincipal: iam.IPrincipal;
    private readonly customRules;
    private readonly environmentVariables;
    private readonly autoBranchEnvironmentVariables;
    constructor(scope: Construct, id: string, props: AppProps);
    /**
     * Adds a custom rewrite/redirect rule to this application
     */
    addCustomRule(rule: CustomRule): this;
    /**
     * Adds an environment variable to this application.
     *
     * All environment variables that you add are encrypted to prevent rogue
     * access so you can use them to store secret information.
     */
    addEnvironment(name: string, value: string): this;
    /**
     * Adds an environment variable to the auto created branch.
     *
     * All environment variables that you add are encrypted to prevent rogue
     * access so you can use them to store secret information.
     */
    addAutoBranchEnvironment(name: string, value: string): this;
    /**
     * Adds a branch to this application
     */
    addBranch(id: string, options?: BranchOptions): Branch;
    /**
     * Adds a domain to this application
     */
    addDomain(id: string, options?: DomainOptions): Domain;
}
/**
 * Auto branch creation configuration
 */
export interface AutoBranchCreation {
    /**
     * Automated branch creation glob patterns
     *
     * @default - all repository branches
     */
    readonly patterns?: string[];
    /**
     * The Basic Auth configuration. Use this to set password protection for
     * the auto created branch.
     *
     * @default - no password protection
     */
    readonly basicAuth?: BasicAuth;
    /**
     * Build spec for the auto created branch.
     *
     * @default - application build spec
     */
    readonly buildSpec?: codebuild.BuildSpec;
    /**
     * Whether to enable auto building for the auto created branch
     *
     * @default true
     */
    readonly autoBuild?: boolean;
    /**
     * Whether to enable pull request preview for the auto created branch.
     *
     * @default true
     */
    readonly pullRequestPreview?: boolean;
    /**
     * Environment variables for the auto created branch.
     *
     * All environment variables that you add are encrypted to prevent rogue
     * access so you can use them to store secret information.
     *
     * @default - application environment variables
     */
    readonly environmentVariables?: {
        [name: string]: string;
    };
    /**
     * The dedicated backend environment for the pull request previews of
     * the auto created branch.
     *
     * @default - automatically provision a temporary backend
     */
    readonly pullRequestEnvironmentName?: string;
    /**
     * Stage for the auto created branch
     *
     * @default - no stage
     */
    readonly stage?: string;
}
/**
 * The status code for a URL rewrite or redirect rule.
 */
export declare enum RedirectStatus {
    /**
     * Rewrite (200)
     */
    REWRITE = "200",
    /**
     * Permanent redirect (301)
     */
    PERMANENT_REDIRECT = "301",
    /**
     * Temporary redirect (302)
     */
    TEMPORARY_REDIRECT = "302",
    /**
     * Not found (404)
     */
    NOT_FOUND = "404",
    /**
     * Not found rewrite (404)
     */
    NOT_FOUND_REWRITE = "404-200"
}
/**
 * Options for a custom rewrite/redirect rule for an Amplify App.
 */
export interface CustomRuleOptions {
    /**
     * The source pattern for a URL rewrite or redirect rule.
     *
     * @see https://docs.aws.amazon.com/amplify/latest/userguide/redirects.html
     */
    readonly source: string;
    /**
     * The target pattern for a URL rewrite or redirect rule.
     *
     * @see https://docs.aws.amazon.com/amplify/latest/userguide/redirects.html
     */
    readonly target: string;
    /**
     * The status code for a URL rewrite or redirect rule.
     *
     * @see https://docs.aws.amazon.com/amplify/latest/userguide/redirects.html
     *
     * @default PERMANENT_REDIRECT
     */
    readonly status?: RedirectStatus;
    /**
     * The condition for a URL rewrite or redirect rule, e.g. country code.
     *
     * @see https://docs.aws.amazon.com/amplify/latest/userguide/redirects.html
     *
     * @default - no condition
     */
    readonly condition?: string;
}
/**
 * Custom rewrite/redirect rule for an Amplify App.
 *
 * @see https://docs.aws.amazon.com/amplify/latest/userguide/redirects.html
 */
export declare class CustomRule {
    /**
     * Sets up a 200 rewrite for all paths to `index.html` except for path
     * containing a file extension.
     */
    static readonly SINGLE_PAGE_APPLICATION_REDIRECT: CustomRule;
    /**
     * The source pattern for a URL rewrite or redirect rule.
     *
     * @see https://docs.aws.amazon.com/amplify/latest/userguide/redirects.html
     */
    readonly source: string;
    /**
     * The target pattern for a URL rewrite or redirect rule.
     *
     * @see https://docs.aws.amazon.com/amplify/latest/userguide/redirects.html
     */
    readonly target: string;
    /**
     * The status code for a URL rewrite or redirect rule.
     *
     * @see https://docs.aws.amazon.com/amplify/latest/userguide/redirects.html
     *
     * @default PERMANENT_REDIRECT
     */
    readonly status?: RedirectStatus;
    /**
     * The condition for a URL rewrite or redirect rule, e.g. country code.
     *
     * @see https://docs.aws.amazon.com/amplify/latest/userguide/redirects.html
     *
     * @default - no condition
     */
    readonly condition?: string;
    constructor(options: CustomRuleOptions);
}
/**
 * Custom response header of an Amplify App.
 */
export interface CustomResponseHeader {
    /**
     * These custom headers will be applied to all URL file paths that match this pattern.
     */
    readonly pattern: string;
    /**
     * The map of custom headers to be applied.
     */
    readonly headers: {
        [key: string]: string;
    };
}
