#!/usr/bin/env python3

import yaml
import toml
import subprocess
import json
import os

base_dir = os.path.join(os.path.dirname(__file__), "..")

# Load explicitly listed dependencies from pixi.toml
with open(os.path.join(base_dir, "pixi.lock"), "rb") as f:
    lock_data = yaml.safe_load(f)

# Load explicitly listed dependencies from pixi.toml
with open(os.path.join(base_dir, "pixi.toml"), "r") as f:
    pixi_data = toml.load(f)

# Gather conda packages and versions from "pixi list --json"
result = subprocess.run(
    ["pixi", "list", "--json"],
    capture_output=True, text=True, check=True
)
# Parse the JSON output
pixi_list = json.loads(result.stdout)
# convert to name -> version
# exclude editable packages because that is this project
pixi_list = {pkg["name"].lower(): pkg["version"] for pkg in pixi_list if 'is_editable' not in pkg or pkg['is_editable'] is not True}

# Write requirements.txt via 'pip list --format json'
result = subprocess.run(
    ["pixi", "run", "pip", "list", "--format", "json"],
    capture_output=True, text=True, check=True
)
# Parse the JSON output
pip_list = json.loads(result.stdout)
# sort
pip_list = sorted(pip_list, key=lambda x: x["name"].lower())

# Explicitly declared packages in pixi.toml
declared_conda = set()
for section in ["dependencies", "host-dependencies"]:
    for pkg, val in pixi_data.get(section, {}).items():
        pkg_lower = pkg.lower()
        declared_conda.add(pkg_lower)

# Prepare environment.yml data
conda_deps = [
    f"{pkg}={pixi_list[pkg]}"
    for pkg in declared_conda if pkg in pixi_list
]
# sort conda_deps
conda_deps.sort()
env_data = {
    "channels": ["conda-forge", "bioconda"],
    "dependencies": conda_deps,
}

# Write environment.yml
with open(os.path.join(base_dir, 'admin', "environment.yml"), "w") as f:
    f.write("# This file was autogenerated by build_dep_defs_from_pixi.py - do not change manually.\n")
    yaml.dump(env_data, f, default_flow_style=False)

print("environment.yml written, with ", len(conda_deps), "dependencies.")

# Find the name of the current project to exclude from requirements.txt
project_name = pixi_data.get("workspace", {}).get("name", "").lower()
if not project_name:
    raise ValueError("Project name not found in pixi.toml.")

print("Excluding project name from requirements.txt:", project_name)

# Write requirements.txt
pip_count = 0
with open(os.path.join(base_dir, 'admin', "requirements.txt"), "w") as f:
    f.write("# This file was autogenerated by build_dep_defs_from_pixi.py - do not change manually.\n")
    for package in pip_list:
        # only include if the package is specified in pixi.toml
        if package["name"].lower() in declared_conda:
            # Skip the project itself
            if package["name"].lower() == project_name:
                continue
            pip_count += 1
            # Sometimes the version is set to 0.0.0 - unsure why - for these ones just set '*'
            if package["version"] == "0.0.0":
                line = f"{package['name']}\n"
            else:
                # If the version is not 0.0.0, write it as well
                line = f"{package['name']}=={package['version']}\n"
            f.write(line)
print("requirements.txt written, with ", pip_count, "dependencies.")
