'''
Generate a graph for testing and documentation purposes.
'''
import logging
from typing import Tuple
import string

import networkx as nx
import numpy as np
import utm

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


def mock_graph(wgs84_coords: bool = False) -> nx.Graph:
    '''
    Prepares a Tutte graph per https://en.wikipedia.org/wiki/Tutte_graph
    :return: NetworkX graph
    '''

    G = nx.Graph()

    nodes = [
        (0, {'x': 700700, 'y': 5719700}),
        (1, {'x': 700610, 'y': 5719780}),
        (2, {'x': 700460, 'y': 5719700}),
        (3, {'x': 700520, 'y': 5719820}),
        (4, {'x': 700620, 'y': 5719905}),
        (5, {'x': 700260, 'y': 5719700}),
        (6, {'x': 700320, 'y': 5719850}),
        (7, {'x': 700420, 'y': 5719880}),
        (8, {'x': 700460, 'y': 5719980}),
        (9, {'x': 700580, 'y': 5720030}),
        (10, {'x': 700100, 'y': 5719810}),
        (11, {'x': 700280, 'y': 5719980}),
        (12, {'x': 700400, 'y': 5720030}),
        (13, {'x': 700460, 'y': 5720130}),
        (14, {'x': 700190, 'y': 5720050}),
        (15, {'x': 700350, 'y': 5720200}),
        (16, {'x': 700800, 'y': 5719750}),
        (17, {'x': 700800, 'y': 5719920}),
        (18, {'x': 700900, 'y': 5719820}),
        (19, {'x': 700910, 'y': 5719690}),
        (20, {'x': 700905, 'y': 5720080}),
        (21, {'x': 701000, 'y': 5719870}),
        (22, {'x': 701040, 'y': 5719660}),
        (23, {'x': 701050, 'y': 5719760}),
        (24, {'x': 701000, 'y': 5719980}),
        (25, {'x': 701130, 'y': 5719950}),
        (26, {'x': 701130, 'y': 5719805}),
        (27, {'x': 701170, 'y': 5719700}),
        (28, {'x': 701100, 'y': 5720200}),
        (29, {'x': 701240, 'y': 5719990}),
        (30, {'x': 701300, 'y': 5719760}),
        (31, {'x': 700690, 'y': 5719590}),
        (32, {'x': 700570, 'y': 5719530}),
        (33, {'x': 700820, 'y': 5719500}),
        (34, {'x': 700700, 'y': 5719480}),
        (35, {'x': 700490, 'y': 5719440}),
        (36, {'x': 700580, 'y': 5719360}),
        (37, {'x': 700690, 'y': 5719370}),
        (38, {'x': 700920, 'y': 5719330}),
        (39, {'x': 700780, 'y': 5719300}),
        (40, {'x': 700680, 'y': 5719200}),
        (41, {'x': 700560, 'y': 5719280}),
        (42, {'x': 700450, 'y': 5719300}),
        (43, {'x': 700440, 'y': 5719150}),
        (44, {'x': 700650, 'y': 5719080}),
        (45, {'x': 700930, 'y': 5719110}),
        # cul-de-sacs
        (46, {'x': 701015, 'y': 5719535}),
        (47, {'x': 701100, 'y': 5719480}),
        (48, {'x': 700917, 'y': 5719517}),
        # isolated node
        (49, {'x': 700400, 'y': 5719550}),
        # isolated edge
        (50, {'x': 700700, 'y': 5720100}),
        (51, {'x': 700700, 'y': 5719900}),
        # disconnected looping component
        (52, {'x': 700400, 'y': 5719650}),
        (53, {'x': 700500, 'y': 5719550}),
        (54, {'x': 700400, 'y': 5719450}),
        (55, {'x': 700300, 'y': 5719550})
    ]

    G.add_nodes_from(nodes)

    edges = [
        (0, 1),
        (0, 16),
        (0, 31),
        (1, 2),
        (1, 4),
        (2, 3),
        (2, 5),
        (3, 4),
        (3, 7),
        (4, 9),
        (5, 6),
        (5, 10),
        (6, 7),
        (6, 11),
        (7, 8),
        (8, 9),
        (8, 12),
        (9, 13),
        (10, 14),
        (10, 43),
        (11, 12),
        (11, 14),
        (12, 13),
        (13, 15),
        (14, 15),
        (15, 28),
        (16, 17),
        (16, 19),
        (17, 18),
        (17, 20),
        (18, 19),
        (18, 21),
        (19, 22),
        (20, 24),
        (20, 28),
        (21, 23),
        (21, 24),
        (22, 23),
        (22, 27),
        (23, 26),
        (24, 25),
        (25, 26),
        (25, 29),
        (26, 27),
        (27, 30),
        (28, 29),
        (29, 30),
        (30, 45),
        (31, 32),
        (31, 33),
        (32, 34),
        (32, 35),
        (33, 34),
        (33, 38),
        (34, 37),
        (35, 36),
        (35, 42),
        (36, 37),
        (36, 41),
        (37, 39),
        (38, 39),
        (38, 45),
        (39, 40),
        (40, 41),
        (40, 44),
        (41, 42),
        (42, 43),
        (43, 44),
        (44, 45),
        # cul-de-sacs
        (22, 46),
        (46, 47),
        (46, 48),
        # isolated edge
        (50, 51),
        # disconnected looping component
        (52, 53),
        (53, 54),
        (54, 55),
        (55, 52)
    ]

    G.add_edges_from(edges)

    if wgs84_coords:
        for n, d in G.nodes(data=True):
            easting = d['x']
            northing = d['y']
            # be cognisant of parameter and return order
            # returns in lat, lng order
            lat, lng = utm.to_latlon(easting, northing, 30, 'U')
            G.nodes[n]['x'] = lng
            G.nodes[n]['y'] = lat

    return G


def get_graph_extents(G: nx.Graph) -> Tuple[float, float, float, float]:

    # get min and maxes for x and y
    min_x = np.inf
    max_x = -np.inf
    min_y = np.inf
    max_y = -np.inf

    for n, d in G.nodes(data=True):
        if d['x'] < min_x:
            min_x = d['x']
        if d['x'] > max_x:
            max_x = d['x']
        if d['y'] < min_y:
            min_y = d['y']
        if d['y'] > max_y:
            max_y = d['y']

    return min_x, min_y, max_x, max_y


def mock_data_dict(G: nx.Graph, length: int = 50, random_seed: int = None) -> dict:
    if random_seed is not None:
        np.random.seed(seed=random_seed)

    min_x, min_y, max_x, max_y = get_graph_extents(G)

    data_dict = {}

    for i in range(length):
        data_dict[i] = {
            'x': np.random.uniform(min_x, max_x),
            'y': np.random.uniform(min_y, max_y),
            'live': bool(np.random.randint(0, 1))
        }

    return data_dict


def mock_categorical_data(length: int, num_classes: int = 10, random_seed: int = None) -> np.ndarray:
    if random_seed is not None:
        np.random.seed(seed=random_seed)

    random_class_str = string.ascii_lowercase
    if num_classes > len(random_class_str):
        raise ValueError(f'The requested {num_classes} classes exceeds the max available categorical classes: {len(random_class_str)}')
    random_class_str = random_class_str[:num_classes]

    d = []
    for i in range(length):
        d.append(random_class_str[np.random.randint(0, len(random_class_str))])

    return np.array(d)


def mock_numerical_data(length: int, min: int = 0, max: int = 100000, num_arrs: int = 1, random_seed: int = None) -> np.ndarray:
    if random_seed is not None:
        np.random.seed(seed=random_seed)

    num_data = []
    for i in range(num_arrs):
        num_data.append(np.random.randint(min, high=max, size=length))
    # return a 2d array
    return np.array(num_data, dtype=float)


def mock_species_data(random_seed: int = None) -> Tuple[np.ndarray, np.ndarray]:
    if random_seed is not None:
        np.random.seed(seed=random_seed)

    for n in range(1, 50, 5):
        data = np.random.randint(1, 10, n)
        unique = np.unique(data)
        counts = np.zeros_like(unique)
        for i, u in enumerate(unique):
            counts[i] = (data == u).sum()
        probs = counts / len(data)

        yield counts, probs


def mock_osm_data(alt=False) -> str:

    if not alt:
        return '{"version":0.6,"generator":"Overpass API 0.7.55.5 2ca3f387","osm3s":{"timestamp_osm_base":"2019-04-01T20:03:02Z","copyright":"The data included in this document is from www.openstreetmap.org. The data is made available under ODbL."},"elements":[{"type":"node","id":108079,"lat":51.5051603,"lon":-0.1380645},{"type":"node","id":108080,"lat":51.5059616,"lon":-0.1387498},{"type":"node","id":108081,"lat":51.5063249,"lon":-0.1390989},{"type":"node","id":108085,"lat":51.5078059,"lon":-0.1406472},{"type":"node","id":9789809,"lat":51.506271,"lon":-0.1378833},{"type":"node","id":9789815,"lat":51.5075804,"lon":-0.1391586},{"type":"node","id":9789816,"lat":51.5072822,"lon":-0.1400548},{"type":"node","id":9789819,"lat":51.5078781,"lon":-0.1382667},{"type":"node","id":25257402,"lat":51.5074375,"lon":-0.141277},{"type":"node","id":25607096,"lat":51.5042674,"lon":-0.1398999},{"type":"node","id":25607097,"lat":51.5043791,"lon":-0.1399722},{"type":"node","id":25607099,"lat":51.504709,"lon":-0.1391244},{"type":"node","id":25607100,"lat":51.5045162,"lon":-0.1392401},{"type":"node","id":25607102,"lat":51.5045613,"lon":-0.1389612},{"type":"node","id":25607125,"lat":51.5052412,"lon":-0.1395689},{"type":"node","id":25607129,"lat":51.5048031,"lon":-0.1383955},{"type":"node","id":26591528,"lat":51.5068821,"lon":-0.1384785},{"type":"node","id":26591529,"lat":51.5068226,"lon":-0.1384205},{"type":"node","id":26591530,"lat":51.5065021,"lon":-0.1392691},{"type":"node","id":26591531,"lat":51.5056551,"lon":-0.1384553},{"type":"node","id":26591532,"lat":51.5060903,"lon":-0.1388759},{"type":"node","id":26591533,"lat":51.5054636,"lon":-0.1403061},{"type":"node","id":26591534,"lat":51.5060639,"lon":-0.1407782},{"type":"node","id":26591535,"lat":51.5066299,"lon":-0.1393986},{"type":"node","id":26591536,"lat":51.5072084,"lon":-0.1399816},{"type":"node","id":26591537,"lat":51.5069197,"lon":-0.1407535},{"type":"node","id":311153342,"lat":51.5061966,"lon":-0.1380917},{"type":"node","id":506328569,"lat":51.5077013,"lon":-0.1387954},{"type":"node","id":677445938,"lat":51.5060908,"lon":-0.1396788},{"type":"node","id":677445939,"lat":51.5055666,"lon":-0.1383791},{"type":"node","id":686339678,"lat":51.5029976,"lon":-0.1377612},{"type":"node","id":812264806,"lat":51.505726,"lon":-0.1396623},{"type":"node","id":844780668,"lat":51.5073535,"lon":-0.1389376},{"type":"node","id":844780674,"lat":51.5074977,"lon":-0.1385238},{"type":"node","id":1352935864,"lat":51.5071042,"lon":-0.1398778},{"type":"node","id":1737601049,"lat":51.5042803,"lon":-0.1399684},{"type":"node","id":1737601058,"lat":51.5043392,"lon":-0.1400025},{"type":"node","id":1737601164,"lat":51.5052407,"lon":-0.1378887},{"type":"node","id":1930510928,"lat":51.5051211,"lon":-0.1378745},{"type":"node","id":1938580015,"lat":51.5052433,"lon":-0.1379634},{"type":"node","id":1938580016,"lat":51.5052782,"lon":-0.1380262},{"type":"node","id":1938580017,"lat":51.5052921,"lon":-0.1377288},{"type":"node","id":1938580021,"lat":51.5054227,"lon":-0.138164},{"type":"node","id":1938580022,"lat":51.5054609,"lon":-0.1383531},{"type":"node","id":1938580025,"lat":51.5055102,"lon":-0.1382436},{"type":"node","id":2038064029,"lat":51.5078458,"lon":-0.14031},{"type":"node","id":2214979325,"lat":51.5051352,"lon":-0.1380107},{"type":"node","id":2214979327,"lat":51.5049977,"lon":-0.1379617},{"type":"node","id":2214979331,"lat":51.5050667,"lon":-0.1378937},{"type":"node","id":2214979337,"lat":51.5050312,"lon":-0.1379143},{"type":"node","id":2214979339,"lat":51.5052557,"lon":-0.1378101},{"type":"node","id":2214979343,"lat":51.5051539,"lon":-0.1377252},{"type":"node","id":2214979344,"lat":51.5053322,"lon":-0.1376419},{"type":"node","id":2214979347,"lat":51.5060222,"lon":-0.1388122},{"type":"node","id":2214979348,"lat":51.5051236,"lon":-0.1379465},{"type":"node","id":2214979350,"lat":51.5051302,"lon":-0.137805},{"type":"node","id":2215472138,"lat":51.5079913,"lon":-0.1399338},{"type":"node","id":2215475643,"lat":51.5078888,"lon":-0.1404451},{"type":"node","id":2215475645,"lat":51.5080183,"lon":-0.1401234},{"type":"node","id":2215475647,"lat":51.5078199,"lon":-0.1403759},{"type":"node","id":2215490962,"lat":51.5075171,"lon":-0.1410896},{"type":"node","id":2215490963,"lat":51.5075381,"lon":-0.1413036},{"type":"node","id":2215490965,"lat":51.5074365,"lon":-0.1415221},{"type":"node","id":2215490966,"lat":51.5077762,"lon":-0.1404814},{"type":"node","id":2215490967,"lat":51.5071711,"lon":-0.1420659},{"type":"node","id":2215502650,"lat":51.5078072,"lon":-0.1405981},{"type":"node","id":2215502651,"lat":51.507702,"lon":-0.1404042},{"type":"node","id":2215502653,"lat":51.507793,"lon":-0.1405121},{"type":"node","id":2215502654,"lat":51.5077354,"lon":-0.1405781},{"type":"node","id":2215502656,"lat":51.5076633,"lon":-0.1405014},{"type":"node","id":2215502657,"lat":51.507802,"lon":-0.1405506},{"type":"node","id":2215503568,"lat":51.5077741,"lon":-0.1407254},{"type":"node","id":2215503569,"lat":51.5075873,"lon":-0.1404168},{"type":"node","id":2622783492,"lat":51.5073392,"lon":-0.1377558},{"type":"node","id":3561459278,"lat":51.5052618,"lon":-0.1395234},{"type":"node","id":3561459279,"lat":51.5044549,"lon":-0.13986},{"type":"node","id":4170421481,"lat":51.506922,"lon":-0.137664},{"type":"node","id":4171766882,"lat":51.5066663,"lon":-0.1388345},{"type":"node","id":4171766883,"lat":51.506722,"lon":-0.1394916},{"type":"node","id":4171766884,"lat":51.5067513,"lon":-0.1389124},{"type":"node","id":4171766885,"lat":51.5067671,"lon":-0.139359},{"type":"node","id":4171766886,"lat":51.5068164,"lon":-0.1392128},{"type":"node","id":4171768189,"lat":51.5070192,"lon":-0.138781},{"type":"node","id":4171768190,"lat":51.5070434,"lon":-0.138726},{"type":"node","id":4171768191,"lat":51.5070701,"lon":-0.1386616},{"type":"node","id":4209492990,"lat":51.5075013,"lon":-0.1379095},{"type":"node","id":4210456238,"lat":51.5073594,"lon":-0.1376923},{"type":"node","id":4535718259,"lat":51.5072342,"lon":-0.1382674},{"type":"node","id":4535718260,"lat":51.5074458,"lon":-0.1386758},{"type":"node","id":4535718261,"lat":51.5073752,"lon":-0.1388756},{"type":"node","id":4919996790,"lat":51.5058688,"lon":-0.1377959},{"type":"node","id":4919996791,"lat":51.5060209,"lon":-0.1379265},{"type":"node","id":107796,"lat":51.5071967,"lon":-0.1323441},{"type":"node","id":107816,"lat":51.507917,"lon":-0.1332211},{"type":"node","id":108078,"lat":51.5054412,"lon":-0.1373969},{"type":"node","id":108084,"lat":51.5066652,"lon":-0.1338229},{"type":"node","id":109625,"lat":51.5075834,"lon":-0.1312575},{"type":"node","id":9789806,"lat":51.5065751,"lon":-0.1370314},{"type":"node","id":9789817,"lat":51.5069432,"lon":-0.1360143},{"type":"node","id":9789818,"lat":51.507439,"lon":-0.1345337},{"type":"node","id":9789820,"lat":51.5076728,"lon":-0.1356737},{"type":"node","id":9789821,"lat":51.506147,"lon":-0.135299},{"type":"node","id":9789822,"lat":51.5074691,"lon":-0.1362638},{"type":"node","id":9789823,"lat":51.5064574,"lon":-0.1355731},{"type":"node","id":9789824,"lat":51.5078796,"lon":-0.1350809},{"type":"node","id":25607360,"lat":51.5063357,"lon":-0.1319233},{"type":"node","id":25607361,"lat":51.5058624,"lon":-0.1332708},{"type":"node","id":25620550,"lat":51.5071559,"lon":-0.1311472},{"type":"node","id":26591526,"lat":51.5079599,"lon":-0.1369995},{"type":"node","id":26591527,"lat":51.5071922,"lon":-0.1376164},{"type":"node","id":26630783,"lat":51.5057222,"lon":-0.1311548},{"type":"node","id":26630784,"lat":51.5057462,"lon":-0.1335854},{"type":"node","id":26630785,"lat":51.5055088,"lon":-0.1343102},{"type":"node","id":28782062,"lat":51.5056797,"lon":-0.1312599},{"type":"node","id":78598321,"lat":51.5057034,"lon":-0.1317359},{"type":"node","id":78598323,"lat":51.5058714,"lon":-0.131323},{"type":"node","id":78598326,"lat":51.5038502,"lon":-0.1361429},{"type":"node","id":271945198,"lat":51.5063684,"lon":-0.1376103},{"type":"node","id":454535722,"lat":51.5052109,"lon":-0.1339252},{"type":"node","id":454535735,"lat":51.5051171,"lon":-0.1337977},{"type":"node","id":454535807,"lat":51.5052588,"lon":-0.1338059},{"type":"node","id":454535808,"lat":51.5051598,"lon":-0.1336947},{"type":"node","id":454535997,"lat":51.5049497,"lon":-0.1335469},{"type":"node","id":454543312,"lat":51.5064143,"lon":-0.1316971},{"type":"node","id":454543496,"lat":51.5071,"lon":-0.1326163},{"type":"node","id":454543585,"lat":51.507582,"lon":-0.132911},{"type":"node","id":454543632,"lat":51.5075651,"lon":-0.1326802},{"type":"node","id":506328021,"lat":51.5066264,"lon":-0.1368842},{"type":"node","id":506328368,"lat":51.5061934,"lon":-0.1364674},{"type":"node","id":853294005,"lat":51.5057995,"lon":-0.1312372},{"type":"node","id":878030001,"lat":51.5054926,"lon":-0.1370941},{"type":"node","id":1188939726,"lat":51.5077609,"lon":-0.1311623},{"type":"node","id":1353135844,"lat":51.5065609,"lon":-0.1368225},{"type":"node","id":1353135845,"lat":51.5064372,"lon":-0.136703},{"type":"node","id":1353135846,"lat":51.5065365,"lon":-0.1371354},{"type":"node","id":1353135847,"lat":51.5063411,"lon":-0.1369535},{"type":"node","id":1353135848,"lat":51.506007,"lon":-0.1366798},{"type":"node","id":1353135849,"lat":51.5064776,"lon":-0.1370777},{"type":"node","id":1423409445,"lat":51.5064182,"lon":-0.1355529},{"type":"node","id":1598214602,"lat":51.5070268,"lon":-0.1328317},{"type":"node","id":1598214612,"lat":51.5072437,"lon":-0.1322089},{"type":"node","id":1637831175,"lat":51.5076925,"lon":-0.1311587},{"type":"node","id":1637831207,"lat":51.5078178,"lon":-0.131196},{"type":"node","id":1637831214,"lat":51.5078465,"lon":-0.1311173},{"type":"node","id":1737601113,"lat":51.5048001,"lon":-0.1333921},{"type":"node","id":2214979323,"lat":51.5054152,"lon":-0.1372466},{"type":"node","id":2214979326,"lat":51.5061958,"lon":-0.1353447},{"type":"node","id":2214979328,"lat":51.5056174,"lon":-0.1369577},{"type":"node","id":2214979329,"lat":51.5071618,"lon":-0.1326672},{"type":"node","id":2214979330,"lat":51.5055449,"lon":-0.1371434},{"type":"node","id":2214979332,"lat":51.5066156,"lon":-0.1339607},{"type":"node","id":2214979333,"lat":51.5067058,"lon":-0.1337028},{"type":"node","id":2214979334,"lat":51.5054592,"lon":-0.1371703},{"type":"node","id":2214979335,"lat":51.506676,"lon":-0.1340102},{"type":"node","id":2214979336,"lat":51.5051995,"lon":-0.137617},{"type":"node","id":2214979338,"lat":51.5070809,"lon":-0.1328814},{"type":"node","id":2214979340,"lat":51.5067245,"lon":-0.1338793},{"type":"node","id":2214979341,"lat":51.5067647,"lon":-0.1337568},{"type":"node","id":2214979342,"lat":51.5053347,"lon":-0.1375266},{"type":"node","id":2214979345,"lat":51.5053452,"lon":-0.1375827},{"type":"node","id":2214979346,"lat":51.5072587,"lon":-0.1323969},{"type":"node","id":2214979349,"lat":51.5052832,"lon":-0.1374569},{"type":"node","id":2214979351,"lat":51.5053192,"lon":-0.1374896},{"type":"node","id":2214988206,"lat":51.5073045,"lon":-0.1322626},{"type":"node","id":2214988209,"lat":51.5076198,"lon":-0.1313654},{"type":"node","id":2214988212,"lat":51.5076414,"lon":-0.131304},{"type":"node","id":2214988215,"lat":51.5074569,"lon":-0.131829},{"type":"node","id":2620932404,"lat":51.5037185,"lon":-0.1360138},{"type":"node","id":2620932405,"lat":51.50442,"lon":-0.1367042},{"type":"node","id":2622783493,"lat":51.5075914,"lon":-0.1369866},{"type":"node","id":2622783494,"lat":51.507745,"lon":-0.1371288},{"type":"node","id":2622783496,"lat":51.5074399,"lon":-0.1374573},{"type":"node","id":3037800597,"lat":51.5075065,"lon":-0.1318654},{"type":"node","id":3439409104,"lat":51.5075829,"lon":-0.1327907},{"type":"node","id":3439409106,"lat":51.5075449,"lon":-0.1328991},{"type":"node","id":3439409107,"lat":51.5074565,"lon":-0.1329274},{"type":"node","id":3439409114,"lat":51.5078258,"lon":-0.1310968},{"type":"node","id":3439409115,"lat":51.5079547,"lon":-0.131203},{"type":"node","id":3439409116,"lat":51.5079977,"lon":-0.1312507},{"type":"node","id":3439409117,"lat":51.508023,"lon":-0.1312826},{"type":"node","id":3439409119,"lat":51.5077983,"lon":-0.1311835},{"type":"node","id":3439409120,"lat":51.5077157,"lon":-0.1311499},{"type":"node","id":3439409121,"lat":51.5077388,"lon":-0.1311552},{"type":"node","id":3557235398,"lat":51.505787,"lon":-0.1334797},{"type":"node","id":3557235418,"lat":51.5053907,"lon":-0.1340912},{"type":"node","id":3557235419,"lat":51.5054246,"lon":-0.1340767},{"type":"node","id":3557235420,"lat":51.5053874,"lon":-0.1337497},{"type":"node","id":3557235421,"lat":51.5053526,"lon":-0.1337554},{"type":"node","id":3557235422,"lat":51.5053207,"lon":-0.1337789},{"type":"node","id":3557235423,"lat":51.5054537,"lon":-0.1340453},{"type":"node","id":3557235424,"lat":51.505486,"lon":-0.1339462},{"type":"node","id":3557235425,"lat":51.5054733,"lon":-0.1338363},{"type":"node","id":3557235426,"lat":51.5054513,"lon":-0.1337925},{"type":"node","id":3557235427,"lat":51.5054216,"lon":-0.1337626},{"type":"node","id":3557235428,"lat":51.5052953,"lon":-0.1338177},{"type":"node","id":3557235429,"lat":51.5052791,"lon":-0.1338677},{"type":"node","id":3557235430,"lat":51.5052739,"lon":-0.1339233},{"type":"node","id":3557235431,"lat":51.5052802,"lon":-0.1339787},{"type":"node","id":3557235432,"lat":51.5052973,"lon":-0.1340279},{"type":"node","id":3557235433,"lat":51.5053235,"lon":-0.1340654},{"type":"node","id":3557235434,"lat":51.5053558,"lon":-0.1340873},{"type":"node","id":3557235435,"lat":51.5054925,"lon":-0.1340199},{"type":"node","id":3557235436,"lat":51.5055034,"lon":-0.1340239},{"type":"node","id":3557235437,"lat":51.5054853,"lon":-0.1338892},{"type":"node","id":3557235438,"lat":51.5055146,"lon":-0.1340221},{"type":"node","id":3557235439,"lat":51.5055247,"lon":-0.1340146},{"type":"node","id":3557235440,"lat":51.5055328,"lon":-0.1340021},{"type":"node","id":3557235441,"lat":51.505538,"lon":-0.1339861},{"type":"node","id":3557235442,"lat":51.5055396,"lon":-0.1339683},{"type":"node","id":3557235443,"lat":51.5055375,"lon":-0.1339507},{"type":"node","id":3557235444,"lat":51.505532,"lon":-0.133935},{"type":"node","id":3557235445,"lat":51.5055237,"lon":-0.1339231},{"type":"node","id":3557235446,"lat":51.5054749,"lon":-0.1340004},{"type":"node","id":3557235449,"lat":51.5055939,"lon":-0.1340504},{"type":"node","id":3561398893,"lat":51.5055965,"lon":-0.1375328},{"type":"node","id":3561398894,"lat":51.5056378,"lon":-0.1375751},{"type":"node","id":3713564059,"lat":51.5068173,"lon":-0.1345893},{"type":"node","id":3783436255,"lat":51.5036243,"lon":-0.1362422},{"type":"node","id":3933844368,"lat":51.5069849,"lon":-0.1341314},{"type":"node","id":3933850766,"lat":51.5072896,"lon":-0.1363027},{"type":"node","id":3933850767,"lat":51.5074324,"lon":-0.1363027},{"type":"node","id":3933850768,"lat":51.5073965,"lon":-0.1363228},{"type":"node","id":3933850769,"lat":51.5073531,"lon":-0.1363268},{"type":"node","id":3933850770,"lat":51.5073314,"lon":-0.1363255},{"type":"node","id":4170421482,"lat":51.5070061,"lon":-0.1374355},{"type":"node","id":4187528682,"lat":51.5064578,"lon":-0.1353104},{"type":"node","id":4187528683,"lat":51.5065289,"lon":-0.1353778},{"type":"node","id":4209437291,"lat":51.5075854,"lon":-0.137589},{"type":"node","id":4210456241,"lat":51.5073964,"lon":-0.1375791},{"type":"node","id":4347914533,"lat":51.5072682,"lon":-0.1321403},{"type":"node","id":4347914534,"lat":51.5072874,"lon":-0.1323113},{"type":"node","id":4347914535,"lat":51.507896,"lon":-0.1311565},{"type":"node","id":4347914541,"lat":51.5078792,"lon":-0.1312386},{"type":"node","id":4580163089,"lat":51.5075944,"lon":-0.1328022},{"type":"node","id":4580163090,"lat":51.5079749,"lon":-0.1331572},{"type":"node","id":5290771955,"lat":51.5039179,"lon":-0.1359775},{"type":"node","id":5357377731,"lat":51.5059831,"lon":-0.1359378},{"type":"node","id":5357377732,"lat":51.5055608,"lon":-0.136907},{"type":"node","id":6266824886,"lat":51.5062406,"lon":-0.1369046},{"type":"node","id":6266824887,"lat":51.5063136,"lon":-0.1369247},{"type":"node","id":6297968144,"lat":51.507875,"lon":-0.1350233},{"type":"node","id":6297968145,"lat":51.5078552,"lon":-0.1351715},{"type":"node","id":6297968146,"lat":51.5078331,"lon":-0.1349521},{"type":"node","id":6329186471,"lat":51.5075814,"lon":-0.1327284},{"type":"node","id":6329186472,"lat":51.5075021,"lon":-0.1329403},{"type":"node","id":6329186510,"lat":51.50608,"lon":-0.131514},{"type":"node","id":6329186511,"lat":51.506103,"lon":-0.1315368},{"type":"node","id":6329186512,"lat":51.5061371,"lon":-0.1315717},{"type":"node","id":6329186513,"lat":51.5061589,"lon":-0.1315923},{"type":"node","id":6329186514,"lat":51.5061909,"lon":-0.1316241},{"type":"node","id":6329186515,"lat":51.5062169,"lon":-0.13165},{"type":"node","id":108054,"lat":51.5139899,"lon":-0.1415321},{"type":"node","id":108055,"lat":51.5131204,"lon":-0.1407061},{"type":"node","id":108057,"lat":51.5110613,"lon":-0.138773},{"type":"node","id":108058,"lat":51.5103796,"lon":-0.1381732},{"type":"node","id":108070,"lat":51.5121078,"lon":-0.1382034},{"type":"node","id":108072,"lat":51.5130185,"lon":-0.1376622},{"type":"node","id":108247,"lat":51.5105569,"lon":-0.1426593},{"type":"node","id":108248,"lat":51.5101355,"lon":-0.1430044},{"type":"node","id":108249,"lat":51.5103572,"lon":-0.1424536},{"type":"node","id":108261,"lat":51.51007,"lon":-0.1376647},{"type":"node","id":108267,"lat":51.5086452,"lon":-0.1414955},{"type":"node","id":109631,"lat":51.5080733,"lon":-0.1399878},{"type":"node","id":9791491,"lat":51.514238,"lon":-0.1393883},{"type":"node","id":9791492,"lat":51.5140159,"lon":-0.1402203},{"type":"node","id":9791494,"lat":51.5136756,"lon":-0.1412668},{"type":"node","id":21665926,"lat":51.5138611,"lon":-0.1394619},{"type":"node","id":21665927,"lat":51.5138004,"lon":-0.1394503},{"type":"node","id":21665929,"lat":51.5134628,"lon":-0.1391495},{"type":"node","id":21665930,"lat":51.513109,"lon":-0.1387747},{"type":"node","id":21665931,"lat":51.5133329,"lon":-0.1390071},{"type":"node","id":21665965,"lat":51.5144698,"lon":-0.1385631},{"type":"node","id":21666002,"lat":51.5145331,"lon":-0.1411287},{"type":"node","id":21666021,"lat":51.5143616,"lon":-0.1417608},{"type":"node","id":25256840,"lat":51.5126324,"lon":-0.1431584},{"type":"node","id":25257039,"lat":51.5127338,"lon":-0.1425571},{"type":"node","id":25257040,"lat":51.5120833,"lon":-0.1415143},{"type":"node","id":25257041,"lat":51.5116974,"lon":-0.1411662},{"type":"node","id":25257042,"lat":51.511573,"lon":-0.1410455},{"type":"node","id":25257043,"lat":51.5112792,"lon":-0.1407705},{"type":"node","id":25257044,"lat":51.5110756,"lon":-0.1412998},{"type":"node","id":25257045,"lat":51.510824,"lon":-0.1419699},{"type":"node","id":25257046,"lat":51.5114768,"lon":-0.1417099},{"type":"node","id":25257047,"lat":51.511177,"lon":-0.1423243},{"type":"node","id":25257321,"lat":51.5083934,"lon":-0.1421623},{"type":"node","id":25257323,"lat":51.5097405,"lon":-0.140896},{"type":"node","id":25257324,"lat":51.5089121,"lon":-0.1408325},{"type":"node","id":25257463,"lat":51.5099924,"lon":-0.1402405},{"type":"node","id":25257616,"lat":51.5083935,"lon":-0.1387437},{"type":"node","id":25257617,"lat":51.5082645,"lon":-0.1394468},{"type":"node","id":25257692,"lat":51.5112806,"lon":-0.1381779},{"type":"node","id":25473286,"lat":51.511712,"lon":-0.137775},{"type":"node","id":25473295,"lat":51.5118794,"lon":-0.138726},{"type":"node","id":25473371,"lat":51.5119972,"lon":-0.1384594},{"type":"node","id":25473373,"lat":51.5127723,"lon":-0.1394587},{"type":"node","id":25473375,"lat":51.513329,"lon":-0.1401668},{"type":"node","id":25473398,"lat":51.5135205,"lon":-0.1404167},{"type":"node","id":25473399,"lat":51.5138453,"lon":-0.1407526},{"type":"node","id":25473512,"lat":51.5122587,"lon":-0.1378479},{"type":"node","id":25473513,"lat":51.5127084,"lon":-0.1383352},{"type":"node","id":25473557,"lat":51.5139318,"lon":-0.1384096},{"type":"node","id":25501320,"lat":51.5138414,"lon":-0.138934},{"type":"node","id":25501325,"lat":51.5136464,"lon":-0.1387422},{"type":"node","id":25501328,"lat":51.5134964,"lon":-0.1385962},{"type":"node","id":25501330,"lat":51.5132667,"lon":-0.138377},{"type":"node","id":25501340,"lat":51.5134078,"lon":-0.1379482},{"type":"node","id":25531001,"lat":51.5132163,"lon":-0.1419371},{"type":"node","id":25531002,"lat":51.5138064,"lon":-0.1424346},{"type":"node","id":25531003,"lat":51.5135172,"lon":-0.1423097},{"type":"node","id":25531180,"lat":51.5129775,"lon":-0.1425402},{"type":"node","id":25531181,"lat":51.5132522,"lon":-0.1426084},{"type":"node","id":25531990,"lat":51.5124377,"lon":-0.1404463},{"type":"node","id":26651415,"lat":51.509957,"lon":-0.1416201},{"type":"node","id":26651416,"lat":51.5100979,"lon":-0.1412561},{"type":"node","id":26652125,"lat":51.5098565,"lon":-0.1378539},{"type":"node","id":26652135,"lat":51.5089549,"lon":-0.1418085},{"type":"node","id":26652137,"lat":51.509218,"lon":-0.1411417},{"type":"node","id":26652140,"lat":51.5086437,"lon":-0.1382156},{"type":"node","id":266481785,"lat":51.5107716,"lon":-0.1395487},{"type":"node","id":266481798,"lat":51.5108338,"lon":-0.1393778},{"type":"node","id":452219937,"lat":51.5128805,"lon":-0.1423406},{"type":"node","id":452589682,"lat":51.5113053,"lon":-0.1397947},{"type":"node","id":452589683,"lat":51.5125346,"lon":-0.140195},{"type":"node","id":452589685,"lat":51.511226,"lon":-0.1400319},{"type":"node","id":452589687,"lat":51.5113998,"lon":-0.1395052},{"type":"node","id":454360126,"lat":51.5101518,"lon":-0.1397786},{"type":"node","id":454360127,"lat":51.5113844,"lon":-0.1419697},{"type":"node","id":454360129,"lat":51.5112637,"lon":-0.1421935},{"type":"node","id":454360130,"lat":51.5110448,"lon":-0.1424787},{"type":"node","id":454360131,"lat":51.5110923,"lon":-0.1424276},{"type":"node","id":454368510,"lat":51.5102583,"lon":-0.1397233},{"type":"node","id":454368514,"lat":51.5121467,"lon":-0.1415983},{"type":"node","id":454368517,"lat":51.512168,"lon":-0.1419632},{"type":"node","id":454368521,"lat":51.5119078,"lon":-0.1422833},{"type":"node","id":454368522,"lat":51.5116854,"lon":-0.1425704},{"type":"node","id":469780509,"lat":51.5083314,"lon":-0.1391095},{"type":"node","id":499350858,"lat":51.5137145,"lon":-0.1381855},{"type":"node","id":701600719,"lat":51.5132356,"lon":-0.1378348},{"type":"node","id":701600720,"lat":51.5132619,"lon":-0.1377316},{"type":"node","id":701600721,"lat":51.5132755,"lon":-0.1376798},{"type":"node","id":701600741,"lat":51.5141723,"lon":-0.1396462},{"type":"node","id":701600743,"lat":51.5139351,"lon":-0.1394271},{"type":"node","id":701600744,"lat":51.5138388,"lon":-0.1390044},{"type":"node","id":844784034,"lat":51.5103355,"lon":-0.142158},{"type":"node","id":881888130,"lat":51.5116591,"lon":-0.1392861},{"type":"node","id":1104463098,"lat":51.5085918,"lon":-0.1391017},{"type":"node","id":1104463100,"lat":51.5084915,"lon":-0.1389985},{"type":"node","id":1104463102,"lat":51.5085848,"lon":-0.1390934},{"type":"node","id":1188939768,"lat":51.5102109,"lon":-0.1379707},{"type":"node","id":1238318167,"lat":51.509727,"lon":-0.1409327},{"type":"node","id":1493328983,"lat":51.5141989,"lon":-0.1395365},{"type":"node","id":1496723557,"lat":51.5134822,"lon":-0.1380072},{"type":"node","id":1496724129,"lat":51.5139223,"lon":-0.1383586},{"type":"node","id":1771472506,"lat":51.5081551,"lon":-0.1400668},{"type":"node","id":1947324460,"lat":51.5101593,"lon":-0.1395111},{"type":"node","id":1947324461,"lat":51.5102629,"lon":-0.1389583},{"type":"node","id":1947324474,"lat":51.5130717,"lon":-0.1408148},{"type":"node","id":1947324488,"lat":51.5140984,"lon":-0.1399307},{"type":"node","id":1947324489,"lat":51.5141243,"lon":-0.1416306},{"type":"node","id":2047806092,"lat":51.5137155,"lon":-0.1411476},{"type":"node","id":2047806095,"lat":51.5141298,"lon":-0.1414889},{"type":"node","id":2047806097,"lat":51.5134683,"lon":-0.1409331},{"type":"node","id":2047806103,"lat":51.5144776,"lon":-0.141692},{"type":"node","id":2047806106,"lat":51.514376,"lon":-0.1416487},{"type":"node","id":2047806108,"lat":51.514031,"lon":-0.1414097},{"type":"node","id":2047806113,"lat":51.5138343,"lon":-0.1412487},{"type":"node","id":2047806126,"lat":51.5133874,"lon":-0.1408576},{"type":"node","id":2047806132,"lat":51.5142437,"lon":-0.1417071},{"type":"node","id":2047806133,"lat":51.5136525,"lon":-0.1413312},{"type":"node","id":2047806134,"lat":51.514177,"lon":-0.1415292},{"type":"node","id":2129184196,"lat":51.511004,"lon":-0.1425176},{"type":"node","id":2129184256,"lat":51.5122287,"lon":-0.1410639},{"type":"node","id":2129184285,"lat":51.5123723,"lon":-0.1412052},{"type":"node","id":2129908162,"lat":51.510139,"lon":-0.1396123},{"type":"node","id":2131603302,"lat":51.5136019,"lon":-0.1414209},{"type":"node","id":2131643979,"lat":51.5116259,"lon":-0.142642},{"type":"node","id":2131643980,"lat":51.5122857,"lon":-0.1418184},{"type":"node","id":2131643981,"lat":51.5123421,"lon":-0.1417455},{"type":"node","id":2131643994,"lat":51.5126647,"lon":-0.1425201},{"type":"node","id":2131644076,"lat":51.513025,"lon":-0.1406265},{"type":"node","id":2214952982,"lat":51.5085071,"lon":-0.1386498},{"type":"node","id":2214952985,"lat":51.5086735,"lon":-0.1378101},{"type":"node","id":2214954341,"lat":51.5084361,"lon":-0.1385888},{"type":"node","id":2215472133,"lat":51.5081944,"lon":-0.1393878},{"type":"node","id":2215472136,"lat":51.508305,"lon":-0.1386698},{"type":"node","id":2215472141,"lat":51.50817,"lon":-0.1397048},{"type":"node","id":2215475640,"lat":51.5084635,"lon":-0.138499},{"type":"node","id":2215502652,"lat":51.5081904,"lon":-0.1401006},{"type":"node","id":2215506073,"lat":51.5083369,"lon":-0.1386969},{"type":"node","id":2323063170,"lat":51.5133973,"lon":-0.1380333},{"type":"node","id":2323063175,"lat":51.5134751,"lon":-0.1386525},{"type":"node","id":2398143017,"lat":51.5099827,"lon":-0.1377347},{"type":"node","id":2398143020,"lat":51.5101373,"lon":-0.1378297},{"type":"node","id":2398143023,"lat":51.5103036,"lon":-0.1380968},{"type":"node","id":2398143027,"lat":51.510366,"lon":-0.138411},{"type":"node","id":2465858762,"lat":51.5133504,"lon":-0.1401927},{"type":"node","id":2480890185,"lat":51.5138633,"lon":-0.1412723},{"type":"node","id":2480891599,"lat":51.5142298,"lon":-0.1415675},{"type":"node","id":2480891600,"lat":51.5139682,"lon":-0.1416094},{"type":"node","id":2480891601,"lat":51.5137505,"lon":-0.1413351},{"type":"node","id":2480891602,"lat":51.5138876,"lon":-0.1414503},{"type":"node","id":2480891605,"lat":51.51375,"lon":-0.141047},{"type":"node","id":2480891606,"lat":51.5136216,"lon":-0.1413936},{"type":"node","id":2480891607,"lat":51.513639,"lon":-0.1413635},{"type":"node","id":2480891608,"lat":51.5142915,"lon":-0.1416102},{"type":"node","id":2480891609,"lat":51.5137719,"lon":-0.1409779},{"type":"node","id":2480891611,"lat":51.5140857,"lon":-0.1414538},{"type":"node","id":2480891612,"lat":51.5135929,"lon":-0.1411909},{"type":"node","id":2480891614,"lat":51.5140516,"lon":-0.1415785},{"type":"node","id":2480891615,"lat":51.5143901,"lon":-0.1416021},{"type":"node","id":2480891617,"lat":51.5137898,"lon":-0.1412114},{"type":"node","id":2480894744,"lat":51.513986,"lon":-0.1417065},{"type":"node","id":2480897574,"lat":51.5139127,"lon":-0.1405439},{"type":"node","id":2770153328,"lat":51.5137349,"lon":-0.1381157},{"type":"node","id":2770153342,"lat":51.5137681,"lon":-0.1379918},{"type":"node","id":2771382066,"lat":51.513634,"lon":-0.1401722},{"type":"node","id":2771382106,"lat":51.51374,"lon":-0.1399023},{"type":"node","id":2771382135,"lat":51.513825,"lon":-0.139667},{"type":"node","id":2771382140,"lat":51.5138564,"lon":-0.1395729},{"type":"node","id":2772491043,"lat":51.5131376,"lon":-0.1387027},{"type":"node","id":2772491048,"lat":51.5133178,"lon":-0.1384281},{"type":"node","id":2772491052,"lat":51.5133571,"lon":-0.1389454},{"type":"node","id":2772491062,"lat":51.5134817,"lon":-0.1386352},{"type":"node","id":2772491064,"lat":51.5134941,"lon":-0.13908},{"type":"node","id":2772491078,"lat":51.5136314,"lon":-0.1387754},{"type":"node","id":2772491086,"lat":51.5138271,"lon":-0.13892},{"type":"node","id":2784434277,"lat":51.5122879,"lon":-0.1378781},{"type":"node","id":2784434310,"lat":51.5127168,"lon":-0.1385986},{"type":"node","id":2784434322,"lat":51.5127713,"lon":-0.1384766},{"type":"node","id":2784434327,"lat":51.5127894,"lon":-0.1384323},{"type":"node","id":2784434339,"lat":51.512799,"lon":-0.1394044},{"type":"node","id":2784434416,"lat":51.5130886,"lon":-0.1388161},{"type":"node","id":2784434451,"lat":51.5133484,"lon":-0.1401225},{"type":"node","id":2784434458,"lat":51.5138461,"lon":-0.1396039},{"type":"node","id":2784682584,"lat":51.5127316,"lon":-0.1382842},{"type":"node","id":2784682596,"lat":51.5127799,"lon":-0.1377108},{"type":"node","id":2784682599,"lat":51.5127869,"lon":-0.1376917},{"type":"node","id":2784682610,"lat":51.5128462,"lon":-0.1380321},{"type":"node","id":2784682614,"lat":51.5128565,"lon":-0.1380093},{"type":"node","id":2784682619,"lat":51.512893,"lon":-0.1380499},{"type":"node","id":2784682621,"lat":51.5129099,"lon":-0.1380696},{"type":"node","id":2784682639,"lat":51.5130376,"lon":-0.1382038},{"type":"node","id":2910874877,"lat":51.5095331,"lon":-0.141448},{"type":"node","id":2932939310,"lat":51.5081223,"lon":-0.1400363},{"type":"node","id":2934851453,"lat":51.5085329,"lon":-0.1417911},{"type":"node","id":2939875788,"lat":51.509898,"lon":-0.1378134},{"type":"node","id":2939875794,"lat":51.5099751,"lon":-0.1377417},{"type":"node","id":2939877210,"lat":51.5103925,"lon":-0.1380555},{"type":"node","id":2939877211,"lat":51.5105168,"lon":-0.138289},{"type":"node","id":2941537158,"lat":51.5110858,"lon":-0.138706},{"type":"node","id":2941537163,"lat":51.5121652,"lon":-0.139664},{"type":"node","id":2941548715,"lat":51.5120217,"lon":-0.1398622},{"type":"node","id":2941614011,"lat":51.5120343,"lon":-0.1396093},{"type":"node","id":2949076333,"lat":51.5115849,"lon":-0.1392211},{"type":"node","id":2949076335,"lat":51.5116976,"lon":-0.1391678},{"type":"node","id":3069482720,"lat":51.5126502,"lon":-0.1430659},{"type":"node","id":3069482721,"lat":51.5126577,"lon":-0.143026},{"type":"node","id":3077397235,"lat":51.512711,"lon":-0.1388995},{"type":"node","id":3458411641,"lat":51.5099799,"lon":-0.1419564},{"type":"node","id":4190435879,"lat":51.5088616,"lon":-0.1376624},{"type":"node","id":4219739061,"lat":51.5134364,"lon":-0.1399213},{"type":"node","id":4219739065,"lat":51.5132492,"lon":-0.1400653},{"type":"node","id":4233926315,"lat":51.5136614,"lon":-0.1394212},{"type":"node","id":4233926316,"lat":51.5136844,"lon":-0.1393628},{"type":"node","id":4233926317,"lat":51.5137053,"lon":-0.1393075},{"type":"node","id":4347805227,"lat":51.5130001,"lon":-0.1406041},{"type":"node","id":4347805229,"lat":51.513057,"lon":-0.1408457},{"type":"node","id":4347805230,"lat":51.5132632,"lon":-0.140732},{"type":"node","id":4347844126,"lat":51.5114758,"lon":-0.1392665},{"type":"node","id":4347844127,"lat":51.5121738,"lon":-0.1397324},{"type":"node","id":4347844128,"lat":51.5130569,"lon":-0.1405283},{"type":"node","id":4347844130,"lat":51.5119766,"lon":-0.1396974},{"type":"node","id":4347844131,"lat":51.5103774,"lon":-0.1383123},{"type":"node","id":4347844133,"lat":51.5117002,"lon":-0.1394631},{"type":"node","id":4347844134,"lat":51.5120577,"lon":-0.1397719},{"type":"node","id":4347844136,"lat":51.5105377,"lon":-0.138432},{"type":"node","id":4347844138,"lat":51.5110099,"lon":-0.1388556},{"type":"node","id":4347844140,"lat":51.5110118,"lon":-0.1388573},{"type":"node","id":4347844142,"lat":51.512647,"lon":-0.1401608},{"type":"node","id":4347844144,"lat":51.5131625,"lon":-0.1405459},{"type":"node","id":4347844146,"lat":51.5120075,"lon":-0.1399059},{"type":"node","id":4347844147,"lat":51.5120969,"lon":-0.1396598},{"type":"node","id":4347844152,"lat":51.5131478,"lon":-0.1406182},{"type":"node","id":4347844154,"lat":51.5101758,"lon":-0.138084},{"type":"node","id":4347844155,"lat":51.5102668,"lon":-0.1382039},{"type":"node","id":4347844156,"lat":51.5099952,"lon":-0.1377237},{"type":"node","id":4347844157,"lat":51.5100721,"lon":-0.1379072},{"type":"node","id":4347844159,"lat":51.512129,"lon":-0.1398373},{"type":"node","id":4347844160,"lat":51.5121667,"lon":-0.1397271},{"type":"node","id":4347844161,"lat":51.5119993,"lon":-0.1397182},{"type":"node","id":4347844162,"lat":51.5105772,"lon":-0.138337},{"type":"node","id":4347870211,"lat":51.5115526,"lon":-0.139331},{"type":"node","id":4347870212,"lat":51.5117646,"lon":-0.1393737},{"type":"node","id":4347870213,"lat":51.5117299,"lon":-0.1394883},{"type":"node","id":4347870214,"lat":51.5115066,"lon":-0.1392938},{"type":"node","id":4347870215,"lat":51.5118031,"lon":-0.1394057},{"type":"node","id":4347870216,"lat":51.5117107,"lon":-0.1391321},{"type":"node","id":4347880277,"lat":51.5103157,"lon":-0.1382542},{"type":"node","id":4347880278,"lat":51.510478,"lon":-0.1383829},{"type":"node","id":4347880280,"lat":51.5103587,"lon":-0.1384514},{"type":"node","id":4394860088,"lat":51.5121402,"lon":-0.1382148},{"type":"node","id":4394862789,"lat":51.5122015,"lon":-0.1382862},{"type":"node","id":4629852537,"lat":51.5144661,"lon":-0.1418015},{"type":"node","id":4629852538,"lat":51.5145536,"lon":-0.141722},{"type":"node","id":4631979548,"lat":51.5125988,"lon":-0.141422},{"type":"node","id":4631979549,"lat":51.5123587,"lon":-0.1400389},{"type":"node","id":4631979550,"lat":51.5123387,"lon":-0.1398817},{"type":"node","id":4631979551,"lat":51.5113339,"lon":-0.1391413},{"type":"node","id":4631979552,"lat":51.5112816,"lon":-0.1389615},{"type":"node","id":4684519081,"lat":51.5145054,"lon":-0.1385889},{"type":"node","id":5112906884,"lat":51.5124325,"lon":-0.1390202},{"type":"node","id":5112906885,"lat":51.5124956,"lon":-0.1390973},{"type":"node","id":5240648782,"lat":51.5094429,"lon":-0.1413603},{"type":"node","id":5568839679,"lat":51.5126242,"lon":-0.1387944},{"type":"node","id":5630578551,"lat":51.5127463,"lon":-0.1425052},{"type":"node","id":5657844054,"lat":51.511586,"lon":-0.1393609},{"type":"node","id":5673161958,"lat":51.5100908,"lon":-0.141792},{"type":"node","id":5673161959,"lat":51.5100915,"lon":-0.1418277},{"type":"node","id":6034520602,"lat":51.5082855,"lon":-0.1386529},{"type":"node","id":6250340166,"lat":51.5137298,"lon":-0.1428142},{"type":"node","id":6250340168,"lat":51.5139445,"lon":-0.1416765},{"type":"node","id":6250340169,"lat":51.5139766,"lon":-0.1417569},{"type":"node","id":6250340171,"lat":51.5139337,"lon":-0.1417689},{"type":"node","id":6250445848,"lat":51.5116241,"lon":-0.139395},{"type":"node","id":6250445849,"lat":51.5110448,"lon":-0.1387581},{"type":"node","id":6266808740,"lat":51.508404,"lon":-0.1387751},{"type":"node","id":6266808741,"lat":51.5084918,"lon":-0.1387008},{"type":"node","id":6266808742,"lat":51.5084174,"lon":-0.1388487},{"type":"node","id":6266808743,"lat":51.5084132,"lon":-0.1388091},{"type":"node","id":6266808744,"lat":51.508136,"lon":-0.1395453},{"type":"node","id":6266808745,"lat":51.508257,"lon":-0.139335},{"type":"node","id":6266808746,"lat":51.5084645,"lon":-0.1387949},{"type":"node","id":6266808747,"lat":51.5083971,"lon":-0.1389049},{"type":"node","id":6266808748,"lat":51.5082764,"lon":-0.1392763},{"type":"node","id":6313645586,"lat":51.5137687,"lon":-0.1426215},{"type":"node","id":6352084888,"lat":51.5133629,"lon":-0.1422359},{"type":"node","id":107793,"lat":51.5110264,"lon":-0.1318385},{"type":"node","id":107797,"lat":51.5092299,"lon":-0.1324028},{"type":"node","id":107798,"lat":51.5101031,"lon":-0.1333483},{"type":"node","id":107799,"lat":51.5102758,"lon":-0.1322859},{"type":"node","id":107800,"lat":51.5102942,"lon":-0.1321586},{"type":"node","id":107801,"lat":51.5095703,"lon":-0.1315707},{"type":"node","id":107802,"lat":51.5084434,"lon":-0.1317124},{"type":"node","id":107807,"lat":51.5121964,"lon":-0.1314662},{"type":"node","id":107809,"lat":51.5114964,"lon":-0.1330266},{"type":"node","id":107812,"lat":51.5113932,"lon":-0.1343278},{"type":"node","id":107813,"lat":51.5106947,"lon":-0.1339402},{"type":"node","id":107815,"lat":51.5100376,"lon":-0.1339621},{"type":"node","id":107818,"lat":51.5097303,"lon":-0.1347729},{"type":"node","id":107820,"lat":51.5107007,"lon":-0.1355377},{"type":"node","id":107821,"lat":51.5103745,"lon":-0.1352468},{"type":"node","id":108059,"lat":51.5105189,"lon":-0.1373792},{"type":"node","id":108060,"lat":51.5108324,"lon":-0.1368041},{"type":"node","id":108061,"lat":51.5104722,"lon":-0.1366159},{"type":"node","id":108073,"lat":51.5134685,"lon":-0.136261},{"type":"node","id":108262,"lat":51.5098797,"lon":-0.1369082},{"type":"node","id":108263,"lat":51.5098718,"lon":-0.1361711},{"type":"node","id":9512915,"lat":51.510432,"lon":-0.1314738},{"type":"node","id":9512924,"lat":51.511814,"lon":-0.1323773},{"type":"node","id":9512925,"lat":51.5124696,"lon":-0.1331364},{"type":"node","id":9512926,"lat":51.5140939,"lon":-0.1344381},{"type":"node","id":9512927,"lat":51.5122307,"lon":-0.1336503},{"type":"node","id":9512928,"lat":51.5127882,"lon":-0.134171},{"type":"node","id":9512930,"lat":51.5138581,"lon":-0.1351405},{"type":"node","id":9512933,"lat":51.5137376,"lon":-0.1325297},{"type":"node","id":9512935,"lat":51.512922,"lon":-0.131952},{"type":"node","id":9512937,"lat":51.5132242,"lon":-0.1311835},{"type":"node","id":9789808,"lat":51.5092153,"lon":-0.1342896},{"type":"node","id":9789811,"lat":51.5084795,"lon":-0.1364565},{"type":"node","id":9789814,"lat":51.5082321,"lon":-0.1362164},{"type":"node","id":21665000,"lat":51.5140093,"lon":-0.1317206},{"type":"node","id":21665006,"lat":51.5145782,"lon":-0.1312216},{"type":"node","id":21665114,"lat":51.5132942,"lon":-0.1329745},{"type":"node","id":21665359,"lat":51.5115619,"lon":-0.1316904},{"type":"node","id":21665360,"lat":51.5117787,"lon":-0.1311235},{"type":"node","id":21665361,"lat":51.5119728,"lon":-0.131281},{"type":"node","id":21665373,"lat":51.5112158,"lon":-0.1313436},{"type":"node","id":21665386,"lat":51.5118004,"lon":-0.1318364},{"type":"node","id":21665387,"lat":51.5118712,"lon":-0.1315925},{"type":"node","id":21665390,"lat":51.5141094,"lon":-0.1336166},{"type":"node","id":21665577,"lat":51.5125844,"lon":-0.1317213},{"type":"node","id":21665582,"lat":51.5114061,"lon":-0.1321188},{"type":"node","id":21666010,"lat":51.513542,"lon":-0.1359177},{"type":"node","id":21666011,"lat":51.5136676,"lon":-0.1355326},{"type":"node","id":21666019,"lat":51.5143631,"lon":-0.1368698},{"type":"node","id":25257618,"lat":51.5088336,"lon":-0.1375893},{"type":"node","id":25257643,"lat":51.510087,"lon":-0.1346493},{"type":"node","id":25257657,"lat":51.5111896,"lon":-0.1361479},{"type":"node","id":25473284,"lat":51.5115852,"lon":-0.1366},{"type":"node","id":25473285,"lat":51.5112745,"lon":-0.1372837},{"type":"node","id":25473287,"lat":51.5120148,"lon":-0.1370657},{"type":"node","id":25473288,"lat":51.5124148,"lon":-0.1374826},{"type":"node","id":25473291,"lat":51.5118479,"lon":-0.1349627},{"type":"node","id":25473293,"lat":51.5133911,"lon":-0.1365329},{"type":"node","id":25473300,"lat":51.5114043,"lon":-0.1357615},{"type":"node","id":25473408,"lat":51.5129141,"lon":-0.1361578},{"type":"node","id":25473409,"lat":51.5131106,"lon":-0.1355887},{"type":"node","id":25473423,"lat":51.5123967,"lon":-0.1338288},{"type":"node","id":25473424,"lat":51.5125964,"lon":-0.1333511},{"type":"node","id":25473556,"lat":51.5126275,"lon":-0.1369615},{"type":"node","id":25473562,"lat":51.512221,"lon":-0.1341851},{"type":"node","id":25475069,"lat":51.5112459,"lon":-0.1320061},{"type":"node","id":25475083,"lat":51.5110557,"lon":-0.1327346},{"type":"node","id":25475478,"lat":51.5127736,"lon":-0.1365231},{"type":"node","id":25475484,"lat":51.511602,"lon":-0.1353984},{"type":"node","id":25501000,"lat":51.5141224,"lon":-0.1313575},{"type":"node","id":25501063,"lat":51.5142148,"lon":-0.1345379},{"type":"node","id":25501065,"lat":51.5142891,"lon":-0.1328865},{"type":"node","id":25501182,"lat":51.5139631,"lon":-0.1360924},{"type":"node","id":25501184,"lat":51.5141847,"lon":-0.135406},{"type":"node","id":25501288,"lat":51.5136253,"lon":-0.1373333},{"type":"node","id":25501289,"lat":51.5132421,"lon":-0.1370545},{"type":"node","id":25502137,"lat":51.5127696,"lon":-0.1352946},{"type":"node","id":25502191,"lat":51.5110997,"lon":-0.1340588},{"type":"node","id":25502540,"lat":51.5134889,"lon":-0.13443},{"type":"node","id":25502541,"lat":51.513994,"lon":-0.1347984},{"type":"node","id":25502545,"lat":51.5142142,"lon":-0.135},{"type":"node","id":25502552,"lat":51.5135471,"lon":-0.1331755},{"type":"node","id":26241038,"lat":51.5099509,"lon":-0.135611},{"type":"node","id":26559362,"lat":51.5098276,"lon":-0.1318316},{"type":"node","id":26559363,"lat":51.510048,"lon":-0.1312256},{"type":"node","id":26559368,"lat":51.50896,"lon":-0.1331558},{"type":"node","id":26559369,"lat":51.5092547,"lon":-0.1333763},{"type":"node","id":26559370,"lat":51.5094939,"lon":-0.1326532},{"type":"node","id":26559371,"lat":51.5094679,"lon":-0.1335766},{"type":"node","id":26559372,"lat":51.5097095,"lon":-0.1328945},{"type":"node","id":26559581,"lat":51.5088294,"lon":-0.1320508},{"type":"node","id":26559582,"lat":51.5091496,"lon":-0.1311865},{"type":"node","id":26559616,"lat":51.5088232,"lon":-0.1330567},{"type":"node","id":26559618,"lat":51.508204,"lon":-0.1324843},{"type":"node","id":26559620,"lat":51.5085863,"lon":-0.1337436},{"type":"node","id":26561311,"lat":51.5101851,"lon":-0.1327368},{"type":"node","id":26573670,"lat":51.5080803,"lon":-0.1313781},{"type":"node","id":26591525,"lat":51.5084692,"lon":-0.1355017},{"type":"node","id":26651413,"lat":51.5088893,"lon":-0.1352497},{"type":"node","id":26652117,"lat":51.5089915,"lon":-0.1370989},{"type":"node","id":26652119,"lat":51.509393,"lon":-0.1374329},{"type":"node","id":26652126,"lat":51.509474,"lon":-0.1371553},{"type":"node","id":26652129,"lat":51.5090849,"lon":-0.1368057},{"type":"node","id":26652133,"lat":51.5085593,"lon":-0.1362216},{"type":"node","id":26652351,"lat":51.5116039,"lon":-0.1345727},{"type":"node","id":26652357,"lat":51.5113116,"lon":-0.1351458},{"type":"node","id":26652362,"lat":51.5112506,"lon":-0.1346211},{"type":"node","id":26699537,"lat":51.5116421,"lon":-0.1322565},{"type":"node","id":26699539,"lat":51.5130413,"lon":-0.1326366},{"type":"node","id":26699543,"lat":51.5120227,"lon":-0.1334456},{"type":"node","id":26699544,"lat":51.5134376,"lon":-0.1323185},{"type":"node","id":26699545,"lat":51.5123195,"lon":-0.1329587},{"type":"node","id":26699549,"lat":51.5129265,"lon":-0.132818},{"type":"node","id":26699551,"lat":51.5132304,"lon":-0.1321707},{"type":"node","id":26845543,"lat":51.5126358,"lon":-0.1345762},{"type":"node","id":26845544,"lat":51.5128962,"lon":-0.1348445},{"type":"node","id":26845545,"lat":51.5124637,"lon":-0.1350495},{"type":"node","id":26845546,"lat":51.5134097,"lon":-0.1353149},{"type":"node","id":26845547,"lat":51.5135551,"lon":-0.1348905},{"type":"node","id":26845548,"lat":51.5132664,"lon":-0.13386},{"type":"node","id":26845551,"lat":51.5139672,"lon":-0.1343351},{"type":"node","id":26846364,"lat":51.5110255,"lon":-0.1311934},{"type":"node","id":26866991,"lat":51.5144916,"lon":-0.1364393},{"type":"node","id":26866992,"lat":51.514255,"lon":-0.1362412},{"type":"node","id":281455002,"lat":51.5121643,"lon":-0.1315444},{"type":"node","id":348875443,"lat":51.5121887,"lon":-0.1336065},{"type":"node","id":452215489,"lat":51.5095524,"lon":-0.1375789},{"type":"node","id":466926153,"lat":51.5114082,"lon":-0.1369723},{"type":"node","id":466926160,"lat":51.5115145,"lon":-0.1375721},{"type":"node","id":466926169,"lat":51.511837,"lon":-0.1374811},{"type":"node","id":466926170,"lat":51.5118816,"lon":-0.1373765},{"type":"node","id":466926176,"lat":51.5117803,"lon":-0.1368287},{"type":"node","id":497457752,"lat":51.5120677,"lon":-0.1326695},{"type":"node","id":677445933,"lat":51.5085082,"lon":-0.1349044},{"type":"node","id":677445934,"lat":51.5086217,"lon":-0.1345826},{"type":"node","id":677445936,"lat":51.508408,"lon":-0.1344002},{"type":"node","id":677445941,"lat":51.5081818,"lon":-0.1373512},{"type":"node","id":700757166,"lat":51.5140778,"lon":-0.1327547},{"type":"node","id":701600704,"lat":51.5127588,"lon":-0.1374227},{"type":"node","id":701600732,"lat":51.5133038,"lon":-0.1375847},{"type":"node","id":701608595,"lat":51.5127025,"lon":-0.1373683},{"type":"node","id":701608597,"lat":51.5121115,"lon":-0.1359892},{"type":"node","id":701608613,"lat":51.5134407,"lon":-0.1371928},{"type":"node","id":701608618,"lat":51.5135041,"lon":-0.1369865},{"type":"node","id":701608620,"lat":51.5129402,"lon":-0.1360946},{"type":"node","id":701608631,"lat":51.514323,"lon":-0.1368436},{"type":"node","id":734320209,"lat":51.5125357,"lon":-0.1371903},{"type":"node","id":771982848,"lat":51.5144072,"lon":-0.1363715},{"type":"node","id":771982865,"lat":51.5142928,"lon":-0.1361092},{"type":"node","id":771982929,"lat":51.5140231,"lon":-0.1358957},{"type":"node","id":771982972,"lat":51.514315,"lon":-0.1360371},{"type":"node","id":772092173,"lat":51.5130839,"lon":-0.1344308},{"type":"node","id":881888313,"lat":51.5142622,"lon":-0.132871},{"type":"node","id":881888314,"lat":51.5143311,"lon":-0.132587},{"type":"node","id":917763911,"lat":51.5099721,"lon":-0.1373847},{"type":"node","id":917763912,"lat":51.5098994,"lon":-0.1358983},{"type":"node","id":917763913,"lat":51.5099262,"lon":-0.1357288},{"type":"node","id":917763920,"lat":51.5096359,"lon":-0.1360197},{"type":"node","id":917763922,"lat":51.5098609,"lon":-0.1367523},{"type":"node","id":1119774074,"lat":51.5107914,"lon":-0.1340625},{"type":"node","id":1119774075,"lat":51.5108906,"lon":-0.1339436},{"type":"node","id":1119774076,"lat":51.5113665,"lon":-0.1329741},{"type":"node","id":1188939742,"lat":51.5119632,"lon":-0.1364251},{"type":"node","id":1188939746,"lat":51.5122346,"lon":-0.135449},{"type":"node","id":1188939750,"lat":51.5118826,"lon":-0.1348938},{"type":"node","id":1188939751,"lat":51.5126173,"lon":-0.1346128},{"type":"node","id":1188939770,"lat":51.5100194,"lon":-0.1350583},{"type":"node","id":1188939774,"lat":51.5101196,"lon":-0.1351105},{"type":"node","id":1188983785,"lat":51.5103061,"lon":-0.1342447},{"type":"node","id":1227938934,"lat":51.5095215,"lon":-0.1345799},{"type":"node","id":1227938936,"lat":51.5095788,"lon":-0.134634},{"type":"node","id":1227938940,"lat":51.5100659,"lon":-0.1336094},{"type":"node","id":1227938942,"lat":51.5095491,"lon":-0.1347454},{"type":"node","id":1227938943,"lat":51.5096815,"lon":-0.1350107},{"type":"node","id":1322321517,"lat":51.5099867,"lon":-0.1346522},{"type":"node","id":1322321549,"lat":51.5099944,"lon":-0.1350094},{"type":"node","id":1322321579,"lat":51.5101219,"lon":-0.1348971},{"type":"node","id":1322321604,"lat":51.5102992,"lon":-0.1349887},{"type":"node","id":1322321619,"lat":51.5101961,"lon":-0.134399},{"type":"node","id":1322321696,"lat":51.509988,"lon":-0.1353185},{"type":"node","id":1324710127,"lat":51.5137234,"lon":-0.1373081},{"type":"node","id":1330788578,"lat":51.513147,"lon":-0.1350656},{"type":"node","id":1330788629,"lat":51.5127603,"lon":-0.135246},{"type":"node","id":1330788638,"lat":51.5129424,"lon":-0.1354464},{"type":"node","id":1334645223,"lat":51.5111211,"lon":-0.1335561},{"type":"node","id":1387235625,"lat":51.5100278,"lon":-0.1351837},{"type":"node","id":1387235630,"lat":51.5100299,"lon":-0.1351183},{"type":"node","id":1387235646,"lat":51.5092923,"lon":-0.1343567},{"type":"node","id":1387235656,"lat":51.509621,"lon":-0.1346715},{"type":"node","id":1493424643,"lat":51.5119338,"lon":-0.1347919},{"type":"node","id":1493424644,"lat":51.5120523,"lon":-0.1349277},{"type":"node","id":1493458889,"lat":51.509559,"lon":-0.1368618},{"type":"node","id":1567896328,"lat":51.509942,"lon":-0.1351052},{"type":"node","id":1598214606,"lat":51.5080506,"lon":-0.1332786},{"type":"node","id":1660481781,"lat":51.5099234,"lon":-0.1348604},{"type":"node","id":1663003820,"lat":51.5086284,"lon":-0.1328803},{"type":"node","id":1663004118,"lat":51.5095745,"lon":-0.1348504},{"type":"node","id":1663004143,"lat":51.509784,"lon":-0.1361165},{"type":"node","id":1663004159,"lat":51.5099511,"lon":-0.1351407},{"type":"node","id":1663004185,"lat":51.5100745,"lon":-0.1352869},{"type":"node","id":1813223023,"lat":51.5098248,"lon":-0.1348626},{"type":"node","id":1906612405,"lat":51.5138997,"lon":-0.135082},{"type":"node","id":1938450896,"lat":51.5098567,"lon":-0.1363361},{"type":"node","id":1938450898,"lat":51.5098925,"lon":-0.1359517},{"type":"node","id":1938580051,"lat":51.5089192,"lon":-0.1352931},{"type":"node","id":1938580055,"lat":51.5098861,"lon":-0.1348645},{"type":"node","id":1938580057,"lat":51.5100376,"lon":-0.1332198},{"type":"node","id":1938580060,"lat":51.5100283,"lon":-0.1341417},{"type":"node","id":1938580061,"lat":51.5101435,"lon":-0.133475},{"type":"node","id":1938580068,"lat":51.5112996,"lon":-0.1333353},{"type":"node","id":1938580071,"lat":51.5116288,"lon":-0.1327707},{"type":"node","id":2205117781,"lat":51.5100568,"lon":-0.1347126},{"type":"node","id":2205117782,"lat":51.5099859,"lon":-0.1348156},{"type":"node","id":2205117785,"lat":51.5100658,"lon":-0.1345062},{"type":"node","id":2205117787,"lat":51.5098247,"lon":-0.1347623},{"type":"node","id":2205117789,"lat":51.5099607,"lon":-0.1346953},{"type":"node","id":2205117791,"lat":51.5099335,"lon":-0.1347274},{"type":"node","id":2205117795,"lat":51.5100357,"lon":-0.1342525},{"type":"node","id":2205117797,"lat":51.5100242,"lon":-0.134768},{"type":"node","id":2205117798,"lat":51.5098756,"lon":-0.134757},{"type":"node","id":2205117799,"lat":51.5099573,"lon":-0.1348404},{"type":"node","id":2205117800,"lat":51.5097812,"lon":-0.1347625},{"type":"node","id":2205117802,"lat":51.509909,"lon":-0.1347449},{"type":"node","id":2205124732,"lat":51.5101379,"lon":-0.1343734},{"type":"node","id":2205124739,"lat":51.5101968,"lon":-0.1343182},{"type":"node","id":2205124744,"lat":51.5101241,"lon":-0.1349756},{"type":"node","id":2205124752,"lat":51.5102732,"lon":-0.1342768},{"type":"node","id":2205124756,"lat":51.5102386,"lon":-0.1343184},{"type":"node","id":2205124760,"lat":51.5104152,"lon":-0.1341711},{"type":"node","id":2205124764,"lat":51.5101102,"lon":-0.1344153},{"type":"node","id":2205124769,"lat":51.5101669,"lon":-0.1343422},{"type":"node","id":2205127953,"lat":51.5100476,"lon":-0.1343736},{"type":"node","id":2205127955,"lat":51.5101129,"lon":-0.134811},{"type":"node","id":2214952978,"lat":51.5090502,"lon":-0.1366425},{"type":"node","id":2214952980,"lat":51.5088794,"lon":-0.1371522},{"type":"node","id":2214952983,"lat":51.5093852,"lon":-0.135944},{"type":"node","id":2214952984,"lat":51.5093139,"lon":-0.1358784},{"type":"node","id":2214952986,"lat":51.508954,"lon":-0.1372112},{"type":"node","id":2214957479,"lat":51.5097692,"lon":-0.1348723},{"type":"node","id":2214957480,"lat":51.5096386,"lon":-0.1348958},{"type":"node","id":2214957481,"lat":51.5096192,"lon":-0.1349513},{"type":"node","id":2214957482,"lat":51.5097095,"lon":-0.1349316},{"type":"node","id":2214957483,"lat":51.5096644,"lon":-0.1348433},{"type":"node","id":2214957484,"lat":51.5096578,"lon":-0.1350844},{"type":"node","id":2214957485,"lat":51.5094128,"lon":-0.1358527},{"type":"node","id":2214957486,"lat":51.5097362,"lon":-0.1348913},{"type":"node","id":2214957487,"lat":51.5096921,"lon":-0.1348058},{"type":"node","id":2214957488,"lat":51.5093661,"lon":-0.1357269},{"type":"node","id":2244856015,"lat":51.5099735,"lon":-0.135397},{"type":"node","id":2244856019,"lat":51.5100134,"lon":-0.1352433},{"type":"node","id":2349657350,"lat":51.5097796,"lon":-0.1348197},{"type":"node","id":2398143012,"lat":51.5098533,"lon":-0.1364907},{"type":"node","id":2398143015,"lat":51.5099258,"lon":-0.1371669},{"type":"node","id":2398143018,"lat":51.5100128,"lon":-0.1375131},{"type":"node","id":2457223323,"lat":51.5127303,"lon":-0.1334622},{"type":"node","id":2465838818,"lat":51.5107383,"lon":-0.1350526},{"type":"node","id":2465838823,"lat":51.5093106,"lon":-0.1322085},{"type":"node","id":2465838826,"lat":51.5085468,"lon":-0.1318032},{"type":"node","id":2476787983,"lat":51.5123456,"lon":-0.1337649},{"type":"node","id":2484858548,"lat":51.5122739,"lon":-0.1314791},{"type":"node","id":2494798964,"lat":51.5103735,"lon":-0.1322087},{"type":"node","id":2587253639,"lat":51.5118195,"lon":-0.1332828},{"type":"node","id":2704097361,"lat":51.5091641,"lon":-0.1344403},{"type":"node","id":2704097362,"lat":51.509155,"lon":-0.1342372},{"type":"node","id":2704097363,"lat":51.5091097,"lon":-0.1341979},{"type":"node","id":2704097364,"lat":51.5092645,"lon":-0.1341508},{"type":"node","id":2704097365,"lat":51.5092878,"lon":-0.134085},{"type":"node","id":2784682598,"lat":51.512776,"lon":-0.1374389},{"type":"node","id":2784682605,"lat":51.512816,"lon":-0.1373404},{"type":"node","id":2784682609,"lat":51.51284,"lon":-0.1375467},{"type":"node","id":2784682613,"lat":51.5128531,"lon":-0.1375108},{"type":"node","id":2784682616,"lat":51.5129155,"lon":-0.1371058},{"type":"node","id":2791710744,"lat":51.5133321,"lon":-0.1327598},{"type":"node","id":2791735965,"lat":51.5132135,"lon":-0.1340464},{"type":"node","id":2791735968,"lat":51.513247,"lon":-0.1339322},{"type":"node","id":2801490747,"lat":51.5116572,"lon":-0.1360722},{"type":"node","id":2801490748,"lat":51.5115324,"lon":-0.1359174},{"type":"node","id":2919558521,"lat":51.5142738,"lon":-0.1328173},{"type":"node","id":2919558531,"lat":51.5143474,"lon":-0.1325256},{"type":"node","id":2939373031,"lat":51.5096895,"lon":-0.1360531},{"type":"node","id":2939373032,"lat":51.5097652,"lon":-0.1361048},{"type":"node","id":2939373037,"lat":51.5099913,"lon":-0.136268},{"type":"node","id":2939373042,"lat":51.5100794,"lon":-0.1363249},{"type":"node","id":3037800598,"lat":51.5080996,"lon":-0.1323997},{"type":"node","id":3054882696,"lat":51.5137781,"lon":-0.1351913},{"type":"node","id":3054882697,"lat":51.513805,"lon":-0.1351597},{"type":"node","id":3054882698,"lat":51.5138145,"lon":-0.1351265},{"type":"node","id":3054882699,"lat":51.5138334,"lon":-0.135155},{"type":"node","id":3054882700,"lat":51.513877,"lon":-0.1351258},{"type":"node","id":3071350728,"lat":51.5138891,"lon":-0.1351666},{"type":"node","id":3071350729,"lat":51.5143757,"lon":-0.1368247},{"type":"node","id":3075938596,"lat":51.5104689,"lon":-0.1365703},{"type":"node","id":3316828729,"lat":51.5116169,"lon":-0.1331341},{"type":"node","id":3316828730,"lat":51.5133381,"lon":-0.1364945},{"type":"node","id":3357928883,"lat":51.5107313,"lon":-0.1355765},{"type":"node","id":3357929264,"lat":51.5104185,"lon":-0.1358639},{"type":"node","id":3439409118,"lat":51.5080549,"lon":-0.1313374},{"type":"node","id":3714663965,"lat":51.5096891,"lon":-0.132952},{"type":"node","id":3729216343,"lat":51.5111122,"lon":-0.1348762},{"type":"node","id":3729216344,"lat":51.5107524,"lon":-0.1348423},{"type":"node","id":3729216345,"lat":51.5110732,"lon":-0.134899},{"type":"node","id":3729362050,"lat":51.5121479,"lon":-0.132756},{"type":"node","id":3729362052,"lat":51.5121673,"lon":-0.1327131},{"type":"node","id":3731994707,"lat":51.5108628,"lon":-0.1348634},{"type":"node","id":3733175891,"lat":51.51121,"lon":-0.1313588},{"type":"node","id":4052098238,"lat":51.5130915,"lon":-0.1344116},{"type":"node","id":4189248446,"lat":51.5110441,"lon":-0.131852},{"type":"node","id":4189248448,"lat":51.5114338,"lon":-0.1320427},{"type":"node","id":4189248606,"lat":51.5114157,"lon":-0.1320923},{"type":"node","id":4189248608,"lat":51.5118821,"lon":-0.131208},{"type":"node","id":4189397636,"lat":51.5110324,"lon":-0.1318431},{"type":"node","id":4190435881,"lat":51.5090514,"lon":-0.1371491},{"type":"node","id":4190435888,"lat":51.5094712,"lon":-0.1358964},{"type":"node","id":4347844132,"lat":51.5098198,"lon":-0.1362681},{"type":"node","id":4347844135,"lat":51.5098218,"lon":-0.1369305},{"type":"node","id":4347844137,"lat":51.509869,"lon":-0.1371955},{"type":"node","id":4347844149,"lat":51.5097953,"lon":-0.1364907},{"type":"node","id":4347844150,"lat":51.5097988,"lon":-0.1363734},{"type":"node","id":4347844151,"lat":51.5099162,"lon":-0.1374172},{"type":"node","id":4347844153,"lat":51.509951,"lon":-0.1375642},{"type":"node","id":4347880279,"lat":51.509828,"lon":-0.1369654},{"type":"node","id":4347885261,"lat":51.5099086,"lon":-0.1358401},{"type":"node","id":4347899047,"lat":51.510063,"lon":-0.1336458},{"type":"node","id":4347899437,"lat":51.509949,"lon":-0.1349617},{"type":"node","id":4347900573,"lat":51.5101776,"lon":-0.1335049},{"type":"node","id":4347914536,"lat":51.5092864,"lon":-0.1324533},{"type":"node","id":4347914537,"lat":51.5085872,"lon":-0.1318387},{"type":"node","id":4347914538,"lat":51.5093312,"lon":-0.1325002},{"type":"node","id":4349648267,"lat":51.5103859,"lon":-0.1316877},{"type":"node","id":4349648269,"lat":51.5104018,"lon":-0.1316044},{"type":"node","id":4349648270,"lat":51.5103575,"lon":-0.1314371},{"type":"node","id":4349648272,"lat":51.5104494,"lon":-0.131394},{"type":"node","id":4349648275,"lat":51.510535,"lon":-0.131526},{"type":"node","id":4349648276,"lat":51.5104777,"lon":-0.1314953},{"type":"node","id":4638404785,"lat":51.5091867,"lon":-0.1362469},{"type":"node","id":4638404786,"lat":51.509282,"lon":-0.1362401},{"type":"node","id":4638404787,"lat":51.5098557,"lon":-0.1371278},{"type":"node","id":4638404788,"lat":51.5098031,"lon":-0.1367566},{"type":"node","id":4638405889,"lat":51.5098546,"lon":-0.1365343},{"type":"node","id":4638405890,"lat":51.5099198,"lon":-0.1357695},{"type":"node","id":4638405891,"lat":51.5100298,"lon":-0.1354105},{"type":"node","id":4641072775,"lat":51.5088809,"lon":-0.1339993},{"type":"node","id":4641072776,"lat":51.5086522,"lon":-0.1338008},{"type":"node","id":4641072777,"lat":51.5080551,"lon":-0.1329243},{"type":"node","id":4641072778,"lat":51.5095761,"lon":-0.1327452},{"type":"node","id":4641072779,"lat":51.5089672,"lon":-0.1321722},{"type":"node","id":4709189057,"lat":51.5144557,"lon":-0.1364092},{"type":"node","id":4709189303,"lat":51.5140633,"lon":-0.1359275},{"type":"node","id":4709189304,"lat":51.5141464,"lon":-0.1359933},{"type":"node","id":4709189403,"lat":51.5142362,"lon":-0.1371533},{"type":"node","id":4709189404,"lat":51.5142673,"lon":-0.1370383},{"type":"node","id":4709189430,"lat":51.514303,"lon":-0.1369133},{"type":"node","id":4718984414,"lat":51.5134621,"lon":-0.1361944},{"type":"node","id":4718985840,"lat":51.5143728,"lon":-0.1368349},{"type":"node","id":4718985841,"lat":51.5144757,"lon":-0.136426},{"type":"node","id":4721392996,"lat":51.5135104,"lon":-0.1362959},{"type":"node","id":4721392997,"lat":51.5135367,"lon":-0.1363079},{"type":"node","id":4721392998,"lat":51.5135668,"lon":-0.136321},{"type":"node","id":4721393038,"lat":51.5133905,"lon":-0.1365802},{"type":"node","id":4728954671,"lat":51.5128882,"lon":-0.1348271},{"type":"node","id":4728954676,"lat":51.5132442,"lon":-0.1351576},{"type":"node","id":4730390156,"lat":51.512532,"lon":-0.1347703},{"type":"node","id":4730390165,"lat":51.5127361,"lon":-0.1353218},{"type":"node","id":4730390415,"lat":51.5125457,"lon":-0.1347848},{"type":"node","id":4730390416,"lat":51.5125544,"lon":-0.1347954},{"type":"node","id":4730390417,"lat":51.5124791,"lon":-0.1349094},{"type":"node","id":4730390418,"lat":51.5124939,"lon":-0.1349259},{"type":"node","id":4730390419,"lat":51.5125041,"lon":-0.134937},{"type":"node","id":4730390426,"lat":51.5125941,"lon":-0.1346862},{"type":"node","id":4730390427,"lat":51.5125979,"lon":-0.1346654},{"type":"node","id":4730390441,"lat":51.5124505,"lon":-0.1350735},{"type":"node","id":4731045392,"lat":51.5130954,"lon":-0.1356363},{"type":"node","id":4731047323,"lat":51.5123114,"lon":-0.1355291},{"type":"node","id":4731047332,"lat":51.5122779,"lon":-0.1354941},{"type":"node","id":4731047333,"lat":51.5123012,"lon":-0.1354418},{"type":"node","id":4731047368,"lat":51.5120173,"lon":-0.1351782},{"type":"node","id":4731047369,"lat":51.5120414,"lon":-0.1351309},{"type":"node","id":4731047383,"lat":51.5127626,"lon":-0.1359998},{"type":"node","id":4731047386,"lat":51.5127801,"lon":-0.1359512},{"type":"node","id":4731047412,"lat":51.512938,"lon":-0.1361782},{"type":"node","id":4737289787,"lat":51.5120803,"lon":-0.1359608},{"type":"node","id":4739043742,"lat":51.5122612,"lon":-0.1366733},{"type":"node","id":4739043743,"lat":51.5122723,"lon":-0.1366366},{"type":"node","id":4739043795,"lat":51.5120615,"lon":-0.1370971},{"type":"node","id":4739043808,"lat":51.5121133,"lon":-0.1365546},{"type":"node","id":4796603767,"lat":51.511338,"lon":-0.1350929},{"type":"node","id":4888988713,"lat":51.5089832,"lon":-0.1331668},{"type":"node","id":5128437130,"lat":51.5132539,"lon":-0.1330896},{"type":"node","id":5244680778,"lat":51.5108597,"lon":-0.1339002},{"type":"node","id":5244680779,"lat":51.5108362,"lon":-0.133831},{"type":"node","id":5244680780,"lat":51.5107578,"lon":-0.1339515},{"type":"node","id":5244680781,"lat":51.5107482,"lon":-0.1339924},{"type":"node","id":5244680782,"lat":51.5108144,"lon":-0.1338974},{"type":"node","id":5244680783,"lat":51.5109306,"lon":-0.1339878},{"type":"node","id":5244680784,"lat":51.5107933,"lon":-0.1338116},{"type":"node","id":5244680785,"lat":51.5107316,"lon":-0.1338589},{"type":"node","id":5244680786,"lat":51.5108252,"lon":-0.1338416},{"type":"node","id":5364578868,"lat":51.512304,"lon":-0.131149},{"type":"node","id":5364578869,"lat":51.5111594,"lon":-0.1335087},{"type":"node","id":5364578870,"lat":51.5105865,"lon":-0.1340296},{"type":"node","id":5364973542,"lat":51.5144238,"lon":-0.1320252},{"type":"node","id":5392928741,"lat":51.5122136,"lon":-0.1314154},{"type":"node","id":5444256381,"lat":51.5082756,"lon":-0.1325385},{"type":"node","id":5444256382,"lat":51.508822,"lon":-0.1330788},{"type":"node","id":5444256383,"lat":51.5088173,"lon":-0.1330413},{"type":"node","id":5444256384,"lat":51.5092608,"lon":-0.1333982},{"type":"node","id":5444256385,"lat":51.5094179,"lon":-0.1335345},{"type":"node","id":5444256386,"lat":51.5088536,"lon":-0.1330727},{"type":"node","id":5444256387,"lat":51.5092578,"lon":-0.1333513},{"type":"node","id":5444256388,"lat":51.5086183,"lon":-0.1328317},{"type":"node","id":5444256389,"lat":51.5092191,"lon":-0.1333516},{"type":"node","id":5453319326,"lat":51.5129869,"lon":-0.1336633},{"type":"node","id":5453319327,"lat":51.5129995,"lon":-0.1336719},{"type":"node","id":5487714089,"lat":51.5121196,"lon":-0.1342607},{"type":"node","id":5487714090,"lat":51.5120465,"lon":-0.1341785},{"type":"node","id":5487714091,"lat":51.512016,"lon":-0.1341439},{"type":"node","id":5518619474,"lat":51.5122945,"lon":-0.1355733},{"type":"node","id":5597443390,"lat":51.5124824,"lon":-0.133186},{"type":"node","id":5597443391,"lat":51.5125752,"lon":-0.1333351},{"type":"node","id":5597443392,"lat":51.5126645,"lon":-0.1334066},{"type":"node","id":5597443393,"lat":51.5124835,"lon":-0.133096},{"type":"node","id":5597443394,"lat":51.5125982,"lon":-0.1332971},{"type":"node","id":5597443395,"lat":51.5125006,"lon":-0.1331333},{"type":"node","id":5686381085,"lat":51.5104992,"lon":-0.1372054},{"type":"node","id":5686382234,"lat":51.5104798,"lon":-0.1367824},{"type":"node","id":5877193899,"lat":51.5106366,"lon":-0.1339885},{"type":"node","id":5877193900,"lat":51.5108019,"lon":-0.134112},{"type":"node","id":5877193901,"lat":51.5108066,"lon":-0.1341502},{"type":"node","id":5928875936,"lat":51.5139649,"lon":-0.1335039},{"type":"node","id":5928875937,"lat":51.5140621,"lon":-0.1335805},{"type":"node","id":5932856688,"lat":51.512816,"lon":-0.1310876},{"type":"node","id":5932875002,"lat":51.5125274,"lon":-0.1316775},{"type":"node","id":5932875003,"lat":51.5126265,"lon":-0.1315974},{"type":"node","id":5932875004,"lat":51.5125575,"lon":-0.1317011},{"type":"node","id":5932875005,"lat":51.5125938,"lon":-0.1315984},{"type":"node","id":6065422037,"lat":51.5121592,"lon":-0.1343134},{"type":"node","id":6214704101,"lat":51.5101596,"lon":-0.1313166},{"type":"node","id":6250445847,"lat":51.5104838,"lon":-0.1375149},{"type":"node","id":6311528475,"lat":51.512056,"lon":-0.133391},{"type":"node","id":6329186022,"lat":51.5086104,"lon":-0.1336756},{"type":"node","id":108053,"lat":51.5156128,"lon":-0.1393138},{"type":"node","id":255562,"lat":51.5166636,"lon":-0.1411994},{"type":"node","id":255563,"lat":51.5168999,"lon":-0.1398643},{"type":"node","id":255564,"lat":51.516115,"lon":-0.1409667},{"type":"node","id":255566,"lat":51.5154803,"lon":-0.1406986},{"type":"node","id":9521018,"lat":51.5172927,"lon":-0.1378773},{"type":"node","id":9521031,"lat":51.5157473,"lon":-0.1376644},{"type":"node","id":9521033,"lat":51.5186144,"lon":-0.1376571},{"type":"node","id":9521034,"lat":51.5180469,"lon":-0.1391714},{"type":"node","id":9791156,"lat":51.5163986,"lon":-0.1396608},{"type":"node","id":21665915,"lat":51.5150973,"lon":-0.1411507},{"type":"node","id":21665932,"lat":51.5154495,"lon":-0.1392228},{"type":"node","id":21665933,"lat":51.5152413,"lon":-0.139099},{"type":"node","id":21665934,"lat":51.5150941,"lon":-0.1390064},{"type":"node","id":21666003,"lat":51.5148179,"lon":-0.1400451},{"type":"node","id":21666007,"lat":51.5152446,"lon":-0.1384346},{"type":"node","id":25257027,"lat":51.5146267,"lon":-0.1418595},{"type":"node","id":25501773,"lat":51.5146539,"lon":-0.1407797},{"type":"node","id":25501940,"lat":51.5155094,"lon":-0.14043},{"type":"node","id":25502834,"lat":51.5162459,"lon":-0.1402318},{"type":"node","id":25502866,"lat":51.5155587,"lon":-0.1398676},{"type":"node","id":25503799,"lat":51.5177731,"lon":-0.1387032},{"type":"node","id":25503801,"lat":51.5179583,"lon":-0.1383073},{"type":"node","id":271949877,"lat":51.5166666,"lon":-0.1378066},{"type":"node","id":333722494,"lat":51.5180196,"lon":-0.1380988},{"type":"node","id":333722495,"lat":51.5181042,"lon":-0.1381634},{"type":"node","id":333722496,"lat":51.5174894,"lon":-0.137754},{"type":"node","id":333722501,"lat":51.5173169,"lon":-0.1379165},{"type":"node","id":333722502,"lat":51.5174391,"lon":-0.1376927},{"type":"node","id":366905603,"lat":51.5167573,"lon":-0.1406916},{"type":"node","id":366906363,"lat":51.5162073,"lon":-0.140467},{"type":"node","id":383974165,"lat":51.5164901,"lon":-0.1388464},{"type":"node","id":383974166,"lat":51.5156825,"lon":-0.1383362},{"type":"node","id":411545972,"lat":51.5167875,"lon":-0.1398144},{"type":"node","id":496704385,"lat":51.5163235,"lon":-0.1401161},{"type":"node","id":496813005,"lat":51.5163738,"lon":-0.1396074},{"type":"node","id":659654482,"lat":51.5150368,"lon":-0.1409834},{"type":"node","id":659654483,"lat":51.515295,"lon":-0.1410174},{"type":"node","id":659654484,"lat":51.515256,"lon":-0.1411282},{"type":"node","id":691084080,"lat":51.5176012,"lon":-0.1416199},{"type":"node","id":691084099,"lat":51.5154581,"lon":-0.1408378},{"type":"node","id":692129349,"lat":51.5177757,"lon":-0.1402454},{"type":"node","id":799284841,"lat":51.5150473,"lon":-0.1401698},{"type":"node","id":1126076446,"lat":51.5162894,"lon":-0.1401948},{"type":"node","id":1126076484,"lat":51.516368,"lon":-0.1395183},{"type":"node","id":1126101190,"lat":51.516071,"lon":-0.1394875},{"type":"node","id":1617713615,"lat":51.5175039,"lon":-0.1415798},{"type":"node","id":1647906601,"lat":51.5162739,"lon":-0.1395683},{"type":"node","id":1684410076,"lat":51.5158167,"lon":-0.1408368},{"type":"node","id":1684410078,"lat":51.5157897,"lon":-0.1407112},{"type":"node","id":1684410080,"lat":51.5159062,"lon":-0.1400802},{"type":"node","id":1684410081,"lat":51.5159168,"lon":-0.1400074},{"type":"node","id":1684410083,"lat":51.5160087,"lon":-0.1395871},{"type":"node","id":1684410084,"lat":51.5160338,"lon":-0.1395188},{"type":"node","id":1684410105,"lat":51.5171329,"lon":-0.1414137},{"type":"node","id":1684410113,"lat":51.5173741,"lon":-0.1400697},{"type":"node","id":1685938594,"lat":51.5169394,"lon":-0.1383308},{"type":"node","id":1685938595,"lat":51.5169609,"lon":-0.1383436},{"type":"node","id":1685938596,"lat":51.5171853,"lon":-0.1383429},{"type":"node","id":1953893721,"lat":51.5178861,"lon":-0.1385268},{"type":"node","id":2047806084,"lat":51.5154167,"lon":-0.1406739},{"type":"node","id":2047806098,"lat":51.5155298,"lon":-0.1396305},{"type":"node","id":2047806102,"lat":51.5154615,"lon":-0.1402899},{"type":"node","id":2047806107,"lat":51.5153396,"lon":-0.1412322},{"type":"node","id":2047806118,"lat":51.5156425,"lon":-0.1385763},{"type":"node","id":2047806121,"lat":51.5146659,"lon":-0.1417645},{"type":"node","id":2198870628,"lat":51.5152614,"lon":-0.1418031},{"type":"node","id":2198870637,"lat":51.5151464,"lon":-0.1419166},{"type":"node","id":2198870639,"lat":51.5153227,"lon":-0.1418216},{"type":"node","id":2198881213,"lat":51.5154963,"lon":-0.140348},{"type":"node","id":2198881214,"lat":51.5154795,"lon":-0.1403175},{"type":"node","id":2198881216,"lat":51.5155053,"lon":-0.1403796},{"type":"node","id":2198881217,"lat":51.5153754,"lon":-0.1414399},{"type":"node","id":2198881218,"lat":51.5153978,"lon":-0.1408108},{"type":"node","id":2409829432,"lat":51.5167382,"lon":-0.1412316},{"type":"node","id":2409829435,"lat":51.5166881,"lon":-0.1410702},{"type":"node","id":2409829444,"lat":51.5166001,"lon":-0.1411725},{"type":"node","id":2414133927,"lat":51.5152704,"lon":-0.141733},{"type":"node","id":2414133930,"lat":51.5155581,"lon":-0.1392874},{"type":"node","id":2467533676,"lat":51.5155993,"lon":-0.1407549},{"type":"node","id":2480891616,"lat":51.5148462,"lon":-0.1418249},{"type":"node","id":3054882705,"lat":51.517262,"lon":-0.1379455},{"type":"node","id":4305326856,"lat":51.5150477,"lon":-0.1409099},{"type":"node","id":4305326859,"lat":51.5152392,"lon":-0.1411347},{"type":"node","id":4305326861,"lat":51.5152582,"lon":-0.1410037},{"type":"node","id":4305326862,"lat":51.515325,"lon":-0.1413637},{"type":"node","id":4305326863,"lat":51.5154842,"lon":-0.1417951},{"type":"node","id":4305326864,"lat":51.5153905,"lon":-0.1407691},{"type":"node","id":4305334389,"lat":51.515246,"lon":-0.1411384},{"type":"node","id":4305334390,"lat":51.5152502,"lon":-0.1411119},{"type":"node","id":4305334391,"lat":51.5152579,"lon":-0.141116},{"type":"node","id":4305343211,"lat":51.5152969,"lon":-0.1413464},{"type":"node","id":4581297847,"lat":51.5154965,"lon":-0.1399498},{"type":"node","id":4581297848,"lat":51.5156523,"lon":-0.1387594},{"type":"node","id":4684517940,"lat":51.5154946,"lon":-0.1392481},{"type":"node","id":4684517980,"lat":51.5153922,"lon":-0.139199},{"type":"node","id":4684517986,"lat":51.5150572,"lon":-0.1393065},{"type":"node","id":4684517987,"lat":51.5150378,"lon":-0.139379},{"type":"node","id":4684519080,"lat":51.5149935,"lon":-0.1392771},{"type":"node","id":4684519082,"lat":51.5150698,"lon":-0.1389892},{"type":"node","id":4684519083,"lat":51.5148668,"lon":-0.1399568},{"type":"node","id":4684519084,"lat":51.5150285,"lon":-0.1393726},{"type":"node","id":4684519085,"lat":51.5150485,"lon":-0.1393005},{"type":"node","id":4684520591,"lat":51.5148308,"lon":-0.1398864},{"type":"node","id":4684520594,"lat":51.5148377,"lon":-0.1400593},{"type":"node","id":4684520597,"lat":51.5151163,"lon":-0.1390204},{"type":"node","id":4684520629,"lat":51.5150176,"lon":-0.139294},{"type":"node","id":4684520630,"lat":51.5149748,"lon":-0.1392639},{"type":"node","id":4684520653,"lat":51.5148098,"lon":-0.1398714},{"type":"node","id":4684520654,"lat":51.5148554,"lon":-0.139904},{"type":"node","id":4684520664,"lat":51.5148456,"lon":-0.1399409},{"type":"node","id":4684520665,"lat":51.5150209,"lon":-0.1392815},{"type":"node","id":4684520666,"lat":51.5150016,"lon":-0.139354},{"type":"node","id":4684520667,"lat":51.5147624,"lon":-0.1387413},{"type":"node","id":4684520668,"lat":51.5147699,"lon":-0.1387141},{"type":"node","id":4684520669,"lat":51.5147554,"lon":-0.1387662},{"type":"node","id":4684520702,"lat":51.5149967,"lon":-0.1401732},{"type":"node","id":4684520703,"lat":51.5149931,"lon":-0.1401902},{"type":"node","id":4684520713,"lat":51.5149887,"lon":-0.1402194},{"type":"node","id":4684520717,"lat":51.5148803,"lon":-0.139967},{"type":"node","id":4684520827,"lat":51.5150193,"lon":-0.1401894},{"type":"node","id":4684520828,"lat":51.5150306,"lon":-0.1401881},{"type":"node","id":4696358371,"lat":51.5157081,"lon":-0.137644},{"type":"node","id":4702603652,"lat":51.5146708,"lon":-0.1378011},{"type":"node","id":4702603653,"lat":51.5147322,"lon":-0.1378403},{"type":"node","id":4702603654,"lat":51.5146978,"lon":-0.1376989},{"type":"node","id":4702603655,"lat":51.5147606,"lon":-0.137741},{"type":"node","id":5216516549,"lat":51.5173662,"lon":-0.1378295},{"type":"node","id":5216516550,"lat":51.517391,"lon":-0.1377843},{"type":"node","id":107316,"lat":51.5188864,"lon":-0.1369442},{"type":"node","id":107318,"lat":51.5200969,"lon":-0.136436},{"type":"node","id":107319,"lat":51.5193831,"lon":-0.1356907},{"type":"node","id":107320,"lat":51.5201219,"lon":-0.1338055},{"type":"node","id":107322,"lat":51.5162463,"lon":-0.1328665},{"type":"node","id":107324,"lat":51.5156469,"lon":-0.1324402},{"type":"node","id":107325,"lat":51.5154946,"lon":-0.1330253},{"type":"node","id":107326,"lat":51.5151306,"lon":-0.1327802},{"type":"node","id":107327,"lat":51.5147635,"lon":-0.132513},{"type":"node","id":107328,"lat":51.5148198,"lon":-0.13229},{"type":"node","id":107848,"lat":51.5189924,"lon":-0.1312083},{"type":"node","id":108045,"lat":51.520841,"lon":-0.1345358},{"type":"node","id":108062,"lat":51.5159337,"lon":-0.1358688},{"type":"node","id":194250,"lat":51.5184705,"lon":-0.1347409},{"type":"node","id":255576,"lat":51.5149786,"lon":-0.1333557},{"type":"node","id":255578,"lat":51.5171097,"lon":-0.1353318},{"type":"node","id":255579,"lat":51.5185315,"lon":-0.1362808},{"type":"node","id":9521006,"lat":51.5167429,"lon":-0.1373585},{"type":"node","id":9521011,"lat":51.5170133,"lon":-0.1359039},{"type":"node","id":9521022,"lat":51.5153055,"lon":-0.135406},{"type":"node","id":9521025,"lat":51.5148315,"lon":-0.1371919},{"type":"node","id":9521029,"lat":51.515842,"lon":-0.1367629},{"type":"node","id":9521035,"lat":51.5160593,"lon":-0.1346358},{"type":"node","id":9791152,"lat":51.5146784,"lon":-0.1358204},{"type":"node","id":9791153,"lat":51.5149112,"lon":-0.1350854},{"type":"node","id":9791484,"lat":51.5187667,"lon":-0.1324378},{"type":"node","id":9791485,"lat":51.5191536,"lon":-0.132826},{"type":"node","id":9791487,"lat":51.5198182,"lon":-0.134583},{"type":"node","id":9791489,"lat":51.520533,"lon":-0.1353561},{"type":"node","id":11863145,"lat":51.5203075,"lon":-0.1340055},{"type":"node","id":11863146,"lat":51.5192314,"lon":-0.1328953},{"type":"node","id":14791185,"lat":51.517445,"lon":-0.1311415},{"type":"node","id":14791188,"lat":51.5185414,"lon":-0.1322033},{"type":"node","id":14791189,"lat":51.5187323,"lon":-0.1317427},{"type":"node","id":21554349,"lat":51.5150678,"lon":-0.1313617},{"type":"node","id":21665026,"lat":51.5160966,"lon":-0.1312572},{"type":"node","id":21665319,"lat":51.5148179,"lon":-0.1339785},{"type":"node","id":21665586,"lat":51.5161312,"lon":-0.134048},{"type":"node","id":21665937,"lat":51.5154249,"lon":-0.1349677},{"type":"node","id":21665938,"lat":51.515588,"lon":-0.1343944},{"type":"node","id":21665945,"lat":51.515055,"lon":-0.1341225},{"type":"node","id":21665946,"lat":51.5160104,"lon":-0.1345994},{"type":"node","id":25417426,"lat":51.5162644,"lon":-0.1326418},{"type":"node","id":25417427,"lat":51.5164755,"lon":-0.132648},{"type":"node","id":25417428,"lat":51.5166644,"lon":-0.1325303},{"type":"node","id":25417429,"lat":51.5168755,"lon":-0.1317148},{"type":"node","id":25417430,"lat":51.516846,"lon":-0.1324109},{"type":"node","id":25417431,"lat":51.5171426,"lon":-0.1317015},{"type":"node","id":25471909,"lat":51.5204601,"lon":-0.1336491},{"type":"node","id":25471910,"lat":51.5205431,"lon":-0.1337081},{"type":"node","id":25471911,"lat":51.5206555,"lon":-0.1336812},{"type":"node","id":25471913,"lat":51.5207258,"lon":-0.1336063},{"type":"node","id":25471914,"lat":51.5207844,"lon":-0.1334892},{"type":"node","id":25471915,"lat":51.5208259,"lon":-0.1333377},{"type":"node","id":25471917,"lat":51.5208318,"lon":-0.1332041},{"type":"node","id":25471918,"lat":51.5208101,"lon":-0.1331044},{"type":"node","id":25471919,"lat":51.5207553,"lon":-0.132969},{"type":"node","id":25472045,"lat":51.5194266,"lon":-0.132478},{"type":"node","id":25472049,"lat":51.5193903,"lon":-0.1324172},{"type":"node","id":25472050,"lat":51.5193621,"lon":-0.1322923},{"type":"node","id":25472051,"lat":51.5193632,"lon":-0.1322253},{"type":"node","id":25472052,"lat":51.5193723,"lon":-0.1321139},{"type":"node","id":25472053,"lat":51.5193997,"lon":-0.1320281},{"type":"node","id":25472054,"lat":51.5194546,"lon":-0.1319305},{"type":"node","id":25472055,"lat":51.5194989,"lon":-0.1318921},{"type":"node","id":25472056,"lat":51.5196114,"lon":-0.1318788},{"type":"node","id":25472057,"lat":51.5196692,"lon":-0.1319236},{"type":"node","id":25495441,"lat":51.5180299,"lon":-0.1342984},{"type":"node","id":25500120,"lat":51.5176741,"lon":-0.1335178},{"type":"node","id":25500123,"lat":51.5174502,"lon":-0.1332031},{"type":"node","id":25500126,"lat":51.5179029,"lon":-0.1324616},{"type":"node","id":25500709,"lat":51.5181391,"lon":-0.1318047},{"type":"node","id":25500758,"lat":51.5189168,"lon":-0.1352038},{"type":"node","id":25500900,"lat":51.5191104,"lon":-0.1354059},{"type":"node","id":25500997,"lat":51.5149257,"lon":-0.1319049},{"type":"node","id":25501005,"lat":51.5198325,"lon":-0.1361599},{"type":"node","id":25501007,"lat":51.5202718,"lon":-0.1350765},{"type":"node","id":25501016,"lat":51.5148023,"lon":-0.1350013},{"type":"node","id":25501064,"lat":51.514647,"lon":-0.1331301},{"type":"node","id":25502544,"lat":51.5147917,"lon":-0.1354644},{"type":"node","id":25503710,"lat":51.5188938,"lon":-0.1336861},{"type":"node","id":25505558,"lat":51.5191427,"lon":-0.1362841},{"type":"node","id":25505566,"lat":51.5186843,"lon":-0.1357833},{"type":"node","id":25530376,"lat":51.5160191,"lon":-0.1319926},{"type":"node","id":25703180,"lat":51.519543,"lon":-0.1342892},{"type":"node","id":30939233,"lat":51.5181376,"lon":-0.1327797},{"type":"node","id":30939234,"lat":51.5169254,"lon":-0.1314992},{"type":"node","id":30939235,"lat":51.5171896,"lon":-0.13174},{"type":"node","id":82710879,"lat":51.5179145,"lon":-0.1358668},{"type":"node","id":82710880,"lat":51.5180276,"lon":-0.1353662},{"type":"node","id":82710881,"lat":51.5175964,"lon":-0.1349251},{"type":"node","id":271949991,"lat":51.5157854,"lon":-0.1373092},{"type":"node","id":332040899,"lat":51.5189538,"lon":-0.1335166},{"type":"node","id":333650005,"lat":51.5198216,"lon":-0.1371227},{"type":"node","id":333650013,"lat":51.5193031,"lon":-0.1365792},{"type":"node","id":333650015,"lat":51.5193746,"lon":-0.1363954},{"type":"node","id":333650017,"lat":51.5191801,"lon":-0.1361983},{"type":"node","id":333670393,"lat":51.5205961,"lon":-0.1333229},{"type":"node","id":333670394,"lat":51.5195556,"lon":-0.1321839},{"type":"node","id":333670399,"lat":51.5193542,"lon":-0.1333152},{"type":"node","id":333670400,"lat":51.5191386,"lon":-0.1328128},{"type":"node","id":333670402,"lat":51.5182756,"lon":-0.1336438},{"type":"node","id":333670409,"lat":51.5203819,"lon":-0.134083},{"type":"node","id":333670411,"lat":51.5189631,"lon":-0.1336836},{"type":"node","id":333670412,"lat":51.518882,"lon":-0.1336964},{"type":"node","id":333719180,"lat":51.5193132,"lon":-0.1328086},{"type":"node","id":333719185,"lat":51.5163834,"lon":-0.1354786},{"type":"node","id":333722488,"lat":51.5171795,"lon":-0.1353772},{"type":"node","id":333722490,"lat":51.5184675,"lon":-0.1368254},{"type":"node","id":333722491,"lat":51.5169781,"lon":-0.1365649},{"type":"node","id":333722497,"lat":51.5174322,"lon":-0.1375942},{"type":"node","id":333722498,"lat":51.5169595,"lon":-0.1371493},{"type":"node","id":333722500,"lat":51.516799,"lon":-0.1370643},{"type":"node","id":367735523,"lat":51.5183857,"lon":-0.1346563},{"type":"node","id":380494805,"lat":51.5170717,"lon":-0.1322998},{"type":"node","id":380494808,"lat":51.5174366,"lon":-0.1318633},{"type":"node","id":380496980,"lat":51.5179723,"lon":-0.1334284},{"type":"node","id":385562683,"lat":51.5161701,"lon":-0.1337251},{"type":"node","id":496812960,"lat":51.5169098,"lon":-0.1365059},{"type":"node","id":565545547,"lat":51.5171918,"lon":-0.1326633},{"type":"node","id":699729408,"lat":51.5196342,"lon":-0.1333214},{"type":"node","id":699729426,"lat":51.5194586,"lon":-0.1337724},{"type":"node","id":734320238,"lat":51.5151179,"lon":-0.1334628},{"type":"node","id":735142648,"lat":51.5171562,"lon":-0.1360122},{"type":"node","id":881889762,"lat":51.5195739,"lon":-0.1312574},{"type":"node","id":881889766,"lat":51.5205651,"lon":-0.131428},{"type":"node","id":881889767,"lat":51.5203859,"lon":-0.1318117},{"type":"node","id":918548280,"lat":51.5187119,"lon":-0.1370091},{"type":"node","id":983839058,"lat":51.5198639,"lon":-0.1344744},{"type":"node","id":1089052858,"lat":51.5153072,"lon":-0.1335846},{"type":"node","id":1089052862,"lat":51.5155821,"lon":-0.1344252},{"type":"node","id":1089052872,"lat":51.5150925,"lon":-0.1341369},{"type":"node","id":1180910960,"lat":51.5158882,"lon":-0.1311246},{"type":"node","id":1448835614,"lat":51.5195419,"lon":-0.1318736},{"type":"node","id":1448835615,"lat":51.5193686,"lon":-0.132335},{"type":"node","id":1448889416,"lat":51.5203347,"lon":-0.1339312},{"type":"node","id":1448894892,"lat":51.5200812,"lon":-0.1339127},{"type":"node","id":1448896143,"lat":51.5205433,"lon":-0.133264},{"type":"node","id":1448896332,"lat":51.5211147,"lon":-0.13202},{"type":"node","id":1448901559,"lat":51.520529,"lon":-0.1315064},{"type":"node","id":1484198280,"lat":51.5173275,"lon":-0.1330185},{"type":"node","id":1484198385,"lat":51.5176977,"lon":-0.1335867},{"type":"node","id":1484205742,"lat":51.516711,"lon":-0.1323789},{"type":"node","id":1484205753,"lat":51.516819,"lon":-0.1324735},{"type":"node","id":1484205770,"lat":51.5165546,"lon":-0.1326165},{"type":"node","id":1484205791,"lat":51.5167686,"lon":-0.1325114},{"type":"node","id":1490441420,"lat":51.5179702,"lon":-0.1322505},{"type":"node","id":1490441421,"lat":51.5180573,"lon":-0.132013},{"type":"node","id":1685751682,"lat":51.516677,"lon":-0.1332334},{"type":"node","id":1685751694,"lat":51.5173934,"lon":-0.1338415},{"type":"node","id":1685751696,"lat":51.5175221,"lon":-0.1339511},{"type":"node","id":1685751700,"lat":51.5177069,"lon":-0.1341511},{"type":"node","id":1685751707,"lat":51.517873,"lon":-0.1346323},{"type":"node","id":1685751711,"lat":51.5178796,"lon":-0.1345847},{"type":"node","id":1685751723,"lat":51.5179385,"lon":-0.1344008},{"type":"node","id":1685751731,"lat":51.5181898,"lon":-0.1351116},{"type":"node","id":1685938592,"lat":51.5169435,"lon":-0.1375167},{"type":"node","id":1694551553,"lat":51.5150241,"lon":-0.1315154},{"type":"node","id":1694551556,"lat":51.5154426,"lon":-0.1316153},{"type":"node","id":1694551558,"lat":51.5157896,"lon":-0.1318682},{"type":"node","id":1716099103,"lat":51.5158244,"lon":-0.1369211},{"type":"node","id":1716099125,"lat":51.515922,"lon":-0.1368297},{"type":"node","id":1785339688,"lat":51.5162967,"lon":-0.1326424},{"type":"node","id":1795369601,"lat":51.5169628,"lon":-0.1311972},{"type":"node","id":1795369610,"lat":51.5169766,"lon":-0.1311219},{"type":"node","id":1906612406,"lat":51.5150882,"lon":-0.1361559},{"type":"node","id":1938953378,"lat":51.5162088,"lon":-0.1332832},{"type":"node","id":2043878233,"lat":51.5153539,"lon":-0.1375107},{"type":"node","id":2249185407,"lat":51.5160552,"lon":-0.1346649},{"type":"node","id":2356344242,"lat":51.5186436,"lon":-0.1358259},{"type":"node","id":2398311994,"lat":51.5193926,"lon":-0.1341318},{"type":"node","id":2398312017,"lat":51.5193486,"lon":-0.1347844},{"type":"node","id":2398312023,"lat":51.5194852,"lon":-0.1342255},{"type":"node","id":2417297378,"lat":51.5162554,"lon":-0.1327468},{"type":"node","id":2417297527,"lat":51.5162258,"lon":-0.1330818},{"type":"node","id":2417297528,"lat":51.516231,"lon":-0.1330209},{"type":"node","id":2417297529,"lat":51.5161654,"lon":-0.1328101},{"type":"node","id":2417297530,"lat":51.5162622,"lon":-0.1326713},{"type":"node","id":2417297531,"lat":51.5163473,"lon":-0.1329612},{"type":"node","id":2417297532,"lat":51.5163147,"lon":-0.1329338},{"type":"node","id":2417298649,"lat":51.5161186,"lon":-0.1346877},{"type":"node","id":2417298650,"lat":51.5160434,"lon":-0.1347872},{"type":"node","id":2417298651,"lat":51.5160712,"lon":-0.1345223},{"type":"node","id":2417299495,"lat":51.5158511,"lon":-0.1366703},{"type":"node","id":2425140532,"lat":51.5158224,"lon":-0.1358035},{"type":"node","id":2425140533,"lat":51.5159473,"lon":-0.1357102},{"type":"node","id":2425140534,"lat":51.515912,"lon":-0.1360763},{"type":"node","id":2425140535,"lat":51.5160656,"lon":-0.1359638},{"type":"node","id":2425141027,"lat":51.5160076,"lon":-0.1359221},{"type":"node","id":2425141028,"lat":51.5159423,"lon":-0.1357688},{"type":"node","id":2425141030,"lat":51.5159185,"lon":-0.1360204},{"type":"node","id":2425141032,"lat":51.515868,"lon":-0.1358336},{"type":"node","id":2440641460,"lat":51.5194328,"lon":-0.1357426},{"type":"node","id":2440641461,"lat":51.5194156,"lon":-0.135599},{"type":"node","id":2440641462,"lat":51.5192997,"lon":-0.1358965},{"type":"node","id":2440641463,"lat":51.5193226,"lon":-0.13584},{"type":"node","id":2440641464,"lat":51.5193018,"lon":-0.1356058},{"type":"node","id":2440641465,"lat":51.5194605,"lon":-0.1357715},{"type":"node","id":2440641466,"lat":51.5193233,"lon":-0.1356282},{"type":"node","id":2440643030,"lat":51.5188371,"lon":-0.1368195},{"type":"node","id":2440643032,"lat":51.5189518,"lon":-0.1367629},{"type":"node","id":2440643033,"lat":51.5189318,"lon":-0.1368164},{"type":"node","id":2440643034,"lat":51.5188174,"lon":-0.1367868},{"type":"node","id":2440643035,"lat":51.5188241,"lon":-0.136798},{"type":"node","id":2440651558,"lat":51.5185535,"lon":-0.1375898},{"type":"node","id":2453179640,"lat":51.5147537,"lon":-0.1358806},{"type":"node","id":2494845029,"lat":51.5156077,"lon":-0.1376098},{"type":"node","id":2494845032,"lat":51.5151188,"lon":-0.1361817},{"type":"node","id":2520452050,"lat":51.5177417,"lon":-0.1341212},{"type":"node","id":2520452052,"lat":51.5177908,"lon":-0.1339883},{"type":"node","id":2522315147,"lat":51.5178225,"lon":-0.1351099},{"type":"node","id":2522315148,"lat":51.5179337,"lon":-0.1357851},{"type":"node","id":2522315149,"lat":51.517969,"lon":-0.1356466},{"type":"node","id":2522315150,"lat":51.5180072,"lon":-0.1354732},{"type":"node","id":2522315151,"lat":51.5180183,"lon":-0.1354164},{"type":"node","id":2833445978,"lat":51.5145988,"lon":-0.1311342},{"type":"node","id":2982330087,"lat":51.5176542,"lon":-0.1313421},{"type":"node","id":2982330111,"lat":51.5184593,"lon":-0.132117},{"type":"node","id":2982330114,"lat":51.5185726,"lon":-0.1321273},{"type":"node","id":2985253988,"lat":51.5205968,"lon":-0.1337131},{"type":"node","id":2985253995,"lat":51.5207513,"lon":-0.1344513},{"type":"node","id":3054882702,"lat":51.5157459,"lon":-0.1366814},{"type":"node","id":3054882703,"lat":51.5168735,"lon":-0.1374497},{"type":"node","id":3054882704,"lat":51.5169244,"lon":-0.1375842},{"type":"node","id":3071350730,"lat":51.5147881,"lon":-0.1371633},{"type":"node","id":3071350731,"lat":51.5150991,"lon":-0.1361181},{"type":"node","id":3071350732,"lat":51.5151081,"lon":-0.1362193},{"type":"node","id":3519133221,"lat":51.5180033,"lon":-0.1316751},{"type":"node","id":3524428937,"lat":51.5186026,"lon":-0.1369166},{"type":"node","id":3524428945,"lat":51.5187685,"lon":-0.1370591},{"type":"node","id":3720552874,"lat":51.5151535,"lon":-0.1320959},{"type":"node","id":3720610535,"lat":51.5180461,"lon":-0.1311138},{"type":"node","id":3822812081,"lat":51.5168625,"lon":-0.1370983},{"type":"node","id":3822896566,"lat":51.5158855,"lon":-0.1373416},{"type":"node","id":3822896567,"lat":51.5159715,"lon":-0.1373793},{"type":"node","id":4222961755,"lat":51.5194244,"lon":-0.1319734},{"type":"node","id":4222962185,"lat":51.5197999,"lon":-0.131628},{"type":"node","id":4222962219,"lat":51.5198654,"lon":-0.1317037},{"type":"node","id":4231049408,"lat":51.5189693,"lon":-0.1312627},{"type":"node","id":4243601481,"lat":51.5179495,"lon":-0.1323813},{"type":"node","id":4243601487,"lat":51.517993,"lon":-0.1323342},{"type":"node","id":4243601491,"lat":51.5180166,"lon":-0.1322968},{"type":"node","id":4243601502,"lat":51.5181081,"lon":-0.132062},{"type":"node","id":4243601512,"lat":51.5181918,"lon":-0.131856},{"type":"node","id":4247304446,"lat":51.5185124,"lon":-0.1330434},{"type":"node","id":4249404263,"lat":51.5185878,"lon":-0.1344421},{"type":"node","id":4249404266,"lat":51.5186099,"lon":-0.1344063},{"type":"node","id":4249404271,"lat":51.5187107,"lon":-0.1341518},{"type":"node","id":4249404273,"lat":51.5187381,"lon":-0.1341801},{"type":"node","id":4249404278,"lat":51.5187645,"lon":-0.1342087},{"type":"node","id":4249404283,"lat":51.5188167,"lon":-0.1338861},{"type":"node","id":4249404285,"lat":51.5188248,"lon":-0.1334078},{"type":"node","id":4249404286,"lat":51.5188248,"lon":-0.1338445},{"type":"node","id":4249404295,"lat":51.5188938,"lon":-0.1334743},{"type":"node","id":4249404300,"lat":51.5189111,"lon":-0.1334927},{"type":"node","id":4249404302,"lat":51.5189134,"lon":-0.1336708},{"type":"node","id":4249404308,"lat":51.5189309,"lon":-0.1335717},{"type":"node","id":4249404309,"lat":51.5189329,"lon":-0.1335159},{"type":"node","id":4249404310,"lat":51.5189329,"lon":-0.1336671},{"type":"node","id":4252666500,"lat":51.5195727,"lon":-0.1342203},{"type":"node","id":4252666501,"lat":51.5195743,"lon":-0.1343227},{"type":"node","id":4252666506,"lat":51.5196009,"lon":-0.1342532},{"type":"node","id":4299259647,"lat":51.5185038,"lon":-0.1362574},{"type":"node","id":4299259661,"lat":51.5185589,"lon":-0.1363144},{"type":"node","id":4301399715,"lat":51.5167257,"lon":-0.1357112},{"type":"node","id":4301399719,"lat":51.5167425,"lon":-0.1356516},{"type":"node","id":4301399731,"lat":51.5168102,"lon":-0.1357686},{"type":"node","id":4301399740,"lat":51.5169448,"lon":-0.1358601},{"type":"node","id":4301399905,"lat":51.5170722,"lon":-0.1359602},{"type":"node","id":4303454975,"lat":51.5173597,"lon":-0.1361889},{"type":"node","id":4303454977,"lat":51.5173728,"lon":-0.1361415},{"type":"node","id":4303454988,"lat":51.517455,"lon":-0.1362511},{"type":"node","id":4303455889,"lat":51.517468,"lon":-0.136201},{"type":"node","id":4303455913,"lat":51.5176746,"lon":-0.1363943},{"type":"node","id":4303455916,"lat":51.5176892,"lon":-0.1363392},{"type":"node","id":4519038134,"lat":51.5194307,"lon":-0.1331043},{"type":"node","id":4581297849,"lat":51.5160081,"lon":-0.1351259},{"type":"node","id":4581297850,"lat":51.5159774,"lon":-0.1354208},{"type":"node","id":4581297851,"lat":51.5162907,"lon":-0.132307},{"type":"node","id":4581297852,"lat":51.5163384,"lon":-0.1316984},{"type":"node","id":4694456248,"lat":51.5155673,"lon":-0.1376052},{"type":"node","id":4694456249,"lat":51.5155923,"lon":-0.1376039},{"type":"node","id":4702603699,"lat":51.515211,"lon":-0.1369976},{"type":"node","id":4702603700,"lat":51.5151442,"lon":-0.1369534},{"type":"node","id":4702603701,"lat":51.5149978,"lon":-0.1368589},{"type":"node","id":4702603702,"lat":51.5149376,"lon":-0.1368188},{"type":"node","id":4710971306,"lat":51.5158016,"lon":-0.1357728},{"type":"node","id":4712556827,"lat":51.5155853,"lon":-0.1344086},{"type":"node","id":4712556846,"lat":51.5155534,"lon":-0.1345125},{"type":"node","id":4712556847,"lat":51.5154849,"lon":-0.1346307},{"type":"node","id":4712556848,"lat":51.5154001,"lon":-0.1350545},{"type":"node","id":4712556854,"lat":51.5153661,"lon":-0.1350285},{"type":"node","id":4712556857,"lat":51.5155207,"lon":-0.1346592},{"type":"node","id":4712556876,"lat":51.5154709,"lon":-0.1350048},{"type":"node","id":4712556877,"lat":51.5153172,"lon":-0.1353446},{"type":"node","id":4712556912,"lat":51.514692,"lon":-0.1353864},{"type":"node","id":4712556913,"lat":51.5147512,"lon":-0.1354344},{"type":"node","id":4712556962,"lat":51.5149471,"lon":-0.1343191},{"type":"node","id":4712556963,"lat":51.5149892,"lon":-0.1343507},{"type":"node","id":4712556977,"lat":51.5151812,"lon":-0.1341839},{"type":"node","id":4718985842,"lat":51.5146681,"lon":-0.1358547},{"type":"node","id":4851370381,"lat":51.5163138,"lon":-0.1320124},{"type":"node","id":5150471750,"lat":51.5193848,"lon":-0.1332317},{"type":"node","id":5239543922,"lat":51.5161986,"lon":-0.1328264},{"type":"node","id":5239556145,"lat":51.5192865,"lon":-0.1328583},{"type":"node","id":5239556146,"lat":51.5191804,"lon":-0.1329464},{"type":"node","id":5239556147,"lat":51.5191627,"lon":-0.132981},{"type":"node","id":5239556407,"lat":51.5192053,"lon":-0.1329167},{"type":"node","id":5239570371,"lat":51.5187198,"lon":-0.1317824},{"type":"node","id":5337345854,"lat":51.5172918,"lon":-0.1337553},{"type":"node","id":5337345855,"lat":51.5169221,"lon":-0.1352077},{"type":"node","id":5434424322,"lat":51.5173765,"lon":-0.1310762},{"type":"node","id":5440856681,"lat":51.519005,"lon":-0.1326796},{"type":"node","id":5461199399,"lat":51.5162233,"lon":-0.1324031},{"type":"node","id":5461199401,"lat":51.5162188,"lon":-0.1324589},{"type":"node","id":5461199403,"lat":51.516217,"lon":-0.1324804},{"type":"node","id":5461199404,"lat":51.5162135,"lon":-0.132524},{"type":"node","id":5461200037,"lat":51.5163021,"lon":-0.1313512},{"type":"node","id":5461200038,"lat":51.5162959,"lon":-0.1314366},{"type":"node","id":5461200039,"lat":51.5162886,"lon":-0.1315333},{"type":"node","id":5461200040,"lat":51.5162812,"lon":-0.1316191},{"type":"node","id":5612180569,"lat":51.5203421,"lon":-0.1340415},{"type":"node","id":5612180570,"lat":51.5200422,"lon":-0.1337264},{"type":"node","id":5612180571,"lat":51.5200453,"lon":-0.1340056},{"type":"node","id":5758343877,"lat":51.5166005,"lon":-0.135002},{"type":"node","id":5768795651,"lat":51.519658,"lon":-0.1323039},{"type":"node","id":5768795652,"lat":51.5195969,"lon":-0.1322323},{"type":"node","id":5768795953,"lat":51.5205114,"lon":-0.1332285},{"type":"node","id":5768795954,"lat":51.5205055,"lon":-0.1331801},{"type":"node","id":5768795955,"lat":51.5196721,"lon":-0.132359},{"type":"node","id":5768795956,"lat":51.5204812,"lon":-0.1332412},{"type":"node","id":5768795957,"lat":51.5196134,"lon":-0.1322516},{"type":"node","id":5768795958,"lat":51.5196923,"lon":-0.1323101},{"type":"node","id":5811143789,"lat":51.5200657,"lon":-0.1337478},{"type":"node","id":5811145644,"lat":51.5187283,"lon":-0.1325389},{"type":"node","id":5877163201,"lat":51.5160931,"lon":-0.1327431},{"type":"node","id":5877163202,"lat":51.5161761,"lon":-0.1327597},{"type":"node","id":5877163203,"lat":51.5161222,"lon":-0.1327192},{"type":"node","id":5877163208,"lat":51.516116,"lon":-0.1327735},{"type":"node","id":5935822696,"lat":51.5160407,"lon":-0.1339999},{"type":"node","id":6032776008,"lat":51.5186998,"lon":-0.1317296},{"type":"node","id":6032776010,"lat":51.5187472,"lon":-0.1317398},{"type":"node","id":6032776011,"lat":51.5187575,"lon":-0.1317296},{"type":"node","id":6032776012,"lat":51.5187177,"lon":-0.1317395},{"type":"node","id":6089225181,"lat":51.5169068,"lon":-0.135198},{"type":"node","id":6089552096,"lat":51.5175016,"lon":-0.1344626},{"type":"node","id":6089552097,"lat":51.5177052,"lon":-0.134672},{"type":"node","id":6089552098,"lat":51.5177303,"lon":-0.1346597},{"type":"node","id":6089552099,"lat":51.5177909,"lon":-0.1346409},{"type":"node","id":6264884980,"lat":51.5155914,"lon":-0.1338544},{"type":"node","id":6264884981,"lat":51.5154824,"lon":-0.134346},{"type":"node","id":6264884982,"lat":51.5155726,"lon":-0.13376},{"type":"node","id":6279571026,"lat":51.5179091,"lon":-0.1343745},{"type":"node","id":6279571027,"lat":51.5179875,"lon":-0.134354},{"type":"node","id":6313624585,"lat":51.5153235,"lon":-0.1335173},{"type":"node","id":6313624586,"lat":51.5153598,"lon":-0.1333737},{"type":"node","id":6313625295,"lat":51.5155932,"lon":-0.1337892},{"type":"node","id":6313625296,"lat":51.5156303,"lon":-0.1337915},{"type":"node","id":6313627510,"lat":51.5153215,"lon":-0.1335932},{"type":"node","id":6315615726,"lat":51.5151407,"lon":-0.1320861},{"type":"node","id":6355365934,"lat":51.5193955,"lon":-0.1325131},{"type":"node","id":6355365935,"lat":51.5186776,"lon":-0.1326671},{"type":"node","id":6355365936,"lat":51.5192871,"lon":-0.1327817},{"type":"node","id":6355365944,"lat":51.5186515,"lon":-0.1327084},{"type":"node","id":6355365946,"lat":51.5194223,"lon":-0.1325304},{"type":"node","id":108030,"lat":51.5211626,"lon":-0.1320351},{"type":"node","id":108033,"lat":51.521388,"lon":-0.1315463},{"type":"node","id":5440856684,"lat":51.5211931,"lon":-0.1313169},{"type":"node","id":107763,"lat":51.5056259,"lon":-0.1266957},{"type":"node","id":107769,"lat":51.5069512,"lon":-0.124963},{"type":"node","id":107770,"lat":51.507179,"lon":-0.1275661},{"type":"node","id":107771,"lat":51.5074002,"lon":-0.1272858},{"type":"node","id":107773,"lat":51.5075061,"lon":-0.1273553},{"type":"node","id":107774,"lat":51.5075453,"lon":-0.1278265},{"type":"node","id":107777,"lat":51.5058624,"lon":-0.130808},{"type":"node","id":107780,"lat":51.5079688,"lon":-0.1265227},{"type":"node","id":109633,"lat":51.505944,"lon":-0.1255486},{"type":"node","id":8916243,"lat":51.5034004,"lon":-0.1291042},{"type":"node","id":8916246,"lat":51.5053418,"lon":-0.1297431},{"type":"node","id":8916255,"lat":51.5049348,"lon":-0.1248356},{"type":"node","id":8916280,"lat":51.5023578,"lon":-0.1297052},{"type":"node","id":21665755,"lat":51.5075248,"lon":-0.1285442},{"type":"node","id":21665756,"lat":51.5075592,"lon":-0.1297484},{"type":"node","id":25377895,"lat":51.5049523,"lon":-0.1252806},{"type":"node","id":25377905,"lat":51.5065071,"lon":-0.1251758},{"type":"node","id":25377942,"lat":51.5032296,"lon":-0.127947},{"type":"node","id":25377946,"lat":51.5048554,"lon":-0.1263571},{"type":"node","id":25504189,"lat":51.5075659,"lon":-0.1277216},{"type":"node","id":25504190,"lat":51.50728,"lon":-0.1280121},{"type":"node","id":25504191,"lat":51.5071532,"lon":-0.1277649},{"type":"node","id":25504193,"lat":51.5075812,"lon":-0.1258899},{"type":"node","id":25504194,"lat":51.5078264,"lon":-0.1255369},{"type":"node","id":25504195,"lat":51.5074928,"lon":-0.1249107},{"type":"node","id":25504196,"lat":51.5072353,"lon":-0.1253526},{"type":"node","id":25504197,"lat":51.5076934,"lon":-0.1246141},{"type":"node","id":25504199,"lat":51.5070488,"lon":-0.1254736},{"type":"node","id":25504201,"lat":51.5071941,"lon":-0.1253816},{"type":"node","id":25504259,"lat":51.5066452,"lon":-0.1273065},{"type":"node","id":25504260,"lat":51.5067262,"lon":-0.1267308},{"type":"node","id":25504261,"lat":51.5066113,"lon":-0.1265543},{"type":"node","id":25504262,"lat":51.5061185,"lon":-0.1265116},{"type":"node","id":25504263,"lat":51.5062427,"lon":-0.126607},{"type":"node","id":25504421,"lat":51.506306,"lon":-0.1257722},{"type":"node","id":25507028,"lat":51.5031978,"lon":-0.1260142},{"type":"node","id":25507047,"lat":51.5064254,"lon":-0.1278809},{"type":"node","id":25507048,"lat":51.5063448,"lon":-0.1281995},{"type":"node","id":25620548,"lat":51.507591,"lon":-0.1300191},{"type":"node","id":25620549,"lat":51.5074232,"lon":-0.1302982},{"type":"node","id":26573667,"lat":51.5079533,"lon":-0.1303096},{"type":"node","id":26630790,"lat":51.5070534,"lon":-0.1299459},{"type":"node","id":26630791,"lat":51.5067679,"lon":-0.129674},{"type":"node","id":26630792,"lat":51.5064783,"lon":-0.1292845},{"type":"node","id":26630793,"lat":51.5076804,"lon":-0.130844},{"type":"node","id":26630839,"lat":51.5077509,"lon":-0.1309304},{"type":"node","id":26671717,"lat":51.5074339,"lon":-0.1280059},{"type":"node","id":26671718,"lat":51.507293,"lon":-0.1273502},{"type":"node","id":26671719,"lat":51.5072559,"lon":-0.1279902},{"type":"node","id":26671720,"lat":51.5075102,"lon":-0.1282287},{"type":"node","id":26671724,"lat":51.5074737,"lon":-0.1289823},{"type":"node","id":26671730,"lat":51.5071336,"lon":-0.1290897},{"type":"node","id":26671733,"lat":51.5072236,"lon":-0.1298745},{"type":"node","id":26671734,"lat":51.5072537,"lon":-0.1301367},{"type":"node","id":130114421,"lat":51.5043785,"lon":-0.1291864},{"type":"node","id":256794566,"lat":51.5058957,"lon":-0.1268357},{"type":"node","id":256794570,"lat":51.5071532,"lon":-0.1277314},{"type":"node","id":256794571,"lat":51.5072493,"lon":-0.1274117},{"type":"node","id":256794572,"lat":51.5068505,"lon":-0.1273961},{"type":"node","id":256794573,"lat":51.5070753,"lon":-0.1275933},{"type":"node","id":256794574,"lat":51.5071183,"lon":-0.1274409},{"type":"node","id":256794575,"lat":51.5067509,"lon":-0.12799},{"type":"node","id":256794576,"lat":51.5070575,"lon":-0.1284367},{"type":"node","id":256794578,"lat":51.5069902,"lon":-0.1281008},{"type":"node","id":256794579,"lat":51.5070393,"lon":-0.1281888},{"type":"node","id":256794580,"lat":51.5071433,"lon":-0.128056},{"type":"node","id":256794581,"lat":51.5070791,"lon":-0.1279399},{"type":"node","id":256794582,"lat":51.5073944,"lon":-0.1282594},{"type":"node","id":256794584,"lat":51.5073141,"lon":-0.1270838},{"type":"node","id":256794588,"lat":51.5076666,"lon":-0.1274017},{"type":"node","id":256794589,"lat":51.5076821,"lon":-0.127359},{"type":"node","id":256794590,"lat":51.5078235,"lon":-0.1271496},{"type":"node","id":256794592,"lat":51.5077491,"lon":-0.1270286},{"type":"node","id":256794593,"lat":51.5076222,"lon":-0.1271163},{"type":"node","id":256794594,"lat":51.5075795,"lon":-0.12716},{"type":"node","id":317182560,"lat":51.5041611,"lon":-0.1263071},{"type":"node","id":345836137,"lat":51.5031437,"lon":-0.1260088},{"type":"node","id":345836394,"lat":51.5031534,"lon":-0.1257591},{"type":"node","id":345838183,"lat":51.5033591,"lon":-0.1289465},{"type":"node","id":345838184,"lat":51.5033186,"lon":-0.1283335},{"type":"node","id":345863587,"lat":51.5063759,"lon":-0.1279826},{"type":"node","id":345863907,"lat":51.5062868,"lon":-0.1278342},{"type":"node","id":345881019,"lat":51.5048151,"lon":-0.1293292},{"type":"node","id":345881288,"lat":51.5057752,"lon":-0.1310236},{"type":"node","id":442691373,"lat":51.5051838,"lon":-0.1294797},{"type":"node","id":442691380,"lat":51.5040164,"lon":-0.1290745},{"type":"node","id":452514328,"lat":51.5078535,"lon":-0.125594},{"type":"node","id":452514368,"lat":51.5080027,"lon":-0.1253629},{"type":"node","id":452515765,"lat":51.5058813,"lon":-0.1269725},{"type":"node","id":452515772,"lat":51.5057968,"lon":-0.1274848},{"type":"node","id":469772106,"lat":51.5040491,"lon":-0.1262846},{"type":"node","id":469772108,"lat":51.5041998,"lon":-0.1261389},{"type":"node","id":469772110,"lat":51.5037854,"lon":-0.1260865},{"type":"node","id":540653726,"lat":51.5026584,"lon":-0.1261966},{"type":"node","id":540653733,"lat":51.502665,"lon":-0.1259753},{"type":"node","id":540653735,"lat":51.5025253,"lon":-0.1259806},{"type":"node","id":603749189,"lat":51.5023277,"lon":-0.129711},{"type":"node","id":632673456,"lat":51.5074604,"lon":-0.1288441},{"type":"node","id":729620161,"lat":51.5031878,"lon":-0.1271718},{"type":"node","id":729620172,"lat":51.5034379,"lon":-0.1271909},{"type":"node","id":729758271,"lat":51.5035537,"lon":-0.1283229},{"type":"node","id":729758277,"lat":51.5034691,"lon":-0.1271838},{"type":"node","id":729758278,"lat":51.5035192,"lon":-0.1271168},{"type":"node","id":729758280,"lat":51.5035594,"lon":-0.1269276},{"type":"node","id":729758282,"lat":51.5035581,"lon":-0.1267993},{"type":"node","id":849731781,"lat":51.5076491,"lon":-0.1246758},{"type":"node","id":853293962,"lat":51.5064203,"lon":-0.1294279},{"type":"node","id":1026069088,"lat":51.5044153,"lon":-0.1256524},{"type":"node","id":1026069092,"lat":51.5043836,"lon":-0.1256002},{"type":"node","id":1026069095,"lat":51.5043559,"lon":-0.1255699},{"type":"node","id":1026069103,"lat":51.5043121,"lon":-0.1255425},{"type":"node","id":1026069133,"lat":51.504279,"lon":-0.125537},{"type":"node","id":1026069135,"lat":51.5042458,"lon":-0.1255499},{"type":"node","id":1026069138,"lat":51.5042149,"lon":-0.1255737},{"type":"node","id":1026069144,"lat":51.5040727,"lon":-0.1255744},{"type":"node","id":1139318696,"lat":51.5063946,"lon":-0.1254089},{"type":"node","id":1139318698,"lat":51.506062,"lon":-0.1267035},{"type":"node","id":1139318714,"lat":51.5060067,"lon":-0.1269031},{"type":"node","id":1188939728,"lat":51.5073717,"lon":-0.1305346},{"type":"node","id":1188939749,"lat":51.5075194,"lon":-0.1300422},{"type":"node","id":1188939752,"lat":51.5077927,"lon":-0.1303634},{"type":"node","id":1188939772,"lat":51.5076885,"lon":-0.1302685},{"type":"node","id":1201169692,"lat":51.507918,"lon":-0.1255257},{"type":"node","id":1201169871,"lat":51.507869,"lon":-0.1255938},{"type":"node","id":1239525647,"lat":51.5075436,"lon":-0.1288187},{"type":"node","id":1239525667,"lat":51.5074889,"lon":-0.1279514},{"type":"node","id":1239525671,"lat":51.5071626,"lon":-0.1278239},{"type":"node","id":1239525686,"lat":51.5076692,"lon":-0.1287627},{"type":"node","id":1239525697,"lat":51.5075742,"lon":-0.1276355},{"type":"node","id":1239525701,"lat":51.5071627,"lon":-0.1276448},{"type":"node","id":1239525705,"lat":51.5069477,"lon":-0.1281776},{"type":"node","id":1239525708,"lat":51.5073626,"lon":-0.1280276},{"type":"node","id":1239525711,"lat":51.5076106,"lon":-0.1287367},{"type":"node","id":1239525719,"lat":51.5066731,"lon":-0.1265863},{"type":"node","id":1239525728,"lat":51.5067113,"lon":-0.1266483},{"type":"node","id":1239525729,"lat":51.5075515,"lon":-0.1274508},{"type":"node","id":1239525734,"lat":51.5072181,"lon":-0.1279469},{"type":"node","id":1239525752,"lat":51.5079528,"lon":-0.1271466},{"type":"node","id":1239525762,"lat":51.5079233,"lon":-0.1270177},{"type":"node","id":1239525771,"lat":51.5075305,"lon":-0.1274043},{"type":"node","id":1274040741,"lat":51.5073218,"lon":-0.1282501},{"type":"node","id":1274040745,"lat":51.5074629,"lon":-0.1272963},{"type":"node","id":1274040747,"lat":51.5075152,"lon":-0.1272252},{"type":"node","id":1274053703,"lat":51.5078666,"lon":-0.1272514},{"type":"node","id":1394485185,"lat":51.5026677,"lon":-0.1296215},{"type":"node","id":1394485239,"lat":51.5028824,"lon":-0.1295228},{"type":"node","id":1465091808,"lat":51.5058467,"lon":-0.1271149},{"type":"node","id":1531257488,"lat":51.5032324,"lon":-0.1279896},{"type":"node","id":1531257497,"lat":51.5031636,"lon":-0.1269907},{"type":"node","id":1531257503,"lat":51.5032332,"lon":-0.1282003},{"type":"node","id":1531257534,"lat":51.5035593,"lon":-0.1269618},{"type":"node","id":1531272560,"lat":51.5068201,"lon":-0.1286466},{"type":"node","id":1531272561,"lat":51.506661,"lon":-0.1286567},{"type":"node","id":1531272586,"lat":51.5067638,"lon":-0.1287712},{"type":"node","id":1538897649,"lat":51.5058393,"lon":-0.1271673},{"type":"node","id":1538931290,"lat":51.5052454,"lon":-0.1266856},{"type":"node","id":1538931292,"lat":51.5051805,"lon":-0.1270238},{"type":"node","id":1538931318,"lat":51.5052128,"lon":-0.1268289},{"type":"node","id":1538956411,"lat":51.5038068,"lon":-0.1290012},{"type":"node","id":1539434357,"lat":51.5048418,"lon":-0.1265272},{"type":"node","id":1539480548,"lat":51.5078693,"lon":-0.1255376},{"type":"node","id":1587616047,"lat":51.505154,"lon":-0.1247072},{"type":"node","id":1587616056,"lat":51.5055651,"lon":-0.1245387},{"type":"node","id":1637789404,"lat":51.5077627,"lon":-0.130353},{"type":"node","id":1637789413,"lat":51.5078616,"lon":-0.1303622},{"type":"node","id":1637831150,"lat":51.507636,"lon":-0.1304486},{"type":"node","id":1637831151,"lat":51.5076518,"lon":-0.1310419},{"type":"node","id":1637831172,"lat":51.5077011,"lon":-0.130382},{"type":"node","id":1660481244,"lat":51.5073469,"lon":-0.1281246},{"type":"node","id":1660481465,"lat":51.5078113,"lon":-0.1273092},{"type":"node","id":1660481466,"lat":51.5078592,"lon":-0.1272709},{"type":"node","id":1697691548,"lat":51.5078623,"lon":-0.1268913},{"type":"node","id":1782020027,"lat":51.5023247,"lon":-0.1261899},{"type":"node","id":1782020193,"lat":51.5021076,"lon":-0.1262032},{"type":"node","id":1782020321,"lat":51.5031667,"lon":-0.1262113},{"type":"node","id":1782020517,"lat":51.5028524,"lon":-0.12598},{"type":"node","id":1782020519,"lat":51.5022247,"lon":-0.1261912},{"type":"node","id":1938580020,"lat":51.5054891,"lon":-0.1267951},{"type":"node","id":1938580023,"lat":51.5055433,"lon":-0.1268041},{"type":"node","id":1938580027,"lat":51.505605,"lon":-0.1267467},{"type":"node","id":1938580032,"lat":51.5071665,"lon":-0.1280361},{"type":"node","id":1938580037,"lat":51.5074511,"lon":-0.1271272},{"type":"node","id":1938580042,"lat":51.5076694,"lon":-0.1309446},{"type":"node","id":1938580044,"lat":51.5077293,"lon":-0.1303206},{"type":"node","id":1938580046,"lat":51.5077974,"lon":-0.1310668},{"type":"node","id":1938580049,"lat":51.5078265,"lon":-0.1303693},{"type":"node","id":1944317624,"lat":51.507419,"lon":-0.1290019},{"type":"node","id":2058516816,"lat":51.506155,"lon":-0.1300842},{"type":"node","id":2058516819,"lat":51.5063938,"lon":-0.1294936},{"type":"node","id":2058516821,"lat":51.5062254,"lon":-0.1293298},{"type":"node","id":2389980687,"lat":51.5029751,"lon":-0.1259924},{"type":"node","id":2389980691,"lat":51.5033469,"lon":-0.1260343},{"type":"node","id":2389980694,"lat":51.5033462,"lon":-0.126222},{"type":"node","id":2389980697,"lat":51.5029683,"lon":-0.1262046},{"type":"node","id":2389980707,"lat":51.502763,"lon":-0.1259739},{"type":"node","id":2389983770,"lat":51.5046224,"lon":-0.1264548},{"type":"node","id":2389983771,"lat":51.5046977,"lon":-0.1262929},{"type":"node","id":2389983772,"lat":51.5037287,"lon":-0.1262462},{"type":"node","id":2389983773,"lat":51.5046725,"lon":-0.1264736},{"type":"node","id":2389983774,"lat":51.5035412,"lon":-0.1262315},{"type":"node","id":2389983775,"lat":51.5043012,"lon":-0.1263381},{"type":"node","id":2389983776,"lat":51.5044661,"lon":-0.126393},{"type":"node","id":2389986332,"lat":51.5045993,"lon":-0.1262543},{"type":"node","id":2389986334,"lat":51.5044522,"lon":-0.1262046},{"type":"node","id":2389986336,"lat":51.5040546,"lon":-0.1261188},{"type":"node","id":2389991891,"lat":51.5049735,"lon":-0.1265758},{"type":"node","id":2389991892,"lat":51.5049532,"lon":-0.1251049},{"type":"node","id":2389991893,"lat":51.5050475,"lon":-0.1264412},{"type":"node","id":2389991894,"lat":51.504796,"lon":-0.1261747},{"type":"node","id":2389991895,"lat":51.5047913,"lon":-0.126483},{"type":"node","id":2389991898,"lat":51.5049961,"lon":-0.1264184},{"type":"node","id":2389991899,"lat":51.504849,"lon":-0.1263952},{"type":"node","id":2389991901,"lat":51.5047765,"lon":-0.1262301},{"type":"node","id":2389991902,"lat":51.5047288,"lon":-0.1264924},{"type":"node","id":2389991903,"lat":51.5047177,"lon":-0.1262838},{"type":"node","id":2389991904,"lat":51.5048157,"lon":-0.1264614},{"type":"node","id":2389991907,"lat":51.5048618,"lon":-0.1260488},{"type":"node","id":2389991908,"lat":51.5048112,"lon":-0.1260919},{"type":"node","id":2389991909,"lat":51.5048383,"lon":-0.1264292},{"type":"node","id":2389991910,"lat":51.5047632,"lon":-0.1264939},{"type":"node","id":2389991911,"lat":51.5047597,"lon":-0.1262555},{"type":"node","id":2389991912,"lat":51.5047789,"lon":-0.1257946},{"type":"node","id":2389991913,"lat":51.5047347,"lon":-0.126273},{"type":"node","id":2389991915,"lat":51.5048888,"lon":-0.126097},{"type":"node","id":2389991917,"lat":51.5048758,"lon":-0.1262016},{"type":"node","id":2390005217,"lat":51.5071707,"lon":-0.1274359},{"type":"node","id":2390005218,"lat":51.5069268,"lon":-0.1274449},{"type":"node","id":2390005220,"lat":51.5071224,"lon":-0.1276653},{"type":"node","id":2390005221,"lat":51.5070436,"lon":-0.127552},{"type":"node","id":2390005222,"lat":51.5071979,"lon":-0.1275044},{"type":"node","id":2390005223,"lat":51.506379,"lon":-0.1271821},{"type":"node","id":2390005956,"lat":51.5050529,"lon":-0.126609},{"type":"node","id":2390005959,"lat":51.5055642,"lon":-0.1268275},{"type":"node","id":2390008564,"lat":51.5060729,"lon":-0.1269407},{"type":"node","id":2390008565,"lat":51.5055808,"lon":-0.1267909},{"type":"node","id":2390008566,"lat":51.5061163,"lon":-0.1269637},{"type":"node","id":2390008567,"lat":51.5058012,"lon":-0.1269324},{"type":"node","id":2390008568,"lat":51.5060426,"lon":-0.1270568},{"type":"node","id":2390008569,"lat":51.5059772,"lon":-0.1270284},{"type":"node","id":2390011979,"lat":51.5031301,"lon":-0.1262101},{"type":"node","id":2390011980,"lat":51.5031644,"lon":-0.1264299},{"type":"node","id":2390011981,"lat":51.5031685,"lon":-0.126011},{"type":"node","id":2390036481,"lat":51.5078589,"lon":-0.126724},{"type":"node","id":2390036490,"lat":51.5077959,"lon":-0.1268331},{"type":"node","id":2390036493,"lat":51.5078037,"lon":-0.1270122},{"type":"node","id":2390036495,"lat":51.5077262,"lon":-0.1269508},{"type":"node","id":2390036497,"lat":51.5079085,"lon":-0.1271448},{"type":"node","id":2390036502,"lat":51.5078525,"lon":-0.1271452},{"type":"node","id":2390036505,"lat":51.50771,"lon":-0.1270414},{"type":"node","id":2390036511,"lat":51.5077566,"lon":-0.1271056},{"type":"node","id":2390039150,"lat":51.5077926,"lon":-0.1271657},{"type":"node","id":2390039151,"lat":51.5077595,"lon":-0.1271991},{"type":"node","id":2390039152,"lat":51.5076407,"lon":-0.1274818},{"type":"node","id":2390039153,"lat":51.5077419,"lon":-0.1272357},{"type":"node","id":2390039155,"lat":51.5077211,"lon":-0.1272069},{"type":"node","id":2390039431,"lat":51.5075654,"lon":-0.1274999},{"type":"node","id":2390039432,"lat":51.5075746,"lon":-0.1275642},{"type":"node","id":2467911855,"lat":51.5067345,"lon":-0.1285201},{"type":"node","id":2494778478,"lat":51.5070115,"lon":-0.1275322},{"type":"node","id":2538676640,"lat":51.5072852,"lon":-0.1268426},{"type":"node","id":2538676644,"lat":51.5069856,"lon":-0.1251663},{"type":"node","id":2549111514,"lat":51.5075481,"lon":-0.1286555},{"type":"node","id":2549111517,"lat":51.5075749,"lon":-0.1287105},{"type":"node","id":2620412360,"lat":51.504818,"lon":-0.1258082},{"type":"node","id":2620412365,"lat":51.504819,"lon":-0.1260654},{"type":"node","id":2726836315,"lat":51.5079955,"lon":-0.1270298},{"type":"node","id":2726849721,"lat":51.5071262,"lon":-0.1258704},{"type":"node","id":2726849722,"lat":51.5073063,"lon":-0.1272878},{"type":"node","id":2726849723,"lat":51.5073156,"lon":-0.1280711},{"type":"node","id":2726849724,"lat":51.5073191,"lon":-0.1272002},{"type":"node","id":2726849725,"lat":51.5073309,"lon":-0.1273162},{"type":"node","id":2726849726,"lat":51.5073351,"lon":-0.1268164},{"type":"node","id":2726849727,"lat":51.5073603,"lon":-0.1273012},{"type":"node","id":2726849728,"lat":51.5073677,"lon":-0.1281784},{"type":"node","id":2726849729,"lat":51.5074109,"lon":-0.1283606},{"type":"node","id":2726849730,"lat":51.5074237,"lon":-0.1284931},{"type":"node","id":2726849731,"lat":51.5074354,"lon":-0.1271837},{"type":"node","id":2726849732,"lat":51.5075017,"lon":-0.1271576},{"type":"node","id":2726852805,"lat":51.5078262,"lon":-0.1272901},{"type":"node","id":2726852811,"lat":51.5078429,"lon":-0.1272766},{"type":"node","id":2726861236,"lat":51.5070338,"lon":-0.128025},{"type":"node","id":2726861242,"lat":51.5070807,"lon":-0.1281244},{"type":"node","id":2726861247,"lat":51.507104,"lon":-0.1278787},{"type":"node","id":2726861258,"lat":51.5071869,"lon":-0.1278914},{"type":"node","id":2726861270,"lat":51.5073991,"lon":-0.1280216},{"type":"node","id":2726866274,"lat":51.5073013,"lon":-0.1281715},{"type":"node","id":2726866276,"lat":51.5073073,"lon":-0.1281885},{"type":"node","id":2726866279,"lat":51.5073136,"lon":-0.1282106},{"type":"node","id":3075334818,"lat":51.5057982,"lon":-0.1306751},{"type":"node","id":3397689899,"lat":51.5058521,"lon":-0.1258807},{"type":"node","id":3399223683,"lat":51.5065085,"lon":-0.1251751},{"type":"node","id":3439409095,"lat":51.5034958,"lon":-0.127157},{"type":"node","id":3439409096,"lat":51.5035359,"lon":-0.1270631},{"type":"node","id":3439409099,"lat":51.5046537,"lon":-0.1292787},{"type":"node","id":3439409101,"lat":51.5052341,"lon":-0.129539},{"type":"node","id":3439409103,"lat":51.5052803,"lon":-0.1296179},{"type":"node","id":3439409112,"lat":51.5077807,"lon":-0.1310419},{"type":"node","id":3439409113,"lat":51.5077653,"lon":-0.1309994},{"type":"node","id":3439409122,"lat":51.5076606,"lon":-0.1310048},{"type":"node","id":3439409123,"lat":51.5076749,"lon":-0.1309092},{"type":"node","id":3439409124,"lat":51.5076782,"lon":-0.1307959},{"type":"node","id":3439409125,"lat":51.5076716,"lon":-0.1307199},{"type":"node","id":3439409126,"lat":51.5076595,"lon":-0.1302137},{"type":"node","id":3439409127,"lat":51.5076364,"lon":-0.1301607},{"type":"node","id":3439409128,"lat":51.507502,"lon":-0.1300563},{"type":"node","id":3439409129,"lat":51.5074877,"lon":-0.1300722},{"type":"node","id":3439409130,"lat":51.5074745,"lon":-0.1300987},{"type":"node","id":3439409131,"lat":51.5074602,"lon":-0.1301377},{"type":"node","id":3439409132,"lat":51.5073544,"lon":-0.1306013},{"type":"node","id":3439409133,"lat":51.5073379,"lon":-0.1306526},{"type":"node","id":3439409134,"lat":51.5075423,"lon":-0.1286274},{"type":"node","id":3439409135,"lat":51.5075373,"lon":-0.1286019},{"type":"node","id":3439409136,"lat":51.5075623,"lon":-0.1286917},{"type":"node","id":3439409137,"lat":51.5075924,"lon":-0.1287279},{"type":"node","id":3439409138,"lat":51.5075189,"lon":-0.1284932},{"type":"node","id":3439409139,"lat":51.5075148,"lon":-0.1284168},{"type":"node","id":3439409140,"lat":51.5075139,"lon":-0.128162},{"type":"node","id":3439409141,"lat":51.5075198,"lon":-0.1280923},{"type":"node","id":3439409143,"lat":51.5075239,"lon":-0.1278857},{"type":"node","id":3439409144,"lat":51.507534,"lon":-0.1278576},{"type":"node","id":3439409145,"lat":51.5075573,"lon":-0.1277771},{"type":"node","id":3439409146,"lat":51.5075072,"lon":-0.1279206},{"type":"node","id":3439409147,"lat":51.5074647,"lon":-0.1279809},{"type":"node","id":3439409148,"lat":51.5074498,"lon":-0.1272803},{"type":"node","id":3439409149,"lat":51.507473,"lon":-0.1273077},{"type":"node","id":3439409150,"lat":51.5074889,"lon":-0.1273265},{"type":"node","id":3439409151,"lat":51.5072201,"lon":-0.1274606},{"type":"node","id":3439409152,"lat":51.5072118,"lon":-0.1280037},{"type":"node","id":3439409153,"lat":51.5071884,"lon":-0.1280171},{"type":"node","id":3783471662,"lat":51.5038288,"lon":-0.1290083},{"type":"node","id":3783505020,"lat":51.5041999,"lon":-0.1291266},{"type":"node","id":3783539225,"lat":51.5050913,"lon":-0.1294181},{"type":"node","id":3807205741,"lat":51.5075158,"lon":-0.129362},{"type":"node","id":3807205742,"lat":51.5075776,"lon":-0.1291592},{"type":"node","id":3807436630,"lat":51.507386,"lon":-0.1270457},{"type":"node","id":3810553685,"lat":51.5072502,"lon":-0.1265101},{"type":"node","id":3810553686,"lat":51.5072878,"lon":-0.1267287},{"type":"node","id":3810553687,"lat":51.5075742,"lon":-0.129873},{"type":"node","id":3810553688,"lat":51.5080213,"lon":-0.1301246},{"type":"node","id":3810553689,"lat":51.5076473,"lon":-0.1298565},{"type":"node","id":3854995565,"lat":51.5033749,"lon":-0.1290071},{"type":"node","id":3854995580,"lat":51.5051467,"lon":-0.1294487},{"type":"node","id":4319985923,"lat":51.507836,"lon":-0.1283232},{"type":"node","id":4319985924,"lat":51.5079491,"lon":-0.1276212},{"type":"node","id":4347914532,"lat":51.5078782,"lon":-0.13054},{"type":"node","id":4347914540,"lat":51.5078955,"lon":-0.1304869},{"type":"node","id":4347917401,"lat":51.507512,"lon":-0.1283026},{"type":"node","id":4347917404,"lat":51.5076369,"lon":-0.1287491},{"type":"node","id":4347922045,"lat":51.5075594,"lon":-0.128977},{"type":"node","id":4347922046,"lat":51.5071114,"lon":-0.1280908},{"type":"node","id":4347925119,"lat":51.5074528,"lon":-0.1287714},{"type":"node","id":4474765621,"lat":51.5077191,"lon":-0.1261252},{"type":"node","id":4650621304,"lat":51.5067902,"lon":-0.1273698},{"type":"node","id":4650621305,"lat":51.506483,"lon":-0.1272307},{"type":"node","id":4650621306,"lat":51.5062798,"lon":-0.1270945},{"type":"node","id":4650621307,"lat":51.5052456,"lon":-0.1265213},{"type":"node","id":4650621308,"lat":51.5049407,"lon":-0.1254576},{"type":"node","id":4707225202,"lat":51.5042655,"lon":-0.126156},{"type":"node","id":4766203722,"lat":51.5072767,"lon":-0.1266637},{"type":"node","id":4766203723,"lat":51.5073998,"lon":-0.1270813},{"type":"node","id":4766203724,"lat":51.5074231,"lon":-0.127091},{"type":"node","id":4813058486,"lat":51.5077498,"lon":-0.1248763},{"type":"node","id":4813058487,"lat":51.5075818,"lon":-0.124584},{"type":"node","id":4897627182,"lat":51.5040137,"lon":-0.1290736},{"type":"node","id":4897627183,"lat":51.5057004,"lon":-0.1304754},{"type":"node","id":4897627185,"lat":51.5056415,"lon":-0.1266372},{"type":"node","id":4897627186,"lat":51.5060128,"lon":-0.1255931},{"type":"node","id":4897627188,"lat":51.5050965,"lon":-0.1247409},{"type":"node","id":4919661397,"lat":51.5070295,"lon":-0.1283192},{"type":"node","id":5034506699,"lat":51.5068868,"lon":-0.1280782},{"type":"node","id":5034506700,"lat":51.5069941,"lon":-0.1282618},{"type":"node","id":5034506707,"lat":51.5068257,"lon":-0.1280232},{"type":"node","id":5034506822,"lat":51.5062957,"lon":-0.1293976},{"type":"node","id":5034506842,"lat":51.5065445,"lon":-0.1289703},{"type":"node","id":5034506843,"lat":51.5066318,"lon":-0.1290514},{"type":"node","id":5034506844,"lat":51.5066967,"lon":-0.1285846},{"type":"node","id":5034506845,"lat":51.5067927,"lon":-0.1287067},{"type":"node","id":5289839774,"lat":51.5047653,"lon":-0.126247},{"type":"node","id":5289839775,"lat":51.5048646,"lon":-0.1262866},{"type":"node","id":5290536464,"lat":51.5060382,"lon":-0.1267896},{"type":"node","id":5323721997,"lat":51.507408,"lon":-0.1257294},{"type":"node","id":5323721998,"lat":51.5074511,"lon":-0.1256632},{"type":"node","id":5323721999,"lat":51.5074866,"lon":-0.125854},{"type":"node","id":5323722000,"lat":51.5075259,"lon":-0.1257936},{"type":"node","id":5323722004,"lat":51.5076071,"lon":-0.1260453},{"type":"node","id":5323722005,"lat":51.5076371,"lon":-0.1259852},{"type":"node","id":5323722009,"lat":51.5076206,"lon":-0.125835},{"type":"node","id":5323722010,"lat":51.5077938,"lon":-0.1255858},{"type":"node","id":5323722017,"lat":51.5079739,"lon":-0.1254952},{"type":"node","id":5323722018,"lat":51.5079459,"lon":-0.1254373},{"type":"node","id":5323722126,"lat":51.5076803,"lon":-0.1258177},{"type":"node","id":5323722127,"lat":51.5076595,"lon":-0.1257806},{"type":"node","id":5323722153,"lat":51.507288,"lon":-0.1253792},{"type":"node","id":5421407179,"lat":51.507846,"lon":-0.1282569},{"type":"node","id":5521427549,"lat":51.5067208,"lon":-0.1288624},{"type":"node","id":5521427550,"lat":51.5068738,"lon":-0.1285279},{"type":"node","id":5521427552,"lat":51.5067877,"lon":-0.1284337},{"type":"node","id":5742335191,"lat":51.5048968,"lon":-0.1258389},{"type":"node","id":6064937878,"lat":51.507673,"lon":-0.1286135},{"type":"node","id":6064937879,"lat":51.5078972,"lon":-0.1273129},{"type":"node","id":6064937881,"lat":51.5076215,"lon":-0.1282476},{"type":"node","id":6064937883,"lat":51.5077834,"lon":-0.1286333},{"type":"node","id":6064939286,"lat":51.5077949,"lon":-0.1273523},{"type":"node","id":6064939287,"lat":51.507994,"lon":-0.1273385},{"type":"node","id":6064986824,"lat":51.5071482,"lon":-0.1292122},{"type":"node","id":6064986825,"lat":51.5071761,"lon":-0.1294541},{"type":"node","id":6075832879,"lat":51.5031814,"lon":-0.1270656},{"type":"node","id":6173728484,"lat":51.507889,"lon":-0.1280007},{"type":"node","id":6337577694,"lat":51.5079839,"lon":-0.1280413},{"type":"node","id":6337577695,"lat":51.5079364,"lon":-0.127707},{"type":"node","id":6337577696,"lat":51.5079174,"lon":-0.1278162},{"type":"node","id":6337577697,"lat":51.5079641,"lon":-0.1279141},{"type":"node","id":6337577698,"lat":51.5079733,"lon":-0.127957},{"type":"node","id":6337577699,"lat":51.5079741,"lon":-0.1280066},{"type":"node","id":6337577700,"lat":51.5079714,"lon":-0.128036},{"type":"node","id":6337577701,"lat":51.5078615,"lon":-0.1281702},{"type":"node","id":6337577702,"lat":51.5078998,"lon":-0.1281595},{"type":"node","id":6337577703,"lat":51.5079257,"lon":-0.1281407},{"type":"node","id":6337577704,"lat":51.5079499,"lon":-0.1281018},{"type":"node","id":6337577705,"lat":51.5079666,"lon":-0.1280535},{"type":"node","id":6337577706,"lat":51.5079449,"lon":-0.1278624},{"type":"node","id":107728,"lat":51.5057051,"lon":-0.1180919},{"type":"node","id":107768,"lat":51.5066734,"lon":-0.1230266},{"type":"node","id":109634,"lat":51.506257,"lon":-0.1243271},{"type":"node","id":25476051,"lat":51.5073162,"lon":-0.1226925},{"type":"node","id":25476053,"lat":51.5078075,"lon":-0.1233542},{"type":"node","id":25476056,"lat":51.5071058,"lon":-0.1221118},{"type":"node","id":25476059,"lat":51.5066318,"lon":-0.1225877},{"type":"node","id":25504187,"lat":51.5065682,"lon":-0.1226158},{"type":"node","id":25504200,"lat":51.5067538,"lon":-0.1237567},{"type":"node","id":25507025,"lat":51.5019646,"lon":-0.123912},{"type":"node","id":26398043,"lat":51.5078506,"lon":-0.1219438},{"type":"node","id":346361417,"lat":51.5031531,"lon":-0.1235888},{"type":"node","id":452514378,"lat":51.5071584,"lon":-0.1238352},{"type":"node","id":452514386,"lat":51.5070163,"lon":-0.1240234},{"type":"node","id":452514407,"lat":51.5072504,"lon":-0.1239935},{"type":"node","id":469762514,"lat":51.5016012,"lon":-0.1239978},{"type":"node","id":566201836,"lat":51.5067358,"lon":-0.1231366},{"type":"node","id":566201846,"lat":51.5071791,"lon":-0.1218338},{"type":"node","id":573320949,"lat":51.5079734,"lon":-0.1238814},{"type":"node","id":573320952,"lat":51.5079444,"lon":-0.1239224},{"type":"node","id":573320955,"lat":51.508006,"lon":-0.124035},{"type":"node","id":720890825,"lat":51.5080031,"lon":-0.1229937},{"type":"node","id":720890829,"lat":51.5077365,"lon":-0.1232292},{"type":"node","id":720890830,"lat":51.5079669,"lon":-0.1223114},{"type":"node","id":826724024,"lat":51.5075689,"lon":-0.1228033},{"type":"node","id":826724213,"lat":51.5074764,"lon":-0.1227707},{"type":"node","id":873549204,"lat":51.5064439,"lon":-0.1236219},{"type":"node","id":873549223,"lat":51.5075267,"lon":-0.1228594},{"type":"node","id":873549227,"lat":51.5076931,"lon":-0.1226308},{"type":"node","id":1139318692,"lat":51.5065108,"lon":-0.1226846},{"type":"node","id":1139318724,"lat":51.5064502,"lon":-0.1223939},{"type":"node","id":1139318726,"lat":51.5063332,"lon":-0.1225951},{"type":"node","id":1139318729,"lat":51.5052513,"lon":-0.122989},{"type":"node","id":1139318732,"lat":51.5065801,"lon":-0.1230013},{"type":"node","id":1537623265,"lat":51.5068789,"lon":-0.1220509},{"type":"node","id":1537623271,"lat":51.5069353,"lon":-0.1220087},{"type":"node","id":1537623283,"lat":51.5065906,"lon":-0.1222857},{"type":"node","id":1537627184,"lat":51.5068133,"lon":-0.1230802},{"type":"node","id":1537627190,"lat":51.5072534,"lon":-0.1227128},{"type":"node","id":1587616049,"lat":51.5057737,"lon":-0.1244532},{"type":"node","id":1587616057,"lat":51.5061633,"lon":-0.1242935},{"type":"node","id":1602135896,"lat":51.5047651,"lon":-0.1238471},{"type":"node","id":1602135897,"lat":51.5048079,"lon":-0.1238028},{"type":"node","id":1602135898,"lat":51.5048915,"lon":-0.1235522},{"type":"node","id":1602135900,"lat":51.5049092,"lon":-0.123698},{"type":"node","id":1602135902,"lat":51.5049397,"lon":-0.1234288},{"type":"node","id":1602135904,"lat":51.5049561,"lon":-0.1237669},{"type":"node","id":1602135905,"lat":51.5050117,"lon":-0.1233895},{"type":"node","id":1602135907,"lat":51.5050199,"lon":-0.1237895},{"type":"node","id":1602135909,"lat":51.5050796,"lon":-0.1234196},{"type":"node","id":1602135917,"lat":51.5050821,"lon":-0.1237571},{"type":"node","id":1602135920,"lat":51.5051265,"lon":-0.123504},{"type":"node","id":1602135921,"lat":51.5051088,"lon":-0.1237193},{"type":"node","id":1602135922,"lat":51.5051374,"lon":-0.1236159},{"type":"node","id":1602135943,"lat":51.5055095,"lon":-0.1233611},{"type":"node","id":1602135949,"lat":51.5055421,"lon":-0.1232291},{"type":"node","id":1602135950,"lat":51.5055383,"lon":-0.1234955},{"type":"node","id":1602135951,"lat":51.5055647,"lon":-0.1231972},{"type":"node","id":1602135953,"lat":51.5055859,"lon":-0.1235533},{"type":"node","id":1602135954,"lat":51.5056147,"lon":-0.1235673},{"type":"node","id":1602135955,"lat":51.5056204,"lon":-0.123163},{"type":"node","id":1602135957,"lat":51.5056742,"lon":-0.1235613},{"type":"node","id":1602135958,"lat":51.5056824,"lon":-0.1231749},{"type":"node","id":1602135960,"lat":51.5057275,"lon":-0.1235093},{"type":"node","id":1602135961,"lat":51.5057467,"lon":-0.1234691},{"type":"node","id":1602135962,"lat":51.5057334,"lon":-0.1232327},{"type":"node","id":1602135963,"lat":51.505764,"lon":-0.1233728},{"type":"node","id":1602135964,"lat":51.50582,"lon":-0.123071},{"type":"node","id":1602135979,"lat":51.5061308,"lon":-0.123009},{"type":"node","id":1602135980,"lat":51.5061349,"lon":-0.1231628},{"type":"node","id":1602135981,"lat":51.5061969,"lon":-0.1228975},{"type":"node","id":1602135982,"lat":51.5061772,"lon":-0.1232416},{"type":"node","id":1602135985,"lat":51.5062714,"lon":-0.1232705},{"type":"node","id":1602135986,"lat":51.5062621,"lon":-0.1228832},{"type":"node","id":1602135987,"lat":51.506304,"lon":-0.123251},{"type":"node","id":1602135989,"lat":51.506346,"lon":-0.1229658},{"type":"node","id":1602135992,"lat":51.5063528,"lon":-0.1231729},{"type":"node","id":1602135993,"lat":51.5063677,"lon":-0.1230678},{"type":"node","id":1602135995,"lat":51.5063725,"lon":-0.1227309},{"type":"node","id":1602135997,"lat":51.5064044,"lon":-0.122592},{"type":"node","id":1645813886,"lat":51.5060333,"lon":-0.1226667},{"type":"node","id":1645813888,"lat":51.5061834,"lon":-0.1225821},{"type":"node","id":1645813894,"lat":51.5067086,"lon":-0.1228636},{"type":"node","id":1645813917,"lat":51.5073907,"lon":-0.1227203},{"type":"node","id":1646043083,"lat":51.5056936,"lon":-0.1180701},{"type":"node","id":1646306166,"lat":51.5048493,"lon":-0.117993},{"type":"node","id":1646306171,"lat":51.5048884,"lon":-0.1179982},{"type":"node","id":1646306174,"lat":51.5050145,"lon":-0.1179969},{"type":"node","id":1646306180,"lat":51.505043,"lon":-0.1180074},{"type":"node","id":1646306182,"lat":51.5050691,"lon":-0.1180244},{"type":"node","id":1646306211,"lat":51.5052782,"lon":-0.1181878},{"type":"node","id":1646306219,"lat":51.5053002,"lon":-0.1181996},{"type":"node","id":1646306223,"lat":51.5053222,"lon":-0.1181996},{"type":"node","id":1694551537,"lat":51.5080294,"lon":-0.1230696},{"type":"node","id":1749442186,"lat":51.5047831,"lon":-0.1233574},{"type":"node","id":1749442188,"lat":51.5056015,"lon":-0.1230828},{"type":"node","id":1749442189,"lat":51.506294,"lon":-0.1228976},{"type":"node","id":1749442190,"lat":51.5064558,"lon":-0.1232441},{"type":"node","id":1749442191,"lat":51.5064879,"lon":-0.123266},{"type":"node","id":1824589995,"lat":51.5047444,"lon":-0.1237318},{"type":"node","id":1936062451,"lat":51.5014932,"lon":-0.1240174},{"type":"node","id":1936084669,"lat":51.5064993,"lon":-0.1223555},{"type":"node","id":1976790118,"lat":51.5069998,"lon":-0.1221983},{"type":"node","id":1976807766,"lat":51.5058157,"lon":-0.1180084},{"type":"node","id":1976807769,"lat":51.5052024,"lon":-0.1185274},{"type":"node","id":1976807775,"lat":51.5057207,"lon":-0.1180801},{"type":"node","id":1976807803,"lat":51.5057535,"lon":-0.1180554},{"type":"node","id":1976807805,"lat":51.5058025,"lon":-0.1180184},{"type":"node","id":1976807806,"lat":51.5058469,"lon":-0.1179858},{"type":"node","id":1976807812,"lat":51.5053256,"lon":-0.1184186},{"type":"node","id":1976807817,"lat":51.505766,"lon":-0.1180459},{"type":"node","id":2145880639,"lat":51.5078211,"lon":-0.1233402},{"type":"node","id":2153822338,"lat":51.5077529,"lon":-0.1216098},{"type":"node","id":2338578977,"lat":51.5055401,"lon":-0.1180449},{"type":"node","id":2494778477,"lat":51.5046786,"lon":-0.1233863},{"type":"node","id":2620430673,"lat":51.5064727,"lon":-0.1235166},{"type":"node","id":2620430674,"lat":51.5065156,"lon":-0.1234518},{"type":"node","id":2725915850,"lat":51.5064769,"lon":-0.1226707},{"type":"node","id":2725915865,"lat":51.5066038,"lon":-0.1233986},{"type":"node","id":2725915866,"lat":51.50662,"lon":-0.1232006},{"type":"node","id":2725915874,"lat":51.506652,"lon":-0.1233767},{"type":"node","id":2725915875,"lat":51.5066776,"lon":-0.1230465},{"type":"node","id":2725915878,"lat":51.5066893,"lon":-0.1236108},{"type":"node","id":2725915880,"lat":51.5066975,"lon":-0.1231598},{"type":"node","id":2725915883,"lat":51.506703,"lon":-0.1231912},{"type":"node","id":2725915891,"lat":51.506723,"lon":-0.123311},{"type":"node","id":2725915896,"lat":51.5067625,"lon":-0.1235772},{"type":"node","id":2725915897,"lat":51.5068153,"lon":-0.1237332},{"type":"node","id":2725915898,"lat":51.5068471,"lon":-0.1237302},{"type":"node","id":2725915899,"lat":51.506872,"lon":-0.1237435},{"type":"node","id":2725915900,"lat":51.5068951,"lon":-0.1237754},{"type":"node","id":2726738171,"lat":51.506482,"lon":-0.1225305},{"type":"node","id":3012176220,"lat":51.5048914,"lon":-0.1236264},{"type":"node","id":3012176221,"lat":51.5049084,"lon":-0.1234832},{"type":"node","id":3012176222,"lat":51.5049291,"lon":-0.1237363},{"type":"node","id":3012176223,"lat":51.5049738,"lon":-0.1234002},{"type":"node","id":3012176224,"lat":51.504987,"lon":-0.1237851},{"type":"node","id":3012176226,"lat":51.505047,"lon":-0.1233967},{"type":"node","id":3012176227,"lat":51.5050523,"lon":-0.12378},{"type":"node","id":3012176228,"lat":51.5051068,"lon":-0.1234564},{"type":"node","id":3012176229,"lat":51.5051277,"lon":-0.1236709},{"type":"node","id":3012176230,"lat":51.5051369,"lon":-0.1235587},{"type":"node","id":3012176231,"lat":51.5055137,"lon":-0.1233134},{"type":"node","id":3012176232,"lat":51.5055124,"lon":-0.1234092},{"type":"node","id":3012177133,"lat":51.5055248,"lon":-0.1232685},{"type":"node","id":3012177134,"lat":51.5055221,"lon":-0.1234548},{"type":"node","id":3012177135,"lat":51.50556,"lon":-0.123529},{"type":"node","id":3012177136,"lat":51.5055913,"lon":-0.1231748},{"type":"node","id":3012177138,"lat":51.5056447,"lon":-0.12357},{"type":"node","id":3012177139,"lat":51.5056519,"lon":-0.1231628},{"type":"node","id":3012177140,"lat":51.5057029,"lon":-0.1235406},{"type":"node","id":3012177141,"lat":51.5057102,"lon":-0.1231987},{"type":"node","id":3012177142,"lat":51.5057508,"lon":-0.1232748},{"type":"node","id":3012177143,"lat":51.5057591,"lon":-0.1234227},{"type":"node","id":3012177144,"lat":51.5057612,"lon":-0.1233225},{"type":"node","id":3012177145,"lat":51.5061235,"lon":-0.1230602},{"type":"node","id":3012177146,"lat":51.5061249,"lon":-0.1231127},{"type":"node","id":3012177147,"lat":51.5061463,"lon":-0.1229627},{"type":"node","id":3012177148,"lat":51.5061527,"lon":-0.1232068},{"type":"node","id":3012177149,"lat":51.5062066,"lon":-0.1232648},{"type":"node","id":3012177150,"lat":51.5062289,"lon":-0.1228832},{"type":"node","id":3012177151,"lat":51.5062388,"lon":-0.1232746},{"type":"node","id":3012177153,"lat":51.506323,"lon":-0.1229259},{"type":"node","id":3012177154,"lat":51.5063319,"lon":-0.1232176},{"type":"node","id":3012177155,"lat":51.5063613,"lon":-0.1230144},{"type":"node","id":3012177156,"lat":51.5063648,"lon":-0.1231221},{"type":"node","id":3012180176,"lat":51.5061688,"lon":-0.1229246},{"type":"node","id":3400241100,"lat":51.5038396,"lon":-0.1230225},{"type":"node","id":3400241138,"lat":51.5041223,"lon":-0.1229489},{"type":"node","id":3729004656,"lat":51.5020299,"lon":-0.1238955},{"type":"node","id":3729004662,"lat":51.5046483,"lon":-0.1231765},{"type":"node","id":3729004663,"lat":51.5046345,"lon":-0.1230457},{"type":"node","id":4084403495,"lat":51.5045426,"lon":-0.1232043},{"type":"node","id":4084403499,"lat":51.5030717,"lon":-0.1236103},{"type":"node","id":4084403508,"lat":51.5019061,"lon":-0.1239286},{"type":"node","id":4168504357,"lat":51.5046087,"lon":-0.1230557},{"type":"node","id":4174199993,"lat":51.5064044,"lon":-0.1224299},{"type":"node","id":4174199994,"lat":51.5065374,"lon":-0.1225043},{"type":"node","id":4174199995,"lat":51.5065669,"lon":-0.1223038},{"type":"node","id":4174199999,"lat":51.5076986,"lon":-0.1214425},{"type":"node","id":4174200000,"lat":51.5079892,"lon":-0.1212011},{"type":"node","id":4194431954,"lat":51.504827,"lon":-0.1231278},{"type":"node","id":4194431956,"lat":51.5055611,"lon":-0.1228906},{"type":"node","id":4194431957,"lat":51.5057882,"lon":-0.1227953},{"type":"node","id":4697525082,"lat":51.5077378,"lon":-0.1215633},{"type":"node","id":4709924030,"lat":51.506378,"lon":-0.1225828},{"type":"node","id":4709924031,"lat":51.5064319,"lon":-0.1226153},{"type":"node","id":4709924035,"lat":51.5063643,"lon":-0.1224625},{"type":"node","id":4709924037,"lat":51.5063556,"lon":-0.1225863},{"type":"node","id":4709924044,"lat":51.5046676,"lon":-0.1233141},{"type":"node","id":4897627184,"lat":51.5066246,"lon":-0.1233891},{"type":"node","id":4897627187,"lat":51.5064389,"lon":-0.1236408},{"type":"node","id":5153538730,"lat":51.5080223,"lon":-0.1242306},{"type":"node","id":5341902060,"lat":51.506201,"lon":-0.1242934},{"type":"node","id":5677722957,"lat":51.5077066,"lon":-0.1231765},{"type":"node","id":5677722961,"lat":51.5079578,"lon":-0.1239035},{"type":"node","id":5677722962,"lat":51.5080193,"lon":-0.1240162},{"type":"node","id":5677722963,"lat":51.5079927,"lon":-0.1240559},{"type":"node","id":5677722964,"lat":51.5079305,"lon":-0.123942},{"type":"node","id":5677722966,"lat":51.5048217,"lon":-0.1237885},{"type":"node","id":5742329950,"lat":51.5046387,"lon":-0.1230736},{"type":"node","id":5742329951,"lat":51.5046114,"lon":-0.1230813},{"type":"node","id":5742332403,"lat":51.504711,"lon":-0.1233606},{"type":"node","id":5742332404,"lat":51.5046932,"lon":-0.1234634},{"type":"node","id":5742332405,"lat":51.5047189,"lon":-0.1234121},{"type":"node","id":6173148413,"lat":51.5076231,"lon":-0.122728},{"type":"node","id":6173148414,"lat":51.507596,"lon":-0.1227657},{"type":"node","id":6173148415,"lat":51.5077444,"lon":-0.1230723},{"type":"node","id":6173148416,"lat":51.5076897,"lon":-0.1231467},{"type":"node","id":6173148417,"lat":51.5077579,"lon":-0.1230532},{"type":"node","id":6173148418,"lat":51.5077679,"lon":-0.1230391},{"type":"node","id":6173148419,"lat":51.5077804,"lon":-0.1230217},{"type":"node","id":107783,"lat":51.5085723,"lon":-0.1272575},{"type":"node","id":107785,"lat":51.5094514,"lon":-0.1273874},{"type":"node","id":107786,"lat":51.5101873,"lon":-0.1282732},{"type":"node","id":107790,"lat":51.511322,"lon":-0.1283895},{"type":"node","id":107791,"lat":51.5118562,"lon":-0.1283797},{"type":"node","id":107792,"lat":51.513281,"lon":-0.1290796},{"type":"node","id":107795,"lat":51.5117551,"lon":-0.1293685},{"type":"node","id":107803,"lat":51.5122405,"lon":-0.1298183},{"type":"node","id":107805,"lat":51.5127816,"lon":-0.129931},{"type":"node","id":107806,"lat":51.5124727,"lon":-0.1306862},{"type":"node","id":108065,"lat":51.5138807,"lon":-0.1282829},{"type":"node","id":109620,"lat":51.5082635,"lon":-0.1290613},{"type":"node","id":109621,"lat":51.5084371,"lon":-0.1282184},{"type":"node","id":9512917,"lat":51.5098278,"lon":-0.131028},{"type":"node","id":9512921,"lat":51.5107986,"lon":-0.1302},{"type":"node","id":9512922,"lat":51.5081488,"lon":-0.1297776},{"type":"node","id":9512939,"lat":51.5134797,"lon":-0.1304698},{"type":"node","id":9512940,"lat":51.5125947,"lon":-0.1271411},{"type":"node","id":9513062,"lat":51.5129287,"lon":-0.1271565},{"type":"node","id":10574748,"lat":51.5098175,"lon":-0.1279834},{"type":"node","id":10574749,"lat":51.5098127,"lon":-0.1254285},{"type":"node","id":21554363,"lat":51.514254,"lon":-0.1278038},{"type":"node","id":21554364,"lat":51.5132922,"lon":-0.1263614},{"type":"node","id":21645238,"lat":51.5138574,"lon":-0.1294521},{"type":"node","id":21665031,"lat":51.5134793,"lon":-0.1296149},{"type":"node","id":21665110,"lat":51.5133278,"lon":-0.1296461},{"type":"node","id":21665118,"lat":51.5134749,"lon":-0.1294408},{"type":"node","id":21665363,"lat":51.5121292,"lon":-0.1308855},{"type":"node","id":21665367,"lat":51.5121687,"lon":-0.1300318},{"type":"node","id":21665372,"lat":51.5119226,"lon":-0.1284646},{"type":"node","id":21665374,"lat":51.511321,"lon":-0.1310544},{"type":"node","id":21665376,"lat":51.5115663,"lon":-0.1303828},{"type":"node","id":21665377,"lat":51.5116836,"lon":-0.129898},{"type":"node","id":21665378,"lat":51.5115216,"lon":-0.1305099},{"type":"node","id":21665382,"lat":51.5115002,"lon":-0.129277},{"type":"node","id":21665384,"lat":51.5112264,"lon":-0.1291967},{"type":"node","id":21665578,"lat":51.5128699,"lon":-0.1309673},{"type":"node","id":21665580,"lat":51.5134667,"lon":-0.1291897},{"type":"node","id":21665581,"lat":51.512586,"lon":-0.1303944},{"type":"node","id":21665583,"lat":51.511951,"lon":-0.1295368},{"type":"node","id":21665584,"lat":51.511142,"lon":-0.1291636},{"type":"node","id":21665585,"lat":51.5116901,"lon":-0.1283715},{"type":"node","id":21665712,"lat":51.5099344,"lon":-0.1304207},{"type":"node","id":21665713,"lat":51.5099686,"lon":-0.1300977},{"type":"node","id":21665714,"lat":51.5094376,"lon":-0.1299635},{"type":"node","id":21665716,"lat":51.5100255,"lon":-0.1295184},{"type":"node","id":21665717,"lat":51.5099506,"lon":-0.129389},{"type":"node","id":21665719,"lat":51.5098152,"lon":-0.1284646},{"type":"node","id":21665720,"lat":51.5099982,"lon":-0.12845},{"type":"node","id":21665724,"lat":51.5102185,"lon":-0.1305743},{"type":"node","id":21665745,"lat":51.5109123,"lon":-0.1286671},{"type":"node","id":21665775,"lat":51.5088689,"lon":-0.1298139},{"type":"node","id":21665859,"lat":51.5095225,"lon":-0.1285865},{"type":"node","id":21665863,"lat":51.5095463,"lon":-0.1284606},{"type":"node","id":21665909,"lat":51.5105992,"lon":-0.1308565},{"type":"node","id":21665910,"lat":51.5109846,"lon":-0.1296644},{"type":"node","id":21665911,"lat":51.5110477,"lon":-0.1295045},{"type":"node","id":21665912,"lat":51.5108353,"lon":-0.1284099},{"type":"node","id":21665913,"lat":51.5103986,"lon":-0.1283648},{"type":"node","id":25476045,"lat":51.5085346,"lon":-0.1246355},{"type":"node","id":25476049,"lat":51.5087885,"lon":-0.1250354},{"type":"node","id":25496892,"lat":51.5127993,"lon":-0.1288244},{"type":"node","id":25496899,"lat":51.5126797,"lon":-0.1271588},{"type":"node","id":25522907,"lat":51.5139675,"lon":-0.1251651},{"type":"node","id":25529877,"lat":51.5145167,"lon":-0.1281325},{"type":"node","id":25529933,"lat":51.5142197,"lon":-0.1285085},{"type":"node","id":25529938,"lat":51.5144872,"lon":-0.1287007},{"type":"node","id":25530236,"lat":51.5141971,"lon":-0.1295961},{"type":"node","id":25530237,"lat":51.5144949,"lon":-0.129721},{"type":"node","id":25533292,"lat":51.5129627,"lon":-0.1269083},{"type":"node","id":25533319,"lat":51.5126128,"lon":-0.1252418},{"type":"node","id":25533321,"lat":51.5123069,"lon":-0.1258314},{"type":"node","id":25533328,"lat":51.5119527,"lon":-0.1253942},{"type":"node","id":25533330,"lat":51.5119059,"lon":-0.1256386},{"type":"node","id":25533435,"lat":51.51221,"lon":-0.1247987},{"type":"node","id":25533506,"lat":51.5131158,"lon":-0.1275677},{"type":"node","id":25533578,"lat":51.5102204,"lon":-0.1269146},{"type":"node","id":25533580,"lat":51.5105164,"lon":-0.1250211},{"type":"node","id":25544113,"lat":51.5104394,"lon":-0.1257014},{"type":"node","id":25544114,"lat":51.510333,"lon":-0.1256798},{"type":"node","id":25544115,"lat":51.509507,"lon":-0.1269241},{"type":"node","id":25544116,"lat":51.509702,"lon":-0.1268893},{"type":"node","id":25544118,"lat":51.509624,"lon":-0.1258647},{"type":"node","id":25544119,"lat":51.5094765,"lon":-0.1258658},{"type":"node","id":25544120,"lat":51.5091724,"lon":-0.1258542},{"type":"node","id":25544123,"lat":51.5110466,"lon":-0.1248071},{"type":"node","id":25544126,"lat":51.5117549,"lon":-0.1260303},{"type":"node","id":25544127,"lat":51.5115762,"lon":-0.126238},{"type":"node","id":25544128,"lat":51.5116631,"lon":-0.1255761},{"type":"node","id":25544130,"lat":51.5114027,"lon":-0.1255354},{"type":"node","id":25544131,"lat":51.5121489,"lon":-0.1262196},{"type":"node","id":25544132,"lat":51.5119268,"lon":-0.1255588},{"type":"node","id":25544134,"lat":51.511974,"lon":-0.1252552},{"type":"node","id":25544135,"lat":51.5128583,"lon":-0.1247861},{"type":"node","id":25544136,"lat":51.5135557,"lon":-0.1257689},{"type":"node","id":25544138,"lat":51.5125444,"lon":-0.1253768},{"type":"node","id":25544141,"lat":51.5118708,"lon":-0.1272292},{"type":"node","id":25544142,"lat":51.5116807,"lon":-0.1274717},{"type":"node","id":25544143,"lat":51.5135637,"lon":-0.1281831},{"type":"node","id":25736633,"lat":51.5137938,"lon":-0.1251952},{"type":"node","id":26559655,"lat":51.5087838,"lon":-0.1302389},{"type":"node","id":26573136,"lat":51.5109673,"lon":-0.1284058},{"type":"node","id":26573143,"lat":51.5109664,"lon":-0.1277854},{"type":"node","id":26573147,"lat":51.5112351,"lon":-0.1277905},{"type":"node","id":26573152,"lat":51.5112341,"lon":-0.1283927},{"type":"node","id":26573672,"lat":51.5083245,"lon":-0.1306422},{"type":"node","id":26573673,"lat":51.5087947,"lon":-0.1310497},{"type":"node","id":26582693,"lat":51.5107126,"lon":-0.1284035},{"type":"node","id":26582694,"lat":51.5107424,"lon":-0.126945},{"type":"node","id":26631965,"lat":51.5112268,"lon":-0.1288388},{"type":"node","id":26846355,"lat":51.5123533,"lon":-0.1295014},{"type":"node","id":26846357,"lat":51.5127307,"lon":-0.1277196},{"type":"node","id":339925031,"lat":51.5082793,"lon":-0.1293015},{"type":"node","id":346430572,"lat":51.508488,"lon":-0.1250322},{"type":"node","id":346430573,"lat":51.5082822,"lon":-0.1253778},{"type":"node","id":346430574,"lat":51.5084256,"lon":-0.1256501},{"type":"node","id":348814978,"lat":51.510804,"lon":-0.1299655},{"type":"node","id":348894359,"lat":51.5106094,"lon":-0.1269163},{"type":"node","id":351509979,"lat":51.5086138,"lon":-0.1256001},{"type":"node","id":370712148,"lat":51.5100845,"lon":-0.1295267},{"type":"node","id":370712150,"lat":51.5099685,"lon":-0.1304395},{"type":"node","id":371564809,"lat":51.513941,"lon":-0.1245616},{"type":"node","id":371564813,"lat":51.514009,"lon":-0.1246596},{"type":"node","id":371564817,"lat":51.5140242,"lon":-0.1247213},{"type":"node","id":401501661,"lat":51.512923,"lon":-0.1296056},{"type":"node","id":401529120,"lat":51.512082,"lon":-0.1284137},{"type":"node","id":401529147,"lat":51.5139776,"lon":-0.1291173},{"type":"node","id":438170612,"lat":51.5141891,"lon":-0.1249218},{"type":"node","id":438170616,"lat":51.5143831,"lon":-0.1251537},{"type":"node","id":438170636,"lat":51.5143234,"lon":-0.1259447},{"type":"node","id":438170643,"lat":51.5139203,"lon":-0.1253959},{"type":"node","id":438170684,"lat":51.5136787,"lon":-0.1275004},{"type":"node","id":438170687,"lat":51.5134722,"lon":-0.1270865},{"type":"node","id":452487859,"lat":51.5107169,"lon":-0.1257457},{"type":"node","id":452488948,"lat":51.5126639,"lon":-0.1273005},{"type":"node","id":452488960,"lat":51.512847,"lon":-0.127013},{"type":"node","id":452488961,"lat":51.5127235,"lon":-0.1270896},{"type":"node","id":452514295,"lat":51.5081647,"lon":-0.1256055},{"type":"node","id":452514296,"lat":51.5082135,"lon":-0.1256923},{"type":"node","id":452514321,"lat":51.5080921,"lon":-0.1260205},{"type":"node","id":488815519,"lat":51.5128164,"lon":-0.1278583},{"type":"node","id":488815520,"lat":51.5126468,"lon":-0.1274753},{"type":"node","id":500486430,"lat":51.5142285,"lon":-0.1309923},{"type":"node","id":566201807,"lat":51.5083536,"lon":-0.1251776},{"type":"node","id":566202757,"lat":51.5089339,"lon":-0.130326},{"type":"node","id":617438710,"lat":51.5123064,"lon":-0.1296391},{"type":"node","id":639259210,"lat":51.5097147,"lon":-0.1300399},{"type":"node","id":639259211,"lat":51.5095111,"lon":-0.1294846},{"type":"node","id":692191040,"lat":51.5136642,"lon":-0.129979},{"type":"node","id":821341516,"lat":51.5103097,"lon":-0.1248324},{"type":"node","id":826724221,"lat":51.5085961,"lon":-0.124607},{"type":"node","id":1089089322,"lat":51.5129751,"lon":-0.1259558},{"type":"node","id":1089089333,"lat":51.512955,"lon":-0.1260043},{"type":"node","id":1094124402,"lat":51.5137736,"lon":-0.1293797},{"type":"node","id":1102994231,"lat":51.5102638,"lon":-0.1283122},{"type":"node","id":1102994236,"lat":51.5085773,"lon":-0.1270533},{"type":"node","id":1102994238,"lat":51.5095352,"lon":-0.1275155},{"type":"node","id":1102994241,"lat":51.5091888,"lon":-0.1272625},{"type":"node","id":1102995756,"lat":51.511415,"lon":-0.1283857},{"type":"node","id":1102995757,"lat":51.5111379,"lon":-0.1283961},{"type":"node","id":1102995758,"lat":51.5113692,"lon":-0.1282621},{"type":"node","id":1102996992,"lat":51.5133942,"lon":-0.1291404},{"type":"node","id":1102996994,"lat":51.513121,"lon":-0.1293038},{"type":"node","id":1102996997,"lat":51.5131669,"lon":-0.1290168},{"type":"node","id":1103212841,"lat":51.5133601,"lon":-0.1289822},{"type":"node","id":1103212860,"lat":51.5136115,"lon":-0.128661},{"type":"node","id":1103212861,"lat":51.513407,"lon":-0.1289239},{"type":"node","id":1104386138,"lat":51.5094712,"lon":-0.1261673},{"type":"node","id":1106056856,"lat":51.5086259,"lon":-0.1254902},{"type":"node","id":1119783271,"lat":51.5135034,"lon":-0.1301526},{"type":"node","id":1156355062,"lat":51.5098477,"lon":-0.1254714},{"type":"node","id":1188939747,"lat":51.5084645,"lon":-0.130068},{"type":"node","id":1239525639,"lat":51.508217,"lon":-0.129008},{"type":"node","id":1239525674,"lat":51.5081562,"lon":-0.1289698},{"type":"node","id":1239525687,"lat":51.5086014,"lon":-0.1262296},{"type":"node","id":1239525704,"lat":51.5082486,"lon":-0.1294722},{"type":"node","id":1239525737,"lat":51.5082846,"lon":-0.1291489},{"type":"node","id":1274040743,"lat":51.5085483,"lon":-0.1259348},{"type":"node","id":1274040746,"lat":51.5087283,"lon":-0.124846},{"type":"node","id":1278032910,"lat":51.5086629,"lon":-0.1247229},{"type":"node","id":1278032912,"lat":51.508153,"lon":-0.1261496},{"type":"node","id":1497135176,"lat":51.508444,"lon":-0.1250543},{"type":"node","id":1497135177,"lat":51.5084938,"lon":-0.1257237},{"type":"node","id":1497135179,"lat":51.5083378,"lon":-0.1254953},{"type":"node","id":1497135187,"lat":51.5087114,"lon":-0.1253286},{"type":"node","id":1497135188,"lat":51.5083212,"lon":-0.1252348},{"type":"node","id":1533393315,"lat":51.5117592,"lon":-0.1269759},{"type":"node","id":1614846675,"lat":51.5101062,"lon":-0.1247733},{"type":"node","id":1616122139,"lat":51.5138641,"lon":-0.1250227},{"type":"node","id":1616122178,"lat":51.514579,"lon":-0.1245721},{"type":"node","id":1637789493,"lat":51.5082855,"lon":-0.1290194},{"type":"node","id":1637789500,"lat":51.5083161,"lon":-0.1289577},{"type":"node","id":1637789501,"lat":51.5083647,"lon":-0.1281863},{"type":"node","id":1637789537,"lat":51.508526,"lon":-0.127622},{"type":"node","id":1637789538,"lat":51.5085551,"lon":-0.1273889},{"type":"node","id":1637789598,"lat":51.5092579,"lon":-0.1287953},{"type":"node","id":1637789599,"lat":51.5092693,"lon":-0.1289963},{"type":"node","id":1637789602,"lat":51.5092913,"lon":-0.1290366},{"type":"node","id":1637789604,"lat":51.5093392,"lon":-0.1290688},{"type":"node","id":1637789608,"lat":51.5094255,"lon":-0.1273622},{"type":"node","id":1637789617,"lat":51.5095255,"lon":-0.129132},{"type":"node","id":1637789621,"lat":51.5095284,"lon":-0.1285009},{"type":"node","id":1637789629,"lat":51.5096702,"lon":-0.1277731},{"type":"node","id":1637789635,"lat":51.5100047,"lon":-0.1281611},{"type":"node","id":1637789636,"lat":51.5100838,"lon":-0.1284319},{"type":"node","id":1637831255,"lat":51.5088988,"lon":-0.1297731},{"type":"node","id":1645919670,"lat":51.508286,"lon":-0.1253169},{"type":"node","id":1645919673,"lat":51.508533,"lon":-0.1259106},{"type":"node","id":1678452739,"lat":51.5094522,"lon":-0.1270223},{"type":"node","id":1678452740,"lat":51.5094603,"lon":-0.1271395},{"type":"node","id":1678452741,"lat":51.5094585,"lon":-0.1272748},{"type":"node","id":1678452742,"lat":51.5094675,"lon":-0.126724},{"type":"node","id":1678452743,"lat":51.5095185,"lon":-0.1270155},{"type":"node","id":1678452744,"lat":51.5095169,"lon":-0.1273453},{"type":"node","id":1678452745,"lat":51.5094946,"lon":-0.1274344},{"type":"node","id":1678452747,"lat":51.5095293,"lon":-0.1272375},{"type":"node","id":1678452749,"lat":51.5095326,"lon":-0.1271421},{"type":"node","id":1678452751,"lat":51.5107974,"lon":-0.1269548},{"type":"node","id":1678452752,"lat":51.5108744,"lon":-0.1269811},{"type":"node","id":1678452754,"lat":51.5110034,"lon":-0.1270252},{"type":"node","id":1678452757,"lat":51.5109925,"lon":-0.1258143},{"type":"node","id":1678452758,"lat":51.5110396,"lon":-0.1270375},{"type":"node","id":1678452766,"lat":51.511128,"lon":-0.1258469},{"type":"node","id":1678452769,"lat":51.5111585,"lon":-0.1255797},{"type":"node","id":1678452776,"lat":51.5112254,"lon":-0.1253489},{"type":"node","id":1678452778,"lat":51.5113011,"lon":-0.1251874},{"type":"node","id":1678452781,"lat":51.511404,"lon":-0.1271201},{"type":"node","id":1678452807,"lat":51.5137037,"lon":-0.1270287},{"type":"node","id":1678452811,"lat":51.5137083,"lon":-0.1269575},{"type":"node","id":1678452813,"lat":51.5137274,"lon":-0.1269215},{"type":"node","id":1678452814,"lat":51.5137469,"lon":-0.1270993},{"type":"node","id":1678452820,"lat":51.5138111,"lon":-0.1269366},{"type":"node","id":1678452821,"lat":51.5137974,"lon":-0.1270877},{"type":"node","id":1678452823,"lat":51.5138263,"lon":-0.1269947},{"type":"node","id":1694551549,"lat":51.5144102,"lon":-0.1257557},{"type":"node","id":1700227411,"lat":51.5141932,"lon":-0.1278838},{"type":"node","id":1729508660,"lat":51.5102971,"lon":-0.126911},{"type":"node","id":1835289526,"lat":51.5092861,"lon":-0.1285555},{"type":"node","id":1835298184,"lat":51.5109578,"lon":-0.1297269},{"type":"node","id":1843385996,"lat":51.5112737,"lon":-0.1250768},{"type":"node","id":1843386487,"lat":51.5138938,"lon":-0.1249654},{"type":"node","id":1857917757,"lat":51.5116674,"lon":-0.1259735},{"type":"node","id":1857917764,"lat":51.5117921,"lon":-0.1260301},{"type":"node","id":1857940093,"lat":51.5142612,"lon":-0.1260635},{"type":"node","id":1857957494,"lat":51.5139081,"lon":-0.1269825},{"type":"node","id":1938580050,"lat":51.5088721,"lon":-0.1272197},{"type":"node","id":1938580054,"lat":51.5094617,"lon":-0.126903},{"type":"node","id":1938580066,"lat":51.5112863,"lon":-0.125119},{"type":"node","id":1938580069,"lat":51.5115959,"lon":-0.1271198},{"type":"node","id":1938580076,"lat":51.5119139,"lon":-0.1269226},{"type":"node","id":1938580079,"lat":51.511945,"lon":-0.1267696},{"type":"node","id":1938580081,"lat":51.5119689,"lon":-0.1270846},{"type":"node","id":1938580082,"lat":51.5131063,"lon":-0.1283016},{"type":"node","id":1938580083,"lat":51.5135547,"lon":-0.1302743},{"type":"node","id":1947257506,"lat":51.5086193,"lon":-0.1255508},{"type":"node","id":1947257509,"lat":51.5090211,"lon":-0.1272356},{"type":"node","id":1947257512,"lat":51.5092847,"lon":-0.1272943},{"type":"node","id":1947257514,"lat":51.5094638,"lon":-0.1272136},{"type":"node","id":1947257515,"lat":51.5105259,"lon":-0.1284012},{"type":"node","id":1953893710,"lat":51.5126623,"lon":-0.1275688},{"type":"node","id":1953893711,"lat":51.5126894,"lon":-0.1276373},{"type":"node","id":1953893713,"lat":51.5127359,"lon":-0.1278808},{"type":"node","id":1953893715,"lat":51.5130703,"lon":-0.126773},{"type":"node","id":1953893718,"lat":51.513156,"lon":-0.1266193},{"type":"node","id":2048921781,"lat":51.5123558,"lon":-0.1258892},{"type":"node","id":2048921782,"lat":51.5126249,"lon":-0.1269394},{"type":"node","id":2048921785,"lat":51.5126283,"lon":-0.1267279},{"type":"node","id":2064602915,"lat":51.5120282,"lon":-0.1261312},{"type":"node","id":2064602918,"lat":51.5120862,"lon":-0.1261576},{"type":"node","id":2105659653,"lat":51.5090425,"lon":-0.1246758},{"type":"node","id":2265298879,"lat":51.5137127,"lon":-0.1270599},{"type":"node","id":2265298881,"lat":51.5138187,"lon":-0.1269541},{"type":"node","id":2265298883,"lat":51.5137729,"lon":-0.1271021},{"type":"node","id":2265298885,"lat":51.5138239,"lon":-0.1269737},{"type":"node","id":2265298886,"lat":51.5137019,"lon":-0.1269924},{"type":"node","id":2265298887,"lat":51.5137711,"lon":-0.126903},{"type":"node","id":2265298888,"lat":51.513793,"lon":-0.1269138},{"type":"node","id":2265298890,"lat":51.5137481,"lon":-0.1269057},{"type":"node","id":2265298892,"lat":51.513814,"lon":-0.1270631},{"type":"node","id":2265298894,"lat":51.5138241,"lon":-0.1270307},{"type":"node","id":2265298896,"lat":51.5134877,"lon":-0.1281425},{"type":"node","id":2265298898,"lat":51.5136288,"lon":-0.127046},{"type":"node","id":2265298900,"lat":51.5137277,"lon":-0.1270844},{"type":"node","id":2265306459,"lat":51.5139239,"lon":-0.1253685},{"type":"node","id":2265306462,"lat":51.5138007,"lon":-0.1251815},{"type":"node","id":2321071816,"lat":51.5090389,"lon":-0.1272385},{"type":"node","id":2321071820,"lat":51.5090567,"lon":-0.1258498},{"type":"node","id":2338578962,"lat":51.5088038,"lon":-0.1250085},{"type":"node","id":2338578968,"lat":51.5088126,"lon":-0.125137},{"type":"node","id":2338578984,"lat":51.5088338,"lon":-0.1250935},{"type":"node","id":2345097899,"lat":51.5089489,"lon":-0.1303356},{"type":"node","id":2345097921,"lat":51.5090032,"lon":-0.1303677},{"type":"node","id":2345097938,"lat":51.5090631,"lon":-0.1304067},{"type":"node","id":2345097948,"lat":51.5091172,"lon":-0.1304476},{"type":"node","id":2356142512,"lat":51.513363,"lon":-0.129594},{"type":"node","id":2356142513,"lat":51.5134155,"lon":-0.12959},{"type":"node","id":2450730396,"lat":51.5140494,"lon":-0.1246745},{"type":"node","id":2450755716,"lat":51.5138599,"lon":-0.1271857},{"type":"node","id":2450755717,"lat":51.5132726,"lon":-0.1264025},{"type":"node","id":2450755718,"lat":51.5133311,"lon":-0.1263898},{"type":"node","id":2450755719,"lat":51.5133213,"lon":-0.1263014},{"type":"node","id":2456267967,"lat":51.5135004,"lon":-0.1288007},{"type":"node","id":2463446566,"lat":51.5109686,"lon":-0.1273922},{"type":"node","id":2476787985,"lat":51.5118765,"lon":-0.1270975},{"type":"node","id":2476805766,"lat":51.5118342,"lon":-0.1258658},{"type":"node","id":2476805770,"lat":51.5117203,"lon":-0.1260809},{"type":"node","id":2476805774,"lat":51.5116313,"lon":-0.1261943},{"type":"node","id":2482696992,"lat":51.5119284,"lon":-0.129516},{"type":"node","id":2482696993,"lat":51.5116993,"lon":-0.129336},{"type":"node","id":2564873525,"lat":51.5087732,"lon":-0.125004},{"type":"node","id":2564873531,"lat":51.5086681,"lon":-0.1258351},{"type":"node","id":2612448628,"lat":51.5085772,"lon":-0.1255779},{"type":"node","id":2612448640,"lat":51.5086103,"lon":-0.1259326},{"type":"node","id":2726834608,"lat":51.5082606,"lon":-0.1271138},{"type":"node","id":2726834609,"lat":51.5083856,"lon":-0.1272868},{"type":"node","id":2726834616,"lat":51.5084172,"lon":-0.1271535},{"type":"node","id":2726834619,"lat":51.5084792,"lon":-0.1273484},{"type":"node","id":2726834627,"lat":51.5085256,"lon":-0.1273548},{"type":"node","id":2726834628,"lat":51.5085383,"lon":-0.1271782},{"type":"node","id":2726834629,"lat":51.5085586,"lon":-0.127352},{"type":"node","id":2726834630,"lat":51.5085956,"lon":-0.1273434},{"type":"node","id":2726834636,"lat":51.5086241,"lon":-0.1273305},{"type":"node","id":2726834637,"lat":51.5086934,"lon":-0.1271845},{"type":"node","id":2726834638,"lat":51.5086941,"lon":-0.1272897},{"type":"node","id":2726834644,"lat":51.5087453,"lon":-0.1272624},{"type":"node","id":2726834645,"lat":51.5087803,"lon":-0.1271798},{"type":"node","id":2726834646,"lat":51.5087823,"lon":-0.1272516},{"type":"node","id":2726834647,"lat":51.5089586,"lon":-0.1272337},{"type":"node","id":2726836316,"lat":51.5080642,"lon":-0.127047},{"type":"node","id":2726836317,"lat":51.5084816,"lon":-0.1272896},{"type":"node","id":2726836318,"lat":51.5085231,"lon":-0.1272934},{"type":"node","id":2726836319,"lat":51.5085459,"lon":-0.1272908},{"type":"node","id":2726836320,"lat":51.5085619,"lon":-0.1272833},{"type":"node","id":2726836321,"lat":51.5085669,"lon":-0.1273501},{"type":"node","id":2726836322,"lat":51.5085736,"lon":-0.1271796},{"type":"node","id":2726838582,"lat":51.5084556,"lon":-0.1273329},{"type":"node","id":2726838583,"lat":51.508459,"lon":-0.1272889},{"type":"node","id":2726838586,"lat":51.5084679,"lon":-0.1271665},{"type":"node","id":2726838589,"lat":51.50865,"lon":-0.1271827},{"type":"node","id":2726838592,"lat":51.5086533,"lon":-0.1273135},{"type":"node","id":2726883549,"lat":51.5081311,"lon":-0.1287836},{"type":"node","id":2726883553,"lat":51.5081724,"lon":-0.1288003},{"type":"node","id":2726883566,"lat":51.5082225,"lon":-0.1287535},{"type":"node","id":2726883568,"lat":51.5082292,"lon":-0.128716},{"type":"node","id":2726883572,"lat":51.5083471,"lon":-0.1274767},{"type":"node","id":2726883574,"lat":51.5083919,"lon":-0.1274941},{"type":"node","id":2726883578,"lat":51.5084171,"lon":-0.1276191},{"type":"node","id":2726883585,"lat":51.5084235,"lon":-0.1275804},{"type":"node","id":2767561431,"lat":51.5130606,"lon":-0.1293895},{"type":"node","id":2767571255,"lat":51.5130915,"lon":-0.1293457},{"type":"node","id":2767582810,"lat":51.5133768,"lon":-0.128962},{"type":"node","id":2767582815,"lat":51.5130821,"lon":-0.1289688},{"type":"node","id":2837565785,"lat":51.5134944,"lon":-0.1298782},{"type":"node","id":2975783589,"lat":51.5086482,"lon":-0.1252792},{"type":"node","id":2975783590,"lat":51.5085482,"lon":-0.1250973},{"type":"node","id":3054882694,"lat":51.5135409,"lon":-0.1302564},{"type":"node","id":3085004818,"lat":51.5117021,"lon":-0.127108},{"type":"node","id":3085004821,"lat":51.511752,"lon":-0.1272721},{"type":"node","id":3085004825,"lat":51.5117774,"lon":-0.1270365},{"type":"node","id":3085004826,"lat":51.5117747,"lon":-0.1272117},{"type":"node","id":3085004827,"lat":51.5117836,"lon":-0.1270621},{"type":"node","id":3085004830,"lat":51.5118696,"lon":-0.1273014},{"type":"node","id":3092290901,"lat":51.508278,"lon":-0.1291057},{"type":"node","id":3092290902,"lat":51.5082822,"lon":-0.1291272},{"type":"node","id":3092290903,"lat":51.5082726,"lon":-0.1290843},{"type":"node","id":3092290904,"lat":51.5082455,"lon":-0.129034},{"type":"node","id":3092290905,"lat":51.5082568,"lon":-0.1290487},{"type":"node","id":3092290906,"lat":51.5082325,"lon":-0.1290212},{"type":"node","id":3092290907,"lat":51.5082872,"lon":-0.1291963},{"type":"node","id":3092290908,"lat":51.5082856,"lon":-0.1292486},{"type":"node","id":3192197694,"lat":51.5134458,"lon":-0.1270902},{"type":"node","id":3192197695,"lat":51.5131458,"lon":-0.1266364},{"type":"node","id":3192197696,"lat":51.5132247,"lon":-0.1267623},{"type":"node","id":3326326761,"lat":51.5083933,"lon":-0.1259023},{"type":"node","id":3326326769,"lat":51.5086065,"lon":-0.1260598},{"type":"node","id":3326326774,"lat":51.5086493,"lon":-0.1254481},{"type":"node","id":3326326775,"lat":51.508679,"lon":-0.1253909},{"type":"node","id":3326339636,"lat":51.5084919,"lon":-0.1258668},{"type":"node","id":3397689895,"lat":51.5080437,"lon":-0.1263773},{"type":"node","id":3397689896,"lat":51.508089,"lon":-0.1264592},{"type":"node","id":3397689897,"lat":51.5082354,"lon":-0.1261746},{"type":"node","id":3397689898,"lat":51.5085416,"lon":-0.1254692},{"type":"node","id":3552338821,"lat":51.5119788,"lon":-0.1293303},{"type":"node","id":3552338822,"lat":51.5119792,"lon":-0.1295635},{"type":"node","id":3552338823,"lat":51.5119797,"lon":-0.1293447},{"type":"node","id":3552338824,"lat":51.5119858,"lon":-0.1294479},{"type":"node","id":3552338869,"lat":51.512668,"lon":-0.1287503},{"type":"node","id":3552452839,"lat":51.5127958,"lon":-0.1287527},{"type":"node","id":3552453346,"lat":51.5132619,"lon":-0.128584},{"type":"node","id":3782162175,"lat":51.5137669,"lon":-0.125164},{"type":"node","id":3810553690,"lat":51.5082525,"lon":-0.1272532},{"type":"node","id":3810553691,"lat":51.5081477,"lon":-0.1270731},{"type":"node","id":3887905011,"lat":51.5114764,"lon":-0.1266811},{"type":"node","id":3887905012,"lat":51.5114771,"lon":-0.1268033},{"type":"node","id":3887905013,"lat":51.5114782,"lon":-0.1270007},{"type":"node","id":3887905014,"lat":51.5114789,"lon":-0.12712},{"type":"node","id":4189248295,"lat":51.5120928,"lon":-0.1302419},{"type":"node","id":4189248449,"lat":51.5121168,"lon":-0.1301754},{"type":"node","id":4347914539,"lat":51.508235,"lon":-0.1295138},{"type":"node","id":4350184736,"lat":51.5092504,"lon":-0.1272829},{"type":"node","id":4350184739,"lat":51.509219,"lon":-0.1272725},{"type":"node","id":4350193651,"lat":51.5084212,"lon":-0.1258526},{"type":"node","id":4350193652,"lat":51.50848,"lon":-0.125582},{"type":"node","id":4350193653,"lat":51.5084082,"lon":-0.1256175},{"type":"node","id":4350193655,"lat":51.5083583,"lon":-0.1257753},{"type":"node","id":4350193657,"lat":51.508389,"lon":-0.1255818},{"type":"node","id":4535154580,"lat":51.5086675,"lon":-0.1248613},{"type":"node","id":4784457197,"lat":51.5127014,"lon":-0.1262984},{"type":"node","id":4784457201,"lat":51.5127796,"lon":-0.126268},{"type":"node","id":4784457202,"lat":51.5127394,"lon":-0.1263329},{"type":"node","id":4784457203,"lat":51.5126305,"lon":-0.1265565},{"type":"node","id":4784457204,"lat":51.5128532,"lon":-0.1262008},{"type":"node","id":4784457205,"lat":51.5131373,"lon":-0.1256638},{"type":"node","id":4784457206,"lat":51.5130337,"lon":-0.1255293},{"type":"node","id":4784457207,"lat":51.5131345,"lon":-0.1253287},{"type":"node","id":4784457208,"lat":51.5130352,"lon":-0.1251998},{"type":"node","id":4784457209,"lat":51.5130684,"lon":-0.1251357},{"type":"node","id":4784457210,"lat":51.5130837,"lon":-0.1251041},{"type":"node","id":4784457211,"lat":51.5131681,"lon":-0.125274},{"type":"node","id":4784457212,"lat":51.5131835,"lon":-0.1252437},{"type":"node","id":4784457213,"lat":51.5129856,"lon":-0.1259666},{"type":"node","id":4784457214,"lat":51.5130096,"lon":-0.1259155},{"type":"node","id":4826960087,"lat":51.5116184,"lon":-0.1259185},{"type":"node","id":4826960088,"lat":51.5116227,"lon":-0.1259504},{"type":"node","id":4826960089,"lat":51.5116256,"lon":-0.1258735},{"type":"node","id":4853048350,"lat":51.5094925,"lon":-0.1295817},{"type":"node","id":5059333678,"lat":51.5134741,"lon":-0.128837},{"type":"node","id":5059333680,"lat":51.5134235,"lon":-0.1291634},{"type":"node","id":5364578859,"lat":51.5098417,"lon":-0.1280064},{"type":"node","id":5364578860,"lat":51.5105591,"lon":-0.1284016},{"type":"node","id":5364578861,"lat":51.5109927,"lon":-0.1284044},{"type":"node","id":5364578862,"lat":51.511599,"lon":-0.1283762},{"type":"node","id":5364578863,"lat":51.513489,"lon":-0.1292033},{"type":"node","id":5364578864,"lat":51.5136864,"lon":-0.1285557},{"type":"node","id":5364578865,"lat":51.5124973,"lon":-0.1286574},{"type":"node","id":5364578866,"lat":51.5143015,"lon":-0.1296399},{"type":"node","id":5364578867,"lat":51.5126406,"lon":-0.130243},{"type":"node","id":5386341215,"lat":51.5099822,"lon":-0.1255622},{"type":"node","id":5386341216,"lat":51.5099755,"lon":-0.1256179},{"type":"node","id":5386345349,"lat":51.509876,"lon":-0.1255086},{"type":"node","id":5386345350,"lat":51.5099085,"lon":-0.1255374},{"type":"node","id":5387300680,"lat":51.5098952,"lon":-0.125282},{"type":"node","id":5393414954,"lat":51.5115433,"lon":-0.1304482},{"type":"node","id":5393414955,"lat":51.5109173,"lon":-0.1298471},{"type":"node","id":5398463416,"lat":51.5126441,"lon":-0.1277589},{"type":"node","id":5398463417,"lat":51.5127324,"lon":-0.1277777},{"type":"node","id":5398463418,"lat":51.5126517,"lon":-0.1277856},{"type":"node","id":5398463419,"lat":51.5127307,"lon":-0.1277569},{"type":"node","id":5398463420,"lat":51.5126437,"lon":-0.1276088},{"type":"node","id":5398465121,"lat":51.5126568,"lon":-0.1275728},{"type":"node","id":5398465122,"lat":51.5126696,"lon":-0.1277921},{"type":"node","id":5398465123,"lat":51.5126472,"lon":-0.1275883},{"type":"node","id":5398465124,"lat":51.5134445,"lon":-0.1291767},{"type":"node","id":5404152138,"lat":51.5145186,"lon":-0.1245437},{"type":"node","id":5420953856,"lat":51.5138447,"lon":-0.1249437},{"type":"node","id":5420953859,"lat":51.5137877,"lon":-0.12521},{"type":"node","id":5444031562,"lat":51.5116121,"lon":-0.1262093},{"type":"node","id":5444031569,"lat":51.5116396,"lon":-0.125569},{"type":"node","id":5444031577,"lat":51.5117965,"lon":-0.1259525},{"type":"node","id":5444031578,"lat":51.51174,"lon":-0.1260262},{"type":"node","id":5444031596,"lat":51.5117738,"lon":-0.1260281},{"type":"node","id":5444031598,"lat":51.511727,"lon":-0.1260202},{"type":"node","id":5444031599,"lat":51.5117035,"lon":-0.1259917},{"type":"node","id":5444031619,"lat":51.5118645,"lon":-0.1257755},{"type":"node","id":5444033120,"lat":51.5127984,"lon":-0.1256363},{"type":"node","id":5444033150,"lat":51.5122501,"lon":-0.1259619},{"type":"node","id":5444033221,"lat":51.5128853,"lon":-0.1254737},{"type":"node","id":5444033222,"lat":51.5128649,"lon":-0.1255114},{"type":"node","id":5444033229,"lat":51.5127739,"lon":-0.1256835},{"type":"node","id":5444033230,"lat":51.5127876,"lon":-0.1256567},{"type":"node","id":5444033231,"lat":51.5128095,"lon":-0.1256159},{"type":"node","id":5444227620,"lat":51.5094552,"lon":-0.1269651},{"type":"node","id":5444227721,"lat":51.5094812,"lon":-0.1269537},{"type":"node","id":5444235310,"lat":51.5098088,"lon":-0.1280487},{"type":"node","id":5444235311,"lat":51.5098105,"lon":-0.1280192},{"type":"node","id":5451506147,"lat":51.5117417,"lon":-0.1269056},{"type":"node","id":5451506148,"lat":51.5120056,"lon":-0.1270858},{"type":"node","id":5451506149,"lat":51.5118686,"lon":-0.1273755},{"type":"node","id":5452395149,"lat":51.5129291,"lon":-0.1272132},{"type":"node","id":5452395150,"lat":51.5129387,"lon":-0.1272862},{"type":"node","id":5454295817,"lat":51.5144009,"lon":-0.1257956},{"type":"node","id":5479634639,"lat":51.5114884,"lon":-0.1283819},{"type":"node","id":5479634644,"lat":51.5117331,"lon":-0.1283705},{"type":"node","id":5479634674,"lat":51.5114114,"lon":-0.1280982},{"type":"node","id":5479634675,"lat":51.511423,"lon":-0.1280651},{"type":"node","id":5479634683,"lat":51.5116471,"lon":-0.1283621},{"type":"node","id":5479634692,"lat":51.5115982,"lon":-0.1283576},{"type":"node","id":5479634694,"lat":51.5114071,"lon":-0.1281755},{"type":"node","id":5479634695,"lat":51.5114056,"lon":-0.1281253},{"type":"node","id":5479634696,"lat":51.511358,"lon":-0.1281594},{"type":"node","id":5479634697,"lat":51.5115078,"lon":-0.1284153},{"type":"node","id":5479634701,"lat":51.5114671,"lon":-0.1284123},{"type":"node","id":5479634704,"lat":51.51134,"lon":-0.1282102},{"type":"node","id":5479635425,"lat":51.5120901,"lon":-0.12847},{"type":"node","id":5479635426,"lat":51.5120498,"lon":-0.1284571},{"type":"node","id":5479635443,"lat":51.5119924,"lon":-0.1284432},{"type":"node","id":5479635444,"lat":51.5120225,"lon":-0.1284511},{"type":"node","id":5479635445,"lat":51.5118724,"lon":-0.1284264},{"type":"node","id":5479635446,"lat":51.5119027,"lon":-0.1284302},{"type":"node","id":5479635447,"lat":51.5118002,"lon":-0.1284164},{"type":"node","id":5479635448,"lat":51.5117711,"lon":-0.1284126},{"type":"node","id":5479635487,"lat":51.5115518,"lon":-0.1276585},{"type":"node","id":5479635489,"lat":51.511526,"lon":-0.1277308},{"type":"node","id":5479635550,"lat":51.511598,"lon":-0.1278801},{"type":"node","id":5479635551,"lat":51.5115196,"lon":-0.1279766},{"type":"node","id":5479635552,"lat":51.5115856,"lon":-0.1278617},{"type":"node","id":5479635553,"lat":51.5115733,"lon":-0.1278432},{"type":"node","id":5479635554,"lat":51.5115077,"lon":-0.1279592},{"type":"node","id":5479635556,"lat":51.5115315,"lon":-0.127994},{"type":"node","id":5479635577,"lat":51.5114997,"lon":-0.1285458},{"type":"node","id":5479635578,"lat":51.5115024,"lon":-0.1285934},{"type":"node","id":5479635579,"lat":51.511489,"lon":-0.1285471},{"type":"node","id":5479635580,"lat":51.5114918,"lon":-0.1285952},{"type":"node","id":5479635598,"lat":51.5114695,"lon":-0.1285988},{"type":"node","id":5479635599,"lat":51.5114668,"lon":-0.1285507},{"type":"node","id":5479635600,"lat":51.511479,"lon":-0.1285972},{"type":"node","id":5479635601,"lat":51.5114763,"lon":-0.1285492},{"type":"node","id":5479635610,"lat":51.5114647,"lon":-0.1285117},{"type":"node","id":5479635611,"lat":51.5114626,"lon":-0.128474},{"type":"node","id":5479635612,"lat":51.5114741,"lon":-0.1285102},{"type":"node","id":5479635613,"lat":51.511472,"lon":-0.1284728},{"type":"node","id":5479635626,"lat":51.5116501,"lon":-0.1283542},{"type":"node","id":5479635627,"lat":51.5117505,"lon":-0.1283102},{"type":"node","id":5479635629,"lat":51.5116452,"lon":-0.1283252},{"type":"node","id":5479635630,"lat":51.5117448,"lon":-0.1282806},{"type":"node","id":5479635631,"lat":51.5116404,"lon":-0.1282968},{"type":"node","id":5479635632,"lat":51.511739,"lon":-0.128251},{"type":"node","id":5479635681,"lat":51.5116251,"lon":-0.1281228},{"type":"node","id":5479635682,"lat":51.5116138,"lon":-0.1280981},{"type":"node","id":5479635683,"lat":51.5116495,"lon":-0.1280949},{"type":"node","id":5479635685,"lat":51.5116387,"lon":-0.1280709},{"type":"node","id":5479635691,"lat":51.5117036,"lon":-0.1281708},{"type":"node","id":5479635692,"lat":51.5116742,"lon":-0.1281155},{"type":"node","id":5479635693,"lat":51.511713,"lon":-0.1281569},{"type":"node","id":5479635694,"lat":51.5116833,"lon":-0.1281022},{"type":"node","id":5479635711,"lat":51.5116218,"lon":-0.1279593},{"type":"node","id":5479635712,"lat":51.5116451,"lon":-0.1280146},{"type":"node","id":5479635713,"lat":51.5116107,"lon":-0.1279693},{"type":"node","id":5479635714,"lat":51.5116343,"lon":-0.1280242},{"type":"node","id":5479635762,"lat":51.5113483,"lon":-0.128086},{"type":"node","id":5479635763,"lat":51.5113706,"lon":-0.1281091},{"type":"node","id":5479635764,"lat":51.5113405,"lon":-0.1281068},{"type":"node","id":5479635765,"lat":51.5113631,"lon":-0.1281296},{"type":"node","id":5479635766,"lat":51.511282,"lon":-0.1282049},{"type":"node","id":5479635767,"lat":51.5112844,"lon":-0.1281286},{"type":"node","id":5479635768,"lat":51.5112679,"lon":-0.1282041},{"type":"node","id":5479635769,"lat":51.5112706,"lon":-0.1281281},{"type":"node","id":5479635834,"lat":51.5115942,"lon":-0.1275372},{"type":"node","id":5479635835,"lat":51.5116141,"lon":-0.1274806},{"type":"node","id":5479635836,"lat":51.5115718,"lon":-0.1276018},{"type":"node","id":5479635837,"lat":51.5115057,"lon":-0.1277884},{"type":"node","id":5481389678,"lat":51.5082684,"lon":-0.1281459},{"type":"node","id":5597434617,"lat":51.5119124,"lon":-0.1285226},{"type":"node","id":5597434619,"lat":51.5119262,"lon":-0.1283876},{"type":"node","id":5695662022,"lat":51.5133948,"lon":-0.1291407},{"type":"node","id":5811221631,"lat":51.5135597,"lon":-0.1282556},{"type":"node","id":5811231652,"lat":51.5135797,"lon":-0.1281503},{"type":"node","id":5811231853,"lat":51.5135412,"lon":-0.1281917},{"type":"node","id":5811231856,"lat":51.5135218,"lon":-0.1281831},{"type":"node","id":5859926240,"lat":51.5119573,"lon":-0.1253545},{"type":"node","id":5859926245,"lat":51.5114872,"lon":-0.1247724},{"type":"node","id":5877184121,"lat":51.5125056,"lon":-0.1306015},{"type":"node","id":5877184122,"lat":51.5128209,"lon":-0.1298405},{"type":"node","id":5932856686,"lat":51.5128811,"lon":-0.1308272},{"type":"node","id":5932856687,"lat":51.5129132,"lon":-0.1308509},{"type":"node","id":5932856689,"lat":51.5127999,"lon":-0.1310552},{"type":"node","id":5932856690,"lat":51.5128391,"lon":-0.1309451},{"type":"node","id":5932856691,"lat":51.5129163,"lon":-0.1308175},{"type":"node","id":5932856692,"lat":51.5128328,"lon":-0.1310641},{"type":"node","id":5932883501,"lat":51.5131069,"lon":-0.1302548},{"type":"node","id":5932883502,"lat":51.5131296,"lon":-0.1302784},{"type":"node","id":5932883503,"lat":51.5131067,"lon":-0.1303323},{"type":"node","id":5932883504,"lat":51.5131372,"lon":-0.1302347},{"type":"node","id":5932883505,"lat":51.5130906,"lon":-0.130349},{"type":"node","id":5932883506,"lat":51.5130861,"lon":-0.1303172},{"type":"node","id":5932899700,"lat":51.5139225,"lon":-0.1294545},{"type":"node","id":5932899701,"lat":51.5139111,"lon":-0.1294181},{"type":"node","id":5932899702,"lat":51.5138056,"lon":-0.1294186},{"type":"node","id":5932899703,"lat":51.5138152,"lon":-0.1293619},{"type":"node","id":5932899704,"lat":51.5138937,"lon":-0.1294696},{"type":"node","id":5932899705,"lat":51.5138766,"lon":-0.1293896},{"type":"node","id":5932959888,"lat":51.5118285,"lon":-0.1271033},{"type":"node","id":5932959890,"lat":51.5117314,"lon":-0.1271059},{"type":"node","id":5932959891,"lat":51.5117883,"lon":-0.1271035},{"type":"node","id":5932959892,"lat":51.5119007,"lon":-0.1269946},{"type":"node","id":5932959894,"lat":51.5118446,"lon":-0.1271023},{"type":"node","id":5932959895,"lat":51.5118708,"lon":-0.1271653},{"type":"node","id":5932959902,"lat":51.5119357,"lon":-0.127087},{"type":"node","id":5932959903,"lat":51.5117852,"lon":-0.1271637},{"type":"node","id":5932960717,"lat":51.5119319,"lon":-0.1268214},{"type":"node","id":6064937880,"lat":51.5082086,"lon":-0.1281216},{"type":"node","id":6064937882,"lat":51.5081,"lon":-0.128768},{"type":"node","id":6064937884,"lat":51.508408,"lon":-0.1276653},{"type":"node","id":6064939285,"lat":51.5082358,"lon":-0.1286723},{"type":"node","id":6064939288,"lat":51.5083192,"lon":-0.1274635},{"type":"node","id":6064939289,"lat":51.5085128,"lon":-0.1277107},{"type":"node","id":6064939290,"lat":51.5083551,"lon":-0.1287195},{"type":"node","id":6064939291,"lat":51.5081402,"lon":-0.1285286},{"type":"node","id":6064939888,"lat":51.5082678,"lon":-0.127778},{"type":"node","id":6173732688,"lat":51.5086089,"lon":-0.1281466},{"type":"node","id":6173732689,"lat":51.5085558,"lon":-0.1284803},{"type":"node","id":6173732694,"lat":51.5089511,"lon":-0.1297892},{"type":"node","id":6173732697,"lat":51.508576,"lon":-0.1283572},{"type":"node","id":6173732717,"lat":51.5086948,"lon":-0.1294453},{"type":"node","id":6173732719,"lat":51.5085145,"lon":-0.1294908},{"type":"node","id":6173732720,"lat":51.5086955,"lon":-0.1295709},{"type":"node","id":6173732723,"lat":51.5085903,"lon":-0.1282648},{"type":"node","id":6173732724,"lat":51.5084998,"lon":-0.1277979},{"type":"node","id":6173732727,"lat":51.5085597,"lon":-0.128458},{"type":"node","id":6173732729,"lat":51.5087831,"lon":-0.1294872},{"type":"node","id":6173740395,"lat":51.5093611,"lon":-0.1302388},{"type":"node","id":6173754809,"lat":51.51081,"lon":-0.1298633},{"type":"node","id":6173783031,"lat":51.5113558,"lon":-0.1250655},{"type":"node","id":6173783033,"lat":51.5107707,"lon":-0.1257591},{"type":"node","id":6173783034,"lat":51.5106495,"lon":-0.1258358},{"type":"node","id":6173783036,"lat":51.5106637,"lon":-0.1257337},{"type":"node","id":6173783039,"lat":51.5112002,"lon":-0.1254164},{"type":"node","id":6173783040,"lat":51.5104891,"lon":-0.125798},{"type":"node","id":6214704100,"lat":51.5093537,"lon":-0.1306325},{"type":"node","id":6214704103,"lat":51.5094866,"lon":-0.1307639},{"type":"node","id":6214704104,"lat":51.5098078,"lon":-0.1310154},{"type":"node","id":6214704105,"lat":51.5095063,"lon":-0.1307789},{"type":"node","id":6214704106,"lat":51.5094629,"lon":-0.1307461},{"type":"node","id":6214704108,"lat":51.5097895,"lon":-0.1310126},{"type":"node","id":6214704109,"lat":51.5094406,"lon":-0.130813},{"type":"node","id":6214704110,"lat":51.5094663,"lon":-0.1308323},{"type":"node","id":6214704111,"lat":51.5097646,"lon":-0.131059},{"type":"node","id":6214704112,"lat":51.5093479,"lon":-0.1306616},{"type":"node","id":6302549987,"lat":51.5112349,"lon":-0.1282657},{"type":"node","id":6311518054,"lat":51.5092483,"lon":-0.1305478},{"type":"node","id":6311518055,"lat":51.5093751,"lon":-0.1301619},{"type":"node","id":6315809114,"lat":51.508192,"lon":-0.1260786},{"type":"node","id":107710,"lat":51.5113631,"lon":-0.1192255},{"type":"node","id":107715,"lat":51.5105677,"lon":-0.1211007},{"type":"node","id":107716,"lat":51.5110821,"lon":-0.1197214},{"type":"node","id":108895,"lat":51.5120546,"lon":-0.1189505},{"type":"node","id":108896,"lat":51.5129401,"lon":-0.1181008},{"type":"node","id":108899,"lat":51.5141677,"lon":-0.1179995},{"type":"node","id":109624,"lat":51.5093363,"lon":-0.1239796},{"type":"node","id":109836,"lat":51.5126513,"lon":-0.1211916},{"type":"node","id":109838,"lat":51.5099588,"lon":-0.1226011},{"type":"node","id":9475243,"lat":51.512203,"lon":-0.117966},{"type":"node","id":9475518,"lat":51.5137124,"lon":-0.1200753},{"type":"node","id":9512908,"lat":51.5107566,"lon":-0.1244755},{"type":"node","id":9513010,"lat":51.5132606,"lon":-0.1240755},{"type":"node","id":9515368,"lat":51.5102051,"lon":-0.1207534},{"type":"node","id":9515370,"lat":51.5110851,"lon":-0.1221837},{"type":"node","id":9515377,"lat":51.5095536,"lon":-0.1221066},{"type":"node","id":9515379,"lat":51.5101698,"lon":-0.1211889},{"type":"node","id":9515387,"lat":51.509656,"lon":-0.1233293},{"type":"node","id":9515388,"lat":51.5097269,"lon":-0.1231649},{"type":"node","id":9515389,"lat":51.5104371,"lon":-0.1214503},{"type":"node","id":9515394,"lat":51.510795,"lon":-0.1205984},{"type":"node","id":10574746,"lat":51.5123941,"lon":-0.1198716},{"type":"node","id":21704021,"lat":51.5145776,"lon":-0.1202961},{"type":"node","id":21990641,"lat":51.5145602,"lon":-0.1192135},{"type":"node","id":25471086,"lat":51.5128101,"lon":-0.1208676},{"type":"node","id":25471089,"lat":51.5122452,"lon":-0.1221231},{"type":"node","id":25471098,"lat":51.511825,"lon":-0.1198595},{"type":"node","id":25471100,"lat":51.5115118,"lon":-0.1206137},{"type":"node","id":25471101,"lat":51.5111013,"lon":-0.1216046},{"type":"node","id":25471371,"lat":51.5133364,"lon":-0.1191124},{"type":"node","id":25471708,"lat":51.5135323,"lon":-0.1225104},{"type":"node","id":25471710,"lat":51.5137818,"lon":-0.1222805},{"type":"node","id":25471711,"lat":51.5138894,"lon":-0.1221964},{"type":"node","id":25471712,"lat":51.5143692,"lon":-0.1216522},{"type":"node","id":25471729,"lat":51.5132772,"lon":-0.1204523},{"type":"node","id":25471747,"lat":51.5140695,"lon":-0.1227897},{"type":"node","id":25472677,"lat":51.5124646,"lon":-0.1187192},{"type":"node","id":25474654,"lat":51.5099951,"lon":-0.1183711},{"type":"node","id":25474657,"lat":51.5106924,"lon":-0.1197944},{"type":"node","id":25474658,"lat":51.5108119,"lon":-0.1194908},{"type":"node","id":25474661,"lat":51.5104919,"lon":-0.1191973},{"type":"node","id":25474662,"lat":51.5103331,"lon":-0.1197875},{"type":"node","id":25474664,"lat":51.5106302,"lon":-0.1200691},{"type":"node","id":25474666,"lat":51.5098945,"lon":-0.120885},{"type":"node","id":25474676,"lat":51.5101884,"lon":-0.119922},{"type":"node","id":25474749,"lat":51.5098277,"lon":-0.1189756},{"type":"node","id":25474750,"lat":51.5096741,"lon":-0.1195388},{"type":"node","id":25474751,"lat":51.5094279,"lon":-0.1203908},{"type":"node","id":25474753,"lat":51.5085239,"lon":-0.1224189},{"type":"node","id":25476001,"lat":51.5090898,"lon":-0.1215406},{"type":"node","id":25476004,"lat":51.5087023,"lon":-0.1224073},{"type":"node","id":25476014,"lat":51.5089387,"lon":-0.1234463},{"type":"node","id":25476017,"lat":51.5086577,"lon":-0.1239202},{"type":"node","id":25476020,"lat":51.5083794,"lon":-0.1243566},{"type":"node","id":25476022,"lat":51.5090305,"lon":-0.1232061},{"type":"node","id":25476025,"lat":51.5092563,"lon":-0.1235019},{"type":"node","id":25476026,"lat":51.509534,"lon":-0.1229122},{"type":"node","id":25476036,"lat":51.5084765,"lon":-0.1226423},{"type":"node","id":25476037,"lat":51.5087793,"lon":-0.123169},{"type":"node","id":25476039,"lat":51.508833,"lon":-0.1228341},{"type":"node","id":25476040,"lat":51.5084783,"lon":-0.1225162},{"type":"node","id":25476041,"lat":51.508196,"lon":-0.1230675},{"type":"node","id":25476042,"lat":51.5087972,"lon":-0.1241401},{"type":"node","id":25476047,"lat":51.5090744,"lon":-0.1244673},{"type":"node","id":25504198,"lat":51.5081545,"lon":-0.1239611},{"type":"node","id":25533436,"lat":51.5124746,"lon":-0.1243081},{"type":"node","id":25533437,"lat":51.5135066,"lon":-0.1224261},{"type":"node","id":25533438,"lat":51.5128031,"lon":-0.1237125},{"type":"node","id":25533439,"lat":51.5132105,"lon":-0.1229734},{"type":"node","id":25533440,"lat":51.513623,"lon":-0.123531},{"type":"node","id":25533605,"lat":51.5119157,"lon":-0.1238339},{"type":"node","id":25533613,"lat":51.5122616,"lon":-0.1230891},{"type":"node","id":25533621,"lat":51.512527,"lon":-0.1224584},{"type":"node","id":27345499,"lat":51.5106773,"lon":-0.1217101},{"type":"node","id":248122264,"lat":51.5100578,"lon":-0.1181431},{"type":"node","id":266558759,"lat":51.5106284,"lon":-0.1233257},{"type":"node","id":266559134,"lat":51.5108149,"lon":-0.1228325},{"type":"node","id":280339753,"lat":51.5111398,"lon":-0.1191486},{"type":"node","id":280412702,"lat":51.5112525,"lon":-0.1190359},{"type":"node","id":280412705,"lat":51.5115572,"lon":-0.1184697},{"type":"node","id":281454941,"lat":51.5111774,"lon":-0.1190266},{"type":"node","id":281454943,"lat":51.5113597,"lon":-0.1189308},{"type":"node","id":281454959,"lat":51.5130033,"lon":-0.1179792},{"type":"node","id":282569742,"lat":51.5117371,"lon":-0.1182491},{"type":"node","id":282569744,"lat":51.5116372,"lon":-0.1184755},{"type":"node","id":282569745,"lat":51.5118156,"lon":-0.118998},{"type":"node","id":282569797,"lat":51.5123422,"lon":-0.1183824},{"type":"node","id":303585905,"lat":51.5092317,"lon":-0.1238356},{"type":"node","id":345886002,"lat":51.5084571,"lon":-0.1244988},{"type":"node","id":346343907,"lat":51.5088639,"lon":-0.124222},{"type":"node","id":351771449,"lat":51.5121022,"lon":-0.1205034},{"type":"node","id":351788581,"lat":51.5145647,"lon":-0.122037},{"type":"node","id":351793985,"lat":51.5132068,"lon":-0.1241638},{"type":"node","id":368279307,"lat":51.512114,"lon":-0.1191511},{"type":"node","id":370711678,"lat":51.5094858,"lon":-0.1242547},{"type":"node","id":371564784,"lat":51.5135112,"lon":-0.1236842},{"type":"node","id":447764957,"lat":51.5084318,"lon":-0.1226933},{"type":"node","id":452486132,"lat":51.5137961,"lon":-0.1233948},{"type":"node","id":452486137,"lat":51.5138042,"lon":-0.1231063},{"type":"node","id":452486520,"lat":51.5138365,"lon":-0.1235687},{"type":"node","id":452486523,"lat":51.5137995,"lon":-0.123453},{"type":"node","id":452486619,"lat":51.5136882,"lon":-0.1234334},{"type":"node","id":452514269,"lat":51.5086764,"lon":-0.1238886},{"type":"node","id":452514276,"lat":51.5088059,"lon":-0.1240511},{"type":"node","id":455705605,"lat":51.5116003,"lon":-0.1188707},{"type":"node","id":455705606,"lat":51.511631,"lon":-0.1188947},{"type":"node","id":469773614,"lat":51.5105475,"lon":-0.1187244},{"type":"node","id":469780726,"lat":51.5103189,"lon":-0.1219193},{"type":"node","id":475966573,"lat":51.513568,"lon":-0.1226125},{"type":"node","id":484375367,"lat":51.5101364,"lon":-0.1182006},{"type":"node","id":568842560,"lat":51.5111252,"lon":-0.1236488},{"type":"node","id":568842573,"lat":51.5116906,"lon":-0.1243188},{"type":"node","id":568842587,"lat":51.5118054,"lon":-0.1235022},{"type":"node","id":568842591,"lat":51.5115623,"lon":-0.1232304},{"type":"node","id":751735187,"lat":51.5101959,"lon":-0.1180989},{"type":"node","id":751735218,"lat":51.5101146,"lon":-0.1180221},{"type":"node","id":751735268,"lat":51.5102098,"lon":-0.1180583},{"type":"node","id":751735372,"lat":51.5102704,"lon":-0.1182987},{"type":"node","id":751735403,"lat":51.5124053,"lon":-0.1223112},{"type":"node","id":826724026,"lat":51.5090944,"lon":-0.1212477},{"type":"node","id":826724148,"lat":51.5091145,"lon":-0.1212034},{"type":"node","id":826724206,"lat":51.5091524,"lon":-0.1215137},{"type":"node","id":826724242,"lat":51.509181,"lon":-0.121282},{"type":"node","id":826724298,"lat":51.5090885,"lon":-0.1211727},{"type":"node","id":826724301,"lat":51.5082272,"lon":-0.1228776},{"type":"node","id":827333150,"lat":51.509057,"lon":-0.1216164},{"type":"node","id":827333154,"lat":51.5087436,"lon":-0.1223151},{"type":"node","id":827333158,"lat":51.508987,"lon":-0.1217725},{"type":"node","id":827333160,"lat":51.5088157,"lon":-0.1221545},{"type":"node","id":857556897,"lat":51.5102257,"lon":-0.1219591},{"type":"node","id":931888889,"lat":51.5135647,"lon":-0.1213382},{"type":"node","id":1104386126,"lat":51.5092061,"lon":-0.1213868},{"type":"node","id":1104386133,"lat":51.5102398,"lon":-0.1212629},{"type":"node","id":1104386144,"lat":51.510209,"lon":-0.1199386},{"type":"node","id":1104386149,"lat":51.5116816,"lon":-0.1195882},{"type":"node","id":1104386151,"lat":51.5091223,"lon":-0.1215802},{"type":"node","id":1104386162,"lat":51.5100524,"lon":-0.1203382},{"type":"node","id":1104386168,"lat":51.5101479,"lon":-0.1200447},{"type":"node","id":1106056843,"lat":51.5104413,"lon":-0.12414},{"type":"node","id":1106056846,"lat":51.5106453,"lon":-0.1210919},{"type":"node","id":1106056847,"lat":51.5098094,"lon":-0.1230932},{"type":"node","id":1106056850,"lat":51.5112438,"lon":-0.1192217},{"type":"node","id":1106056855,"lat":51.5102319,"lon":-0.1221404},{"type":"node","id":1106056858,"lat":51.5104884,"lon":-0.1215005},{"type":"node","id":1106056861,"lat":51.5096923,"lon":-0.123358},{"type":"node","id":1106056863,"lat":51.5110098,"lon":-0.1201147},{"type":"node","id":1106056866,"lat":51.5100094,"lon":-0.1226605},{"type":"node","id":1106056874,"lat":51.5114169,"lon":-0.1191523},{"type":"node","id":1106056876,"lat":51.5093037,"lon":-0.1241422},{"type":"node","id":1106056878,"lat":51.5108331,"lon":-0.1205906},{"type":"node","id":1106056883,"lat":51.511178,"lon":-0.1193906},{"type":"node","id":1106056886,"lat":51.5115288,"lon":-0.118661},{"type":"node","id":1139345972,"lat":51.5097993,"lon":-0.1191582},{"type":"node","id":1155428970,"lat":51.5100671,"lon":-0.1225304},{"type":"node","id":1163257575,"lat":51.508815,"lon":-0.1228927},{"type":"node","id":1163257577,"lat":51.5088261,"lon":-0.1230404},{"type":"node","id":1163257578,"lat":51.5088178,"lon":-0.1229717},{"type":"node","id":1163257579,"lat":51.508888,"lon":-0.1229021},{"type":"node","id":1163257580,"lat":51.508814,"lon":-0.1231074},{"type":"node","id":1164840729,"lat":51.5102758,"lon":-0.1213008},{"type":"node","id":1174970393,"lat":51.5084998,"lon":-0.1224703},{"type":"node","id":1237514459,"lat":51.5113407,"lon":-0.1190744},{"type":"node","id":1237514461,"lat":51.5113183,"lon":-0.119103},{"type":"node","id":1237514475,"lat":51.5144976,"lon":-0.1195046},{"type":"node","id":1237514480,"lat":51.5113354,"lon":-0.1189726},{"type":"node","id":1237543413,"lat":51.5100819,"lon":-0.118449},{"type":"node","id":1237543434,"lat":51.5100672,"lon":-0.1184893},{"type":"node","id":1375625379,"lat":51.5136437,"lon":-0.11991},{"type":"node","id":1375625382,"lat":51.5137649,"lon":-0.1202131},{"type":"node","id":1375634504,"lat":51.5138763,"lon":-0.1230857},{"type":"node","id":1614846605,"lat":51.509403,"lon":-0.1239434},{"type":"node","id":1614846607,"lat":51.5094758,"lon":-0.1243749},{"type":"node","id":1614846610,"lat":51.5095143,"lon":-0.1241951},{"type":"node","id":1614846616,"lat":51.5095758,"lon":-0.124167},{"type":"node","id":1614846678,"lat":51.5103371,"lon":-0.1240707},{"type":"node","id":1614846683,"lat":51.5103737,"lon":-0.124098},{"type":"node","id":1614846684,"lat":51.5104526,"lon":-0.1237776},{"type":"node","id":1614846692,"lat":51.5111839,"lon":-0.1222702},{"type":"node","id":1614846695,"lat":51.5115778,"lon":-0.1213925},{"type":"node","id":1614846699,"lat":51.5116094,"lon":-0.1213262},{"type":"node","id":1614846702,"lat":51.5116733,"lon":-0.1212},{"type":"node","id":1614846707,"lat":51.5119194,"lon":-0.1217894},{"type":"node","id":1614846708,"lat":51.5119358,"lon":-0.1217567},{"type":"node","id":1614846711,"lat":51.5120743,"lon":-0.1204274},{"type":"node","id":1614926285,"lat":51.5115213,"lon":-0.1193825},{"type":"node","id":1614926320,"lat":51.5132029,"lon":-0.1187687},{"type":"node","id":1614926322,"lat":51.5134874,"lon":-0.118499},{"type":"node","id":1614926323,"lat":51.5135217,"lon":-0.1184944},{"type":"node","id":1614926333,"lat":51.5138483,"lon":-0.119235},{"type":"node","id":1614926334,"lat":51.513864,"lon":-0.1190806},{"type":"node","id":1614926336,"lat":51.5139464,"lon":-0.119},{"type":"node","id":1614926337,"lat":51.5140277,"lon":-0.1189201},{"type":"node","id":1614926338,"lat":51.5140464,"lon":-0.1188527},{"type":"node","id":1614926339,"lat":51.5140634,"lon":-0.1188844},{"type":"node","id":1614926340,"lat":51.5141805,"lon":-0.1226025},{"type":"node","id":1614926341,"lat":51.5143003,"lon":-0.1194796},{"type":"node","id":1614926342,"lat":51.5144725,"lon":-0.1230148},{"type":"node","id":1616122145,"lat":51.5143001,"lon":-0.1242333},{"type":"node","id":1616162948,"lat":51.5113938,"lon":-0.1188482},{"type":"node","id":1678452782,"lat":51.5115797,"lon":-0.1227167},{"type":"node","id":1678452794,"lat":51.5132115,"lon":-0.1218515},{"type":"node","id":1678452804,"lat":51.5135667,"lon":-0.1214674},{"type":"node","id":1678452805,"lat":51.5135759,"lon":-0.1214233},{"type":"node","id":1678452806,"lat":51.5135891,"lon":-0.1213562},{"type":"node","id":1678452810,"lat":51.5137033,"lon":-0.1234186},{"type":"node","id":1678452824,"lat":51.5140442,"lon":-0.1209393},{"type":"node","id":1685903440,"lat":51.511442,"lon":-0.1191232},{"type":"node","id":1685903441,"lat":51.5116345,"lon":-0.1190235},{"type":"node","id":1685903444,"lat":51.5123397,"lon":-0.1188187},{"type":"node","id":1685903445,"lat":51.5115815,"lon":-0.1188317},{"type":"node","id":1685903447,"lat":51.5122004,"lon":-0.1188958},{"type":"node","id":1685903455,"lat":51.5126247,"lon":-0.1185681},{"type":"node","id":1697727370,"lat":51.5112136,"lon":-0.1190403},{"type":"node","id":1697727377,"lat":51.5113,"lon":-0.1190098},{"type":"node","id":1697727378,"lat":51.5113743,"lon":-0.1190384},{"type":"node","id":1782729408,"lat":51.5111041,"lon":-0.1191192},{"type":"node","id":1819091297,"lat":51.5107642,"lon":-0.1196044},{"type":"node","id":1934197346,"lat":51.511304,"lon":-0.1192182},{"type":"node","id":1934197347,"lat":51.5112578,"lon":-0.1194569},{"type":"node","id":1938953360,"lat":51.5100132,"lon":-0.1182687},{"type":"node","id":1938953361,"lat":51.511585,"lon":-0.1187516},{"type":"node","id":1947257511,"lat":51.5092926,"lon":-0.1240328},{"type":"node","id":1947257513,"lat":51.5093344,"lon":-0.1240796},{"type":"node","id":1947257516,"lat":51.5107786,"lon":-0.1188737},{"type":"node","id":1947257517,"lat":51.511205,"lon":-0.1195975},{"type":"node","id":1947257518,"lat":51.5112688,"lon":-0.1190318},{"type":"node","id":1947257529,"lat":51.5112666,"lon":-0.1190854},{"type":"node","id":1947257531,"lat":51.5113552,"lon":-0.1191509},{"type":"node","id":1947257536,"lat":51.5114547,"lon":-0.1188932},{"type":"node","id":1947276617,"lat":51.5104611,"lon":-0.1213792},{"type":"node","id":1947276620,"lat":51.5104584,"lon":-0.1215798},{"type":"node","id":1953686428,"lat":51.51022,"lon":-0.1184517},{"type":"node","id":1953893720,"lat":51.513806,"lon":-0.1235079},{"type":"node","id":2069620510,"lat":51.5126008,"lon":-0.1211339},{"type":"node","id":2092250094,"lat":51.5097171,"lon":-0.120674},{"type":"node","id":2105667816,"lat":51.5110285,"lon":-0.1189091},{"type":"node","id":2126997613,"lat":51.510266,"lon":-0.1199075},{"type":"node","id":2126997628,"lat":51.5103126,"lon":-0.1198009},{"type":"node","id":2126997633,"lat":51.5102887,"lon":-0.1198563},{"type":"node","id":2126997649,"lat":51.510235,"lon":-0.1199399},{"type":"node","id":2126997658,"lat":51.5104023,"lon":-0.1195631},{"type":"node","id":2140823714,"lat":51.5098244,"lon":-0.1189052},{"type":"node","id":2140823741,"lat":51.5098221,"lon":-0.1190603},{"type":"node","id":2159454166,"lat":51.5143191,"lon":-0.1217043},{"type":"node","id":2159454167,"lat":51.5138637,"lon":-0.1221338},{"type":"node","id":2184965485,"lat":51.5112017,"lon":-0.1193055},{"type":"node","id":2184965488,"lat":51.5110494,"lon":-0.1190796},{"type":"node","id":2184966600,"lat":51.5113084,"lon":-0.119231},{"type":"node","id":2238463476,"lat":51.5111739,"lon":-0.1191786},{"type":"node","id":2238463480,"lat":51.5112252,"lon":-0.119209},{"type":"node","id":2337311458,"lat":51.510742,"lon":-0.1218795},{"type":"node","id":2337311461,"lat":51.5107712,"lon":-0.1218188},{"type":"node","id":2338587824,"lat":51.5105061,"lon":-0.1236425},{"type":"node","id":2407843602,"lat":51.5109666,"lon":-0.1201147},{"type":"node","id":2444562449,"lat":51.5134912,"lon":-0.1210163},{"type":"node","id":2444565402,"lat":51.5132565,"lon":-0.1217915},{"type":"node","id":2444565403,"lat":51.5139867,"lon":-0.1209842},{"type":"node","id":2444571508,"lat":51.5138055,"lon":-0.1232032},{"type":"node","id":2444571509,"lat":51.5141988,"lon":-0.1226317},{"type":"node","id":2450397874,"lat":51.5113881,"lon":-0.1192338},{"type":"node","id":2450397879,"lat":51.5114709,"lon":-0.1193165},{"type":"node","id":2450730397,"lat":51.5138,"lon":-0.1232622},{"type":"node","id":2453519182,"lat":51.5126991,"lon":-0.1210749},{"type":"node","id":2453519185,"lat":51.5126018,"lon":-0.1213033},{"type":"node","id":2476805753,"lat":51.5121543,"lon":-0.1206141},{"type":"node","id":2476805758,"lat":51.5129525,"lon":-0.1215299},{"type":"node","id":2482696991,"lat":51.5123119,"lon":-0.1219673},{"type":"node","id":2502903587,"lat":51.5138928,"lon":-0.1205272},{"type":"node","id":2502903588,"lat":51.5144974,"lon":-0.121917},{"type":"node","id":2512940572,"lat":51.5131251,"lon":-0.1243124},{"type":"node","id":2521654793,"lat":51.511403,"lon":-0.119245},{"type":"node","id":2521654794,"lat":51.5114143,"lon":-0.1192298},{"type":"node","id":2564873543,"lat":51.5093773,"lon":-0.1241323},{"type":"node","id":2612448636,"lat":51.5112896,"lon":-0.119343},{"type":"node","id":2612448638,"lat":51.5113019,"lon":-0.1192777},{"type":"node","id":2612451170,"lat":51.5115927,"lon":-0.1186346},{"type":"node","id":2663048864,"lat":51.5112868,"lon":-0.1191493},{"type":"node","id":2663048865,"lat":51.5113035,"lon":-0.1192312},{"type":"node","id":2831321470,"lat":51.5115898,"lon":-0.1209064},{"type":"node","id":2975783588,"lat":51.5091557,"lon":-0.1229455},{"type":"node","id":3054882690,"lat":51.5118371,"lon":-0.1198076},{"type":"node","id":3054882691,"lat":51.5123714,"lon":-0.119919},{"type":"node","id":3054882693,"lat":51.5134121,"lon":-0.1222019},{"type":"node","id":3054882695,"lat":51.5137557,"lon":-0.1200469},{"type":"node","id":3085005035,"lat":51.5137601,"lon":-0.1233988},{"type":"node","id":3085005036,"lat":51.513796,"lon":-0.1233254},{"type":"node","id":3085005037,"lat":51.5138073,"lon":-0.1231603},{"type":"node","id":3085005038,"lat":51.5138353,"lon":-0.123104},{"type":"node","id":3189218645,"lat":51.5098594,"lon":-0.1229812},{"type":"node","id":3497406876,"lat":51.5124367,"lon":-0.1187414},{"type":"node","id":3764054180,"lat":51.5112083,"lon":-0.1182146},{"type":"node","id":3764054181,"lat":51.5112392,"lon":-0.1181074},{"type":"node","id":3764074072,"lat":51.5104712,"lon":-0.1184603},{"type":"node","id":3764074088,"lat":51.5107516,"lon":-0.1186861},{"type":"node","id":3790287116,"lat":51.5116415,"lon":-0.1224039},{"type":"node","id":3811129601,"lat":51.5116899,"lon":-0.1233745},{"type":"node","id":3876825295,"lat":51.5084311,"lon":-0.1208027},{"type":"node","id":3876825296,"lat":51.5081181,"lon":-0.1210908},{"type":"node","id":3895412044,"lat":51.5143585,"lon":-0.1181288},{"type":"node","id":3911044741,"lat":51.5096833,"lon":-0.1184958},{"type":"node","id":4059394509,"lat":51.5108879,"lon":-0.1203313},{"type":"node","id":4059394510,"lat":51.5097201,"lon":-0.1231786},{"type":"node","id":4059394511,"lat":51.5103773,"lon":-0.1217815},{"type":"node","id":4068974160,"lat":51.512723,"lon":-0.1184583},{"type":"node","id":4079423542,"lat":51.5122403,"lon":-0.1194761},{"type":"node","id":4079423543,"lat":51.5119203,"lon":-0.1189772},{"type":"node","id":4079423544,"lat":51.5127404,"lon":-0.1209882},{"type":"node","id":4082282109,"lat":51.5105726,"lon":-0.118542},{"type":"node","id":4174200003,"lat":51.508613,"lon":-0.1205865},{"type":"node","id":4174200004,"lat":51.5087116,"lon":-0.1204447},{"type":"node","id":4174200005,"lat":51.5088306,"lon":-0.1202677},{"type":"node","id":4174200009,"lat":51.509037,"lon":-0.1199627},{"type":"node","id":4174200013,"lat":51.5094415,"lon":-0.1192796},{"type":"node","id":4174200018,"lat":51.5095967,"lon":-0.1189639},{"type":"node","id":4174200021,"lat":51.5097164,"lon":-0.1186628},{"type":"node","id":4174200022,"lat":51.5097522,"lon":-0.1185588},{"type":"node","id":4174200023,"lat":51.5097624,"lon":-0.11872},{"type":"node","id":4174200024,"lat":51.5098005,"lon":-0.118785},{"type":"node","id":4174200025,"lat":51.5098235,"lon":-0.1183503},{"type":"node","id":4174200026,"lat":51.5098728,"lon":-0.1186722},{"type":"node","id":4174365778,"lat":51.50982,"lon":-0.1188499},{"type":"node","id":4174365779,"lat":51.5096198,"lon":-0.1189059},{"type":"node","id":4174365780,"lat":51.509543,"lon":-0.1188258},{"type":"node","id":4302887173,"lat":51.5127795,"lon":-0.1183857},{"type":"node","id":4349663550,"lat":51.5138574,"lon":-0.123096},{"type":"node","id":4349663551,"lat":51.510523,"lon":-0.1215356},{"type":"node","id":4350208168,"lat":51.5104961,"lon":-0.1210344},{"type":"node","id":4350208169,"lat":51.5106894,"lon":-0.1211321},{"type":"node","id":4350208170,"lat":51.5106078,"lon":-0.1211895},{"type":"node","id":4350208171,"lat":51.5106503,"lon":-0.1209784},{"type":"node","id":4350220819,"lat":51.5114409,"lon":-0.1192727},{"type":"node","id":4350220820,"lat":51.5113853,"lon":-0.119193},{"type":"node","id":4350239457,"lat":51.5117486,"lon":-0.1196927},{"type":"node","id":4350239459,"lat":51.5115171,"lon":-0.1185693},{"type":"node","id":4350239460,"lat":51.51155,"lon":-0.1185853},{"type":"node","id":4350239461,"lat":51.5116699,"lon":-0.1184023},{"type":"node","id":4350239462,"lat":51.5114606,"lon":-0.1186846},{"type":"node","id":4350239463,"lat":51.5115088,"lon":-0.1187229},{"type":"node","id":4350239464,"lat":51.5115801,"lon":-0.1187793},{"type":"node","id":4350239467,"lat":51.5113902,"lon":-0.1190162},{"type":"node","id":4350239468,"lat":51.51151,"lon":-0.1190736},{"type":"node","id":4415640809,"lat":51.511567,"lon":-0.1225733},{"type":"node","id":4457798822,"lat":51.511391,"lon":-0.1230437},{"type":"node","id":4457798823,"lat":51.5115505,"lon":-0.1226805},{"type":"node","id":4637615417,"lat":51.5135984,"lon":-0.1224525},{"type":"node","id":4686140571,"lat":51.5139367,"lon":-0.1225855},{"type":"node","id":4686140572,"lat":51.5140265,"lon":-0.1227249},{"type":"node","id":4697525065,"lat":51.5099548,"lon":-0.1183988},{"type":"node","id":4697525067,"lat":51.5098408,"lon":-0.1186421},{"type":"node","id":4697525068,"lat":51.5097872,"lon":-0.1187623},{"type":"node","id":4806684750,"lat":51.5123518,"lon":-0.1224314},{"type":"node","id":4967033601,"lat":51.5129536,"lon":-0.120711},{"type":"node","id":4967033604,"lat":51.5136685,"lon":-0.120095},{"type":"node","id":4967033605,"lat":51.5132947,"lon":-0.119158},{"type":"node","id":4967033606,"lat":51.5128565,"lon":-0.1208043},{"type":"node","id":5011388929,"lat":51.5122497,"lon":-0.1231171},{"type":"node","id":5011388934,"lat":51.5119946,"lon":-0.1237046},{"type":"node","id":5011556345,"lat":51.5121881,"lon":-0.1220554},{"type":"node","id":5011556351,"lat":51.5123036,"lon":-0.1221922},{"type":"node","id":5011556356,"lat":51.5118523,"lon":-0.123552},{"type":"node","id":5011556360,"lat":51.5118966,"lon":-0.1234476},{"type":"node","id":5011796248,"lat":51.5123554,"lon":-0.1224238},{"type":"node","id":5011796252,"lat":51.5117529,"lon":-0.1234442},{"type":"node","id":5011796256,"lat":51.511928,"lon":-0.1236317},{"type":"node","id":5011796257,"lat":51.511636,"lon":-0.1233133},{"type":"node","id":5011796258,"lat":51.5120294,"lon":-0.1220548},{"type":"node","id":5011796259,"lat":51.5115311,"lon":-0.123192},{"type":"node","id":5011796261,"lat":51.5120175,"lon":-0.1218537},{"type":"node","id":5011796263,"lat":51.5120829,"lon":-0.1219313},{"type":"node","id":5011796264,"lat":51.5120323,"lon":-0.1220482},{"type":"node","id":5011796269,"lat":51.5124775,"lon":-0.1223985},{"type":"node","id":5011796271,"lat":51.5114664,"lon":-0.1231213},{"type":"node","id":5012087056,"lat":51.5119134,"lon":-0.1224804},{"type":"node","id":5012087059,"lat":51.5119076,"lon":-0.1221256},{"type":"node","id":5012087061,"lat":51.511963,"lon":-0.1221881},{"type":"node","id":5012087062,"lat":51.5119285,"lon":-0.1224445},{"type":"node","id":5012087065,"lat":51.5118703,"lon":-0.1227007},{"type":"node","id":5012087066,"lat":51.5118356,"lon":-0.1226624},{"type":"node","id":5012087069,"lat":51.5119465,"lon":-0.1225187},{"type":"node","id":5012087071,"lat":51.5118197,"lon":-0.1226964},{"type":"node","id":5012087073,"lat":51.5117009,"lon":-0.1227854},{"type":"node","id":5012087075,"lat":51.5116464,"lon":-0.1227213},{"type":"node","id":5012149898,"lat":51.5118716,"lon":-0.1220845},{"type":"node","id":5012149899,"lat":51.5117686,"lon":-0.1235847},{"type":"node","id":5012149900,"lat":51.5115166,"lon":-0.1233277},{"type":"node","id":5012149901,"lat":51.5124535,"lon":-0.1226328},{"type":"node","id":5012149903,"lat":51.5122674,"lon":-0.1222727},{"type":"node","id":5012149904,"lat":51.5115982,"lon":-0.122667},{"type":"node","id":5012149905,"lat":51.5115157,"lon":-0.1230014},{"type":"node","id":5012149906,"lat":51.5120895,"lon":-0.1234834},{"type":"node","id":5012149907,"lat":51.51181,"lon":-0.1220118},{"type":"node","id":5012149908,"lat":51.5120618,"lon":-0.1234532},{"type":"node","id":5012149909,"lat":51.5115822,"lon":-0.1230717},{"type":"node","id":5012149910,"lat":51.5119794,"lon":-0.1219402},{"type":"node","id":5012149911,"lat":51.5122337,"lon":-0.1230976},{"type":"node","id":5012149912,"lat":51.5120483,"lon":-0.1220113},{"type":"node","id":5012149913,"lat":51.5121549,"lon":-0.1221291},{"type":"node","id":5012149914,"lat":51.5123733,"lon":-0.1223835},{"type":"node","id":5012149915,"lat":51.5124317,"lon":-0.1226086},{"type":"node","id":5012149916,"lat":51.5124432,"lon":-0.1224772},{"type":"node","id":5012149917,"lat":51.5118505,"lon":-0.1219232},{"type":"node","id":5012149918,"lat":51.5116659,"lon":-0.1224316},{"type":"node","id":5012149919,"lat":51.5117721,"lon":-0.1231856},{"type":"node","id":5012149920,"lat":51.5119743,"lon":-0.1235242},{"type":"node","id":5012151521,"lat":51.5118313,"lon":-0.1232576},{"type":"node","id":5012151522,"lat":51.5117161,"lon":-0.123123},{"type":"node","id":5012151523,"lat":51.5116106,"lon":-0.1226808},{"type":"node","id":5012151525,"lat":51.511496,"lon":-0.1228076},{"type":"node","id":5012151526,"lat":51.5119086,"lon":-0.1219967},{"type":"node","id":5012183933,"lat":51.5123321,"lon":-0.1231708},{"type":"node","id":5012183938,"lat":51.5123277,"lon":-0.1219319},{"type":"node","id":5153538727,"lat":51.5081787,"lon":-0.1240052},{"type":"node","id":5153538728,"lat":51.5081204,"lon":-0.1240893},{"type":"node","id":5325427428,"lat":51.5081811,"lon":-0.1230001},{"type":"node","id":5325427429,"lat":51.508202,"lon":-0.1229826},{"type":"node","id":5325427430,"lat":51.5081598,"lon":-0.1230215},{"type":"node","id":5325427437,"lat":51.508914,"lon":-0.1234084},{"type":"node","id":5325427460,"lat":51.5087045,"lon":-0.1238959},{"type":"node","id":5333858450,"lat":51.5092746,"lon":-0.123902},{"type":"node","id":5387300688,"lat":51.5097733,"lon":-0.1234654},{"type":"node","id":5391163021,"lat":51.5103476,"lon":-0.1240441},{"type":"node","id":5391163022,"lat":51.5110698,"lon":-0.122212},{"type":"node","id":5391163030,"lat":51.511027,"lon":-0.1222913},{"type":"node","id":5397216223,"lat":51.512307,"lon":-0.1231407},{"type":"node","id":5397216224,"lat":51.5127588,"lon":-0.1236601},{"type":"node","id":5404151675,"lat":51.5145362,"lon":-0.123732},{"type":"node","id":5404151676,"lat":51.5144919,"lon":-0.1236579},{"type":"node","id":5404151677,"lat":51.514569,"lon":-0.1237865},{"type":"node","id":5404152924,"lat":51.5144848,"lon":-0.123047},{"type":"node","id":5404153140,"lat":51.5142641,"lon":-0.1242761},{"type":"node","id":5479717489,"lat":51.513036,"lon":-0.1239729},{"type":"node","id":5479777428,"lat":51.5130426,"lon":-0.1242047},{"type":"node","id":5479777429,"lat":51.5130378,"lon":-0.1242058},{"type":"node","id":5479777430,"lat":51.5130331,"lon":-0.1242038},{"type":"node","id":5479777431,"lat":51.5130293,"lon":-0.1241992},{"type":"node","id":5479777432,"lat":51.5130269,"lon":-0.1241925},{"type":"node","id":5479777433,"lat":51.5130262,"lon":-0.1241849},{"type":"node","id":5479777434,"lat":51.5130274,"lon":-0.1241774},{"type":"node","id":5479777435,"lat":51.5130303,"lon":-0.1241712},{"type":"node","id":5479777436,"lat":51.5130345,"lon":-0.1241673},{"type":"node","id":5479777437,"lat":51.5130393,"lon":-0.1241663},{"type":"node","id":5479777438,"lat":51.5130439,"lon":-0.1241682},{"type":"node","id":5479777439,"lat":51.5130477,"lon":-0.1241729},{"type":"node","id":5479777440,"lat":51.5130502,"lon":-0.1241796},{"type":"node","id":5479777441,"lat":51.5130508,"lon":-0.1241872},{"type":"node","id":5479777442,"lat":51.5130496,"lon":-0.1241947},{"type":"node","id":5479777443,"lat":51.5130467,"lon":-0.1242008},{"type":"node","id":5479777634,"lat":51.5129295,"lon":-0.1241616},{"type":"node","id":5479777651,"lat":51.5129198,"lon":-0.124182},{"type":"node","id":5479777652,"lat":51.5129208,"lon":-0.1241946},{"type":"node","id":5479777653,"lat":51.5129372,"lon":-0.1242161},{"type":"node","id":5479777654,"lat":51.5129443,"lon":-0.1242163},{"type":"node","id":5479777655,"lat":51.512951,"lon":-0.1242088},{"type":"node","id":5479777902,"lat":51.5130443,"lon":-0.1243383},{"type":"node","id":5479777903,"lat":51.5130218,"lon":-0.1243764},{"type":"node","id":5479777908,"lat":51.5131348,"lon":-0.1241912},{"type":"node","id":5479777909,"lat":51.5131587,"lon":-0.1241537},{"type":"node","id":5479777910,"lat":51.5131407,"lon":-0.1242008},{"type":"node","id":5479777911,"lat":51.513164,"lon":-0.1241623},{"type":"node","id":5479777912,"lat":51.5130505,"lon":-0.124348},{"type":"node","id":5479777913,"lat":51.513028,"lon":-0.124386},{"type":"node","id":5677722958,"lat":51.5081131,"lon":-0.1238861},{"type":"node","id":5677722965,"lat":51.5080357,"lon":-0.1239954},{"type":"node","id":5746109250,"lat":51.5106358,"lon":-0.1226023},{"type":"node","id":5746109251,"lat":51.5106782,"lon":-0.1226561},{"type":"node","id":5934126091,"lat":51.5132367,"lon":-0.121891},{"type":"node","id":6111301383,"lat":51.5085474,"lon":-0.122449},{"type":"node","id":6111301384,"lat":51.5084587,"lon":-0.1226641},{"type":"node","id":6161772316,"lat":51.5102417,"lon":-0.120125},{"type":"node","id":6161772317,"lat":51.5102799,"lon":-0.1201577},{"type":"node","id":6161772318,"lat":51.5103558,"lon":-0.1202226},{"type":"node","id":6173148421,"lat":51.508203,"lon":-0.1228136},{"type":"node","id":6173783037,"lat":51.512032,"lon":-0.1236174},{"type":"node","id":6173783042,"lat":51.5119501,"lon":-0.1236559},{"type":"node","id":6256910043,"lat":51.5114235,"lon":-0.1234085},{"type":"node","id":6256910044,"lat":51.5114548,"lon":-0.1233371},{"type":"node","id":6256910045,"lat":51.5114961,"lon":-0.1234904},{"type":"node","id":6256910046,"lat":51.5115264,"lon":-0.1234234},{"type":"node","id":6277660685,"lat":51.5116545,"lon":-0.1211371},{"type":"node","id":6277660702,"lat":51.5112031,"lon":-0.1222274},{"type":"node","id":6282772554,"lat":51.5106682,"lon":-0.1216998},{"type":"node","id":6282772555,"lat":51.5107558,"lon":-0.1218509},{"type":"node","id":6282772556,"lat":51.5115165,"lon":-0.1226424},{"type":"node","id":6282772557,"lat":51.5120264,"lon":-0.120522},{"type":"node","id":6282772563,"lat":51.5124484,"lon":-0.1198398},{"type":"node","id":6282772575,"lat":51.5129881,"lon":-0.1194118},{"type":"node","id":6283479888,"lat":51.5144973,"lon":-0.1193596},{"type":"node","id":6283479982,"lat":51.513224,"lon":-0.1187488},{"type":"node","id":6283521240,"lat":51.5115945,"lon":-0.1190338},{"type":"node","id":6291190465,"lat":51.5104184,"lon":-0.1198684},{"type":"node","id":6307073859,"lat":51.5095732,"lon":-0.119888},{"type":"node","id":6329942070,"lat":51.5123891,"lon":-0.1185113},{"type":"node","id":12354829,"lat":51.5057122,"lon":-0.1154558},{"type":"node","id":12354831,"lat":51.5054606,"lon":-0.1158416},{"type":"node","id":21554353,"lat":51.5059899,"lon":-0.1148421},{"type":"node","id":25472945,"lat":51.5055107,"lon":-0.1140774},{"type":"node","id":26398105,"lat":51.5052556,"lon":-0.1165808},{"type":"node","id":26398235,"lat":51.5056797,"lon":-0.1179269},{"type":"node","id":26405445,"lat":51.507452,"lon":-0.1154951},{"type":"node","id":251952341,"lat":51.5079318,"lon":-0.1136921},{"type":"node","id":282686493,"lat":51.5061035,"lon":-0.1150266},{"type":"node","id":282686494,"lat":51.5061371,"lon":-0.1150436},{"type":"node","id":282686496,"lat":51.5061273,"lon":-0.1150687},{"type":"node","id":282686497,"lat":51.5061197,"lon":-0.1150426},{"type":"node","id":282686499,"lat":51.5060624,"lon":-0.116226},{"type":"node","id":469785645,"lat":51.5063455,"lon":-0.1149583},{"type":"node","id":590146497,"lat":51.505597,"lon":-0.1156397},{"type":"node","id":590146512,"lat":51.5051717,"lon":-0.1161701},{"type":"node","id":805522608,"lat":51.5070225,"lon":-0.1154123},{"type":"node","id":805522625,"lat":51.50699,"lon":-0.1155278},{"type":"node","id":805522627,"lat":51.5069267,"lon":-0.115691},{"type":"node","id":805522630,"lat":51.5068362,"lon":-0.115613},{"type":"node","id":805522632,"lat":51.5068231,"lon":-0.1156557},{"type":"node","id":805522634,"lat":51.5069123,"lon":-0.1157324},{"type":"node","id":805522640,"lat":51.5070471,"lon":-0.1153454},{"type":"node","id":805522643,"lat":51.5069496,"lon":-0.1152566},{"type":"node","id":805522645,"lat":51.5069657,"lon":-0.1152088},{"type":"node","id":805522646,"lat":51.5070642,"lon":-0.1152961},{"type":"node","id":805522648,"lat":51.5071066,"lon":-0.1151745},{"type":"node","id":914309874,"lat":51.5058741,"lon":-0.1145263},{"type":"node","id":914309891,"lat":51.5063536,"lon":-0.114495},{"type":"node","id":914309954,"lat":51.5059015,"lon":-0.1144244},{"type":"node","id":915410128,"lat":51.5062558,"lon":-0.1144047},{"type":"node","id":1630605810,"lat":51.5061858,"lon":-0.1143415},{"type":"node","id":1646024762,"lat":51.5053103,"lon":-0.1167358},{"type":"node","id":1646024772,"lat":51.5056134,"lon":-0.1158756},{"type":"node","id":1646024795,"lat":51.5057195,"lon":-0.1157512},{"type":"node","id":1646024810,"lat":51.5058414,"lon":-0.1174825},{"type":"node","id":1646024813,"lat":51.505928,"lon":-0.1176822},{"type":"node","id":1646024847,"lat":51.5060909,"lon":-0.1165135},{"type":"node","id":1646024849,"lat":51.5061012,"lon":-0.1150069},{"type":"node","id":1646024870,"lat":51.5061131,"lon":-0.1149999},{"type":"node","id":1646024871,"lat":51.5061222,"lon":-0.1165767},{"type":"node","id":1646024873,"lat":51.5061247,"lon":-0.1160183},{"type":"node","id":1646024878,"lat":51.5061365,"lon":-0.1150231},{"type":"node","id":1646024880,"lat":51.506138,"lon":-0.1160338},{"type":"node","id":1646024882,"lat":51.5061385,"lon":-0.1159814},{"type":"node","id":1646024884,"lat":51.5061405,"lon":-0.1171517},{"type":"node","id":1646024887,"lat":51.5061466,"lon":-0.1159814},{"type":"node","id":1646024890,"lat":51.5061512,"lon":-0.1159937},{"type":"node","id":1646024897,"lat":51.5061864,"lon":-0.1167044},{"type":"node","id":1646024907,"lat":51.5062148,"lon":-0.1163741},{"type":"node","id":1646024911,"lat":51.50622,"lon":-0.1167704},{"type":"node","id":1646024923,"lat":51.50622,"lon":-0.1163998},{"type":"node","id":1646024925,"lat":51.5062267,"lon":-0.1173531},{"type":"node","id":1646024937,"lat":51.506323,"lon":-0.1157047},{"type":"node","id":1646024941,"lat":51.5063275,"lon":-0.1157243},{"type":"node","id":1646024943,"lat":51.5063337,"lon":-0.1156719},{"type":"node","id":1646024946,"lat":51.5063377,"lon":-0.1157219},{"type":"node","id":1646024948,"lat":51.5063515,"lon":-0.1156842},{"type":"node","id":1646024957,"lat":51.5064662,"lon":-0.1168984},{"type":"node","id":1646024959,"lat":51.5064702,"lon":-0.1169173},{"type":"node","id":1646024962,"lat":51.5064743,"lon":-0.1168886},{"type":"node","id":1646024964,"lat":51.5064962,"lon":-0.1169296},{"type":"node","id":1646024966,"lat":51.5064978,"lon":-0.1168993},{"type":"node","id":1646024968,"lat":51.5065069,"lon":-0.1169214},{"type":"node","id":1646024969,"lat":51.5065144,"lon":-0.1168737},{"type":"node","id":1646024971,"lat":51.5065538,"lon":-0.1155991},{"type":"node","id":1646024973,"lat":51.5065569,"lon":-0.1156171},{"type":"node","id":1646024975,"lat":51.5065681,"lon":-0.1155549},{"type":"node","id":1646024977,"lat":51.5065686,"lon":-0.1156163},{"type":"node","id":1646025000,"lat":51.5065929,"lon":-0.1155472},{"type":"node","id":1646025038,"lat":51.5069794,"lon":-0.1161773},{"type":"node","id":1646025042,"lat":51.5069873,"lon":-0.1161116},{"type":"node","id":1646043096,"lat":51.5068628,"lon":-0.115792},{"type":"node","id":1646043098,"lat":51.506877,"lon":-0.1158125},{"type":"node","id":1646077134,"lat":51.5053083,"lon":-0.1160296},{"type":"node","id":1646306110,"lat":51.5042777,"lon":-0.1169132},{"type":"node","id":1646306117,"lat":51.5043749,"lon":-0.1172925},{"type":"node","id":1646306121,"lat":51.5044449,"lon":-0.1170787},{"type":"node","id":1646306124,"lat":51.5045108,"lon":-0.1177903},{"type":"node","id":1646306128,"lat":51.5045303,"lon":-0.1178348},{"type":"node","id":1646306130,"lat":51.5045414,"lon":-0.1170192},{"type":"node","id":1646306132,"lat":51.504558,"lon":-0.1178701},{"type":"node","id":1646306134,"lat":51.5045762,"lon":-0.1169978},{"type":"node","id":1646306138,"lat":51.504584,"lon":-0.1178884},{"type":"node","id":1646306140,"lat":51.5046014,"lon":-0.1177398},{"type":"node","id":1646306149,"lat":51.5046365,"lon":-0.1172566},{"type":"node","id":1646306150,"lat":51.504648,"lon":-0.1173056},{"type":"node","id":1646306154,"lat":51.5047296,"lon":-0.1176698},{"type":"node","id":1646306156,"lat":51.5047594,"lon":-0.117703},{"type":"node","id":1646306158,"lat":51.5047892,"lon":-0.117833},{"type":"node","id":1646306164,"lat":51.5048737,"lon":-0.116445},{"type":"node","id":1646306191,"lat":51.5051757,"lon":-0.1178145},{"type":"node","id":1646306203,"lat":51.5052621,"lon":-0.1152267},{"type":"node","id":1822628006,"lat":51.5071825,"lon":-0.1152457},{"type":"node","id":1953721171,"lat":51.5054737,"lon":-0.1141701},{"type":"node","id":2105690372,"lat":51.5073614,"lon":-0.1157181},{"type":"node","id":2105690380,"lat":51.5068471,"lon":-0.1152579},{"type":"node","id":2140823543,"lat":51.5067166,"lon":-0.1171521},{"type":"node","id":2140823569,"lat":51.5066245,"lon":-0.1172836},{"type":"node","id":2140823620,"lat":51.5067678,"lon":-0.1170814},{"type":"node","id":2140823777,"lat":51.5065744,"lon":-0.1173573},{"type":"node","id":2334377717,"lat":51.5060691,"lon":-0.1145702},{"type":"node","id":2533348197,"lat":51.5065425,"lon":-0.1170078},{"type":"node","id":2533348198,"lat":51.5065479,"lon":-0.1170613},{"type":"node","id":2533348199,"lat":51.5066462,"lon":-0.1170369},{"type":"node","id":2885402275,"lat":51.5059733,"lon":-0.1148859},{"type":"node","id":3328660559,"lat":51.5059678,"lon":-0.116018},{"type":"node","id":3497406877,"lat":51.5065029,"lon":-0.1150956},{"type":"node","id":3519058521,"lat":51.5069628,"lon":-0.1153598},{"type":"node","id":4082316448,"lat":51.5059115,"lon":-0.1150396},{"type":"node","id":4082316449,"lat":51.5056623,"lon":-0.1155355},{"type":"node","id":4752385702,"lat":51.5055309,"lon":-0.1157376},{"type":"node","id":5094593323,"lat":51.50767,"lon":-0.1134066},{"type":"node","id":5094593324,"lat":51.5074375,"lon":-0.1132935},{"type":"node","id":5094593326,"lat":51.5075468,"lon":-0.1133838},{"type":"node","id":5094593327,"lat":51.5067432,"lon":-0.1127343},{"type":"node","id":5094593332,"lat":51.5068734,"lon":-0.1128437},{"type":"node","id":5094593335,"lat":51.5075204,"lon":-0.113282},{"type":"node","id":5094593340,"lat":51.5076599,"lon":-0.1133971},{"type":"node","id":5110757790,"lat":51.5076772,"lon":-0.113441},{"type":"node","id":5110757792,"lat":51.5076768,"lon":-0.1134229},{"type":"node","id":107697,"lat":51.5127888,"lon":-0.1144666},{"type":"node","id":107698,"lat":51.511267,"lon":-0.1136754},{"type":"node","id":107699,"lat":51.5109203,"lon":-0.1135266},{"type":"node","id":107708,"lat":51.5129232,"lon":-0.1136998},{"type":"node","id":107714,"lat":51.5124406,"lon":-0.1157494},{"type":"node","id":9475245,"lat":51.5132746,"lon":-0.1169498},{"type":"node","id":21590616,"lat":51.5137181,"lon":-0.1167035},{"type":"node","id":21590618,"lat":51.5140257,"lon":-0.1164522},{"type":"node","id":25378117,"lat":51.5124813,"lon":-0.1159902},{"type":"node","id":25378118,"lat":51.5128305,"lon":-0.1161189},{"type":"node","id":25378119,"lat":51.5131246,"lon":-0.1133298},{"type":"node","id":25378121,"lat":51.511858,"lon":-0.1121861},{"type":"node","id":25378123,"lat":51.5114144,"lon":-0.1120409},{"type":"node","id":25378124,"lat":51.5110492,"lon":-0.1119464},{"type":"node","id":25497907,"lat":51.5136936,"lon":-0.1144903},{"type":"node","id":25497908,"lat":51.5133459,"lon":-0.114317},{"type":"node","id":25497910,"lat":51.5143802,"lon":-0.1170921},{"type":"node","id":25497912,"lat":51.5143271,"lon":-0.1168995},{"type":"node","id":25497917,"lat":51.5144371,"lon":-0.115859},{"type":"node","id":25533129,"lat":51.5125927,"lon":-0.1127959},{"type":"node","id":25533132,"lat":51.5123991,"lon":-0.1133075},{"type":"node","id":25533150,"lat":51.5128956,"lon":-0.1139613},{"type":"node","id":25533164,"lat":51.5119869,"lon":-0.1127412},{"type":"node","id":25533170,"lat":51.5117804,"lon":-0.1126039},{"type":"node","id":25533174,"lat":51.5117168,"lon":-0.112512},{"type":"node","id":25533180,"lat":51.5117196,"lon":-0.1120831},{"type":"node","id":25533203,"lat":51.511009,"lon":-0.1149317},{"type":"node","id":25533211,"lat":51.5109361,"lon":-0.1152509},{"type":"node","id":25533215,"lat":51.510817,"lon":-0.1153629},{"type":"node","id":25533217,"lat":51.5107966,"lon":-0.1153791},{"type":"node","id":25737250,"lat":51.5121071,"lon":-0.1130375},{"type":"node","id":25737266,"lat":51.5119745,"lon":-0.1140585},{"type":"node","id":33141175,"lat":51.5135312,"lon":-0.1124974},{"type":"node","id":255747713,"lat":51.5145171,"lon":-0.116903},{"type":"node","id":255747714,"lat":51.5144123,"lon":-0.1161856},{"type":"node","id":255747716,"lat":51.5145626,"lon":-0.1160948},{"type":"node","id":255747728,"lat":51.5145115,"lon":-0.117504},{"type":"node","id":255747733,"lat":51.5144729,"lon":-0.116966},{"type":"node","id":255747735,"lat":51.5142133,"lon":-0.1163002},{"type":"node","id":256339270,"lat":51.5143322,"lon":-0.1173727},{"type":"node","id":270112539,"lat":51.5128152,"lon":-0.1119036},{"type":"node","id":270112581,"lat":51.5122424,"lon":-0.1118487},{"type":"node","id":270112583,"lat":51.5123548,"lon":-0.1119413},{"type":"node","id":280412704,"lat":51.5117872,"lon":-0.1178146},{"type":"node","id":281454958,"lat":51.5136353,"lon":-0.1178552},{"type":"node","id":282569738,"lat":51.5126553,"lon":-0.115046},{"type":"node","id":282569739,"lat":51.5126175,"lon":-0.1154463},{"type":"node","id":282569740,"lat":51.5123293,"lon":-0.1168605},{"type":"node","id":282569741,"lat":51.5121997,"lon":-0.1172043},{"type":"node","id":282569746,"lat":51.5120619,"lon":-0.1174156},{"type":"node","id":282569747,"lat":51.5119809,"lon":-0.1175801},{"type":"node","id":282569749,"lat":51.5119354,"lon":-0.1176979},{"type":"node","id":282569803,"lat":51.5128351,"lon":-0.1162515},{"type":"node","id":282569804,"lat":51.5128337,"lon":-0.1162229},{"type":"node","id":282569814,"lat":51.5140927,"lon":-0.1171847},{"type":"node","id":321255910,"lat":51.5128687,"lon":-0.1116799},{"type":"node","id":321255911,"lat":51.5125614,"lon":-0.1115108},{"type":"node","id":321255913,"lat":51.5131805,"lon":-0.1122093},{"type":"node","id":321255914,"lat":51.5132413,"lon":-0.1122563},{"type":"node","id":396093936,"lat":51.5136134,"lon":-0.1176276},{"type":"node","id":396093938,"lat":51.5131682,"lon":-0.1174642},{"type":"node","id":396093944,"lat":51.5130359,"lon":-0.1150867},{"type":"node","id":396093947,"lat":51.5129752,"lon":-0.1148936},{"type":"node","id":396093955,"lat":51.5128497,"lon":-0.1148341},{"type":"node","id":455705604,"lat":51.5118736,"lon":-0.1174976},{"type":"node","id":455705609,"lat":51.5133368,"lon":-0.1176555},{"type":"node","id":455705610,"lat":51.5133795,"lon":-0.1175232},{"type":"node","id":484375487,"lat":51.5109249,"lon":-0.1160653},{"type":"node","id":751735130,"lat":51.511471,"lon":-0.1173051},{"type":"node","id":751735148,"lat":51.5120269,"lon":-0.1168548},{"type":"node","id":751735214,"lat":51.5116685,"lon":-0.1166151},{"type":"node","id":751735253,"lat":51.5118633,"lon":-0.1167378},{"type":"node","id":751735259,"lat":51.5118224,"lon":-0.1176932},{"type":"node","id":751735296,"lat":51.5114072,"lon":-0.1175259},{"type":"node","id":751735355,"lat":51.5116233,"lon":-0.1167778},{"type":"node","id":751735365,"lat":51.5114386,"lon":-0.1174173},{"type":"node","id":1139387510,"lat":51.5126157,"lon":-0.113524},{"type":"node","id":1140700492,"lat":51.5129009,"lon":-0.1145953},{"type":"node","id":1140700560,"lat":51.5126778,"lon":-0.1161645},{"type":"node","id":1140700574,"lat":51.5130357,"lon":-0.1147384},{"type":"node","id":1237543404,"lat":51.5107108,"lon":-0.1166566},{"type":"node","id":1237543425,"lat":51.5103395,"lon":-0.1166673},{"type":"node","id":1238198746,"lat":51.510774,"lon":-0.1167037},{"type":"node","id":1238198769,"lat":51.5107503,"lon":-0.1167789},{"type":"node","id":1238198792,"lat":51.5106885,"lon":-0.1167327},{"type":"node","id":1419565097,"lat":51.5137245,"lon":-0.1116247},{"type":"node","id":1422415852,"lat":51.5117548,"lon":-0.115835},{"type":"node","id":1528450579,"lat":51.5132589,"lon":-0.1176697},{"type":"node","id":1528450580,"lat":51.513325,"lon":-0.1174684},{"type":"node","id":1538881758,"lat":51.51235,"lon":-0.1166843},{"type":"node","id":1538881759,"lat":51.5122446,"lon":-0.1171296},{"type":"node","id":1538881761,"lat":51.512502,"lon":-0.1167512},{"type":"node","id":1538881762,"lat":51.5123806,"lon":-0.1172307},{"type":"node","id":1630605982,"lat":51.5090358,"lon":-0.1172324},{"type":"node","id":1685903437,"lat":51.5122959,"lon":-0.1170255},{"type":"node","id":1685903439,"lat":51.5129905,"lon":-0.1135138},{"type":"node","id":1685903442,"lat":51.5131705,"lon":-0.1177276},{"type":"node","id":1685903446,"lat":51.5131312,"lon":-0.1175934},{"type":"node","id":1685903451,"lat":51.5128816,"lon":-0.1148274},{"type":"node","id":1685903452,"lat":51.5123693,"lon":-0.1164315},{"type":"node","id":1685903456,"lat":51.5121431,"lon":-0.1172841},{"type":"node","id":1697772135,"lat":51.513661,"lon":-0.1119679},{"type":"node","id":1730303400,"lat":51.5113039,"lon":-0.1134497},{"type":"node","id":1730303419,"lat":51.5113683,"lon":-0.1129224},{"type":"node","id":1730303422,"lat":51.5117096,"lon":-0.1122031},{"type":"node","id":1730303447,"lat":51.5114044,"lon":-0.1123186},{"type":"node","id":1734545131,"lat":51.5123199,"lon":-0.1169288},{"type":"node","id":1881001588,"lat":51.5090058,"lon":-0.1173477},{"type":"node","id":1938953358,"lat":51.5093471,"lon":-0.117503},{"type":"node","id":1938953359,"lat":51.5094726,"lon":-0.117617},{"type":"node","id":1938953363,"lat":51.5123508,"lon":-0.115989},{"type":"node","id":1938953364,"lat":51.5125495,"lon":-0.1154135},{"type":"node","id":1938953366,"lat":51.5127382,"lon":-0.1149052},{"type":"node","id":1938953367,"lat":51.5128049,"lon":-0.1136606},{"type":"node","id":1938953369,"lat":51.5128587,"lon":-0.114542},{"type":"node","id":1938953370,"lat":51.5129349,"lon":-0.1148536},{"type":"node","id":1938953371,"lat":51.5130024,"lon":-0.1149644},{"type":"node","id":1938953372,"lat":51.5132359,"lon":-0.1132352},{"type":"node","id":1938953374,"lat":51.5133094,"lon":-0.1130797},{"type":"node","id":1938953375,"lat":51.5134345,"lon":-0.113164},{"type":"node","id":1954581192,"lat":51.5132742,"lon":-0.1172695},{"type":"node","id":2140823576,"lat":51.5107244,"lon":-0.1123724},{"type":"node","id":2140823590,"lat":51.5107488,"lon":-0.1154282},{"type":"node","id":2140823631,"lat":51.5105905,"lon":-0.1155412},{"type":"node","id":2140823680,"lat":51.5106852,"lon":-0.1154847},{"type":"node","id":2140823733,"lat":51.5107747,"lon":-0.1123838},{"type":"node","id":2184963115,"lat":51.5101017,"lon":-0.1179572},{"type":"node","id":2184970824,"lat":51.5119693,"lon":-0.1170831},{"type":"node","id":2441990077,"lat":51.513308,"lon":-0.1174248},{"type":"node","id":2441990078,"lat":51.5134206,"lon":-0.1175508},{"type":"node","id":2441990080,"lat":51.5132872,"lon":-0.117351},{"type":"node","id":2441990081,"lat":51.5132004,"lon":-0.1176985},{"type":"node","id":2441990082,"lat":51.5134666,"lon":-0.1175738},{"type":"node","id":2533371661,"lat":51.5109352,"lon":-0.1147549},{"type":"node","id":2533371664,"lat":51.5109646,"lon":-0.1147702},{"type":"node","id":2533371666,"lat":51.5110343,"lon":-0.1148087},{"type":"node","id":2533371669,"lat":51.5111427,"lon":-0.1137434},{"type":"node","id":2533371671,"lat":51.5111767,"lon":-0.1137624},{"type":"node","id":2533371673,"lat":51.5112418,"lon":-0.1137979},{"type":"node","id":2586614735,"lat":51.5135635,"lon":-0.1123696},{"type":"node","id":2586614741,"lat":51.5136619,"lon":-0.1124221},{"type":"node","id":2586616722,"lat":51.5136892,"lon":-0.1118127},{"type":"node","id":2586616724,"lat":51.5137743,"lon":-0.1118596},{"type":"node","id":2586621166,"lat":51.5132907,"lon":-0.1138229},{"type":"node","id":2586621168,"lat":51.5134721,"lon":-0.1139208},{"type":"node","id":2586635273,"lat":51.5133671,"lon":-0.1129479},{"type":"node","id":2586635274,"lat":51.512067,"lon":-0.1140745},{"type":"node","id":2586635275,"lat":51.5128152,"lon":-0.1143485},{"type":"node","id":2586635276,"lat":51.5127712,"lon":-0.1145398},{"type":"node","id":2586635279,"lat":51.5128355,"lon":-0.1145134},{"type":"node","id":2586635280,"lat":51.5126281,"lon":-0.1143755},{"type":"node","id":2586635281,"lat":51.5128103,"lon":-0.1148519},{"type":"node","id":2586635283,"lat":51.5127271,"lon":-0.1146028},{"type":"node","id":2586635284,"lat":51.5135032,"lon":-0.1126044},{"type":"node","id":2586635285,"lat":51.512046,"lon":-0.1141441},{"type":"node","id":2586635288,"lat":51.5126814,"lon":-0.1145397},{"type":"node","id":2586635291,"lat":51.5127523,"lon":-0.1144421},{"type":"node","id":2586635292,"lat":51.5128257,"lon":-0.1143011},{"type":"node","id":2586635293,"lat":51.5126526,"lon":-0.1145121},{"type":"node","id":2586635295,"lat":51.5127405,"lon":-0.114681},{"type":"node","id":2586635296,"lat":51.5134781,"lon":-0.1129213},{"type":"node","id":3140855372,"lat":51.5132632,"lon":-0.1171477},{"type":"node","id":3140855373,"lat":51.5132454,"lon":-0.114265},{"type":"node","id":3140855374,"lat":51.5136927,"lon":-0.1144898},{"type":"node","id":3140855390,"lat":51.5140168,"lon":-0.1178948},{"type":"node","id":3140855394,"lat":51.5132922,"lon":-0.1166529},{"type":"node","id":3140855395,"lat":51.5132222,"lon":-0.1156703},{"type":"node","id":3140855396,"lat":51.513103,"lon":-0.1149205},{"type":"node","id":3140855397,"lat":51.5137215,"lon":-0.1147764},{"type":"node","id":3140855398,"lat":51.5137333,"lon":-0.114595},{"type":"node","id":3140855399,"lat":51.5139601,"lon":-0.1154066},{"type":"node","id":3140965156,"lat":51.5132812,"lon":-0.1160301},{"type":"node","id":3141011326,"lat":51.5130988,"lon":-0.1148455},{"type":"node","id":3141558547,"lat":51.5138465,"lon":-0.1150356},{"type":"node","id":3141587863,"lat":51.5141234,"lon":-0.1157245},{"type":"node","id":3142660158,"lat":51.5141863,"lon":-0.1163214},{"type":"node","id":3142670557,"lat":51.5137471,"lon":-0.1166798},{"type":"node","id":3142789923,"lat":51.5142828,"lon":-0.1155872},{"type":"node","id":3148313371,"lat":51.5142666,"lon":-0.1173226},{"type":"node","id":3149029365,"lat":51.5131105,"lon":-0.1147169},{"type":"node","id":3149029366,"lat":51.5132127,"lon":-0.1142481},{"type":"node","id":3149804732,"lat":51.5132975,"lon":-0.1163592},{"type":"node","id":3153598126,"lat":51.5137287,"lon":-0.1147963},{"type":"node","id":3155351702,"lat":51.5143805,"lon":-0.1160149},{"type":"node","id":3497448050,"lat":51.5132376,"lon":-0.1141126},{"type":"node","id":3519042187,"lat":51.5119122,"lon":-0.1173289},{"type":"node","id":3581332535,"lat":51.5141059,"lon":-0.1156764},{"type":"node","id":3581332539,"lat":51.5141182,"lon":-0.1157095},{"type":"node","id":3581332546,"lat":51.5135769,"lon":-0.1166869},{"type":"node","id":3585676160,"lat":51.5144704,"lon":-0.1161515},{"type":"node","id":3644748765,"lat":51.5132918,"lon":-0.1161216},{"type":"node","id":3811094139,"lat":51.5109689,"lon":-0.1130423},{"type":"node","id":3811094140,"lat":51.510841,"lon":-0.1140818},{"type":"node","id":3895360876,"lat":51.5123833,"lon":-0.1178216},{"type":"node","id":3895360879,"lat":51.5122581,"lon":-0.1179036},{"type":"node","id":3911044721,"lat":51.5105677,"lon":-0.1153036},{"type":"node","id":4012411306,"lat":51.5128475,"lon":-0.1165058},{"type":"node","id":4016579385,"lat":51.5128929,"lon":-0.1141343},{"type":"node","id":4059354472,"lat":51.5131944,"lon":-0.1143319},{"type":"node","id":4059354473,"lat":51.5121453,"lon":-0.1164972},{"type":"node","id":4068974161,"lat":51.5130614,"lon":-0.1178918},{"type":"node","id":4174200027,"lat":51.5099885,"lon":-0.1178541},{"type":"node","id":4174200028,"lat":51.5104898,"lon":-0.1160235},{"type":"node","id":4174200032,"lat":51.5106185,"lon":-0.1153971},{"type":"node","id":4174200033,"lat":51.5106305,"lon":-0.1153352},{"type":"node","id":4174200034,"lat":51.5106452,"lon":-0.1152615},{"type":"node","id":4174200035,"lat":51.5106685,"lon":-0.1152957},{"type":"node","id":4174200036,"lat":51.5106731,"lon":-0.115116},{"type":"node","id":4174200038,"lat":51.5107004,"lon":-0.1153704},{"type":"node","id":4174200039,"lat":51.5107054,"lon":-0.1153217},{"type":"node","id":4174200040,"lat":51.5107505,"lon":-0.1146622},{"type":"node","id":4174200043,"lat":51.5109612,"lon":-0.1119498},{"type":"node","id":4174200045,"lat":51.5110135,"lon":-0.1124777},{"type":"node","id":4174200046,"lat":51.5110476,"lon":-0.1114128},{"type":"node","id":4284097308,"lat":51.5143586,"lon":-0.1173926},{"type":"node","id":4285841903,"lat":51.5144704,"lon":-0.1158733},{"type":"node","id":4431563560,"lat":51.5131759,"lon":-0.1142674},{"type":"node","id":4431563561,"lat":51.5130916,"lon":-0.1142894},{"type":"node","id":4431563562,"lat":51.5130042,"lon":-0.114272},{"type":"node","id":4431563563,"lat":51.5129381,"lon":-0.1142306},{"type":"node","id":4431563564,"lat":51.5128704,"lon":-0.1140839},{"type":"node","id":4502770262,"lat":51.5112148,"lon":-0.1136771},{"type":"node","id":4502770263,"lat":51.5111667,"lon":-0.1136525},{"type":"node","id":4502770264,"lat":51.5112464,"lon":-0.1136916},{"type":"node","id":4684502368,"lat":51.5118031,"lon":-0.1167046},{"type":"node","id":4684502369,"lat":51.511914,"lon":-0.1167718},{"type":"node","id":4697525039,"lat":51.5111735,"lon":-0.1119786},{"type":"node","id":4697525047,"lat":51.5109812,"lon":-0.1135577},{"type":"node","id":4697525053,"lat":51.510695,"lon":-0.1153677},{"type":"node","id":4697525059,"lat":51.5102698,"lon":-0.1169247},{"type":"node","id":4697525062,"lat":51.5109874,"lon":-0.1152873},{"type":"node","id":4781199330,"lat":51.5126893,"lon":-0.112137},{"type":"node","id":4781199331,"lat":51.512641,"lon":-0.1120989},{"type":"node","id":4781199332,"lat":51.5125635,"lon":-0.1120961},{"type":"node","id":4781199338,"lat":51.5124484,"lon":-0.1120169},{"type":"node","id":4781199360,"lat":51.5117591,"lon":-0.1120969},{"type":"node","id":4781199361,"lat":51.5118199,"lon":-0.1121522},{"type":"node","id":4833195584,"lat":51.5144062,"lon":-0.116153},{"type":"node","id":4833195585,"lat":51.5144423,"lon":-0.116168},{"type":"node","id":5501130885,"lat":51.5121248,"lon":-0.1130539},{"type":"node","id":5532720347,"lat":51.5116026,"lon":-0.1138532},{"type":"node","id":5532720348,"lat":51.5116784,"lon":-0.1135143},{"type":"node","id":5532720349,"lat":51.5119956,"lon":-0.1136826},{"type":"node","id":5623296806,"lat":51.5142455,"lon":-0.1165645},{"type":"node","id":6139961783,"lat":51.5119265,"lon":-0.114032},{"type":"node","id":6140802454,"lat":51.5117181,"lon":-0.113535},{"type":"node","id":26374229,"lat":51.5110579,"lon":-0.1107141},{"type":"node","id":270109327,"lat":51.5126202,"lon":-0.1112064},{"type":"node","id":270112540,"lat":51.512947,"lon":-0.1113649},{"type":"node","id":2341656968,"lat":51.511034,"lon":-0.1079968},{"type":"node","id":3265436260,"lat":51.5110373,"lon":-0.1081398},{"type":"node","id":3811094141,"lat":51.5110588,"lon":-0.1093242},{"type":"node","id":3811094142,"lat":51.5110586,"lon":-0.110198},{"type":"node","id":4085747014,"lat":51.5110473,"lon":-0.1083646},{"type":"node","id":4174200047,"lat":51.5110579,"lon":-0.1100864},{"type":"node","id":6327557047,"lat":51.5110393,"lon":-0.1081838},{"type":"node","id":104302,"lat":51.5203627,"lon":-0.1303394},{"type":"node","id":107823,"lat":51.5152865,"lon":-0.1300895},{"type":"node","id":107824,"lat":51.5164132,"lon":-0.1303768},{"type":"node","id":107831,"lat":51.5169182,"lon":-0.1268899},{"type":"node","id":107839,"lat":51.5205708,"lon":-0.1263434},{"type":"node","id":107842,"lat":51.5196863,"lon":-0.1295881},{"type":"node","id":107843,"lat":51.5200794,"lon":-0.1286719},{"type":"node","id":107844,"lat":51.5207567,"lon":-0.1294315},{"type":"node","id":107851,"lat":51.5189481,"lon":-0.128798},{"type":"node","id":107852,"lat":51.5182278,"lon":-0.1261146},{"type":"node","id":107853,"lat":51.5178022,"lon":-0.1276215},{"type":"node","id":108014,"lat":51.5209576,"lon":-0.1268219},{"type":"node","id":108028,"lat":51.5201055,"lon":-0.1309273},{"type":"node","id":108067,"lat":51.5146586,"lon":-0.1273566},{"type":"node","id":108068,"lat":51.5155167,"lon":-0.1267532},{"type":"node","id":9791482,"lat":51.517086,"lon":-0.1307776},{"type":"node","id":10574891,"lat":51.5151842,"lon":-0.1269339},{"type":"node","id":13250138,"lat":51.5165933,"lon":-0.1289161},{"type":"node","id":14727249,"lat":51.5179251,"lon":-0.1257703},{"type":"node","id":14791174,"lat":51.5182485,"lon":-0.1304367},{"type":"node","id":14791182,"lat":51.5185576,"lon":-0.1283787},{"type":"node","id":14791186,"lat":51.51785,"lon":-0.130067},{"type":"node","id":14791190,"lat":51.5176222,"lon":-0.1306738},{"type":"node","id":14791193,"lat":51.5173455,"lon":-0.1296518},{"type":"node","id":21554368,"lat":51.5170477,"lon":-0.1258407},{"type":"node","id":21665008,"lat":51.5147637,"lon":-0.1303598},{"type":"node","id":21665009,"lat":51.5149397,"lon":-0.1304417},{"type":"node","id":21665080,"lat":51.5156257,"lon":-0.1309507},{"type":"node","id":21665081,"lat":51.5148654,"lon":-0.1298854},{"type":"node","id":21665130,"lat":51.5157625,"lon":-0.1290374},{"type":"node","id":21665131,"lat":51.5156023,"lon":-0.1285481},{"type":"node","id":21665132,"lat":51.5155971,"lon":-0.1284294},{"type":"node","id":21665133,"lat":51.5155689,"lon":-0.1279044},{"type":"node","id":21665134,"lat":51.515566,"lon":-0.1276367},{"type":"node","id":21665135,"lat":51.5155741,"lon":-0.1274219},{"type":"node","id":21665136,"lat":51.5156466,"lon":-0.1268929},{"type":"node","id":21665137,"lat":51.5156675,"lon":-0.1267779},{"type":"node","id":21665139,"lat":51.5151401,"lon":-0.129674},{"type":"node","id":21665140,"lat":51.515489,"lon":-0.1289488},{"type":"node","id":21665141,"lat":51.5156,"lon":-0.1286889},{"type":"node","id":21665247,"lat":51.5174796,"lon":-0.126224},{"type":"node","id":21665248,"lat":51.5177418,"lon":-0.1255735},{"type":"node","id":21665587,"lat":51.5164685,"lon":-0.1296463},{"type":"node","id":21665588,"lat":51.5156901,"lon":-0.1266742},{"type":"node","id":21665589,"lat":51.5150428,"lon":-0.129961},{"type":"node","id":21665590,"lat":51.5172613,"lon":-0.1250452},{"type":"node","id":25375812,"lat":51.5170905,"lon":-0.1279102},{"type":"node","id":25375954,"lat":51.5167017,"lon":-0.1295242},{"type":"node","id":25375955,"lat":51.5165075,"lon":-0.1294551},{"type":"node","id":25375957,"lat":51.5167927,"lon":-0.1275886},{"type":"node","id":25375958,"lat":51.5176238,"lon":-0.1284127},{"type":"node","id":25375960,"lat":51.517317,"lon":-0.1272267},{"type":"node","id":25375976,"lat":51.5165546,"lon":-0.1272374},{"type":"node","id":25376002,"lat":51.5166393,"lon":-0.1266652},{"type":"node","id":25376033,"lat":51.5170149,"lon":-0.1309929},{"type":"node","id":25470853,"lat":51.5188018,"lon":-0.1247317},{"type":"node","id":25471038,"lat":51.5210277,"lon":-0.1289512},{"type":"node","id":25472654,"lat":51.518006,"lon":-0.1267345},{"type":"node","id":25472666,"lat":51.5176279,"lon":-0.1263439},{"type":"node","id":25472672,"lat":51.5179111,"lon":-0.1271056},{"type":"node","id":25472674,"lat":51.5174708,"lon":-0.126793},{"type":"node","id":25529874,"lat":51.5153857,"lon":-0.127463},{"type":"node","id":25529875,"lat":51.5151287,"lon":-0.1275756},{"type":"node","id":25529876,"lat":51.5148716,"lon":-0.1277419},{"type":"node","id":25530024,"lat":51.5152105,"lon":-0.1287464},{"type":"node","id":25530033,"lat":51.5151599,"lon":-0.1287578},{"type":"node","id":25530034,"lat":51.5149717,"lon":-0.1288041},{"type":"node","id":25530036,"lat":51.5148769,"lon":-0.1289883},{"type":"node","id":25530312,"lat":51.5159823,"lon":-0.128653},{"type":"node","id":25736719,"lat":51.5146306,"lon":-0.1268704},{"type":"node","id":26699540,"lat":51.5151557,"lon":-0.1306564},{"type":"node","id":26699559,"lat":51.5158247,"lon":-0.1259847},{"type":"node","id":26699560,"lat":51.5162859,"lon":-0.1263453},{"type":"node","id":26699561,"lat":51.5159739,"lon":-0.1254008},{"type":"node","id":26699562,"lat":51.5167508,"lon":-0.1260371},{"type":"node","id":26699563,"lat":51.5166088,"lon":-0.1253786},{"type":"node","id":26699565,"lat":51.5158289,"lon":-0.1265786},{"type":"node","id":26848675,"lat":51.516493,"lon":-0.1271734},{"type":"node","id":26848680,"lat":51.5167885,"lon":-0.1247963},{"type":"node","id":257810133,"lat":51.5211132,"lon":-0.1250351},{"type":"node","id":281455009,"lat":51.5171097,"lon":-0.1249763},{"type":"node","id":392132821,"lat":51.5153007,"lon":-0.1300965},{"type":"node","id":438170606,"lat":51.5145865,"lon":-0.125394},{"type":"node","id":447581785,"lat":51.5164042,"lon":-0.1309312},{"type":"node","id":469760236,"lat":51.5166878,"lon":-0.1282597},{"type":"node","id":469760241,"lat":51.5155747,"lon":-0.1283265},{"type":"node","id":490256481,"lat":51.5147054,"lon":-0.1288634},{"type":"node","id":499122575,"lat":51.5155829,"lon":-0.1309275},{"type":"node","id":499122580,"lat":51.5152266,"lon":-0.1307033},{"type":"node","id":501766800,"lat":51.5147009,"lon":-0.1283691},{"type":"node","id":501766819,"lat":51.5147086,"lon":-0.1283762},{"type":"node","id":634568066,"lat":51.5183299,"lon":-0.1258618},{"type":"node","id":634568069,"lat":51.5183907,"lon":-0.1257112},{"type":"node","id":634568087,"lat":51.5188618,"lon":-0.1262531},{"type":"node","id":764870818,"lat":51.5193182,"lon":-0.1291999},{"type":"node","id":764870828,"lat":51.5185859,"lon":-0.129657},{"type":"node","id":764870843,"lat":51.5193076,"lon":-0.1304639},{"type":"node","id":764870861,"lat":51.5186097,"lon":-0.1308025},{"type":"node","id":866311216,"lat":51.5169444,"lon":-0.1257317},{"type":"node","id":866311227,"lat":51.5167158,"lon":-0.1274183},{"type":"node","id":866311234,"lat":51.5171855,"lon":-0.1249994},{"type":"node","id":874823434,"lat":51.5198409,"lon":-0.1254851},{"type":"node","id":881889759,"lat":51.5200926,"lon":-0.1300417},{"type":"node","id":949045792,"lat":51.5166181,"lon":-0.1254125},{"type":"node","id":949045824,"lat":51.5163273,"lon":-0.1256402},{"type":"node","id":949045884,"lat":51.5163967,"lon":-0.1247461},{"type":"node","id":949045924,"lat":51.5164413,"lon":-0.1245997},{"type":"node","id":949076158,"lat":51.5173439,"lon":-0.1271599},{"type":"node","id":949076160,"lat":51.5174565,"lon":-0.1268377},{"type":"node","id":970857026,"lat":51.5173915,"lon":-0.1294488},{"type":"node","id":970857045,"lat":51.516863,"lon":-0.129042},{"type":"node","id":976973186,"lat":51.5206128,"lon":-0.1262374},{"type":"node","id":983836443,"lat":51.5204093,"lon":-0.1280631},{"type":"node","id":1080591766,"lat":51.5162602,"lon":-0.1278291},{"type":"node","id":1080591780,"lat":51.5156363,"lon":-0.128477},{"type":"node","id":1103001639,"lat":51.5156025,"lon":-0.1286079},{"type":"node","id":1103268830,"lat":51.5154569,"lon":-0.1280911},{"type":"node","id":1103268833,"lat":51.5154819,"lon":-0.1280949},{"type":"node","id":1103268839,"lat":51.5155731,"lon":-0.1281103},{"type":"node","id":1104448381,"lat":51.5146346,"lon":-0.1282854},{"type":"node","id":1105928793,"lat":51.5160497,"lon":-0.128387},{"type":"node","id":1105928825,"lat":51.5163016,"lon":-0.1288216},{"type":"node","id":1109736448,"lat":51.518829,"lon":-0.1262079},{"type":"node","id":1109736452,"lat":51.5184775,"lon":-0.1258062},{"type":"node","id":1115431333,"lat":51.5150102,"lon":-0.1259852},{"type":"node","id":1115431335,"lat":51.5150709,"lon":-0.1260849},{"type":"node","id":1115431337,"lat":51.5150886,"lon":-0.1260637},{"type":"node","id":1115431346,"lat":51.5151866,"lon":-0.1258369},{"type":"node","id":1115431348,"lat":51.5152321,"lon":-0.1257453},{"type":"node","id":1156353853,"lat":51.520767,"lon":-0.1308153},{"type":"node","id":1230855633,"lat":51.5171486,"lon":-0.1263374},{"type":"node","id":1230855643,"lat":51.5170901,"lon":-0.1262976},{"type":"node","id":1230855652,"lat":51.516989,"lon":-0.1262636},{"type":"node","id":1392029481,"lat":51.5197928,"lon":-0.1254294},{"type":"node","id":1413151219,"lat":51.5175115,"lon":-0.1262504},{"type":"node","id":1413151220,"lat":51.517587,"lon":-0.1263119},{"type":"node","id":1448889397,"lat":51.5207392,"lon":-0.1294118},{"type":"node","id":1448889398,"lat":51.5207908,"lon":-0.1294694},{"type":"node","id":1448889399,"lat":51.5207316,"lon":-0.1294867},{"type":"node","id":1448890899,"lat":51.5203471,"lon":-0.1303752},{"type":"node","id":1604752665,"lat":51.5181228,"lon":-0.1278957},{"type":"node","id":1616122196,"lat":51.5145964,"lon":-0.1246165},{"type":"node","id":1616122219,"lat":51.5146226,"lon":-0.1246835},{"type":"node","id":1616122237,"lat":51.5148259,"lon":-0.1249822},{"type":"node","id":1616122241,"lat":51.5148628,"lon":-0.1250542},{"type":"node","id":1616122244,"lat":51.5149405,"lon":-0.1252164},{"type":"node","id":1616122245,"lat":51.5150709,"lon":-0.1254803},{"type":"node","id":1616122247,"lat":51.5151403,"lon":-0.1256114},{"type":"node","id":1616122250,"lat":51.5152919,"lon":-0.1258195},{"type":"node","id":1616122251,"lat":51.5153823,"lon":-0.1259242},{"type":"node","id":1616122252,"lat":51.5155689,"lon":-0.1260995},{"type":"node","id":1616122253,"lat":51.5157712,"lon":-0.126344},{"type":"node","id":1700106533,"lat":51.5169353,"lon":-0.126756},{"type":"node","id":1700106576,"lat":51.5170025,"lon":-0.1269616},{"type":"node","id":1700106707,"lat":51.5171136,"lon":-0.1258871},{"type":"node","id":1700140611,"lat":51.5161805,"lon":-0.1262571},{"type":"node","id":1700140613,"lat":51.5162174,"lon":-0.1261594},{"type":"node","id":1700140638,"lat":51.5161945,"lon":-0.1262671},{"type":"node","id":1700140653,"lat":51.5162575,"lon":-0.1260128},{"type":"node","id":1700140657,"lat":51.5162162,"lon":-0.1260229},{"type":"node","id":1795369607,"lat":51.5169145,"lon":-0.1306375},{"type":"node","id":1795369609,"lat":51.5164438,"lon":-0.1297656},{"type":"node","id":1795369613,"lat":51.5164052,"lon":-0.1301129},{"type":"node","id":1843345666,"lat":51.5210377,"lon":-0.1252105},{"type":"node","id":1917794487,"lat":51.5161324,"lon":-0.1295094},{"type":"node","id":1917794490,"lat":51.5163419,"lon":-0.1295918},{"type":"node","id":1927220454,"lat":51.5160009,"lon":-0.1293131},{"type":"node","id":1937391902,"lat":51.5160699,"lon":-0.1283255},{"type":"node","id":1937391977,"lat":51.5170336,"lon":-0.1259745},{"type":"node","id":1938953380,"lat":51.516407,"lon":-0.1302478},{"type":"node","id":1938953381,"lat":51.51642,"lon":-0.1299552},{"type":"node","id":1938953382,"lat":51.5164162,"lon":-0.1305351},{"type":"node","id":2094955592,"lat":51.5168061,"lon":-0.1274908},{"type":"node","id":2096473199,"lat":51.5172093,"lon":-0.1251464},{"type":"node","id":2096561060,"lat":51.5171398,"lon":-0.1247897},{"type":"node","id":2096561065,"lat":51.5170594,"lon":-0.1249533},{"type":"node","id":2096561071,"lat":51.5170234,"lon":-0.1249347},{"type":"node","id":2096561074,"lat":51.5171337,"lon":-0.1248611},{"type":"node","id":2096561079,"lat":51.5171011,"lon":-0.1250695},{"type":"node","id":2159428766,"lat":51.5179042,"lon":-0.1257481},{"type":"node","id":2159428767,"lat":51.5177789,"lon":-0.1256148},{"type":"node","id":2284375270,"lat":51.5206597,"lon":-0.1264584},{"type":"node","id":2414156537,"lat":51.5153918,"lon":-0.1301424},{"type":"node","id":2425138404,"lat":51.5164185,"lon":-0.1306113},{"type":"node","id":2425138405,"lat":51.5164763,"lon":-0.1303836},{"type":"node","id":2425138406,"lat":51.5164037,"lon":-0.1301536},{"type":"node","id":2425138407,"lat":51.5165349,"lon":-0.1304},{"type":"node","id":2441998875,"lat":51.5150593,"lon":-0.1268119},{"type":"node","id":2479767093,"lat":51.5166189,"lon":-0.1287483},{"type":"node","id":2479767095,"lat":51.5165709,"lon":-0.1290699},{"type":"node","id":2479767098,"lat":51.5164848,"lon":-0.1288748},{"type":"node","id":2493654751,"lat":51.5146355,"lon":-0.1309004},{"type":"node","id":2496862506,"lat":51.5157684,"lon":-0.1289698},{"type":"node","id":2496863224,"lat":51.5155844,"lon":-0.1287513},{"type":"node","id":2496863225,"lat":51.5155541,"lon":-0.1288158},{"type":"node","id":2502928029,"lat":51.5179281,"lon":-0.1301386},{"type":"node","id":2556043427,"lat":51.5182982,"lon":-0.1281178},{"type":"node","id":2622506576,"lat":51.5186212,"lon":-0.1245373},{"type":"node","id":2833445980,"lat":51.5145936,"lon":-0.1308776},{"type":"node","id":2833445982,"lat":51.514649,"lon":-0.1309281},{"type":"node","id":2833448424,"lat":51.5147522,"lon":-0.1304102},{"type":"node","id":2833448429,"lat":51.5148133,"lon":-0.1304376},{"type":"node","id":2833448450,"lat":51.5149247,"lon":-0.1304922},{"type":"node","id":2833448458,"lat":51.5149878,"lon":-0.1299344},{"type":"node","id":2833448475,"lat":51.5151295,"lon":-0.1300069},{"type":"node","id":2961254826,"lat":51.5154339,"lon":-0.1290734},{"type":"node","id":2971399439,"lat":51.5155825,"lon":-0.1302206},{"type":"node","id":2971399440,"lat":51.5157563,"lon":-0.1302817},{"type":"node","id":2971399443,"lat":51.5162564,"lon":-0.1301449},{"type":"node","id":2971399444,"lat":51.5158904,"lon":-0.1297749},{"type":"node","id":3054882706,"lat":51.5173317,"lon":-0.1271934},{"type":"node","id":3062596155,"lat":51.5168924,"lon":-0.1290613},{"type":"node","id":3254337787,"lat":51.518744,"lon":-0.1246724},{"type":"node","id":3254337788,"lat":51.5188714,"lon":-0.1245726},{"type":"node","id":3519181426,"lat":51.5184396,"lon":-0.1282467},{"type":"node","id":3645263633,"lat":51.5203848,"lon":-0.1302918},{"type":"node","id":3645263634,"lat":51.5204185,"lon":-0.1303964},{"type":"node","id":3782162182,"lat":51.5151139,"lon":-0.126009},{"type":"node","id":4222962166,"lat":51.5197366,"lon":-0.1294707},{"type":"node","id":4222962171,"lat":51.5197464,"lon":-0.1296598},{"type":"node","id":4231049411,"lat":51.5196638,"lon":-0.1296402},{"type":"node","id":4231050400,"lat":51.5182317,"lon":-0.130421},{"type":"node","id":4231050414,"lat":51.5182671,"lon":-0.1303938},{"type":"node","id":4231050595,"lat":51.5189294,"lon":-0.1288465},{"type":"node","id":4236640051,"lat":51.5182034,"lon":-0.1279771},{"type":"node","id":4236640057,"lat":51.5182413,"lon":-0.1280574},{"type":"node","id":4236640064,"lat":51.5182626,"lon":-0.1279982},{"type":"node","id":4236640084,"lat":51.5183194,"lon":-0.1280579},{"type":"node","id":4236640087,"lat":51.5183317,"lon":-0.1281556},{"type":"node","id":4236640531,"lat":51.5177391,"lon":-0.127573},{"type":"node","id":4236640538,"lat":51.5178235,"lon":-0.1274763},{"type":"node","id":4236640539,"lat":51.5178409,"lon":-0.1276504},{"type":"node","id":4236640689,"lat":51.5183392,"lon":-0.1280787},{"type":"node","id":4236640693,"lat":51.5183513,"lon":-0.1281093},{"type":"node","id":4236640701,"lat":51.5183703,"lon":-0.1281668},{"type":"node","id":4237160296,"lat":51.5177571,"lon":-0.1278624},{"type":"node","id":4237160305,"lat":51.5177833,"lon":-0.1277368},{"type":"node","id":4237160366,"lat":51.5181697,"lon":-0.1290601},{"type":"node","id":4237160372,"lat":51.51824,"lon":-0.1291364},{"type":"node","id":4237160373,"lat":51.5182593,"lon":-0.1288399},{"type":"node","id":4237160380,"lat":51.5183306,"lon":-0.1289203},{"type":"node","id":4239163351,"lat":51.5173075,"lon":-0.1281157},{"type":"node","id":4239163353,"lat":51.5173115,"lon":-0.1294491},{"type":"node","id":4239163378,"lat":51.5173811,"lon":-0.1295007},{"type":"node","id":4239163380,"lat":51.5173789,"lon":-0.1281833},{"type":"node","id":4239163381,"lat":51.5173802,"lon":-0.1282096},{"type":"node","id":4239163382,"lat":51.5173805,"lon":-0.1291722},{"type":"node","id":4239163385,"lat":51.5173851,"lon":-0.128307},{"type":"node","id":4239163388,"lat":51.5173957,"lon":-0.1291044},{"type":"node","id":4239163469,"lat":51.5170762,"lon":-0.1278948},{"type":"node","id":4239163474,"lat":51.51711,"lon":-0.1279287},{"type":"node","id":4239163997,"lat":51.5165764,"lon":-0.1294809},{"type":"node","id":4239164016,"lat":51.5166904,"lon":-0.1295255},{"type":"node","id":4239164026,"lat":51.5167148,"lon":-0.1294987},{"type":"node","id":4239164036,"lat":51.5168477,"lon":-0.1276526},{"type":"node","id":4239164053,"lat":51.5170029,"lon":-0.129139},{"type":"node","id":4239164192,"lat":51.5172033,"lon":-0.1293859},{"type":"node","id":4239164207,"lat":51.5172612,"lon":-0.128254},{"type":"node","id":4239164219,"lat":51.5173014,"lon":-0.1281338},{"type":"node","id":4239164289,"lat":51.5173965,"lon":-0.1283295},{"type":"node","id":4239164293,"lat":51.5174112,"lon":-0.1290283},{"type":"node","id":4239164301,"lat":51.5174326,"lon":-0.1283539},{"type":"node","id":4239164309,"lat":51.5174552,"lon":-0.1288525},{"type":"node","id":4239164311,"lat":51.5174663,"lon":-0.1288686},{"type":"node","id":4239164319,"lat":51.5174899,"lon":-0.1288835},{"type":"node","id":4239164327,"lat":51.517505,"lon":-0.1286317},{"type":"node","id":4239164330,"lat":51.5175144,"lon":-0.1286269},{"type":"node","id":4239164331,"lat":51.5175156,"lon":-0.1288998},{"type":"node","id":4239164333,"lat":51.5175285,"lon":-0.1286284},{"type":"node","id":4239164339,"lat":51.5175435,"lon":-0.1286349},{"type":"node","id":4239164344,"lat":51.5175726,"lon":-0.1286475},{"type":"node","id":4239164347,"lat":51.5176575,"lon":-0.1282557},{"type":"node","id":4239164350,"lat":51.5176897,"lon":-0.1281683},{"type":"node","id":4239701213,"lat":51.5174345,"lon":-0.1298691},{"type":"node","id":4239701247,"lat":51.5174675,"lon":-0.1297499},{"type":"node","id":4239701265,"lat":51.517576,"lon":-0.1299797},{"type":"node","id":4239701276,"lat":51.5176118,"lon":-0.1298687},{"type":"node","id":4242112079,"lat":51.5171016,"lon":-0.1307217},{"type":"node","id":4242112322,"lat":51.5173871,"lon":-0.1296852},{"type":"node","id":4242112323,"lat":51.5176002,"lon":-0.1283906},{"type":"node","id":4242112325,"lat":51.5177925,"lon":-0.1276895},{"type":"node","id":4356572310,"lat":51.5155687,"lon":-0.1266926},{"type":"node","id":4356572314,"lat":51.5161922,"lon":-0.1246545},{"type":"node","id":4356572316,"lat":51.515634,"lon":-0.1265505},{"type":"node","id":4356572321,"lat":51.5155715,"lon":-0.1268483},{"type":"node","id":4356572322,"lat":51.5157651,"lon":-0.1261915},{"type":"node","id":4356577280,"lat":51.5162414,"lon":-0.1245306},{"type":"node","id":4356609718,"lat":51.5173151,"lon":-0.1249176},{"type":"node","id":4360406513,"lat":51.5172236,"lon":-0.1250225},{"type":"node","id":4360406521,"lat":51.5173016,"lon":-0.1250895},{"type":"node","id":4360435541,"lat":51.5181869,"lon":-0.1260681},{"type":"node","id":4360435542,"lat":51.5183729,"lon":-0.1259008},{"type":"node","id":4360451851,"lat":51.5206345,"lon":-0.1261822},{"type":"node","id":4360451852,"lat":51.5205222,"lon":-0.1262872},{"type":"node","id":4360451853,"lat":51.5204697,"lon":-0.1262298},{"type":"node","id":4360454515,"lat":51.5207108,"lon":-0.12652},{"type":"node","id":4360457244,"lat":51.5208426,"lon":-0.1249886},{"type":"node","id":4360457245,"lat":51.5209758,"lon":-0.1253596},{"type":"node","id":4380848928,"lat":51.5150818,"lon":-0.1266792},{"type":"node","id":4409200420,"lat":51.5162269,"lon":-0.1287839},{"type":"node","id":4409200422,"lat":51.5167082,"lon":-0.1281262},{"type":"node","id":4409200424,"lat":51.5165419,"lon":-0.1265767},{"type":"node","id":4409200425,"lat":51.5167544,"lon":-0.1278336},{"type":"node","id":4409200426,"lat":51.517434,"lon":-0.1246359},{"type":"node","id":4676738223,"lat":51.5150846,"lon":-0.1270126},{"type":"node","id":4682210575,"lat":51.5185322,"lon":-0.1253766},{"type":"node","id":4715890870,"lat":51.5207761,"lon":-0.1294512},{"type":"node","id":4715890871,"lat":51.5206811,"lon":-0.1293492},{"type":"node","id":4715890872,"lat":51.5209264,"lon":-0.1288423},{"type":"node","id":4860880474,"lat":51.5197521,"lon":-0.1294345},{"type":"node","id":4942576593,"lat":51.5155974,"lon":-0.1272288},{"type":"node","id":5128425114,"lat":51.5147361,"lon":-0.1304926},{"type":"node","id":5339077113,"lat":51.5187766,"lon":-0.1247921},{"type":"node","id":5339970474,"lat":51.5177655,"lon":-0.1255991},{"type":"node","id":5339970475,"lat":51.5179088,"lon":-0.125753},{"type":"node","id":5339970477,"lat":51.5179368,"lon":-0.1257417},{"type":"node","id":5344928729,"lat":51.5159019,"lon":-0.1297424},{"type":"node","id":5350044101,"lat":51.5175821,"lon":-0.1259714},{"type":"node","id":5350044723,"lat":51.5177586,"lon":-0.1255339},{"type":"node","id":5350044724,"lat":51.5177301,"lon":-0.1256028},{"type":"node","id":5350044725,"lat":51.5172961,"lon":-0.1250834},{"type":"node","id":5355634693,"lat":51.5170995,"lon":-0.1258764},{"type":"node","id":5355634694,"lat":51.5174964,"lon":-0.1262396},{"type":"node","id":5355634695,"lat":51.5174919,"lon":-0.1261937},{"type":"node","id":5359989224,"lat":51.5175856,"lon":-0.1268906},{"type":"node","id":5359989234,"lat":51.5176854,"lon":-0.1269677},{"type":"node","id":5362237836,"lat":51.5176138,"lon":-0.1263843},{"type":"node","id":5362237839,"lat":51.5174641,"lon":-0.1268139},{"type":"node","id":5362237840,"lat":51.5178827,"lon":-0.1270882},{"type":"node","id":5362237842,"lat":51.5174922,"lon":-0.1268112},{"type":"node","id":5362237856,"lat":51.5179439,"lon":-0.1269775},{"type":"node","id":5404583125,"lat":51.5145893,"lon":-0.124554},{"type":"node","id":5413338307,"lat":51.5148704,"lon":-0.1249078},{"type":"node","id":5419519686,"lat":51.5157569,"lon":-0.1310376},{"type":"node","id":5420170453,"lat":51.5146793,"lon":-0.1252384},{"type":"node","id":5420170454,"lat":51.5146534,"lon":-0.1253007},{"type":"node","id":5434424321,"lat":51.5172648,"lon":-0.130006},{"type":"node","id":5454295758,"lat":51.515097,"lon":-0.126807},{"type":"node","id":5454295759,"lat":51.5151041,"lon":-0.1267518},{"type":"node","id":5454295779,"lat":51.5146562,"lon":-0.1268661},{"type":"node","id":5461196612,"lat":51.5164055,"lon":-0.130116},{"type":"node","id":5461196613,"lat":51.5158714,"lon":-0.1298502},{"type":"node","id":5461196614,"lat":51.5165669,"lon":-0.1301739},{"type":"node","id":5461196618,"lat":51.5164081,"lon":-0.1301579},{"type":"node","id":5461199021,"lat":51.5162756,"lon":-0.1300088},{"type":"node","id":5461199345,"lat":51.5159021,"lon":-0.1302484},{"type":"node","id":5461199346,"lat":51.5156855,"lon":-0.1301534},{"type":"node","id":5461199354,"lat":51.51622,"lon":-0.1305461},{"type":"node","id":5461199355,"lat":51.5161924,"lon":-0.1306528},{"type":"node","id":5461199356,"lat":51.5161758,"lon":-0.1306417},{"type":"node","id":5461199357,"lat":51.5162034,"lon":-0.130535},{"type":"node","id":5461199376,"lat":51.5163814,"lon":-0.1305017},{"type":"node","id":5461199380,"lat":51.5162917,"lon":-0.13045},{"type":"node","id":5461199381,"lat":51.5163575,"lon":-0.1304665},{"type":"node","id":5461199570,"lat":51.5162476,"lon":-0.130481},{"type":"node","id":5461199572,"lat":51.5162599,"lon":-0.1304331},{"type":"node","id":5461199578,"lat":51.5162,"lon":-0.1303464},{"type":"node","id":5461199579,"lat":51.5162453,"lon":-0.1303765},{"type":"node","id":5461199585,"lat":51.5161546,"lon":-0.1303877},{"type":"node","id":5461199591,"lat":51.5163771,"lon":-0.1305462},{"type":"node","id":5461199592,"lat":51.516362,"lon":-0.1307625},{"type":"node","id":5461199593,"lat":51.516364,"lon":-0.1307176},{"type":"node","id":5461199597,"lat":51.5160844,"lon":-0.1303519},{"type":"node","id":5461199722,"lat":51.5163575,"lon":-0.1308629},{"type":"node","id":5461199723,"lat":51.5163582,"lon":-0.130846},{"type":"node","id":5461200025,"lat":51.5163518,"lon":-0.1307901},{"type":"node","id":5461200026,"lat":51.5162928,"lon":-0.130775},{"type":"node","id":5461200041,"lat":51.5163122,"lon":-0.1309597},{"type":"node","id":5461200042,"lat":51.5163046,"lon":-0.1310658},{"type":"node","id":5461200147,"lat":51.5154639,"lon":-0.1300511},{"type":"node","id":5461200149,"lat":51.5154768,"lon":-0.1300572},{"type":"node","id":5461200150,"lat":51.5155271,"lon":-0.1300817},{"type":"node","id":5461200151,"lat":51.5153651,"lon":-0.130003},{"type":"node","id":5461200152,"lat":51.5152265,"lon":-0.1298999},{"type":"node","id":5461200163,"lat":51.5153148,"lon":-0.1299785},{"type":"node","id":5461200166,"lat":51.5151759,"lon":-0.129877},{"type":"node","id":5461200168,"lat":51.5153019,"lon":-0.1299725},{"type":"node","id":5461200169,"lat":51.515163,"lon":-0.1298711},{"type":"node","id":5461200171,"lat":51.5161915,"lon":-0.1306252},{"type":"node","id":5461200172,"lat":51.5161865,"lon":-0.1306588},{"type":"node","id":5461200173,"lat":51.5161817,"lon":-0.1306916},{"type":"node","id":5461200178,"lat":51.5159236,"lon":-0.1299029},{"type":"node","id":5461200179,"lat":51.5159286,"lon":-0.129868},{"type":"node","id":5461200180,"lat":51.5159388,"lon":-0.129837},{"type":"node","id":5461200182,"lat":51.5159964,"lon":-0.1298598},{"type":"node","id":5461200183,"lat":51.5160094,"lon":-0.129865},{"type":"node","id":5461200184,"lat":51.5160669,"lon":-0.1298884},{"type":"node","id":5461200186,"lat":51.5162193,"lon":-0.129989},{"type":"node","id":5461200188,"lat":51.5162063,"lon":-0.1299838},{"type":"node","id":5461200189,"lat":51.5154102,"lon":-0.130025},{"type":"node","id":5461200190,"lat":51.5152516,"lon":-0.129948},{"type":"node","id":5461200191,"lat":51.5151124,"lon":-0.1298482},{"type":"node","id":5461200356,"lat":51.515801,"lon":-0.1299363},{"type":"node","id":5461200357,"lat":51.5154706,"lon":-0.1297659},{"type":"node","id":5461200358,"lat":51.5157952,"lon":-0.129971},{"type":"node","id":5461200359,"lat":51.515464,"lon":-0.1298},{"type":"node","id":5461200360,"lat":51.5157891,"lon":-0.1300072},{"type":"node","id":5461200361,"lat":51.5154573,"lon":-0.1298349},{"type":"node","id":5461200393,"lat":51.5165009,"lon":-0.1301673},{"type":"node","id":5461200394,"lat":51.5160551,"lon":-0.1299182},{"type":"node","id":5461200395,"lat":51.5160501,"lon":-0.129953},{"type":"node","id":5461200400,"lat":51.5160652,"lon":-0.1305739},{"type":"node","id":5461200401,"lat":51.5160602,"lon":-0.1306075},{"type":"node","id":5461200402,"lat":51.5160554,"lon":-0.1306403},{"type":"node","id":5461200544,"lat":51.5162508,"lon":-0.1302842},{"type":"node","id":5461200545,"lat":51.5162233,"lon":-0.1306714},{"type":"node","id":5461200546,"lat":51.5162836,"lon":-0.1302914},{"type":"node","id":5461200547,"lat":51.5162561,"lon":-0.1306787},{"type":"node","id":5461200548,"lat":51.516315,"lon":-0.1302984},{"type":"node","id":5461200549,"lat":51.5162875,"lon":-0.1306856},{"type":"node","id":5461200553,"lat":51.5161453,"lon":-0.1299591},{"type":"node","id":5461200554,"lat":51.516474,"lon":-0.1301646},{"type":"node","id":5463724741,"lat":51.5202382,"lon":-0.1259587},{"type":"node","id":5463724742,"lat":51.5207719,"lon":-0.1249082},{"type":"node","id":5465557380,"lat":51.5208872,"lon":-0.1255689},{"type":"node","id":5630569824,"lat":51.5175577,"lon":-0.1308223},{"type":"node","id":5630706682,"lat":51.5156336,"lon":-0.1287806},{"type":"node","id":5630761778,"lat":51.5169249,"lon":-0.1259071},{"type":"node","id":5630766895,"lat":51.5161262,"lon":-0.1264268},{"type":"node","id":5630841550,"lat":51.5158772,"lon":-0.1257977},{"type":"node","id":5630841551,"lat":51.5157458,"lon":-0.1262707},{"type":"node","id":5630841552,"lat":51.5158675,"lon":-0.1258355},{"type":"node","id":5630841553,"lat":51.5157039,"lon":-0.1263146},{"type":"node","id":5630841554,"lat":51.5159164,"lon":-0.12605},{"type":"node","id":5630841555,"lat":51.5159375,"lon":-0.1260655},{"type":"node","id":5630940176,"lat":51.5163307,"lon":-0.1270114},{"type":"node","id":5631066029,"lat":51.517605,"lon":-0.126329},{"type":"node","id":5811146367,"lat":51.5178756,"lon":-0.1300059},{"type":"node","id":5811146368,"lat":51.5177821,"lon":-0.1300105},{"type":"node","id":5811146369,"lat":51.51789,"lon":-0.1299627},{"type":"node","id":5811146370,"lat":51.5178083,"lon":-0.1301777},{"type":"node","id":5811146371,"lat":51.5178235,"lon":-0.1301374},{"type":"node","id":5811146372,"lat":51.5176602,"lon":-0.1307149},{"type":"node","id":5811148944,"lat":51.517116,"lon":-0.1306587},{"type":"node","id":5811148945,"lat":51.5170594,"lon":-0.1308591},{"type":"node","id":5811148946,"lat":51.5167874,"lon":-0.1305593},{"type":"node","id":5811148947,"lat":51.5166339,"lon":-0.130454},{"type":"node","id":5811148948,"lat":51.5165819,"lon":-0.1304232},{"type":"node","id":5811153023,"lat":51.5163575,"lon":-0.1300988},{"type":"node","id":5811164165,"lat":51.5162415,"lon":-0.1303417},{"type":"node","id":5811174034,"lat":51.5158559,"lon":-0.1300034},{"type":"node","id":5811174036,"lat":51.5162433,"lon":-0.1302805},{"type":"node","id":5811245210,"lat":51.5157883,"lon":-0.1266024},{"type":"node","id":5811245211,"lat":51.516296,"lon":-0.1270977},{"type":"node","id":5811252635,"lat":51.5155304,"lon":-0.1268757},{"type":"node","id":5811252637,"lat":51.5157313,"lon":-0.1265586},{"type":"node","id":5811252641,"lat":51.5155998,"lon":-0.1266278},{"type":"node","id":5811263955,"lat":51.5154882,"lon":-0.1268352},{"type":"node","id":5811263956,"lat":51.5155472,"lon":-0.1267293},{"type":"node","id":5811263957,"lat":51.5153769,"lon":-0.1268731},{"type":"node","id":5840695132,"lat":51.515177,"lon":-0.1287175},{"type":"node","id":5840695133,"lat":51.5152659,"lon":-0.1287835},{"type":"node","id":5840695134,"lat":51.5152697,"lon":-0.1286905},{"type":"node","id":5840695135,"lat":51.5152707,"lon":-0.1286632},{"type":"node","id":5840695143,"lat":51.5151783,"lon":-0.1286915},{"type":"node","id":5840695145,"lat":51.514724,"lon":-0.1283909},{"type":"node","id":5840695151,"lat":51.5151752,"lon":-0.1287545},{"type":"node","id":5877171660,"lat":51.5159986,"lon":-0.1285888},{"type":"node","id":5877171661,"lat":51.5165778,"lon":-0.1290247},{"type":"node","id":5877171662,"lat":51.5165277,"lon":-0.1288905},{"type":"node","id":5877171663,"lat":51.5166073,"lon":-0.1288203},{"type":"node","id":5877177127,"lat":51.5174054,"lon":-0.1297007},{"type":"node","id":5932928493,"lat":51.517158,"lon":-0.1252614},{"type":"node","id":5935736817,"lat":51.5175146,"lon":-0.1307777},{"type":"node","id":5935769749,"lat":51.5162987,"lon":-0.1301576},{"type":"node","id":6219635599,"lat":51.5171008,"lon":-0.1252201},{"type":"node","id":6219635602,"lat":51.5171228,"lon":-0.1253224},{"type":"node","id":6219635605,"lat":51.5170761,"lon":-0.1256006},{"type":"node","id":6219635609,"lat":51.517196,"lon":-0.1251797},{"type":"node","id":6219646000,"lat":51.5170421,"lon":-0.1258989},{"type":"node","id":6250477238,"lat":51.5170601,"lon":-0.1257277},{"type":"node","id":6313755481,"lat":51.516226,"lon":-0.1303393},{"type":"node","id":6313786638,"lat":51.5163442,"lon":-0.1299529},{"type":"node","id":6313786639,"lat":51.516147,"lon":-0.1298778},{"type":"node","id":6319587582,"lat":51.5150709,"lon":-0.1298468},{"type":"node","id":6319600650,"lat":51.5151529,"lon":-0.1300194},{"type":"node","id":6319600651,"lat":51.5149604,"lon":-0.1299216},{"type":"node","id":6319618545,"lat":51.5155804,"lon":-0.1283794},{"type":"node","id":6319748547,"lat":51.5163624,"lon":-0.1264145},{"type":"node","id":6319748550,"lat":51.5158569,"lon":-0.1260061},{"type":"node","id":6319748554,"lat":51.5168459,"lon":-0.1268377},{"type":"node","id":6319779695,"lat":51.5156157,"lon":-0.1265915},{"type":"node","id":107828,"lat":51.5162721,"lon":-0.1244536},{"type":"node","id":107829,"lat":51.5172868,"lon":-0.1224044},{"type":"node","id":107830,"lat":51.517622,"lon":-0.1203947},{"type":"node","id":107833,"lat":51.5182239,"lon":-0.1227975},{"type":"node","id":107835,"lat":51.5195906,"lon":-0.1229193},{"type":"node","id":107836,"lat":51.519406,"lon":-0.123355},{"type":"node","id":107837,"lat":51.5191826,"lon":-0.1238394},{"type":"node","id":107838,"lat":51.5189418,"lon":-0.1244106},{"type":"node","id":107855,"lat":51.5199122,"lon":-0.1193925},{"type":"node","id":107860,"lat":51.5209401,"lon":-0.122119},{"type":"node","id":107863,"lat":51.5198922,"lon":-0.122169},{"type":"node","id":108400,"lat":51.5182884,"lon":-0.1205951},{"type":"node","id":108403,"lat":51.5162653,"lon":-0.1194575},{"type":"node","id":108404,"lat":51.5190565,"lon":-0.1209862},{"type":"node","id":108901,"lat":51.5177057,"lon":-0.1191781},{"type":"node","id":9475515,"lat":51.5146873,"lon":-0.1222068},{"type":"node","id":9475520,"lat":51.5166192,"lon":-0.1186009},{"type":"node","id":9475528,"lat":51.5158337,"lon":-0.1205489},{"type":"node","id":9476441,"lat":51.5198395,"lon":-0.1195794},{"type":"node","id":9476442,"lat":51.5187578,"lon":-0.1194746},{"type":"node","id":9476443,"lat":51.5192007,"lon":-0.1180577},{"type":"node","id":10616173,"lat":51.519158,"lon":-0.1197585},{"type":"node","id":10616174,"lat":51.5195482,"lon":-0.1183398},{"type":"node","id":11544693,"lat":51.5184051,"lon":-0.1189092},{"type":"node","id":11544696,"lat":51.5189795,"lon":-0.118748},{"type":"node","id":11544697,"lat":51.5185247,"lon":-0.1184183},{"type":"node","id":11767206,"lat":51.5207708,"lon":-0.1232431},{"type":"node","id":11767207,"lat":51.5211382,"lon":-0.1215687},{"type":"node","id":11767209,"lat":51.5209755,"lon":-0.1228281},{"type":"node","id":13355224,"lat":51.5153605,"lon":-0.1231981},{"type":"node","id":13491058,"lat":51.5193924,"lon":-0.119029},{"type":"node","id":14727254,"lat":51.5184879,"lon":-0.1243933},{"type":"node","id":21563009,"lat":51.5168435,"lon":-0.1187904},{"type":"node","id":21590630,"lat":51.5166609,"lon":-0.1197187},{"type":"node","id":21590631,"lat":51.5171213,"lon":-0.1200421},{"type":"node","id":21590632,"lat":51.5172086,"lon":-0.1191449},{"type":"node","id":21704017,"lat":51.5148612,"lon":-0.1217461},{"type":"node","id":21990639,"lat":51.5151696,"lon":-0.1194266},{"type":"node","id":25470388,"lat":51.5158209,"lon":-0.1238188},{"type":"node","id":25470464,"lat":51.5146406,"lon":-0.1206541},{"type":"node","id":25470465,"lat":51.5157155,"lon":-0.1190897},{"type":"node","id":25470678,"lat":51.5161815,"lon":-0.1209735},{"type":"node","id":25470679,"lat":51.5173904,"lon":-0.1218496},{"type":"node","id":25470680,"lat":51.5149787,"lon":-0.1226154},{"type":"node","id":25470681,"lat":51.516531,"lon":-0.1213895},{"type":"node","id":25470694,"lat":51.5154564,"lon":-0.1233384},{"type":"node","id":25470697,"lat":51.5162435,"lon":-0.12192},{"type":"node","id":25470700,"lat":51.5162091,"lon":-0.1231405},{"type":"node","id":25470776,"lat":51.5151287,"lon":-0.122849},{"type":"node","id":25470835,"lat":51.5183139,"lon":-0.1242077},{"type":"node","id":25472757,"lat":51.5184393,"lon":-0.1222951},{"type":"node","id":25472758,"lat":51.5174847,"lon":-0.1212578},{"type":"node","id":25472759,"lat":51.517454,"lon":-0.1233657},{"type":"node","id":25472762,"lat":51.5176856,"lon":-0.1227708},{"type":"node","id":25472764,"lat":51.5180605,"lon":-0.1231692},{"type":"node","id":25472766,"lat":51.5180503,"lon":-0.121872},{"type":"node","id":25472769,"lat":51.5172898,"lon":-0.1232071},{"type":"node","id":25472865,"lat":51.5205746,"lon":-0.12106},{"type":"node","id":25472874,"lat":51.5206514,"lon":-0.1211352},{"type":"node","id":25472875,"lat":51.5204849,"lon":-0.1217041},{"type":"node","id":25472903,"lat":51.5193845,"lon":-0.1181918},{"type":"node","id":25472906,"lat":51.5183182,"lon":-0.1192465},{"type":"node","id":25472992,"lat":51.5209508,"lon":-0.1214018},{"type":"node","id":25503720,"lat":51.5184871,"lon":-0.1200387},{"type":"node","id":25503723,"lat":51.5182251,"lon":-0.1195984},{"type":"node","id":25503725,"lat":51.5180558,"lon":-0.1205297},{"type":"node","id":264515752,"lat":51.5178702,"lon":-0.1190605},{"type":"node","id":281454961,"lat":51.5147236,"lon":-0.1186063},{"type":"node","id":281454988,"lat":51.517691,"lon":-0.1189673},{"type":"node","id":281454989,"lat":51.5177487,"lon":-0.1191347},{"type":"node","id":293864446,"lat":51.5195995,"lon":-0.1208533},{"type":"node","id":296441984,"lat":51.5188644,"lon":-0.1235011},{"type":"node","id":317930302,"lat":51.517407,"lon":-0.1180878},{"type":"node","id":317930352,"lat":51.5187548,"lon":-0.1181995},{"type":"node","id":317930353,"lat":51.518662,"lon":-0.1185163},{"type":"node","id":338505530,"lat":51.5177815,"lon":-0.1182357},{"type":"node","id":354671172,"lat":51.5158371,"lon":-0.1180388},{"type":"node","id":366518767,"lat":51.5205716,"lon":-0.1214048},{"type":"node","id":371400554,"lat":51.5158866,"lon":-0.1225676},{"type":"node","id":396093922,"lat":51.5172774,"lon":-0.1227002},{"type":"node","id":396093929,"lat":51.517389,"lon":-0.1224447},{"type":"node","id":438170520,"lat":51.5157016,"lon":-0.1239295},{"type":"node","id":438170530,"lat":51.5156092,"lon":-0.1235559},{"type":"node","id":438170536,"lat":51.5156051,"lon":-0.1238175},{"type":"node","id":438170563,"lat":51.5152372,"lon":-0.1241795},{"type":"node","id":438170568,"lat":51.5154384,"lon":-0.1244462},{"type":"node","id":438170585,"lat":51.515218,"lon":-0.1242274},{"type":"node","id":438170591,"lat":51.5154129,"lon":-0.1244808},{"type":"node","id":455705618,"lat":51.5152884,"lon":-0.1189761},{"type":"node","id":455705620,"lat":51.5155377,"lon":-0.1191402},{"type":"node","id":455705621,"lat":51.5160816,"lon":-0.1195124},{"type":"node","id":455705622,"lat":51.5163453,"lon":-0.1196821},{"type":"node","id":455705623,"lat":51.5166292,"lon":-0.1198697},{"type":"node","id":455705624,"lat":51.5171268,"lon":-0.1201982},{"type":"node","id":455705625,"lat":51.5176431,"lon":-0.1204035},{"type":"node","id":455705626,"lat":51.518043,"lon":-0.1206375},{"type":"node","id":455705627,"lat":51.5183257,"lon":-0.1208331},{"type":"node","id":455705628,"lat":51.5188608,"lon":-0.1210839},{"type":"node","id":456363900,"lat":51.5172746,"lon":-0.122873},{"type":"node","id":456363901,"lat":51.5172414,"lon":-0.1228362},{"type":"node","id":456363902,"lat":51.5172161,"lon":-0.1228279},{"type":"node","id":456363903,"lat":51.5170914,"lon":-0.1228778},{"type":"node","id":490256621,"lat":51.515964,"lon":-0.1213852},{"type":"node","id":534993848,"lat":51.515497,"lon":-0.1224433},{"type":"node","id":534993850,"lat":51.515528,"lon":-0.1224018},{"type":"node","id":534993859,"lat":51.5155769,"lon":-0.1225998},{"type":"node","id":534993866,"lat":51.5153395,"lon":-0.1221695},{"type":"node","id":534993869,"lat":51.51641,"lon":-0.1216119},{"type":"node","id":534993963,"lat":51.5159493,"lon":-0.1206814},{"type":"node","id":727447379,"lat":51.5150012,"lon":-0.1217573},{"type":"node","id":734773333,"lat":51.5201034,"lon":-0.1203025},{"type":"node","id":734773335,"lat":51.5201829,"lon":-0.1203679},{"type":"node","id":734773337,"lat":51.5201137,"lon":-0.1205593},{"type":"node","id":742854790,"lat":51.5190236,"lon":-0.1224608},{"type":"node","id":742854809,"lat":51.5187256,"lon":-0.123222},{"type":"node","id":742854814,"lat":51.5187874,"lon":-0.1233364},{"type":"node","id":742854830,"lat":51.5186188,"lon":-0.1232351},{"type":"node","id":742854844,"lat":51.5187802,"lon":-0.1233122},{"type":"node","id":742854854,"lat":51.5188063,"lon":-0.1233983},{"type":"node","id":742854866,"lat":51.5189755,"lon":-0.122396},{"type":"node","id":742854871,"lat":51.5189572,"lon":-0.1236016},{"type":"node","id":742854873,"lat":51.5189423,"lon":-0.1223733},{"type":"node","id":742854879,"lat":51.5185459,"lon":-0.1231498},{"type":"node","id":742854884,"lat":51.5193196,"lon":-0.1226091},{"type":"node","id":742854887,"lat":51.5189928,"lon":-0.1223911},{"type":"node","id":742854889,"lat":51.5186556,"lon":-0.1231419},{"type":"node","id":742854891,"lat":51.5189357,"lon":-0.1235456},{"type":"node","id":742854911,"lat":51.5190361,"lon":-0.1224009},{"type":"node","id":742854915,"lat":51.5189541,"lon":-0.1224213},{"type":"node","id":742854924,"lat":51.5192431,"lon":-0.1226264},{"type":"node","id":742854934,"lat":51.5189406,"lon":-0.1221637},{"type":"node","id":742854954,"lat":51.5190133,"lon":-0.122246},{"type":"node","id":742854960,"lat":51.5187931,"lon":-0.123261},{"type":"node","id":764867047,"lat":51.5192631,"lon":-0.1236649},{"type":"node","id":764867054,"lat":51.5194902,"lon":-0.1231689},{"type":"node","id":869389491,"lat":51.5193447,"lon":-0.1183174},{"type":"node","id":869389523,"lat":51.5193633,"lon":-0.1182593},{"type":"node","id":877060602,"lat":51.5179619,"lon":-0.1190936},{"type":"node","id":895113296,"lat":51.5194176,"lon":-0.1189051},{"type":"node","id":968291529,"lat":51.5171206,"lon":-0.1190198},{"type":"node","id":973525755,"lat":51.5185866,"lon":-0.1190187},{"type":"node","id":1104473061,"lat":51.5189475,"lon":-0.1211642},{"type":"node","id":1104473062,"lat":51.5192864,"lon":-0.1214838},{"type":"node","id":1104496288,"lat":51.5193599,"lon":-0.1199738},{"type":"node","id":1104496292,"lat":51.5195029,"lon":-0.1202452},{"type":"node","id":1104496295,"lat":51.5188521,"lon":-0.1213807},{"type":"node","id":1104496296,"lat":51.5193843,"lon":-0.1200655},{"type":"node","id":1104496309,"lat":51.5194633,"lon":-0.1199017},{"type":"node","id":1104496313,"lat":51.5196728,"lon":-0.1197915},{"type":"node","id":1104496316,"lat":51.5196404,"lon":-0.1199826},{"type":"node","id":1104496319,"lat":51.5194834,"lon":-0.1201032},{"type":"node","id":1104496326,"lat":51.5193307,"lon":-0.1198792},{"type":"node","id":1104496341,"lat":51.5185106,"lon":-0.1193157},{"type":"node","id":1104496342,"lat":51.5177728,"lon":-0.1191159},{"type":"node","id":1104496344,"lat":51.5192973,"lon":-0.1205631},{"type":"node","id":1104496346,"lat":51.5193989,"lon":-0.1202374},{"type":"node","id":1104496349,"lat":51.5177989,"lon":-0.1188985},{"type":"node","id":1104508800,"lat":51.5188472,"lon":-0.1209534},{"type":"node","id":1104509637,"lat":51.5196661,"lon":-0.1201403},{"type":"node","id":1115431307,"lat":51.5176659,"lon":-0.1240862},{"type":"node","id":1115431319,"lat":51.5175966,"lon":-0.1240076},{"type":"node","id":1126375339,"lat":51.5168824,"lon":-0.1215847},{"type":"node","id":1126375365,"lat":51.516797,"lon":-0.1220079},{"type":"node","id":1126375369,"lat":51.5168281,"lon":-0.1220202},{"type":"node","id":1126375486,"lat":51.518332,"lon":-0.1207193},{"type":"node","id":1161243563,"lat":51.5189056,"lon":-0.1208595},{"type":"node","id":1161243567,"lat":51.5188694,"lon":-0.1209624},{"type":"node","id":1161243569,"lat":51.5188395,"lon":-0.1210673},{"type":"node","id":1237514473,"lat":51.5148861,"lon":-0.1187049},{"type":"node","id":1237514492,"lat":51.5147945,"lon":-0.1186499},{"type":"node","id":1392042899,"lat":51.519706,"lon":-0.1209649},{"type":"node","id":1392042915,"lat":51.5199296,"lon":-0.1203959},{"type":"node","id":1413151217,"lat":51.5160032,"lon":-0.1235149},{"type":"node","id":1588852215,"lat":51.516594,"lon":-0.1224611},{"type":"node","id":1611771520,"lat":51.5176824,"lon":-0.119601},{"type":"node","id":1611771527,"lat":51.5171932,"lon":-0.1194631},{"type":"node","id":1614926348,"lat":51.514759,"lon":-0.1234717},{"type":"node","id":1614926358,"lat":51.5148391,"lon":-0.1224198},{"type":"node","id":1614949561,"lat":51.5161524,"lon":-0.1226831},{"type":"node","id":1614949580,"lat":51.5162949,"lon":-0.1228945},{"type":"node","id":1614949585,"lat":51.516324,"lon":-0.1229227},{"type":"node","id":1614949587,"lat":51.5164109,"lon":-0.1230463},{"type":"node","id":1614949613,"lat":51.5167448,"lon":-0.1235294},{"type":"node","id":1614978560,"lat":51.5155845,"lon":-0.1235249},{"type":"node","id":1614978621,"lat":51.5163844,"lon":-0.1195321},{"type":"node","id":1614978658,"lat":51.5165326,"lon":-0.1201341},{"type":"node","id":1614978668,"lat":51.5165645,"lon":-0.1189019},{"type":"node","id":1614978671,"lat":51.5166055,"lon":-0.1187439},{"type":"node","id":1806593433,"lat":51.5167276,"lon":-0.1235685},{"type":"node","id":1806593437,"lat":51.5172125,"lon":-0.123938},{"type":"node","id":1937391982,"lat":51.5173025,"lon":-0.1230464},{"type":"node","id":1938953384,"lat":51.5166699,"lon":-0.1186642},{"type":"node","id":1938953392,"lat":51.5172989,"lon":-0.1229633},{"type":"node","id":1938953393,"lat":51.5174046,"lon":-0.1203754},{"type":"node","id":1938953394,"lat":51.5176826,"lon":-0.1190722},{"type":"node","id":1938953397,"lat":51.517777,"lon":-0.1186992},{"type":"node","id":1938953398,"lat":51.5177801,"lon":-0.1188174},{"type":"node","id":1938953401,"lat":51.5183027,"lon":-0.1193409},{"type":"node","id":1938953428,"lat":51.5187249,"lon":-0.1195714},{"type":"node","id":1938953440,"lat":51.5194992,"lon":-0.1199087},{"type":"node","id":1947276623,"lat":51.5149299,"lon":-0.1198011},{"type":"node","id":1947276625,"lat":51.5152132,"lon":-0.1214992},{"type":"node","id":1947276628,"lat":51.5154261,"lon":-0.1211852},{"type":"node","id":1947276630,"lat":51.5156376,"lon":-0.1208631},{"type":"node","id":1947276635,"lat":51.5161046,"lon":-0.1201112},{"type":"node","id":1953676002,"lat":51.5191061,"lon":-0.1213315},{"type":"node","id":1954794473,"lat":51.518407,"lon":-0.1222582},{"type":"node","id":1954794476,"lat":51.5187981,"lon":-0.1213151},{"type":"node","id":1954794477,"lat":51.5188843,"lon":-0.1211052},{"type":"node","id":1954794478,"lat":51.518981,"lon":-0.1209101},{"type":"node","id":2094965921,"lat":51.5180895,"lon":-0.119155},{"type":"node","id":2096436908,"lat":51.5193652,"lon":-0.120371},{"type":"node","id":2096436909,"lat":51.5193003,"lon":-0.120403},{"type":"node","id":2096436910,"lat":51.5193362,"lon":-0.1204467},{"type":"node","id":2096436911,"lat":51.5191541,"lon":-0.120808},{"type":"node","id":2096436912,"lat":51.518672,"lon":-0.1216108},{"type":"node","id":2096443242,"lat":51.5198251,"lon":-0.1195582},{"type":"node","id":2096443243,"lat":51.5197529,"lon":-0.119766},{"type":"node","id":2096443244,"lat":51.5196074,"lon":-0.1199004},{"type":"node","id":2096443245,"lat":51.5197588,"lon":-0.1196679},{"type":"node","id":2096450587,"lat":51.5189012,"lon":-0.1212633},{"type":"node","id":2096450589,"lat":51.5195745,"lon":-0.1198959},{"type":"node","id":2096450591,"lat":51.519512,"lon":-0.1200567},{"type":"node","id":2096450593,"lat":51.5193975,"lon":-0.1201415},{"type":"node","id":2096450595,"lat":51.5196116,"lon":-0.1198633},{"type":"node","id":2096450597,"lat":51.5190154,"lon":-0.1209453},{"type":"node","id":2096450599,"lat":51.5193999,"lon":-0.1198888},{"type":"node","id":2096450603,"lat":51.5188457,"lon":-0.1212023},{"type":"node","id":2117429474,"lat":51.5189003,"lon":-0.1209782},{"type":"node","id":2117429475,"lat":51.5189374,"lon":-0.1210017},{"type":"node","id":2117444161,"lat":51.5190847,"lon":-0.1213116},{"type":"node","id":2117444163,"lat":51.5191824,"lon":-0.1207586},{"type":"node","id":2117444166,"lat":51.5190655,"lon":-0.1207621},{"type":"node","id":2117444169,"lat":51.5190412,"lon":-0.1212643},{"type":"node","id":2117444170,"lat":51.519118,"lon":-0.1211111},{"type":"node","id":2117444174,"lat":51.5191403,"lon":-0.1211621},{"type":"node","id":2117447071,"lat":51.5190841,"lon":-0.1207269},{"type":"node","id":2117467718,"lat":51.5187988,"lon":-0.1210486},{"type":"node","id":2117469204,"lat":51.518118,"lon":-0.1205532},{"type":"node","id":2117469205,"lat":51.5181054,"lon":-0.1206737},{"type":"node","id":2117470720,"lat":51.5175069,"lon":-0.1202784},{"type":"node","id":2117470721,"lat":51.5177727,"lon":-0.1203999},{"type":"node","id":2117470722,"lat":51.5174894,"lon":-0.1204175},{"type":"node","id":2117470723,"lat":51.5175907,"lon":-0.1205799},{"type":"node","id":2117470724,"lat":51.5176656,"lon":-0.1199908},{"type":"node","id":2117470725,"lat":51.5177274,"lon":-0.1205047},{"type":"node","id":2117470726,"lat":51.5176513,"lon":-0.120156},{"type":"node","id":2117470727,"lat":51.5177443,"lon":-0.1203868},{"type":"node","id":2117475740,"lat":51.5164076,"lon":-0.1197235},{"type":"node","id":2117475741,"lat":51.5187472,"lon":-0.1219271},{"type":"node","id":2117475743,"lat":51.5164474,"lon":-0.119578},{"type":"node","id":2249185409,"lat":51.5177246,"lon":-0.1186995},{"type":"node","id":2392997817,"lat":51.5186263,"lon":-0.1193842},{"type":"node","id":2402713067,"lat":51.517178,"lon":-0.1227188},{"type":"node","id":2402713068,"lat":51.5172745,"lon":-0.1227614},{"type":"node","id":2402713069,"lat":51.517227,"lon":-0.1225913},{"type":"node","id":2402713070,"lat":51.5171613,"lon":-0.1228247},{"type":"node","id":2402713071,"lat":51.5171274,"lon":-0.1228405},{"type":"node","id":2441993335,"lat":51.5176607,"lon":-0.1200612},{"type":"node","id":2441993340,"lat":51.5149309,"lon":-0.121763},{"type":"node","id":2441993343,"lat":51.5149573,"lon":-0.121767},{"type":"node","id":2441993346,"lat":51.5148825,"lon":-0.1217449},{"type":"node","id":2441993349,"lat":51.5146023,"lon":-0.1219933},{"type":"node","id":2441993353,"lat":51.5149796,"lon":-0.1217652},{"type":"node","id":2441993356,"lat":51.5148381,"lon":-0.1217566},{"type":"node","id":2441993358,"lat":51.5149045,"lon":-0.1217512},{"type":"node","id":2441993361,"lat":51.5163037,"lon":-0.1197614},{"type":"node","id":2441993365,"lat":51.517449,"lon":-0.1204008},{"type":"node","id":2441998874,"lat":51.5146157,"lon":-0.124492},{"type":"node","id":2502928026,"lat":51.5154052,"lon":-0.1232624},{"type":"node","id":2502928027,"lat":51.515941,"lon":-0.123993},{"type":"node","id":2512940590,"lat":51.5157927,"lon":-0.1237809},{"type":"node","id":2683449617,"lat":51.5160534,"lon":-0.1208139},{"type":"node","id":3085005039,"lat":51.5171875,"lon":-0.1228225},{"type":"node","id":3085005040,"lat":51.5172581,"lon":-0.1228491},{"type":"node","id":3085005041,"lat":51.517279,"lon":-0.122551},{"type":"node","id":3085005042,"lat":51.517282,"lon":-0.1224778},{"type":"node","id":3085005043,"lat":51.517303,"lon":-0.1230032},{"type":"node","id":3085005044,"lat":51.5172917,"lon":-0.1218072},{"type":"node","id":3085005046,"lat":51.517312,"lon":-0.1232206},{"type":"node","id":3085005049,"lat":51.5173128,"lon":-0.121819},{"type":"node","id":3085005050,"lat":51.5173341,"lon":-0.1232384},{"type":"node","id":3085005052,"lat":51.5173476,"lon":-0.1228923},{"type":"node","id":3085005053,"lat":51.5173589,"lon":-0.1227724},{"type":"node","id":3085005054,"lat":51.5173756,"lon":-0.1219475},{"type":"node","id":3085005055,"lat":51.5174032,"lon":-0.1217852},{"type":"node","id":3085005056,"lat":51.5174191,"lon":-0.1216958},{"type":"node","id":3085005057,"lat":51.5174285,"lon":-0.1218616},{"type":"node","id":3085005058,"lat":51.5174371,"lon":-0.1218709},{"type":"node","id":3085005059,"lat":51.5174423,"lon":-0.1218854},{"type":"node","id":3085005060,"lat":51.5174428,"lon":-0.1219031},{"type":"node","id":3085005061,"lat":51.5174697,"lon":-0.1233268},{"type":"node","id":3085005063,"lat":51.5178578,"lon":-0.1237507},{"type":"node","id":3085005064,"lat":51.5178492,"lon":-0.1236642},{"type":"node","id":3085005065,"lat":51.5178782,"lon":-0.1237662},{"type":"node","id":3085005066,"lat":51.5179069,"lon":-0.123727},{"type":"node","id":3085005067,"lat":51.517944,"lon":-0.1237673},{"type":"node","id":3085005068,"lat":51.5179563,"lon":-0.1238045},{"type":"node","id":3085005069,"lat":51.5182977,"lon":-0.1242616},{"type":"node","id":3140855388,"lat":51.5157345,"lon":-0.1179666},{"type":"node","id":3149232618,"lat":51.5154878,"lon":-0.118945},{"type":"node","id":3519217057,"lat":51.5175842,"lon":-0.1242798},{"type":"node","id":3519233743,"lat":51.5182308,"lon":-0.1192076},{"type":"node","id":3644748772,"lat":51.516851,"lon":-0.1198595},{"type":"node","id":3895412046,"lat":51.5150932,"lon":-0.1181053},{"type":"node","id":3895422567,"lat":51.5149794,"lon":-0.1185526},{"type":"node","id":3895422571,"lat":51.5150897,"lon":-0.122903},{"type":"node","id":3895422574,"lat":51.5148876,"lon":-0.1231973},{"type":"node","id":4008046109,"lat":51.5171024,"lon":-0.1190801},{"type":"node","id":4008046110,"lat":51.5166847,"lon":-0.1196105},{"type":"node","id":4034416983,"lat":51.5168782,"lon":-0.1189394},{"type":"node","id":4216274089,"lat":51.5186801,"lon":-0.1184545},{"type":"node","id":4216274114,"lat":51.5187252,"lon":-0.1183007},{"type":"node","id":4216274127,"lat":51.5187441,"lon":-0.1185751},{"type":"node","id":4216274136,"lat":51.5187614,"lon":-0.1185101},{"type":"node","id":4216274149,"lat":51.5188069,"lon":-0.1193229},{"type":"node","id":4218360149,"lat":51.519544,"lon":-0.1184304},{"type":"node","id":4218360151,"lat":51.5195511,"lon":-0.1183863},{"type":"node","id":4218360345,"lat":51.5193623,"lon":-0.1206267},{"type":"node","id":4218360357,"lat":51.5193856,"lon":-0.1204457},{"type":"node","id":4218360364,"lat":51.5193916,"lon":-0.1202902},{"type":"node","id":4218360423,"lat":51.5192039,"lon":-0.1196145},{"type":"node","id":4218360491,"lat":51.5194362,"lon":-0.1201844},{"type":"node","id":4218360529,"lat":51.5195051,"lon":-0.118292},{"type":"node","id":4218365467,"lat":51.5192604,"lon":-0.1194014},{"type":"node","id":4218365468,"lat":51.519317,"lon":-0.1194462},{"type":"node","id":4305311206,"lat":51.5168361,"lon":-0.1200139},{"type":"node","id":4356554727,"lat":51.5147263,"lon":-0.1221424},{"type":"node","id":4356572311,"lat":51.5163455,"lon":-0.1243109},{"type":"node","id":4356572312,"lat":51.5162258,"lon":-0.1243952},{"type":"node","id":4356572315,"lat":51.5163312,"lon":-0.1245043},{"type":"node","id":4356572319,"lat":51.5163208,"lon":-0.1243585},{"type":"node","id":4356572320,"lat":51.5161747,"lon":-0.1243258},{"type":"node","id":4356591931,"lat":51.5173183,"lon":-0.1217863},{"type":"node","id":4356591932,"lat":51.517254,"lon":-0.1217881},{"type":"node","id":4356620356,"lat":51.5184144,"lon":-0.1223535},{"type":"node","id":4356620357,"lat":51.5185227,"lon":-0.1221038},{"type":"node","id":4356620358,"lat":51.518163,"lon":-0.1229323},{"type":"node","id":4356620359,"lat":51.5183927,"lon":-0.1224044},{"type":"node","id":4356620360,"lat":51.5177363,"lon":-0.1236643},{"type":"node","id":4356620361,"lat":51.5181901,"lon":-0.1228855},{"type":"node","id":4356620362,"lat":51.5183443,"lon":-0.1221903},{"type":"node","id":4356620363,"lat":51.518372,"lon":-0.1222203},{"type":"node","id":4356620365,"lat":51.5178846,"lon":-0.1237778},{"type":"node","id":4356620366,"lat":51.5177371,"lon":-0.123926},{"type":"node","id":4356620367,"lat":51.5182115,"lon":-0.1228329},{"type":"node","id":4360406511,"lat":51.5179082,"lon":-0.1238044},{"type":"node","id":4360451847,"lat":51.5189875,"lon":-0.124307},{"type":"node","id":4360451848,"lat":51.5194595,"lon":-0.1234158},{"type":"node","id":4360469338,"lat":51.5207953,"lon":-0.1232733},{"type":"node","id":4360469341,"lat":51.5207146,"lon":-0.1231855},{"type":"node","id":4411113474,"lat":51.5169256,"lon":-0.1200664},{"type":"node","id":4411120201,"lat":51.5164687,"lon":-0.1240715},{"type":"node","id":4587152382,"lat":51.5174495,"lon":-0.1214866},{"type":"node","id":4587152383,"lat":51.51725,"lon":-0.1235785},{"type":"node","id":4681962044,"lat":51.519594,"lon":-0.1218382},{"type":"node","id":4681962045,"lat":51.5195173,"lon":-0.1217531},{"type":"node","id":4681962046,"lat":51.5210196,"lon":-0.1235498},{"type":"node","id":4682210576,"lat":51.5191206,"lon":-0.123983},{"type":"node","id":4720710035,"lat":51.515677,"lon":-0.119232},{"type":"node","id":4962063593,"lat":51.5164505,"lon":-0.1212981},{"type":"node","id":4962138929,"lat":51.5172676,"lon":-0.1217643},{"type":"node","id":4962172562,"lat":51.5177674,"lon":-0.1238551},{"type":"node","id":4962172563,"lat":51.5177799,"lon":-0.1237098},{"type":"node","id":4962172564,"lat":51.5178893,"lon":-0.1237069},{"type":"node","id":4967033602,"lat":51.515062,"lon":-0.1217003},{"type":"node","id":4967033603,"lat":51.5146218,"lon":-0.1221244},{"type":"node","id":5339077112,"lat":51.5178143,"lon":-0.1237457},{"type":"node","id":5339077114,"lat":51.5184682,"lon":-0.1244409},{"type":"node","id":5339077115,"lat":51.5183008,"lon":-0.1242514},{"type":"node","id":5339077116,"lat":51.5179324,"lon":-0.1238098},{"type":"node","id":5339078763,"lat":51.5177105,"lon":-0.1239864},{"type":"node","id":5371270743,"lat":51.5190225,"lon":-0.1245079},{"type":"node","id":5371270746,"lat":51.5189788,"lon":-0.1244557},{"type":"node","id":5404582018,"lat":51.5153439,"lon":-0.1232267},{"type":"node","id":5404582019,"lat":51.5151031,"lon":-0.123568},{"type":"node","id":5404582020,"lat":51.5151078,"lon":-0.1236801},{"type":"node","id":5404582918,"lat":51.514857,"lon":-0.1240924},{"type":"node","id":5404582919,"lat":51.5147894,"lon":-0.1241635},{"type":"node","id":5404583121,"lat":51.5150756,"lon":-0.1237867},{"type":"node","id":5404583122,"lat":51.5150602,"lon":-0.1237648},{"type":"node","id":5404583123,"lat":51.5150458,"lon":-0.1237429},{"type":"node","id":5404583124,"lat":51.5148376,"lon":-0.1241726},{"type":"node","id":5404583131,"lat":51.5151264,"lon":-0.1236018},{"type":"node","id":5404583132,"lat":51.5150807,"lon":-0.1235353},{"type":"node","id":5404583172,"lat":51.5147611,"lon":-0.1233877},{"type":"node","id":5404583187,"lat":51.5150939,"lon":-0.1236603},{"type":"node","id":5404583188,"lat":51.5151233,"lon":-0.123702},{"type":"node","id":5404583195,"lat":51.5149576,"lon":-0.1238946},{"type":"node","id":5404583196,"lat":51.5149372,"lon":-0.1239382},{"type":"node","id":5404583197,"lat":51.5148523,"lon":-0.124194},{"type":"node","id":5404583198,"lat":51.5148137,"lon":-0.124138},{"type":"node","id":5408675096,"lat":51.5152856,"lon":-0.1243147},{"type":"node","id":5408675753,"lat":51.5155948,"lon":-0.123672},{"type":"node","id":5413338302,"lat":51.5149534,"lon":-0.1243372},{"type":"node","id":5413338303,"lat":51.514992,"lon":-0.1243919},{"type":"node","id":5413338304,"lat":51.5150473,"lon":-0.1244702},{"type":"node","id":5413338305,"lat":51.5150598,"lon":-0.1244882},{"type":"node","id":5413338306,"lat":51.5150745,"lon":-0.1245104},{"type":"node","id":5445808813,"lat":51.5174964,"lon":-0.1181208},{"type":"node","id":5445808814,"lat":51.5175871,"lon":-0.118156},{"type":"node","id":5458472045,"lat":51.5185469,"lon":-0.1220483},{"type":"node","id":5458472046,"lat":51.5186575,"lon":-0.1217949},{"type":"node","id":5458472047,"lat":51.5189412,"lon":-0.1196093},{"type":"node","id":5466793561,"lat":51.5177473,"lon":-0.1195132},{"type":"node","id":5466793562,"lat":51.5175567,"lon":-0.1194623},{"type":"node","id":5466793563,"lat":51.5175925,"lon":-0.119139},{"type":"node","id":5466793564,"lat":51.5177829,"lon":-0.1191887},{"type":"node","id":5466793575,"lat":51.5168861,"lon":-0.1200471},{"type":"node","id":5466793581,"lat":51.5167691,"lon":-0.1202781},{"type":"node","id":5466793585,"lat":51.5168254,"lon":-0.1203089},{"type":"node","id":5466793786,"lat":51.517021,"lon":-0.1203947},{"type":"node","id":5466793789,"lat":51.5172616,"lon":-0.1205246},{"type":"node","id":5466793796,"lat":51.5169644,"lon":-0.1203656},{"type":"node","id":5466793797,"lat":51.5172049,"lon":-0.120495},{"type":"node","id":5466793798,"lat":51.5170493,"lon":-0.1201043},{"type":"node","id":5466793799,"lat":51.5170966,"lon":-0.1201347},{"type":"node","id":5466793800,"lat":51.517227,"lon":-0.1201737},{"type":"node","id":5466793801,"lat":51.5172443,"lon":-0.1200958},{"type":"node","id":5466793802,"lat":51.517426,"lon":-0.1203481},{"type":"node","id":5466793803,"lat":51.5173759,"lon":-0.1203156},{"type":"node","id":5466793943,"lat":51.517774,"lon":-0.1192703},{"type":"node","id":5466793944,"lat":51.5177562,"lon":-0.1194321},{"type":"node","id":5466793963,"lat":51.5172919,"lon":-0.1198566},{"type":"node","id":5466793964,"lat":51.5173091,"lon":-0.1198639},{"type":"node","id":5466793983,"lat":51.5173266,"lon":-0.1198714},{"type":"node","id":5466793984,"lat":51.5173671,"lon":-0.1193991},{"type":"node","id":5466793985,"lat":51.5173843,"lon":-0.1194064},{"type":"node","id":5466793986,"lat":51.5174018,"lon":-0.1194138},{"type":"node","id":5466793988,"lat":51.5175828,"lon":-0.119224},{"type":"node","id":5466793992,"lat":51.5175654,"lon":-0.1193816},{"type":"node","id":5466793995,"lat":51.517459,"lon":-0.1193563},{"type":"node","id":5466793996,"lat":51.5174785,"lon":-0.1193646},{"type":"node","id":5466793997,"lat":51.5174977,"lon":-0.1193727},{"type":"node","id":5466794222,"lat":51.5175172,"lon":-0.119381},{"type":"node","id":5466794224,"lat":51.5173893,"lon":-0.1197805},{"type":"node","id":5466794225,"lat":51.5174088,"lon":-0.1197888},{"type":"node","id":5466794226,"lat":51.517428,"lon":-0.1197969},{"type":"node","id":5466794227,"lat":51.5174475,"lon":-0.1198052},{"type":"node","id":5466794228,"lat":51.5175501,"lon":-0.1213288},{"type":"node","id":5466794231,"lat":51.5175972,"lon":-0.1205404},{"type":"node","id":5466794238,"lat":51.5171498,"lon":-0.1199081},{"type":"node","id":5517569296,"lat":51.5196246,"lon":-0.1191957},{"type":"node","id":5986562231,"lat":51.5183341,"lon":-0.122934},{"type":"node","id":5986763688,"lat":51.5176013,"lon":-0.1203847},{"type":"node","id":6028641764,"lat":51.5187143,"lon":-0.12188},{"type":"node","id":6032800919,"lat":51.5194056,"lon":-0.1182072},{"type":"node","id":6058284063,"lat":51.5183022,"lon":-0.122892},{"type":"node","id":6060011973,"lat":51.5196534,"lon":-0.1207097},{"type":"node","id":6060011974,"lat":51.5197759,"lon":-0.1204213},{"type":"node","id":6060011975,"lat":51.519683,"lon":-0.1206625},{"type":"node","id":6060034392,"lat":51.5195429,"lon":-0.1199086},{"type":"node","id":6060045960,"lat":51.5178215,"lon":-0.1189654},{"type":"node","id":6060045961,"lat":51.5178237,"lon":-0.1190856},{"type":"node","id":6060045967,"lat":51.5177025,"lon":-0.1188972},{"type":"node","id":6060061452,"lat":51.517788,"lon":-0.1188755},{"type":"node","id":6060061453,"lat":51.517755,"lon":-0.1185701},{"type":"node","id":6060084833,"lat":51.517752,"lon":-0.118635},{"type":"node","id":6287138677,"lat":51.5169007,"lon":-0.118838},{"type":"node","id":6287158541,"lat":51.516434,"lon":-0.120383},{"type":"node","id":108002,"lat":51.5219019,"lon":-0.127833},{"type":"node","id":108017,"lat":51.5214017,"lon":-0.1301587},{"type":"node","id":25471043,"lat":51.5215345,"lon":-0.1275294},{"type":"node","id":1448876659,"lat":51.521386,"lon":-0.1301425},{"type":"node","id":1879123343,"lat":51.5217235,"lon":-0.1277614},{"type":"node","id":1879123345,"lat":51.5218252,"lon":-0.1278316},{"type":"node","id":4955757603,"lat":51.5218673,"lon":-0.1278334},{"type":"node","id":5373783321,"lat":51.5217736,"lon":-0.1278073},{"type":"node","id":107864,"lat":51.5214846,"lon":-0.1241233},{"type":"node","id":11767208,"lat":51.5211582,"lon":-0.1223146},{"type":"node","id":4360460586,"lat":51.521368,"lon":-0.1239795},{"type":"node","id":4360460587,"lat":51.5214235,"lon":-0.1240479},{"type":"node","id":107856,"lat":51.5203943,"lon":-0.1176259},{"type":"node","id":108902,"lat":51.5179846,"lon":-0.1164696},{"type":"node","id":9475519,"lat":51.5172232,"lon":-0.1154272},{"type":"node","id":10572786,"lat":51.5152552,"lon":-0.1115215},{"type":"node","id":11544695,"lat":51.5189013,"lon":-0.1168804},{"type":"node","id":14728072,"lat":51.5202355,"lon":-0.1175414},{"type":"node","id":20599921,"lat":51.5149585,"lon":-0.1158941},{"type":"node","id":21563015,"lat":51.5175425,"lon":-0.115624},{"type":"node","id":21563698,"lat":51.5146167,"lon":-0.1147208},{"type":"node","id":21564825,"lat":51.5148091,"lon":-0.1138536},{"type":"node","id":21583361,"lat":51.5151411,"lon":-0.1122514},{"type":"node","id":25472904,"lat":51.5196747,"lon":-0.1172364},{"type":"node","id":25503707,"lat":51.5181775,"lon":-0.1147225},{"type":"node","id":25503708,"lat":51.5181037,"lon":-0.1154342},{"type":"node","id":317930304,"lat":51.5174804,"lon":-0.1176573},{"type":"node","id":317930325,"lat":51.5171721,"lon":-0.1174991},{"type":"node","id":317930344,"lat":51.5191424,"lon":-0.1170014},{"type":"node","id":317930350,"lat":51.5192219,"lon":-0.1174636},{"type":"node","id":317930358,"lat":51.5195025,"lon":-0.1178662},{"type":"node","id":317930360,"lat":51.5194545,"lon":-0.1178327},{"type":"node","id":317930362,"lat":51.5194989,"lon":-0.1176175},{"type":"node","id":323256856,"lat":51.5150331,"lon":-0.1127684},{"type":"node","id":338501922,"lat":51.5166156,"lon":-0.1150552},{"type":"node","id":338501923,"lat":51.5188792,"lon":-0.1168667},{"type":"node","id":338505531,"lat":51.5178207,"lon":-0.1177863},{"type":"node","id":338569568,"lat":51.515131,"lon":-0.112308},{"type":"node","id":338576378,"lat":51.5159558,"lon":-0.1145538},{"type":"node","id":354671266,"lat":51.5168989,"lon":-0.1170238},{"type":"node","id":354671417,"lat":51.5155329,"lon":-0.1161151},{"type":"node","id":704139941,"lat":51.5151861,"lon":-0.117732},{"type":"node","id":892620498,"lat":51.5195059,"lon":-0.117644},{"type":"node","id":892620500,"lat":51.5190133,"lon":-0.1175122},{"type":"node","id":892620508,"lat":51.5191648,"lon":-0.117481},{"type":"node","id":892620510,"lat":51.5190984,"lon":-0.1176165},{"type":"node","id":893431458,"lat":51.5191308,"lon":-0.1175171},{"type":"node","id":893489165,"lat":51.5192851,"lon":-0.1174971},{"type":"node","id":965267525,"lat":51.5180519,"lon":-0.1159226},{"type":"node","id":973525136,"lat":51.5178626,"lon":-0.1174284},{"type":"node","id":973525143,"lat":51.5174764,"lon":-0.1172287},{"type":"node","id":1302605368,"lat":51.5149313,"lon":-0.117412},{"type":"node","id":1551117103,"lat":51.5195039,"lon":-0.1171645},{"type":"node","id":1783558256,"lat":51.516163,"lon":-0.1147848},{"type":"node","id":3140855378,"lat":51.5153882,"lon":-0.1142743},{"type":"node","id":3140855379,"lat":51.5159335,"lon":-0.1146477},{"type":"node","id":3140855380,"lat":51.5157064,"lon":-0.1154678},{"type":"node","id":3140855387,"lat":51.5151778,"lon":-0.1175765},{"type":"node","id":3141558549,"lat":51.5147277,"lon":-0.1149553},{"type":"node","id":3142630783,"lat":51.5151726,"lon":-0.1150946},{"type":"node","id":3142630784,"lat":51.5150472,"lon":-0.115572},{"type":"node","id":3148247689,"lat":51.5152133,"lon":-0.1175997},{"type":"node","id":3156898687,"lat":51.5147627,"lon":-0.1164509},{"type":"node","id":3398910717,"lat":51.5190966,"lon":-0.1171851},{"type":"node","id":3519269750,"lat":51.5181303,"lon":-0.1151676},{"type":"node","id":3644748766,"lat":51.5181465,"lon":-0.1150249},{"type":"node","id":4013524084,"lat":51.5146026,"lon":-0.1174572},{"type":"node","id":4013524085,"lat":51.5146711,"lon":-0.1166911},{"type":"node","id":4013524086,"lat":51.5147055,"lon":-0.1166562},{"type":"node","id":4013524087,"lat":51.5154027,"lon":-0.1166231},{"type":"node","id":4215728338,"lat":51.5186185,"lon":-0.1176844},{"type":"node","id":4215728347,"lat":51.5186487,"lon":-0.1175684},{"type":"node","id":4215728357,"lat":51.5186998,"lon":-0.1177418},{"type":"node","id":4215728363,"lat":51.5187314,"lon":-0.1176235},{"type":"node","id":4216272817,"lat":51.5183365,"lon":-0.1167328},{"type":"node","id":4216272819,"lat":51.5183564,"lon":-0.1166346},{"type":"node","id":4216272873,"lat":51.5186644,"lon":-0.1178765},{"type":"node","id":4216273229,"lat":51.5188599,"lon":-0.1178773},{"type":"node","id":4216273232,"lat":51.518891,"lon":-0.117913},{"type":"node","id":4216274099,"lat":51.5186956,"lon":-0.1177577},{"type":"node","id":4216274105,"lat":51.5187047,"lon":-0.117902},{"type":"node","id":4216274120,"lat":51.5187347,"lon":-0.1177861},{"type":"node","id":4285841901,"lat":51.5148471,"lon":-0.1163085},{"type":"node","id":4927544936,"lat":51.514896,"lon":-0.1173004},{"type":"node","id":4927544938,"lat":51.5147519,"lon":-0.1167171},{"type":"node","id":4927544939,"lat":51.5147494,"lon":-0.1166666},{"type":"node","id":4927544940,"lat":51.5147583,"lon":-0.1166327},{"type":"node","id":4927544942,"lat":51.5147944,"lon":-0.1165048},{"type":"node","id":4927544944,"lat":51.5148123,"lon":-0.1164378},{"type":"node","id":5409072334,"lat":51.5202962,"lon":-0.1175716},{"type":"node","id":5416565257,"lat":51.517846,"lon":-0.1175679},{"type":"node","id":5445808815,"lat":51.5176655,"lon":-0.1177274},{"type":"node","id":5445808816,"lat":51.5175733,"lon":-0.1176925},{"type":"node","id":5787555029,"lat":51.514862,"lon":-0.1135974},{"type":"node","id":5803957060,"lat":51.5158674,"lon":-0.1141733},{"type":"node","id":5803957063,"lat":51.5157098,"lon":-0.1131549},{"type":"node","id":5803957066,"lat":51.5155604,"lon":-0.1130792},{"type":"node","id":5803957067,"lat":51.5155548,"lon":-0.1131003},{"type":"node","id":5803957070,"lat":51.5151977,"lon":-0.1135122},{"type":"node","id":5803957071,"lat":51.5151778,"lon":-0.1134697},{"type":"node","id":5803957072,"lat":51.5152449,"lon":-0.1134167},{"type":"node","id":5803957073,"lat":51.5152166,"lon":-0.1134011},{"type":"node","id":5803957075,"lat":51.5153063,"lon":-0.1131486},{"type":"node","id":5803957076,"lat":51.5152807,"lon":-0.1131331},{"type":"node","id":5803957079,"lat":51.5152712,"lon":-0.1132968},{"type":"node","id":5803957080,"lat":51.5152425,"lon":-0.1132797},{"type":"node","id":5803957081,"lat":51.5152051,"lon":-0.1133486},{"type":"node","id":5803957082,"lat":51.5152287,"lon":-0.1133401},{"type":"node","id":5803957083,"lat":51.5152575,"lon":-0.113218},{"type":"node","id":5803957084,"lat":51.5152434,"lon":-0.1131889},{"type":"node","id":5803957085,"lat":51.5151613,"lon":-0.1136538},{"type":"node","id":5803957086,"lat":51.5151931,"lon":-0.1136533},{"type":"node","id":5803957089,"lat":51.5153112,"lon":-0.1137098},{"type":"node","id":5803957090,"lat":51.5152996,"lon":-0.1137558},{"type":"node","id":5803957093,"lat":51.5154991,"lon":-0.1138301},{"type":"node","id":5803957094,"lat":51.5154863,"lon":-0.1138815},{"type":"node","id":5803957096,"lat":51.5157102,"lon":-0.1139628},{"type":"node","id":5803957097,"lat":51.5156932,"lon":-0.1140277},{"type":"node","id":5804044775,"lat":51.5150046,"lon":-0.1136769},{"type":"node","id":5804044776,"lat":51.5149857,"lon":-0.1136665},{"type":"node","id":5817963322,"lat":51.5157363,"lon":-0.113171},{"type":"node","id":5817963323,"lat":51.5151436,"lon":-0.1136959},{"type":"node","id":5817963324,"lat":51.5151976,"lon":-0.1136351},{"type":"node","id":5817963325,"lat":51.5151677,"lon":-0.1136498},{"type":"node","id":5817963327,"lat":51.5153234,"lon":-0.1129477},{"type":"node","id":5817963328,"lat":51.5153275,"lon":-0.1129321},{"type":"node","id":5817963330,"lat":51.5152878,"lon":-0.1129229},{"type":"node","id":5817963331,"lat":51.515282,"lon":-0.1129453},{"type":"node","id":5817963345,"lat":51.515817,"lon":-0.1145677},{"type":"node","id":5817963347,"lat":51.5159766,"lon":-0.1146741},{"type":"node","id":5817964155,"lat":51.5158626,"lon":-0.1140649},{"type":"node","id":5817964156,"lat":51.515848,"lon":-0.1141235},{"type":"node","id":5817964165,"lat":51.5155896,"lon":-0.1139655},{"type":"node","id":5817964166,"lat":51.5155453,"lon":-0.1139378},{"type":"node","id":5817964167,"lat":51.515516,"lon":-0.1139367},{"type":"node","id":5817964170,"lat":51.5156345,"lon":-0.1139981},{"type":"node","id":5817964171,"lat":51.515651,"lon":-0.11403},{"type":"node","id":5821464938,"lat":51.5159004,"lon":-0.1141566},{"type":"node","id":5821464939,"lat":51.5158924,"lon":-0.1141892},{"type":"node","id":6284930123,"lat":51.5180886,"lon":-0.1155925},{"type":"node","id":6287138678,"lat":51.517414,"lon":-0.1163793},{"type":"node","id":6354436150,"lat":51.518063,"lon":-0.1158226},{"type":"way","id":4266560,"nodes":[26591531,3561459278,25607125]},{"type":"way","id":4362702,"nodes":[9789809,26591529,26591528,4171768191,844780668,9789815]},{"type":"way","id":4362733,"nodes":[26591530,4171766882,26591529]},{"type":"way","id":4362736,"nodes":[26591534,26591535]},{"type":"way","id":4362737,"nodes":[26591536,26591537]},{"type":"way","id":53638587,"nodes":[108081,677445938]},{"type":"way","id":182693028,"nodes":[1930510928,2214979331,2214979337,2214979327,25607129,25607102,25607100,25607096,1737601049,1737601058,25607097,3561459279,25607099]},{"type":"way","id":183479838,"nodes":[677445939,1938580025,1938580021,1938580016,1938580015]},{"type":"way","id":183479839,"nodes":[677445939,26591531,108080]},{"type":"way","id":211489815,"nodes":[1938580015,1737601164,2214979339,1938580017,2214979344]},{"type":"way","id":211489820,"nodes":[26591532,108081,26591530,26591535,4171766883,1352935864,26591536,9789816]},{"type":"way","id":211560909,"nodes":[2215490967,2215490965,2215490963,2215503568,108085]},{"type":"way","id":211561187,"nodes":[2215502654,2215490962,25257402]},{"type":"way","id":211561191,"nodes":[9789816,2215503569,2215502656,2215502654]},{"type":"way","id":211561194,"nodes":[108085,2215502650,2215502657,2215502653,2215490966]},{"type":"way","id":211561198,"nodes":[2215490966,2215502651,9789816]},{"type":"way","id":211561202,"nodes":[2215502654,108085]},{"type":"way","id":211561206,"nodes":[2215490966,2215502654]},{"type":"way","id":299395401,"nodes":[108080,2214979347,26591532]},{"type":"way","id":306191312,"nodes":[1930510928,2214979348,2214979325,108079,1938580022,677445939]},{"type":"way","id":350378150,"nodes":[4919996790,4919996791]},{"type":"way","id":350378151,"nodes":[4919996791,311153342]},{"type":"way","id":416435096,"nodes":[4171766886,4171766885]},{"type":"way","id":416435097,"nodes":[4171768189,4171768190]},{"type":"way","id":416435098,"nodes":[4171766882,4171766884]},{"type":"way","id":421258904,"nodes":[2622783492,4210456238]},{"type":"way","id":457420530,"nodes":[4535718259,844780674,4535718260]},{"type":"way","id":457420531,"nodes":[4535718261,844780668]},{"type":"way","id":457420532,"nodes":[4535718260,4535718261]},{"type":"way","id":677965609,"nodes":[26591533,812264806,26591532]},{"type":"way","id":2263973,"nodes":[9789823,9789817,3933850766,3933850770,3933850769,3933850768,3933850767,9789822,9789820,6297968145,9789824,6297968144]},{"type":"way","id":2263979,"nodes":[107796,454543496]},{"type":"way","id":4266579,"nodes":[2214979332,2214979335]},{"type":"way","id":4266580,"nodes":[107796,454543312]},{"type":"way","id":4362704,"nodes":[9789817,506328021,9789806]},{"type":"way","id":4362705,"nodes":[9789821,2214979326,1423409445,9789823]},{"type":"way","id":4362706,"nodes":[3933844368,2214979340]},{"type":"way","id":4369035,"nodes":[2214979346,107796]},{"type":"way","id":4369036,"nodes":[26630784,3557235398]},{"type":"way","id":4369038,"nodes":[454535735,454535722]},{"type":"way","id":4369040,"nodes":[109625,4347914533,1598214612,107796]},{"type":"way","id":4369410,"nodes":[26630784,3557235449,26630785]},{"type":"way","id":25015587,"nodes":[2214979328,271945198]},{"type":"way","id":28333674,"nodes":[108078,3561398893]},{"type":"way","id":38446257,"nodes":[454535807,454535808]},{"type":"way","id":38446963,"nodes":[25607360,454543496]},{"type":"way","id":38446964,"nodes":[454543496,2214979329]},{"type":"way","id":41401256,"nodes":[506328021,1353135844]},{"type":"way","id":53876143,"nodes":[107816,9789818]},{"type":"way","id":120750848,"nodes":[1353135844,1353135845]},{"type":"way","id":120750849,"nodes":[1353135847,1353135849]},{"type":"way","id":120750850,"nodes":[1353135849,1353135846]},{"type":"way","id":120750851,"nodes":[1353135845,506328368]},{"type":"way","id":120750852,"nodes":[1353135848,6266824886,6266824887,1353135847]},{"type":"way","id":183056157,"nodes":[2214979329,3439409107,6329186472,3439409106,454543585,107816]},{"type":"way","id":211489807,"nodes":[454543496,1598214602,2214979333,108084]},{"type":"way","id":211489808,"nodes":[4580163090,4580163089,3439409104,6329186471,454543632,2214979346]},{"type":"way","id":211489813,"nodes":[2214979340,108084]},{"type":"way","id":211489816,"nodes":[2214979332,25607361]},{"type":"way","id":211489821,"nodes":[2214979346,4347914534,2214988206,2214988215,2214988209,2214988212,1637831175]},{"type":"way","id":211489823,"nodes":[2214979329,2214979346]},{"type":"way","id":211490174,"nodes":[1637831175,109625]},{"type":"way","id":232725462,"nodes":[9789821,5357377732,878030001,2214979334,2214979323,2214979349]},{"type":"way","id":256633379,"nodes":[4210456241,2622783496,2622783493,2622783494,4209437291,2622783496]},{"type":"way","id":297206317,"nodes":[6297968144,6297968146,9789818,3933844368]},{"type":"way","id":297206318,"nodes":[108084,2214979332,9789821]},{"type":"way","id":303196145,"nodes":[78598323,853294005,26630783]},{"type":"way","id":349937186,"nodes":[3557235428,3557235429,3557235430,3557235431,3557235432,3557235433,3557235434,3557235418,3557235419,3557235423,3557235446,3557235424,3557235437,3557235425,3557235426,3557235427,3557235420,3557235421,3557235422,3557235428]},{"type":"way","id":349937187,"nodes":[3557235446,3557235435,3557235436,3557235438,3557235439,3557235440,3557235441,3557235442,3557235443,3557235444,3557235445,3557235437]},{"type":"way","id":349937188,"nodes":[3557235449,3557235440]},{"type":"way","id":350378152,"nodes":[3561398893,3561398894]},{"type":"way","id":418321219,"nodes":[4187528683,4187528682]},{"type":"way","id":462641799,"nodes":[3439409106,3439409104]},{"type":"way","id":462662569,"nodes":[4580163090,107816]},{"type":"way","id":547545593,"nodes":[5290771955,454535997,78598321,78598323]},{"type":"way","id":642019682,"nodes":[78598326,2620932405,2214979349]},{"type":"way","id":675336438,"nodes":[3933844368,3713564059,4187528683,9789823]},{"type":"way","id":675836640,"nodes":[6329186510,6329186511]},{"type":"way","id":675836642,"nodes":[6329186512,6329186513]},{"type":"way","id":675836644,"nodes":[6329186514,6329186515]},{"type":"way","id":4082553,"nodes":[21665582,4189248606,4189248448,21665359,21665360,4189248295,4189248449,21665367]},{"type":"way","id":4082558,"nodes":[107795,21665377,21665376,5393414954,21665378,21665374,21665373,3733175891,107793]},{"type":"way","id":4082561,"nodes":[21665361,21665363]},{"type":"way","id":4082563,"nodes":[1694551553,21665006,500486430,9512939]},{"type":"way","id":4082564,"nodes":[107328,5364973542,21665000,9512937,21665578,5932856690,107806]},{"type":"way","id":4082566,"nodes":[9512935,26699551,26699544,9512933,700757166,881888313,25501065,25501064,255576]},{"type":"way","id":4082609,"nodes":[9791492,25501773]},{"type":"way","id":4082612,"nodes":[108073,4721392996,4721392997,4721392998,701608631,21666019,3071350730,9521025,2043878233,4694456248,4694456249,2494845029]},{"type":"way","id":4253462,"nodes":[25473424,5597443392,2457223323,5453319326,5453319327,26845548,26845551,9512926,25501063,25501016]},{"type":"way","id":4253830,"nodes":[108059,25257692,25473295]},{"type":"way","id":4253890,"nodes":[108058,2939877210,6250445847,108059]},{"type":"way","id":4259144,"nodes":[108085,108267,26652135,108248]},{"type":"way","id":4259245,"nodes":[9791152,4718985842,26866991,3071350729,4718985840,21666019]},{"type":"way","id":4259250,"nodes":[4712556912,25502545]},{"type":"way","id":4259296,"nodes":[25501773,21666002,2480891615,2047806106]},{"type":"way","id":4356102,"nodes":[107816,1598214606]},{"type":"way","id":4358432,"nodes":[9512915,4349648270,6214704101,26559363,9512917,6214704104,6214704108]},{"type":"way","id":4358459,"nodes":[9789806,1353135846,271945198,9789809,311153342,108080]},{"type":"way","id":4358462,"nodes":[9789811,9789814,9789820]},{"type":"way","id":4358790,"nodes":[6214704108,6214704111,107801]},{"type":"way","id":4360147,"nodes":[26573672,26573670]},{"type":"way","id":4362701,"nodes":[9789811,677445941,9789819,506328569,9789815,9789816]},{"type":"way","id":4362731,"nodes":[26591526,9789814]},{"type":"way","id":4362734,"nodes":[26591527,26591528]},{"type":"way","id":4370888,"nodes":[1947324461,4190435879,25257618]},{"type":"way","id":4373939,"nodes":[25620548,1188939749,3439409128,3439409129,3439409130,3439409131,25620549,1188939728,3439409132,3439409133,25620550]},{"type":"way","id":4376565,"nodes":[21665000,25501000,500486430]},{"type":"way","id":4393281,"nodes":[3733175891,26846364,21665909]},{"type":"way","id":5237969,"nodes":[26559618,3037800598,3037800597,2214988215]},{"type":"way","id":38748498,"nodes":[4347844147,2941614011,4347870215,4347870212,881888130,2949076333,4631979552,108057,6250445849,4347844162,2939877211,108058,2398143023,1188939768,2398143020,108261,2398143018,917763911,2398143015,108262,917763922]},{"type":"way","id":40976543,"nodes":[25473293,4721393038,25501289,108072]},{"type":"way","id":43491848,"nodes":[26573670,4347914541,1637831207,3439409119,1188939726,3439409121,3439409120,1637831175]},{"type":"way","id":55873939,"nodes":[701600720,701600721,701600732]},{"type":"way","id":57007567,"nodes":[21665934,4684519082,4684520669,4684519081,21665965]},{"type":"way","id":136456829,"nodes":[108072,2784682613,2784682598,701600704]},{"type":"way","id":155313766,"nodes":[686339678,3783436255,2620932404,1737601113,28782062,26630783]},{"type":"way","id":209690395,"nodes":[2198870637,2480891616,2047806121,4629852538,2047806103,2047806106]},{"type":"way","id":210500025,"nodes":[26573670,3439409118,3439409117,3439409116,3439409115,4347914535,1637831214,3439409114,1938580046,3439409112,3439409113,26630839]},{"type":"way","id":211486426,"nodes":[107818,2214957487,2214957483,2214957480,2214957481,2214957488,2214952984,4638404785,2214952978,2214952980,2214952985,2215475640,2214954341,25257616]},{"type":"way","id":211489809,"nodes":[2214979344,108078,2214979330,2214979328,5357377731,2214979326,2214979335,2214979340,2214979341,2214979338,2214979329]},{"type":"way","id":211489814,"nodes":[2214979349,2214979336,2214979343,2214979350,1930510928]},{"type":"way","id":211489817,"nodes":[2214979344,2214979345,2214979342,2214979351,2214979349]},{"type":"way","id":211490210,"nodes":[1637831175,26630839]},{"type":"way","id":211557607,"nodes":[108085,2215475643,2215475645,109631]},{"type":"way","id":211557608,"nodes":[25257616,2215506073,2215472136,6034520602,9789819,4209492990,2622783492,26591527,4170421482,9789806]},{"type":"way","id":211557609,"nodes":[2214952982,26652140,25257618,2214952986,26652117,26652129,4638404786,2214952983,2214957485,2214957484,1227938943,2214957482,2214957486,2214957479,1813223023]},{"type":"way","id":240343183,"nodes":[2480897574,9791492,1947324488,701600741,1493328983,9791491,21665965,4702603652,4702603654,9521025]},{"type":"way","id":273631916,"nodes":[2784682599,2784682609]},{"type":"way","id":279088480,"nodes":[2833445978,21665006]},{"type":"way","id":290461915,"nodes":[2939875788,26652125,452215489]},{"type":"way","id":302690919,"nodes":[108054,2480891614,1947324489,2047806132,21666021,4629852537,25257027]},{"type":"way","id":304118931,"nodes":[9791152,25501184,3071350728,9512930]},{"type":"way","id":349937184,"nodes":[3557235398,25607361,25607360,454543312,26630790]},{"type":"way","id":350378153,"nodes":[3561398894,4919996790]},{"type":"way","id":416223959,"nodes":[4170421482,4170421481]},{"type":"way","id":421258903,"nodes":[4210456238,4210456241]},{"type":"way","id":423007010,"nodes":[734320209,25473288,25473512,108070,25473371,25473295]},{"type":"way","id":436938220,"nodes":[4638404788,4347844135,4347880279,4638404787,4347844137,4347844151,4347844153,4347844156,4347844157,4347844154,4347844155,4347880277,4347844131,4347880278,4347844136,4347844138,4347844140,4631979551,4347844126,4347870214,4347870211,5657844054,6250445848,4347844133,4347870213,4347844130,4347844161,4347844134,4347844159,4631979549,452589683,4347805227,2131644076,108055]},{"type":"way","id":437139417,"nodes":[9512915,4349648272,21665909]},{"type":"way","id":462641798,"nodes":[4580163090,1598214606]},{"type":"way","id":469810338,"nodes":[4641072777,4580163090]},{"type":"way","id":542471105,"nodes":[5244680786,5244680779,1334645223,5364578869,1938580068,107809,1938580071,9512924,281455002,107807,5392928741,5364578868,107806]},{"type":"way","id":603198588,"nodes":[25620548,1637831150,3439409125,3439409124,26630793,3439409123,1938580042,3439409122,1637831151,109625]},{"type":"way","id":615092345,"nodes":[26630783,345881288]},{"type":"way","id":616192639,"nodes":[9512935,9512937,9512939]},{"type":"way","id":628422482,"nodes":[5932856688,5932856692,21665578,5932856687,5932856691]},{"type":"way","id":628426435,"nodes":[5932856691,5932856686,5932856690,5932856689,5932856688]},{"type":"way","id":647037413,"nodes":[26559582,6214704112]},{"type":"way","id":669229894,"nodes":[6266808745,2215472133,6266808744,2215472138,2038064029,2215475647,2215490966]},{"type":"way","id":673025496,"nodes":[25473286,466926169,466926170,25473287]},{"type":"way","id":673569249,"nodes":[25473285,466926160,25473286]},{"type":"way","id":4082605,"nodes":[2047806092,9791494]},{"type":"way","id":4253790,"nodes":[25473286,108070]},{"type":"way","id":4253807,"nodes":[25473399,25473398,2465858762,25473375]},{"type":"way","id":4253858,"nodes":[25473512,2784434277,25473513,2784434327,21665930,21665931,21665929,4233926316,21665927]},{"type":"way","id":4253869,"nodes":[25473557,1496724129,499350858]},{"type":"way","id":4258932,"nodes":[108054,2480891600]},{"type":"way","id":4259090,"nodes":[25531001,6352084888,25531003]},{"type":"way","id":4259091,"nodes":[9791494,2047806133,2480891607,2480891606,2131603302,25531001,452219937,5630578551,25257039]},{"type":"way","id":4259122,"nodes":[452219937,25531180,25531181]},{"type":"way","id":4259126,"nodes":[2131643980,2131643994,25257039]},{"type":"way","id":4259249,"nodes":[2131643980,454368514,25257040,25257041,25257042]},{"type":"way","id":4259262,"nodes":[25257046,25257041]},{"type":"way","id":4259266,"nodes":[108247,25257045,25257044]},{"type":"way","id":4259268,"nodes":[25257046,25257044,25257463]},{"type":"way","id":4259269,"nodes":[25257323,26651416,25257045]},{"type":"way","id":4259271,"nodes":[25257042,4347844146,2941548715,4347844134]},{"type":"way","id":4259311,"nodes":[25531990,2129184256,25257040]},{"type":"way","id":4259346,"nodes":[25473398,2771382066,2771382106,2771382135,2784434458,2771382140,21665926]},{"type":"way","id":4259348,"nodes":[21665927,21665926,701600743,9791491]},{"type":"way","id":4259349,"nodes":[4233926315,4233926316,4233926317,701600744,25501320]},{"type":"way","id":4259350,"nodes":[25501330,2772491048,25501328,25501325,2772491086,25501320]},{"type":"way","id":4259351,"nodes":[21665929,2772491064,2772491078,25501325]},{"type":"way","id":4259352,"nodes":[21665931,2772491052,2323063175,2772491062,25501328]},{"type":"way","id":4259353,"nodes":[25473373,2784434339,2784434416,21665930]},{"type":"way","id":4259355,"nodes":[5568839679,4394862789]},{"type":"way","id":4370889,"nodes":[2129908162,1947324460,1947324461,4347880280,2398143027,4347844131]},{"type":"way","id":4370890,"nodes":[2910874877,1238318167,25257323,25257463,454360126,2129908162]},{"type":"way","id":4370939,"nodes":[25257044,25257043]},{"type":"way","id":4370942,"nodes":[108249,3458411641,2910874877]},{"type":"way","id":4370943,"nodes":[2910874877,5240648782,26652137,25257324,2215502652,1771472506,2932939310,109631]},{"type":"way","id":4370944,"nodes":[25257324,108267,2934851453,25257321]},{"type":"way","id":4370945,"nodes":[25257043,25257042]},{"type":"way","id":4370947,"nodes":[4347844156,2398143017,2939875794]},{"type":"way","id":18791974,"nodes":[25257043,454368510,2129908162]},{"type":"way","id":18791975,"nodes":[2131643979,454368522,454368521,454368517,2131643980]},{"type":"way","id":22760594,"nodes":[2131643980,2131643981,4631979548,4347805229,1947324474,108055]},{"type":"way","id":24503625,"nodes":[266481785,266481798]},{"type":"way","id":24503662,"nodes":[452589682,266481798,4347844140]},{"type":"way","id":38329791,"nodes":[25257046,454360127,454360129,25257047,454360131,454360130,2129184196]},{"type":"way","id":38329792,"nodes":[452589685,452589682,452589687]},{"type":"way","id":40976540,"nodes":[25501320,25473557]},{"type":"way","id":40976541,"nodes":[499350858,1496723557,25501340,701600719,108072]},{"type":"way","id":40976542,"nodes":[25501330,2323063170,25501340]},{"type":"way","id":43365534,"nodes":[108055,2480891612,9791494]},{"type":"way","id":70854122,"nodes":[26651416,26651415,5673161958,5673161959,844784034]},{"type":"way","id":95207348,"nodes":[6266808747,1104463100]},{"type":"way","id":95207350,"nodes":[1104463100,1104463102]},{"type":"way","id":96228569,"nodes":[5568839679,3077397235]},{"type":"way","id":96228570,"nodes":[5568839679,5112906885]},{"type":"way","id":202958486,"nodes":[2129184256,2129184285]},{"type":"way","id":209692628,"nodes":[2047806108,2480890185,2047806113,2480891617,2047806092]},{"type":"way","id":216153513,"nodes":[2047806108,108054]},{"type":"way","id":232486356,"nodes":[2047806092,2480891605,2480891609,25473399,2480897574]},{"type":"way","id":232724498,"nodes":[2047806092,2047806097,2047806126,4347805230,4347844152]},{"type":"way","id":232725808,"nodes":[6266808742,6266808746,6266808741,2214952982]},{"type":"way","id":232725809,"nodes":[109631,2215472141,25257617,6266808745]},{"type":"way","id":238942185,"nodes":[25473375,4219739065,25473373,5112906885,5112906884,25473371]},{"type":"way","id":240343066,"nodes":[9791494,2480891601,2480891602,108054]},{"type":"way","id":240343067,"nodes":[2047806106,2480891608,2480891599,2047806134,2047806095,2480891611,2047806108]},{"type":"way","id":262200662,"nodes":[25473295,4347870216,2949076335,881888130,6250445848]},{"type":"way","id":262200664,"nodes":[108072,2784682614]},{"type":"way","id":272084501,"nodes":[21665930,2772491043,25501330]},{"type":"way","id":272084533,"nodes":[499350858,2770153328]},{"type":"way","id":272084534,"nodes":[2770153328,2770153342]},{"type":"way","id":273631893,"nodes":[2784682614,2784682610,2784682584,25473513]},{"type":"way","id":273631917,"nodes":[2784682596,2784682599]},{"type":"way","id":273631919,"nodes":[2784682614,2784682619]},{"type":"way","id":273631921,"nodes":[2784682619,2784682621,2784682639]},{"type":"way","id":282036542,"nodes":[25257039,3069482721,3069482720,25256840]},{"type":"way","id":290461914,"nodes":[2939875794,2939875788]},{"type":"way","id":303385359,"nodes":[5568839679,2784434310,2784434322,2784434327]},{"type":"way","id":436938218,"nodes":[4347844152,4347844128,4347844142,4631979550,4347844127,4347844160,4347844147]},{"type":"way","id":436941633,"nodes":[4347844131,108058]},{"type":"way","id":440868727,"nodes":[1104463102,1104463098]},{"type":"way","id":441784313,"nodes":[4394860088,108070]},{"type":"way","id":441784316,"nodes":[4394862789,4394860088]},{"type":"way","id":468720688,"nodes":[4347844134,4347844147]},{"type":"way","id":468720689,"nodes":[108055,4347844152]},{"type":"way","id":667552911,"nodes":[6250340171,25531002,6313645586,6250340166]},{"type":"way","id":667552915,"nodes":[2480891600,6250340168,6250340171]},{"type":"way","id":667552930,"nodes":[6250340171,6250340169,2480894744,2480891600]},{"type":"way","id":667558951,"nodes":[2941537163,5112906884]},{"type":"way","id":667558962,"nodes":[4347844144,25473375,2784434451,4219739061,4233926315]},{"type":"way","id":667562259,"nodes":[4347844140,6250445849]},{"type":"way","id":667562260,"nodes":[2941537158,25257692]},{"type":"way","id":669229892,"nodes":[25257616,6266808740,6266808743,6266808742]},{"type":"way","id":669229893,"nodes":[6266808745,6266808748,469780509,6266808747,6266808742]},{"type":"way","id":674173948,"nodes":[25531003,25531002]},{"type":"way","id":2263976,"nodes":[107802,26559618,4641072777]},{"type":"way","id":4082560,"nodes":[21665361,107807]},{"type":"way","id":4082597,"nodes":[21665000,9512933]},{"type":"way","id":4253776,"nodes":[108060,25473285]},{"type":"way","id":4253785,"nodes":[25473284,25257657]},{"type":"way","id":4253792,"nodes":[25473288,4739043795,25473287]},{"type":"way","id":4253794,"nodes":[25473284,466926153,25473285]},{"type":"way","id":4253812,"nodes":[108059,108060,25257657,25473300,25475484,25473291]},{"type":"way","id":4253875,"nodes":[25473300,2801490748]},{"type":"way","id":4253891,"nodes":[108263,2939373037]},{"type":"way","id":4253894,"nodes":[25257657,3357928883,107820]},{"type":"way","id":4253927,"nodes":[9512926,25502541,1906612405,3054882700,9512930]},{"type":"way","id":4253972,"nodes":[107800,2494798964,25475083,1119774076,107809]},{"type":"way","id":4253984,"nodes":[9512925,5597443393,9512935]},{"type":"way","id":4254009,"nodes":[25475478,701608597,4737289787,25475484]},{"type":"way","id":4254010,"nodes":[25473291,4731047368,1188939746,4731047332,4731047323,4731047383,25473408]},{"type":"way","id":4254044,"nodes":[25473408,4731047412,3316828730]},{"type":"way","id":4254981,"nodes":[107820,2465838818,3729216344,5877193901,5877193900,1119774074,5244680781]},{"type":"way","id":4256251,"nodes":[25502541,25502540]},{"type":"way","id":4259354,"nodes":[1324710127,25501288,701608613,25501289]},{"type":"way","id":4356098,"nodes":[107798,26561311,107799,107800,4349648267,4349648269,9512915]},{"type":"way","id":4356147,"nodes":[107801,2465838823]},{"type":"way","id":4358420,"nodes":[26559369,5444256387,26559370]},{"type":"way","id":4358421,"nodes":[9789808,2704097364,2704097365,26559371,3714663965,26559372]},{"type":"way","id":4358430,"nodes":[26559582,107801]},{"type":"way","id":4358431,"nodes":[26559581,26559582]},{"type":"way","id":4358438,"nodes":[26559371,5444256385,5444256384,26559369]},{"type":"way","id":4358439,"nodes":[26559368,4888988713,5444256389,26559369]},{"type":"way","id":4358440,"nodes":[26559368,5444256386,26559616]},{"type":"way","id":4358441,"nodes":[26559620,6329186022,5444256382,26559616]},{"type":"way","id":4358454,"nodes":[107813,4347900573,1938580061,107798]},{"type":"way","id":4358461,"nodes":[9789808,2704097361,26651413,26652133,9789811]},{"type":"way","id":4360148,"nodes":[26559616,5444256383,1663003820]},{"type":"way","id":4362732,"nodes":[9789814,26591525]},{"type":"way","id":4369048,"nodes":[107798,1938580057,26559372]},{"type":"way","id":4369189,"nodes":[25257643,2205117785]},{"type":"way","id":4370886,"nodes":[2214957485,4190435888,917763920,2939373031]},{"type":"way","id":4370946,"nodes":[2214957488,1938580051,26651413]},{"type":"way","id":4370948,"nodes":[26652119,26652126,1493458889]},{"type":"way","id":4370986,"nodes":[25473408,701608620,4731045392,25473409]},{"type":"way","id":4370987,"nodes":[107812,2587253639]},{"type":"way","id":4370988,"nodes":[26652357,4796603767,26652351]},{"type":"way","id":4370989,"nodes":[26652362,107812]},{"type":"way","id":4376564,"nodes":[26845545,4730390419,4730390416,4730390426,4730390427,1188939751,26845543,9512928,5453319326]},{"type":"way","id":4376567,"nodes":[21665577,5932875003]},{"type":"way","id":4376569,"nodes":[5453319327,5128437130,21665114]},{"type":"way","id":4376570,"nodes":[6311528475,26699545]},{"type":"way","id":4376571,"nodes":[1813223023,1938580055,1660481781,2205117799,2205117782,2205117797,2205117781,25257643]},{"type":"way","id":4376572,"nodes":[26699537,21665386,21665387,21665361]},{"type":"way","id":4376573,"nodes":[5392928741,2484858548,5932875002,5932875004]},{"type":"way","id":4376575,"nodes":[21665360,4189248608,21665361]},{"type":"way","id":4376577,"nodes":[26699549,26699539,26699551]},{"type":"way","id":4376578,"nodes":[21665114,2791710744,26699544]},{"type":"way","id":4376580,"nodes":[107813,5244680780,5244680782,5244680786]},{"type":"way","id":4393236,"nodes":[107821,1322321604]},{"type":"way","id":4393237,"nodes":[9512925,9512927]},{"type":"way","id":4393238,"nodes":[26699543,2587253639]},{"type":"way","id":4393240,"nodes":[26845543,4728954671,26845544,1330788578,4728954676,26845546,21666011]},{"type":"way","id":4393241,"nodes":[26845544,1330788629]},{"type":"way","id":4393243,"nodes":[772092173,4052098238,2791735965]},{"type":"way","id":4395793,"nodes":[26866991,4718985841,4709189057]},{"type":"way","id":4395794,"nodes":[21665390,25501065]},{"type":"way","id":4395795,"nodes":[21665390,5928875937]},{"type":"way","id":4480716,"nodes":[107793,4189397636,4189248446,25475069,21665582,26699537,9512924]},{"type":"way","id":4480717,"nodes":[9512915,4349648276,4349648275,107793]},{"type":"way","id":23198010,"nodes":[107801,26559362,107799]},{"type":"way","id":23198011,"nodes":[107797,26559368]},{"type":"way","id":35085535,"nodes":[26559372,4641072778,26559370,4347914538,4347914536,107797]},{"type":"way","id":35085536,"nodes":[9789808,1387235646,1227938934,1227938936,1387235656,107818]},{"type":"way","id":35085537,"nodes":[107818,2349657350,1813223023]},{"type":"way","id":38310154,"nodes":[452215489,26652119,4190435881,26652117]},{"type":"way","id":43118736,"nodes":[26241038,4638405891,1663004185,1188939774,2205124744,1322321579,2205127955,25257643]},{"type":"way","id":43486254,"nodes":[25257643,1322321619,2205124756,2205124752,1188983785]},{"type":"way","id":53638574,"nodes":[26651413,677445933,677445934,677445936]},{"type":"way","id":53876136,"nodes":[107802,26573670]},{"type":"way","id":55873935,"nodes":[701600704,701608595,734320209]},{"type":"way","id":55874494,"nodes":[701608613,701608618]},{"type":"way","id":61926916,"nodes":[771982848,26866992,771982865,771982972]},{"type":"way","id":61926920,"nodes":[771982865,4709189304]},{"type":"way","id":68543461,"nodes":[9512924,497457752,3729362050,26699545,9512925]},{"type":"way","id":68543462,"nodes":[9512925,5597443390,5597443391,25473424]},{"type":"way","id":68880084,"nodes":[108059,5686381085]},{"type":"way","id":74662590,"nodes":[2919558531,881888314]},{"type":"way","id":96690719,"nodes":[107820,107821]},{"type":"way","id":105600260,"nodes":[26699543,348875443,9512927,2476787983,25473423]},{"type":"way","id":133730761,"nodes":[21665577,9512935]},{"type":"way","id":136097421,"nodes":[1493424643,1493424644]},{"type":"way","id":153661834,"nodes":[1663004118,1227938942]},{"type":"way","id":153661837,"nodes":[1663004159,1567896328]},{"type":"way","id":180217865,"nodes":[9512930,3054882698,26845547,772092173,9512928]},{"type":"way","id":188281273,"nodes":[108263,1938450898,917763912,4347885261,4638405890,917763913,26241038]},{"type":"way","id":193420534,"nodes":[1813223023,4347899437]},{"type":"way","id":210447794,"nodes":[1188983785,2205124739,2205124769,2205124732,2205124764,2205117785,1322321517,2205117789,2205117791,2205117802,2205117798,2205117787,2205117800,107818]},{"type":"way","id":211487026,"nodes":[1188983785,2205124760,5364578870,5877193899,107813]},{"type":"way","id":215065448,"nodes":[2205117785,2205127953,2205117795,1938580060,107815,4347899047,1227938940,107798]},{"type":"way","id":240505256,"nodes":[25473408,25475478,25473556,734320209]},{"type":"way","id":246902443,"nodes":[108073,25473293]},{"type":"way","id":262200663,"nodes":[25473291,1188939750,1493424643,6065422037,25473562,25473423,25473424]},{"type":"way","id":262200665,"nodes":[9512930,3054882699,3054882697,3054882696,21666011,21666010,4718984414,108073]},{"type":"way","id":262200666,"nodes":[1330788629,25502137,1330788638,25473409,21666010]},{"type":"way","id":273631918,"nodes":[2784682609,2784682613]},{"type":"way","id":273631920,"nodes":[2784682598,2784682605,2784682616]},{"type":"way","id":274458828,"nodes":[2791735965,2791735968]},{"type":"way","id":274458829,"nodes":[2791735968,26845548]},{"type":"way","id":275510215,"nodes":[2801490748,2801490747]},{"type":"way","id":275510216,"nodes":[2801490747,1188939742,4739043808,4739043742,25473556]},{"type":"way","id":288394413,"nodes":[881888314,2919558521]},{"type":"way","id":288394414,"nodes":[2919558521,881888313]},{"type":"way","id":290405992,"nodes":[2939373031,2939373032]},{"type":"way","id":290405993,"nodes":[2939373037,2939373042]},{"type":"way","id":290405994,"nodes":[2939373042,108061,108060]},{"type":"way","id":290405995,"nodes":[2939373032,1663004143,108263]},{"type":"way","id":325008373,"nodes":[3316828730,25473293]},{"type":"way","id":325008374,"nodes":[3316828729,107809]},{"type":"way","id":369099971,"nodes":[26652362,3729216343,3729216345,3731994707,3729216344]},{"type":"way","id":369118442,"nodes":[3729362050,3729362052]},{"type":"way","id":371306107,"nodes":[25501182,771982929,25501184]},{"type":"way","id":399696957,"nodes":[917763922,4638405889,2398143012,1938450896,108263]},{"type":"way","id":436938229,"nodes":[108263,4347844132,4347844150,4347844149,4638404788]},{"type":"way","id":436943364,"nodes":[4347899437,1322321549,1188939770,1387235630,1387235625,2244856019,1322321696,2244856015,26241038]},{"type":"way","id":436945579,"nodes":[107797,4641072779,26559581,4347914537,2465838826,107802]},{"type":"way","id":462641800,"nodes":[1598214606,26559620,4641072776,4641072775,2704097363,2704097362,9789808]},{"type":"way","id":477598144,"nodes":[4709189303,771982929]},{"type":"way","id":477598147,"nodes":[4709189304,4709189303]},{"type":"way","id":477598199,"nodes":[4709189057,771982848]},{"type":"way","id":477598215,"nodes":[4709189404,4709189403]},{"type":"way","id":477598216,"nodes":[4709189430,4709189404]},{"type":"way","id":477598224,"nodes":[701608631,4709189430]},{"type":"way","id":480012223,"nodes":[1330788629,4730390165]},{"type":"way","id":480012233,"nodes":[4730390416,4730390415,4730390156]},{"type":"way","id":480012234,"nodes":[4730390419,4730390418,4730390417]},{"type":"way","id":480012236,"nodes":[4730390441,26845545]},{"type":"way","id":480077795,"nodes":[5518619474,4731047323]},{"type":"way","id":480077798,"nodes":[4731047333,4731047332]},{"type":"way","id":480077807,"nodes":[4731047369,4731047368]},{"type":"way","id":480077811,"nodes":[4731047386,4731047383]},{"type":"way","id":480954362,"nodes":[4739043743,4739043742]},{"type":"way","id":542468803,"nodes":[2465838823,107797]},{"type":"way","id":542471106,"nodes":[5244680786,5244680784,5244680785,107813]},{"type":"way","id":542471107,"nodes":[5244680779,5244680778,1119774075,5244680783,25502191,107812,26652351,1188939750]},{"type":"way","id":565206742,"nodes":[1663003820,5444256388,5444256381,26559618]},{"type":"way","id":570827697,"nodes":[6065422037,5487714089]},{"type":"way","id":570827698,"nodes":[5487714089,5487714090]},{"type":"way","id":570827699,"nodes":[5487714090,5487714091]},{"type":"way","id":585788252,"nodes":[25473424,5597443394,5597443395,9512925]},{"type":"way","id":590640978,"nodes":[5244680781,107813]},{"type":"way","id":597111399,"nodes":[108061,3075938596,3357929264,107821]},{"type":"way","id":597111400,"nodes":[5686382234,108061]},{"type":"way","id":597111497,"nodes":[5686381085,5686382234]},{"type":"way","id":628033667,"nodes":[5928875936,25502552]},{"type":"way","id":628033668,"nodes":[5928875937,5928875936]},{"type":"way","id":628424979,"nodes":[5932875003,5932875005,5932875004]},{"type":"way","id":628424980,"nodes":[5932875003,5932856688]},{"type":"way","id":628424981,"nodes":[5932875004,21665577]},{"type":"way","id":673569250,"nodes":[25473287,466926176,25473284]},{"type":"way","id":673956939,"nodes":[6311528475,26699543]},{"type":"way","id":673956940,"nodes":[2587253639,3316828729]},{"type":"way","id":4082610,"nodes":[21665932,4684517980,21665933,4684520597,21665934]},{"type":"way","id":4256301,"nodes":[25502866,1684410081]},{"type":"way","id":4256487,"nodes":[25503799,1953893721,25503801,333722494]},{"type":"way","id":4259293,"nodes":[21666003,4684520664,4684520654,4684520666,4684520629,4684520665,21665934,21666007]},{"type":"way","id":4259294,"nodes":[25501940,2198881216,2198881213,2198881214,2047806102]},{"type":"way","id":4259295,"nodes":[2047806107,21665915,25501773]},{"type":"way","id":30279889,"nodes":[333722502,5216516550]},{"type":"way","id":30279890,"nodes":[255563,1685938596,3054882705,9521018]},{"type":"way","id":32629148,"nodes":[255564,366906363,25502834]},{"type":"way","id":32629353,"nodes":[9791156,496704385,1126076446,25502834,1684410080]},{"type":"way","id":33608060,"nodes":[1684410076,2467533676,255566]},{"type":"way","id":33608061,"nodes":[1684410084,108053]},{"type":"way","id":33608388,"nodes":[383974165,383974166]},{"type":"way","id":54918283,"nodes":[255563,366905603,2409829435,255562]},{"type":"way","id":55026149,"nodes":[692129349,1684410113,255563]},{"type":"way","id":104867630,"nodes":[9521031,383974166]},{"type":"way","id":156155819,"nodes":[1126101190,1647906601,496813005,9791156,411545972,255563]},{"type":"way","id":156155830,"nodes":[1684410076,1684410078,1684410080,1684410081,1684410083,1684410084,1126101190]},{"type":"way","id":194259717,"nodes":[383974166,2047806118,2414133930,2047806098,4581297847,2047806102,2047806084]},{"type":"way","id":209690397,"nodes":[2047806084,2198881218,2047806107,2414133927,2198870628]},{"type":"way","id":209691453,"nodes":[2047806084,255566]},{"type":"way","id":209691454,"nodes":[25501940,25502866,108053,4581297848,383974166]},{"type":"way","id":209691455,"nodes":[799284841,4684520828,4684520827,4684520702,4684520594,21666003]},{"type":"way","id":209691456,"nodes":[255566,25501940]},{"type":"way","id":209692646,"nodes":[691084080,1617713615,1684410105,2409829432,255562,2409829444,255564,1684410076]},{"type":"way","id":233070705,"nodes":[2198870639,2198881217,691084099,255566]},{"type":"way","id":431325293,"nodes":[4305326859,4305334389,4305334390,4305334391,659654484]},{"type":"way","id":431325294,"nodes":[4305326864,4305326862]},{"type":"way","id":431325295,"nodes":[4305326863,4305343211]},{"type":"way","id":432503409,"nodes":[4305326861,659654483]},{"type":"way","id":432503416,"nodes":[4305326856,659654482]},{"type":"way","id":474509251,"nodes":[4684520630,4684519080,4684520629]},{"type":"way","id":474509257,"nodes":[4684520654,4684520591,4684520653]},{"type":"way","id":474509258,"nodes":[4684520717,4684519083,4684520664]},{"type":"way","id":474509259,"nodes":[4684517986,4684519085,4684520665]},{"type":"way","id":474509260,"nodes":[4684517987,4684519084,4684520666]},{"type":"way","id":474509261,"nodes":[4684520669,4684520667,4684520668]},{"type":"way","id":474509268,"nodes":[4684517940,21665932]},{"type":"way","id":474509289,"nodes":[2047806102,799284841]},{"type":"way","id":474509290,"nodes":[4684520713,4684520703,4684520702]},{"type":"way","id":475957302,"nodes":[4696358371,9521031]},{"type":"way","id":476776082,"nodes":[4702603653,4702603652]},{"type":"way","id":476776083,"nodes":[4702603655,4702603654]},{"type":"way","id":539031897,"nodes":[5216516550,5216516549]},{"type":"way","id":539031901,"nodes":[5216516549,333722501]},{"type":"way","id":2644236,"nodes":[11863146,5239556145,333719180,6355365946,25472045]},{"type":"way","id":2644707,"nodes":[11863145,1448889416,25471909,333670393]},{"type":"way","id":4068454,"nodes":[25500758,194250,367735523,25495441]},{"type":"way","id":4082521,"nodes":[107324,1694551558]},{"type":"way","id":4082565,"nodes":[21665319,255576]},{"type":"way","id":4082603,"nodes":[9521022,4712556877,4712556848,21665937,4712556857,4712556846,1089052862,4712556827,21665938]},{"type":"way","id":4082604,"nodes":[9521025,4702603702,3071350732,2494845032]},{"type":"way","id":4082607,"nodes":[9521029,3054882702,2494845032]},{"type":"way","id":4253366,"nodes":[25471919,25471918,25471917,25471915,25471914,25471913,25471911,2985253988,25471910,25471909]},{"type":"way","id":4253380,"nodes":[25472045,25472049,1448835615,25472050,25472051,25472052,25472053,4222961755,25472054,25472055,1448835614,25472056,25472057]},{"type":"way","id":4253386,"nodes":[107848,4231049408,6032776011,6032776010,14791189]},{"type":"way","id":4253395,"nodes":[3720610535,6032776008,6032776012,14791189]},{"type":"way","id":4253925,"nodes":[1906612406,2453179640,9791152]},{"type":"way","id":4256032,"nodes":[1685751696,1484198385,25500120,25500123,1484198280,565545547]},{"type":"way","id":4256033,"nodes":[25500123,25500126,4243601481]},{"type":"way","id":4256034,"nodes":[25417426,1785339688,25417427,1484205770,25417428]},{"type":"way","id":4256035,"nodes":[25417428,1484205791,1484205753,25417430,25417431]},{"type":"way","id":4256036,"nodes":[25495441,333670402,4247304446,6355365944,6355365935,5811145644,9791484]},{"type":"way","id":4256090,"nodes":[25503710,333670412,4249404286,4249404283,4249404271,4249404266,4249404263,194250]},{"type":"way","id":4256302,"nodes":[496812960,333722500,9521006]},{"type":"way","id":4256397,"nodes":[108062,2425141027,2425140535,496812960]},{"type":"way","id":4256398,"nodes":[9521035,2417298649,5758343877,6089225181,5337345855,255578]},{"type":"way","id":4256565,"nodes":[25495441,6279571027,1685751723]},{"type":"way","id":4256728,"nodes":[25505558,25505566]},{"type":"way","id":4257851,"nodes":[9791152,25502544,9791153]},{"type":"way","id":4259243,"nodes":[21665945,21665319]},{"type":"way","id":4259244,"nodes":[21665945,4712556963]},{"type":"way","id":4272915,"nodes":[25500900,2398312017,25703180]},{"type":"way","id":4395791,"nodes":[255576,107326]},{"type":"way","id":4395792,"nodes":[9521022,3071350731,1906612406]},{"type":"way","id":4816641,"nodes":[1685751731,2356344242,25505566,25500758]},{"type":"way","id":4816642,"nodes":[25500120,30939233,25500126]},{"type":"way","id":4816643,"nodes":[30939234,25417431]},{"type":"way","id":10064546,"nodes":[82710879,2522315148]},{"type":"way","id":24865179,"nodes":[107319,2440641461,9791487,983839058]},{"type":"way","id":30131460,"nodes":[25503710,4249404308,332040899]},{"type":"way","id":30131461,"nodes":[332040899,5239556147,5239556146,5239556407,11863146]},{"type":"way","id":30273083,"nodes":[333650005,333650013,333650015,333650017]},{"type":"way","id":30273505,"nodes":[255578,333722488,82710879,4299259647,255579,4299259661,2440643034,2440643035,2440643030,107316]},{"type":"way","id":30275207,"nodes":[333670393,1448896143,5768795953]},{"type":"way","id":30275210,"nodes":[4519038134,5150471750]},{"type":"way","id":30279582,"nodes":[107318,25501005,2440641465,2440641460,107319,2440641466,2440641464,25500900,25500758]},{"type":"way","id":30279583,"nodes":[9521011,4301399740]},{"type":"way","id":30279885,"nodes":[918548280,3524428945]},{"type":"way","id":30279887,"nodes":[255578,9521011,496812960]},{"type":"way","id":33096424,"nodes":[107322,2417297528,2417297527,1938953378,385562683,21665586]},{"type":"way","id":33096425,"nodes":[21665938,6264884981,4712556977,1089052872]},{"type":"way","id":33437336,"nodes":[565545547,380494805,380494808]},{"type":"way","id":33437477,"nodes":[1685751700,2520452050]},{"type":"way","id":33722369,"nodes":[21665586,5935822696,6313625296,6313625295]},{"type":"way","id":41660591,"nodes":[333650017,2440641462,2440641463,107319]},{"type":"way","id":41660593,"nodes":[107316,2440643033,2440643032,25505558,333650017]},{"type":"way","id":48465025,"nodes":[107320,11863145]},{"type":"way","id":48465026,"nodes":[11863145,5612180569,333670409,2985253995,108045]},{"type":"way","id":55687720,"nodes":[699729426,699729408]},{"type":"way","id":59207523,"nodes":[734320238,255576]},{"type":"way","id":59207650,"nodes":[107324,5877163201]},{"type":"way","id":59287906,"nodes":[9521011,4301399905]},{"type":"way","id":66151802,"nodes":[1694551553,25500997,107328,107327]},{"type":"way","id":70039739,"nodes":[9791489,25501007,9791487,4252666501,25703180,2398312023,2398311994,333670411,4249404310,4249404302,25503710]},{"type":"way","id":74662675,"nodes":[881889767,1448901559,881889766]},{"type":"way","id":78201726,"nodes":[735142648,4303454977,4303455889,4303455916,333722490,3524428937,918548280]},{"type":"way","id":84697110,"nodes":[983839058,5612180571,1448894892,107320]},{"type":"way","id":93773946,"nodes":[1089052858,734320238]},{"type":"way","id":93773948,"nodes":[1089052872,21665945]},{"type":"way","id":93773949,"nodes":[21665586,2417298651,9521035]},{"type":"way","id":135765459,"nodes":[25500709,1490441421]},{"type":"way","id":135765461,"nodes":[1490441421,1490441420]},{"type":"way","id":156310006,"nodes":[1685751723,6279571026,1685751700,1685751696,1685751694,5337345854,1685751682,2417297531,2417297532,107322]},{"type":"way","id":156310009,"nodes":[1685751723,1685751711,1685751707,1685751731]},{"type":"way","id":232724478,"nodes":[9521022,4710971306,2425140532,2425141032,108062]},{"type":"way","id":232724479,"nodes":[108062,2425141030,2425140534,2417299495,9521029]},{"type":"way","id":233334612,"nodes":[9521035,2249185407,2417298650,4581297849,4581297850,2425140533,2425141028,108062]},{"type":"way","id":241929531,"nodes":[9791153,9521022]},{"type":"way","id":244215096,"nodes":[2494845032,1906612406]},{"type":"way","id":244493378,"nodes":[25417431,30939235]},{"type":"way","id":244772123,"nodes":[2520452050,2520452052]},{"type":"way","id":244772124,"nodes":[2520452052,380496980]},{"type":"way","id":244963578,"nodes":[2522315148,2522315149]},{"type":"way","id":244963579,"nodes":[2522315149,2522315150,2522315151,82710880,2522315147,82710881]},{"type":"way","id":264965256,"nodes":[14791188,9791484]},{"type":"way","id":264965257,"nodes":[9791484,5440856681,333670400,9791485,11863146]},{"type":"way","id":299682132,"nodes":[25500709,4243601512]},{"type":"way","id":299682133,"nodes":[11863146,4519038134,699729408,5612180570,5811143789,107320]},{"type":"way","id":368200740,"nodes":[3720552874,6315615726]},{"type":"way","id":378924364,"nodes":[3822812081,333722500]},{"type":"way","id":378924366,"nodes":[333722498,3822812081]},{"type":"way","id":378930974,"nodes":[3822896566,271949991]},{"type":"way","id":378930975,"nodes":[3822896567,3822896566]},{"type":"way","id":422540693,"nodes":[4222962219,4222962185]},{"type":"way","id":425013568,"nodes":[4243601491,4243601502]},{"type":"way","id":425013571,"nodes":[4243601502,4243601512]},{"type":"way","id":425013575,"nodes":[1490441420,4243601481]},{"type":"way","id":425013579,"nodes":[4243601481,4243601487,4243601491]},{"type":"way","id":425651109,"nodes":[4249404271,4249404273,4249404278]},{"type":"way","id":425651110,"nodes":[332040899,4249404309,4249404300,4249404295,4249404285]},{"type":"way","id":425936628,"nodes":[4252666500,25703180]},{"type":"way","id":425936629,"nodes":[4252666506,4252666501]},{"type":"way","id":430876285,"nodes":[4301399905,735142648]},{"type":"way","id":430876286,"nodes":[4301399740,4301399731]},{"type":"way","id":430876288,"nodes":[4301399731,4301399715,333719185]},{"type":"way","id":430876299,"nodes":[4301399719,4301399715]},{"type":"way","id":431096159,"nodes":[4303455916,4303455913]},{"type":"way","id":431096160,"nodes":[4303454977,4303454975]},{"type":"way","id":431096161,"nodes":[4303455889,4303454988]},{"type":"way","id":476776097,"nodes":[4702603700,4702603699]},{"type":"way","id":476776098,"nodes":[4702603701,4702603700]},{"type":"way","id":476776099,"nodes":[4702603702,4702603701]},{"type":"way","id":477968031,"nodes":[4712556913,4712556912]},{"type":"way","id":477968032,"nodes":[25502544,4712556913]},{"type":"way","id":477968043,"nodes":[4712556963,4712556962]},{"type":"way","id":477968054,"nodes":[4712556857,4712556847]},{"type":"way","id":477968055,"nodes":[4712556848,4712556854]},{"type":"way","id":477968058,"nodes":[21665937,4712556876]},{"type":"way","id":494310512,"nodes":[14791189,5239570371,2982330114,14791188]},{"type":"way","id":564099715,"nodes":[5434424322,14791185,2982330087,3519133221,25500709]},{"type":"way","id":567293270,"nodes":[5461199401,5461199399]},{"type":"way","id":567293275,"nodes":[5461199404,5461199403]},{"type":"way","id":567293412,"nodes":[5461200038,5461200037]},{"type":"way","id":567293413,"nodes":[5461200040,5461200039]},{"type":"way","id":609004176,"nodes":[5768795953,5768795954,5768795958,5768795651]},{"type":"way","id":609004177,"nodes":[5768795651,5768795955,5768795956,5768795953]},{"type":"way","id":609004178,"nodes":[5768795651,5768795957,5768795652,333670394]},{"type":"way","id":622159392,"nodes":[5239543922,107322]},{"type":"way","id":622159393,"nodes":[5877163201,5877163208,2417297529,5239543922]},{"type":"way","id":622159394,"nodes":[5239543922,5877163202,5877163203,5877163201]},{"type":"way","id":628701125,"nodes":[21665938,21665946,9521035]},{"type":"way","id":648671556,"nodes":[6089552096,6089552097,6089552098]},{"type":"way","id":648671557,"nodes":[6089552098,6089552099]},{"type":"way","id":648671558,"nodes":[1685751707,6089552099]},{"type":"way","id":667745342,"nodes":[4243601512,2982330111,14791188]},{"type":"way","id":669017520,"nodes":[6313625295,6264884982,6313627510,1089052858]},{"type":"way","id":669017521,"nodes":[4712556963,25501016]},{"type":"way","id":669017522,"nodes":[6264884981,6264884980,6313625295]},{"type":"way","id":669019465,"nodes":[25501016,9791153]},{"type":"way","id":669859107,"nodes":[107325,107324]},{"type":"way","id":669859108,"nodes":[1694551558,1694551556,21554349]},{"type":"way","id":670243525,"nodes":[5150471750,333670399]},{"type":"way","id":670336914,"nodes":[107327,107326,107325]},{"type":"way","id":670336915,"nodes":[21554349,1694551553]},{"type":"way","id":670565232,"nodes":[9521006,1716099125,9521029]},{"type":"way","id":674171225,"nodes":[6313624586,6313624585]},{"type":"way","id":674171226,"nodes":[1089052858,6313624585]},{"type":"way","id":675092245,"nodes":[21665026,25530376]},{"type":"way","id":678745735,"nodes":[25472045,333670394]},{"type":"way","id":678745738,"nodes":[25472045,6355365934,6355365936,11863146]},{"type":"way","id":2644237,"nodes":[108028,881889766,1448896332,108030]},{"type":"way","id":4082523,"nodes":[1694551556,21665080]},{"type":"way","id":25016188,"nodes":[271949877,3822896567]},{"type":"way","id":30279886,"nodes":[9521033,2440651558,333722491,496812960]},{"type":"way","id":30279888,"nodes":[333722495,333722494,333722496,333722502,333722497,333722498]},{"type":"way","id":33096423,"nodes":[107824,1938953382,2425138404,447581785,4581297852,4851370381,4581297851,25417426,2417297530,2417297378,107322]},{"type":"way","id":48465024,"nodes":[108033,5440856684,1156353853,3645263634,104302]},{"type":"way","id":74662674,"nodes":[881889762,881889759]},{"type":"way","id":101982008,"nodes":[9791482,5434424322]},{"type":"way","id":102308087,"nodes":[21665080,5419519686,1180910960,21665026]},{"type":"way","id":129148997,"nodes":[107848,764870843,4231049411,107842]},{"type":"way","id":156333052,"nodes":[1685938592,3054882704,1685938594,1685938595,1685938596]},{"type":"way","id":232724480,"nodes":[9521029,1716099103,271949991,9521031]},{"type":"way","id":233334347,"nodes":[25417428,1484205742,25417429,30939234,1795369601,1795369610,25376033,5811148945,9791482]},{"type":"way","id":234239649,"nodes":[9521006,271949877,383974165,1126076484,496813005]},{"type":"way","id":241773308,"nodes":[2833445982,2833445978]},{"type":"way","id":368203568,"nodes":[14791190,5811146372,3720610535]},{"type":"way","id":475957303,"nodes":[2494845029,4696358371]},{"type":"way","id":590298787,"nodes":[333670394,25472057,4222962185,108028,1448890899,104302]},{"type":"way","id":615664503,"nodes":[9521034,25503799,333722501,9521018,1685938592,3054882703,9521006]},{"type":"way","id":675079826,"nodes":[14791174,764870861,107848]},{"type":"way","id":2263971,"nodes":[26630839,4347914532,4347914540,26573667]},{"type":"way","id":2263981,"nodes":[107777,3075334818,4897627183,8916246,3439409103,3439409101,442691373,3854995580,3783539225,345881019,3439409099,130114421,3783505020,442691380,4897627182,3783471662,1538956411,8916243,1394485239,1394485185,8916280,603749189]},{"type":"way","id":4244999,"nodes":[25377942,729620161,6075832879,1531257497,2390011980]},{"type":"way","id":4256531,"nodes":[256794570,25504191]},{"type":"way","id":4256533,"nodes":[25504197,849731781]},{"type":"way","id":4256536,"nodes":[5323722009,5323722127,5323722010]},{"type":"way","id":4256538,"nodes":[25504199,25504201,25504196,5323722153]},{"type":"way","id":4256544,"nodes":[25504259,25504260,1239525728,1239525719,25504261]},{"type":"way","id":4256545,"nodes":[2390008569,1139318714]},{"type":"way","id":4256546,"nodes":[25504263,25504262]},{"type":"way","id":4256796,"nodes":[25504421,4897627186,109633]},{"type":"way","id":4256797,"nodes":[107763,4897627185,3397689899]},{"type":"way","id":4369049,"nodes":[25504190,2726849723,1660481244,2726849728,256794582,2726849729,2726849730,4347925119,632673456]},{"type":"way","id":4369050,"nodes":[25620548,3439409127,3439409126,1188939772,1938580044,1637789404,1188939752,1938580049,1637789413,26573667]},{"type":"way","id":4373934,"nodes":[21665755,3439409138,3439409139,4347917401,26671720,3439409140,3439409141,107774]},{"type":"way","id":4373938,"nodes":[26630839,1637831172,1188939772,3810553689]},{"type":"way","id":4373940,"nodes":[26671724,1944317624,26671730]},{"type":"way","id":4373942,"nodes":[26671730,6064986824]},{"type":"way","id":23715300,"nodes":[2389991902,1539434357,2389991891]},{"type":"way","id":23715304,"nodes":[256794572,2390005218,2494778478,2390005221,256794573,2390005220,256794570]},{"type":"way","id":23715305,"nodes":[256794571,2390005217,256794574,256794572]},{"type":"way","id":23715307,"nodes":[25504191,2726861247]},{"type":"way","id":23715308,"nodes":[256794579,2726861242,4347922046,256794580,1938580032,3439409153,3439409152,26671719]},{"type":"way","id":23715310,"nodes":[256794594,2726849732,1938580037,4766203724,3807436630]},{"type":"way","id":23715312,"nodes":[25504189,2390039152,256794588,256794589]},{"type":"way","id":23715315,"nodes":[256794589,2390039155,2390036511,2390036493]},{"type":"way","id":31117321,"nodes":[8916243,3854995565,345838183]},{"type":"way","id":31119233,"nodes":[26671730,256794576,4919661397,5034506700]},{"type":"way","id":38325246,"nodes":[25504194,1539480548,5323722018,452514368]},{"type":"way","id":38325327,"nodes":[452515765,1465091808,1538897649,452515772]},{"type":"way","id":39093506,"nodes":[256794572,4650621304,25504259,4650621305,2390005223]},{"type":"way","id":43118733,"nodes":[25504189,1239525697,2390039432,2390039431,1239525729,1239525771,107773,3439409150,3439409149,1274040745]},{"type":"way","id":43118734,"nodes":[1274040745,3439409148]},{"type":"way","id":43118737,"nodes":[256794571,3439409151,2390005222,107770,1239525701,256794570]},{"type":"way","id":43118738,"nodes":[25504190,1239525708,2726861270,26671717,3439409147,1239525667,3439409146,3439409143,3439409144,107774]},{"type":"way","id":43118739,"nodes":[107774,3439409145,25504189]},{"type":"way","id":43154948,"nodes":[256794593,256794594]},{"type":"way","id":43157301,"nodes":[25377946,2389983771]},{"type":"way","id":43365536,"nodes":[26671718,256794571]},{"type":"way","id":48049632,"nodes":[345881288,107777]},{"type":"way","id":49796163,"nodes":[26671719,25504190]},{"type":"way","id":58880129,"nodes":[729620161,729620172,729758277,3439409095,729758278,3439409096,1531257534,729758280,729758282]},{"type":"way","id":88334956,"nodes":[5742335191,2620412360,2389991912,1026069088,1026069092,1026069095,1026069103,1026069133,1026069135,1026069138,1026069144]},{"type":"way","id":98477125,"nodes":[2390005223,4650621306]},{"type":"way","id":104080999,"nodes":[1201169871,1201169692]},{"type":"way","id":111971057,"nodes":[3439409148,2726849731,4766203723,3807436630]},{"type":"way","id":111972509,"nodes":[1274053703,1660481466]},{"type":"way","id":139726516,"nodes":[1531257503,1531257488]},{"type":"way","id":139726517,"nodes":[2390011980,1782020321]},{"type":"way","id":139726521,"nodes":[345838183,345838184,729758271]},{"type":"way","id":140469332,"nodes":[1538931290,1538931318,1538931292]},{"type":"way","id":140517898,"nodes":[452514368,4813058486]},{"type":"way","id":145344753,"nodes":[1587616047,4897627188,8916255]},{"type":"way","id":153342407,"nodes":[1660481466,2726852811,2726852805,1660481465]},{"type":"way","id":166759002,"nodes":[2389991902,2389991910,2389991895,2389991904,2389991909,2389991899,25377946]},{"type":"way","id":183479842,"nodes":[1938580020,1938580023,2390008565,1938580027,107763]},{"type":"way","id":195406732,"nodes":[25507047,345863587,25507048,2058516821,5034506822,2058516819]},{"type":"way","id":199201258,"nodes":[1782020321,2390011981]},{"type":"way","id":217573431,"nodes":[107763,4650621307]},{"type":"way","id":230500145,"nodes":[2389980687,1782020517,2389980707,540653733,540653735]},{"type":"way","id":230500146,"nodes":[2389980697,2390011979,1782020321]},{"type":"way","id":230500147,"nodes":[345836394,345836137]},{"type":"way","id":230500556,"nodes":[2389980694,2389983774,2389983772,469772106]},{"type":"way","id":230500560,"nodes":[2389983776,2389983770,2389983773,2389991902]},{"type":"way","id":230500846,"nodes":[2389983771,2389986332,2389986334,4707225202,469772108,2389986336,469772110,2389980691,25507028,2390011981,345836137]},{"type":"way","id":230501412,"nodes":[25377946,5289839775,2389991917,2389991915,2389991907]},{"type":"way","id":230501416,"nodes":[2389991907,5742335191,4650621308,25377895,2389991892,8916255]},{"type":"way","id":230501417,"nodes":[2389991907,2620412365,2389991908,2389991894,2389991901,5289839774,2389991911,2389991913,2389991903,2389983771]},{"type":"way","id":230502961,"nodes":[2390005959,2390008567,452515765,2390008569]},{"type":"way","id":230502962,"nodes":[2389991891,2390005956,1538931290,1938580020,2390005959]},{"type":"way","id":230503366,"nodes":[25504262,25504421,1139318696,25377905,3399223683,107769]},{"type":"way","id":230503367,"nodes":[1139318714,256794566,107763]},{"type":"way","id":230503369,"nodes":[1139318714,5290536464,1139318698,25504262]},{"type":"way","id":230503556,"nodes":[1782020193,1782020519,1782020027,540653726,2389980697]},{"type":"way","id":230503678,"nodes":[345836137,2390011979]},{"type":"way","id":230503680,"nodes":[1782020321,2389980694]},{"type":"way","id":230503683,"nodes":[345836137,2389980687]},{"type":"way","id":230506722,"nodes":[25504191,1239525671,2726861258,1239525734,26671719]},{"type":"way","id":230506725,"nodes":[2390036493,256794592,2390036505,256794593]},{"type":"way","id":230506728,"nodes":[256794594,1274040747,1274040745]},{"type":"way","id":267215482,"nodes":[2726861247,256794581,2726861236,256794578]},{"type":"way","id":267215816,"nodes":[2726866274,2726866276,2726866279,1274040741]},{"type":"way","id":300869140,"nodes":[2726849721,3810553685]},{"type":"way","id":300869141,"nodes":[107769,2538676644,25504199,2726849721]},{"type":"way","id":300869142,"nodes":[2390008569,2390008568,2390005223]},{"type":"way","id":332620705,"nodes":[3397689899,109633]},{"type":"way","id":377346227,"nodes":[3807205741,21665756,3810553687]},{"type":"way","id":377346229,"nodes":[3807205742,4347922045,1239525647,21665755]},{"type":"way","id":377367118,"nodes":[3807436630,2726849726,3810553686]},{"type":"way","id":450558287,"nodes":[4474765621,5323722005,25504193]},{"type":"way","id":451852747,"nodes":[3439409148,107771,2726849727,2726849725,26671718]},{"type":"way","id":480094739,"nodes":[3810553687,25620548]},{"type":"way","id":480094740,"nodes":[3810553686,2538676640,256794584,2726849724,2726849722,26671718]},{"type":"way","id":480094741,"nodes":[3810553685,4766203722]},{"type":"way","id":482974060,"nodes":[469772106,317182560,2389983775,2389983776]},{"type":"way","id":482974061,"nodes":[4650621307,2389991893,2389991898,25377946]},{"type":"way","id":483872216,"nodes":[4766203722,3810553686]},{"type":"way","id":483872217,"nodes":[4650621306,2390008566,2390008564,1139318714]},{"type":"way","id":489017709,"nodes":[4813058486,4813058487]},{"type":"way","id":498958797,"nodes":[632673456,26671724]},{"type":"way","id":498958807,"nodes":[1239525705,5034506699,5034506707,256794575,25507047,345863907]},{"type":"way","id":498958808,"nodes":[26671724,3807205741]},{"type":"way","id":498958812,"nodes":[107777,2058516816,2058516819,853293962,26630792]},{"type":"way","id":498958816,"nodes":[1239525711,3439409137,2549111517,3439409136,2549111514,3439409134,3439409135,21665755]},{"type":"way","id":507776882,"nodes":[3810553689,3807205742]},{"type":"way","id":515540039,"nodes":[256794578,1239525705]},{"type":"way","id":515540040,"nodes":[26630792,5034506843,5521427549]},{"type":"way","id":522735403,"nodes":[5034506700,256794579]},{"type":"way","id":522735404,"nodes":[1239525705,5521427552]},{"type":"way","id":537932902,"nodes":[1531272560,5521427550]},{"type":"way","id":537932905,"nodes":[1531272586,5034506845,1531272560]},{"type":"way","id":537932911,"nodes":[2467911855,5034506844,1531272561]},{"type":"way","id":537932914,"nodes":[1531272561,5034506842,26630792]},{"type":"way","id":551258344,"nodes":[5323721998,5323721997]},{"type":"way","id":551258346,"nodes":[5323722000,5323721999]},{"type":"way","id":551258350,"nodes":[5323722005,5323722004]},{"type":"way","id":551258352,"nodes":[5323722010,25504194]},{"type":"way","id":551258353,"nodes":[25504193,5323722009]},{"type":"way","id":551258355,"nodes":[5323722018,5323722017]},{"type":"way","id":551258356,"nodes":[5323722127,5323722126]},{"type":"way","id":575394076,"nodes":[5521427552,2467911855]},{"type":"way","id":575394078,"nodes":[5521427550,5034506700]},{"type":"way","id":575394079,"nodes":[5521427549,1531272586]},{"type":"way","id":603992626,"nodes":[5323722153,5323721998,5323722000,25504193]},{"type":"way","id":620100136,"nodes":[25620549,26671734,26630790,26630791]},{"type":"way","id":645091879,"nodes":[4319985924,6064939286]},{"type":"way","id":645091887,"nodes":[6064937881,4319985923]},{"type":"way","id":645091891,"nodes":[6064937883,4319985923,5421407179,6337577701,6337577702,6337577703,6337577704,6337577705,6337577700]},{"type":"way","id":645099808,"nodes":[6064986824,6064986825]},{"type":"way","id":645099809,"nodes":[6064986825,26671733,26671734]},{"type":"way","id":676780197,"nodes":[6337577700,6337577699,6337577698,6337577697,6337577706,6337577696,6337577695,4319985924,6064939287]},{"type":"way","id":4254119,"nodes":[25476053,2145880639]},{"type":"way","id":4254121,"nodes":[25476056,1976790118]},{"type":"way","id":4256528,"nodes":[25476051,1537627190]},{"type":"way","id":4338128,"nodes":[1976807775,1976807803]},{"type":"way","id":38325256,"nodes":[452514378,452514386]},{"type":"way","id":45114498,"nodes":[3729004662,4084403495,346361417,4084403499,3729004656,25507025,4084403508,469762514,1936062451]},{"type":"way","id":73812517,"nodes":[873549227,6173148413]},{"type":"way","id":98477127,"nodes":[1139318724,2726738171,1139318692]},{"type":"way","id":98477128,"nodes":[1645813888,1139318726,4709924037,4709924030,1602135997,4709924031,2725915850,1139318692]},{"type":"way","id":140340148,"nodes":[1537623265,1537623283]},{"type":"way","id":140340149,"nodes":[1537623283,4174199995,1936084669]},{"type":"way","id":140340864,"nodes":[1537627184,566201836,2725915880,2725915866]},{"type":"way","id":140340865,"nodes":[1537627190,1537627184]},{"type":"way","id":145344751,"nodes":[873549204,2620430673,2620430674,2725915865,4897627184,2725915874]},{"type":"way","id":147018275,"nodes":[1602135896,1602135897]},{"type":"way","id":147018276,"nodes":[1602135905,1602135955,1602135964,1602135981]},{"type":"way","id":147018282,"nodes":[1602135917,1602135957,1602135985]},{"type":"way","id":147018283,"nodes":[1749442189,1602135995]},{"type":"way","id":151744359,"nodes":[1645813894,25476059]},{"type":"way","id":163195340,"nodes":[1749442191,1749442190,1602135992]},{"type":"way","id":163195341,"nodes":[1602135955,1749442188]},{"type":"way","id":163195342,"nodes":[1602135902,1749442186]},{"type":"way","id":183238431,"nodes":[1936084669,1139318724]},{"type":"way","id":186931866,"nodes":[1976807769,1976807812]},{"type":"way","id":186931868,"nodes":[1976807766,1976807806]},{"type":"way","id":186931878,"nodes":[1976807817,1976807805]},{"type":"way","id":205408321,"nodes":[4174199999,4697525082,2153822338,26398043,720890830]},{"type":"way","id":220019919,"nodes":[1139318692,1139318732,2725915866,2725915874]},{"type":"way","id":267123479,"nodes":[2725915874,2725915878,25504200]},{"type":"way","id":267123480,"nodes":[25504200,2725915896,2725915891,2725915883,2725915880,2725915875,107768,25504187,4174199994,1936084669]},{"type":"way","id":267123485,"nodes":[2725915874,2725915891]},{"type":"way","id":297344119,"nodes":[1602135900,3012176220,1602135898,3012176221,1602135902,3012176223,1602135905,3012176226,1602135909,3012176228,1602135920,3012176230,1602135922,3012176229,1602135921,1602135917,3012176227,1602135907,3012176224,1602135904,3012176222,1602135900]},{"type":"way","id":297344120,"nodes":[1602135981,3012177150,1602135986,1749442189,3012177153,1602135989,3012177155,1602135993,3012177156,1602135992,3012177154,1602135987,1602135985,3012177151,3012177149,1602135982,3012177148,1602135980,3012177146,3012177145,1602135979,3012177147,3012180176,1602135981]},{"type":"way","id":297344121,"nodes":[1602135960,3012177140,1602135957,3012177138,1602135954,1602135953,3012177135,1602135950,3012177134,3012176232,1602135943,3012176231,3012177133,1602135949,1602135951,3012177136,1602135955,3012177139,1602135958,3012177141,1602135962,3012177142,3012177144,1602135963,3012177143,1602135961,1602135960]},{"type":"way","id":332905497,"nodes":[3400241100,3400241138]},{"type":"way","id":406364955,"nodes":[4168504357,5742329951,3729004662]},{"type":"way","id":415935333,"nodes":[3729004662,5742329950,3729004663]},{"type":"way","id":416690552,"nodes":[1139318724,4174199993,4709924035,1645813888]},{"type":"way","id":419359922,"nodes":[1645813888,1645813886,4194431957,4194431956,1139318729,4194431954,3729004662]},{"type":"way","id":595824513,"nodes":[5677722961,5677722962]},{"type":"way","id":595824515,"nodes":[5677722964,5677722963]},{"type":"way","id":595824516,"nodes":[573320955,573320952]},{"type":"way","id":595824518,"nodes":[1602135897,5677722966]},{"type":"way","id":595824520,"nodes":[5677722966,1602135900]},{"type":"way","id":605417126,"nodes":[5742332404,5742332405,5742332403,4709924044]},{"type":"way","id":605417127,"nodes":[5742332404,2494778477,4709924044,3729004662]},{"type":"way","id":659336493,"nodes":[826724024,6173148414]},{"type":"way","id":659336495,"nodes":[6173148415,6173148417]},{"type":"way","id":659336497,"nodes":[6173148419,6173148418]},{"type":"way","id":659336499,"nodes":[720890825,1694551537]},{"type":"way","id":4082588,"nodes":[109620,3092290905,3092290904,3092290906,1239525639,1239525674,1239525686,4347917404,1239525711]},{"type":"way","id":4236991,"nodes":[21990641,6283479888,1237514475,21704021,25470464,21704017]},{"type":"way","id":4253153,"nodes":[455705618,21990639,1947276623,21704021]},{"type":"way","id":4253397,"nodes":[351788581,2441993349,2441993356,21704017,2441993346]},{"type":"way","id":4256534,"nodes":[25504194,25504195,452514386,2725915900,2725915899,2725915898,2725915897,25504200]},{"type":"way","id":4258925,"nodes":[21554363,25529877,1104448381,501766800]},{"type":"way","id":4258962,"nodes":[21665135,25529874,25529875,25529876,25529877,25529933]},{"type":"way","id":4258963,"nodes":[108065,25529933,25529938,490256481]},{"type":"way","id":4260266,"nodes":[25544127,5444031562,2476805774,2476805770,25544126,5444031577,2476805766,5444031619,25533330,25544132,25533328,5859926240,25544134,25533435,25533436,25533438]},{"type":"way","id":4260277,"nodes":[1694551549,4380848928,5454295759,5454295758]},{"type":"way","id":4276391,"nodes":[1616122178,5404583125,2441998874,5404582919,5404583198,5404582918,5404583196,5404583195,5404583123,5404583187,5404583131,5404582018,13355224]},{"type":"way","id":4276392,"nodes":[10574891,5454295758,2441998875,5454295779,25736719,1857957494,1678452823]},{"type":"way","id":4337023,"nodes":[26398235,1646043083,107728]},{"type":"way","id":4360146,"nodes":[26573667,26573672]},{"type":"way","id":4615506,"nodes":[9475518,3054882695,1614926341,21990641,281454961]},{"type":"way","id":17933743,"nodes":[351788581,4967033603,9475515,1614926358,25470680]},{"type":"way","id":25712484,"nodes":[3895360879,9475243,282569749]},{"type":"way","id":25717522,"nodes":[280412704,280412705,4350239459,4350239462,1616162948,281454943,1237514480,1697727377,1947257518,280412702]},{"type":"way","id":25803163,"nodes":[281454959,4068974161,1685903442,2441990081,1528450579,455705609,281454958,281454961,1237514492,1237514473,455705618]},{"type":"way","id":25904236,"nodes":[280412704,4350239460,1106056886]},{"type":"way","id":28355074,"nodes":[108899,256339270,25497910,255747733,255747713,4013524085,4013524086,4927544939]},{"type":"way","id":31120027,"nodes":[345886002,25476020,5153538727,25504198,5677722958,25476053,720890829,5677722957,6173148416,873549223,826724213,1645813917,25476051]},{"type":"way","id":34437380,"nodes":[282569738,282569739,25378117,1685903452,1538881758,282569740,1734545131,1685903437,1538881759,282569741,1685903456,282569746,282569747,282569749,282569742,4350239461,282569744,1106056886]},{"type":"way","id":37458969,"nodes":[438170616,438170606]},{"type":"way","id":38325244,"nodes":[1278032912,452514321,452514328,25504194]},{"type":"way","id":43892838,"nodes":[281454959,1685903446,396093938,3140855372]},{"type":"way","id":60368359,"nodes":[751735355,751735130,751735365,751735296,3764054181,3764054180]},{"type":"way","id":72140279,"nodes":[9515387,109624,1947257511,25476047,2338578962,25476049,2975783589]},{"type":"way","id":145344750,"nodes":[109634,5341902060,1587616057,1587616049,1587616056,1587616047]},{"type":"way","id":148377820,"nodes":[1614846616,1614846675]},{"type":"way","id":148377822,"nodes":[1614846683,1614846675,5387300680,10574749,25544118,25544119]},{"type":"way","id":148386199,"nodes":[1614926348,5404152924,1614926342,2444571509,1614926340]},{"type":"way","id":148386200,"nodes":[1614926342,1614926358]},{"type":"way","id":148519535,"nodes":[1616122178,5404152138,1616122145]},{"type":"way","id":148519545,"nodes":[1616122145,5404153140,2450730396,371564817,1843386487]},{"type":"way","id":148519546,"nodes":[1616122145,5404151677,1614926348]},{"type":"way","id":151793293,"nodes":[2338578977,1646306223,1646306219,1646306211,1646306182,1646306180,1646306174,1646306171,1646306166,1646306138,1646306132,1646306128,1646306124,1646306117,1646306110]},{"type":"way","id":200596450,"nodes":[2105659653,1106056876,1947257513,1614846605,1106056861,1106056847,3189218645,1106056866,1155428970,1106056855,469780726,4059394511,1947276620,1106056858,4350208170,1106056846,1106056878,1106056863]},{"type":"way","id":200596931,"nodes":[1881001588,1938953360,1953686428]},{"type":"way","id":208169476,"nodes":[2184963115,751735268,751735187,751735218]},{"type":"way","id":217242019,"nodes":[9513010,5420953856,1616122139]},{"type":"way","id":217573432,"nodes":[25504200,107769]},{"type":"way","id":220019917,"nodes":[8916255,1602135896,1824589995,5742332404]},{"type":"way","id":225025827,"nodes":[248122264,1938953359,1938953358,1630605982]},{"type":"way","id":230502391,"nodes":[109633,109634,4897627187,873549204]},{"type":"way","id":230506730,"nodes":[2390036493,1697691548,3397689896,3397689897]},{"type":"way","id":236140104,"nodes":[108899,3140855390,396093936,2441990082]},{"type":"way","id":237462806,"nodes":[1678452778,1938580066,1843385996,25544123,9512908]},{"type":"way","id":249958677,"nodes":[2564873525,4535154580,25476045,345886002]},{"type":"way","id":251325041,"nodes":[3895422567,3895412044,108899]},{"type":"way","id":267213247,"nodes":[256794589,2390039153,2390039151,2390039150,256794590,2390036502,2390036497,1239525752,3810553690]},{"type":"way","id":288506479,"nodes":[2476787985,5932959892,1938580076,5932960717,1938580079,25544131,5444033150,25533321,25544138,25533319,25544135,2512940572,351793985,9513010,371564784,25533440,452486619,1678452810,3085005035,452486132]},{"type":"way","id":424889275,"nodes":[2140823631,4174200028,1237543425,4697525059,4174200027,4174200025,4174200022,4174200021,4174365779,4174200018,4174200013,4174200009,4174200005,4174200004,4174200003,3876825295,3876825296,4174200000,4174199999,566201846,1537623271,1537623265]},{"type":"way","id":438301430,"nodes":[1616122237,5420170453,5420170454,438170606,1694551549]},{"type":"way","id":438301431,"nodes":[1616122237,1616122219,1616122196,1616122178]},{"type":"way","id":480094743,"nodes":[3810553691,2726836316,2726836315,1239525762,2390036493]},{"type":"way","id":489017706,"nodes":[4813058487,452514407,452514378]},{"type":"way","id":498185199,"nodes":[3397689895,107780,2390036481,2390036490,2390036495,256794593]},{"type":"way","id":530799878,"nodes":[5153538730,5153538728]},{"type":"way","id":595824517,"nodes":[5677722965,573320949]},{"type":"way","id":603198589,"nodes":[26573667,3810553688,9512922]},{"type":"way","id":628429131,"nodes":[5932899700,25530236,5364578866,25530237,21665081]},{"type":"way","id":645091883,"nodes":[6064939288,6064939287,6064937879,6064939286]},{"type":"way","id":645091889,"nodes":[6064937878,6064937883,6064937882]},{"type":"way","id":645091895,"nodes":[6064937880,6337577694,6337577700,6173728484]},{"type":"way","id":647037417,"nodes":[25544119,1614846607,370711678,1614846610,1614846616]},{"type":"way","id":647062692,"nodes":[1103212861,5059333678,2456267967,1103212860,5364578864,108065,1700227411,21554363,108067,4676738223,10574891,5811263957,5811263955]},{"type":"way","id":659410652,"nodes":[6173783031,5859926245,568842573,25533605]},{"type":"way","id":2198814,"nodes":[2726834609,2726838583,2726836317,2726836318,2726836319,2726836320,107783,2726836322]},{"type":"way","id":2263970,"nodes":[109620,1637789493,1637789500,6064939290,109621,6173732724,6064939289,1637789537,1637789538,2726836321]},{"type":"way","id":4082552,"nodes":[21665581,21665367]},{"type":"way","id":4082554,"nodes":[21665583,3552338824,3552338823,3552338821,5597434617,21665372,5597434619]},{"type":"way","id":4082555,"nodes":[107795,2482696992,21665583,3552338822,107803]},{"type":"way","id":4082556,"nodes":[21665585,107795]},{"type":"way","id":4082557,"nodes":[107795,2482696993,21665382,21665384,21665584]},{"type":"way","id":4082567,"nodes":[21665031,2356142513,2356142512,21665110,5932883504]},{"type":"way","id":4082592,"nodes":[21665911,21665745,21665912]},{"type":"way","id":4082596,"nodes":[107790,26573152,1102995757,5364578861,26573136,21665912,26582693]},{"type":"way","id":4236997,"nodes":[1678452814,438170684,5811231652,25544143]},{"type":"way","id":4258964,"nodes":[25530236,25529938]},{"type":"way","id":4260254,"nodes":[25544113,25533580,821341516]},{"type":"way","id":4260256,"nodes":[25533578,25544114]},{"type":"way","id":4260259,"nodes":[25544119,25544120]},{"type":"way","id":4260265,"nodes":[5932959891,3085004827,3085004825,1533393315]},{"type":"way","id":4260269,"nodes":[25544131,2064602918]},{"type":"way","id":4260278,"nodes":[25544136,4784457212,4784457210,25544135]},{"type":"way","id":4260281,"nodes":[5932959891,5932959903,3085004826,3085004821,25544142,1102995758,107790]},{"type":"way","id":4260282,"nodes":[107791,5451506149,3085004830,25544141,5932959895,2476787985]},{"type":"way","id":4260283,"nodes":[9513062,25496899]},{"type":"way","id":4260284,"nodes":[1678452807,2265298898,438170687,3192197694,9513062]},{"type":"way","id":4260285,"nodes":[25544143,5811231853,5811231856,2265298896,25533506,5452395150,5452395149,9513062]},{"type":"way","id":4260286,"nodes":[25544138,5444033229,1089089322,4784457213,21554364]},{"type":"way","id":4260287,"nodes":[1678452813,438170643,2265306459]},{"type":"way","id":4276376,"nodes":[25736633,3782162175]},{"type":"way","id":4356104,"nodes":[9512922,1188939747,26559655]},{"type":"way","id":4356148,"nodes":[9512917,21665712]},{"type":"way","id":4358414,"nodes":[107790,26631965,21665584,21665911]},{"type":"way","id":4358416,"nodes":[21665724,21665909]},{"type":"way","id":4358455,"nodes":[26559655,21665775,1637831255,6173732694]},{"type":"way","id":4358456,"nodes":[21665713,639259210,21665714]},{"type":"way","id":4358457,"nodes":[21665716,21665717,21665719]},{"type":"way","id":4360035,"nodes":[21665714,4853048350,639259211,1637789617,21665859,1637789621,21665863]},{"type":"way","id":4360036,"nodes":[21665863,21665719]},{"type":"way","id":4360037,"nodes":[21665719,5444235310,5444235311,10574748]},{"type":"way","id":4360038,"nodes":[26573136,26573143,2463446566,1678452754]},{"type":"way","id":4360039,"nodes":[26573152,6302549987,26573147,26573143]},{"type":"way","id":4361223,"nodes":[26582693,26582694]},{"type":"way","id":4373994,"nodes":[26573673,26573672]},{"type":"way","id":4376566,"nodes":[1938580083,3054882694,1119783271,2837565785,21665031]},{"type":"way","id":4382813,"nodes":[107790,1102995756,5479634639,5364578862,21665585,5479634644,107791]},{"type":"way","id":4382814,"nodes":[107791,5597434619,401529120,5364578865,3552338869,25496892]},{"type":"way","id":4393280,"nodes":[1678452821,2450755716]},{"type":"way","id":4718200,"nodes":[21665911,21665910]},{"type":"way","id":4992945,"nodes":[1103212861,3552453346,1938580082,488815519,26846357]},{"type":"way","id":4992946,"nodes":[25496892,3552452839,1953893713,5398463417,5398463419,26846357]},{"type":"way","id":5003469,"nodes":[107792,5695662022,1102996992,5059333680,5398465124,21665580]},{"type":"way","id":23715311,"nodes":[3397689898,4350193652,346430574,4350193655,6315809114,1278032912,3397689895]},{"type":"way","id":31467248,"nodes":[2564873531,2321071820,25544120]},{"type":"way","id":32984563,"nodes":[371564809,371564813]},{"type":"way","id":37458967,"nodes":[371564817,438170612,438170616]},{"type":"way","id":38325242,"nodes":[452514295,452514296]},{"type":"way","id":40412562,"nodes":[26846357,1953893711,1953893710,488815520,452488948,25496899]},{"type":"way","id":48616855,"nodes":[107803,617438710,26846355]},{"type":"way","id":49796159,"nodes":[6173732694,21665714]},{"type":"way","id":49796160,"nodes":[339925031,3092290908,3092290907,1239525737,3092290902,3092290901,3092290903,109620]},{"type":"way","id":49796164,"nodes":[21665712,370712150]},{"type":"way","id":49796165,"nodes":[21665713,21665712]},{"type":"way","id":49796168,"nodes":[370712150,21665724]},{"type":"way","id":49797402,"nodes":[1678452766,1678452758]},{"type":"way","id":49797408,"nodes":[10574749,1156355062,5386345349,5386345350,5386341215,25544114,25544113,6173783036,452487859,6173783033]},{"type":"way","id":49797412,"nodes":[1678452751,26582694]},{"type":"way","id":49797415,"nodes":[26582694,348894359,1729508660,25533578,25544116,25544115]},{"type":"way","id":49798167,"nodes":[107786,1637789635,5364578859,10574748]},{"type":"way","id":49798170,"nodes":[21665719,21665720,1637789636,107786]},{"type":"way","id":49798172,"nodes":[370712148,21665716]},{"type":"way","id":49798175,"nodes":[21665716,21665713]},{"type":"way","id":49798450,"nodes":[26582693,5364578860,1947257515,21665913,1102994231,107786]},{"type":"way","id":93776538,"nodes":[1089089333,1089089322]},{"type":"way","id":93776541,"nodes":[4784457204,1089089333]},{"type":"way","id":93776546,"nodes":[9512940,2048921782]},{"type":"way","id":95094733,"nodes":[107792,1103212841,2767582810,1103212861]},{"type":"way","id":95100742,"nodes":[107805,5877184122,401501661,2767561431,2767571255,1102996994]},{"type":"way","id":112448576,"nodes":[3326339636,1645919673,1274040743]},{"type":"way","id":112448577,"nodes":[1278032910,1274040746]},{"type":"way","id":150915471,"nodes":[1637789501,5481389678]},{"type":"way","id":150915477,"nodes":[1637789617,1637789604,1637789602,1637789599,1637789598,1835289526]},{"type":"way","id":155450993,"nodes":[1678452778,1678452776,6173783039]},{"type":"way","id":155450996,"nodes":[5932959891,5932959890,3085004818]},{"type":"way","id":155450998,"nodes":[1678452823,2265298885,2265298881,1678452820,2265298888,2265298887,2265298890,1678452813,1678452811,2265298886,1678452807,2265298879,2265298900,1678452814,2265298883,1678452821,2265298892,2265298894,1678452823]},{"type":"way","id":155451004,"nodes":[1678452742,1938580054,5444227620,1678452739,1678452740,1947257514,1678452741,1637789608]},{"type":"way","id":155451005,"nodes":[1678452745,1678452744,1678452747,1678452749,1678452743,25544115,1678452742]},{"type":"way","id":155451007,"nodes":[1678452742,1104386138,25544119]},{"type":"way","id":194374616,"nodes":[2048921785,4784457203,4784457202,4784457201,4784457204]},{"type":"way","id":194374619,"nodes":[4784457202,4784457197,2048921781,25533321]},{"type":"way","id":196087786,"nodes":[2064602915,1857917764,5444031596,25544126]},{"type":"way","id":196087789,"nodes":[2064602918,2064602915]},{"type":"way","id":209060806,"nodes":[25522907,1616122139]},{"type":"way","id":209060807,"nodes":[25522907,1694551549]},{"type":"way","id":215065133,"nodes":[21554364,2450755718]},{"type":"way","id":217241358,"nodes":[1694551549,5454295817,438170636,1857940093,1678452820]},{"type":"way","id":217242011,"nodes":[2265306462,25736633,5420953859]},{"type":"way","id":217242014,"nodes":[2265306459,25522907]},{"type":"way","id":217242015,"nodes":[1616122139,2265306462]},{"type":"way","id":217242377,"nodes":[21554364,2450755717]},{"type":"way","id":237084255,"nodes":[21665118,21665580]},{"type":"way","id":237127652,"nodes":[1843386487,1616122139]},{"type":"way","id":237130353,"nodes":[2450755716,21554363]},{"type":"way","id":237130354,"nodes":[2450755717,1953893718,3192197695,1953893715,25533292,452488960,452488961,25496899]},{"type":"way","id":237130356,"nodes":[2450755719,21554364]},{"type":"way","id":237130357,"nodes":[2450755718,1678452811]},{"type":"way","id":237462805,"nodes":[1678452778,6173783031]},{"type":"way","id":237493874,"nodes":[21665367,107803]},{"type":"way","id":237493875,"nodes":[21665581,5364578867,107805]},{"type":"way","id":237493876,"nodes":[107806,5877184121,21665581]},{"type":"way","id":239318974,"nodes":[1106056856,3326326774,3326326775,1497135187,2338578968,2338578984,2105659653]},{"type":"way","id":264966078,"nodes":[1094124402,5932899702,21645238,5932899704,5932899700]},{"type":"way","id":264966079,"nodes":[9512939,1938580083]},{"type":"way","id":264966080,"nodes":[9512939,5932883504]},{"type":"way","id":267213248,"nodes":[1938580050,2726834645,2726834637,2726838589,2726836322]},{"type":"way","id":267213403,"nodes":[2726836321,2726834630,2726834636,2726838592,2726834638,2726834644,2726834646,1938580050]},{"type":"way","id":267213404,"nodes":[2726836322,2726834628,2726838586,2726834616,2726834608,3810553691]},{"type":"way","id":267213537,"nodes":[2726834609,2726838582,2726834619,2726834627,2726834629,2726836321]},{"type":"way","id":267217552,"nodes":[2726883566,2726883568]},{"type":"way","id":267217553,"nodes":[2726883572,2726883574]},{"type":"way","id":267217554,"nodes":[2726883585,2726883578]},{"type":"way","id":267217555,"nodes":[2726883549,2726883553]},{"type":"way","id":271816292,"nodes":[1102996994,107792]},{"type":"way","id":271816293,"nodes":[21665580,5364578863,1094124402]},{"type":"way","id":288506485,"nodes":[2476787985,5932959894,5932959888,5932959891]},{"type":"way","id":294010665,"nodes":[2975783589,2975783590,346430572,1497135176,566201807,1497135188,1645919670,346430573,1497135179,4350193657,4350193653,346430574]},{"type":"way","id":303982846,"nodes":[1533393315,5451506147,25544127,25544130,1678452778]},{"type":"way","id":303982852,"nodes":[3085004818,1938580069,3887905014,1678452781,1678452758,1678452754,1678452752,1678452751]},{"type":"way","id":303982854,"nodes":[25496899,9512940,5451506148,1938580081,5932959902,2476787985]},{"type":"way","id":305976269,"nodes":[25544143,5811221631,2456267967]},{"type":"way","id":313310307,"nodes":[3192197695,3192197696]},{"type":"way","id":318825723,"nodes":[26559655,566202757,2345097899,2345097921,2345097938,2345097948,6311518054,6214704100]},{"type":"way","id":326018187,"nodes":[3326326775,2975783589]},{"type":"way","id":326019004,"nodes":[1278032910,826724221]},{"type":"way","id":332620704,"nodes":[3397689897,3326326761,4350193651,1497135177,2612448628,1106056856]},{"type":"way","id":385444407,"nodes":[3887905014,3887905013,3887905012,3887905011]},{"type":"way","id":431893905,"nodes":[2975783589,3397689898]},{"type":"way","id":437200581,"nodes":[346430574,1497135177]},{"type":"way","id":437201225,"nodes":[2726836322,1102994236,1239525687]},{"type":"way","id":480094742,"nodes":[3810553690,2726834609]},{"type":"way","id":485671960,"nodes":[2048921782,2048921785]},{"type":"way","id":485671961,"nodes":[4784457212,4784457211,4784457207]},{"type":"way","id":485671962,"nodes":[4784457210,4784457209,4784457208,4784457207,4784457206,4784457205,4784457214,4784457213]},{"type":"way","id":490512547,"nodes":[25544126,5444031578,5444031598,5444031599,1857917757,4826960088,4826960087,4826960089,25544128,5444031569,25544130]},{"type":"way","id":497441317,"nodes":[1239525687,3326326769,2612448640,351509979,1947257506,1106056856]},{"type":"way","id":498958810,"nodes":[9512922,4347914539,1239525704,339925031]},{"type":"way","id":498958813,"nodes":[6214704108,6214704105]},{"type":"way","id":558503735,"nodes":[5386341216,5386341215]},{"type":"way","id":559142261,"nodes":[5393414954,5393414955]},{"type":"way","id":559766159,"nodes":[1953893710,5398465121,5398465123,5398463420,5398463416,5398463418,5398465122,5398463417]},{"type":"way","id":559766160,"nodes":[21665031,21665118]},{"type":"way","id":565171625,"nodes":[5444033222,5444033231]},{"type":"way","id":565171626,"nodes":[5444033221,5444033222]},{"type":"way","id":565171629,"nodes":[5444033231,5444033120,5444033230,5444033229]},{"type":"way","id":565199557,"nodes":[25544115,5444227721,5444227620]},{"type":"way","id":569860515,"nodes":[5479635835,5479635834]},{"type":"way","id":569860516,"nodes":[5479635836,5479635487]},{"type":"way","id":569860517,"nodes":[5479635837,5479635489]},{"type":"way","id":569860522,"nodes":[5479634675,5479634674]},{"type":"way","id":569860523,"nodes":[5479634692,5479634683]},{"type":"way","id":569860525,"nodes":[5479634695,5479634694]},{"type":"way","id":569860526,"nodes":[5479634696,5479634704]},{"type":"way","id":569860527,"nodes":[5479634701,5479634697]},{"type":"way","id":569860535,"nodes":[5479635426,5479635425]},{"type":"way","id":569860553,"nodes":[5479635444,5479635443]},{"type":"way","id":569860556,"nodes":[5479635446,5479635445]},{"type":"way","id":569860560,"nodes":[5479635448,5479635447]},{"type":"way","id":569860599,"nodes":[5479635550,5479635556]},{"type":"way","id":569860601,"nodes":[5479635552,5479635551]},{"type":"way","id":569860602,"nodes":[5479635553,5479635554]},{"type":"way","id":569860612,"nodes":[5479635578,5479635577]},{"type":"way","id":569860614,"nodes":[5479635580,5479635579]},{"type":"way","id":569860616,"nodes":[5479635599,5479635598]},{"type":"way","id":569860617,"nodes":[5479635601,5479635600]},{"type":"way","id":569860620,"nodes":[5479635611,5479635610]},{"type":"way","id":569860621,"nodes":[5479635613,5479635612]},{"type":"way","id":569860628,"nodes":[5479635627,5479635626]},{"type":"way","id":569860629,"nodes":[5479635630,5479635629]},{"type":"way","id":569860630,"nodes":[5479635632,5479635631]},{"type":"way","id":569860658,"nodes":[5479635681,5479635683]},{"type":"way","id":569860659,"nodes":[5479635682,5479635685]},{"type":"way","id":569860664,"nodes":[5479635692,5479635691]},{"type":"way","id":569860665,"nodes":[5479635694,5479635693]},{"type":"way","id":569860670,"nodes":[5479635712,5479635711]},{"type":"way","id":569860671,"nodes":[5479635714,5479635713]},{"type":"way","id":569860679,"nodes":[5479635763,5479635762]},{"type":"way","id":569860680,"nodes":[5479635765,5479635764]},{"type":"way","id":569860681,"nodes":[5479635767,5479635766]},{"type":"way","id":569860682,"nodes":[5479635769,5479635768]},{"type":"way","id":585785134,"nodes":[1938580083,692191040,21645238]},{"type":"way","id":585785135,"nodes":[25496892,2767582815,1102996997,107792]},{"type":"way","id":615662723,"nodes":[5932883504,107805]},{"type":"way","id":624848791,"nodes":[5420953859,25544136,2450755719]},{"type":"way","id":628422480,"nodes":[5932856691,5932883505]},{"type":"way","id":628426436,"nodes":[5932883504,5932883501,5932883506,5932883505]},{"type":"way","id":628426437,"nodes":[5932883505,5932883503,5932883502,5932883504]},{"type":"way","id":628429129,"nodes":[5932899700,5932899701,5932899705,5932899703,1094124402]},{"type":"way","id":628429130,"nodes":[5932899705,401529147]},{"type":"way","id":645091880,"nodes":[6064937882,6064939291,6064937880,6064939888,6064939288]},{"type":"way","id":645091893,"nodes":[6064939289,6064937884]},{"type":"way","id":645091894,"nodes":[6064939290,6064939285]},{"type":"way","id":647037414,"nodes":[1637789608,1947257512,4350184736,4350184739,1102994241,2321071816,1947257509,2726834647,1938580050]},{"type":"way","id":647037415,"nodes":[1678452745,107785,1637789608]},{"type":"way","id":647037416,"nodes":[10574748,1637789629,1102994238,1678452745]},{"type":"way","id":659405416,"nodes":[6173732688,6173732723]},{"type":"way","id":659405422,"nodes":[6173732719,6173732720]},{"type":"way","id":659405423,"nodes":[6173732697,6173732727,6173732689]},{"type":"way","id":659405429,"nodes":[6173732717,6173732729]},{"type":"way","id":659406290,"nodes":[6214704100,21665714]},{"type":"way","id":659407958,"nodes":[6173754809,348814978]},{"type":"way","id":659410650,"nodes":[6173783034,6173783040]},{"type":"way","id":659410651,"nodes":[6173783036,6173783034]},{"type":"way","id":659410653,"nodes":[6173783033,1678452757,1678452766]},{"type":"way","id":659410654,"nodes":[6173783039,1678452769,1678452766]},{"type":"way","id":663969749,"nodes":[6214704105,6214704103,6214704106,6214704112]},{"type":"way","id":663969754,"nodes":[6214704106,6214704109]},{"type":"way","id":663969760,"nodes":[6214704112,6214704100]},{"type":"way","id":663969766,"nodes":[6214704110,6214704103]},{"type":"way","id":673567884,"nodes":[21665910,370712148]},{"type":"way","id":673956057,"nodes":[6311518055,6173740395]},{"type":"way","id":675079681,"nodes":[21665909,9512921,5393414955,1835298184,21665910]},{"type":"way","id":2190803,"nodes":[107710,4350220820,1106056874,1685903440,4350239468,6283521240,1685903441,282569745,4079423543,108895]},{"type":"way","id":2198816,"nodes":[107716,25474658,25474661,4174200026]},{"type":"way","id":2424941,"nodes":[25471089,2482696991,5012183938,2453519185]},{"type":"way","id":4236979,"nodes":[108896,1614926320,25471371]},{"type":"way","id":4253267,"nodes":[1106056863,25471100]},{"type":"way","id":4253347,"nodes":[25471711,2159454167,1678452805,931888889,2444562449,25471729]},{"type":"way","id":4253930,"nodes":[25474658,1819091297,25474657]},{"type":"way","id":4253931,"nodes":[25474676,1104386168,1104386162]},{"type":"way","id":4253933,"nodes":[1104386168,6161772316]},{"type":"way","id":4253951,"nodes":[25474662,6291190465,25474664]},{"type":"way","id":4253952,"nodes":[25476040,1174970393,25474753,826724298,25474751,6307073859,25474750]},{"type":"way","id":4253953,"nodes":[25474661,2126997658,25474662,2126997628,2126997633,2126997613,2126997649,1104386144,25474676,25474750]},{"type":"way","id":4253954,"nodes":[9515368,4350208168,107715]},{"type":"way","id":4253955,"nodes":[9515379,25474666]},{"type":"way","id":4254100,"nodes":[1104386133,1164840729]},{"type":"way","id":4254101,"nodes":[9515377,1104386151,25476001]},{"type":"way","id":4254105,"nodes":[25476001,827333150]},{"type":"way","id":4254106,"nodes":[25476004,2975783588]},{"type":"way","id":4254108,"nodes":[9515377,2975783588]},{"type":"way","id":4254109,"nodes":[25476022,25476025,25476026]},{"type":"way","id":4254110,"nodes":[9515388,25476026]},{"type":"way","id":4254112,"nodes":[25476037,1163257580]},{"type":"way","id":4254113,"nodes":[25476014,5325427437,25476037,25476036]},{"type":"way","id":4260275,"nodes":[25533438,25533439,25533437]},{"type":"way","id":4277058,"nodes":[1614846692,9515370]},{"type":"way","id":4369165,"nodes":[4457798822,5011796271,5011796259,568842591,5011796257,3811129601,5011796252,568842587,5011556356,5011796256,6173783042,5011388934]},{"type":"way","id":4451477,"nodes":[27345499,6282772554,4349663551]},{"type":"way","id":4615508,"nodes":[25471747,4686140572]},{"type":"way","id":4615509,"nodes":[25471712,2159454166,25471711,25471710,4637615417,25471708]},{"type":"way","id":17933745,"nodes":[25471371,1375625379,9475518]},{"type":"way","id":17933748,"nodes":[109836,2069620510,2476805753,351771449]},{"type":"way","id":17933751,"nodes":[351771449,1614846711,25471098]},{"type":"way","id":23013973,"nodes":[751735372,484375367,248122264]},{"type":"way","id":24511395,"nodes":[1106056866,109838]},{"type":"way","id":25717521,"nodes":[280412702,1697727370,281454941,2105667816]},{"type":"way","id":25720713,"nodes":[107710,2450397874,2521654793,2450397879]},{"type":"way","id":25904237,"nodes":[282569744,2612451170,1938953361,4350239464,1685903445,455705605,455705606,282569745]},{"type":"way","id":25904240,"nodes":[9475243,282569797]},{"type":"way","id":25904241,"nodes":[282569797,6329942070]},{"type":"way","id":31142270,"nodes":[346343907,25476042]},{"type":"way","id":31476268,"nodes":[351771449,3054882691,10574746]},{"type":"way","id":31476941,"nodes":[351788581,1614926340,25471747,1375634504,4349663550,3085005038,452486137]},{"type":"way","id":31477085,"nodes":[351793985,5479717489,25533438]},{"type":"way","id":32728727,"nodes":[368279307,3054882690,25471098]},{"type":"way","id":32728729,"nodes":[108895,368279307,4079423542]},{"type":"way","id":38069220,"nodes":[6111301384,447764957]},{"type":"way","id":38325238,"nodes":[452514269,25476041]},{"type":"way","id":38325240,"nodes":[452514269,5325427460,452514276]},{"type":"way","id":49797427,"nodes":[25471708,25533437,3054882693,5934126091,1678452794,2476805758,109836]},{"type":"way","id":49797434,"nodes":[25471089,5011556345,5011796263,5011796261,1614846708]},{"type":"way","id":49797435,"nodes":[25533605,5011388934,6173783037,5012149906,5011388929,25533613,5012149901,25533621,5011796269,751735403,5011556351,25471089]},{"type":"way","id":49797441,"nodes":[25471712,2502903588,351788581]},{"type":"way","id":49797442,"nodes":[25471086,4079423544,2453519182]},{"type":"way","id":60368363,"nodes":[751735403,5012149914]},{"type":"way","id":68807975,"nodes":[826724298,826724148,826724242]},{"type":"way","id":68807986,"nodes":[826724206,1104386126,826724026]},{"type":"way","id":68808009,"nodes":[5325427429,5325427428]},{"type":"way","id":68883404,"nodes":[827333150,827333158]},{"type":"way","id":68883405,"nodes":[109838,9515388,4059394510,9515387]},{"type":"way","id":68883406,"nodes":[827333160,827333154]},{"type":"way","id":68883407,"nodes":[827333158,827333160]},{"type":"way","id":68883408,"nodes":[827333154,25476004]},{"type":"way","id":71507464,"nodes":[25476020,25476017,452514269]},{"type":"way","id":77771481,"nodes":[1614926285,2450397879]},{"type":"way","id":83247596,"nodes":[25533438,5397216224,5012183933,5397216223,25533613]},{"type":"way","id":94576669,"nodes":[25471086,4967033606,4967033601,25471729,4967033604,9475518]},{"type":"way","id":100077167,"nodes":[4457798823,5012151525,4457798822,568842560,9512908]},{"type":"way","id":100619426,"nodes":[1163257579,25476039,6111301383]},{"type":"way","id":100619427,"nodes":[6111301383,25474753]},{"type":"way","id":100619431,"nodes":[1163257580,1163257577,1163257578,1163257575,25476039]},{"type":"way","id":109393880,"nodes":[266559134,5391163030,5391163022,9515370]},{"type":"way","id":109393881,"nodes":[1614846678,5391163021,1614846684,2338587824]},{"type":"way","id":139420244,"nodes":[452486137,475966573,25471708]},{"type":"way","id":148377821,"nodes":[1614846683,1614846678]},{"type":"way","id":148377825,"nodes":[1614846711,6282772557,1614846702]},{"type":"way","id":148386201,"nodes":[1614926341,1614926339,1614926338,1614926323,1614926322,6283479982,1614926320]},{"type":"way","id":148386209,"nodes":[1614926339,1614926337,1614926336,1614926334,1614926333]},{"type":"way","id":148519536,"nodes":[1616122145,452486520,1953893720,452486523,452486132]},{"type":"way","id":154986825,"nodes":[1106056850,2184965485,1106056883]},{"type":"way","id":155450989,"nodes":[452486132,3085005036,2450730397,2444571508,3085005037,452486137]},{"type":"way","id":155450991,"nodes":[1678452824,2444565403,1678452806,1678452805]},{"type":"way","id":183059896,"nodes":[1934197347,2612448636,2612448638,2663048865]},{"type":"way","id":195542899,"nodes":[1106056886,4350239463,1947257536,4350239467,1697727378,1237514459,1237514461,2663048864]},{"type":"way","id":199200283,"nodes":[25474666,2092250094,25474751]},{"type":"way","id":200596105,"nodes":[4059394509,9515394,4350208171,107715,1947276617,9515389,857556897,109838]},{"type":"way","id":200596451,"nodes":[1106056863,1947257517,1934197347]},{"type":"way","id":200596544,"nodes":[1106056850,2663048865,2184966600]},{"type":"way","id":200596928,"nodes":[1947257516,2184965488,1782729408]},{"type":"way","id":200596960,"nodes":[2105667816,3764074088,4082282109,3764074072,751735372]},{"type":"way","id":208169450,"nodes":[1953686428,469773614]},{"type":"way","id":208169795,"nodes":[1106056883,107716,2407843602,4059394509]},{"type":"way","id":208169807,"nodes":[2521654794,4350220820]},{"type":"way","id":216137760,"nodes":[1934197347,107710]},{"type":"way","id":224887250,"nodes":[2337311461,6282772555,2337311458]},{"type":"way","id":225030185,"nodes":[2338587824,266558759,266559134]},{"type":"way","id":237089711,"nodes":[2450397879,4350220819,2521654794]},{"type":"way","id":237462804,"nodes":[2453519185,109836]},{"type":"way","id":237462807,"nodes":[2453519182,109836]},{"type":"way","id":249958671,"nodes":[2564873543,370711678]},{"type":"way","id":255562922,"nodes":[1106056846,107715]},{"type":"way","id":260741726,"nodes":[2663048865,1934197346,2663048864]},{"type":"way","id":260741727,"nodes":[2663048864,1947257529,280412702]},{"type":"way","id":260741728,"nodes":[2184966600,107710]},{"type":"way","id":260741729,"nodes":[2663048864,1106056850]},{"type":"way","id":294010666,"nodes":[2975783588,25476022,25476014,452514269]},{"type":"way","id":324319421,"nodes":[10574746,6282772563,6282772575,4967033605,25471371]},{"type":"way","id":324319422,"nodes":[10574746,25471086]},{"type":"way","id":387883962,"nodes":[4174200022,3911044741]},{"type":"way","id":405787811,"nodes":[108895,1685903447,1685903444,3497406876,25472677,1685903455,4068974160]},{"type":"way","id":405787812,"nodes":[4079423542,10574746]},{"type":"way","id":416716056,"nodes":[4174200026,4697525067,4174200022]},{"type":"way","id":416716060,"nodes":[4174365780,4174365779]},{"type":"way","id":416716062,"nodes":[4174200021,4174200023,4697525068,4174200024,4174365778,2140823714,25474749,2140823741,1139345972,25474750]},{"type":"way","id":416716067,"nodes":[4174200024,4174200026]},{"type":"way","id":431040155,"nodes":[4068974160,4302887173,108896,281454959]},{"type":"way","id":437141999,"nodes":[25471098,4350239457,1104386149,1614926285]},{"type":"way","id":437142000,"nodes":[1614846708,1614846707,5012149917,5012149907,3790287116,4415640809,4457798823]},{"type":"way","id":437142002,"nodes":[9475518,1375625382,2502903587,1678452824,25471712]},{"type":"way","id":437203852,"nodes":[4350220820,1947257531,1237514461]},{"type":"way","id":437203853,"nodes":[1237514461,280412702]},{"type":"way","id":437877198,"nodes":[469773614,1947257516]},{"type":"way","id":437877199,"nodes":[1782729408,280339753,2238463476,2238463480,1106056850]},{"type":"way","id":437877474,"nodes":[25471098,25471100]},{"type":"way","id":437877475,"nodes":[1678452805,1678452804,2444565402,1678452794]},{"type":"way","id":474718697,"nodes":[4686140571,25471710]},{"type":"way","id":474718698,"nodes":[4686140572,4686140571]},{"type":"way","id":476102905,"nodes":[4697525065,1237543434,1237543413,25474654]},{"type":"way","id":488354629,"nodes":[5011556356,5011556360]},{"type":"way","id":512564969,"nodes":[5011556345,5012149913]},{"type":"way","id":512579148,"nodes":[5011796257,5012151522]},{"type":"way","id":512579150,"nodes":[5011796269,5012149916]},{"type":"way","id":512579151,"nodes":[5011796261,5012149910]},{"type":"way","id":512579153,"nodes":[5011796263,5012149912]},{"type":"way","id":512579159,"nodes":[5011796259,5012149909]},{"type":"way","id":512579166,"nodes":[5011796252,5012151521]},{"type":"way","id":512579175,"nodes":[3811129601,5012149919]},{"type":"way","id":512579178,"nodes":[5011796271,5012149905]},{"type":"way","id":512579179,"nodes":[5011796264,5011796258]},{"type":"way","id":512579189,"nodes":[5011796256,5012149920]},{"type":"way","id":512579190,"nodes":[5011796248,4806684750]},{"type":"way","id":512596970,"nodes":[5012087062,5012087056,5012087069]},{"type":"way","id":512596976,"nodes":[5012087071,5012087066,5012087065]},{"type":"way","id":512596983,"nodes":[5012087061,5012087059]},{"type":"way","id":512596990,"nodes":[5012087075,5012087073]},{"type":"way","id":512601074,"nodes":[5012149904,5012151523]},{"type":"way","id":512601075,"nodes":[5012149917,5012151526]},{"type":"way","id":512601078,"nodes":[5012149898,5012149907]},{"type":"way","id":512601081,"nodes":[5012149906,5012149908]},{"type":"way","id":512601087,"nodes":[5012149903,5011556351]},{"type":"way","id":512601090,"nodes":[5012149911,5011388929]},{"type":"way","id":512601091,"nodes":[5012149915,5012149901]},{"type":"way","id":512601095,"nodes":[5012149918,3790287116]},{"type":"way","id":512601099,"nodes":[5012149899,568842587]},{"type":"way","id":512601100,"nodes":[5012149900,568842591]},{"type":"way","id":551430780,"nodes":[5325427430,5325427428]},{"type":"way","id":552451099,"nodes":[5333858450,303585905]},{"type":"way","id":559566159,"nodes":[4457798823,1678452782]},{"type":"way","id":559566164,"nodes":[1614846692,6282772556,4457798823]},{"type":"way","id":560380576,"nodes":[5404151677,5404151675,5404151676]},{"type":"way","id":569883260,"nodes":[5479777431,5479777432,5479777433,5479777434,5479777435,5479777436,5479777437,5479777438,5479777439,5479777440,5479777441,5479777442,5479777443,5479777428,5479777429,5479777430]},{"type":"way","id":569883295,"nodes":[5479777651,5479777634]},{"type":"way","id":569883298,"nodes":[5479777653,5479777652]},{"type":"way","id":569883301,"nodes":[5479777655,5479777654]},{"type":"way","id":569883372,"nodes":[5479777903,5479777902]},{"type":"way","id":569883379,"nodes":[5479777909,5479777908]},{"type":"way","id":569883382,"nodes":[5479777911,5479777910]},{"type":"way","id":569883386,"nodes":[5479777913,5479777912]},{"type":"way","id":584351062,"nodes":[9515377,109838]},{"type":"way","id":585735067,"nodes":[9512908,1106056843,1614846683]},{"type":"way","id":585735068,"nodes":[1614846702,1614846699,1614846695,6277660702,1614846692]},{"type":"way","id":585735069,"nodes":[25471100,25471101,4350208169,1106056846]},{"type":"way","id":585735070,"nodes":[25471100,2831321470,6277660685,1614846702]},{"type":"way","id":605953617,"nodes":[5746109250,5746109251]},{"type":"way","id":647037412,"nodes":[1614846678,5387300688,1106056861,9515387]},{"type":"way","id":647037418,"nodes":[9515370,2337311461,27345499]},{"type":"way","id":657937579,"nodes":[6161772317,6161772318]},{"type":"way","id":659336503,"nodes":[6173148421,826724301]},{"type":"way","id":668157200,"nodes":[6256910043,6256910044]},{"type":"way","id":668157201,"nodes":[6256910045,6256910046]},{"type":"way","id":25803166,"nodes":[26405445,1822628006,805522648,914309891,915410128,1630605810]},{"type":"way","id":25913655,"nodes":[282686497,282686493,1646024849,1646024870,1646024878,282686494,282686496]},{"type":"way","id":25913656,"nodes":[12354829,3328660559,282686499]},{"type":"way","id":66535223,"nodes":[805522625,805522627,805522630]},{"type":"way","id":66535226,"nodes":[805522608,805522640,805522643,805522645,805522646,805522648]},{"type":"way","id":151764561,"nodes":[1646024810,1646024813]},{"type":"way","id":151764562,"nodes":[1646024907,1646024923]},{"type":"way","id":151764563,"nodes":[1646024871,1646024847]},{"type":"way","id":151764564,"nodes":[1646024884,1646024925]},{"type":"way","id":151764566,"nodes":[1646024772,1646024795]},{"type":"way","id":151764568,"nodes":[1646024911,1646024897]},{"type":"way","id":151764569,"nodes":[1646025042,1646025038]},{"type":"way","id":151764570,"nodes":[805522632,805522634]},{"type":"way","id":151764572,"nodes":[1646025000,1646024977,1646024973,1646024971,1646024975]},{"type":"way","id":151764573,"nodes":[1646024880,1646024890,1646024887,1646024882,1646024873]},{"type":"way","id":151764574,"nodes":[1646024948,1646024946,1646024941,1646024937,1646024943]},{"type":"way","id":151764575,"nodes":[1646024969,1646024968,1646024964,1646024959,1646024957,1646024962,1646024966]},{"type":"way","id":151766576,"nodes":[1646043096,1646043098]},{"type":"way","id":151793294,"nodes":[26398105,1646024762]},{"type":"way","id":151793297,"nodes":[1646306164,1646306191]},{"type":"way","id":151793298,"nodes":[1646306154,1646306156,1646306158]},{"type":"way","id":151793299,"nodes":[12354831,1646306203]},{"type":"way","id":151793302,"nodes":[1646306130,1646306121,1646306140,1646306154,1646306150,1646306149,1646306134,1646306130]},{"type":"way","id":200598189,"nodes":[2105690372,805522608,3519058521,2105690380,2334377717,914309954,25472945]},{"type":"way","id":204064498,"nodes":[2140823620,2140823543]},{"type":"way","id":204064499,"nodes":[2140823777,2140823569]},{"type":"way","id":223569640,"nodes":[590146512,1646077134,12354831,4752385702,590146497,4082316449,12354829,4082316448,2885402275,21554353]},{"type":"way","id":224637155,"nodes":[1953721171,914309874,469785645,3497406877]},{"type":"way","id":246278862,"nodes":[2533348197,2533348198,2533348199]},{"type":"way","id":523267186,"nodes":[5094593335,5094593324]},{"type":"way","id":523267189,"nodes":[5094593326,251952341]},{"type":"way","id":523267190,"nodes":[5094593327,5094593332,5094593324,5094593326]},{"type":"way","id":523267192,"nodes":[5094593335,5094593340]},{"type":"way","id":523267193,"nodes":[5094593340,5094593323,5110757792,5110757790]},{"type":"way","id":23613967,"nodes":[4927544942,3156898687,255747716,4285841903,25497917]},{"type":"way","id":23613976,"nodes":[255747733,4013524084]},{"type":"way","id":24858856,"nodes":[270112539,321255910,270112540]},{"type":"way","id":24858866,"nodes":[4781199338,321255911,270109327]},{"type":"way","id":200596577,"nodes":[3497406877,805522625,1881001588]},{"type":"way","id":200598186,"nodes":[1630605982,2105690372]},{"type":"way","id":314282955,"nodes":[4927544936,4013524084,255747728,4284097308,256339270]},{"type":"way","id":320040501,"nodes":[2341656968,3265436260,6327557047,4085747014,3811094141,4174200047,3811094142,26374229,4174200046,25378124,4174200045,3811094139,107699,3811094140,4174200040,4174200036,4174200034]},{"type":"way","id":2424005,"nodes":[107699,4697525047,4502770263]},{"type":"way","id":4075810,"nodes":[3140855394,3581332546,21590616]},{"type":"way","id":4255475,"nodes":[25497910,25497912]},{"type":"way","id":4255477,"nodes":[3141587863,3581332539,3581332535]},{"type":"way","id":4255578,"nodes":[3141011326,3149029365,4059354472,3149029366,3497448050,2586621166]},{"type":"way","id":4276413,"nodes":[25378119,25533129,25378121]},{"type":"way","id":4276414,"nodes":[25533132,25533129]},{"type":"way","id":4277004,"nodes":[107708,1938953367,1139387510,25533132,5501130885,25737250,25533164,25533170,25533174,1730303422,25533180]},{"type":"way","id":4277054,"nodes":[3140965156,25378118,1140700560,25378117]},{"type":"way","id":4299212,"nodes":[107714,25533203]},{"type":"way","id":4615512,"nodes":[2586635284,2586635273,1938953374,1938953372,25378119]},{"type":"way","id":23613971,"nodes":[255747713,255747714,4833195584,3155351702,3142789923]},{"type":"way","id":23613972,"nodes":[255747716,3585676160]},{"type":"way","id":23659749,"nodes":[25497912,5623296806,3142660158]},{"type":"way","id":23659750,"nodes":[3581332535,3140855399,3141558547,3153598126,3140855397,3140855398,25497907,3140855374,25497908,3140855373,3149029366]},{"type":"way","id":24858865,"nodes":[270112583,270112581]},{"type":"way","id":25904244,"nodes":[25378118,282569804,282569803]},{"type":"way","id":25904248,"nodes":[256339270,3148313371,282569814]},{"type":"way","id":29206494,"nodes":[321255913,321255914]},{"type":"way","id":34437379,"nodes":[3140855395,396093944,1938953371,396093947,1938953370,1685903451,396093955,2586635281,1938953366,282569738]},{"type":"way","id":38901195,"nodes":[3140855372,9475245,3140855394,3149804732,3644748765,3140965156,3140855395]},{"type":"way","id":38901196,"nodes":[3140855395,3140855396,3141011326]},{"type":"way","id":38901197,"nodes":[282569738,1938953364,107714,1938953363]},{"type":"way","id":39123705,"nodes":[3141011326,1140700574,1140700492,1938953369,2586635279,107697]},{"type":"way","id":60368354,"nodes":[751735214,751735355]},{"type":"way","id":60368370,"nodes":[4684502368,751735214,484375487]},{"type":"way","id":107790222,"nodes":[1238198746,1238198769,1238198792,1237543404]},{"type":"way","id":128805224,"nodes":[25533211,4697525062,1422415852]},{"type":"way","id":140464461,"nodes":[1538881761,1538881758]},{"type":"way","id":140464462,"nodes":[1538881759,1538881762]},{"type":"way","id":156329913,"nodes":[107698,5532720347,6139961783,25737266]},{"type":"way","id":157541665,"nodes":[1697772135,2586614735,33141175,2586635284]},{"type":"way","id":204064489,"nodes":[2140823733,2140823576]},{"type":"way","id":208169961,"nodes":[1938953363,4059354473]},{"type":"way","id":208169962,"nodes":[455705604,751735259,280412704]},{"type":"way","id":236139689,"nodes":[25533217,25533215,25533211,25533203,2533371666,2533371673,107698,1730303400,1730303419,1730303447,25378123,4697525039,25378124]},{"type":"way","id":236140105,"nodes":[2441990082,2441990078,455705610,1528450580,2441990077,2441990080,1954581192,3140855372]},{"type":"way","id":246281847,"nodes":[2533371661,2533371664]},{"type":"way","id":246281849,"nodes":[2533371669,2533371671]},{"type":"way","id":252613091,"nodes":[2586614735,2586614741]},{"type":"way","id":252613323,"nodes":[2586616724,2586616722]},{"type":"way","id":252613701,"nodes":[2586621168,2586621166]},{"type":"way","id":252614800,"nodes":[107697,2586635291,2586635280,2586635274,25737266]},{"type":"way","id":252614802,"nodes":[2586621166,1938953375,2586635296,2586635284]},{"type":"way","id":252614806,"nodes":[25737266,2586635285,2586635293,2586635288,2586635283,2586635295]},{"type":"way","id":252614807,"nodes":[25378119,1685903439,107708,25533150,4431563564,2586635292,2586635275,107697,2586635276,2586635295,282569738]},{"type":"way","id":357709277,"nodes":[21590616,3142670557,21590618,3142660158,255747735]},{"type":"way","id":398489181,"nodes":[282569803,4012411306]},{"type":"way","id":398489182,"nodes":[3895360879,3895360876]},{"type":"way","id":403581840,"nodes":[4059354473,751735148,2184970824,3519042187,455705604]},{"type":"way","id":416690535,"nodes":[25378124,4174200043]},{"type":"way","id":416690543,"nodes":[2140823631,2140823680,2140823590,25533217]},{"type":"way","id":416690605,"nodes":[4174200034,4174200035,4174200039,25533217]},{"type":"way","id":416690610,"nodes":[3911044721,4174200033,4697525053,4174200038,2140823590]},{"type":"way","id":424889285,"nodes":[4174200034,4174200033,4174200032,2140823631]},{"type":"way","id":445807203,"nodes":[3149029366,4431563560,4431563561,4431563562,4431563563,4016579385,4431563564]},{"type":"way","id":453558580,"nodes":[4502770262,4502770264,107698]},{"type":"way","id":453558581,"nodes":[4502770263,4502770262]},{"type":"way","id":474502536,"nodes":[4684502369,751735253,4684502368]},{"type":"way","id":474502537,"nodes":[751735148,4684502369]},{"type":"way","id":485346072,"nodes":[4781199330,4781199331]},{"type":"way","id":485346073,"nodes":[4781199331,4781199332,4781199338,270112583]},{"type":"way","id":485346080,"nodes":[4781199361,4781199360]},{"type":"way","id":491173007,"nodes":[4833195585,255747714]},{"type":"way","id":491173008,"nodes":[3585676160,4833195585]},{"type":"way","id":572619930,"nodes":[6139961783,5532720349]},{"type":"way","id":576992785,"nodes":[5532720347,5532720348,6140802454,5532720349]},{"type":"way","id":608788572,"nodes":[1419565097,2586616722,1697772135]},{"type":"way","id":655423770,"nodes":[5532720349,5501130885]},{"type":"way","id":31039,"nodes":[107824,2425138405,2425138407,5811148948,5811148947,5811148946,1795369607,9791482]},{"type":"way","id":2644235,"nodes":[104302,3645263633,1448889399,107844]},{"type":"way","id":2644765,"nodes":[107843,4860880474]},{"type":"way","id":2876427,"nodes":[9791482,4242112079,5811148944,5434424321]},{"type":"way","id":2876428,"nodes":[13250138,5877171663,2479767093,469760236,4409200422,4409200425]},{"type":"way","id":2876551,"nodes":[107851,4231050595,764870828,4231050414,14791174]},{"type":"way","id":4082526,"nodes":[21665081,21665008,2833448424,5128425114]},{"type":"way","id":4082527,"nodes":[21665008,21665009]},{"type":"way","id":4244822,"nodes":[25375812,970857045,4239164026,25375954,4239164016,4239163997,25375955]},{"type":"way","id":4244823,"nodes":[21665132,1080591780,25530312,4409200420]},{"type":"way","id":4253217,"nodes":[21665247,5355634695,5350044101,5350044724,21665248]},{"type":"way","id":4253403,"nodes":[14791182,4237160380,4237160372,5811146369,5811146367,14791186]},{"type":"way","id":4253546,"nodes":[25375957,4239164036,4239163469,25375812,4239163474,4239163351,4239163380,4242112323,25375958]},{"type":"way","id":4253548,"nodes":[14791174,4231050400,2502928029,14791186,5811146368,4239701276,4239701247,5877177127,4242112322,14791193]},{"type":"way","id":4253551,"nodes":[25472666,5362237836,25472674]},{"type":"way","id":4253554,"nodes":[25472674,5362237842,5359989224,5359989234,5362237840,25472672]},{"type":"way","id":4253562,"nodes":[14727249,4360435541,107852]},{"type":"way","id":4258961,"nodes":[25530033,5840695151,25530024,5840695133,21665140]},{"type":"way","id":4259161,"nodes":[25530312,5877171660,1105928793,1937391902,1080591766,25375976]},{"type":"way","id":4259178,"nodes":[21665141,2496863224,2496863225,21665140,2961254826,21665139,6319587582,21665589]},{"type":"way","id":4376579,"nodes":[499122580,26699540,107823]},{"type":"way","id":4376581,"nodes":[107824,1938953380,2425138406,1795369613,5461196612,1938953381,1795369609,21665587,25375955]},{"type":"way","id":4376582,"nodes":[21665588,5811245210,26699565]},{"type":"way","id":4376583,"nodes":[107831,6319748554]},{"type":"way","id":4376585,"nodes":[26699562,949045824,26699561]},{"type":"way","id":4376586,"nodes":[21665588,21665137,21665136,4942576593]},{"type":"way","id":4393568,"nodes":[26848675,25376002]},{"type":"way","id":4393581,"nodes":[21554368,866311216]},{"type":"way","id":4615507,"nodes":[21665247,1700106707,5355634693,21554368]},{"type":"way","id":7970884,"nodes":[107839,2284375270,4360454515,108014]},{"type":"way","id":7970885,"nodes":[1843345666,4360457245]},{"type":"way","id":17513384,"nodes":[107843,983836443]},{"type":"way","id":17513386,"nodes":[107844,1448889397,4715890871,107843]},{"type":"way","id":24465844,"nodes":[21665248,5339970474,2159428767,2159428766,5339970475,14727249]},{"type":"way","id":24465847,"nodes":[21665590,5350044725,4360406521,21665248]},{"type":"way","id":31309016,"nodes":[107852,25472654]},{"type":"way","id":38060395,"nodes":[21665132,21665131,1103001639,21665141]},{"type":"way","id":38060397,"nodes":[21665141,5630706682,21665130]},{"type":"way","id":40470530,"nodes":[490256481,25530036]},{"type":"way","id":40964939,"nodes":[499122575,21665080]},{"type":"way","id":41117904,"nodes":[26699565,5630940176]},{"type":"way","id":43573213,"nodes":[107831,1700106533,1230855652,1937391977,6219646000,21554368]},{"type":"way","id":49934757,"nodes":[634568069,634568066,107852]},{"type":"way","id":49934761,"nodes":[25470853,5339077113]},{"type":"way","id":49934784,"nodes":[634568087,1109736448]},{"type":"way","id":49934785,"nodes":[983836443,108014]},{"type":"way","id":60494630,"nodes":[104302,881889759,4222962171,107842]},{"type":"way","id":60531130,"nodes":[107853,4236640531,25375960,1700106576,107831]},{"type":"way","id":67551926,"nodes":[25472674,5362237839,949076160,949076158,3054882706,25375960]},{"type":"way","id":67551927,"nodes":[25472654,25472666]},{"type":"way","id":67551929,"nodes":[25472666,5631066029,1413151220,1413151219,5355634694,21665247]},{"type":"way","id":67551930,"nodes":[25375960,25375812]},{"type":"way","id":72989034,"nodes":[281455009,2096561079,6219635599,6219635602]},{"type":"way","id":81459090,"nodes":[949045924,949045884]},{"type":"way","id":94892317,"nodes":[26699560,1700140638,1700140611,5630841555,5630841554,6319748550,26699559]},{"type":"way","id":94892319,"nodes":[26699560,26699562]},{"type":"way","id":95100184,"nodes":[1103268830,1103268833]},{"type":"way","id":96228580,"nodes":[1115431333,1115431335,1115431337]},{"type":"way","id":96228583,"nodes":[1115431337,3782162182]},{"type":"way","id":96228585,"nodes":[1115431346,1115431348]},{"type":"way","id":107057663,"nodes":[1230855633,1230855643]},{"type":"way","id":107057664,"nodes":[1230855643,1230855652]},{"type":"way","id":116615382,"nodes":[21665589,2833448475,6319600650,107823,392132821]},{"type":"way","id":147248799,"nodes":[1604752665,4236640539,107853]},{"type":"way","id":147248800,"nodes":[107842,764870818]},{"type":"way","id":148519534,"nodes":[4356572322,1616122252,1616122251,1616122250,1115431348,1616122247,1616122245,1616122244,1616122241,1616122237]},{"type":"way","id":157759512,"nodes":[1700140613,1700140653]},{"type":"way","id":157759513,"nodes":[1700140653,1700140657]},{"type":"way","id":157759514,"nodes":[1700140611,1700140613]},{"type":"way","id":181353817,"nodes":[1917794487,1917794490]},{"type":"way","id":199511595,"nodes":[2094955592,866311227,25375976,26848675]},{"type":"way","id":199658589,"nodes":[25375955,2479767095,5877171661,13250138]},{"type":"way","id":199661227,"nodes":[21554368,6250477238,6219635605,6219635602]},{"type":"way","id":199664979,"nodes":[281455009,866311234,4360406513,21665590]},{"type":"way","id":233075058,"nodes":[392132821,2414156537]},{"type":"way","id":241773309,"nodes":[2493654751,2833445982]},{"type":"way","id":242166996,"nodes":[21665081,6319600651,2833448458,21665589]},{"type":"way","id":279088492,"nodes":[2493654751,2833445980]},{"type":"way","id":279088493,"nodes":[2833448424,2833448429,2833448450]},{"type":"way","id":291536486,"nodes":[25530036,25530034]},{"type":"way","id":293637093,"nodes":[2414156537,2971399439,2971399440]},{"type":"way","id":302026658,"nodes":[5434424321,14791193,4239163378,970857026,4239164331,4239164344,25375958,4239164347,4239164350,4237160296,4237160305,4242112325,107853]},{"type":"way","id":315631030,"nodes":[2971399440,6313755481,5811164165,107824]},{"type":"way","id":319017443,"nodes":[25472654,5362237856,25472672,4236640538,107853]},{"type":"way","id":322767216,"nodes":[14791186,5811146371,5811146370,14791190]},{"type":"way","id":374835686,"nodes":[3782162182,1115431346]},{"type":"way","id":401189740,"nodes":[107851,14791182,3519181426,4236640701]},{"type":"way","id":424202400,"nodes":[764870818,107851]},{"type":"way","id":424202401,"nodes":[4236640051,1604752665]},{"type":"way","id":424202402,"nodes":[4236640701,4236640087,2556043427,4236640057,4236640051]},{"type":"way","id":424202403,"nodes":[4236640701,4236640693,4236640689,4236640084,4236640064,4236640051]},{"type":"way","id":424252149,"nodes":[4237160372,4237160366]},{"type":"way","id":424252150,"nodes":[4237160373,4237160380]},{"type":"way","id":424507427,"nodes":[4239163382,4239163388]},{"type":"way","id":424507429,"nodes":[4239164293,4239163388]},{"type":"way","id":424507430,"nodes":[4239163378,4239163353,4239164192]},{"type":"way","id":424507431,"nodes":[4239164207,4239164219,4239163351]},{"type":"way","id":424507432,"nodes":[4239164301,4239164289,4239163385,4239163381,4239163380]},{"type":"way","id":424507433,"nodes":[4239164331,4239164319,4239164311,4239164309,4239164327,4239164330,4239164333,4239164339,4239164344]},{"type":"way","id":424560510,"nodes":[4239701265,4239701276]},{"type":"way","id":424560512,"nodes":[4239701247,4239701213]},{"type":"way","id":437880011,"nodes":[26699561,5630841550,5630841552]},{"type":"way","id":438310842,"nodes":[5465557380,4360451851,976973186,107839]},{"type":"way","id":438310843,"nodes":[4360457244,1843345666]},{"type":"way","id":438310844,"nodes":[257810133,1843345666]},{"type":"way","id":443322181,"nodes":[4409200420,1105928825,2479767098,5877171662,13250138]},{"type":"way","id":443322182,"nodes":[4409200425,25375957,2094955592,107831]},{"type":"way","id":473673101,"nodes":[1109736448,1109736452,4360435542]},{"type":"way","id":478363077,"nodes":[4715890870,25471038]},{"type":"way","id":478363078,"nodes":[25471038,4715890872,4715890871]},{"type":"way","id":491003673,"nodes":[4942576593,21665135,21665134,21665133,1103268839,469760241]},{"type":"way","id":494310513,"nodes":[4860880474,4222962166,107842]},{"type":"way","id":527537175,"nodes":[5128425114,2493654751]},{"type":"way","id":560896052,"nodes":[5339077113,4682210575,634568069]},{"type":"way","id":561558982,"nodes":[469760241,6319618545,21665132]},{"type":"way","id":564099716,"nodes":[4409200424,6319748547]},{"type":"way","id":567293138,"nodes":[5461200184,5461200183]},{"type":"way","id":567293140,"nodes":[5461200553,5461200188]},{"type":"way","id":567293146,"nodes":[5461200393,5461196614]},{"type":"way","id":567293187,"nodes":[5461199346,5461199345]},{"type":"way","id":567293206,"nodes":[5461199357,5461199356]},{"type":"way","id":567293213,"nodes":[5461199354,5461199355]},{"type":"way","id":567293230,"nodes":[5461199376,5461199591]},{"type":"way","id":567293245,"nodes":[5461199380,5461199381]},{"type":"way","id":567293355,"nodes":[5461199572,5461199570]},{"type":"way","id":567293358,"nodes":[5461199578,5461199579]},{"type":"way","id":567293361,"nodes":[5461199593,5461199592]},{"type":"way","id":567293363,"nodes":[5461199597,5461199585]},{"type":"way","id":567293375,"nodes":[5461199722,5461199723]},{"type":"way","id":567293406,"nodes":[5461200025,5461200026]},{"type":"way","id":567293414,"nodes":[5461200042,5461200041]},{"type":"way","id":567293424,"nodes":[5461200150,5461200149]},{"type":"way","id":567293425,"nodes":[5461200151,5461200163]},{"type":"way","id":567293426,"nodes":[5461200152,5461200166]},{"type":"way","id":567293460,"nodes":[5461200182,5461200180]},{"type":"way","id":567293466,"nodes":[5461200186,5461199021]},{"type":"way","id":567293473,"nodes":[5461200147,5461200189]},{"type":"way","id":567293476,"nodes":[5461200168,5461200190]},{"type":"way","id":567293480,"nodes":[5461200169,5461200191]},{"type":"way","id":567293493,"nodes":[5461200357,5461200356]},{"type":"way","id":567293496,"nodes":[5461200359,5461200358]},{"type":"way","id":567293500,"nodes":[5461200361,5461200360]},{"type":"way","id":567293535,"nodes":[5461200394,5461200179]},{"type":"way","id":567293538,"nodes":[5461200395,5461200178]},{"type":"way","id":567293545,"nodes":[5461200400,5461200171]},{"type":"way","id":567293548,"nodes":[5461200401,5461200172]},{"type":"way","id":567293552,"nodes":[5461200402,5461200173]},{"type":"way","id":567293568,"nodes":[5461200545,5461200544]},{"type":"way","id":567293575,"nodes":[5461200547,5461200546]},{"type":"way","id":567293582,"nodes":[5461200549,5461200548]},{"type":"way","id":567293593,"nodes":[5461196618,5461200554]},{"type":"way","id":567941583,"nodes":[4360457245,5465557380]},{"type":"way","id":570577008,"nodes":[14791190,5630569824]},{"type":"way","id":585743671,"nodes":[4356572316,6319779695,5811252641,4356572310]},{"type":"way","id":589653019,"nodes":[1917794490,21665587]},{"type":"way","id":589658496,"nodes":[26699562,5630761778]},{"type":"way","id":589658504,"nodes":[5630761778,21554368]},{"type":"way","id":589658510,"nodes":[866311216,949045792,26699563,26848680]},{"type":"way","id":589659240,"nodes":[5630766895,26699560]},{"type":"way","id":589660342,"nodes":[4356572310,5811263956,108068,5811263955]},{"type":"way","id":589665715,"nodes":[5630841551,5630841553,4356572316]},{"type":"way","id":589665716,"nodes":[26699559,4356572322]},{"type":"way","id":589665717,"nodes":[4356572322,5630841551]},{"type":"way","id":589665718,"nodes":[5630841551,1616122253]},{"type":"way","id":589674457,"nodes":[5630940176,26848675]},{"type":"way","id":614321610,"nodes":[5630940176,5811245211]},{"type":"way","id":614324183,"nodes":[5811263955,5811252635,4356572321,21665588]},{"type":"way","id":617901192,"nodes":[5840695134,5840695135]},{"type":"way","id":617901199,"nodes":[5840695143,5840695132]},{"type":"way","id":617901200,"nodes":[501766819,5840695145]},{"type":"way","id":622160507,"nodes":[4239164053,3062596155,970857045]},{"type":"way","id":628689167,"nodes":[5935736817,5630569824]},{"type":"way","id":628693434,"nodes":[5811174034,2971399443,5935769749,5811153023]},{"type":"way","id":628693436,"nodes":[5811174036,2971399443]},{"type":"way","id":674188214,"nodes":[5811174034,5461196613,2971399444,5344928729,1927220454]},{"type":"way","id":674188216,"nodes":[2496862506,1927220454,1917794487]},{"type":"way","id":674188943,"nodes":[2971399444,6313786639,6313786638,5811153023]},{"type":"way","id":674816213,"nodes":[25376002,4409200424]},{"type":"way","id":674816215,"nodes":[26699565,5630766895]},{"type":"way","id":674823272,"nodes":[6319748554,25376002]},{"type":"way","id":674823275,"nodes":[5630841552,26699559]},{"type":"way","id":674823281,"nodes":[1616122253,5811252637,21665588]},{"type":"way","id":674823289,"nodes":[6319748547,26699560]},{"type":"way","id":2425193,"nodes":[107830,5986763688]},{"type":"way","id":2876973,"nodes":[107837,742854871]},{"type":"way","id":3088166,"nodes":[107835,107863]},{"type":"way","id":3188938,"nodes":[5458472046,455705627]},{"type":"way","id":4236981,"nodes":[107829,3085005042,3085005041,396093922,2402713068,456363900]},{"type":"way","id":4236993,"nodes":[5458472046,1104496295,2096450587,1104473061]},{"type":"way","id":4253171,"nodes":[9475528,534993963]},{"type":"way","id":4253185,"nodes":[25470388,1413151217]},{"type":"way","id":4253600,"nodes":[5339077112,4962172563,4356620360,25472759]},{"type":"way","id":4253606,"nodes":[25472759,3085005061,25472762,25472766]},{"type":"way","id":4253607,"nodes":[25472764,25472762]},{"type":"way","id":4253659,"nodes":[13491058,5517569296]},{"type":"way","id":4253664,"nodes":[11544697,317930353]},{"type":"way","id":4254072,"nodes":[11767207,25472992,25472874,25472865,734773337,1392042915,1104509637,1104496316]},{"type":"way","id":4255493,"nodes":[21590631,5466794238]},{"type":"way","id":4256871,"nodes":[10616173,4218360423,4218365467,13491058,895113296,4218360149,4218360151,10616174,4218360529,6032800919,25472903,9476443]},{"type":"way","id":4256876,"nodes":[9476442,2392997817,1104496341,25472906,3519233743,2094965921]},{"type":"way","id":4256878,"nodes":[25472906,1938953401,25503723]},{"type":"way","id":4256991,"nodes":[108404,2096436911,2117444163,1104496344]},{"type":"way","id":4393564,"nodes":[25472769,4587152383]},{"type":"way","id":4393577,"nodes":[107829,2402713069,2402713067,456363903]},{"type":"way","id":4615510,"nodes":[25472758,5466794228,25472766,4356620362,4356620363,1954794473,25472757]},{"type":"way","id":4615513,"nodes":[25470694,371400554,25470697]},{"type":"way","id":9346429,"nodes":[108901,1611771520,2117470724,2441993335,2117470726,107830]},{"type":"way","id":17513388,"nodes":[107838,4360451847]},{"type":"way","id":17513390,"nodes":[25470679,3085005054,107829]},{"type":"way","id":17513393,"nodes":[9475528,1947276635,2441993361,455705622]},{"type":"way","id":24378722,"nodes":[25470678,4962063593]},{"type":"way","id":24378840,"nodes":[264515752,6060045961,1104496342]},{"type":"way","id":24936211,"nodes":[108404,2096450597,1954794478]},{"type":"way","id":25803171,"nodes":[264515752,6060045960,1104496349,6060061452,1938953398,1938953397,6060084833]},{"type":"way","id":26789076,"nodes":[293864446,1392042899,25472875,107860]},{"type":"way","id":28921460,"nodes":[338505530,5445808814]},{"type":"way","id":32592744,"nodes":[366518767,25472874]},{"type":"way","id":37458962,"nodes":[25470388,438170520]},{"type":"way","id":37458963,"nodes":[438170530,5408675753,438170536]},{"type":"way","id":37458965,"nodes":[438170563,438170568]},{"type":"way","id":37458966,"nodes":[438170585,5408675096,438170591]},{"type":"way","id":38550744,"nodes":[21590630,455705623]},{"type":"way","id":38550745,"nodes":[456363900,3085005040,456363901,456363902,3085005039,2402713070,2402713071,456363903]},{"type":"way","id":38900540,"nodes":[6060084833,2249185409,6060045967,281454988,1938953394,108901]},{"type":"way","id":42786632,"nodes":[534993866,534993848,534993850]},{"type":"way","id":42786633,"nodes":[534993848,534993859]},{"type":"way","id":42786634,"nodes":[534993869,25470681]},{"type":"way","id":49934743,"nodes":[455705625,2117470725]},{"type":"way","id":49934748,"nodes":[107830,5466794231,2117470723,25472758,4587152382,3085005056,3085005055,25470679]},{"type":"way","id":49934749,"nodes":[1614978621,108403,25470465,3149232618,3895422567]},{"type":"way","id":49939371,"nodes":[455705627,2117467718,1161243569,455705628,1954794477]},{"type":"way","id":56062184,"nodes":[317930353,4216274127,11544696]},{"type":"way","id":56062185,"nodes":[317930352,4216274114]},{"type":"way","id":56062186,"nodes":[10616173,5458472047]},{"type":"way","id":58704856,"nodes":[727447379,4356554727]},{"type":"way","id":59235953,"nodes":[734773333,734773335,734773337]},{"type":"way","id":59847117,"nodes":[742854887,742854911,742854924,742854884]},{"type":"way","id":59847118,"nodes":[107837,764867047,107836]},{"type":"way","id":59847119,"nodes":[742854814,742854844,742854960]},{"type":"way","id":59847121,"nodes":[742854871,742854891,742854854,742854814]},{"type":"way","id":59847123,"nodes":[742854873,742854790]},{"type":"way","id":59847124,"nodes":[742854915,742854866,742854887]},{"type":"way","id":59847125,"nodes":[742854889,742854809]},{"type":"way","id":59847127,"nodes":[5458472046,6028641764,2117475741]},{"type":"way","id":74141482,"nodes":[877060602,264515752]},{"type":"way","id":75821291,"nodes":[869389523,869389491]},{"type":"way","id":83225329,"nodes":[968291529,6287138677,21563009,1938953384,9475520]},{"type":"way","id":83592711,"nodes":[11544693,973525755]},{"type":"way","id":95208509,"nodes":[1104473061,108404]},{"type":"way","id":95208510,"nodes":[1104473061,2117444169,2117444161,1953676002,1104473062]},{"type":"way","id":95209412,"nodes":[25503720,1938953428,9476442]},{"type":"way","id":95209413,"nodes":[1104473062,2117444174,2117444170,108404]},{"type":"way","id":95211064,"nodes":[1104496313,2096450595,2096450589,6060034392,1938953440,1104496309,2096450599,1104496326]},{"type":"way","id":95211065,"nodes":[1104496313,2096443244,2096450591,1104496319,4218360491,1104496346]},{"type":"way","id":95211066,"nodes":[1104496344,2096436910,2096436908,4218360364,1104496346]},{"type":"way","id":95211067,"nodes":[1104496344,4218360357,1104496292,1104496316]},{"type":"way","id":95212401,"nodes":[1104496326,10616173]},{"type":"way","id":95212402,"nodes":[1126375486,1104508800]},{"type":"way","id":96228577,"nodes":[1115431307,1115431319]},{"type":"way","id":97237927,"nodes":[1126375339,1126375365,1126375369]},{"type":"way","id":100451442,"nodes":[2117429475,2117429474,1161243567,1104508800]},{"type":"way","id":125295715,"nodes":[1392042915,1392042899]},{"type":"way","id":127732676,"nodes":[1413151217,25470700,1614949585]},{"type":"way","id":148386206,"nodes":[1614926348,5404583172,3895422574,3895422571,25470776]},{"type":"way","id":148388406,"nodes":[1614949613,1614949587,1614949585]},{"type":"way","id":148391188,"nodes":[1614978560,438170530,2512940590,25470388,2502928027,4356572320,4356572312,107828]},{"type":"way","id":148391192,"nodes":[9475520,1614978671,1614978668,1614978621]},{"type":"way","id":148391193,"nodes":[6287138677,4034416983,4008046110,21590630]},{"type":"way","id":183506561,"nodes":[1104496342,281454989,108901]},{"type":"way","id":184908620,"nodes":[1954794477,2096450603,1954794476,2096436912,5458472046]},{"type":"way","id":199512871,"nodes":[2094965921,877060602]},{"type":"way","id":199661224,"nodes":[1104496344,4218360345,293864446]},{"type":"way","id":199661226,"nodes":[25472764,4356620358,4356620361,4356620367,107833,4356620359,4356620356,25472757,4356620357,5458472045]},{"type":"way","id":199662044,"nodes":[107855,2096443242,2096443245,1104496313]},{"type":"way","id":199662045,"nodes":[1104496346,2096436909,2117447071,2117444166,1954794478]},{"type":"way","id":199662046,"nodes":[1104496346,2096450593,1104496296,1104496288,1104496326]},{"type":"way","id":199662047,"nodes":[1104496316,2096443243,9476441,107855]},{"type":"way","id":199672360,"nodes":[456363900,1938953392,3085005043]},{"type":"way","id":201725616,"nodes":[1954794477,1104473061]},{"type":"way","id":201725617,"nodes":[1954794478,2117429475,1954794477]},{"type":"way","id":201725618,"nodes":[1954794478,1161243563,108400,2117469204,25503725]},{"type":"way","id":201726450,"nodes":[25503725,2117470721,2117470727,455705625]},{"type":"way","id":201731222,"nodes":[455705622,1614978621]},{"type":"way","id":210074377,"nodes":[455705623,1614978658,6287158541]},{"type":"way","id":215300904,"nodes":[107836,764867054,107835]},{"type":"way","id":230159008,"nodes":[25472759,3085005050,3085005046,25472769]},{"type":"way","id":231949987,"nodes":[456363903,1614949613]},{"type":"way","id":251325040,"nodes":[455705618,455705620,4720710035,455705621,455705622,2117475740,455705623,4411113474,455705624]},{"type":"way","id":251325042,"nodes":[2441993346,2441993358,2441993340,2441993343,2441993353,727447379,4967033602,1947276625,1947276628,1947276630,9475528]},{"type":"way","id":251334875,"nodes":[25470680,25470776,13355224,2502928026,25470694,1614978560]},{"type":"way","id":262678527,"nodes":[2683449617,25470678]},{"type":"way","id":262678528,"nodes":[534993963,2683449617]},{"type":"way","id":277929210,"nodes":[25470678,490256621,534993866,25470680]},{"type":"way","id":303982842,"nodes":[3085005064,25472764]},{"type":"way","id":319017441,"nodes":[742854954,742854884]},{"type":"way","id":319017442,"nodes":[742854830,742854879,5986562231]},{"type":"way","id":421882363,"nodes":[4216274127,4216274136]},{"type":"way","id":421882364,"nodes":[4216274089,317930353]},{"type":"way","id":421882365,"nodes":[4216274114,4216274089]},{"type":"way","id":422109694,"nodes":[4218365468,4218365467]},{"type":"way","id":431299972,"nodes":[5466793963,5466793984]},{"type":"way","id":431299974,"nodes":[5466793801,5466793800]},{"type":"way","id":431320437,"nodes":[5466793575,4305311206]},{"type":"way","id":432474409,"nodes":[5466793944,5466793561]},{"type":"way","id":432474410,"nodes":[5466793988,5466793563]},{"type":"way","id":432474411,"nodes":[5466793995,5466794224]},{"type":"way","id":432474412,"nodes":[5466793943,5466793564]},{"type":"way","id":432474413,"nodes":[5466793992,5466793562]},{"type":"way","id":432474414,"nodes":[5466793986,5466793983]},{"type":"way","id":432474417,"nodes":[5466793789,5466793797]},{"type":"way","id":432479175,"nodes":[5466793803,5466793802]},{"type":"way","id":432479180,"nodes":[5466793799,5466793798]},{"type":"way","id":432479188,"nodes":[5466793786,5466793796]},{"type":"way","id":437880591,"nodes":[1614949585,1588852215,25470697]},{"type":"way","id":437880592,"nodes":[1614949585,1614949580,1614949561]},{"type":"way","id":437881810,"nodes":[25470679,4356591931]},{"type":"way","id":437881811,"nodes":[1614949613,1806593433,4411120201,4356572311,4356572319,107828]},{"type":"way","id":437881814,"nodes":[25470697,534993869]},{"type":"way","id":438304614,"nodes":[3085005068,3085005067,3085005066,4962172564,3085005064]},{"type":"way","id":438309960,"nodes":[742854884,107835]},{"type":"way","id":438309961,"nodes":[742854871,296441984,742854830]},{"type":"way","id":438309962,"nodes":[4360451847,4682210576,107837]},{"type":"way","id":438311806,"nodes":[11767206,4360469341,107863,4681962044,4681962045,1104473062]},{"type":"way","id":506482252,"nodes":[4962063593,25470681]},{"type":"way","id":506482253,"nodes":[25470681,1126375339,4356591932]},{"type":"way","id":506489661,"nodes":[4356591931,4962138929,4356591932]},{"type":"way","id":506489662,"nodes":[3085005043,3085005052,3085005053,396093929,3085005060,3085005059,3085005058,3085005057]},{"type":"way","id":506489666,"nodes":[3085005043,1937391982,25472769]},{"type":"way","id":506493656,"nodes":[5339077112,3085005063,3085005065,4356620365,4360406511]},{"type":"way","id":517266374,"nodes":[5458472047,9476442]},{"type":"way","id":533219692,"nodes":[5458472045,5458472046]},{"type":"way","id":545042168,"nodes":[4356591932,3085005044,3085005049,25470679]},{"type":"way","id":553063208,"nodes":[3085005068,5339077116,4360406511]},{"type":"way","id":556859792,"nodes":[5371270743,5371270746,107838]},{"type":"way","id":560417473,"nodes":[5404583123,5404583122,5404583121]},{"type":"way","id":560417480,"nodes":[5404583132,5404582019,5404583131]},{"type":"way","id":560417515,"nodes":[5404583188,5404582020,5404583187]},{"type":"way","id":560417518,"nodes":[5404583198,5404583124,5404583197]},{"type":"way","id":561385725,"nodes":[5413338304,5413338303]},{"type":"way","id":561385726,"nodes":[5413338302,5404583197]},{"type":"way","id":561385727,"nodes":[5413338303,5413338302]},{"type":"way","id":561385728,"nodes":[5413338306,5413338305,5413338304]},{"type":"way","id":565466193,"nodes":[5445808814,5445808813]},{"type":"way","id":567941589,"nodes":[4681962046,4360469338,11767206]},{"type":"way","id":568170861,"nodes":[5466793585,5466793581]},{"type":"way","id":568170922,"nodes":[5466793985,5466793964]},{"type":"way","id":568170931,"nodes":[5466793996,5466794225]},{"type":"way","id":568170936,"nodes":[5466794226,5466793997]},{"type":"way","id":568170940,"nodes":[5466794227,5466794222]},{"type":"way","id":574900775,"nodes":[5517569296,107855]},{"type":"way","id":634314088,"nodes":[2117475741,742854934,742854954]},{"type":"way","id":634328292,"nodes":[5986562231,6058284063,107833]},{"type":"way","id":634363121,"nodes":[5466794238,1611771527,21590632,4008046109,968291529]},{"type":"way","id":634363129,"nodes":[5986763688,2117470720,21590631]},{"type":"way","id":634363133,"nodes":[2117470725,455705626,2117469205,455705627]},{"type":"way","id":634363135,"nodes":[21590631,3644748772,21590630,2117475743,1614978621]},{"type":"way","id":634363138,"nodes":[455705625,107830]},{"type":"way","id":639234346,"nodes":[455705624,1938953393,2441993365,2117470722,5986763688]},{"type":"way","id":639234347,"nodes":[3519217057,1115431307,5339078763,4356620366,4962172562,5339077112,3085005064]},{"type":"way","id":644302879,"nodes":[6060011974,6060011975,6060011973,293864446]},{"type":"way","id":673432299,"nodes":[9476443,11544696,4216274149,9476442]},{"type":"way","id":2592181,"nodes":[11544693,11544697,4216272873,4216274099,4215728357,4215728363,11544695]},{"type":"way","id":2644766,"nodes":[107844,4715890870,1448889398,1448876659,108017]},{"type":"way","id":2661843,"nodes":[11767206,11767209,11767208]},{"type":"way","id":2876891,"nodes":[107864,4360460587,4360460586,4681962046]},{"type":"way","id":2876892,"nodes":[107839,4360451852,4360451853,5463724741,874823434,1392029481,5371270743]},{"type":"way","id":4074189,"nodes":[21563015,6287138678,317930325,21563009]},{"type":"way","id":4253215,"nodes":[14727254,5339077114,5339970477,14727249]},{"type":"way","id":4253218,"nodes":[25470853,3254337787,2622506576,14727254,25470835,3085005068]},{"type":"way","id":4254070,"nodes":[1614978560,438170563,438170585,5413338306,5413338307,1616122237]},{"type":"way","id":4376584,"nodes":[107828,4356577280,4356572314,26699561]},{"type":"way","id":4393584,"nodes":[107828,4356572315,949045924,26848680,2096561071,2096561065,281455009]},{"type":"way","id":17513387,"nodes":[108014,25471043,1879123343,5373783321,1879123345,4955757603,108002]},{"type":"way","id":17513389,"nodes":[107838,3254337788,25470853]},{"type":"way","id":35787177,"nodes":[9475520,354671266,9475519]},{"type":"way","id":67551928,"nodes":[21665248,5350044723,3085005069,5339077115,25470835]},{"type":"way","id":148386202,"nodes":[3140855387,3148247689,3140855388,354671172,9475520]},{"type":"way","id":148386205,"nodes":[3140855387,704139941,3895412046,3895422567]},{"type":"way","id":173540092,"nodes":[107836,4360451848,5463724742,4360457244]},{"type":"way","id":533016674,"nodes":[4587152383,1806593437,2096561060,2096561074,281455009]},{"type":"way","id":553063207,"nodes":[4360406511,5339077115,5339077114,5339077113]},{"type":"way","id":565466196,"nodes":[5445808813,317930302,317930304]},{"type":"way","id":662531827,"nodes":[108902,973525136,5416565257,338505531,338505530,6060061453,6060084833]},{"type":"way","id":664521275,"nodes":[6219635602,5932928493,6219635609,2096473199,21665590,4356609718,4409200426,3519217057]},{"type":"way","id":2592180,"nodes":[108902,4216272819,338501923,11544695,317930344,1551117103,25472904,14728072,5409072334,107856]},{"type":"way","id":4074253,"nodes":[3142630784,3141558549,21563698]},{"type":"way","id":4074309,"nodes":[3140855379,5817963345,3140855378,21564825]},{"type":"way","id":4075106,"nodes":[9475519,21563015]},{"type":"way","id":4256982,"nodes":[9475519,338501922,1783558256,5817963347,3140855379]},{"type":"way","id":28921461,"nodes":[338505531,5445808815]},{"type":"way","id":28921470,"nodes":[317930358,317930360,892620498,317930362,893489165,317930350,892620508,893431458,892620510]},{"type":"way","id":35787178,"nodes":[3140855379,3140855380,354671417,4013524087,3140855387]},{"type":"way","id":49797451,"nodes":[21564825,5787555029,323256856,338569568,21583361,10572786]},{"type":"way","id":49932212,"nodes":[25503707,3644748766,3519269750,25503708,6284930123,6354436150,965267525]},{"type":"way","id":56107800,"nodes":[4927544939,4927544940,4927544942,4927544944,4285841901,20599921,3142630784]},{"type":"way","id":75606036,"nodes":[317930344,3398910717,892620500]},{"type":"way","id":83592298,"nodes":[973525143,973525136]},{"type":"way","id":148065616,"nodes":[3140855379,338576378]},{"type":"way","id":148386203,"nodes":[3140855387,1302605368,4927544936,4927544938,4927544939]},{"type":"way","id":185312887,"nodes":[3142630784,3142630783,3140855378]},{"type":"way","id":421836546,"nodes":[4215728347,4215728363]},{"type":"way","id":421836548,"nodes":[4215728338,4215728357]},{"type":"way","id":421882362,"nodes":[4216274105,4216272873]},{"type":"way","id":421882366,"nodes":[4216272819,4216272817]},{"type":"way","id":421882367,"nodes":[4216274099,4216274120,4216273229,4216273232]},{"type":"way","id":565466197,"nodes":[5445808816,317930304,317930325]},{"type":"way","id":565466199,"nodes":[5445808815,5445808816]},{"type":"way","id":613394393,"nodes":[5803957067,5803957066]},{"type":"way","id":613394401,"nodes":[5817963325,5803957070,5803957071]},{"type":"way","id":613394407,"nodes":[5803957072,5803957073]},{"type":"way","id":613394416,"nodes":[5803957075,5803957076]},{"type":"way","id":613394425,"nodes":[5803957079,5803957080]},{"type":"way","id":613394428,"nodes":[5803957081,5803957082,5803957080]},{"type":"way","id":613394432,"nodes":[5803957080,5803957083,5803957084]},{"type":"way","id":613394436,"nodes":[5817963325,5803957085]},{"type":"way","id":613394439,"nodes":[5817963324,5803957086,5817963323]},{"type":"way","id":613394444,"nodes":[5803957089,5803957090]},{"type":"way","id":613394453,"nodes":[5803957093,5803957094]},{"type":"way","id":613394459,"nodes":[5803957096,5803957097]},{"type":"way","id":613406344,"nodes":[5804044776,5804044775]},{"type":"way","id":615235187,"nodes":[5803957063,5817963322]},{"type":"way","id":615235190,"nodes":[5817963327,5817963328]},{"type":"way","id":615235192,"nodes":[5817963330,5817963331]},{"type":"way","id":615235204,"nodes":[5817964155,5817964156]},{"type":"way","id":615235213,"nodes":[5817964165,5817964166,5817964167]},{"type":"way","id":615235215,"nodes":[5817964165,5817964170,5817964171]},{"type":"way","id":615635336,"nodes":[5821464938,5821464939,5803957060]},{"type":"way","id":671146173,"nodes":[965267525,108902]},{"type":"way","id":676541201,"nodes":[21563698,21564825]}]}'
    
    else:
        return '{  "version": 0.6,  "generator": "Overpass API 0.7.55.6 486819c8",  "osm3s": {    "timestamp_osm_base": "2019-04-19T22:43:02Z",    "copyright": "The data included in this document is from www.openstreetmap.org. The data is made available under ODbL."  },  "elements": [{  "type": "node",  "id": 246858445,  "lat": 40.7203353,  "lon": -74.0122755},{  "type": "node",  "id": 246891302,  "lat": 40.7204605,  "lon": -74.0122493},{  "type": "node",  "id": 42428417,  "lat": 40.7168710,  "lon": -74.0040260},{  "type": "node",  "id": 42428420,  "lat": 40.7174490,  "lon": -74.0035330},{  "type": "node",  "id": 42430126,  "lat": 40.7206388,  "lon": -74.0052198},{  "type": "node",  "id": 42434839,  "lat": 40.7203782,  "lon": -74.0059661},{  "type": "node",  "id": 42440825,  "lat": 40.7205570,  "lon": -74.0034834},{  "type": "node",  "id": 42447246,  "lat": 40.7200394,  "lon": -74.0039146},{  "type": "node",  "id": 42447249,  "lat": 40.7194841,  "lon": -74.0043702},{  "type": "node",  "id": 42453777,  "lat": 40.7188593,  "lon": -74.0048352},{  "type": "node",  "id": 4143851142,  "lat": 40.7179523,  "lon": -74.0055874},{  "type": "node",  "id": 4143851144,  "lat": 40.7182558,  "lon": -74.0052693},{  "type": "node",  "id": 4143851145,  "lat": 40.7181077,  "lon": -74.0053833},{  "type": "node",  "id": 4143859857,  "lat": 40.7180022,  "lon": -74.0054952},{  "type": "node",  "id": 4143859859,  "lat": 40.7180669,  "lon": -74.0054176},{  "type": "node",  "id": 4143859862,  "lat": 40.7180308,  "lon": -74.0054547},{  "type": "node",  "id": 4143859869,  "lat": 40.7179752,  "lon": -74.0055400},{  "type": "node",  "id": 4207150078,  "lat": 40.7167984,  "lon": -74.0040877},{  "type": "node",  "id": 4754110667,  "lat": 40.7173337,  "lon": -74.0036313},{  "type": "node",  "id": 4890621095,  "lat": 40.7208005,  "lon": -74.0050887},{  "type": "node",  "id": 5804849353,  "lat": 40.7204365,  "lon": -74.0059561},{  "type": "node",  "id": 42436021,  "lat": 40.7235239,  "lon": -74.0103391},{  "type": "node",  "id": 42436024,  "lat": 40.7236291,  "lon": -74.0113646},{  "type": "node",  "id": 42436313,  "lat": 40.7215308,  "lon": -74.0106804},{  "type": "node",  "id": 42436316,  "lat": 40.7216343,  "lon": -74.0117703},{  "type": "node",  "id": 42438060,  "lat": 40.7228729,  "lon": -74.0104496},{  "type": "node",  "id": 42440850,  "lat": 40.7253004,  "lon": -74.0100907},{  "type": "node",  "id": 42440854,  "lat": 40.7258067,  "lon": -74.0109300},{  "type": "node",  "id": 42440873,  "lat": 40.7250652,  "lon": -74.0100724},{  "type": "node",  "id": 42440916,  "lat": 40.7222194,  "lon": -74.0105578},{  "type": "node",  "id": 42440918,  "lat": 40.7223272,  "lon": -74.0116275},{  "type": "node",  "id": 42442291,  "lat": 40.7260808,  "lon": -74.0099015},{  "type": "node",  "id": 42442294,  "lat": 40.7261761,  "lon": -74.0108723},{  "type": "node",  "id": 42445171,  "lat": 40.7241745,  "lon": -74.0102257},{  "type": "node",  "id": 42445174,  "lat": 40.7242763,  "lon": -74.0112346},{  "type": "node",  "id": 246881559,  "lat": 40.7251489,  "lon": -74.0117691},{  "type": "node",  "id": 246881639,  "lat": 40.7250996,  "lon": -74.0113069},{  "type": "node",  "id": 246881640,  "lat": 40.7250747,  "lon": -74.0110732},{  "type": "node",  "id": 246881975,  "lat": 40.7244018,  "lon": -74.0114395},{  "type": "node",  "id": 246881976,  "lat": 40.7243766,  "lon": -74.0112144},{  "type": "node",  "id": 246882084,  "lat": 40.7242189,  "lon": -74.0114742},{  "type": "node",  "id": 246882085,  "lat": 40.7241935,  "lon": -74.0112514},{  "type": "node",  "id": 246882133,  "lat": 40.7229034,  "lon": -74.0117389},{  "type": "node",  "id": 246882134,  "lat": 40.7228793,  "lon": -74.0115172},{  "type": "node",  "id": 246882152,  "lat": 40.7224870,  "lon": -74.0122926},{  "type": "node",  "id": 246882203,  "lat": 40.7224421,  "lon": -74.0118352},{  "type": "node",  "id": 246882204,  "lat": 40.7224203,  "lon": -74.0116100},{  "type": "node",  "id": 246882215,  "lat": 40.7223061,  "lon": -74.0123275},{  "type": "node",  "id": 246882230,  "lat": 40.7222476,  "lon": -74.0118758},{  "type": "node",  "id": 246882231,  "lat": 40.7222210,  "lon": -74.0116494},{  "type": "node",  "id": 246889362,  "lat": 40.7257225,  "lon": -74.0112014},{  "type": "node",  "id": 246889572,  "lat": 40.7252103,  "lon": -74.0112853},{  "type": "node",  "id": 246890098,  "lat": 40.7223501,  "lon": -74.0118544},{  "type": "node",  "id": 246890279,  "lat": 40.7229994,  "lon": -74.0117188},{  "type": "node",  "id": 1701898668,  "lat": 40.7257810,  "lon": -74.0111019},{  "type": "node",  "id": 1701898669,  "lat": 40.7258036,  "lon": -74.0110174},{  "type": "node",  "id": 1701898682,  "lat": 40.7250071,  "lon": -74.0098511},{  "type": "node",  "id": 1701898687,  "lat": 40.7257943,  "lon": -74.0110628},{  "type": "node",  "id": 1701898699,  "lat": 40.7250479,  "lon": -74.0099926},{  "type": "node",  "id": 1701898705,  "lat": 40.7258074,  "lon": -74.0109692},{  "type": "node",  "id": 1701898709,  "lat": 40.7250294,  "lon": -74.0099220},{  "type": "node",  "id": 1701898730,  "lat": 40.7257406,  "lon": -74.0107127},{  "type": "node",  "id": 3196459695,  "lat": 40.7253392,  "lon": -74.0117297},{  "type": "node",  "id": 3196459700,  "lat": 40.7254474,  "lon": -74.0116350},{  "type": "node",  "id": 3196459701,  "lat": 40.7254460,  "lon": -74.0116151},{  "type": "node",  "id": 3196459708,  "lat": 40.7252922,  "lon": -74.0116828},{  "type": "node",  "id": 3196459709,  "lat": 40.7251822,  "lon": -74.0117038},{  "type": "node",  "id": 3196459710,  "lat": 40.7252449,  "lon": -74.0117492},{  "type": "node",  "id": 3196459711,  "lat": 40.7252391,  "lon": -74.0116930},{  "type": "node",  "id": 3196459867,  "lat": 40.7224477,  "lon": -74.0122624},{  "type": "node",  "id": 3196459868,  "lat": 40.7223437,  "lon": -74.0122839},{  "type": "node",  "id": 3196459869,  "lat": 40.7223994,  "lon": -74.0122724},{  "type": "node",  "id": 3196459870,  "lat": 40.7224036,  "lon": -74.0123087},{  "type": "node",  "id": 3316533531,  "lat": 40.7250876,  "lon": -74.0101590},{  "type": "node",  "id": 3316533532,  "lat": 40.7253929,  "lon": -74.0100228},{  "type": "node",  "id": 3748942433,  "lat": 40.7252568,  "lon": -74.0105408},{  "type": "node",  "id": 3748942434,  "lat": 40.7252582,  "lon": -74.0105032},{  "type": "node",  "id": 3748942435,  "lat": 40.7252582,  "lon": -74.0106179},{  "type": "node",  "id": 3748942436,  "lat": 40.7252626,  "lon": -74.0106606},{  "type": "node",  "id": 3748942437,  "lat": 40.7252664,  "lon": -74.0104969},{  "type": "node",  "id": 3748942438,  "lat": 40.7252693,  "lon": -74.0104542},{  "type": "node",  "id": 3748942439,  "lat": 40.7252693,  "lon": -74.0106944},{  "type": "node",  "id": 3748942440,  "lat": 40.7252771,  "lon": -74.0103573},{  "type": "node",  "id": 3748942441,  "lat": 40.7252795,  "lon": -74.0107186},{  "type": "node",  "id": 3748942442,  "lat": 40.7252882,  "lon": -74.0103159},{  "type": "node",  "id": 3748942443,  "lat": 40.7252964,  "lon": -74.0107467},{  "type": "node",  "id": 3748942444,  "lat": 40.7253100,  "lon": -74.0102631},{  "type": "node",  "id": 3748942445,  "lat": 40.7253128,  "lon": -74.0107741},{  "type": "node",  "id": 3748942446,  "lat": 40.7253150,  "lon": -74.0104676},{  "type": "node",  "id": 3748942447,  "lat": 40.7253157,  "lon": -74.0104344},{  "type": "node",  "id": 3748942448,  "lat": 40.7253191,  "lon": -74.0103745},{  "type": "node",  "id": 3748942449,  "lat": 40.7253254,  "lon": -74.0104669},{  "type": "node",  "id": 3748942450,  "lat": 40.7253258,  "lon": -74.0108066},{  "type": "node",  "id": 3748942451,  "lat": 40.7253275,  "lon": -74.0105077},{  "type": "node",  "id": 3748942452,  "lat": 40.7253360,  "lon": -74.0105529},{  "type": "node",  "id": 3748942453,  "lat": 40.7253379,  "lon": -74.0108461},{  "type": "node",  "id": 3748942454,  "lat": 40.7253389,  "lon": -74.0103052},{  "type": "node",  "id": 3748942455,  "lat": 40.7253457,  "lon": -74.0108997},{  "type": "node",  "id": 3748942456,  "lat": 40.7253554,  "lon": -74.0101749},{  "type": "node",  "id": 3748942457,  "lat": 40.7253577,  "lon": -74.0106128},{  "type": "node",  "id": 3748942459,  "lat": 40.7253801,  "lon": -74.0102092},{  "type": "node",  "id": 3748942460,  "lat": 40.7253848,  "lon": -74.0106638},{  "type": "node",  "id": 3748942461,  "lat": 40.7254142,  "lon": -74.0107046},{  "type": "node",  "id": 3748942462,  "lat": 40.7254471,  "lon": -74.0107377},{  "type": "node",  "id": 3748942463,  "lat": 40.7254871,  "lon": -74.0107830},{  "type": "node",  "id": 3748942464,  "lat": 40.7255094,  "lon": -74.0108187},{  "type": "node",  "id": 3748942465,  "lat": 40.7255233,  "lon": -74.0108639},{  "type": "node",  "id": 3874650168,  "lat": 40.7223168,  "lon": -74.0115237},{  "type": "node",  "id": 3874650174,  "lat": 40.7229632,  "lon": -74.0113692},{  "type": "node",  "id": 3874650177,  "lat": 40.7236155,  "lon": -74.0112320},{  "type": "node",  "id": 3874650180,  "lat": 40.7242627,  "lon": -74.0111079},{  "type": "node",  "id": 3874650183,  "lat": 40.7251664,  "lon": -74.0109277},{  "type": "node",  "id": 4141586731,  "lat": 40.7257806,  "lon": -74.0107932},{  "type": "node",  "id": 4141586733,  "lat": 40.7268827,  "lon": -74.0107716},{  "type": "node",  "id": 4141586734,  "lat": 40.7257630,  "lon": -74.0107503},{  "type": "node",  "id": 4141586735,  "lat": 40.7269928,  "lon": -74.0107609},{  "type": "node",  "id": 4141586736,  "lat": 40.7257925,  "lon": -74.0108344},{  "type": "node",  "id": 4141586737,  "lat": 40.7267742,  "lon": -74.0107840},{  "type": "node",  "id": 4141593653,  "lat": 40.7272444,  "lon": -74.0109768},{  "type": "node",  "id": 4141593654,  "lat": 40.7273528,  "lon": -74.0109643},{  "type": "node",  "id": 4141593655,  "lat": 40.7271336,  "lon": -74.0109912},{  "type": "node",  "id": 4142073860,  "lat": 40.7259771,  "lon": -74.0111586},{  "type": "node",  "id": 4142073861,  "lat": 40.7257466,  "lon": -74.0111678},{  "type": "node",  "id": 4142073862,  "lat": 40.7258010,  "lon": -74.0108784},{  "type": "node",  "id": 4142073863,  "lat": 40.7270219,  "lon": -74.0110076},{  "type": "node",  "id": 4142073864,  "lat": 40.7257650,  "lon": -74.0111379},{  "type": "node",  "id": 4142105820,  "lat": 40.7253366,  "lon": -74.0100532},{  "type": "node",  "id": 4142105821,  "lat": 40.7253660,  "lon": -74.0100319},{  "type": "node",  "id": 4142105827,  "lat": 40.7241895,  "lon": -74.0103858},{  "type": "node",  "id": 4143812553,  "lat": 40.7216671,  "lon": -74.0119908},{  "type": "node",  "id": 4207143421,  "lat": 40.7229769,  "lon": -74.0114958},{  "type": "node",  "id": 4207143422,  "lat": 40.7235721,  "lon": -74.0116039},{  "type": "node",  "id": 4207143428,  "lat": 40.7251813,  "lon": -74.0110507},{  "type": "node",  "id": 4207143432,  "lat": 40.7255580,  "lon": -74.0104576},{  "type": "node",  "id": 4207143435,  "lat": 40.7264727,  "lon": -74.0110870},{  "type": "node",  "id": 4629486881,  "lat": 40.7252315,  "lon": -74.0099820},{  "type": "node",  "id": 4872871134,  "lat": 40.7249727,  "lon": -74.0100883},{  "type": "node",  "id": 5366406002,  "lat": 40.7214942,  "lon": -74.0098779},{  "type": "node",  "id": 5708245833,  "lat": 40.7261603,  "lon": -74.0107116},{  "type": "node",  "id": 42423456,  "lat": 40.7237914,  "lon": -74.0061130},{  "type": "node",  "id": 42424401,  "lat": 40.7216450,  "lon": -74.0064375},{  "type": "node",  "id": 42424408,  "lat": 40.7219848,  "lon": -74.0082309},{  "type": "node",  "id": 42424619,  "lat": 40.7237578,  "lon": -74.0079297},{  "type": "node",  "id": 42424813,  "lat": 40.7218618,  "lon": -74.0071010},{  "type": "node",  "id": 42424864,  "lat": 40.7244132,  "lon": -74.0059302},{  "type": "node",  "id": 42426060,  "lat": 40.7239092,  "lon": -74.0075216},{  "type": "node",  "id": 42426089,  "lat": 40.7234376,  "lon": -74.0075182},{  "type": "node",  "id": 42430102,  "lat": 40.7255633,  "lon": -74.0045513},{  "type": "node",  "id": 42430108,  "lat": 40.7242670,  "lon": -74.0047390},{  "type": "node",  "id": 42430131,  "lat": 40.7210250,  "lon": -74.0052480},{  "type": "node",  "id": 42430136,  "lat": 40.7222274,  "lon": -74.0053546},{  "type": "node",  "id": 42430143,  "lat": 40.7236250,  "lon": -74.0047860},{  "type": "node",  "id": 42430147,  "lat": 40.7241730,  "lon": -74.0045750},{  "type": "node",  "id": 42430151,  "lat": 40.7248717,  "lon": -74.0042981},{  "type": "node",  "id": 42430154,  "lat": 40.7255130,  "lon": -74.0040310},{  "type": "node",  "id": 42430160,  "lat": 40.7262500,  "lon": -74.0037300},{  "type": "node",  "id": 42430164,  "lat": 40.7269580,  "lon": -74.0034450},{  "type": "node",  "id": 42432116,  "lat": 40.7245757,  "lon": -74.0077807},{  "type": "node",  "id": 42432127,  "lat": 40.7240550,  "lon": -74.0043800},{  "type": "node",  "id": 42432135,  "lat": 40.7236820,  "lon": -74.0036350},{  "type": "node",  "id": 42433573,  "lat": 40.7208687,  "lon": -74.0039184},{  "type": "node",  "id": 42436014,  "lat": 40.7232849,  "lon": -74.0080097},{  "type": "node",  "id": 42436017,  "lat": 40.7234416,  "lon": -74.0095375},{  "type": "node",  "id": 42436064,  "lat": 40.7221396,  "lon": -74.0097659},{  "type": "node",  "id": 42436069,  "lat": 40.7227945,  "lon": -74.0096510},{  "type": "node",  "id": 42436073,  "lat": 40.7241027,  "lon": -74.0094239},{  "type": "node",  "id": 42436083,  "lat": 40.7247725,  "lon": -74.0092994},{  "type": "node",  "id": 42436088,  "lat": 40.7260025,  "lon": -74.0090917},{  "type": "node",  "id": 42436093,  "lat": 40.7267460,  "lon": -74.0089670},{  "type": "node",  "id": 42436308,  "lat": 40.7213301,  "lon": -74.0083433},{  "type": "node",  "id": 42436340,  "lat": 40.7226441,  "lon": -74.0081187},{  "type": "node",  "id": 42436353,  "lat": 40.7251920,  "lon": -74.0076690},{  "type": "node",  "id": 42436355,  "lat": 40.7258538,  "lon": -74.0075547},{  "type": "node",  "id": 42436359,  "lat": 40.7266010,  "lon": -74.0074233},{  "type": "node",  "id": 42436364,  "lat": 40.7273280,  "lon": -74.0073052},{  "type": "node",  "id": 42436567,  "lat": 40.7250234,  "lon": -74.0058148},{  "type": "node",  "id": 42436761,  "lat": 40.7220398,  "lon": -74.0050505},{  "type": "node",  "id": 42436763,  "lat": 40.7227417,  "lon": -74.0044354},{  "type": "node",  "id": 42436770,  "lat": 40.7234490,  "lon": -74.0038300},{  "type": "node",  "id": 42438779,  "lat": 40.7210830,  "lon": -74.0058450},{  "type": "node",  "id": 42438781,  "lat": 40.7217538,  "lon": -74.0057181},{  "type": "node",  "id": 42439340,  "lat": 40.7269410,  "lon": -74.0032780},{  "type": "node",  "id": 42439343,  "lat": 40.7269534,  "lon": -74.0033735},{  "type": "node",  "id": 42439345,  "lat": 40.7271570,  "lon": -74.0055450},{  "type": "node",  "id": 42439524,  "lat": 40.7241840,  "lon": -74.0042720},{  "type": "node",  "id": 42439527,  "lat": 40.7252520,  "lon": -74.0033760},{  "type": "node",  "id": 42439550,  "lat": 40.7222803,  "lon": -74.0059159},{  "type": "node",  "id": 42439551,  "lat": 40.7230395,  "lon": -74.0052482},{  "type": "node",  "id": 42440829,  "lat": 40.7213697,  "lon": -74.0046175},{  "type": "node",  "id": 42440838,  "lat": 40.7225596,  "lon": -74.0063091},{  "type": "node",  "id": 42440847,  "lat": 40.7243115,  "lon": -74.0086756},{  "type": "node",  "id": 42440913,  "lat": 40.7220680,  "lon": -74.0090561},{  "type": "node",  "id": 42442282,  "lat": 40.7255004,  "lon": -74.0038894},{  "type": "node",  "id": 42442286,  "lat": 40.7256753,  "lon": -74.0057082},{  "type": "node",  "id": 42442288,  "lat": 40.7259304,  "lon": -74.0083464},{  "type": "node",  "id": 42445574,  "lat": 40.7264350,  "lon": -74.0056580},{  "type": "node",  "id": 42452015,  "lat": 40.7231321,  "lon": -74.0062154},{  "type": "node",  "id": 42452026,  "lat": 40.7223840,  "lon": -74.0037680},{  "type": "node",  "id": 246884508,  "lat": 40.7218794,  "lon": -74.0067853},{  "type": "node",  "id": 246885162,  "lat": 40.7224747,  "lon": -74.0067245},{  "type": "node",  "id": 246886236,  "lat": 40.7224022,  "lon": -74.0066881},{  "type": "node",  "id": 246886237,  "lat": 40.7224308,  "lon": -74.0066965},{  "type": "node",  "id": 246886238,  "lat": 40.7224534,  "lon": -74.0067065},{  "type": "node",  "id": 562073774,  "lat": 40.7217674,  "lon": -74.0069814},{  "type": "node",  "id": 562073788,  "lat": 40.7215506,  "lon": -74.0066882},{  "type": "node",  "id": 592484776,  "lat": 40.7249131,  "lon": -74.0046459},{  "type": "node",  "id": 1104165608,  "lat": 40.7211966,  "lon": -74.0083678},{  "type": "node",  "id": 1701898651,  "lat": 40.7219961,  "lon": -74.0067185},{  "type": "node",  "id": 1701898678,  "lat": 40.7223582,  "lon": -74.0066805},{  "type": "node",  "id": 1701898690,  "lat": 40.7223028,  "lon": -74.0066758},{  "type": "node",  "id": 1701898695,  "lat": 40.7222546,  "lon": -74.0066766},{  "type": "node",  "id": 1701898701,  "lat": 40.7249238,  "lon": -74.0096653},{  "type": "node",  "id": 1701898704,  "lat": 40.7220318,  "lon": -74.0067103},{  "type": "node",  "id": 1701898714,  "lat": 40.7218983,  "lon": -74.0067651},{  "type": "node",  "id": 1701898723,  "lat": 40.7219251,  "lon": -74.0067462},{  "type": "node",  "id": 2151385470,  "lat": 40.7209116,  "lon": -74.0039782},{  "type": "node",  "id": 2151385472,  "lat": 40.7209574,  "lon": -74.0038422},{  "type": "node",  "id": 2151385487,  "lat": 40.7212566,  "lon": -74.0047117},{  "type": "node",  "id": 2151385490,  "lat": 40.7213143,  "lon": -74.0045402},{  "type": "node",  "id": 2151385497,  "lat": 40.7214352,  "lon": -74.0047146},{  "type": "node",  "id": 2151385500,  "lat": 40.7214627,  "lon": -74.0045435},{  "type": "node",  "id": 2151385504,  "lat": 40.7217590,  "lon": -74.0051786},{  "type": "node",  "id": 2151385505,  "lat": 40.7217295,  "lon": -74.0054398},{  "type": "node",  "id": 2151385506,  "lat": 40.7219805,  "lon": -74.0054938},{  "type": "node",  "id": 2151385507,  "lat": 40.7220726,  "lon": -74.0053863},{  "type": "node",  "id": 2477507992,  "lat": 40.7220988,  "lon": -74.0049921},{  "type": "node",  "id": 2477560540,  "lat": 40.7264452,  "lon": -74.0035410},{  "type": "node",  "id": 2915066906,  "lat": 40.7216216,  "lon": -74.0066097},{  "type": "node",  "id": 2915066907,  "lat": 40.7217797,  "lon": -74.0071130},{  "type": "node",  "id": 3252582131,  "lat": 40.7263055,  "lon": -74.0056109},{  "type": "node",  "id": 3252582132,  "lat": 40.7264217,  "lon": -74.0055199},{  "type": "node",  "id": 3252582133,  "lat": 40.7264490,  "lon": -74.0058082},{  "type": "node",  "id": 3252582134,  "lat": 40.7265128,  "lon": -74.0056458},{  "type": "node",  "id": 3316533533,  "lat": 40.7249786,  "lon": -74.0095890},{  "type": "node",  "id": 3546151526,  "lat": 40.7243133,  "lon": -74.0049570},{  "type": "node",  "id": 3546151527,  "lat": 40.7242913,  "lon": -74.0048191},{  "type": "node",  "id": 3567807957,  "lat": 40.7210964,  "lon": -74.0065527},{  "type": "node",  "id": 3788364698,  "lat": 40.7237329,  "lon": -74.0058495},{  "type": "node",  "id": 3788364700,  "lat": 40.7237441,  "lon": -74.0059300},{  "type": "node",  "id": 3788364701,  "lat": 40.7237622,  "lon": -74.0060099},{  "type": "node",  "id": 3788379684,  "lat": 40.7219830,  "lon": -74.0081915},{  "type": "node",  "id": 3788379685,  "lat": 40.7219844,  "lon": -74.0081489},{  "type": "node",  "id": 3788379686,  "lat": 40.7219852,  "lon": -74.0080912},{  "type": "node",  "id": 3788379729,  "lat": 40.7229508,  "lon": -74.0048836},{  "type": "node",  "id": 3788379730,  "lat": 40.7229215,  "lon": -74.0048080},{  "type": "node",  "id": 3788379732,  "lat": 40.7219836,  "lon": -74.0053789},{  "type": "node",  "id": 3788379733,  "lat": 40.7218880,  "lon": -74.0053637},{  "type": "node",  "id": 4136469550,  "lat": 40.7219829,  "lon": -74.0080405},{  "type": "node",  "id": 4136469551,  "lat": 40.7218632,  "lon": -74.0068081},{  "type": "node",  "id": 4136469552,  "lat": 40.7219200,  "lon": -74.0076796},{  "type": "node",  "id": 4136469553,  "lat": 40.7219581,  "lon": -74.0067307},{  "type": "node",  "id": 4136469554,  "lat": 40.7219277,  "lon": -74.0077405},{  "type": "node",  "id": 4136469555,  "lat": 40.7219740,  "lon": -74.0079461},{  "type": "node",  "id": 4136469556,  "lat": 40.7218435,  "lon": -74.0069106},{  "type": "node",  "id": 4136469557,  "lat": 40.7218510,  "lon": -74.0068374},{  "type": "node",  "id": 4136469558,  "lat": 40.7219657,  "lon": -74.0078952},{  "type": "node",  "id": 4136469559,  "lat": 40.7218442,  "lon": -74.0068696},{  "type": "node",  "id": 4136469560,  "lat": 40.7219384,  "lon": -74.0077957},{  "type": "node",  "id": 4136476380,  "lat": 40.7222015,  "lon": -74.0066821},{  "type": "node",  "id": 4136500817,  "lat": 40.7225191,  "lon": -74.0068462},{  "type": "node",  "id": 4136500818,  "lat": 40.7225057,  "lon": -74.0067702},{  "type": "node",  "id": 4136500819,  "lat": 40.7225144,  "lon": -74.0068034},{  "type": "node",  "id": 4136500820,  "lat": 40.7224915,  "lon": -74.0067447},{  "type": "node",  "id": 4136508181,  "lat": 40.7221797,  "lon": -74.0053673},{  "type": "node",  "id": 4136508184,  "lat": 40.7221176,  "lon": -74.0053809},{  "type": "node",  "id": 4139510037,  "lat": 40.7244917,  "lon": -74.0068861},{  "type": "node",  "id": 4139510041,  "lat": 40.7238049,  "lon": -74.0065257},{  "type": "node",  "id": 4142105791,  "lat": 40.7231841,  "lon": -74.0071539},{  "type": "node",  "id": 4142105792,  "lat": 40.7248879,  "lon": -74.0096126},{  "type": "node",  "id": 4142105795,  "lat": 40.7249837,  "lon": -74.0097860},{  "type": "node",  "id": 4142105796,  "lat": 40.7243621,  "lon": -74.0086327},{  "type": "node",  "id": 4142105797,  "lat": 40.7245729,  "lon": -74.0091795},{  "type": "node",  "id": 4142105799,  "lat": 40.7240498,  "lon": -74.0087744},{  "type": "node",  "id": 4142105800,  "lat": 40.7244717,  "lon": -74.0088835},{  "type": "node",  "id": 4142105801,  "lat": 40.7243936,  "lon": -74.0086182},{  "type": "node",  "id": 4142105802,  "lat": 40.7250700,  "lon": -74.0097275},{  "type": "node",  "id": 4142105803,  "lat": 40.7231293,  "lon": -74.0070839},{  "type": "node",  "id": 4142105805,  "lat": 40.7240638,  "lon": -74.0090026},{  "type": "node",  "id": 4142105806,  "lat": 40.7241741,  "lon": -74.0084958},{  "type": "node",  "id": 4142105809,  "lat": 40.7230846,  "lon": -74.0071203},{  "type": "node",  "id": 4142105810,  "lat": 40.7240432,  "lon": -74.0086818},{  "type": "node",  "id": 4142105811,  "lat": 40.7240219,  "lon": -74.0084406},{  "type": "node",  "id": 4142105812,  "lat": 40.7246766,  "lon": -74.0093172},{  "type": "node",  "id": 4142105813,  "lat": 40.7228854,  "lon": -74.0067677},{  "type": "node",  "id": 4142105814,  "lat": 40.7243346,  "lon": -74.0086518},{  "type": "node",  "id": 4142105815,  "lat": 40.7244196,  "lon": -74.0088134},{  "type": "node",  "id": 4142105816,  "lat": 40.7244313,  "lon": -74.0086075},{  "type": "node",  "id": 4142105817,  "lat": 40.7231822,  "lon": -74.0072765},{  "type": "node",  "id": 4142105818,  "lat": 40.7249550,  "lon": -74.0097227},{  "type": "node",  "id": 4142105819,  "lat": 40.7231317,  "lon": -74.0071979},{  "type": "node",  "id": 4142105822,  "lat": 40.7236541,  "lon": -74.0079472},{  "type": "node",  "id": 4142105823,  "lat": 40.7242199,  "lon": -74.0085571},{  "type": "node",  "id": 4142105824,  "lat": 40.7240580,  "lon": -74.0089167},{  "type": "node",  "id": 4142105828,  "lat": 40.7250255,  "lon": -74.0096582},{  "type": "node",  "id": 4143859865,  "lat": 40.7217139,  "lon": -74.0053281},{  "type": "node",  "id": 4143859866,  "lat": 40.7217363,  "lon": -74.0055245},{  "type": "node",  "id": 4143859873,  "lat": 40.7216796,  "lon": -74.0050647},{  "type": "node",  "id": 4207143424,  "lat": 40.7241763,  "lon": -74.0086477},{  "type": "node",  "id": 4207143426,  "lat": 40.7245185,  "lon": -74.0089482},{  "type": "node",  "id": 4207143427,  "lat": 40.7251113,  "lon": -74.0097926},{  "type": "node",  "id": 4215665529,  "lat": 40.7229860,  "lon": -74.0062383},{  "type": "node",  "id": 4215665530,  "lat": 40.7230894,  "lon": -74.0070322},{  "type": "node",  "id": 4387367047,  "lat": 40.7224246,  "lon": -74.0046876},{  "type": "node",  "id": 4560056040,  "lat": 40.7257750,  "lon": -74.0067426},{  "type": "node",  "id": 4602410642,  "lat": 40.7238371,  "lon": -74.0068328},{  "type": "node",  "id": 4602410644,  "lat": 40.7238826,  "lon": -74.0072677},{  "type": "node",  "id": 4602414021,  "lat": 40.7237608,  "lon": -74.0061172},{  "type": "node",  "id": 4602429319,  "lat": 40.7238557,  "lon": -74.0079119},{  "type": "node",  "id": 4602429320,  "lat": 40.7235580,  "lon": -74.0078111},{  "type": "node",  "id": 4602429322,  "lat": 40.7238372,  "lon": -74.0080314},{  "type": "node",  "id": 4602429325,  "lat": 40.7235171,  "lon": -74.0076225},{  "type": "node",  "id": 4602429327,  "lat": 40.7236141,  "lon": -74.0077465},{  "type": "node",  "id": 4602429329,  "lat": 40.7239530,  "lon": -74.0078914},{  "type": "node",  "id": 4602429333,  "lat": 40.7237334,  "lon": -74.0080515},{  "type": "node",  "id": 4602429335,  "lat": 40.7234746,  "lon": -74.0076928},{  "type": "node",  "id": 4616536348,  "lat": 40.7244231,  "lon": -74.0060550},{  "type": "node",  "id": 4616536349,  "lat": 40.7238386,  "lon": -74.0060397},{  "type": "node",  "id": 4616536350,  "lat": 40.7263219,  "lon": -74.0057596},{  "type": "node",  "id": 4616536351,  "lat": 40.7256907,  "lon": -74.0058541},{  "type": "node",  "id": 4616536352,  "lat": 40.7250366,  "lon": -74.0059578},{  "type": "node",  "id": 4616536353,  "lat": 40.7243487,  "lon": -74.0060445},{  "type": "node",  "id": 4616536354,  "lat": 40.7238653,  "lon": -74.0061272},{  "type": "node",  "id": 4629486875,  "lat": 40.7234208,  "lon": -74.0078877},{  "type": "node",  "id": 4629486876,  "lat": 40.7234353,  "lon": -74.0078733},{  "type": "node",  "id": 4629486877,  "lat": 40.7234522,  "lon": -74.0078430},{  "type": "node",  "id": 4629486878,  "lat": 40.7234617,  "lon": -74.0078162},{  "type": "node",  "id": 4629486879,  "lat": 40.7234955,  "lon": -74.0077224},{  "type": "node",  "id": 4676469047,  "lat": 40.7229262,  "lon": -74.0033067},{  "type": "node",  "id": 4778174559,  "lat": 40.7224205,  "lon": -74.0063224},{  "type": "node",  "id": 4778174560,  "lat": 40.7224766,  "lon": -74.0061922},{  "type": "node",  "id": 4778174561,  "lat": 40.7227040,  "lon": -74.0062824},{  "type": "node",  "id": 4778174562,  "lat": 40.7226542,  "lon": -74.0064422},{  "type": "node",  "id": 4778174563,  "lat": 40.7216947,  "lon": -74.0051794},{  "type": "node",  "id": 4778174564,  "lat": 40.7218157,  "lon": -74.0064128},{  "type": "node",  "id": 4872871121,  "lat": 40.7247434,  "lon": -74.0094063},{  "type": "node",  "id": 4872871132,  "lat": 40.7248436,  "lon": -74.0093971},{  "type": "node",  "id": 4890320280,  "lat": 40.7237029,  "lon": -74.0047567},{  "type": "node",  "id": 4890320281,  "lat": 40.7236003,  "lon": -74.0046519},{  "type": "node",  "id": 4890320282,  "lat": 40.7240295,  "lon": -74.0043294},{  "type": "node",  "id": 4890320283,  "lat": 40.7236399,  "lon": -74.0049331},{  "type": "node",  "id": 4890320289,  "lat": 40.7241156,  "lon": -74.0043300},{  "type": "node",  "id": 4890320290,  "lat": 40.7243594,  "lon": -74.0047256},{  "type": "node",  "id": 4890320291,  "lat": 40.7242692,  "lon": -74.0045370},{  "type": "node",  "id": 4890320293,  "lat": 40.7254514,  "lon": -74.0040567},{  "type": "node",  "id": 4890320296,  "lat": 40.7255712,  "lon": -74.0046328},{  "type": "node",  "id": 4890320299,  "lat": 40.7249810,  "lon": -74.0046360},{  "type": "node",  "id": 4890320302,  "lat": 40.7255902,  "lon": -74.0039997},{  "type": "node",  "id": 4890320304,  "lat": 40.7255324,  "lon": -74.0042309},{  "type": "node",  "id": 4890320305,  "lat": 40.7240064,  "lon": -74.0044261},{  "type": "node",  "id": 4890320314,  "lat": 40.7254978,  "lon": -74.0045608},{  "type": "node",  "id": 4890320315,  "lat": 40.7249209,  "lon": -74.0047294},{  "type": "node",  "id": 4890320316,  "lat": 40.7236959,  "lon": -74.0047204},{  "type": "node",  "id": 4890320319,  "lat": 40.7242414,  "lon": -74.0046929},{  "type": "node",  "id": 4890320320,  "lat": 40.7248552,  "lon": -74.0046556},{  "type": "node",  "id": 4890333632,  "lat": 40.7251077,  "lon": -74.0059465},{  "type": "node",  "id": 4890333634,  "lat": 40.7250948,  "lon": -74.0058031},{  "type": "node",  "id": 4890333635,  "lat": 40.7243131,  "lon": -74.0060385},{  "type": "node",  "id": 4890333636,  "lat": 40.7249575,  "lon": -74.0058273},{  "type": "node",  "id": 4890333638,  "lat": 40.7250439,  "lon": -74.0060381},{  "type": "node",  "id": 4890333641,  "lat": 40.7244853,  "lon": -74.0060451},{  "type": "node",  "id": 4890333642,  "lat": 40.7249705,  "lon": -74.0059682},{  "type": "node",  "id": 4890333643,  "lat": 40.7250169,  "lon": -74.0057455},{  "type": "node",  "id": 4890333647,  "lat": 40.7244316,  "lon": -74.0061583},{  "type": "node",  "id": 4890333652,  "lat": 40.7244056,  "lon": -74.0058566},{  "type": "node",  "id": 4890333653,  "lat": 40.7243040,  "lon": -74.0059510},{  "type": "node",  "id": 4890333654,  "lat": 40.7244737,  "lon": -74.0059188},{  "type": "node",  "id": 4890333656,  "lat": 40.7256047,  "lon": -74.0057197},{  "type": "node",  "id": 4890418800,  "lat": 40.7223815,  "lon": -74.0058271},{  "type": "node",  "id": 4890418814,  "lat": 40.7216107,  "lon": -74.0053154},{  "type": "node",  "id": 4890419521,  "lat": 40.7218058,  "lon": -74.0063016},{  "type": "node",  "id": 4890535524,  "lat": 40.7223126,  "lon": -74.0063385},{  "type": "node",  "id": 4890553386,  "lat": 40.7216706,  "lon": -74.0057338},{  "type": "node",  "id": 4890621100,  "lat": 40.7210381,  "lon": -74.0053833},{  "type": "node",  "id": 4890633688,  "lat": 40.7231903,  "lon": -74.0062063},{  "type": "node",  "id": 4890633690,  "lat": 40.7237079,  "lon": -74.0061255},{  "type": "node",  "id": 4890633694,  "lat": 40.7229496,  "lon": -74.0053273},{  "type": "node",  "id": 4890633695,  "lat": 40.7230818,  "lon": -74.0062210},{  "type": "node",  "id": 4890633697,  "lat": 40.7237529,  "lon": -74.0059688},{  "type": "node",  "id": 4890633699,  "lat": 40.7229610,  "lon": -74.0049189},{  "type": "node",  "id": 4890633706,  "lat": 40.7235436,  "lon": -74.0048193},{  "type": "node",  "id": 4890633707,  "lat": 40.7231191,  "lon": -74.0060797},{  "type": "node",  "id": 4890633710,  "lat": 40.7230286,  "lon": -74.0051934},{  "type": "node",  "id": 4890633717,  "lat": 40.7221621,  "lon": -74.0052488},{  "type": "node",  "id": 4890633719,  "lat": 40.7237739,  "lon": -74.0062381},{  "type": "node",  "id": 4890633720,  "lat": 40.7238548,  "lon": -74.0060366},{  "type": "node",  "id": 4892579962,  "lat": 40.7248374,  "lon": -74.0092884},{  "type": "node",  "id": 4892579966,  "lat": 40.7246115,  "lon": -74.0092306},{  "type": "node",  "id": 4892579967,  "lat": 40.7240370,  "lon": -74.0086120},{  "type": "node",  "id": 4892579968,  "lat": 40.7247113,  "lon": -74.0092121},{  "type": "node",  "id": 5135800965,  "lat": 40.7257139,  "lon": -74.0039491},{  "type": "node",  "id": 5135800967,  "lat": 40.7244857,  "lon": -74.0044512},{  "type": "node",  "id": 5135824727,  "lat": 40.7232923,  "lon": -74.0049328},{  "type": "node",  "id": 5135824729,  "lat": 40.7225214,  "lon": -74.0052551},{  "type": "node",  "id": 5135896922,  "lat": 40.7271533,  "lon": -74.0033648},{  "type": "node",  "id": 5155817221,  "lat": 40.7211425,  "lon": -74.0052615},{  "type": "node",  "id": 5173374390,  "lat": 40.7256246,  "lon": -74.0058646},{  "type": "node",  "id": 5173374392,  "lat": 40.7256675,  "lon": -74.0056278},{  "type": "node",  "id": 5173374393,  "lat": 40.7256968,  "lon": -74.0059182},{  "type": "node",  "id": 5216470312,  "lat": 40.7240961,  "lon": -74.0046040},{  "type": "node",  "id": 5216470316,  "lat": 40.7230671,  "lon": -74.0050372},{  "type": "node",  "id": 5216470727,  "lat": 40.7230023,  "lon": -74.0050623},{  "type": "node",  "id": 5216470728,  "lat": 40.7229144,  "lon": -74.0050987},{  "type": "node",  "id": 5708229593,  "lat": 40.7235735,  "lon": -74.0079608},{  "type": "node",  "id": 5708230097,  "lat": 40.7241010,  "lon": -74.0044559},{  "type": "node",  "id": 5708233911,  "lat": 40.7238753,  "lon": -74.0061252},{  "type": "node",  "id": 5708239345,  "lat": 40.7245570,  "lon": -74.0093394},{  "type": "node",  "id": 42435434,  "lat": 40.7139472,  "lon": -73.9995380},{  "type": "node",  "id": 588455744,  "lat": 40.7136598,  "lon": -73.9986598},{  "type": "node",  "id": 1773055865,  "lat": 40.7135233,  "lon": -73.9986438},{  "type": "node",  "id": 42427316,  "lat": 40.7180170,  "lon": -73.9999560},{  "type": "node",  "id": 42427319,  "lat": 40.7186723,  "lon": -73.9994292},{  "type": "node",  "id": 42427322,  "lat": 40.7189130,  "lon": -73.9992340},{  "type": "node",  "id": 42427324,  "lat": 40.7197860,  "lon": -73.9985240},{  "type": "node",  "id": 42428425,  "lat": 40.7180338,  "lon": -74.0030430},{  "type": "node",  "id": 42428428,  "lat": 40.7186389,  "lon": -74.0025285},{  "type": "node",  "id": 42428431,  "lat": 40.7191530,  "lon": -74.0020910},{  "type": "node",  "id": 42428433,  "lat": 40.7194058,  "lon": -74.0018769},{  "type": "node",  "id": 42428434,  "lat": 40.7199380,  "lon": -74.0014250},{  "type": "node",  "id": 42432171,  "lat": 40.7207050,  "lon": -73.9973490},{  "type": "node",  "id": 42432174,  "lat": 40.7204864,  "lon": -73.9968212},{  "type": "node",  "id": 42432238,  "lat": 40.7145118,  "lon": -73.9987971},{  "type": "node",  "id": 42432241,  "lat": 40.7149950,  "lon": -73.9988660},{  "type": "node",  "id": 42432244,  "lat": 40.7156303,  "lon": -73.9984926},{  "type": "node",  "id": 42432246,  "lat": 40.7168256,  "lon": -73.9977531},{  "type": "node",  "id": 42432250,  "lat": 40.7178782,  "lon": -73.9971031},{  "type": "node",  "id": 42432255,  "lat": 40.7181330,  "lon": -73.9969360},{  "type": "node",  "id": 42432260,  "lat": 40.7184170,  "lon": -73.9967710},{  "type": "node",  "id": 42435435,  "lat": 40.7144649,  "lon": -73.9995926},{  "type": "node",  "id": 42435440,  "lat": 40.7151670,  "lon": -73.9996530},{  "type": "node",  "id": 42435448,  "lat": 40.7159173,  "lon": -73.9993214},{  "type": "node",  "id": 42435451,  "lat": 40.7171225,  "lon": -73.9986047},{  "type": "node",  "id": 42435456,  "lat": 40.7181633,  "lon": -73.9979382},{  "type": "node",  "id": 42435461,  "lat": 40.7184700,  "lon": -73.9977400},{  "type": "node",  "id": 42435465,  "lat": 40.7187940,  "lon": -73.9975590},{  "type": "node",  "id": 42435468,  "lat": 40.7191690,  "lon": -73.9973730},{  "type": "node",  "id": 42435470,  "lat": 40.7193300,  "lon": -73.9973036},{  "type": "node",  "id": 42438034,  "lat": 40.7153613,  "lon": -73.9977158},{  "type": "node",  "id": 42439207,  "lat": 40.7195980,  "lon": -73.9980891},{  "type": "node",  "id": 42439580,  "lat": 40.7191930,  "lon": -73.9998410},{  "type": "node",  "id": 42439581,  "lat": 40.7195520,  "lon": -74.0006220},{  "type": "node",  "id": 42439583,  "lat": 40.7203410,  "lon": -74.0022210},{  "type": "node",  "id": 42440804,  "lat": 40.7165615,  "lon": -73.9969813},{  "type": "node",  "id": 42440810,  "lat": 40.7175500,  "lon": -73.9993130},{  "type": "node",  "id": 42440815,  "lat": 40.7184300,  "lon": -74.0005320},{  "type": "node",  "id": 42440820,  "lat": 40.7199160,  "lon": -74.0025889},{  "type": "node",  "id": 42440823,  "lat": 40.7203895,  "lon": -74.0032496},{  "type": "node",  "id": 42445489,  "lat": 40.7200440,  "lon": -73.9990610},{  "type": "node",  "id": 42445612,  "lat": 40.7184420,  "lon": -73.9987548},{  "type": "node",  "id": 42452048,  "lat": 40.7204322,  "lon": -73.9998680},{  "type": "node",  "id": 42454170,  "lat": 40.7189870,  "lon": -73.9983950},{  "type": "node",  "id": 42454175,  "lat": 40.7195401,  "lon": -73.9981618},{  "type": "node",  "id": 591995262,  "lat": 40.7186551,  "lon": -74.0008461},{  "type": "node",  "id": 591995273,  "lat": 40.7189610,  "lon": -74.0012730},{  "type": "node",  "id": 1905965196,  "lat": 40.7162135,  "lon": -73.9971948},{  "type": "node",  "id": 2052618391,  "lat": 40.7191435,  "lon": -74.0015172},{  "type": "node",  "id": 3316533507,  "lat": 40.7183861,  "lon": -73.9996593},{  "type": "node",  "id": 4202984493,  "lat": 40.7187331,  "lon": -74.0002575},{  "type": "node",  "id": 4754110661,  "lat": 40.7187523,  "lon": -74.0024323},{  "type": "node",  "id": 4754110663,  "lat": 40.7185043,  "lon": -74.0026429},{  "type": "node",  "id": 4754110665,  "lat": 40.7181394,  "lon": -74.0029533},{  "type": "node",  "id": 4754117950,  "lat": 40.7205319,  "lon": -74.0009219},{  "type": "node",  "id": 4755831442,  "lat": 40.7207439,  "lon": -74.0005093},{  "type": "node",  "id": 5297484683,  "lat": 40.7200487,  "lon": -73.9983066},{  "type": "node",  "id": 5481883674,  "lat": 40.7190476,  "lon": -74.0021807},{  "type": "node",  "id": 5805243783,  "lat": 40.7196555,  "lon": -73.9980169},{  "type": "node",  "id": 42429637,  "lat": 40.7153550,  "lon": -73.9934000},{  "type": "node",  "id": 42429641,  "lat": 40.7164855,  "lon": -73.9928237},{  "type": "node",  "id": 42429643,  "lat": 40.7176680,  "lon": -73.9922172},{  "type": "node",  "id": 42429644,  "lat": 40.7185836,  "lon": -73.9917491},{  "type": "node",  "id": 42429645,  "lat": 40.7195624,  "lon": -73.9912489},{  "type": "node",  "id": 42429646,  "lat": 40.7207678,  "lon": -73.9906322},{  "type": "node",  "id": 42429752,  "lat": 40.7203456,  "lon": -73.9941173},{  "type": "node",  "id": 42429754,  "lat": 40.7199073,  "lon": -73.9929178},{  "type": "node",  "id": 42429756,  "lat": 40.7198216,  "lon": -73.9921032},{  "type": "node",  "id": 42430030,  "lat": 40.7206316,  "lon": -73.9949663},{  "type": "node",  "id": 42432085,  "lat": 40.7190700,  "lon": -73.9933600},{  "type": "node",  "id": 42432108,  "lat": 40.7183480,  "lon": -73.9909740},{  "type": "node",  "id": 42432111,  "lat": 40.7188510,  "lon": -73.9926210},{  "type": "node",  "id": 42432178,  "lat": 40.7201656,  "lon": -73.9960312},{  "type": "node",  "id": 42432181,  "lat": 40.7198670,  "lon": -73.9952600},{  "type": "node",  "id": 42432184,  "lat": 40.7194567,  "lon": -73.9943905},{  "type": "node",  "id": 42432265,  "lat": 40.7186910,  "lon": -73.9966320},{  "type": "node",  "id": 42432282,  "lat": 40.7190494,  "lon": -73.9964811},{  "type": "node",  "id": 42437580,  "lat": 40.7173076,  "lon": -73.9954317},{  "type": "node",  "id": 42437589,  "lat": 40.7184702,  "lon": -73.9948092},{  "type": "node",  "id": 42445603,  "lat": 40.7169745,  "lon": -73.9944558},{  "type": "node",  "id": 42445606,  "lat": 40.7176103,  "lon": -73.9963184},{  "type": "node",  "id": 42451660,  "lat": 40.7179326,  "lon": -73.9930898},{  "type": "node",  "id": 42452056,  "lat": 40.7187820,  "lon": -73.9956974},{  "type": "node",  "id": 42452067,  "lat": 40.7181594,  "lon": -73.9938242},{  "type": "node",  "id": 486867405,  "lat": 40.7194205,  "lon": -73.9913214},{  "type": "node",  "id": 486867432,  "lat": 40.7196806,  "lon": -73.9921746},{  "type": "node",  "id": 486867578,  "lat": 40.7201970,  "lon": -73.9938678},{  "type": "node",  "id": 486868873,  "lat": 40.7193292,  "lon": -73.9904805},{  "type": "node",  "id": 486869282,  "lat": 40.7200465,  "lon": -73.9928443},{  "type": "node",  "id": 486869287,  "lat": 40.7203336,  "lon": -73.9937906},{  "type": "node",  "id": 1773069828,  "lat": 40.7195597,  "lon": -73.9945080},{  "type": "node",  "id": 1773069829,  "lat": 40.7193561,  "lon": -73.9942647},{  "type": "node",  "id": 1773076511,  "lat": 40.7203048,  "lon": -73.9939823},{  "type": "node",  "id": 1773076513,  "lat": 40.7200948,  "lon": -73.9941321},{  "type": "node",  "id": 1773076514,  "lat": 40.7201802,  "lon": -73.9940362},{  "type": "node",  "id": 1991894114,  "lat": 40.7181462,  "lon": -73.9959808},{  "type": "node",  "id": 3120728297,  "lat": 40.7196932,  "lon": -73.9927529},{  "type": "node",  "id": 3212472843,  "lat": 40.7191862,  "lon": -73.9905533},{  "type": "node",  "id": 4202930753,  "lat": 40.7198870,  "lon": -73.9923187},{  "type": "node",  "id": 4202930754,  "lat": 40.7201786,  "lon": -73.9932663},{  "type": "node",  "id": 4202950101,  "lat": 40.7172381,  "lon": -73.9954766},{  "type": "node",  "id": 4215624277,  "lat": 40.7200901,  "lon": -73.9935174},{  "type": "node",  "id": 4300912711,  "lat": 40.7183742,  "lon": -73.9910601},{  "type": "node",  "id": 4539951607,  "lat": 40.7190409,  "lon": -73.9932637},{  "type": "node",  "id": 4539951608,  "lat": 40.7198827,  "lon": -73.9928316},{  "type": "node",  "id": 4539951609,  "lat": 40.7200212,  "lon": -73.9927581},{  "type": "node",  "id": 4539951617,  "lat": 40.7190219,  "lon": -73.9931980},{  "type": "node",  "id": 4539951620,  "lat": 40.7197355,  "lon": -73.9927347},{  "type": "node",  "id": 4539951622,  "lat": 40.7198688,  "lon": -73.9927915},{  "type": "node",  "id": 4539951623,  "lat": 40.7200088,  "lon": -73.9927201},{  "type": "node",  "id": 4956305806,  "lat": 40.7202552,  "lon": -73.9941522},{  "type": "node",  "id": 5043087532,  "lat": 40.7177404,  "lon": -73.9921802},{  "type": "node",  "id": 5043087536,  "lat": 40.7175911,  "lon": -73.9922566},{  "type": "node",  "id": 5159452342,  "lat": 40.7198509,  "lon": -73.9921998},{  "type": "node",  "id": 5159452344,  "lat": 40.7201526,  "lon": -73.9927882},{  "type": "node",  "id": 5159452347,  "lat": 40.7195863,  "lon": -73.9922253},{  "type": "node",  "id": 5159452348,  "lat": 40.7197108,  "lon": -73.9922734},{  "type": "node",  "id": 5159452349,  "lat": 40.7198050,  "lon": -73.9929718},{  "type": "node",  "id": 5159452350,  "lat": 40.7201263,  "lon": -73.9927096},{  "type": "node",  "id": 5159452352,  "lat": 40.7199155,  "lon": -73.9920566},{  "type": "node",  "id": 5161246305,  "lat": 40.7172591,  "lon": -73.9952894},{  "type": "node",  "id": 5296792744,  "lat": 40.7194410,  "lon": -73.9904224},{  "type": "node",  "id": 5296795594,  "lat": 40.7204616,  "lon": -73.9939240},{  "type": "node",  "id": 5296795595,  "lat": 40.7204711,  "lon": -73.9940704},{  "type": "node",  "id": 5296795596,  "lat": 40.7203711,  "lon": -73.9941931},{  "type": "node",  "id": 5296795597,  "lat": 40.7203237,  "lon": -73.9938564},{  "type": "node",  "id": 5296795598,  "lat": 40.7202258,  "lon": -73.9938984},{  "type": "node",  "id": 5296795599,  "lat": 40.7199566,  "lon": -73.9930796},{  "type": "node",  "id": 5296795600,  "lat": 40.7200976,  "lon": -73.9930127},{  "type": "node",  "id": 5296795754,  "lat": 40.7193500,  "lon": -73.9905492},{  "type": "node",  "id": 5296795755,  "lat": 40.7192072,  "lon": -73.9906219},{  "type": "node",  "id": 5296795758,  "lat": 40.7194014,  "lon": -73.9912590},{  "type": "node",  "id": 5296795759,  "lat": 40.7195475,  "lon": -73.9911998},{  "type": "node",  "id": 5296795760,  "lat": 40.7196567,  "lon": -73.9912004},{  "type": "node",  "id": 5296795761,  "lat": 40.7195880,  "lon": -73.9913334},{  "type": "node",  "id": 5296795762,  "lat": 40.7194440,  "lon": -73.9913984},{  "type": "node",  "id": 5296795763,  "lat": 40.7193351,  "lon": -73.9913648},{  "type": "node",  "id": 5805243784,  "lat": 40.7184199,  "lon": -73.9946870},{  "type": "node",  "id": 5805243785,  "lat": 40.7185226,  "lon": -73.9949369},{  "type": "node",  "id": 5854381564,  "lat": 40.7194242,  "lon": -73.9904311},{  "type": "node",  "id": 6330061743,  "lat": 40.7197014,  "lon": -73.9922428},{  "type": "node",  "id": 6330061745,  "lat": 40.7198421,  "lon": -73.9921708},{  "type": "node",  "id": 6330061746,  "lat": 40.7188656,  "lon": -73.9926704},{  "type": "node",  "id": 6330061754,  "lat": 40.7188979,  "lon": -73.9927792},{  "type": "node",  "id": 6369613028,  "lat": 40.7170077,  "lon": -73.9945532},{  "type": "node",  "id": 6369613034,  "lat": 40.7173516,  "lon": -73.9955607},{  "type": "node",  "id": 6369613035,  "lat": 40.7173365,  "lon": -73.9954162},{  "type": "node",  "id": 6369613048,  "lat": 40.7181957,  "lon": -73.9939445},{  "type": "node",  "id": 6369613050,  "lat": 40.7182270,  "lon": -73.9937897},{  "type": "node",  "id": 6369613059,  "lat": 40.7180105,  "lon": -73.9930500},{  "type": "node",  "id": 6369613316,  "lat": 40.7154385,  "lon": -73.9933574},{  "type": "node",  "id": 6369625008,  "lat": 40.7165324,  "lon": -73.9927997},{  "type": "node",  "id": 6369625010,  "lat": 40.7164357,  "lon": -73.9928491},{  "type": "node",  "id": 42427327,  "lat": 40.7208360,  "lon": -73.9976550},{  "type": "node",  "id": 42428436,  "lat": 40.7208634,  "lon": -74.0006449},{  "type": "node",  "id": 42428438,  "lat": 40.7217891,  "lon": -73.9998554},{  "type": "node",  "id": 42428441,  "lat": 40.7230210,  "lon": -73.9988120},{  "type": "node",  "id": 42428444,  "lat": 40.7242350,  "lon": -73.9977850},{  "type": "node",  "id": 42428447,  "lat": 40.7255012,  "lon": -73.9967146},{  "type": "node",  "id": 42430041,  "lat": 40.7214509,  "lon": -73.9973979},{  "type": "node",  "id": 42430044,  "lat": 40.7215815,  "lon": -73.9977854},{  "type": "node",  "id": 42432142,  "lat": 40.7233191,  "lon": -74.0029712},{  "type": "node",  "id": 42432148,  "lat": 40.7229283,  "lon": -74.0021754},{  "type": "node",  "id": 42432152,  "lat": 40.7225511,  "lon": -74.0014071},{  "type": "node",  "id": 42432156,  "lat": 40.7221828,  "lon": -74.0006571},{  "type": "node",  "id": 42432161,  "lat": 40.7214210,  "lon": -73.9990310},{  "type": "node",  "id": 42432165,  "lat": 40.7210630,  "lon": -73.9981960},{  "type": "node",  "id": 42433574,  "lat": 40.7219950,  "lon": -74.0029710},{  "type": "node",  "id": 42433577,  "lat": 40.7241660,  "lon": -74.0011270},{  "type": "node",  "id": 42433578,  "lat": 40.7253820,  "lon": -74.0001010},{  "type": "node",  "id": 42433581,  "lat": 40.7266087,  "lon": -73.9990701},{  "type": "node",  "id": 42436779,  "lat": 40.7248800,  "lon": -74.0026190},{  "type": "node",  "id": 42436784,  "lat": 40.7261286,  "lon": -74.0015804},{  "type": "node",  "id": 42436788,  "lat": 40.7273529,  "lon": -74.0005803},{  "type": "node",  "id": 42439335,  "lat": 40.7268780,  "lon": -74.0031220},{  "type": "node",  "id": 42439337,  "lat": 40.7269160,  "lon": -74.0032010},{  "type": "node",  "id": 42439530,  "lat": 40.7264960,  "lon": -74.0023500},{  "type": "node",  "id": 42442247,  "lat": 40.7223037,  "lon": -73.9971275},{  "type": "node",  "id": 42442255,  "lat": 40.7226691,  "lon": -73.9979746},{  "type": "node",  "id": 42442269,  "lat": 40.7234230,  "lon": -73.9996090},{  "type": "node",  "id": 42442273,  "lat": 40.7237950,  "lon": -74.0003670},{  "type": "node",  "id": 42442276,  "lat": 40.7245520,  "lon": -74.0019330},{  "type": "node",  "id": 42448552,  "lat": 40.7258712,  "lon": -73.9975736},{  "type": "node",  "id": 42448554,  "lat": 40.7262337,  "lon": -73.9983091},{  "type": "node",  "id": 42448558,  "lat": 40.7270026,  "lon": -73.9998695},{  "type": "node",  "id": 42449570,  "lat": 40.7212520,  "lon": -74.0014470},{  "type": "node",  "id": 42449576,  "lat": 40.7246320,  "lon": -73.9985850},{  "type": "node",  "id": 42452040,  "lat": 40.7216210,  "lon": -74.0022080},{  "type": "node",  "id": 42454423,  "lat": 40.7257730,  "lon": -74.0008980},{  "type": "node",  "id": 42456197,  "lat": 40.7250080,  "lon": -73.9993450},{  "type": "node",  "id": 42458267,  "lat": 40.7238800,  "lon": -73.9969520},{  "type": "node",  "id": 1919595915,  "lat": 40.7253844,  "lon": -73.9968134},{  "type": "node",  "id": 1919595917,  "lat": 40.7261180,  "lon": -73.9984070},{  "type": "node",  "id": 1919595918,  "lat": 40.7272378,  "lon": -74.0006749},{  "type": "node",  "id": 1919595919,  "lat": 40.7254675,  "lon": -73.9970892},{  "type": "node",  "id": 1919595925,  "lat": 40.7257539,  "lon": -73.9976695},{  "type": "node",  "id": 1919595926,  "lat": 40.7264932,  "lon": -73.9991668},{  "type": "node",  "id": 1927715353,  "lat": 40.7256397,  "lon": -74.0006264},{  "type": "node",  "id": 2154628332,  "lat": 40.7222025,  "lon": -73.9971007},{  "type": "node",  "id": 2154628339,  "lat": 40.7222637,  "lon": -73.9972486},{  "type": "node",  "id": 2477560535,  "lat": 40.7268029,  "lon": -74.0031947},{  "type": "node",  "id": 3236571730,  "lat": 40.7272964,  "lon": -73.9974022},{  "type": "node",  "id": 3546151525,  "lat": 40.7233966,  "lon": -74.0030760},{  "type": "node",  "id": 3573482108,  "lat": 40.7229774,  "lon": -73.9988489},{  "type": "node",  "id": 3573482109,  "lat": 40.7229794,  "lon": -73.9987153},{  "type": "node",  "id": 3573482110,  "lat": 40.7230717,  "lon": -73.9987691},{  "type": "node",  "id": 3573482111,  "lat": 40.7230754,  "lon": -73.9989199},{  "type": "node",  "id": 3594715134,  "lat": 40.7255850,  "lon": -73.9969929},{  "type": "node",  "id": 4202908172,  "lat": 40.7257096,  "lon": -73.9972458},{  "type": "node",  "id": 4202908173,  "lat": 40.7267939,  "lon": -73.9994460},{  "type": "node",  "id": 4745829026,  "lat": 40.7269662,  "lon": -73.9987723},{  "type": "node",  "id": 4755831435,  "lat": 40.7239729,  "lon": -73.9980067},{  "type": "node",  "id": 4755831437,  "lat": 40.7227353,  "lon": -73.9990540},{  "type": "node",  "id": 4755831438,  "lat": 40.7217467,  "lon": -73.9997691},{  "type": "node",  "id": 4755831440,  "lat": 40.7216054,  "lon": -74.0000122},{  "type": "node",  "id": 4755898719,  "lat": 40.7250916,  "lon": -73.9970609},{  "type": "node",  "id": 5152337872,  "lat": 40.7213762,  "lon": -73.9974291},{  "type": "node",  "id": 5152337873,  "lat": 40.7215331,  "lon": -73.9973657},{  "type": "node",  "id": 5152337874,  "lat": 40.7215160,  "lon": -73.9978373},{  "type": "node",  "id": 5152337875,  "lat": 40.7215407,  "lon": -73.9976644},{  "type": "node",  "id": 5152337880,  "lat": 40.7216465,  "lon": -73.9977343},{  "type": "node",  "id": 5152337881,  "lat": 40.7214753,  "lon": -73.9974703},{  "type": "node",  "id": 5243716990,  "lat": 40.7259447,  "lon": -73.9980558},{  "type": "node",  "id": 5481866323,  "lat": 40.7273031,  "lon": -73.9984916},{  "type": "node",  "id": 5481866326,  "lat": 40.7270319,  "lon": -73.9987176},{  "type": "node",  "id": 5799037570,  "lat": 40.7254165,  "lon": -73.9969323},{  "type": "node",  "id": 5799117240,  "lat": 40.7268876,  "lon": -73.9999657},{  "type": "node",  "id": 5799117244,  "lat": 40.7262327,  "lon": -73.9983072},{  "type": "node",  "id": 5829827195,  "lat": 40.7266787,  "lon": -73.9990118},{  "type": "node",  "id": 5829827213,  "lat": 40.7259465,  "lon": -73.9975089},{  "type": "node",  "id": 5829827214,  "lat": 40.7270788,  "lon": -73.9998041},{  "type": "node",  "id": 5829844672,  "lat": 40.7268152,  "lon": -74.0000263},{  "type": "node",  "id": 5829844675,  "lat": 40.7253161,  "lon": -73.9968711},{  "type": "node",  "id": 5829847185,  "lat": 40.7256842,  "lon": -73.9977263},{  "type": "node",  "id": 5829847188,  "lat": 40.7271721,  "lon": -74.0007286},{  "type": "node",  "id": 5829847195,  "lat": 40.7264210,  "lon": -73.9992275},{  "type": "node",  "id": 5829880518,  "lat": 40.7272709,  "lon": -74.0007420},{  "type": "node",  "id": 5829880519,  "lat": 40.7273212,  "lon": -74.0005161},{  "type": "node",  "id": 5829880528,  "lat": 40.7272064,  "lon": -74.0006113},{  "type": "node",  "id": 5829907363,  "lat": 40.7261997,  "lon": -73.9982402},{  "type": "node",  "id": 5829907364,  "lat": 40.7257800,  "lon": -73.9977223},{  "type": "node",  "id": 5829907365,  "lat": 40.7257140,  "lon": -73.9975887},{  "type": "node",  "id": 5829907366,  "lat": 40.7262669,  "lon": -73.9983764},{  "type": "node",  "id": 5829907367,  "lat": 40.7261506,  "lon": -73.9984730},{  "type": "node",  "id": 5829907368,  "lat": 40.7268446,  "lon": -73.9998766},{  "type": "node",  "id": 5829907369,  "lat": 40.7269580,  "lon": -73.9997790},{  "type": "node",  "id": 5829907370,  "lat": 40.7269305,  "lon": -74.0000526},{  "type": "node",  "id": 5829907371,  "lat": 40.7258299,  "lon": -73.9974897},{  "type": "node",  "id": 5829907372,  "lat": 40.7259022,  "lon": -73.9976365},{  "type": "node",  "id": 5829907374,  "lat": 40.7270451,  "lon": -73.9999557},{  "type": "node",  "id": 5829907375,  "lat": 40.7260865,  "lon": -73.9983431},{  "type": "node",  "id": 5829927763,  "lat": 40.7254177,  "lon": -73.9969361},{  "type": "node",  "id": 5829927764,  "lat": 40.7253581,  "lon": -73.9967028},{  "type": "node",  "id": 5829927772,  "lat": 40.7255342,  "lon": -73.9968361},{  "type": "node",  "id": 6024767164,  "lat": 40.7260527,  "lon": -73.9984624},{  "type": "node",  "id": 42421927,  "lat": 40.7262630,  "lon": -73.9944030},{  "type": "node",  "id": 42428454,  "lat": 40.7272110,  "lon": -73.9952700},{  "type": "node",  "id": 42429672,  "lat": 40.7269405,  "lon": -73.9947295},{  "type": "node",  "id": 42430034,  "lat": 40.7208958,  "lon": -73.9957503},{  "type": "node",  "id": 42430038,  "lat": 40.7211736,  "lon": -73.9965747},{  "type": "node",  "id": 42431165,  "lat": 40.7272967,  "lon": -73.9936333},{  "type": "node",  "id": 42431168,  "lat": 40.7264167,  "lon": -73.9918576},{  "type": "node",  "id": 42431712,  "lat": 40.7213883,  "lon": -73.9927159},{  "type": "node",  "id": 42431822,  "lat": 40.7254670,  "lon": -73.9930401},{  "type": "node",  "id": 42432054,  "lat": 40.7221825,  "lon": -73.9908948},{  "type": "node",  "id": 42432082,  "lat": 40.7228551,  "lon": -73.9931797},{  "type": "node",  "id": 42432083,  "lat": 40.7224089,  "lon": -73.9916453},{  "type": "node",  "id": 42432302,  "lat": 40.7216242,  "lon": -73.9954604},{  "type": "node",  "id": 42432305,  "lat": 40.7230360,  "lon": -73.9949090},{  "type": "node",  "id": 42432308,  "lat": 40.7247613,  "lon": -73.9941439},{  "type": "node",  "id": 42434919,  "lat": 40.7246140,  "lon": -73.9916085},{  "type": "node",  "id": 42435477,  "lat": 40.7233679,  "lon": -73.9957075},{  "type": "node",  "id": 42435485,  "lat": 40.7240720,  "lon": -73.9953510},{  "type": "node",  "id": 42435491,  "lat": 40.7250181,  "lon": -73.9948703},{  "type": "node",  "id": 42437111,  "lat": 40.7210302,  "lon": -73.9914809},{  "type": "node",  "id": 42437113,  "lat": 40.7212480,  "lon": -73.9922370},{  "type": "node",  "id": 42437114,  "lat": 40.7216587,  "lon": -73.9936267},{  "type": "node",  "id": 42437605,  "lat": 40.7209767,  "lon": -73.9938815},{  "type": "node",  "id": 42437608,  "lat": 40.7223908,  "lon": -73.9933532},{  "type": "node",  "id": 42437612,  "lat": 40.7242393,  "lon": -73.9926626},{  "type": "node",  "id": 42437613,  "lat": 40.7249181,  "lon": -73.9924089},{  "type": "node",  "id": 42437618,  "lat": 40.7256721,  "lon": -73.9921271},{  "type": "node",  "id": 42442230,  "lat": 40.7213148,  "lon": -73.9947014},{  "type": "node",  "id": 42442836,  "lat": 40.7238043,  "lon": -73.9912273},{  "type": "node",  "id": 42442838,  "lat": 40.7241854,  "lon": -73.9909044},{  "type": "node",  "id": 42442842,  "lat": 40.7242892,  "lon": -73.9908277},{  "type": "node",  "id": 42442843,  "lat": 40.7249392,  "lon": -73.9903384},{  "type": "node",  "id": 42443127,  "lat": 40.7237426,  "lon": -73.9910273},{  "type": "node",  "id": 42445494,  "lat": 40.7234670,  "lon": -73.9966610},{  "type": "node",  "id": 42445498,  "lat": 40.7236860,  "lon": -73.9964760},{  "type": "node",  "id": 42445505,  "lat": 40.7244030,  "lon": -73.9958820},{  "type": "node",  "id": 42445511,  "lat": 40.7251537,  "lon": -73.9952714},{  "type": "node",  "id": 42445520,  "lat": 40.7266400,  "lon": -73.9941290},{  "type": "node",  "id": 42451665,  "lat": 40.7235035,  "lon": -73.9902408},{  "type": "node",  "id": 42453327,  "lat": 40.7246620,  "lon": -73.9963010},{  "type": "node",  "id": 42457319,  "lat": 40.7252790,  "lon": -73.9957977},{  "type": "node",  "id": 42457325,  "lat": 40.7245091,  "lon": -73.9934284},{  "type": "node",  "id": 42458313,  "lat": 40.7227230,  "lon": -73.9941530},{  "type": "node",  "id": 428628885,  "lat": 40.7238123,  "lon": -73.9945675},{  "type": "node",  "id": 428833475,  "lat": 40.7237959,  "lon": -73.9955011},{  "type": "node",  "id": 428833476,  "lat": 40.7235599,  "lon": -73.9946785},{  "type": "node",  "id": 428833477,  "lat": 40.7235292,  "lon": -73.9956298},{  "type": "node",  "id": 428833478,  "lat": 40.7232875,  "lon": -73.9948037},{  "type": "node",  "id": 494509182,  "lat": 40.7237318,  "lon": -73.9915040},{  "type": "node",  "id": 1692433907,  "lat": 40.7252621,  "lon": -73.9922803},{  "type": "node",  "id": 1692433911,  "lat": 40.7256586,  "lon": -73.9937515},{  "type": "node",  "id": 1692433914,  "lat": 40.7258505,  "lon": -73.9944634},{  "type": "node",  "id": 1692433915,  "lat": 40.7258521,  "lon": -73.9947115},{  "type": "node",  "id": 1692433916,  "lat": 40.7259345,  "lon": -73.9946480},{  "type": "node",  "id": 1692433917,  "lat": 40.7260180,  "lon": -73.9945838},{  "type": "node",  "id": 1692433918,  "lat": 40.7261433,  "lon": -73.9950705},{  "type": "node",  "id": 1692433919,  "lat": 40.7265273,  "lon": -73.9958473},{  "type": "node",  "id": 1692433920,  "lat": 40.7269315,  "lon": -73.9966626},{  "type": "node",  "id": 1773076509,  "lat": 40.7209391,  "lon": -73.9937464},{  "type": "node",  "id": 1773076778,  "lat": 40.7216185,  "lon": -73.9934936},{  "type": "node",  "id": 1773078006,  "lat": 40.7223456,  "lon": -73.9932232},{  "type": "node",  "id": 1773082408,  "lat": 40.7228172,  "lon": -73.9930477},{  "type": "node",  "id": 1773082410,  "lat": 40.7241983,  "lon": -73.9925341},{  "type": "node",  "id": 1773084391,  "lat": 40.7252355,  "lon": -73.9921466},{  "type": "node",  "id": 1773084392,  "lat": 40.7255967,  "lon": -73.9919013},{  "type": "node",  "id": 1773084402,  "lat": 40.7270946,  "lon": -73.9915448},{  "type": "node",  "id": 1773084405,  "lat": 40.7263628,  "lon": -73.9917292},{  "type": "node",  "id": 1773084407,  "lat": 40.7256296,  "lon": -73.9919998},{  "type": "node",  "id": 1773084410,  "lat": 40.7248826,  "lon": -73.9922795},{  "type": "node",  "id": 1773084413,  "lat": 40.7257013,  "lon": -73.9922144},{  "type": "node",  "id": 1918039864,  "lat": 40.7240975,  "lon": -73.9927155},{  "type": "node",  "id": 1918039869,  "lat": 40.7248879,  "lon": -73.9949407},{  "type": "node",  "id": 1918039877,  "lat": 40.7236025,  "lon": -73.9910658},{  "type": "node",  "id": 1918039880,  "lat": 40.7251668,  "lon": -73.9959002},{  "type": "node",  "id": 1918039896,  "lat": 40.7233701,  "lon": -73.9903071},{  "type": "node",  "id": 1918039897,  "lat": 40.7250362,  "lon": -73.9953664},{  "type": "node",  "id": 1918039898,  "lat": 40.7243691,  "lon": -73.9934855},{  "type": "node",  "id": 1918039901,  "lat": 40.7246263,  "lon": -73.9942070},{  "type": "node",  "id": 1918039904,  "lat": 40.7240579,  "lon": -73.9925863},{  "type": "node",  "id": 1918039935,  "lat": 40.7239860,  "lon": -73.9909867},{  "type": "node",  "id": 3316523989,  "lat": 40.7221574,  "lon": -73.9908118},{  "type": "node",  "id": 3316523990,  "lat": 40.7222372,  "lon": -73.9908675},{  "type": "node",  "id": 3463814535,  "lat": 40.7229867,  "lon": -73.9949283},{  "type": "node",  "id": 3463814536,  "lat": 40.7230137,  "lon": -73.9948552},{  "type": "node",  "id": 3463814537,  "lat": 40.7230639,  "lon": -73.9949760},{  "type": "node",  "id": 3463814538,  "lat": 40.7230918,  "lon": -73.9948857},{  "type": "node",  "id": 3463814539,  "lat": 40.7233192,  "lon": -73.9957268},{  "type": "node",  "id": 3463814540,  "lat": 40.7233421,  "lon": -73.9956454},{  "type": "node",  "id": 3463814541,  "lat": 40.7233939,  "lon": -73.9957703},{  "type": "node",  "id": 3463814542,  "lat": 40.7234264,  "lon": -73.9956793},{  "type": "node",  "id": 3927743126,  "lat": 40.7238399,  "lon": -73.9910084},{  "type": "node",  "id": 3927743127,  "lat": 40.7237784,  "lon": -73.9911453},{  "type": "node",  "id": 3927743131,  "lat": 40.7236984,  "lon": -73.9908833},{  "type": "node",  "id": 3927743135,  "lat": 40.7236387,  "lon": -73.9911949},{  "type": "node",  "id": 4202908169,  "lat": 40.7242438,  "lon": -73.9931340},{  "type": "node",  "id": 4202908170,  "lat": 40.7251474,  "lon": -73.9958189},{  "type": "node",  "id": 4202908171,  "lat": 40.7253562,  "lon": -73.9961141},{  "type": "node",  "id": 4202950102,  "lat": 40.7208209,  "lon": -73.9939397},{  "type": "node",  "id": 4202950103,  "lat": 40.7233388,  "lon": -73.9928538},{  "type": "node",  "id": 4202950104,  "lat": 40.7248713,  "lon": -73.9924264},{  "type": "node",  "id": 4202957549,  "lat": 40.7246308,  "lon": -73.9905696},{  "type": "node",  "id": 4202984494,  "lat": 40.7242102,  "lon": -73.9960417},{  "type": "node",  "id": 4202984495,  "lat": 40.7248122,  "lon": -73.9955552},{  "type": "node",  "id": 4202984496,  "lat": 40.7270190,  "lon": -73.9938434},{  "type": "node",  "id": 4215580881,  "lat": 40.7259076,  "lon": -73.9963711},{  "type": "node",  "id": 4501924441,  "lat": 40.7220811,  "lon": -73.9944038},{  "type": "node",  "id": 4501924445,  "lat": 40.7223076,  "lon": -73.9951943},{  "type": "node",  "id": 4539951610,  "lat": 40.7212219,  "lon": -73.9921517},{  "type": "node",  "id": 4539951611,  "lat": 40.7223769,  "lon": -73.9915498},{  "type": "node",  "id": 4539951613,  "lat": 40.7235730,  "lon": -73.9909775},{  "type": "node",  "id": 4591871814,  "lat": 40.7216649,  "lon": -73.9901735},{  "type": "node",  "id": 4690139671,  "lat": 40.7215690,  "lon": -73.9902225},{  "type": "node",  "id": 4755898713,  "lat": 40.7262327,  "lon": -73.9960963},{  "type": "node",  "id": 4755898715,  "lat": 40.7252582,  "lon": -73.9962837},{  "type": "node",  "id": 4776041123,  "lat": 40.7249036,  "lon": -73.9902539},{  "type": "node",  "id": 4776041124,  "lat": 40.7242516,  "lon": -73.9907394},{  "type": "node",  "id": 4776041130,  "lat": 40.7237141,  "lon": -73.9909348},{  "type": "node",  "id": 4892845897,  "lat": 40.7238510,  "lon": -73.9913862},{  "type": "node",  "id": 5159433792,  "lat": 40.7219524,  "lon": -73.9962656},{  "type": "node",  "id": 5243701466,  "lat": 40.7258321,  "lon": -73.9920691},{  "type": "node",  "id": 5243701467,  "lat": 40.7255094,  "lon": -73.9920463},{  "type": "node",  "id": 5277800701,  "lat": 40.7216873,  "lon": -73.9901620},{  "type": "node",  "id": 5277800703,  "lat": 40.7210393,  "lon": -73.9904934},{  "type": "node",  "id": 5286255072,  "lat": 40.7269199,  "lon": -73.9916740},{  "type": "node",  "id": 5286255073,  "lat": 40.7268740,  "lon": -73.9915386},{  "type": "node",  "id": 5355267632,  "lat": 40.7267777,  "lon": -73.9963523},{  "type": "node",  "id": 5481937333,  "lat": 40.7270546,  "lon": -73.9931487},{  "type": "node",  "id": 5783022064,  "lat": 40.7234853,  "lon": -73.9909975},{  "type": "node",  "id": 5783022067,  "lat": 40.7232861,  "lon": -73.9903484},{  "type": "node",  "id": 5783022072,  "lat": 40.7235149,  "lon": -73.9910985},{  "type": "node",  "id": 5783028385,  "lat": 40.7233966,  "lon": -73.9903945},{  "type": "node",  "id": 5783028389,  "lat": 40.7233420,  "lon": -73.9902140},{  "type": "node",  "id": 5783028393,  "lat": 40.7235289,  "lon": -73.9903247},{  "type": "node",  "id": 5784518530,  "lat": 40.7271722,  "lon": -73.9917059},{  "type": "node",  "id": 5798966627,  "lat": 40.7235556,  "lon": -73.9909199},{  "type": "node",  "id": 5798966628,  "lat": 40.7238133,  "lon": -73.9912618},{  "type": "node",  "id": 5799037566,  "lat": 40.7250027,  "lon": -73.9952627},{  "type": "node",  "id": 5799037574,  "lat": 40.7251844,  "lon": -73.9953979},{  "type": "node",  "id": 5799463409,  "lat": 40.7254397,  "lon": -73.9956638},{  "type": "node",  "id": 5799559461,  "lat": 40.7212811,  "lon": -73.9936191},{  "type": "node",  "id": 5824300186,  "lat": 40.7264550,  "lon": -73.9919487},{  "type": "node",  "id": 5824300192,  "lat": 40.7257251,  "lon": -73.9939981},{  "type": "node",  "id": 5829753046,  "lat": 40.7249570,  "lon": -73.9954294},{  "type": "node",  "id": 5829753047,  "lat": 40.7252274,  "lon": -73.9952076},{  "type": "node",  "id": 5829779485,  "lat": 40.7242700,  "lon": -73.9927536},{  "type": "node",  "id": 5829779486,  "lat": 40.7238899,  "lon": -73.9911667},{  "type": "node",  "id": 5829779489,  "lat": 40.7246530,  "lon": -73.9942819},{  "type": "node",  "id": 5829779490,  "lat": 40.7243454,  "lon": -73.9934186},{  "type": "node",  "id": 5829779491,  "lat": 40.7247379,  "lon": -73.9940774},{  "type": "node",  "id": 5829779492,  "lat": 40.7241603,  "lon": -73.9924085},{  "type": "node",  "id": 5829779493,  "lat": 40.7248120,  "lon": -73.9949789},{  "type": "node",  "id": 5829779494,  "lat": 40.7239782,  "lon": -73.9926159},{  "type": "node",  "id": 5829779496,  "lat": 40.7242940,  "lon": -73.9935159},{  "type": "node",  "id": 5829779497,  "lat": 40.7244837,  "lon": -73.9933563},{  "type": "node",  "id": 5829779498,  "lat": 40.7251044,  "lon": -73.9948281},{  "type": "node",  "id": 5829779501,  "lat": 40.7247917,  "lon": -73.9942271},{  "type": "node",  "id": 5829779504,  "lat": 40.7249125,  "lon": -73.9950096},{  "type": "node",  "id": 5829779506,  "lat": 40.7248507,  "lon": -73.9941048},{  "type": "node",  "id": 5829779507,  "lat": 40.7245485,  "lon": -73.9942415},{  "type": "node",  "id": 5829779509,  "lat": 40.7249969,  "lon": -73.9948103},{  "type": "node",  "id": 5829779510,  "lat": 40.7245358,  "lon": -73.9935042},{  "type": "node",  "id": 5829779511,  "lat": 40.7241269,  "lon": -73.9928058},{  "type": "node",  "id": 5829779512,  "lat": 40.7240179,  "lon": -73.9927452},{  "type": "node",  "id": 5829779515,  "lat": 40.7248650,  "lon": -73.9948763},{  "type": "node",  "id": 5829779516,  "lat": 40.7246029,  "lon": -73.9941417},{  "type": "node",  "id": 5829779518,  "lat": 40.7242867,  "lon": -73.9925011},{  "type": "node",  "id": 5829779519,  "lat": 40.7240211,  "lon": -73.9924641},{  "type": "node",  "id": 5829779524,  "lat": 40.7250435,  "lon": -73.9949422},{  "type": "node",  "id": 5829779527,  "lat": 40.7245956,  "lon": -73.9933933},{  "type": "node",  "id": 5829779529,  "lat": 40.7243945,  "lon": -73.9935567},{  "type": "node",  "id": 5829779530,  "lat": 40.7250045,  "lon": -73.9952683},{  "type": "node",  "id": 5829779531,  "lat": 40.7243258,  "lon": -73.9926296},{  "type": "node",  "id": 5829779533,  "lat": 40.7251232,  "lon": -73.9951675},{  "type": "node",  "id": 5829827197,  "lat": 40.7253535,  "lon": -73.9957357},{  "type": "node",  "id": 5829827207,  "lat": 40.7255725,  "lon": -73.9966543},{  "type": "node",  "id": 5829927762,  "lat": 40.7251824,  "lon": -73.9959673},{  "type": "node",  "id": 5829927765,  "lat": 40.7251832,  "lon": -73.9953927},{  "type": "node",  "id": 5829927766,  "lat": 40.7252985,  "lon": -73.9958775},{  "type": "node",  "id": 5829927767,  "lat": 40.7254757,  "lon": -73.9966037},{  "type": "node",  "id": 5829927768,  "lat": 40.7251450,  "lon": -73.9958079},{  "type": "node",  "id": 5829927769,  "lat": 40.7252584,  "lon": -73.9957104},{  "type": "node",  "id": 5829927770,  "lat": 40.7250701,  "lon": -73.9954894},{  "type": "node",  "id": 5991033038,  "lat": 40.7234367,  "lon": -73.9910096},{  "type": "node",  "id": 5991033039,  "lat": 40.7234721,  "lon": -73.9911206},{  "type": "node",  "id": 6213204045,  "lat": 40.7265461,  "lon": -73.9958315},{  "type": "node",  "id": 6213204049,  "lat": 40.7269476,  "lon": -73.9966489},{  "type": "node",  "id": 6213204060,  "lat": 40.7259553,  "lon": -73.9946320},{  "type": "node",  "id": 6330061740,  "lat": 40.7222030,  "lon": -73.9909628},{  "type": "node",  "id": 6330061741,  "lat": 40.7210508,  "lon": -73.9915524},{  "type": "node",  "id": 6330061765,  "lat": 40.7223580,  "lon": -73.9914763},{  "type": "node",  "id": 6330061766,  "lat": 40.7212015,  "lon": -73.9920765},{  "type": "node",  "id": 6330061769,  "lat": 40.7222262,  "lon": -73.9910397},{  "type": "node",  "id": 6330061770,  "lat": 40.7210725,  "lon": -73.9916280},{  "type": "node",  "id": 6336172705,  "lat": 40.7213060,  "lon": -73.9921115},{  "type": "node",  "id": 6336172706,  "lat": 40.7214118,  "lon": -73.9920442},{  "type": "node",  "id": 42437108,  "lat": 40.7204518,  "lon": -73.9895903},{  "type": "node",  "id": 42437109,  "lat": 40.7205327,  "lon": -73.9898571},{  "type": "node",  "id": 2349482103,  "lat": 40.7204930,  "lon": -73.9897264},{  "type": "node",  "id": 3314293709,  "lat": 40.7205561,  "lon": -73.9899344},{  "type": "node",  "id": 3314293710,  "lat": 40.7205937,  "lon": -73.9898237},{  "type": "node",  "id": 3314293711,  "lat": 40.7204915,  "lon": -73.9898783},{  "type": "node",  "id": 3314293714,  "lat": 40.7205141,  "lon": -73.9895582},{  "type": "node",  "id": 592509222,  "lat": 40.7203935,  "lon": -73.9799914},{  "type": "node",  "id": 3783071256,  "lat": 40.7205489,  "lon": -73.9800059},{  "type": "node",  "id": 3783073165,  "lat": 40.7204545,  "lon": -73.9800308},{  "type": "node",  "id": 3783073197,  "lat": 40.7205864,  "lon": -73.9800272},{  "type": "node",  "id": 3783073245,  "lat": 40.7204964,  "lon": -73.9800246},{  "type": "node",  "id": 42429647,  "lat": 40.7219247,  "lon": -73.9900405},{  "type": "node",  "id": 42432057,  "lat": 40.7216919,  "lon": -73.9892727},{  "type": "node",  "id": 42432060,  "lat": 40.7216077,  "lon": -73.9889948},{  "type": "node",  "id": 42432063,  "lat": 40.7214626,  "lon": -73.9885157},{  "type": "node",  "id": 42434205,  "lat": 40.7269880,  "lon": -73.9857530},{  "type": "node",  "id": 42437333,  "lat": 40.7255820,  "lon": -73.9898700},{  "type": "node",  "id": 42437339,  "lat": 40.7245840,  "lon": -73.9875020},{  "type": "node",  "id": 42437343,  "lat": 40.7236490,  "lon": -73.9852860},{  "type": "node",  "id": 42440702,  "lat": 40.7226898,  "lon": -73.9870797},{  "type": "node",  "id": 42442848,  "lat": 40.7261880,  "lon": -73.9894290},{  "type": "node",  "id": 42442850,  "lat": 40.7267997,  "lon": -73.9889783},{  "type": "node",  "id": 42443513,  "lat": 40.7232739,  "lon": -73.9884433},{  "type": "node",  "id": 42444827,  "lat": 40.7264090,  "lon": -73.9861760},{  "type": "node",  "id": 42444829,  "lat": 40.7254730,  "lon": -73.9839580},{  "type": "node",  "id": 42453166,  "lat": 40.7257960,  "lon": -73.9866230},{  "type": "node",  "id": 42453601,  "lat": 40.7239420,  "lon": -73.9879680},{  "type": "node",  "id": 42453604,  "lat": 40.7230090,  "lon": -73.9857520},{  "type": "node",  "id": 42455039,  "lat": 40.7222945,  "lon": -73.9862562},{  "type": "node",  "id": 42455051,  "lat": 40.7242690,  "lon": -73.9848370},{  "type": "node",  "id": 42455867,  "lat": 40.7252030,  "lon": -73.9870530},{  "type": "node",  "id": 42456331,  "lat": 40.7227753,  "lon": -73.9878409},{  "type": "node",  "id": 775996460,  "lat": 40.7252102,  "lon": -73.9901407},{  "type": "node",  "id": 805771586,  "lat": 40.7228473,  "lon": -73.9885668},{  "type": "node",  "id": 1918039876,  "lat": 40.7231066,  "lon": -73.9894363},{  "type": "node",  "id": 1919595911,  "lat": 40.7224176,  "lon": -73.9871602},{  "type": "node",  "id": 1919595914,  "lat": 40.7226441,  "lon": -73.9879084},{  "type": "node",  "id": 1919595929,  "lat": 40.7221671,  "lon": -73.9863325},{  "type": "node",  "id": 2672569868,  "lat": 40.7233520,  "lon": -73.9886290},{  "type": "node",  "id": 3212472829,  "lat": 40.7227520,  "lon": -73.9885918},{  "type": "node",  "id": 3212472830,  "lat": 40.7228246,  "lon": -73.9888378},{  "type": "node",  "id": 3291987532,  "lat": 40.7236239,  "lon": -73.9872124},{  "type": "node",  "id": 3314285365,  "lat": 40.7223718,  "lon": -73.9890116},{  "type": "node",  "id": 3314293703,  "lat": 40.7215865,  "lon": -73.9889254},{  "type": "node",  "id": 3314293704,  "lat": 40.7217190,  "lon": -73.9893625},{  "type": "node",  "id": 3314293705,  "lat": 40.7215180,  "lon": -73.9884874},{  "type": "node",  "id": 3316523995,  "lat": 40.7248995,  "lon": -73.9863431},{  "type": "node",  "id": 3316524005,  "lat": 40.7257039,  "lon": -73.9845051},{  "type": "node",  "id": 3420766627,  "lat": 40.7267555,  "lon": -73.9888745},{  "type": "node",  "id": 3420766628,  "lat": 40.7273592,  "lon": -73.9884206},{  "type": "node",  "id": 3531834102,  "lat": 40.7221609,  "lon": -73.9887981},{  "type": "node",  "id": 3834459525,  "lat": 40.7239859,  "lon": -73.9880724},{  "type": "node",  "id": 3834459528,  "lat": 40.7246273,  "lon": -73.9876048},{  "type": "node",  "id": 3834459532,  "lat": 40.7258398,  "lon": -73.9867259},{  "type": "node",  "id": 3834459535,  "lat": 40.7270310,  "lon": -73.9858541},{  "type": "node",  "id": 3834459537,  "lat": 40.7252462,  "lon": -73.9871571},{  "type": "node",  "id": 3834459541,  "lat": 40.7234181,  "lon": -73.9884865},{  "type": "node",  "id": 3834459544,  "lat": 40.7264540,  "lon": -73.9862822},{  "type": "node",  "id": 3875091620,  "lat": 40.7252373,  "lon": -73.9871665},{  "type": "node",  "id": 3875091621,  "lat": 40.7258269,  "lon": -73.9867382},{  "type": "node",  "id": 3875091622,  "lat": 40.7264259,  "lon": -73.9863054},{  "type": "node",  "id": 3875091623,  "lat": 40.7270161,  "lon": -73.9858681},{  "type": "node",  "id": 4202899501,  "lat": 40.7228579,  "lon": -73.9881131},{  "type": "node",  "id": 4202908168,  "lat": 40.7230690,  "lon": -73.9893122},{  "type": "node",  "id": 4216045795,  "lat": 40.7260961,  "lon": -73.9836370},{  "type": "node",  "id": 4241743151,  "lat": 40.7233635,  "lon": -73.9884670},{  "type": "node",  "id": 4241743152,  "lat": 40.7229528,  "lon": -73.9884258},{  "type": "node",  "id": 4241743153,  "lat": 40.7232182,  "lon": -73.9883119},{  "type": "node",  "id": 4241743156,  "lat": 40.7229367,  "lon": -73.9888750},{  "type": "node",  "id": 4241743158,  "lat": 40.7233345,  "lon": -73.9884002},{  "type": "node",  "id": 4241743160,  "lat": 40.7232212,  "lon": -73.9884808},{  "type": "node",  "id": 4241743161,  "lat": 40.7228120,  "lon": -73.9884630},{  "type": "node",  "id": 4241743162,  "lat": 40.7233357,  "lon": -73.9885904},{  "type": "node",  "id": 4241743163,  "lat": 40.7230624,  "lon": -73.9887870},{  "type": "node",  "id": 4241748518,  "lat": 40.7230460,  "lon": -73.9885308},{  "type": "node",  "id": 4275822495,  "lat": 40.7273467,  "lon": -73.9885793},{  "type": "node",  "id": 4332491210,  "lat": 40.7227492,  "lon": -73.9878545},{  "type": "node",  "id": 4376122099,  "lat": 40.7222061,  "lon": -73.9887842},{  "type": "node",  "id": 4376122101,  "lat": 40.7255486,  "lon": -73.9897908},{  "type": "node",  "id": 4376122102,  "lat": 40.7261557,  "lon": -73.9893510},{  "type": "node",  "id": 4376122103,  "lat": 40.7267680,  "lon": -73.9889038},{  "type": "node",  "id": 4478001045,  "lat": 40.7225484,  "lon": -73.9870932},{  "type": "node",  "id": 4491355922,  "lat": 40.7229692,  "lon": -73.9877313},{  "type": "node",  "id": 4505948472,  "lat": 40.7233374,  "lon": -73.9865325},{  "type": "node",  "id": 4557539295,  "lat": 40.7229150,  "lon": -73.9888033},{  "type": "node",  "id": 4598752833,  "lat": 40.7226761,  "lon": -73.9880142},{  "type": "node",  "id": 4598752834,  "lat": 40.7227536,  "lon": -73.9877694},{  "type": "node",  "id": 4598752835,  "lat": 40.7227987,  "lon": -73.9879243},{  "type": "node",  "id": 4598752838,  "lat": 40.7229229,  "lon": -73.9877574},{  "type": "node",  "id": 4598752841,  "lat": 40.7226240,  "lon": -73.9878420},{  "type": "node",  "id": 4598752849,  "lat": 40.7225541,  "lon": -73.9879547},{  "type": "node",  "id": 4598752850,  "lat": 40.7226152,  "lon": -73.9870603},{  "type": "node",  "id": 4598752856,  "lat": 40.7223338,  "lon": -73.9863859},{  "type": "node",  "id": 4598752858,  "lat": 40.7228326,  "lon": -73.9878084},{  "type": "node",  "id": 4598752866,  "lat": 40.7222065,  "lon": -73.9864626},{  "type": "node",  "id": 4625620217,  "lat": 40.7219466,  "lon": -73.9901130},{  "type": "node",  "id": 4625620218,  "lat": 40.7219022,  "lon": -73.9899663},{  "type": "node",  "id": 4690139672,  "lat": 40.7217378,  "lon": -73.9901361},{  "type": "node",  "id": 4886266066,  "lat": 40.7218307,  "lon": -73.9883266},{  "type": "node",  "id": 4886404633,  "lat": 40.7241515,  "lon": -73.9884659},{  "type": "node",  "id": 4886404634,  "lat": 40.7237504,  "lon": -73.9875128},{  "type": "node",  "id": 4968494266,  "lat": 40.7238855,  "lon": -73.9880082},{  "type": "node",  "id": 4968494267,  "lat": 40.7240211,  "lon": -73.9879106},{  "type": "node",  "id": 4968494268,  "lat": 40.7240029,  "lon": -73.9881127},{  "type": "node",  "id": 4968494269,  "lat": 40.7238868,  "lon": -73.9878369},{  "type": "node",  "id": 5341393940,  "lat": 40.7254313,  "lon": -73.9868874},{  "type": "node",  "id": 5341393941,  "lat": 40.7244472,  "lon": -73.9876013},{  "type": "node",  "id": 5481937325,  "lat": 40.7262906,  "lon": -73.9858955},{  "type": "node",  "id": 5481937330,  "lat": 40.7263005,  "lon": -73.9878068},{  "type": "node",  "id": 5783020900,  "lat": 40.7226564,  "lon": -73.9870487},{  "type": "node",  "id": 5783020902,  "lat": 40.7224382,  "lon": -73.9872281},{  "type": "node",  "id": 5783020907,  "lat": 40.7225710,  "lon": -73.9871676},{  "type": "node",  "id": 5783020909,  "lat": 40.7225134,  "lon": -73.9869778},{  "type": "node",  "id": 5783020910,  "lat": 40.7223884,  "lon": -73.9870607},{  "type": "node",  "id": 5783022066,  "lat": 40.7230210,  "lon": -73.9894800},{  "type": "node",  "id": 5783028387,  "lat": 40.7234727,  "lon": -73.9901394},{  "type": "node",  "id": 5798966629,  "lat": 40.7229899,  "lon": -73.9885442},{  "type": "node",  "id": 5854325121,  "lat": 40.7216227,  "lon": -73.9890443},{  "type": "node",  "id": 5854381562,  "lat": 40.7216605,  "lon": -73.9889760},{  "type": "node",  "id": 5854381570,  "lat": 40.7215604,  "lon": -73.9890192},{  "type": "node",  "id": 5854381578,  "lat": 40.7216452,  "lon": -73.9892962},{  "type": "node",  "id": 5854382586,  "lat": 40.7217480,  "lon": -73.9892512},{  "type": "node",  "id": 6031953401,  "lat": 40.7250677,  "lon": -73.9867400},{  "type": "node",  "id": 6031953403,  "lat": 40.7261306,  "lon": -73.9855183},{  "type": "node",  "id": 6031953405,  "lat": 40.7258071,  "lon": -73.9847559},{  "type": "node",  "id": 6133355297,  "lat": 40.7256129,  "lon": -73.9867557},{  "type": "node",  "id": 6133355298,  "lat": 40.7243978,  "lon": -73.9876372},{  "type": "node",  "id": 6133388359,  "lat": 40.7235167,  "lon": -73.9882705},{  "type": "node",  "id": 6139477283,  "lat": 40.7225994,  "lon": -73.9889242},{  "type": "node",  "id": 6162064820,  "lat": 40.7218266,  "lon": -73.9889169},{  "type": "node",  "id": 42434148,  "lat": 40.7266660,  "lon": -73.9830900},{  "type": "node",  "id": 42434215,  "lat": 40.7260530,  "lon": -73.9835350},{  "type": "node",  "id": 42434226,  "lat": 40.7250730,  "lon": -73.9812110},{  "type": "node",  "id": 42434233,  "lat": 40.7240764,  "lon": -73.9788469},{  "type": "node",  "id": 42437346,  "lat": 40.7226710,  "lon": -73.9829640},{  "type": "node",  "id": 42437349,  "lat": 40.7216783,  "lon": -73.9806076},{  "type": "node",  "id": 42444832,  "lat": 40.7244940,  "lon": -73.9816370},{  "type": "node",  "id": 42444833,  "lat": 40.7234973,  "lon": -73.9792659},{  "type": "node",  "id": 42449023,  "lat": 40.7272610,  "lon": -73.9826570},{  "type": "node",  "id": 42453607,  "lat": 40.7220190,  "lon": -73.9834440},{  "type": "node",  "id": 42453610,  "lat": 40.7210227,  "lon": -73.9810814},{  "type": "node",  "id": 42455013,  "lat": 40.7264706,  "lon": -73.9770983},{  "type": "node",  "id": 42455083,  "lat": 40.7232780,  "lon": -73.9825240},{  "type": "node",  "id": 42455175,  "lat": 40.7222810,  "lon": -73.9801561},{  "type": "node",  "id": 42455396,  "lat": 40.7212770,  "lon": -73.9777880},{  "type": "node",  "id": 274306324,  "lat": 40.7255646,  "lon": -73.9823587},{  "type": "node",  "id": 3783146167,  "lat": 40.7210599,  "lon": -73.9811698},{  "type": "node",  "id": 4216045804,  "lat": 40.7273076,  "lon": -73.9827680},{  "type": "node",  "id": 4216045822,  "lat": 40.7251064,  "lon": -73.9812889},{  "type": "node",  "id": 4216045844,  "lat": 40.7250374,  "lon": -73.9811265},{  "type": "node",  "id": 4216045849,  "lat": 40.7267118,  "lon": -73.9831989},{  "type": "node",  "id": 4216055441,  "lat": 40.7260063,  "lon": -73.9834252},{  "type": "node",  "id": 4219836772,  "lat": 40.7209945,  "lon": -73.9810133},{  "type": "node",  "id": 5575804213,  "lat": 40.7240299,  "lon": -73.9805329},{  "type": "node",  "id": 42436899,  "lat": 40.7324705,  "lon": -74.0098682},{  "type": "node",  "id": 42436903,  "lat": 40.7324182,  "lon": -74.0102359},{  "type": "node",  "id": 42440532,  "lat": 40.7298231,  "lon": -74.0104777},{  "type": "node",  "id": 42448570,  "lat": 40.7291112,  "lon": -74.0105554},{  "type": "node",  "id": 42452008,  "lat": 40.7320066,  "lon": -74.0102717},{  "type": "node",  "id": 42452613,  "lat": 40.7338689,  "lon": -74.0100969},{  "type": "node",  "id": 42453442,  "lat": 40.7305689,  "lon": -74.0104027},{  "type": "node",  "id": 42453457,  "lat": 40.7331448,  "lon": -74.0101602},{  "type": "node",  "id": 246649346,  "lat": 40.7338017,  "lon": -74.0103147},{  "type": "node",  "id": 246857083,  "lat": 40.7312000,  "lon": -74.0105690},{  "type": "node",  "id": 246857084,  "lat": 40.7291341,  "lon": -74.0107672},{  "type": "node",  "id": 246874702,  "lat": 40.7337949,  "lon": -74.0101030},{  "type": "node",  "id": 246876728,  "lat": 40.7313887,  "lon": -74.0105464},{  "type": "node",  "id": 246876730,  "lat": 40.7313757,  "lon": -74.0103318},{  "type": "node",  "id": 246877762,  "lat": 40.7311878,  "lon": -74.0103450},{  "type": "node",  "id": 246878196,  "lat": 40.7325892,  "lon": -74.0104251},{  "type": "node",  "id": 246878284,  "lat": 40.7323082,  "lon": -74.0104519},{  "type": "node",  "id": 246878875,  "lat": 40.7297412,  "lon": -74.0106892},{  "type": "node",  "id": 246878876,  "lat": 40.7297271,  "lon": -74.0104850},{  "type": "node",  "id": 246890579,  "lat": 40.7298427,  "lon": -74.0106787},{  "type": "node",  "id": 591454081,  "lat": 40.7312869,  "lon": -74.0103381},{  "type": "node",  "id": 593816735,  "lat": 40.7297645,  "lon": -74.0108395},{  "type": "node",  "id": 1812608108,  "lat": 40.7290935,  "lon": -74.0103915},{  "type": "node",  "id": 3874650186,  "lat": 40.7298110,  "lon": -74.0103484},{  "type": "node",  "id": 3874650189,  "lat": 40.7305862,  "lon": -74.0106092},{  "type": "node",  "id": 3874650195,  "lat": 40.7338631,  "lon": -74.0099688},{  "type": "node",  "id": 4207143436,  "lat": 40.7283922,  "lon": -74.0106251},{  "type": "node",  "id": 4207143437,  "lat": 40.7294675,  "lon": -74.0107244},{  "type": "node",  "id": 4207143439,  "lat": 40.7304282,  "lon": -74.0106214},{  "type": "node",  "id": 4379778905,  "lat": 40.7297629,  "lon": -74.0107815},{  "type": "node",  "id": 4379778906,  "lat": 40.7297697,  "lon": -74.0107474},{  "type": "node",  "id": 4379778907,  "lat": 40.7297848,  "lon": -74.0107256},{  "type": "node",  "id": 4379778908,  "lat": 40.7298271,  "lon": -74.0107185},{  "type": "node",  "id": 4379778909,  "lat": 40.7298401,  "lon": -74.0107032},{  "type": "node",  "id": 4599130009,  "lat": 40.7305476,  "lon": -74.0101928},{  "type": "node",  "id": 4599130010,  "lat": 40.7312967,  "lon": -74.0105574},{  "type": "node",  "id": 4599130011,  "lat": 40.7319895,  "lon": -74.0100853},{  "type": "node",  "id": 4599130012,  "lat": 40.7324448,  "lon": -74.0104389},{  "type": "node",  "id": 4599130013,  "lat": 40.7338857,  "lon": -74.0103050},{  "type": "node",  "id": 4629486886,  "lat": 40.7292347,  "lon": -74.0107542},{  "type": "node",  "id": 4629486887,  "lat": 40.7292171,  "lon": -74.0105434},{  "type": "node",  "id": 4629488489,  "lat": 40.7290974,  "lon": -74.0104275},{  "type": "node",  "id": 4629488490,  "lat": 40.7312751,  "lon": -74.0102122},{  "type": "node",  "id": 4872777160,  "lat": 40.7305563,  "lon": -74.0102789},{  "type": "node",  "id": 4872777164,  "lat": 40.7319940,  "lon": -74.0101346},{  "type": "node",  "id": 4872777172,  "lat": 40.7324382,  "lon": -74.0100954},{  "type": "node",  "id": 4872777176,  "lat": 40.7325768,  "lon": -74.0102194},{  "type": "node",  "id": 4872777179,  "lat": 40.7331652,  "lon": -74.0100360},{  "type": "node",  "id": 5708228060,  "lat": 40.7323023,  "lon": -74.0102460},{  "type": "node",  "id": 42427898,  "lat": 40.7323690,  "lon": -74.0058160},{  "type": "node",  "id": 42430761,  "lat": 40.7294709,  "lon": -74.0051447},{  "type": "node",  "id": 42430770,  "lat": 40.7303210,  "lon": -74.0046880},{  "type": "node",  "id": 42430782,  "lat": 40.7307190,  "lon": -74.0044600},{  "type": "node",  "id": 42430791,  "lat": 40.7311289,  "lon": -74.0042257},{  "type": "node",  "id": 42430796,  "lat": 40.7318924,  "lon": -74.0037892},{  "type": "node",  "id": 42431273,  "lat": 40.7315444,  "lon": -74.0049377},{  "type": "node",  "id": 42431274,  "lat": 40.7313550,  "lon": -74.0055320},{  "type": "node",  "id": 42431275,  "lat": 40.7316248,  "lon": -74.0060166},{  "type": "node",  "id": 42431515,  "lat": 40.7287130,  "lon": -74.0032970},{  "type": "node",  "id": 42431517,  "lat": 40.7293440,  "lon": -74.0036570},{  "type": "node",  "id": 42431519,  "lat": 40.7298634,  "lon": -74.0039573},{  "type": "node",  "id": 42431524,  "lat": 40.7304470,  "lon": -74.0043030},{  "type": "node",  "id": 42431528,  "lat": 40.7310034,  "lon": -74.0046246},{  "type": "node",  "id": 42431531,  "lat": 40.7318361,  "lon": -74.0051065},{  "type": "node",  "id": 42431536,  "lat": 40.7324485,  "lon": -74.0054610},{  "type": "node",  "id": 42431703,  "lat": 40.7333700,  "lon": -74.0073990},{  "type": "node",  "id": 42431706,  "lat": 40.7333340,  "lon": -74.0070250},{  "type": "node",  "id": 42431789,  "lat": 40.7317466,  "lon": -74.0032895},{  "type": "node",  "id": 42431797,  "lat": 40.7329250,  "lon": -74.0039756},{  "type": "node",  "id": 42431802,  "lat": 40.7333991,  "lon": -74.0042514},{  "type": "node",  "id": 42436097,  "lat": 40.7274700,  "lon": -74.0088450},{  "type": "node",  "id": 42436100,  "lat": 40.7281950,  "lon": -74.0087240},{  "type": "node",  "id": 42436109,  "lat": 40.7289084,  "lon": -74.0085943},{  "type": "node",  "id": 42436115,  "lat": 40.7296297,  "lon": -74.0083997},{  "type": "node",  "id": 42436119,  "lat": 40.7303430,  "lon": -74.0082070},{  "type": "node",  "id": 42436122,  "lat": 40.7310668,  "lon": -74.0080090},{  "type": "node",  "id": 42436126,  "lat": 40.7317805,  "lon": -74.0078076},{  "type": "node",  "id": 42436129,  "lat": 40.7328706,  "lon": -74.0075146},{  "type": "node",  "id": 42436134,  "lat": 40.7335880,  "lon": -74.0073440},{  "type": "node",  "id": 42436369,  "lat": 40.7280500,  "lon": -74.0072030},{  "type": "node",  "id": 42436371,  "lat": 40.7287662,  "lon": -74.0070745},{  "type": "node",  "id": 42436374,  "lat": 40.7294927,  "lon": -74.0069526},{  "type": "node",  "id": 42436377,  "lat": 40.7302239,  "lon": -74.0068382},{  "type": "node",  "id": 42436381,  "lat": 40.7309450,  "lon": -74.0067209},{  "type": "node",  "id": 42436384,  "lat": 40.7316700,  "lon": -74.0066039},{  "type": "node",  "id": 42436390,  "lat": 40.7324250,  "lon": -74.0064889},{  "type": "node",  "id": 42436393,  "lat": 40.7330550,  "lon": -74.0063829},{  "type": "node",  "id": 42436396,  "lat": 40.7337430,  "lon": -74.0062792},{  "type": "node",  "id": 42436894,  "lat": 40.7326715,  "lon": -74.0086061},{  "type": "node",  "id": 42436968,  "lat": 40.7288533,  "lon": -74.0052717},{  "type": "node",  "id": 42439352,  "lat": 40.7275570,  "lon": -74.0096230},{  "type": "node",  "id": 42440527,  "lat": 40.7297049,  "lon": -74.0092082},{  "type": "node",  "id": 42444267,  "lat": 40.7332040,  "lon": -74.0098000},{  "type": "node",  "id": 42448563,  "lat": 40.7286179,  "lon": -74.0053112},{  "type": "node",  "id": 42448568,  "lat": 40.7289809,  "lon": -74.0093502},{  "type": "node",  "id": 42452001,  "lat": 40.7316336,  "lon": -74.0061759},{  "type": "node",  "id": 42452006,  "lat": 40.7318701,  "lon": -74.0087845},{  "type": "node",  "id": 42454635,  "lat": 40.7307980,  "lon": -74.0052950},{  "type": "node",  "id": 42454642,  "lat": 40.7311535,  "lon": -74.0089252},{  "type": "node",  "id": 42455200,  "lat": 40.7278806,  "lon": -74.0054267},{  "type": "node",  "id": 42455207,  "lat": 40.7282630,  "lon": -74.0094870},{  "type": "node",  "id": 42456464,  "lat": 40.7301040,  "lon": -74.0054750},{  "type": "node",  "id": 42456986,  "lat": 40.7304340,  "lon": -74.0090700},{  "type": "node",  "id": 42457735,  "lat": 40.7334010,  "lon": -74.0085540},{  "type": "node",  "id": 339044224,  "lat": 40.7331346,  "lon": -74.0058581},{  "type": "node",  "id": 529556011,  "lat": 40.7329359,  "lon": -74.0039421},{  "type": "node",  "id": 1319476955,  "lat": 40.7293802,  "lon": -74.0053534},{  "type": "node",  "id": 1319476957,  "lat": 40.7293676,  "lon": -74.0054420},{  "type": "node",  "id": 1812608107,  "lat": 40.7289905,  "lon": -74.0094387},{  "type": "node",  "id": 2073622000,  "lat": 40.7294177,  "lon": -74.0061177},{  "type": "node",  "id": 2073622011,  "lat": 40.7301645,  "lon": -74.0061488},{  "type": "node",  "id": 2522434121,  "lat": 40.7286872,  "lon": -74.0032616},{  "type": "node",  "id": 3546132354,  "lat": 40.7295395,  "lon": -74.0049873},{  "type": "node",  "id": 4205565493,  "lat": 40.7327034,  "lon": -74.0033268},{  "type": "node",  "id": 4205565494,  "lat": 40.7328151,  "lon": -74.0032632},{  "type": "node",  "id": 4207862995,  "lat": 40.7313315,  "lon": -74.0066591},{  "type": "node",  "id": 4207862996,  "lat": 40.7327338,  "lon": -74.0064364},{  "type": "node",  "id": 4320035374,  "lat": 40.7322874,  "lon": -74.0037175},{  "type": "node",  "id": 4578212489,  "lat": 40.7329144,  "lon": -74.0073439},{  "type": "node",  "id": 4578212490,  "lat": 40.7328136,  "lon": -74.0077427},{  "type": "node",  "id": 4618691808,  "lat": 40.7321786,  "lon": -74.0035391},{  "type": "node",  "id": 4679523599,  "lat": 40.7323894,  "lon": -74.0036613},{  "type": "node",  "id": 4685597889,  "lat": 40.7301502,  "lon": -74.0059895},{  "type": "node",  "id": 4816151662,  "lat": 40.7316676,  "lon": -74.0056551},{  "type": "node",  "id": 4816151663,  "lat": 40.7316331,  "lon": -74.0058828},{  "type": "node",  "id": 4816151664,  "lat": 40.7316345,  "lon": -74.0058028},{  "type": "node",  "id": 4816151665,  "lat": 40.7316473,  "lon": -74.0057227},{  "type": "node",  "id": 4816151666,  "lat": 40.7315061,  "lon": -74.0056091},{  "type": "node",  "id": 4816151667,  "lat": 40.7315886,  "lon": -74.0058754},{  "type": "node",  "id": 4816151668,  "lat": 40.7315712,  "lon": -74.0057204},{  "type": "node",  "id": 4816151669,  "lat": 40.7315542,  "lon": -74.0056482},{  "type": "node",  "id": 4816151670,  "lat": 40.7315638,  "lon": -74.0056778},{  "type": "node",  "id": 4816151671,  "lat": 40.7315362,  "lon": -74.0056262},{  "type": "node",  "id": 4816151672,  "lat": 40.7324376,  "lon": -74.0032654},{  "type": "node",  "id": 4841945993,  "lat": 40.7337978,  "lon": -74.0044748},{  "type": "node",  "id": 4841945994,  "lat": 40.7336883,  "lon": -74.0044645},{  "type": "node",  "id": 4841945995,  "lat": 40.7333791,  "lon": -74.0043727},{  "type": "node",  "id": 5151708185,  "lat": 40.7276779,  "lon": -74.0033030},{  "type": "node",  "id": 5151983466,  "lat": 40.7324316,  "lon": -74.0033287},{  "type": "node",  "id": 5151983467,  "lat": 40.7321384,  "lon": -74.0036485},{  "type": "node",  "id": 5151983468,  "lat": 40.7323187,  "lon": -74.0035456},{  "type": "node",  "id": 5357273833,  "lat": 40.7336649,  "lon": -74.0068154},{  "type": "node",  "id": 5357282419,  "lat": 40.7333670,  "lon": -74.0069518},{  "type": "node",  "id": 5646503062,  "lat": 40.7329943,  "lon": -74.0067979},{  "type": "node",  "id": 5690453353,  "lat": 40.7321213,  "lon": -74.0035058},{  "type": "node",  "id": 5837088104,  "lat": 40.7325066,  "lon": -74.0034387},{  "type": "node",  "id": 5837088105,  "lat": 40.7322539,  "lon": -74.0035825},{  "type": "node",  "id": 5837101292,  "lat": 40.7322903,  "lon": -74.0036384},{  "type": "node",  "id": 5837101293,  "lat": 40.7322962,  "lon": -74.0036887},{  "type": "node",  "id": 5837101294,  "lat": 40.7324479,  "lon": -74.0033705},{  "type": "node",  "id": 5837109293,  "lat": 40.7293669,  "lon": -74.0055533},{  "type": "node",  "id": 5837116451,  "lat": 40.7295765,  "lon": -74.0051048},{  "type": "node",  "id": 5837116452,  "lat": 40.7293655,  "lon": -74.0051845},{  "type": "node",  "id": 5866337889,  "lat": 40.7293755,  "lon": -74.0035569},{  "type": "node",  "id": 5866337890,  "lat": 40.7303743,  "lon": -74.0042600},{  "type": "node",  "id": 6210448835,  "lat": 40.7314385,  "lon": -74.0066848},{  "type": "node",  "id": 6210448838,  "lat": 40.7329657,  "lon": -74.0064980},{  "type": "node",  "id": 6210448839,  "lat": 40.7328399,  "lon": -74.0064642},{  "type": "node",  "id": 6210448840,  "lat": 40.7330397,  "lon": -74.0064875},{  "type": "node",  "id": 6210448846,  "lat": 40.7302326,  "lon": -74.0069382},{  "type": "node",  "id": 6210448848,  "lat": 40.7288792,  "lon": -74.0071529},{  "type": "node",  "id": 6210448853,  "lat": 40.7295023,  "lon": -74.0070541},{  "type": "node",  "id": 6210448859,  "lat": 40.7315804,  "lon": -74.0067210},{  "type": "node",  "id": 6210448861,  "lat": 40.7299859,  "lon": -74.0069166},{  "type": "node",  "id": 6210448862,  "lat": 40.7316793,  "lon": -74.0067051},{  "type": "node",  "id": 6210448864,  "lat": 40.7301256,  "lon": -74.0069564},{  "type": "node",  "id": 6210448866,  "lat": 40.7298840,  "lon": -74.0068914},{  "type": "node",  "id": 6210448867,  "lat": 40.7309534,  "lon": -74.0068238},{  "type": "node",  "id": 6210448868,  "lat": 40.7337293,  "lon": -74.0063785},{  "type": "node",  "id": 6213204036,  "lat": 40.7322862,  "lon": -74.0035644},{  "type": "node",  "id": 6213204050,  "lat": 40.7317582,  "lon": -74.0032564},{  "type": "node",  "id": 6213204054,  "lat": 40.7334051,  "lon": -74.0042100},{  "type": "node",  "id": 42439099,  "lat": 40.7375567,  "lon": -74.0098066},{  "type": "node",  "id": 42440946,  "lat": 40.7359739,  "lon": -74.0099198},{  "type": "node",  "id": 42442081,  "lat": 40.7342193,  "lon": -74.0100656},{  "type": "node",  "id": 42449098,  "lat": 40.7353148,  "lon": -74.0099713},{  "type": "node",  "id": 42453468,  "lat": 40.7345430,  "lon": -74.0100371},{  "type": "node",  "id": 42453472,  "lat": 40.7369494,  "lon": -74.0098340},{  "type": "node",  "id": 246647917,  "lat": 40.7380225,  "lon": -74.0098426},{  "type": "node",  "id": 246647920,  "lat": 40.7392972,  "lon": -74.0098512},{  "type": "node",  "id": 246647957,  "lat": 40.7396791,  "lon": -74.0098864},{  "type": "node",  "id": 246647958,  "lat": 40.7394965,  "lon": -74.0099809},{  "type": "node",  "id": 246647959,  "lat": 40.7388302,  "lon": -74.0101345},{  "type": "node",  "id": 246647962,  "lat": 40.7378777,  "lon": -74.0100502},{  "type": "node",  "id": 246648627,  "lat": 40.7393286,  "lon": -74.0100471},{  "type": "node",  "id": 246649342,  "lat": 40.7376401,  "lon": -74.0100301},{  "type": "node",  "id": 246649343,  "lat": 40.7374760,  "lon": -74.0100259},{  "type": "node",  "id": 246649344,  "lat": 40.7354128,  "lon": -74.0101760},{  "type": "node",  "id": 246649345,  "lat": 40.7352410,  "lon": -74.0101936},{  "type": "node",  "id": 246649427,  "lat": 40.7385169,  "lon": -74.0101186},{  "type": "node",  "id": 246649429,  "lat": 40.7388125,  "lon": -74.0099059},{  "type": "node",  "id": 246861652,  "lat": 40.7385168,  "lon": -74.0098941},{  "type": "node",  "id": 246862616,  "lat": 40.7353986,  "lon": -74.0099632},{  "type": "node",  "id": 246862852,  "lat": 40.7352298,  "lon": -74.0099784},{  "type": "node",  "id": 246874387,  "lat": 40.7339775,  "lon": -74.0102945},{  "type": "node",  "id": 246874388,  "lat": 40.7339655,  "lon": -74.0100879},{  "type": "node",  "id": 373895089,  "lat": 40.7389902,  "lon": -74.0099051},{  "type": "node",  "id": 373895090,  "lat": 40.7394207,  "lon": -74.0098083},{  "type": "node",  "id": 373895306,  "lat": 40.7390280,  "lon": -74.0101158},{  "type": "node",  "id": 593816397,  "lat": 40.7400455,  "lon": -74.0098956},{  "type": "node",  "id": 593816490,  "lat": 40.7376362,  "lon": -74.0098109},{  "type": "node",  "id": 2376384353,  "lat": 40.7389136,  "lon": -74.0101314},{  "type": "node",  "id": 2376384363,  "lat": 40.7391183,  "lon": -74.0100995},{  "type": "node",  "id": 2376384365,  "lat": 40.7394219,  "lon": -74.0100145},{  "type": "node",  "id": 3884622507,  "lat": 40.7386486,  "lon": -74.0100624},{  "type": "node",  "id": 3884622508,  "lat": 40.7387332,  "lon": -74.0100126},{  "type": "node",  "id": 3884622509,  "lat": 40.7387858,  "lon": -74.0099658},{  "type": "node",  "id": 4599130014,  "lat": 40.7353185,  "lon": -74.0101858},{  "type": "node",  "id": 4599130016,  "lat": 40.7375621,  "lon": -74.0100281},{  "type": "node",  "id": 4872761059,  "lat": 40.7345636,  "lon": -74.0098981},{  "type": "node",  "id": 4872761060,  "lat": 40.7353316,  "lon": -74.0098437},{  "type": "node",  "id": 42429390,  "lat": 40.7394180,  "lon": -74.0038650},{  "type": "node",  "id": 42429394,  "lat": 40.7400297,  "lon": -74.0053043},{  "type": "node",  "id": 42431149,  "lat": 40.7394110,  "lon": -74.0057758},{  "type": "node",  "id": 42431151,  "lat": 40.7396310,  "lon": -74.0063340},{  "type": "node",  "id": 42431154,  "lat": 40.7402946,  "lon": -74.0079644},{  "type": "node",  "id": 42431497,  "lat": 40.7387661,  "lon": -74.0034031},{  "type": "node",  "id": 42431500,  "lat": 40.7387561,  "lon": -74.0038600},{  "type": "node",  "id": 42431503,  "lat": 40.7387641,  "lon": -74.0055102},{  "type": "node",  "id": 42431505,  "lat": 40.7387622,  "lon": -74.0064600},{  "type": "node",  "id": 42431508,  "lat": 40.7387590,  "lon": -74.0081030},{  "type": "node",  "id": 42431511,  "lat": 40.7387628,  "lon": -74.0095413},{  "type": "node",  "id": 42431804,  "lat": 40.7340100,  "lon": -74.0045610},{  "type": "node",  "id": 42431805,  "lat": 40.7346517,  "lon": -74.0047458},{  "type": "node",  "id": 42431812,  "lat": 40.7352971,  "lon": -74.0049302},{  "type": "node",  "id": 42431814,  "lat": 40.7359654,  "lon": -74.0051271},{  "type": "node",  "id": 42431829,  "lat": 40.7367040,  "lon": -74.0053520},{  "type": "node",  "id": 42431831,  "lat": 40.7367882,  "lon": -74.0054231},{  "type": "node",  "id": 42431844,  "lat": 40.7387583,  "lon": -74.0035485},{  "type": "node",  "id": 42435578,  "lat": 40.7375429,  "lon": -74.0047793},{  "type": "node",  "id": 42435581,  "lat": 40.7381289,  "lon": -74.0041378},{  "type": "node",  "id": 42435583,  "lat": 40.7383899,  "lon": -74.0038576},{  "type": "node",  "id": 42436135,  "lat": 40.7342960,  "lon": -74.0071620},{  "type": "node",  "id": 42436139,  "lat": 40.7349954,  "lon": -74.0069783},{  "type": "node",  "id": 42436142,  "lat": 40.7357317,  "lon": -74.0068010},{  "type": "node",  "id": 42436146,  "lat": 40.7364224,  "lon": -74.0066187},{  "type": "node",  "id": 42436149,  "lat": 40.7369570,  "lon": -74.0064980},{  "type": "node",  "id": 42436152,  "lat": 40.7375500,  "lon": -74.0064870},{  "type": "node",  "id": 42436155,  "lat": 40.7381422,  "lon": -74.0064700},{  "type": "node",  "id": 42436159,  "lat": 40.7394050,  "lon": -74.0064490},{  "type": "node",  "id": 42436400,  "lat": 40.7344390,  "lon": -74.0061609},{  "type": "node",  "id": 42436404,  "lat": 40.7351330,  "lon": -74.0060475},{  "type": "node",  "id": 42436407,  "lat": 40.7358460,  "lon": -74.0059318},{  "type": "node",  "id": 42436411,  "lat": 40.7365313,  "lon": -74.0058168},{  "type": "node",  "id": 42436414,  "lat": 40.7371946,  "lon": -74.0057723},{  "type": "node",  "id": 42436423,  "lat": 40.7375527,  "lon": -74.0057088},{  "type": "node",  "id": 42436427,  "lat": 40.7381397,  "lon": -74.0056100},{  "type": "node",  "id": 42436431,  "lat": 40.7394127,  "lon": -74.0054071},{  "type": "node",  "id": 42439078,  "lat": 40.7375440,  "lon": -74.0038660},{  "type": "node",  "id": 42439096,  "lat": 40.7375540,  "lon": -74.0082060},{  "type": "node",  "id": 42440580,  "lat": 40.7404482,  "lon": -74.0090873},{  "type": "node",  "id": 42440940,  "lat": 40.7368390,  "lon": -74.0037140},{  "type": "node",  "id": 42440943,  "lat": 40.7361909,  "lon": -74.0083227},{  "type": "node",  "id": 42442078,  "lat": 40.7344770,  "lon": -74.0084620},{  "type": "node",  "id": 42449076,  "lat": 40.7362021,  "lon": -74.0035332},{  "type": "node",  "id": 42449089,  "lat": 40.7355248,  "lon": -74.0083743},{  "type": "node",  "id": 42452610,  "lat": 40.7340940,  "lon": -74.0084910},{  "type": "node",  "id": 42453477,  "lat": 40.7381515,  "lon": -74.0096057},{  "type": "node",  "id": 42453485,  "lat": 40.7394724,  "lon": -74.0096207},{  "type": "node",  "id": 42453489,  "lat": 40.7395950,  "lon": -74.0095747},{  "type": "node",  "id": 42453502,  "lat": 40.7399410,  "lon": -74.0093877},{  "type": "node",  "id": 42453510,  "lat": 40.7400841,  "lon": -74.0092933},{  "type": "node",  "id": 42453517,  "lat": 40.7402401,  "lon": -74.0091989},{  "type": "node",  "id": 42454742,  "lat": 40.7381290,  "lon": -74.0038740},{  "type": "node",  "id": 42454758,  "lat": 40.7381472,  "lon": -74.0081540},{  "type": "node",  "id": 42455357,  "lat": 40.7393804,  "lon": -74.0080510},{  "type": "node",  "id": 42457421,  "lat": 40.7355308,  "lon": -74.0033399},{  "type": "node",  "id": 42457426,  "lat": 40.7347802,  "lon": -74.0084332},{  "type": "node",  "id": 42459436,  "lat": 40.7369530,  "lon": -74.0082540},{  "type": "node",  "id": 246647921,  "lat": 40.7395638,  "lon": -74.0097482},{  "type": "node",  "id": 246647922,  "lat": 40.7399150,  "lon": -74.0095508},{  "type": "node",  "id": 246647923,  "lat": 40.7403377,  "lon": -74.0092847},{  "type": "node",  "id": 246647955,  "lat": 40.7404589,  "lon": -74.0094104},{  "type": "node",  "id": 246647956,  "lat": 40.7398505,  "lon": -74.0097818},{  "type": "node",  "id": 246647994,  "lat": 40.7402587,  "lon": -74.0097239},{  "type": "node",  "id": 246649433,  "lat": 40.7390267,  "lon": -74.0096742},{  "type": "node",  "id": 246882782,  "lat": 40.7382209,  "lon": -74.0096665},{  "type": "node",  "id": 246882783,  "lat": 40.7387482,  "lon": -74.0095808},{  "type": "node",  "id": 246882949,  "lat": 40.7374682,  "lon": -74.0097994},{  "type": "node",  "id": 247402950,  "lat": 40.7369428,  "lon": -74.0058581},{  "type": "node",  "id": 247403144,  "lat": 40.7369934,  "lon": -74.0058325},{  "type": "node",  "id": 247403145,  "lat": 40.7369625,  "lon": -74.0058390},{  "type": "node",  "id": 247403147,  "lat": 40.7370312,  "lon": -74.0056895},{  "type": "node",  "id": 247403149,  "lat": 40.7370734,  "lon": -74.0057185},{  "type": "node",  "id": 373896719,  "lat": 40.7397134,  "lon": -74.0096709},{  "type": "node",  "id": 373896721,  "lat": 40.7400906,  "lon": -74.0094392},{  "type": "node",  "id": 373896889,  "lat": 40.7401886,  "lon": -74.0095717},{  "type": "node",  "id": 529555931,  "lat": 40.7366174,  "lon": -74.0052767},{  "type": "node",  "id": 593816275,  "lat": 40.7393541,  "lon": -74.0096237},{  "type": "node",  "id": 593816391,  "lat": 40.7401018,  "lon": -74.0096253},{  "type": "node",  "id": 719516045,  "lat": 40.7393884,  "lon": -74.0079007},{  "type": "node",  "id": 1042457810,  "lat": 40.7394682,  "lon": -74.0080399},{  "type": "node",  "id": 1042457827,  "lat": 40.7393633,  "lon": -74.0082108},{  "type": "node",  "id": 1096111017,  "lat": 40.7397385,  "lon": -74.0080185},{  "type": "node",  "id": 1272608956,  "lat": 40.7394833,  "lon": -74.0082042},{  "type": "node",  "id": 1412724862,  "lat": 40.7367223,  "lon": -74.0057039},{  "type": "node",  "id": 1412724863,  "lat": 40.7366771,  "lon": -74.0057465},{  "type": "node",  "id": 1412724864,  "lat": 40.7366357,  "lon": -74.0057694},{  "type": "node",  "id": 1578334770,  "lat": 40.7393596,  "lon": -74.0092351},{  "type": "node",  "id": 1578334788,  "lat": 40.7393584,  "lon": -74.0095224},{  "type": "node",  "id": 1701920956,  "lat": 40.7386932,  "lon": -74.0096319},{  "type": "node",  "id": 1701920959,  "lat": 40.7367728,  "lon": -74.0056494},{  "type": "node",  "id": 1701920974,  "lat": 40.7388417,  "lon": -74.0097885},{  "type": "node",  "id": 1701920975,  "lat": 40.7387256,  "lon": -74.0096090},{  "type": "node",  "id": 1701920986,  "lat": 40.7381820,  "lon": -74.0096495},{  "type": "node",  "id": 1701920988,  "lat": 40.7381607,  "lon": -74.0096264},{  "type": "node",  "id": 1701921010,  "lat": 40.7365961,  "lon": -74.0057912},{  "type": "node",  "id": 1825739972,  "lat": 40.7367426,  "lon": -74.0053777},{  "type": "node",  "id": 1825739992,  "lat": 40.7369314,  "lon": -74.0059199},{  "type": "node",  "id": 1825739995,  "lat": 40.7369336,  "lon": -74.0058869},{  "type": "node",  "id": 1825739998,  "lat": 40.7369372,  "lon": -74.0058726},{  "type": "node",  "id": 1825740002,  "lat": 40.7369512,  "lon": -74.0058466},{  "type": "node",  "id": 1825740008,  "lat": 40.7370500,  "lon": -74.0057052},{  "type": "node",  "id": 1825740011,  "lat": 40.7370996,  "lon": -74.0057304},{  "type": "node",  "id": 2076950449,  "lat": 40.7362525,  "lon": -74.0058746},{  "type": "node",  "id": 2076950450,  "lat": 40.7363345,  "lon": -74.0052340},{  "type": "node",  "id": 2208862762,  "lat": 40.7404046,  "lon": -74.0095837},{  "type": "node",  "id": 2823299558,  "lat": 40.7398424,  "lon": -74.0082761},{  "type": "node",  "id": 2823299559,  "lat": 40.7396539,  "lon": -74.0082925},{  "type": "node",  "id": 2823299560,  "lat": 40.7396498,  "lon": -74.0081980},{  "type": "node",  "id": 2823299561,  "lat": 40.7395077,  "lon": -74.0082076},{  "type": "node",  "id": 2823299562,  "lat": 40.7395067,  "lon": -74.0081831},{  "type": "node",  "id": 2823299564,  "lat": 40.7398195,  "lon": -74.0081818},{  "type": "node",  "id": 2823299565,  "lat": 40.7397398,  "lon": -74.0081865},{  "type": "node",  "id": 2823299566,  "lat": 40.7396910,  "lon": -74.0081897},{  "type": "node",  "id": 2823299567,  "lat": 40.7396240,  "lon": -74.0081956},{  "type": "node",  "id": 2823299568,  "lat": 40.7395317,  "lon": -74.0082015},{  "type": "node",  "id": 2823299569,  "lat": 40.7398416,  "lon": -74.0082385},{  "type": "node",  "id": 2823299570,  "lat": 40.7400784,  "lon": -74.0082200},{  "type": "node",  "id": 2823299571,  "lat": 40.7398395,  "lon": -74.0081812},{  "type": "node",  "id": 2823299572,  "lat": 40.7401304,  "lon": -74.0082177},{  "type": "node",  "id": 2823299573,  "lat": 40.7401296,  "lon": -74.0081826},{  "type": "node",  "id": 2823299577,  "lat": 40.7404181,  "lon": -74.0080646},{  "type": "node",  "id": 2823300805,  "lat": 40.7401469,  "lon": -74.0081787},{  "type": "node",  "id": 2845912197,  "lat": 40.7403260,  "lon": -74.0081403},{  "type": "node",  "id": 2921544667,  "lat": 40.7398244,  "lon": -74.0082777},{  "type": "node",  "id": 2921544680,  "lat": 40.7399344,  "lon": -74.0082313},{  "type": "node",  "id": 3874650201,  "lat": 40.7359933,  "lon": -74.0097774},{  "type": "node",  "id": 3874650204,  "lat": 40.7369497,  "lon": -74.0097126},{  "type": "node",  "id": 3874650207,  "lat": 40.7375543,  "lon": -74.0096737},{  "type": "node",  "id": 3884622510,  "lat": 40.7388887,  "lon": -74.0097245},{  "type": "node",  "id": 3884622511,  "lat": 40.7389561,  "lon": -74.0096913},{  "type": "node",  "id": 4015418428,  "lat": 40.7401803,  "lon": -74.0059242},{  "type": "node",  "id": 4207862997,  "lat": 40.7341347,  "lon": -74.0062078},{  "type": "node",  "id": 4207862998,  "lat": 40.7355264,  "lon": -74.0059831},{  "type": "node",  "id": 4492821528,  "lat": 40.7385088,  "lon": -74.0037131},{  "type": "node",  "id": 4557517555,  "lat": 40.7402610,  "lon": -74.0058640},{  "type": "node",  "id": 4599130015,  "lat": 40.7369498,  "lon": -74.0096501},{  "type": "node",  "id": 4599130017,  "lat": 40.7393596,  "lon": -74.0094516},{  "type": "node",  "id": 4634753632,  "lat": 40.7404039,  "lon": -74.0081162},{  "type": "node",  "id": 4685453632,  "lat": 40.7399958,  "lon": -74.0052246},{  "type": "node",  "id": 4685453633,  "lat": 40.7394123,  "lon": -74.0053066},{  "type": "node",  "id": 4685453634,  "lat": 40.7387636,  "lon": -74.0054159},{  "type": "node",  "id": 4685453635,  "lat": 40.7381390,  "lon": -74.0055107},{  "type": "node",  "id": 4685453636,  "lat": 40.7375516,  "lon": -74.0056066},{  "type": "node",  "id": 4685453637,  "lat": 40.7371329,  "lon": -74.0056753},{  "type": "node",  "id": 4685453638,  "lat": 40.7370995,  "lon": -74.0056787},{  "type": "node",  "id": 4685453639,  "lat": 40.7370731,  "lon": -74.0056765},{  "type": "node",  "id": 4685453640,  "lat": 40.7370526,  "lon": -74.0056584},{  "type": "node",  "id": 4685453641,  "lat": 40.7370102,  "lon": -74.0056046},{  "type": "node",  "id": 4685453642,  "lat": 40.7369833,  "lon": -74.0055727},{  "type": "node",  "id": 4685462237,  "lat": 40.7390464,  "lon": -74.0054685},{  "type": "node",  "id": 4685462238,  "lat": 40.7388658,  "lon": -74.0053974},{  "type": "node",  "id": 4685462239,  "lat": 40.7389270,  "lon": -74.0054197},{  "type": "node",  "id": 4685462240,  "lat": 40.7378489,  "lon": -74.0056556},{  "type": "node",  "id": 4685462241,  "lat": 40.7376743,  "lon": -74.0055866},{  "type": "node",  "id": 4685462242,  "lat": 40.7377371,  "lon": -74.0056114},{  "type": "node",  "id": 4685465960,  "lat": 40.7371253,  "lon": -74.0058038},{  "type": "node",  "id": 4741865080,  "lat": 40.7364619,  "lon": -74.0082979},{  "type": "node",  "id": 4816137911,  "lat": 40.7386619,  "lon": -74.0036491},{  "type": "node",  "id": 5244251222,  "lat": 40.7369496,  "lon": -74.0063301},{  "type": "node",  "id": 5331842131,  "lat": 40.7377954,  "lon": -74.0045029},{  "type": "node",  "id": 5849926991,  "lat": 40.7387557,  "lon": -74.0036197},{  "type": "node",  "id": 5849926992,  "lat": 40.7387707,  "lon": -74.0032748},{  "type": "node",  "id": 5849948634,  "lat": 40.7386815,  "lon": -74.0054272},{  "type": "node",  "id": 5875541578,  "lat": 40.7394359,  "lon": -74.0081865},{  "type": "node",  "id": 6210448836,  "lat": 40.7366994,  "lon": -74.0058430},{  "type": "node",  "id": 6210448837,  "lat": 40.7364742,  "lon": -74.0059417},{  "type": "node",  "id": 6210448842,  "lat": 40.7342232,  "lon": -74.0062407},{  "type": "node",  "id": 6210448843,  "lat": 40.7365165,  "lon": -74.0059266},{  "type": "node",  "id": 6210448845,  "lat": 40.7365100,  "lon": -74.0059745},{  "type": "node",  "id": 6210448847,  "lat": 40.7366211,  "lon": -74.0058909},{  "type": "node",  "id": 6210448849,  "lat": 40.7367598,  "lon": -74.0057931},{  "type": "node",  "id": 6210448850,  "lat": 40.7368941,  "lon": -74.0056420},{  "type": "node",  "id": 6210448851,  "lat": 40.7364606,  "lon": -74.0058314},{  "type": "node",  "id": 6210448855,  "lat": 40.7344236,  "lon": -74.0062687},{  "type": "node",  "id": 6210448856,  "lat": 40.7356141,  "lon": -74.0060139},{  "type": "node",  "id": 6210448857,  "lat": 40.7358319,  "lon": -74.0060412},{  "type": "node",  "id": 6210448860,  "lat": 40.7351165,  "lon": -74.0061590},{  "type": "node",  "id": 6210448863,  "lat": 40.7357274,  "lon": -74.0060555},{  "type": "node",  "id": 6210448865,  "lat": 40.7343337,  "lon": -74.0062796},{  "type": "node",  "id": 6211332476,  "lat": 40.7387090,  "lon": -74.0034703},{  "type": "node",  "id": 6211332477,  "lat": 40.7381299,  "lon": -74.0042697},{  "type": "node",  "id": 6211332478,  "lat": 40.7368399,  "lon": -74.0055757},{  "type": "node",  "id": 6211334185,  "lat": 40.7370068,  "lon": -74.0055470},{  "type": "node",  "id": 6211334192,  "lat": 40.7369752,  "lon": -74.0056338},{  "type": "node",  "id": 6211334195,  "lat": 40.7369482,  "lon": -74.0056010},{  "type": "node",  "id": 6211334197,  "lat": 40.7387113,  "lon": -74.0036261},{  "type": "node",  "id": 6211334198,  "lat": 40.7375446,  "lon": -74.0049372},{  "type": "node",  "id": 6211334201,  "lat": 40.7369344,  "lon": -74.0054703},{  "type": "node",  "id": 6211334202,  "lat": 40.7368828,  "lon": -74.0055274},{  "type": "node",  "id": 6211334203,  "lat": 40.7387091,  "lon": -74.0035998},{  "type": "node",  "id": 6211334204,  "lat": 40.7385029,  "lon": -74.0038583},{  "type": "node",  "id": 6211527559,  "lat": 40.7368364,  "lon": -74.0054745},{  "type": "node",  "id": 6213204034,  "lat": 40.7353019,  "lon": -74.0048972},{  "type": "node",  "id": 6213204051,  "lat": 40.7340147,  "lon": -74.0045285},{  "type": "node",  "id": 6213204061,  "lat": 40.7366110,  "lon": -74.0053140},{  "type": "node",  "id": 6213204071,  "lat": 40.7359709,  "lon": -74.0050902},{  "type": "node",  "id": 6213204074,  "lat": 40.7346569,  "lon": -74.0047111},{  "type": "node",  "id": 6213315375,  "lat": 40.7388162,  "lon": -74.0034834},{  "type": "node",  "id": 6213315380,  "lat": 40.7388221,  "lon": -74.0033378},{  "type": "node",  "id": 42427390,  "lat": 40.7458160,  "lon": -74.0055350},{  "type": "node",  "id": 42427393,  "lat": 40.7467950,  "lon": -74.0078599},{  "type": "node",  "id": 42429412,  "lat": 40.7409399,  "lon": -74.0074994},{  "type": "node",  "id": 42429431,  "lat": 40.7414704,  "lon": -74.0087400},{  "type": "node",  "id": 42429663,  "lat": 40.7434761,  "lon": -74.0035806},{  "type": "node",  "id": 42429664,  "lat": 40.7446460,  "lon": -74.0063850},{  "type": "node",  "id": 42429666,  "lat": 40.7453774,  "lon": -74.0081260},{  "type": "node",  "id": 42431156,  "lat": 40.7407313,  "lon": -74.0089600},{  "type": "node",  "id": 42436439,  "lat": 40.7408505,  "lon": -74.0051265},{  "type": "node",  "id": 42439280,  "lat": 40.7428829,  "lon": -74.0039992},{  "type": "node",  "id": 42439286,  "lat": 40.7440610,  "lon": -74.0068110},{  "type": "node",  "id": 42439295,  "lat": 40.7446947,  "lon": -74.0082461},{  "type": "node",  "id": 42443676,  "lat": 40.7463948,  "lon": -74.0051067},{  "type": "node",  "id": 42444897,  "lat": 40.7409462,  "lon": -74.0053437},{  "type": "node",  "id": 42444900,  "lat": 40.7416507,  "lon": -74.0049404},{  "type": "node",  "id": 42444909,  "lat": 40.7422885,  "lon": -74.0044325},{  "type": "node",  "id": 42445766,  "lat": 40.7428340,  "lon": -74.0077050},{  "type": "node",  "id": 42445769,  "lat": 40.7431693,  "lon": -74.0085372},{  "type": "node",  "id": 42449341,  "lat": 40.7434700,  "lon": -74.0072380},{  "type": "node",  "id": 42449345,  "lat": 40.7439579,  "lon": -74.0083751},{  "type": "node",  "id": 42454325,  "lat": 40.7416402,  "lon": -74.0069928},{  "type": "node",  "id": 42454328,  "lat": 40.7421216,  "lon": -74.0081809},{  "type": "node",  "id": 42454381,  "lat": 40.7452310,  "lon": -74.0059600},{  "type": "node",  "id": 246579724,  "lat": 40.7418659,  "lon": -74.0087955},{  "type": "node",  "id": 246579759,  "lat": 40.7440325,  "lon": -74.0085628},{  "type": "node",  "id": 246579760,  "lat": 40.7419941,  "lon": -74.0089797},{  "type": "node",  "id": 246580346,  "lat": 40.7447675,  "lon": -74.0084362},{  "type": "node",  "id": 246580982,  "lat": 40.7441719,  "lon": -74.0085360},{  "type": "node",  "id": 246647924,  "lat": 40.7406563,  "lon": -74.0091302},{  "type": "node",  "id": 246647925,  "lat": 40.7409750,  "lon": -74.0090272},{  "type": "node",  "id": 246647926,  "lat": 40.7415408,  "lon": -74.0088727},{  "type": "node",  "id": 246647952,  "lat": 40.7415260,  "lon": -74.0090788},{  "type": "node",  "id": 246647953,  "lat": 40.7412671,  "lon": -74.0091342},{  "type": "node",  "id": 246647954,  "lat": 40.7406853,  "lon": -74.0093028},{  "type": "node",  "id": 246647987,  "lat": 40.7436939,  "lon": -74.0088136},{  "type": "node",  "id": 246647988,  "lat": 40.7433386,  "lon": -74.0088829},{  "type": "node",  "id": 246647992,  "lat": 40.7407362,  "lon": -74.0093933},{  "type": "node",  "id": 246647993,  "lat": 40.7404878,  "lon": -74.0095204},{  "type": "node",  "id": 246648063,  "lat": 40.7430653,  "lon": -74.0087517},{  "type": "node",  "id": 246648107,  "lat": 40.7426120,  "lon": -74.0088346},{  "type": "node",  "id": 246648217,  "lat": 40.7423868,  "lon": -74.0081882},{  "type": "node",  "id": 246648218,  "lat": 40.7424335,  "lon": -74.0081011},{  "type": "node",  "id": 246648219,  "lat": 40.7425410,  "lon": -74.0079278},{  "type": "node",  "id": 246648824,  "lat": 40.7417860,  "lon": -74.0091460},{  "type": "node",  "id": 246652085,  "lat": 40.7468920,  "lon": -74.0080401},{  "type": "node",  "id": 246652086,  "lat": 40.7468695,  "lon": -74.0078489},{  "type": "node",  "id": 246652326,  "lat": 40.7452402,  "lon": -74.0081519},{  "type": "node",  "id": 246652365,  "lat": 40.7452662,  "lon": -74.0083460},{  "type": "node",  "id": 246652628,  "lat": 40.7445650,  "lon": -74.0086109},{  "type": "node",  "id": 246652629,  "lat": 40.7445689,  "lon": -74.0084694},{  "type": "node",  "id": 246652630,  "lat": 40.7445743,  "lon": -74.0082721},{  "type": "node",  "id": 246652649,  "lat": 40.7439169,  "lon": -74.0087318},{  "type": "node",  "id": 246652650,  "lat": 40.7438948,  "lon": -74.0085895},{  "type": "node",  "id": 246652651,  "lat": 40.7438428,  "lon": -74.0084006},{  "type": "node",  "id": 246652738,  "lat": 40.7430854,  "lon": -74.0089263},{  "type": "node",  "id": 246652740,  "lat": 40.7430426,  "lon": -74.0085544},{  "type": "node",  "id": 246652754,  "lat": 40.7422418,  "lon": -74.0090627},{  "type": "node",  "id": 246652755,  "lat": 40.7422243,  "lon": -74.0089327},{  "type": "node",  "id": 371247825,  "lat": 40.7420721,  "lon": -74.0077552},{  "type": "node",  "id": 373896216,  "lat": 40.7405783,  "lon": -74.0090186},{  "type": "node",  "id": 373896722,  "lat": 40.7405003,  "lon": -74.0092074},{  "type": "node",  "id": 373896723,  "lat": 40.7408189,  "lon": -74.0090701},{  "type": "node",  "id": 373897122,  "lat": 40.7410826,  "lon": -74.0091740},{  "type": "node",  "id": 373897123,  "lat": 40.7409100,  "lon": -74.0092215},{  "type": "node",  "id": 373897299,  "lat": 40.7412286,  "lon": -74.0089585},{  "type": "node",  "id": 373901771,  "lat": 40.7442053,  "lon": -74.0086736},{  "type": "node",  "id": 373903786,  "lat": 40.7454546,  "lon": -74.0083061},{  "type": "node",  "id": 387180916,  "lat": 40.7416415,  "lon": -74.0048213},{  "type": "node",  "id": 402475097,  "lat": 40.7423524,  "lon": -74.0086875},{  "type": "node",  "id": 402480678,  "lat": 40.7424626,  "lon": -74.0088706},{  "type": "node",  "id": 429568770,  "lat": 40.7460452,  "lon": -74.0060754},{  "type": "node",  "id": 593816329,  "lat": 40.7421951,  "lon": -74.0087163},{  "type": "node",  "id": 1578345404,  "lat": 40.7412818,  "lon": -74.0087979},{  "type": "node",  "id": 1701920922,  "lat": 40.7431247,  "lon": -74.0087145},{  "type": "node",  "id": 1701920951,  "lat": 40.7439738,  "lon": -74.0084363},{  "type": "node",  "id": 1701920961,  "lat": 40.7440579,  "lon": -74.0085109},{  "type": "node",  "id": 1701920982,  "lat": 40.7441031,  "lon": -74.0085235},{  "type": "node",  "id": 1701920994,  "lat": 40.7460201,  "lon": -74.0077745},{  "type": "node",  "id": 1701921000,  "lat": 40.7460005,  "lon": -74.0077088},{  "type": "node",  "id": 1701921001,  "lat": 40.7431694,  "lon": -74.0086427},{  "type": "node",  "id": 1701921009,  "lat": 40.7431761,  "lon": -74.0085923},{  "type": "node",  "id": 1701921016,  "lat": 40.7439989,  "lon": -74.0084777},{  "type": "node",  "id": 1701921023,  "lat": 40.7431541,  "lon": -74.0086835},{  "type": "node",  "id": 1701921024,  "lat": 40.7440246,  "lon": -74.0084968},{  "type": "node",  "id": 2034609413,  "lat": 40.7427942,  "lon": -74.0089583},{  "type": "node",  "id": 2208862779,  "lat": 40.7413818,  "lon": -74.0092319},{  "type": "node",  "id": 2208862783,  "lat": 40.7438414,  "lon": -74.0087478},{  "type": "node",  "id": 2208862797,  "lat": 40.7425663,  "lon": -74.0090008},{  "type": "node",  "id": 2376384435,  "lat": 40.7405637,  "lon": -74.0094710},{  "type": "node",  "id": 2376384437,  "lat": 40.7405756,  "lon": -74.0093513},{  "type": "node",  "id": 2376384443,  "lat": 40.7406446,  "lon": -74.0094259},{  "type": "node",  "id": 2376384447,  "lat": 40.7408444,  "lon": -74.0093723},{  "type": "node",  "id": 2376384451,  "lat": 40.7423267,  "lon": -74.0084082},{  "type": "node",  "id": 2376384453,  "lat": 40.7423674,  "lon": -74.0087288},{  "type": "node",  "id": 2376384455,  "lat": 40.7423971,  "lon": -74.0087918},{  "type": "node",  "id": 2376384457,  "lat": 40.7424261,  "lon": -74.0088354},{  "type": "node",  "id": 2376384459,  "lat": 40.7424744,  "lon": -74.0080222},{  "type": "node",  "id": 2376384461,  "lat": 40.7428398,  "lon": -74.0089585},{  "type": "node",  "id": 2376384463,  "lat": 40.7428835,  "lon": -74.0089588},{  "type": "node",  "id": 2376384465,  "lat": 40.7429331,  "lon": -74.0089568},{  "type": "node",  "id": 2376384467,  "lat": 40.7437274,  "lon": -74.0088058},{  "type": "node",  "id": 2376384469,  "lat": 40.7437582,  "lon": -74.0087950},{  "type": "node",  "id": 2376384471,  "lat": 40.7438116,  "lon": -74.0087571},{  "type": "node",  "id": 2671667947,  "lat": 40.7418974,  "lon": -74.0076148},{  "type": "node",  "id": 2671667948,  "lat": 40.7433226,  "lon": -74.0068872},{  "type": "node",  "id": 2671667952,  "lat": 40.7447926,  "lon": -74.0067347},{  "type": "node",  "id": 2671667954,  "lat": 40.7459824,  "lon": -74.0061081},{  "type": "node",  "id": 2671667955,  "lat": 40.7459887,  "lon": -74.0059426},{  "type": "node",  "id": 2823299575,  "lat": 40.7404792,  "lon": -74.0081420},{  "type": "node",  "id": 2823299576,  "lat": 40.7409905,  "lon": -74.0079865},{  "type": "node",  "id": 2823299578,  "lat": 40.7405423,  "lon": -74.0080095},{  "type": "node",  "id": 2823299579,  "lat": 40.7407983,  "lon": -74.0079565},{  "type": "node",  "id": 2823299580,  "lat": 40.7409185,  "lon": -74.0080871},{  "type": "node",  "id": 2823299581,  "lat": 40.7412371,  "lon": -74.0080171},{  "type": "node",  "id": 2823299582,  "lat": 40.7408811,  "lon": -74.0080464},{  "type": "node",  "id": 2823299583,  "lat": 40.7413576,  "lon": -74.0079048},{  "type": "node",  "id": 2823299584,  "lat": 40.7420625,  "lon": -74.0077106},{  "type": "node",  "id": 2823299585,  "lat": 40.7414912,  "lon": -74.0078781},{  "type": "node",  "id": 2823299586,  "lat": 40.7415507,  "lon": -74.0078104},{  "type": "node",  "id": 2823299587,  "lat": 40.7418576,  "lon": -74.0078322},{  "type": "node",  "id": 2823299588,  "lat": 40.7418069,  "lon": -74.0076842},{  "type": "node",  "id": 2823299589,  "lat": 40.7419384,  "lon": -74.0078140},{  "type": "node",  "id": 2823299590,  "lat": 40.7420480,  "lon": -74.0077875},{  "type": "node",  "id": 2823299591,  "lat": 40.7420845,  "lon": -74.0078126},{  "type": "node",  "id": 2823299592,  "lat": 40.7423491,  "lon": -74.0077534},{  "type": "node",  "id": 2823300817,  "lat": 40.7407960,  "lon": -74.0080667},{  "type": "node",  "id": 2823300820,  "lat": 40.7407577,  "lon": -74.0079637},{  "type": "node",  "id": 2823300821,  "lat": 40.7409528,  "lon": -74.0080104},{  "type": "node",  "id": 2823300822,  "lat": 40.7409967,  "lon": -74.0080710},{  "type": "node",  "id": 2823300823,  "lat": 40.7414500,  "lon": -74.0078863},{  "type": "node",  "id": 2823300832,  "lat": 40.7419138,  "lon": -74.0079300},{  "type": "node",  "id": 2823300833,  "lat": 40.7418372,  "lon": -74.0077528},{  "type": "node",  "id": 2823300834,  "lat": 40.7418498,  "lon": -74.0078821},{  "type": "node",  "id": 2823300835,  "lat": 40.7418992,  "lon": -74.0078761},{  "type": "node",  "id": 2823300836,  "lat": 40.7420775,  "lon": -74.0077802},{  "type": "node",  "id": 2823833421,  "lat": 40.7407910,  "lon": -74.0080674},{  "type": "node",  "id": 2823833422,  "lat": 40.7407242,  "lon": -74.0079715},{  "type": "node",  "id": 2823833423,  "lat": 40.7409419,  "lon": -74.0080159},{  "type": "node",  "id": 2823833424,  "lat": 40.7409856,  "lon": -74.0080733},{  "type": "node",  "id": 2823833425,  "lat": 40.7415167,  "lon": -74.0078491},{  "type": "node",  "id": 2823833426,  "lat": 40.7418037,  "lon": -74.0077595},{  "type": "node",  "id": 2823833428,  "lat": 40.7421502,  "lon": -74.0077262},{  "type": "node",  "id": 2823833429,  "lat": 40.7422244,  "lon": -74.0077076},{  "type": "node",  "id": 2823833430,  "lat": 40.7423017,  "lon": -74.0076921},{  "type": "node",  "id": 2823833431,  "lat": 40.7423704,  "lon": -74.0076726},{  "type": "node",  "id": 2823833432,  "lat": 40.7424235,  "lon": -74.0076499},{  "type": "node",  "id": 2823833433,  "lat": 40.7424758,  "lon": -74.0076179},{  "type": "node",  "id": 2823833434,  "lat": 40.7426070,  "lon": -74.0075180},{  "type": "node",  "id": 2823833435,  "lat": 40.7424047,  "lon": -74.0077272},{  "type": "node",  "id": 2823833436,  "lat": 40.7424711,  "lon": -74.0076901},{  "type": "node",  "id": 2823833437,  "lat": 40.7433724,  "lon": -74.0069617},{  "type": "node",  "id": 2823833443,  "lat": 40.7432546,  "lon": -74.0070436},{  "type": "node",  "id": 2823833444,  "lat": 40.7434083,  "lon": -74.0068900},{  "type": "node",  "id": 2823833445,  "lat": 40.7437417,  "lon": -74.0068913},{  "type": "node",  "id": 2823833446,  "lat": 40.7438843,  "lon": -74.0069672},{  "type": "node",  "id": 2823833447,  "lat": 40.7439448,  "lon": -74.0069730},{  "type": "node",  "id": 2823833449,  "lat": 40.7441511,  "lon": -74.0069793},{  "type": "node",  "id": 2823833450,  "lat": 40.7441839,  "lon": -74.0069509},{  "type": "node",  "id": 2823833451,  "lat": 40.7443840,  "lon": -74.0069699},{  "type": "node",  "id": 2823833452,  "lat": 40.7444840,  "lon": -74.0070064},{  "type": "node",  "id": 2823833453,  "lat": 40.7445414,  "lon": -74.0069715},{  "type": "node",  "id": 2823833454,  "lat": 40.7445876,  "lon": -74.0069953},{  "type": "node",  "id": 2823833455,  "lat": 40.7446771,  "lon": -74.0069663},{  "type": "node",  "id": 2823833456,  "lat": 40.7448558,  "lon": -74.0068371},{  "type": "node",  "id": 2823833457,  "lat": 40.7449293,  "lon": -74.0068384},{  "type": "node",  "id": 2823833458,  "lat": 40.7451393,  "lon": -74.0066862},{  "type": "node",  "id": 2823833459,  "lat": 40.7451781,  "lon": -74.0066260},{  "type": "node",  "id": 2823833460,  "lat": 40.7453907,  "lon": -74.0064653},{  "type": "node",  "id": 2823833461,  "lat": 40.7454770,  "lon": -74.0064246},{  "type": "node",  "id": 2823833462,  "lat": 40.7455958,  "lon": -74.0064187},{  "type": "node",  "id": 2823833463,  "lat": 40.7463024,  "lon": -74.0059068},{  "type": "node",  "id": 2823833464,  "lat": 40.7437606,  "lon": -74.0069013},{  "type": "node",  "id": 2823833467,  "lat": 40.7467191,  "lon": -74.0056018},{  "type": "node",  "id": 2823833468,  "lat": 40.7468175,  "lon": -74.0054787},{  "type": "node",  "id": 2823833470,  "lat": 40.7459719,  "lon": -74.0060845},{  "type": "node",  "id": 2823833471,  "lat": 40.7459923,  "lon": -74.0061315},{  "type": "node",  "id": 2823833473,  "lat": 40.7459472,  "lon": -74.0059901},{  "type": "node",  "id": 2823833474,  "lat": 40.7459684,  "lon": -74.0060447},{  "type": "node",  "id": 2823833475,  "lat": 40.7459588,  "lon": -74.0060521},{  "type": "node",  "id": 2823833476,  "lat": 40.7447519,  "lon": -74.0068821},{  "type": "node",  "id": 2823833477,  "lat": 40.7447693,  "lon": -74.0068669},{  "type": "node",  "id": 2823833479,  "lat": 40.7447478,  "lon": -74.0067820},{  "type": "node",  "id": 2823833481,  "lat": 40.7447546,  "lon": -74.0069103},{  "type": "node",  "id": 2823833482,  "lat": 40.7447701,  "lon": -74.0068406},{  "type": "node",  "id": 2823833483,  "lat": 40.7447614,  "lon": -74.0068468},{  "type": "node",  "id": 2823833489,  "lat": 40.7459369,  "lon": -74.0059976},{  "type": "node",  "id": 2823833490,  "lat": 40.7447384,  "lon": -74.0067874},{  "type": "node",  "id": 2823833496,  "lat": 40.7439818,  "lon": -74.0069657},{  "type": "node",  "id": 2823833518,  "lat": 40.7446582,  "lon": -74.0069729},{  "type": "node",  "id": 2823895884,  "lat": 40.7426799,  "lon": -74.0075397},{  "type": "node",  "id": 2823895885,  "lat": 40.7428627,  "lon": -74.0074114},{  "type": "node",  "id": 2823895886,  "lat": 40.7432605,  "lon": -74.0071379},{  "type": "node",  "id": 2823895887,  "lat": 40.7428156,  "lon": -74.0074444},{  "type": "node",  "id": 2823895888,  "lat": 40.7427870,  "lon": -74.0073862},{  "type": "node",  "id": 2823895889,  "lat": 40.7432893,  "lon": -74.0070195},{  "type": "node",  "id": 2845937086,  "lat": 40.7418532,  "lon": -74.0077500},{  "type": "node",  "id": 2845937087,  "lat": 40.7419044,  "lon": -74.0078938},{  "type": "node",  "id": 2985573626,  "lat": 40.7410838,  "lon": -74.0053798},{  "type": "node",  "id": 2985573628,  "lat": 40.7414956,  "lon": -74.0050584},{  "type": "node",  "id": 3236324362,  "lat": 40.7425849,  "lon": -74.0051318},{  "type": "node",  "id": 3236324368,  "lat": 40.7428292,  "lon": -74.0056997},{  "type": "node",  "id": 3564892565,  "lat": 40.7412230,  "lon": -74.0050583},{  "type": "node",  "id": 3564892566,  "lat": 40.7413187,  "lon": -74.0050382},{  "type": "node",  "id": 3564892567,  "lat": 40.7415276,  "lon": -74.0049008},{  "type": "node",  "id": 3564892568,  "lat": 40.7415813,  "lon": -74.0049889},{  "type": "node",  "id": 3564892571,  "lat": 40.7424314,  "lon": -74.0043288},{  "type": "node",  "id": 3564892572,  "lat": 40.7425589,  "lon": -74.0040771},{  "type": "node",  "id": 3564892573,  "lat": 40.7427126,  "lon": -74.0039564},{  "type": "node",  "id": 3874589908,  "lat": 40.7416247,  "lon": -74.0047815},{  "type": "node",  "id": 3874650225,  "lat": 40.7420863,  "lon": -74.0080669},{  "type": "node",  "id": 3874650227,  "lat": 40.7420635,  "lon": -74.0082682},{  "type": "node",  "id": 3874650232,  "lat": 40.7423150,  "lon": -74.0085986},{  "type": "node",  "id": 3874650236,  "lat": 40.7429040,  "lon": -74.0076536},{  "type": "node",  "id": 3874650238,  "lat": 40.7428916,  "lon": -74.0078479},{  "type": "node",  "id": 3874650241,  "lat": 40.7427587,  "lon": -74.0077623},{  "type": "node",  "id": 3874653357,  "lat": 40.7439276,  "lon": -74.0083057},{  "type": "node",  "id": 3874653359,  "lat": 40.7446613,  "lon": -74.0081705},{  "type": "node",  "id": 3874653362,  "lat": 40.7453350,  "lon": -74.0080269},{  "type": "node",  "id": 3874653365,  "lat": 40.7467605,  "lon": -74.0077763},{  "type": "node",  "id": 4133206253,  "lat": 40.7415787,  "lon": -74.0086844},{  "type": "node",  "id": 4133206254,  "lat": 40.7422986,  "lon": -74.0085428},{  "type": "node",  "id": 4133206257,  "lat": 40.7419649,  "lon": -74.0083536},{  "type": "node",  "id": 4207862999,  "lat": 40.7430491,  "lon": -74.0082389},{  "type": "node",  "id": 4207863000,  "lat": 40.7443704,  "lon": -74.0085024},{  "type": "node",  "id": 4379966358,  "lat": 40.7419998,  "lon": -74.0091058},{  "type": "node",  "id": 4379966360,  "lat": 40.7431323,  "lon": -74.0084454},{  "type": "node",  "id": 4379966362,  "lat": 40.7433241,  "lon": -74.0085058},{  "type": "node",  "id": 4379966363,  "lat": 40.7434298,  "lon": -74.0086801},{  "type": "node",  "id": 4379966364,  "lat": 40.7435182,  "lon": -74.0088448},{  "type": "node",  "id": 4555500073,  "lat": 40.7437702,  "lon": -74.0042856},{  "type": "node",  "id": 4555500078,  "lat": 40.7440714,  "lon": -74.0050076},{  "type": "node",  "id": 4555500086,  "lat": 40.7442058,  "lon": -74.0035154},{  "type": "node",  "id": 4555501893,  "lat": 40.7443288,  "lon": -74.0038087},{  "type": "node",  "id": 4555501898,  "lat": 40.7444218,  "lon": -74.0040304},{  "type": "node",  "id": 4555501902,  "lat": 40.7445592,  "lon": -74.0043580},{  "type": "node",  "id": 4555542092,  "lat": 40.7437363,  "lon": -74.0042042},{  "type": "node",  "id": 4555542098,  "lat": 40.7439569,  "lon": -74.0047332},{  "type": "node",  "id": 4555542099,  "lat": 40.7440004,  "lon": -74.0048374},{  "type": "node",  "id": 4555542101,  "lat": 40.7445384,  "lon": -74.0043084},{  "type": "node",  "id": 4555581029,  "lat": 40.7425029,  "lon": -74.0042769},{  "type": "node",  "id": 4555581031,  "lat": 40.7426700,  "lon": -74.0041555},{  "type": "node",  "id": 4555581039,  "lat": 40.7431274,  "lon": -74.0045827},{  "type": "node",  "id": 4555581040,  "lat": 40.7431561,  "lon": -74.0046512},{  "type": "node",  "id": 4555581043,  "lat": 40.7432519,  "lon": -74.0048798},{  "type": "node",  "id": 4555581045,  "lat": 40.7433415,  "lon": -74.0050937},{  "type": "node",  "id": 4557517550,  "lat": 40.7416809,  "lon": -74.0049194},{  "type": "node",  "id": 4557517551,  "lat": 40.7420532,  "lon": -74.0046519},{  "type": "node",  "id": 4557517552,  "lat": 40.7421252,  "lon": -74.0045542},{  "type": "node",  "id": 4557517553,  "lat": 40.7420267,  "lon": -74.0045863},{  "type": "node",  "id": 4557517554,  "lat": 40.7409922,  "lon": -74.0054494},{  "type": "node",  "id": 4592029853,  "lat": 40.7433486,  "lon": -74.0035005},{  "type": "node",  "id": 4592029854,  "lat": 40.7427619,  "lon": -74.0039255},{  "type": "node",  "id": 4599130018,  "lat": 40.7413845,  "lon": -74.0085357},{  "type": "node",  "id": 4601466495,  "lat": 40.7422452,  "lon": -74.0043294},{  "type": "node",  "id": 4601466496,  "lat": 40.7428306,  "lon": -74.0038757},{  "type": "node",  "id": 4634753633,  "lat": 40.7407485,  "lon": -74.0051553},{  "type": "node",  "id": 4634753634,  "lat": 40.7408128,  "lon": -74.0050370},{  "type": "node",  "id": 4634753635,  "lat": 40.7408639,  "lon": -74.0054062},{  "type": "node",  "id": 4634753636,  "lat": 40.7408830,  "lon": -74.0052002},{  "type": "node",  "id": 4634753637,  "lat": 40.7409637,  "lon": -74.0051058},{  "type": "node",  "id": 4685453631,  "lat": 40.7407280,  "lon": -74.0051046},{  "type": "node",  "id": 4685465959,  "lat": 40.7405180,  "lon": -74.0052346},{  "type": "node",  "id": 4783358909,  "lat": 40.7431589,  "lon": -74.0074665},{  "type": "node",  "id": 4783358910,  "lat": 40.7431034,  "lon": -74.0075072},{  "type": "node",  "id": 4872623855,  "lat": 40.7460345,  "lon": -74.0078863},{  "type": "node",  "id": 4872623863,  "lat": 40.7459892,  "lon": -74.0080171},{  "type": "node",  "id": 4872623868,  "lat": 40.7460215,  "lon": -74.0079697},{  "type": "node",  "id": 4872626641,  "lat": 40.7413999,  "lon": -74.0085723},{  "type": "node",  "id": 4872626650,  "lat": 40.7406876,  "lon": -74.0088565},{  "type": "node",  "id": 5103334974,  "lat": 40.7436413,  "lon": -74.0058093},{  "type": "node",  "id": 5244251221,  "lat": 40.7411486,  "lon": -74.0058234},{  "type": "node",  "id": 5545401595,  "lat": 40.7434201,  "lon": -74.0034486},{  "type": "node",  "id": 6401966213,  "lat": 40.7456418,  "lon": -74.0056615},{  "type": "node",  "id": 42422258,  "lat": 40.7482137,  "lon": -74.0074126},{  "type": "node",  "id": 42427483,  "lat": 40.7483550,  "lon": -74.0036840},{  "type": "node",  "id": 42427494,  "lat": 40.7502561,  "lon": -74.0082754},{  "type": "node",  "id": 42430589,  "lat": 40.7488769,  "lon": -74.0069320},{  "type": "node",  "id": 42434285,  "lat": 40.7470230,  "lon": -74.0046530},{  "type": "node",  "id": 42437914,  "lat": 40.7476900,  "lon": -74.0041680},{  "type": "node",  "id": 42437915,  "lat": 40.7494285,  "lon": -74.0080316},{  "type": "node",  "id": 42443677,  "lat": 40.7475038,  "lon": -74.0077312},{  "type": "node",  "id": 246579582,  "lat": 40.7495976,  "lon": -74.0081174},{  "type": "node",  "id": 246579597,  "lat": 40.7477566,  "lon": -74.0076956},{  "type": "node",  "id": 246579608,  "lat": 40.7484596,  "lon": -74.0076282},{  "type": "node",  "id": 246579610,  "lat": 40.7489201,  "lon": -74.0077701},{  "type": "node",  "id": 246579614,  "lat": 40.7498056,  "lon": -74.0082032},{  "type": "node",  "id": 246579746,  "lat": 40.7501763,  "lon": -74.0084779},{  "type": "node",  "id": 246579748,  "lat": 40.7495271,  "lon": -74.0083007},{  "type": "node",  "id": 246579749,  "lat": 40.7491879,  "lon": -74.0081002},{  "type": "node",  "id": 246579750,  "lat": 40.7489213,  "lon": -74.0079543},{  "type": "node",  "id": 246579751,  "lat": 40.7485897,  "lon": -74.0078427},{  "type": "node",  "id": 246579752,  "lat": 40.7482202,  "lon": -74.0078160},{  "type": "node",  "id": 246579753,  "lat": 40.7480596,  "lon": -74.0078318},{  "type": "node",  "id": 246580301,  "lat": 40.7487054,  "lon": -74.0076805},{  "type": "node",  "id": 246580443,  "lat": 40.7493971,  "lon": -74.0065543},{  "type": "node",  "id": 246580445,  "lat": 40.7494817,  "lon": -74.0063483},{  "type": "node",  "id": 246580465,  "lat": 40.7495012,  "lon": -74.0065372},{  "type": "node",  "id": 246580466,  "lat": 40.7495415,  "lon": -74.0065691},{  "type": "node",  "id": 246580736,  "lat": 40.7481843,  "lon": -74.0076496},{  "type": "node",  "id": 246580758,  "lat": 40.7489204,  "lon": -74.0070372},{  "type": "node",  "id": 246580797,  "lat": 40.7503864,  "lon": -74.0084826},{  "type": "node",  "id": 246650838,  "lat": 40.7492790,  "lon": -74.0079456},{  "type": "node",  "id": 246650840,  "lat": 40.7493779,  "lon": -74.0082324},{  "type": "node",  "id": 246651360,  "lat": 40.7500228,  "lon": -74.0084704},{  "type": "node",  "id": 246651361,  "lat": 40.7500370,  "lon": -74.0082557},{  "type": "node",  "id": 246651647,  "lat": 40.7484857,  "lon": -74.0078256},{  "type": "node",  "id": 246651648,  "lat": 40.7482882,  "lon": -74.0075083},{  "type": "node",  "id": 373910553,  "lat": 40.7498237,  "lon": -74.0084243},{  "type": "node",  "id": 373910795,  "lat": 40.7499227,  "lon": -74.0082376},{  "type": "node",  "id": 1701844592,  "lat": 40.7482421,  "lon": -74.0075636},{  "type": "node",  "id": 1701844602,  "lat": 40.7479155,  "lon": -74.0076331},{  "type": "node",  "id": 1701844672,  "lat": 40.7480415,  "lon": -74.0075565},{  "type": "node",  "id": 1701844780,  "lat": 40.7481058,  "lon": -74.0075054},{  "type": "node",  "id": 1701844809,  "lat": 40.7479943,  "lon": -74.0075911},{  "type": "node",  "id": 1701920933,  "lat": 40.7495329,  "lon": -74.0065581},{  "type": "node",  "id": 1701920939,  "lat": 40.7487514,  "lon": -74.0071395},{  "type": "node",  "id": 1701920946,  "lat": 40.7484197,  "lon": -74.0073753},{  "type": "node",  "id": 1701920957,  "lat": 40.7488510,  "lon": -74.0070872},{  "type": "node",  "id": 1701920999,  "lat": 40.7486658,  "lon": -74.0071902},{  "type": "node",  "id": 1701921027,  "lat": 40.7494645,  "lon": -74.0065300},{  "type": "node",  "id": 2376384580,  "lat": 40.7496792,  "lon": -74.0083735},{  "type": "node",  "id": 2823833469,  "lat": 40.7470778,  "lon": -74.0052889},{  "type": "node",  "id": 2823833519,  "lat": 40.7473090,  "lon": -74.0051241},{  "type": "node",  "id": 2823833520,  "lat": 40.7475476,  "lon": -74.0050390},{  "type": "node",  "id": 2823833523,  "lat": 40.7471776,  "lon": -74.0052669},{  "type": "node",  "id": 2823833529,  "lat": 40.7474221,  "lon": -74.0051274},{  "type": "node",  "id": 2823833530,  "lat": 40.7473025,  "lon": -74.0052106},{  "type": "node",  "id": 2823833532,  "lat": 40.7472757,  "lon": -74.0051849},{  "type": "node",  "id": 2823833535,  "lat": 40.7473935,  "lon": -74.0050626},{  "type": "node",  "id": 2823833536,  "lat": 40.7484060,  "lon": -74.0043203},{  "type": "node",  "id": 2823833537,  "lat": 40.7480085,  "lon": -74.0046655},{  "type": "node",  "id": 2823833538,  "lat": 40.7479284,  "lon": -74.0047240},{  "type": "node",  "id": 2823833539,  "lat": 40.7480567,  "lon": -74.0046324},{  "type": "node",  "id": 2823833540,  "lat": 40.7481046,  "lon": -74.0045986},{  "type": "node",  "id": 2823833541,  "lat": 40.7481596,  "lon": -74.0045594},{  "type": "node",  "id": 2823833542,  "lat": 40.7481745,  "lon": -74.0045485},{  "type": "node",  "id": 2823833543,  "lat": 40.7481564,  "lon": -74.0045028},{  "type": "node",  "id": 2823833544,  "lat": 40.7485309,  "lon": -74.0042912},{  "type": "node",  "id": 2823833545,  "lat": 40.7486733,  "lon": -74.0041880},{  "type": "node",  "id": 2823833546,  "lat": 40.7487572,  "lon": -74.0040667},{  "type": "node",  "id": 2823833547,  "lat": 40.7491360,  "lon": -74.0037909},{  "type": "node",  "id": 2823833548,  "lat": 40.7493003,  "lon": -74.0036699},{  "type": "node",  "id": 2823833549,  "lat": 40.7495032,  "lon": -74.0036020},{  "type": "node",  "id": 2823833602,  "lat": 40.7497098,  "lon": -74.0034532},{  "type": "node",  "id": 2840192832,  "lat": 40.7479572,  "lon": -74.0046505},{  "type": "node",  "id": 2840192833,  "lat": 40.7480224,  "lon": -74.0046206},{  "type": "node",  "id": 2840192834,  "lat": 40.7481280,  "lon": -74.0045429},{  "type": "node",  "id": 2845975411,  "lat": 40.7480248,  "lon": -74.0046543},{  "type": "node",  "id": 3146796134,  "lat": 40.7495624,  "lon": -74.0035594},{  "type": "node",  "id": 3146796136,  "lat": 40.7494330,  "lon": -74.0036255},{  "type": "node",  "id": 3146796138,  "lat": 40.7493116,  "lon": -74.0036661},{  "type": "node",  "id": 3563148493,  "lat": 40.7498326,  "lon": -74.0033648},{  "type": "node",  "id": 3563148495,  "lat": 40.7498204,  "lon": -74.0033736},{  "type": "node",  "id": 3914863394,  "lat": 40.7487664,  "lon": -74.0070121},{  "type": "node",  "id": 3914863396,  "lat": 40.7489606,  "lon": -74.0068712},{  "type": "node",  "id": 3914863399,  "lat": 40.7490000,  "lon": -74.0069740},{  "type": "node",  "id": 3914863402,  "lat": 40.7494831,  "lon": -74.0066127},{  "type": "node",  "id": 3914863404,  "lat": 40.7488082,  "lon": -74.0071106},{  "type": "node",  "id": 3914863405,  "lat": 40.7488448,  "lon": -74.0068573},{  "type": "node",  "id": 4207978494,  "lat": 40.7493471,  "lon": -74.0067145},{  "type": "node",  "id": 4207978495,  "lat": 40.7492584,  "lon": -74.0066550},{  "type": "node",  "id": 4872623861,  "lat": 40.7481852,  "lon": -74.0073464},{  "type": "node",  "id": 4872623865,  "lat": 40.7481481,  "lon": -74.0074690},{  "type": "node",  "id": 4872623873,  "lat": 40.7474696,  "lon": -74.0076504},{  "type": "node",  "id": 42429700,  "lat": 40.7317982,  "lon": -74.0011515},{  "type": "node",  "id": 42429701,  "lat": 40.7311376,  "lon": -74.0029381},{  "type": "node",  "id": 42430167,  "lat": 40.7276640,  "lon": -74.0031550},{  "type": "node",  "id": 42430187,  "lat": 40.7297532,  "lon": -74.0021622},{  "type": "node",  "id": 42430205,  "lat": 40.7305810,  "lon": -74.0016840},{  "type": "node",  "id": 42430217,  "lat": 40.7310650,  "lon": -74.0014130},{  "type": "node",  "id": 42430231,  "lat": 40.7317060,  "lon": -74.0009540},{  "type": "node",  "id": 42430233,  "lat": 40.7323150,  "lon": -74.0005110},{  "type": "node",  "id": 42430235,  "lat": 40.7329290,  "lon": -74.0000660},{  "type": "node",  "id": 42430237,  "lat": 40.7336634,  "lon": -73.9995234},{  "type": "node",  "id": 42430468,  "lat": 40.7283942,  "lon": -74.0030858},{  "type": "node",  "id": 42430470,  "lat": 40.7283268,  "lon": -74.0030875},{  "type": "node",  "id": 42430803,  "lat": 40.7332265,  "lon": -74.0030292},{  "type": "node",  "id": 42430805,  "lat": 40.7336094,  "lon": -74.0028114},{  "type": "node",  "id": 42431773,  "lat": 40.7303374,  "lon": -74.0024762},{  "type": "node",  "id": 42431777,  "lat": 40.7309159,  "lon": -74.0028101},{  "type": "node",  "id": 42431785,  "lat": 40.7314683,  "lon": -74.0031289},{  "type": "node",  "id": 42433260,  "lat": 40.7334390,  "lon": -74.0011030},{  "type": "node",  "id": 42433261,  "lat": 40.7338430,  "lon": -74.0012050},{  "type": "node",  "id": 42433281,  "lat": 40.7322150,  "lon": -73.9986120},{  "type": "node",  "id": 42433282,  "lat": 40.7332427,  "lon": -74.0007095},{  "type": "node",  "id": 42433286,  "lat": 40.7336966,  "lon": -74.0016383},{  "type": "node",  "id": 42433821,  "lat": 40.7329627,  "lon": -74.0017902},{  "type": "node",  "id": 42433822,  "lat": 40.7333795,  "lon": -74.0025764},{  "type": "node",  "id": 42436793,  "lat": 40.7296860,  "lon": -73.9986042},{  "type": "node",  "id": 42436796,  "lat": 40.7302877,  "lon": -73.9980696},{  "type": "node",  "id": 42436844,  "lat": 40.7298963,  "lon": -74.0013313},{  "type": "node",  "id": 42436848,  "lat": 40.7302460,  "lon": -74.0010200},{  "type": "node",  "id": 42436963,  "lat": 40.7297360,  "lon": -74.0024100},{  "type": "node",  "id": 42437951,  "lat": 40.7329460,  "lon": -73.9979870},{  "type": "node",  "id": 42439533,  "lat": 40.7277329,  "lon": -74.0013516},{  "type": "node",  "id": 42439537,  "lat": 40.7300598,  "lon": -73.9993655},{  "type": "node",  "id": 42439542,  "lat": 40.7306690,  "lon": -73.9988420},{  "type": "node",  "id": 42440553,  "lat": 40.7293265,  "lon": -73.9978721},{  "type": "node",  "id": 42446902,  "lat": 40.7323180,  "lon": -74.0016630},{  "type": "node",  "id": 42448549,  "lat": 40.7281021,  "lon": -74.0021007},{  "type": "node",  "id": 42450325,  "lat": 40.7276010,  "lon": -74.0025130},{  "type": "node",  "id": 42450333,  "lat": 40.7326101,  "lon": -73.9982742},{  "type": "node",  "id": 42451987,  "lat": 40.7328410,  "lon": -74.0021240},{  "type": "node",  "id": 42452173,  "lat": 40.7337653,  "lon": -74.0003266},{  "type": "node",  "id": 42452396,  "lat": 40.7299438,  "lon": -73.9973576},{  "type": "node",  "id": 42452964,  "lat": 40.7287340,  "lon": -73.9984000},{  "type": "node",  "id": 529556095,  "lat": 40.7296913,  "lon": -74.0020271},{  "type": "node",  "id": 529556119,  "lat": 40.7281073,  "lon": -73.9989223},{  "type": "node",  "id": 1692433924,  "lat": 40.7276837,  "lon": -73.9981746},{  "type": "node",  "id": 1692433927,  "lat": 40.7280098,  "lon": -73.9990044},{  "type": "node",  "id": 1692433928,  "lat": 40.7280695,  "lon": -73.9989542},{  "type": "node",  "id": 1692433932,  "lat": 40.7284251,  "lon": -73.9996723},{  "type": "node",  "id": 1692433935,  "lat": 40.7288005,  "lon": -74.0004308},{  "type": "node",  "id": 1692433936,  "lat": 40.7289446,  "lon": -73.9970942},{  "type": "node",  "id": 1692433937,  "lat": 40.7289751,  "lon": -73.9971564},{  "type": "node",  "id": 1692433938,  "lat": 40.7291711,  "lon": -74.0011907},{  "type": "node",  "id": 1692433939,  "lat": 40.7299867,  "lon": -74.0004975},{  "type": "node",  "id": 1692433940,  "lat": 40.7304314,  "lon": -74.0001225},{  "type": "node",  "id": 1692433941,  "lat": 40.7310432,  "lon": -73.9996039},{  "type": "node",  "id": 1692433942,  "lat": 40.7316274,  "lon": -73.9991063},{  "type": "node",  "id": 1701920971,  "lat": 40.7337951,  "lon": -74.0003264},{  "type": "node",  "id": 1701920981,  "lat": 40.7337333,  "lon": -74.0003380},{  "type": "node",  "id": 1760570971,  "lat": 40.7335491,  "lon": -73.9994511},{  "type": "node",  "id": 1760570973,  "lat": 40.7335740,  "lon": -73.9994820},{  "type": "node",  "id": 1760570975,  "lat": 40.7335985,  "lon": -73.9994986},{  "type": "node",  "id": 1760570977,  "lat": 40.7335394,  "lon": -73.9994285},{  "type": "node",  "id": 1760570981,  "lat": 40.7336228,  "lon": -73.9995092},{  "type": "node",  "id": 1919595927,  "lat": 40.7276175,  "lon": -74.0014439},{  "type": "node",  "id": 1919595928,  "lat": 40.7279861,  "lon": -74.0021905},{  "type": "node",  "id": 2477560563,  "lat": 40.7295131,  "lon": -74.0018856},{  "type": "node",  "id": 2477560572,  "lat": 40.7289129,  "lon": -74.0025832},{  "type": "node",  "id": 2550012966,  "lat": 40.7288832,  "lon": -74.0024384},{  "type": "node",  "id": 2580430244,  "lat": 40.7318551,  "lon": -74.0008455},{  "type": "node",  "id": 3573482131,  "lat": 40.7283536,  "lon": -74.0027215},{  "type": "node",  "id": 3584752204,  "lat": 40.7298918,  "lon": -73.9974010},{  "type": "node",  "id": 3584752205,  "lat": 40.7298929,  "lon": -73.9972523},{  "type": "node",  "id": 3584752208,  "lat": 40.7299982,  "lon": -73.9974703},{  "type": "node",  "id": 3584752228,  "lat": 40.7302398,  "lon": -73.9981121},{  "type": "node",  "id": 3584752231,  "lat": 40.7302563,  "lon": -73.9980046},{  "type": "node",  "id": 3584752242,  "lat": 40.7303233,  "lon": -73.9981433},{  "type": "node",  "id": 3584752262,  "lat": 40.7306184,  "lon": -73.9988855},{  "type": "node",  "id": 3584752265,  "lat": 40.7306425,  "lon": -73.9987885},{  "type": "node",  "id": 3584752281,  "lat": 40.7307066,  "lon": -73.9989164},{  "type": "node",  "id": 3584752496,  "lat": 40.7309915,  "lon": -73.9996477},{  "type": "node",  "id": 3584752497,  "lat": 40.7310158,  "lon": -73.9995479},{  "type": "node",  "id": 3584752506,  "lat": 40.7313700,  "lon": -73.9968855},{  "type": "node",  "id": 3584752508,  "lat": 40.7314143,  "lon": -73.9969760},{  "type": "node",  "id": 3584752515,  "lat": 40.7314650,  "lon": -73.9970715},{  "type": "node",  "id": 3584752516,  "lat": 40.7314685,  "lon": -73.9969322},{  "type": "node",  "id": 3584752520,  "lat": 40.7321658,  "lon": -73.9986534},{  "type": "node",  "id": 3584752522,  "lat": 40.7321895,  "lon": -73.9985601},{  "type": "node",  "id": 3584752526,  "lat": 40.7322730,  "lon": -73.9985624},{  "type": "node",  "id": 3891950553,  "lat": 40.7295972,  "lon": -74.0020563},{  "type": "node",  "id": 3891950555,  "lat": 40.7297288,  "lon": -74.0023169},{  "type": "node",  "id": 3891950890,  "lat": 40.7288886,  "lon": -74.0024715},{  "type": "node",  "id": 3891950891,  "lat": 40.7294642,  "lon": -74.0019312},{  "type": "node",  "id": 3891960758,  "lat": 40.7298798,  "lon": -74.0020941},{  "type": "node",  "id": 3891960759,  "lat": 40.7296032,  "lon": -74.0022430},{  "type": "node",  "id": 3891960760,  "lat": 40.7298973,  "lon": -74.0022288},{  "type": "node",  "id": 3891963908,  "lat": 40.7284047,  "lon": -74.0031587},{  "type": "node",  "id": 3891963909,  "lat": 40.7285331,  "lon": -74.0027704},{  "type": "node",  "id": 3891963913,  "lat": 40.7282330,  "lon": -74.0027659},{  "type": "node",  "id": 3891963914,  "lat": 40.7285127,  "lon": -74.0028916},{  "type": "node",  "id": 3891963915,  "lat": 40.7280358,  "lon": -74.0030050},{  "type": "node",  "id": 3891969467,  "lat": 40.7303659,  "lon": -74.0023835},{  "type": "node",  "id": 3891969468,  "lat": 40.7304611,  "lon": -74.0017514},{  "type": "node",  "id": 3891969470,  "lat": 40.7302656,  "lon": -74.0024359},{  "type": "node",  "id": 3891969471,  "lat": 40.7305230,  "lon": -74.0018726},{  "type": "node",  "id": 4016646249,  "lat": 40.7309192,  "lon": -74.0011159},{  "type": "node",  "id": 4016646252,  "lat": 40.7313397,  "lon": -74.0012163},{  "type": "node",  "id": 4202908174,  "lat": 40.7282488,  "lon": -74.0023984},{  "type": "node",  "id": 4302903239,  "lat": 40.7311040,  "lon": -73.9995521},{  "type": "node",  "id": 4302910548,  "lat": 40.7316610,  "lon": -73.9991749},{  "type": "node",  "id": 4302910552,  "lat": 40.7315668,  "lon": -73.9991581},{  "type": "node",  "id": 4302910555,  "lat": 40.7310757,  "lon": -73.9996702},{  "type": "node",  "id": 4302910557,  "lat": 40.7316880,  "lon": -73.9990553},{  "type": "node",  "id": 4320028807,  "lat": 40.7332834,  "lon": -74.0029968},{  "type": "node",  "id": 4320028809,  "lat": 40.7334148,  "lon": -74.0026142},{  "type": "node",  "id": 4320028815,  "lat": 40.7338195,  "lon": -74.0026919},{  "type": "node",  "id": 4320028816,  "lat": 40.7332919,  "lon": -74.0028356},{  "type": "node",  "id": 4320028818,  "lat": 40.7335060,  "lon": -74.0028702},{  "type": "node",  "id": 4320028821,  "lat": 40.7339063,  "lon": -74.0012232},{  "type": "node",  "id": 4320028822,  "lat": 40.7333520,  "lon": -74.0026579},{  "type": "node",  "id": 4320028823,  "lat": 40.7338049,  "lon": -74.0011954},{  "type": "node",  "id": 4320028826,  "lat": 40.7337334,  "lon": -74.0027408},{  "type": "node",  "id": 4320028827,  "lat": 40.7333343,  "lon": -74.0025065},{  "type": "node",  "id": 4320028830,  "lat": 40.7331641,  "lon": -74.0032143},{  "type": "node",  "id": 4320028833,  "lat": 40.7331500,  "lon": -74.0030727},{  "type": "node",  "id": 4320028836,  "lat": 40.7335868,  "lon": -74.0029633},{  "type": "node",  "id": 4320028838,  "lat": 40.7334099,  "lon": -74.0024864},{  "type": "node",  "id": 4320028840,  "lat": 40.7338203,  "lon": -74.0028301},{  "type": "node",  "id": 4320028842,  "lat": 40.7335188,  "lon": -74.0027002},{  "type": "node",  "id": 4320028844,  "lat": 40.7336713,  "lon": -74.0015848},{  "type": "node",  "id": 4320028845,  "lat": 40.7333258,  "lon": -74.0025340},{  "type": "node",  "id": 4320032274,  "lat": 40.7329457,  "lon": -74.0017612},{  "type": "node",  "id": 4320032276,  "lat": 40.7328203,  "lon": -74.0021831},{  "type": "node",  "id": 4320032277,  "lat": 40.7329278,  "lon": -74.0018684},{  "type": "node",  "id": 4320032286,  "lat": 40.7328070,  "lon": -74.0020930},{  "type": "node",  "id": 4320105472,  "lat": 40.7318515,  "lon": -74.0012321},{  "type": "node",  "id": 4320105473,  "lat": 40.7317515,  "lon": -74.0012192},{  "type": "node",  "id": 4320108382,  "lat": 40.7315541,  "lon": -74.0006446},{  "type": "node",  "id": 4320108384,  "lat": 40.7305210,  "lon": -74.0015651},{  "type": "node",  "id": 4320108387,  "lat": 40.7316490,  "lon": -74.0008377},{  "type": "node",  "id": 4320108989,  "lat": 40.7310090,  "lon": -74.0012990},{  "type": "node",  "id": 4320194257,  "lat": 40.7337619,  "lon": -73.9994532},{  "type": "node",  "id": 4320194261,  "lat": 40.7338556,  "lon": -74.0011225},{  "type": "node",  "id": 4320194276,  "lat": 40.7338271,  "lon": -73.9995876},{  "type": "node",  "id": 4320194279,  "lat": 40.7335729,  "lon": -73.9995902},{  "type": "node",  "id": 4321677346,  "lat": 40.7278344,  "lon": -74.0030836},{  "type": "node",  "id": 4321677347,  "lat": 40.7280062,  "lon": -74.0031190},{  "type": "node",  "id": 4321748234,  "lat": 40.7281016,  "lon": -74.0028952},{  "type": "node",  "id": 4321748235,  "lat": 40.7284155,  "lon": -74.0026719},{  "type": "node",  "id": 4321748236,  "lat": 40.7281737,  "lon": -74.0025704},{  "type": "node",  "id": 4321748237,  "lat": 40.7284697,  "lon": -74.0028017},{  "type": "node",  "id": 4321748238,  "lat": 40.7283741,  "lon": -74.0028488},{  "type": "node",  "id": 4375703337,  "lat": 40.7316428,  "lon": -74.0009986},{  "type": "node",  "id": 4375703339,  "lat": 40.7317509,  "lon": -74.0012210},{  "type": "node",  "id": 4375703341,  "lat": 40.7317628,  "lon": -74.0010787},{  "type": "node",  "id": 4375703342,  "lat": 40.7317685,  "lon": -74.0009086},{  "type": "node",  "id": 4387363208,  "lat": 40.7277828,  "lon": -74.0023522},{  "type": "node",  "id": 4679523598,  "lat": 40.7336361,  "lon": -74.0026246},{  "type": "node",  "id": 4679523600,  "lat": 40.7282179,  "lon": -74.0026820},{  "type": "node",  "id": 4679523601,  "lat": 40.7281828,  "lon": -74.0028462},{  "type": "node",  "id": 5151708181,  "lat": 40.7276061,  "lon": -74.0031788},{  "type": "node",  "id": 5151708183,  "lat": 40.7276497,  "lon": -74.0030089},{  "type": "node",  "id": 5151708184,  "lat": 40.7277347,  "lon": -74.0031249},{  "type": "node",  "id": 5151711951,  "lat": 40.7310116,  "lon": -74.0014429},{  "type": "node",  "id": 5151711953,  "lat": 40.7311244,  "lon": -74.0013705},{  "type": "node",  "id": 5151719389,  "lat": 40.7305903,  "lon": -74.0016787},{  "type": "node",  "id": 5155816417,  "lat": 40.7331688,  "lon": -73.9998888},{  "type": "node",  "id": 5155816418,  "lat": 40.7308272,  "lon": -74.0015461},{  "type": "node",  "id": 5155816419,  "lat": 40.7286951,  "lon": -74.0026905},{  "type": "node",  "id": 5244228879,  "lat": 40.7338007,  "lon": -74.0014950},{  "type": "node",  "id": 5481866330,  "lat": 40.7278041,  "lon": -74.0018218},{  "type": "node",  "id": 5690447869,  "lat": 40.7322466,  "lon": -73.9986763},{  "type": "node",  "id": 5690465658,  "lat": 40.7328610,  "lon": -74.0020650},{  "type": "node",  "id": 5796876484,  "lat": 40.7283848,  "lon": -74.0029568},{  "type": "node",  "id": 5799117241,  "lat": 40.7283335,  "lon": -74.0025124},{  "type": "node",  "id": 5799117243,  "lat": 40.7282206,  "lon": -74.0028105},{  "type": "node",  "id": 5829827193,  "lat": 40.7274265,  "lon": -74.0005180},{  "type": "node",  "id": 5829827215,  "lat": 40.7278066,  "lon": -74.0012880},{  "type": "node",  "id": 5829827216,  "lat": 40.7281811,  "lon": -74.0020335},{  "type": "node",  "id": 5829844668,  "lat": 40.7275476,  "lon": -74.0015004},{  "type": "node",  "id": 5829844674,  "lat": 40.7279214,  "lon": -74.0022420},{  "type": "node",  "id": 5829880520,  "lat": 40.7280203,  "lon": -74.0022597},{  "type": "node",  "id": 5829880521,  "lat": 40.7275804,  "lon": -74.0013688},{  "type": "node",  "id": 5829880522,  "lat": 40.7281362,  "lon": -74.0021699},{  "type": "node",  "id": 5829880523,  "lat": 40.7276962,  "lon": -74.0012772},{  "type": "node",  "id": 5829880524,  "lat": 40.7280663,  "lon": -74.0020281},{  "type": "node",  "id": 5829880525,  "lat": 40.7273864,  "lon": -74.0006483},{  "type": "node",  "id": 5829880526,  "lat": 40.7276461,  "lon": -74.0015018},{  "type": "node",  "id": 5829880527,  "lat": 40.7277609,  "lon": -74.0014083},{  "type": "node",  "id": 5829880529,  "lat": 40.7279536,  "lon": -74.0021247},{  "type": "node",  "id": 5854302552,  "lat": 40.7298561,  "lon": -74.0022057},{  "type": "node",  "id": 5854302553,  "lat": 40.7293347,  "lon": -74.0015231},{  "type": "node",  "id": 5854302554,  "lat": 40.7296922,  "lon": -74.0020718},{  "type": "node",  "id": 5854302555,  "lat": 40.7297332,  "lon": -74.0022009},{  "type": "node",  "id": 5854302556,  "lat": 40.7297007,  "lon": -74.0021128},{  "type": "node",  "id": 5854302557,  "lat": 40.7297242,  "lon": -74.0022615},{  "type": "node",  "id": 5854302558,  "lat": 40.7296215,  "lon": -74.0021058},{  "type": "node",  "id": 5854302559,  "lat": 40.7297236,  "lon": -74.0021418},{  "type": "node",  "id": 5854302560,  "lat": 40.7297393,  "lon": -74.0023661},{  "type": "node",  "id": 5866337888,  "lat": 40.7282993,  "lon": -74.0028830},{  "type": "node",  "id": 6213204038,  "lat": 40.7280896,  "lon": -73.9989372},{  "type": "node",  "id": 6213204043,  "lat": 40.7291933,  "lon": -74.0011718},{  "type": "node",  "id": 6213204047,  "lat": 40.7303491,  "lon": -74.0024382},{  "type": "node",  "id": 6213204056,  "lat": 40.7288194,  "lon": -74.0004148},{  "type": "node",  "id": 6213204062,  "lat": 40.7284422,  "lon": -73.9996578},{  "type": "node",  "id": 6213204069,  "lat": 40.7311489,  "lon": -74.0029063},{  "type": "node",  "id": 42421877,  "lat": 40.7328593,  "lon": -73.9959038},{  "type": "node",  "id": 42428458,  "lat": 40.7278266,  "lon": -73.9947496},{  "type": "node",  "id": 42428460,  "lat": 40.7284120,  "lon": -73.9942520},{  "type": "node",  "id": 42428464,  "lat": 40.7290080,  "lon": -73.9937470},{  "type": "node",  "id": 42428468,  "lat": 40.7295800,  "lon": -73.9932620},{  "type": "node",  "id": 42428471,  "lat": 40.7300790,  "lon": -73.9928410},{  "type": "node",  "id": 42428473,  "lat": 40.7306011,  "lon": -73.9923997},{  "type": "node",  "id": 42428476,  "lat": 40.7311660,  "lon": -73.9919210},{  "type": "node",  "id": 42428480,  "lat": 40.7317445,  "lon": -73.9914619},{  "type": "node",  "id": 42428483,  "lat": 40.7324490,  "lon": -73.9912890},{  "type": "node",  "id": 42428489,  "lat": 40.7332140,  "lon": -73.9911140},{  "type": "node",  "id": 42429676,  "lat": 40.7275728,  "lon": -73.9941859},{  "type": "node",  "id": 42430857,  "lat": 40.7334370,  "lon": -73.9954890},{  "type": "node",  "id": 42430861,  "lat": 40.7327310,  "lon": -73.9938140},{  "type": "node",  "id": 42430872,  "lat": 40.7312807,  "lon": -73.9903509},{  "type": "node",  "id": 42433292,  "lat": 40.7299780,  "lon": -73.9940640},{  "type": "node",  "id": 42433295,  "lat": 40.7303500,  "lon": -73.9948220},{  "type": "node",  "id": 42433298,  "lat": 40.7307232,  "lon": -73.9955715},{  "type": "node",  "id": 42433354,  "lat": 40.7318560,  "lon": -73.9966390},{  "type": "node",  "id": 42433356,  "lat": 40.7312090,  "lon": -73.9951190},{  "type": "node",  "id": 42433587,  "lat": 40.7295629,  "lon": -73.9965691},{  "type": "node",  "id": 42434194,  "lat": 40.7292206,  "lon": -73.9909718},{  "type": "node",  "id": 42434196,  "lat": 40.7289400,  "lon": -73.9903444},{  "type": "node",  "id": 42437949,  "lat": 40.7322550,  "lon": -73.9963510},{  "type": "node",  "id": 42439236,  "lat": 40.7298614,  "lon": -73.9914198},{  "type": "node",  "id": 42439243,  "lat": 40.7297950,  "lon": -73.9907804},{  "type": "node",  "id": 42439249,  "lat": 40.7299009,  "lon": -73.9907440},{  "type": "node",  "id": 42440545,  "lat": 40.7282036,  "lon": -73.9955850},{  "type": "node",  "id": 42440548,  "lat": 40.7285692,  "lon": -73.9963296},{  "type": "node",  "id": 42443799,  "lat": 40.7281239,  "lon": -73.9902269},{  "type": "node",  "id": 42445534,  "lat": 40.7278670,  "lon": -73.9931470},{  "type": "node",  "id": 42445543,  "lat": 40.7300532,  "lon": -73.9910884},{  "type": "node",  "id": 42445548,  "lat": 40.7306150,  "lon": -73.9906460},{  "type": "node",  "id": 42445558,  "lat": 40.7305570,  "lon": -73.9905029},{  "type": "node",  "id": 42449045,  "lat": 40.7321650,  "lon": -73.9942970},{  "type": "node",  "id": 42449580,  "lat": 40.7288100,  "lon": -73.9950610},{  "type": "node",  "id": 42449582,  "lat": 40.7294060,  "lon": -73.9945470},{  "type": "node",  "id": 42449589,  "lat": 40.7309542,  "lon": -73.9932433},{  "type": "node",  "id": 42449817,  "lat": 40.7301550,  "lon": -73.9960630},{  "type": "node",  "id": 42453104,  "lat": 40.7319645,  "lon": -73.9902032},{  "type": "node",  "id": 42455026,  "lat": 40.7333040,  "lon": -73.9933220},{  "type": "node",  "id": 42456205,  "lat": 40.7291800,  "lon": -73.9958120},{  "type": "node",  "id": 42456206,  "lat": 40.7297760,  "lon": -73.9953080},{  "type": "node",  "id": 42456211,  "lat": 40.7312837,  "lon": -73.9940304},{  "type": "node",  "id": 42456303,  "lat": 40.7316000,  "lon": -73.9947861},{  "type": "node",  "id": 42457697,  "lat": 40.7281520,  "lon": -73.9937240},{  "type": "node",  "id": 595407624,  "lat": 40.7286200,  "lon": -73.9911572},{  "type": "node",  "id": 595407639,  "lat": 40.7296855,  "lon": -73.9902106},{  "type": "node",  "id": 595407648,  "lat": 40.7310019,  "lon": -73.9904097},{  "type": "node",  "id": 1110619966,  "lat": 40.7291987,  "lon": -73.9909228},{  "type": "node",  "id": 1110619975,  "lat": 40.7289950,  "lon": -73.9904675},{  "type": "node",  "id": 1773121031,  "lat": 40.7275016,  "lon": -73.9914029},{  "type": "node",  "id": 1773121034,  "lat": 40.7277700,  "lon": -73.9912550},{  "type": "node",  "id": 3584752194,  "lat": 40.7295094,  "lon": -73.9966145},{  "type": "node",  "id": 3584752195,  "lat": 40.7295311,  "lon": -73.9965063},{  "type": "node",  "id": 3584752198,  "lat": 40.7295960,  "lon": -73.9966376},{  "type": "node",  "id": 3584752200,  "lat": 40.7296183,  "lon": -73.9965218},{  "type": "node",  "id": 3584752212,  "lat": 40.7300940,  "lon": -73.9961152},{  "type": "node",  "id": 3584752215,  "lat": 40.7301318,  "lon": -73.9960167},{  "type": "node",  "id": 3584752226,  "lat": 40.7302298,  "lon": -73.9959983},{  "type": "node",  "id": 3584752273,  "lat": 40.7306699,  "lon": -73.9956200},{  "type": "node",  "id": 3584752276,  "lat": 40.7306805,  "lon": -73.9954858},{  "type": "node",  "id": 3584752291,  "lat": 40.7307629,  "lon": -73.9956525},{  "type": "node",  "id": 3584752292,  "lat": 40.7307838,  "lon": -73.9955151},{  "type": "node",  "id": 3627218171,  "lat": 40.7299697,  "lon": -73.9907218},{  "type": "node",  "id": 3919350745,  "lat": 40.7299992,  "lon": -73.9911621},{  "type": "node",  "id": 3919350754,  "lat": 40.7299383,  "lon": -73.9908141},{  "type": "node",  "id": 3919350757,  "lat": 40.7298553,  "lon": -73.9906619},{  "type": "node",  "id": 3919350758,  "lat": 40.7298852,  "lon": -73.9915756},{  "type": "node",  "id": 3919350763,  "lat": 40.7297872,  "lon": -73.9915064},{  "type": "node",  "id": 3919350764,  "lat": 40.7300060,  "lon": -73.9909770},{  "type": "node",  "id": 3919359296,  "lat": 40.7305046,  "lon": -73.9905176},{  "type": "node",  "id": 3919359299,  "lat": 40.7305679,  "lon": -73.9906806},{  "type": "node",  "id": 3957463101,  "lat": 40.7284628,  "lon": -73.9961129},{  "type": "node",  "id": 3957463103,  "lat": 40.7290097,  "lon": -73.9954637},{  "type": "node",  "id": 4015190382,  "lat": 40.7283580,  "lon": -73.9907799},{  "type": "node",  "id": 4117109189,  "lat": 40.7288579,  "lon": -73.9950197},{  "type": "node",  "id": 4117109190,  "lat": 40.7288393,  "lon": -73.9951205},{  "type": "node",  "id": 4202984497,  "lat": 40.7296573,  "lon": -73.9916165},{  "type": "node",  "id": 4234425009,  "lat": 40.7306923,  "lon": -73.9905994},{  "type": "node",  "id": 4234425018,  "lat": 40.7305240,  "lon": -73.9904242},{  "type": "node",  "id": 4234425037,  "lat": 40.7306563,  "lon": -73.9907415},{  "type": "node",  "id": 4234425039,  "lat": 40.7306410,  "lon": -73.9904794},{  "type": "node",  "id": 4234425046,  "lat": 40.7299192,  "lon": -73.9913204},{  "type": "node",  "id": 4234578190,  "lat": 40.7313496,  "lon": -73.9903368},{  "type": "node",  "id": 4234578192,  "lat": 40.7313608,  "lon": -73.9905446},{  "type": "node",  "id": 4234578193,  "lat": 40.7312044,  "lon": -73.9903654},{  "type": "node",  "id": 4234578194,  "lat": 40.7312330,  "lon": -73.9902382},{  "type": "node",  "id": 4500010367,  "lat": 40.7316143,  "lon": -73.9932047},{  "type": "node",  "id": 4500010369,  "lat": 40.7316306,  "lon": -73.9931928},{  "type": "node",  "id": 4500010374,  "lat": 40.7316861,  "lon": -73.9931521},{  "type": "node",  "id": 4500010382,  "lat": 40.7318225,  "lon": -73.9934783},{  "type": "node",  "id": 4500033344,  "lat": 40.7314439,  "lon": -73.9907462},{  "type": "node",  "id": 4500033348,  "lat": 40.7315866,  "lon": -73.9910925},{  "type": "node",  "id": 4577294519,  "lat": 40.7276571,  "lon": -73.9909901},{  "type": "node",  "id": 4621333639,  "lat": 40.7284873,  "lon": -73.9910846},{  "type": "node",  "id": 4621333640,  "lat": 40.7279568,  "lon": -73.9913790},{  "type": "node",  "id": 4621333641,  "lat": 40.7278592,  "lon": -73.9914013},{  "type": "node",  "id": 4621333642,  "lat": 40.7278215,  "lon": -73.9913614},{  "type": "node",  "id": 4684550288,  "lat": 40.7321691,  "lon": -73.9913510},{  "type": "node",  "id": 4745685357,  "lat": 40.7287673,  "lon": -73.9950979},{  "type": "node",  "id": 4745685361,  "lat": 40.7285264,  "lon": -73.9960601},{  "type": "node",  "id": 4745685363,  "lat": 40.7285559,  "lon": -73.9960365},{  "type": "node",  "id": 4745685367,  "lat": 40.7291140,  "lon": -73.9958676},{  "type": "node",  "id": 4745685370,  "lat": 40.7291011,  "lon": -73.9958788},{  "type": "node",  "id": 4745686172,  "lat": 40.7292126,  "lon": -73.9958764},{  "type": "node",  "id": 4745686174,  "lat": 40.7291585,  "lon": -73.9957680},{  "type": "node",  "id": 4747401774,  "lat": 40.7316651,  "lon": -73.9915097},{  "type": "node",  "id": 4747401775,  "lat": 40.7318254,  "lon": -73.9914263},{  "type": "node",  "id": 4755898678,  "lat": 40.7307956,  "lon": -73.9928643},{  "type": "node",  "id": 4755898680,  "lat": 40.7297053,  "lon": -73.9931563},{  "type": "node",  "id": 4755898682,  "lat": 40.7293218,  "lon": -73.9934809},{  "type": "node",  "id": 4755898710,  "lat": 40.7274317,  "lon": -73.9950852},{  "type": "node",  "id": 4762558061,  "lat": 40.7313573,  "lon": -73.9923739},{  "type": "node",  "id": 5165772929,  "lat": 40.7284937,  "lon": -73.9906779},{  "type": "node",  "id": 5357086126,  "lat": 40.7329392,  "lon": -73.9904648},{  "type": "node",  "id": 5357196527,  "lat": 40.7337971,  "lon": -73.9944891},{  "type": "node",  "id": 5481937335,  "lat": 40.7274473,  "lon": -73.9935050},{  "type": "node",  "id": 5590991384,  "lat": 40.7291332,  "lon": -73.9902061},{  "type": "node",  "id": 5780009413,  "lat": 40.7305228,  "lon": -73.9904213},{  "type": "node",  "id": 5784518512,  "lat": 40.7285599,  "lon": -73.9911429},{  "type": "node",  "id": 5784518517,  "lat": 40.7284980,  "lon": -73.9911980},{  "type": "node",  "id": 5784518520,  "lat": 40.7284232,  "lon": -73.9907309},{  "type": "node",  "id": 5784518525,  "lat": 40.7284123,  "lon": -73.9909133},{  "type": "node",  "id": 5784518526,  "lat": 40.7282829,  "lon": -73.9908405},{  "type": "node",  "id": 5784518536,  "lat": 40.7301013,  "lon": -73.9912041},{  "type": "node",  "id": 5784518538,  "lat": 40.7297841,  "lon": -73.9915098},{  "type": "node",  "id": 5784518539,  "lat": 40.7291667,  "lon": -73.9908514},{  "type": "node",  "id": 5784518540,  "lat": 40.7290053,  "lon": -73.9904905},{  "type": "node",  "id": 5784518544,  "lat": 40.7278127,  "lon": -73.9930344},{  "type": "node",  "id": 5784518545,  "lat": 40.7301062,  "lon": -73.9910435},{  "type": "node",  "id": 5824300189,  "lat": 40.7278545,  "lon": -73.9948739},{  "type": "node",  "id": 5824300190,  "lat": 40.7278038,  "lon": -73.9946479},{  "type": "node",  "id": 6100764861,  "lat": 40.7275654,  "lon": -73.9913677},{  "type": "node",  "id": 6255601607,  "lat": 40.7319388,  "lon": -73.9955957},{  "type": "node",  "id": 6255601608,  "lat": 40.7319696,  "lon": -73.9956692},{  "type": "node",  "id": 42421882,  "lat": 40.7341472,  "lon": -73.9990143},{  "type": "node",  "id": 42421889,  "lat": 40.7341463,  "lon": -73.9991816},{  "type": "node",  "id": 42421901,  "lat": 40.7347830,  "lon": -73.9992480},{  "type": "node",  "id": 42429374,  "lat": 40.7366680,  "lon": -73.9973410},{  "type": "node",  "id": 42429375,  "lat": 40.7378614,  "lon": -74.0001742},{  "type": "node",  "id": 42429377,  "lat": 40.7386731,  "lon": -74.0021545},{  "type": "node",  "id": 42429378,  "lat": 40.7387436,  "lon": -74.0024001},{  "type": "node",  "id": 42429388,  "lat": 40.7390674,  "lon": -74.0030522},{  "type": "node",  "id": 42430241,  "lat": 40.7347840,  "lon": -73.9987140},{  "type": "node",  "id": 42430244,  "lat": 40.7350306,  "lon": -73.9985393},{  "type": "node",  "id": 42430247,  "lat": 40.7353910,  "lon": -73.9982740},{  "type": "node",  "id": 42430249,  "lat": 40.7360320,  "lon": -73.9978050},{  "type": "node",  "id": 42430253,  "lat": 40.7373610,  "lon": -73.9968400},{  "type": "node",  "id": 42430811,  "lat": 40.7343413,  "lon": -74.0023950},{  "type": "node",  "id": 42430813,  "lat": 40.7350691,  "lon": -74.0019694},{  "type": "node",  "id": 42430818,  "lat": 40.7355352,  "lon": -74.0016967},{  "type": "node",  "id": 42430824,  "lat": 40.7357944,  "lon": -74.0015452},{  "type": "node",  "id": 42430828,  "lat": 40.7365640,  "lon": -74.0010950},{  "type": "node",  "id": 42432191,  "lat": 40.7340688,  "lon": -73.9996618},{  "type": "node",  "id": 42432194,  "lat": 40.7346880,  "lon": -73.9999730},{  "type": "node",  "id": 42432199,  "lat": 40.7353070,  "lon": -74.0003560},{  "type": "node",  "id": 42432201,  "lat": 40.7359260,  "lon": -74.0007220},{  "type": "node",  "id": 42432205,  "lat": 40.7371680,  "lon": -74.0014630},{  "type": "node",  "id": 42432208,  "lat": 40.7376609,  "lon": -74.0017550},{  "type": "node",  "id": 42432211,  "lat": 40.7381296,  "lon": -74.0020342},{  "type": "node",  "id": 42432214,  "lat": 40.7393750,  "lon": -74.0027775},{  "type": "node",  "id": 42433264,  "lat": 40.7344830,  "lon": -74.0013890},{  "type": "node",  "id": 42433269,  "lat": 40.7351280,  "lon": -74.0015740},{  "type": "node",  "id": 42433272,  "lat": 40.7357626,  "lon": -74.0017616},{  "type": "node",  "id": 42433275,  "lat": 40.7364365,  "lon": -74.0019538},{  "type": "node",  "id": 42433278,  "lat": 40.7370660,  "lon": -74.0021360},{  "type": "node",  "id": 42435596,  "lat": 40.7397447,  "lon": -74.0024906},{  "type": "node",  "id": 42435598,  "lat": 40.7404441,  "lon": -74.0019879},{  "type": "node",  "id": 42436856,  "lat": 40.7339701,  "lon": -74.0003713},{  "type": "node",  "id": 42439073,  "lat": 40.7372222,  "lon": -74.0006278},{  "type": "node",  "id": 42439823,  "lat": 40.7385520,  "lon": -73.9996840},{  "type": "node",  "id": 42439826,  "lat": 40.7392430,  "lon": -73.9991850},{  "type": "node",  "id": 42439830,  "lat": 40.7398810,  "lon": -73.9987130},{  "type": "node",  "id": 42452598,  "lat": 40.7348912,  "lon": -74.0031527},{  "type": "node",  "id": 42457721,  "lat": 40.7348465,  "lon": -73.9988468},{  "type": "node",  "id": 42457728,  "lat": 40.7342533,  "lon": -74.0029703},{  "type": "node",  "id": 1701920968,  "lat": 40.7376511,  "lon": -74.0016965},{  "type": "node",  "id": 1701920972,  "lat": 40.7376412,  "lon": -74.0016365},{  "type": "node",  "id": 1701921005,  "lat": 40.7376112,  "lon": -74.0015555},{  "type": "node",  "id": 1701921025,  "lat": 40.7387178,  "lon": -74.0022742},{  "type": "node",  "id": 1760570976,  "lat": 40.7342309,  "lon": -73.9997137},{  "type": "node",  "id": 1760570978,  "lat": 40.7343394,  "lon": -73.9997690},{  "type": "node",  "id": 1760570979,  "lat": 40.7343702,  "lon": -73.9997878},{  "type": "node",  "id": 1760570980,  "lat": 40.7342725,  "lon": -73.9997331},{  "type": "node",  "id": 1760570982,  "lat": 40.7343109,  "lon": -73.9997543},{  "type": "node",  "id": 1825740014,  "lat": 40.7387702,  "lon": -74.0025336},{  "type": "node",  "id": 1825740017,  "lat": 40.7391685,  "lon": -74.0029414},{  "type": "node",  "id": 4044849083,  "lat": 40.7382552,  "lon": -74.0021073},{  "type": "node",  "id": 4205565508,  "lat": 40.7397974,  "lon": -74.0026235},{  "type": "node",  "id": 4320194260,  "lat": 40.7341264,  "lon": -73.9996743},{  "type": "node",  "id": 4320194262,  "lat": 40.7342160,  "lon": -73.9991284},{  "type": "node",  "id": 4320194264,  "lat": 40.7339647,  "lon": -74.0004070},{  "type": "node",  "id": 4320194265,  "lat": 40.7339788,  "lon": -74.0003088},{  "type": "node",  "id": 4320194267,  "lat": 40.7340793,  "lon": -73.9992264},{  "type": "node",  "id": 4320194269,  "lat": 40.7340536,  "lon": -73.9997710},{  "type": "node",  "id": 4320194273,  "lat": 40.7339257,  "lon": -74.0003599},{  "type": "node",  "id": 4320194277,  "lat": 40.7347038,  "lon": -73.9998521},{  "type": "node",  "id": 4816137912,  "lat": 40.7390552,  "lon": -74.0032105},{  "type": "node",  "id": 4841945996,  "lat": 40.7341355,  "lon": -73.9989395},{  "type": "node",  "id": 5155816415,  "lat": 40.7349892,  "lon": -73.9985687},{  "type": "node",  "id": 5244228880,  "lat": 40.7339959,  "lon": -74.0025915},{  "type": "node",  "id": 5244228881,  "lat": 40.7342885,  "lon": -74.0027404},{  "type": "node",  "id": 5244228882,  "lat": 40.7353477,  "lon": -74.0018064},{  "type": "node",  "id": 5244228883,  "lat": 40.7368080,  "lon": -74.0009218},{  "type": "node",  "id": 5244228884,  "lat": 40.7381978,  "lon": -73.9999354},{  "type": "node",  "id": 5244228885,  "lat": 40.7383446,  "lon": -73.9991888},{  "type": "node",  "id": 5244228886,  "lat": 40.7375550,  "lon": -73.9973033},{  "type": "node",  "id": 5244228888,  "lat": 40.7362935,  "lon": -73.9976142},{  "type": "node",  "id": 5413254261,  "lat": 40.7373712,  "lon": -73.9968326},{  "type": "node",  "id": 5413254263,  "lat": 40.7373421,  "lon": -73.9968536},{  "type": "node",  "id": 5554351996,  "lat": 40.7358838,  "lon": -74.0014929},{  "type": "node",  "id": 5690441975,  "lat": 40.7340154,  "lon": -73.9996494},{  "type": "node",  "id": 5690441976,  "lat": 40.7340938,  "lon": -73.9995275},{  "type": "node",  "id": 5690441977,  "lat": 40.7341288,  "lon": -73.9993539},{  "type": "node",  "id": 5849907289,  "lat": 40.7376868,  "lon": -74.0019143},{  "type": "node",  "id": 6211332474,  "lat": 40.7392716,  "lon": -74.0027163},{  "type": "node",  "id": 6211332483,  "lat": 40.7391128,  "lon": -74.0031635},{  "type": "node",  "id": 6211334191,  "lat": 40.7391292,  "lon": -74.0032012},{  "type": "node",  "id": 6211334199,  "lat": 40.7392425,  "lon": -74.0028827},{  "type": "node",  "id": 6211484429,  "lat": 40.7392926,  "lon": -74.0029857},{  "type": "node",  "id": 6211484437,  "lat": 40.7391992,  "lon": -74.0030625},{  "type": "node",  "id": 6213315371,  "lat": 40.7388533,  "lon": -74.0024658},{  "type": "node",  "id": 6213315373,  "lat": 40.7386728,  "lon": -74.0023576},{  "type": "node",  "id": 6213315378,  "lat": 40.7395009,  "lon": -74.0028363},{  "type": "node",  "id": 6213315379,  "lat": 40.7381296,  "lon": -74.0021367},{  "type": "node",  "id": 6254325748,  "lat": 40.7347578,  "lon": -73.9994401},{  "type": "node",  "id": 42427369,  "lat": 40.7396730,  "lon": -73.9909450},{  "type": "node",  "id": 42428170,  "lat": 40.7371143,  "lon": -73.9903427},{  "type": "node",  "id": 42428491,  "lat": 40.7339780,  "lon": -73.9909450},{  "type": "node",  "id": 42428493,  "lat": 40.7347938,  "lon": -73.9907171},{  "type": "node",  "id": 42429373,  "lat": 40.7353130,  "lon": -73.9941180},{  "type": "node",  "id": 42429657,  "lat": 40.7385065,  "lon": -73.9917962},{  "type": "node",  "id": 42429659,  "lat": 40.7398600,  "lon": -73.9950180},{  "type": "node",  "id": 42430255,  "lat": 40.7380440,  "lon": -73.9963370},{  "type": "node",  "id": 42430257,  "lat": 40.7386860,  "lon": -73.9958740},{  "type": "node",  "id": 42430259,  "lat": 40.7392720,  "lon": -73.9954460},{  "type": "node",  "id": 42430263,  "lat": 40.7404420,  "lon": -73.9945930},{  "type": "node",  "id": 42431490,  "lat": 40.7366910,  "lon": -73.9931200},{  "type": "node",  "id": 42431491,  "lat": 40.7358871,  "lon": -73.9912296},{  "type": "node",  "id": 42436935,  "lat": 40.7345320,  "lon": -73.9922660},{  "type": "node",  "id": 42437009,  "lat": 40.7349589,  "lon": -73.9904530},{  "type": "node",  "id": 42438784,  "lat": 40.7352048,  "lon": -73.9916914},{  "type": "node",  "id": 42438785,  "lat": 40.7365258,  "lon": -73.9907376},{  "type": "node",  "id": 42439070,  "lat": 40.7346720,  "lon": -73.9945850},{  "type": "node",  "id": 42439272,  "lat": 40.7379170,  "lon": -73.9922250},{  "type": "node",  "id": 42440710,  "lat": 40.7360129,  "lon": -73.9936067},{  "type": "node",  "id": 42443671,  "lat": 40.7402492,  "lon": -73.9905314},{  "type": "node",  "id": 42446521,  "lat": 40.7339220,  "lon": -73.9927920},{  "type": "node",  "id": 42449067,  "lat": 40.7340340,  "lon": -73.9950500},{  "type": "node",  "id": 42449333,  "lat": 40.7373300,  "lon": -73.9926540},{  "type": "node",  "id": 42454378,  "lat": 40.7390885,  "lon": -73.9913746},{  "type": "node",  "id": 1251605769,  "lat": 40.7348662,  "lon": -73.9906445},{  "type": "node",  "id": 1412706986,  "lat": 40.7370515,  "lon": -73.9903063},{  "type": "node",  "id": 2077984658,  "lat": 40.7349890,  "lon": -73.9904039},{  "type": "node",  "id": 2077984663,  "lat": 40.7350107,  "lon": -73.9903683},{  "type": "node",  "id": 3670742435,  "lat": 40.7347488,  "lon": -73.9906110},{  "type": "node",  "id": 3670742436,  "lat": 40.7346900,  "lon": -73.9907843},{  "type": "node",  "id": 3670742441,  "lat": 40.7351705,  "lon": -73.9916100},{  "type": "node",  "id": 3670742462,  "lat": 40.7351048,  "lon": -73.9917768},{  "type": "node",  "id": 3670742474,  "lat": 40.7348419,  "lon": -73.9908310},{  "type": "node",  "id": 3670742486,  "lat": 40.7353126,  "lon": -73.9916533},{  "type": "node",  "id": 3875000763,  "lat": 40.7370741,  "lon": -73.9902472},{  "type": "node",  "id": 3875000767,  "lat": 40.7370531,  "lon": -73.9903837},{  "type": "node",  "id": 3884600523,  "lat": 40.7370353,  "lon": -73.9902704},{  "type": "node",  "id": 4205565500,  "lat": 40.7351648,  "lon": -73.9901645},{  "type": "node",  "id": 4285055690,  "lat": 40.7371495,  "lon": -73.9904257},{  "type": "node",  "id": 4285055691,  "lat": 40.7371847,  "lon": -73.9903067},{  "type": "node",  "id": 4500010387,  "lat": 40.7343472,  "lon": -73.9948217},{  "type": "node",  "id": 4500012796,  "lat": 40.7348564,  "lon": -73.9950217},{  "type": "node",  "id": 4500012798,  "lat": 40.7369452,  "lon": -73.9937245},{  "type": "node",  "id": 4762558055,  "lat": 40.7345959,  "lon": -73.9908055},{  "type": "node",  "id": 4762558057,  "lat": 40.7344395,  "lon": -73.9908408},{  "type": "node",  "id": 4763882045,  "lat": 40.7377240,  "lon": -73.9901713},{  "type": "node",  "id": 4983147326,  "lat": 40.7352187,  "lon": -73.9910327},{  "type": "node",  "id": 4983147327,  "lat": 40.7352732,  "lon": -73.9910599},{  "type": "node",  "id": 4983147822,  "lat": 40.7362511,  "lon": -73.9904902},{  "type": "node",  "id": 4983147823,  "lat": 40.7362999,  "lon": -73.9904406},{  "type": "node",  "id": 4983147824,  "lat": 40.7364774,  "lon": -73.9902565},{  "type": "node",  "id": 4983147825,  "lat": 40.7364241,  "lon": -73.9903004},{  "type": "node",  "id": 4983147826,  "lat": 40.7350722,  "lon": -73.9906117},{  "type": "node",  "id": 4983147827,  "lat": 40.7350059,  "lon": -73.9906354},{  "type": "node",  "id": 4983147828,  "lat": 40.7350822,  "lon": -73.9906120},{  "type": "node",  "id": 4983147829,  "lat": 40.7351538,  "lon": -73.9905868},{  "type": "node",  "id": 4983147830,  "lat": 40.7353229,  "lon": -73.9905261},{  "type": "node",  "id": 4983147831,  "lat": 40.7353769,  "lon": -73.9905116},{  "type": "node",  "id": 4983147832,  "lat": 40.7361182,  "lon": -73.9902768},{  "type": "node",  "id": 4983147833,  "lat": 40.7360687,  "lon": -73.9902888},{  "type": "node",  "id": 4983147834,  "lat": 40.7362237,  "lon": -73.9902406},{  "type": "node",  "id": 4983147835,  "lat": 40.7361663,  "lon": -73.9902604},{  "type": "node",  "id": 4983147836,  "lat": 40.7363127,  "lon": -73.9903769},{  "type": "node",  "id": 4983147837,  "lat": 40.7361959,  "lon": -73.9904096},{  "type": "node",  "id": 4983147838,  "lat": 40.7361480,  "lon": -73.9904229},{  "type": "node",  "id": 4983147839,  "lat": 40.7360957,  "lon": -73.9904389},{  "type": "node",  "id": 4983147840,  "lat": 40.7353571,  "lon": -73.9906864},{  "type": "node",  "id": 4983147841,  "lat": 40.7354070,  "lon": -73.9906695},{  "type": "node",  "id": 4983147842,  "lat": 40.7351103,  "lon": -73.9907616},{  "type": "node",  "id": 4983147843,  "lat": 40.7351765,  "lon": -73.9907387},{  "type": "node",  "id": 4983147844,  "lat": 40.7350960,  "lon": -73.9907602},{  "type": "node",  "id": 4983147845,  "lat": 40.7350344,  "lon": -73.9907831},{  "type": "node",  "id": 4983147846,  "lat": 40.7352084,  "lon": -73.9905607},{  "type": "node",  "id": 4983147847,  "lat": 40.7351850,  "lon": -73.9905662},{  "type": "node",  "id": 4983147848,  "lat": 40.7351906,  "lon": -73.9905921},{  "type": "node",  "id": 4983147849,  "lat": 40.7352135,  "lon": -73.9905844},{  "type": "node",  "id": 4983147850,  "lat": 40.7352352,  "lon": -73.9907039},{  "type": "node",  "id": 4983147851,  "lat": 40.7352129,  "lon": -73.9907106},{  "type": "node",  "id": 4983147852,  "lat": 40.7352188,  "lon": -73.9907362},{  "type": "node",  "id": 4983147853,  "lat": 40.7352416,  "lon": -73.9907290},{  "type": "node",  "id": 4983147854,  "lat": 40.7349210,  "lon": -73.9906305},{  "type": "node",  "id": 4983147895,  "lat": 40.7348696,  "lon": -73.9911996},{  "type": "node",  "id": 4983147896,  "lat": 40.7348500,  "lon": -73.9911607},{  "type": "node",  "id": 4983147897,  "lat": 40.7348419,  "lon": -73.9911215},{  "type": "node",  "id": 4983147898,  "lat": 40.7348284,  "lon": -73.9910913},{  "type": "node",  "id": 4983147899,  "lat": 40.7347606,  "lon": -73.9909753},{  "type": "node",  "id": 4983147900,  "lat": 40.7347702,  "lon": -73.9909991},{  "type": "node",  "id": 4983147901,  "lat": 40.7347951,  "lon": -73.9910196},{  "type": "node",  "id": 4983147902,  "lat": 40.7348081,  "lon": -73.9910424},{  "type": "node",  "id": 4983147903,  "lat": 40.7352229,  "lon": -73.9903105},{  "type": "node",  "id": 4983147904,  "lat": 40.7352593,  "lon": -73.9902696},{  "type": "node",  "id": 4983147905,  "lat": 40.7352349,  "lon": -73.9903259},{  "type": "node",  "id": 4983147906,  "lat": 40.7352649,  "lon": -73.9902887},{  "type": "node",  "id": 4983147907,  "lat": 40.7352443,  "lon": -73.9903400},{  "type": "node",  "id": 4983147908,  "lat": 40.7352755,  "lon": -73.9902954},{  "type": "node",  "id": 4983147909,  "lat": 40.7352534,  "lon": -73.9903591},{  "type": "node",  "id": 4983147910,  "lat": 40.7352865,  "lon": -73.9903212},{  "type": "node",  "id": 4983147912,  "lat": 40.7349888,  "lon": -73.9912165},{  "type": "node",  "id": 4983147913,  "lat": 40.7349707,  "lon": -73.9911770},{  "type": "node",  "id": 4983147914,  "lat": 40.7346267,  "lon": -73.9903739},{  "type": "node",  "id": 4983147915,  "lat": 40.7346478,  "lon": -73.9904240},{  "type": "node",  "id": 4983147916,  "lat": 40.7345867,  "lon": -73.9902858},{  "type": "node",  "id": 4983147917,  "lat": 40.7345675,  "lon": -73.9902375},{  "type": "node",  "id": 4983147924,  "lat": 40.7347566,  "lon": -73.9906969},{  "type": "node",  "id": 4983147925,  "lat": 40.7347733,  "lon": -73.9907424},{  "type": "node",  "id": 4983147926,  "lat": 40.7349179,  "lon": -73.9908051},{  "type": "node",  "id": 4983147927,  "lat": 40.7348195,  "lon": -73.9908482},{  "type": "node",  "id": 4983147928,  "lat": 40.7347989,  "lon": -73.9908028},{  "type": "node",  "id": 4983147933,  "lat": 40.7345081,  "lon": -73.9903301},{  "type": "node",  "id": 4983147934,  "lat": 40.7344912,  "lon": -73.9902887},{  "type": "node",  "id": 4983147940,  "lat": 40.7350431,  "lon": -73.9913461},{  "type": "node",  "id": 4983147941,  "lat": 40.7350192,  "lon": -73.9912985},{  "type": "node",  "id": 4983147942,  "lat": 40.7348674,  "lon": -73.9910365},{  "type": "node",  "id": 4983147943,  "lat": 40.7347988,  "lon": -73.9910867},{  "type": "node",  "id": 4983147944,  "lat": 40.7346609,  "lon": -73.9911012},{  "type": "node",  "id": 4983147945,  "lat": 40.7346781,  "lon": -73.9911420},{  "type": "node",  "id": 4983147946,  "lat": 40.7347130,  "lon": -73.9911643},{  "type": "node",  "id": 4983147947,  "lat": 40.7347279,  "lon": -73.9911984},{  "type": "node",  "id": 4983147948,  "lat": 40.7347722,  "lon": -73.9914042},{  "type": "node",  "id": 4983147949,  "lat": 40.7347640,  "lon": -73.9913365},{  "type": "node",  "id": 4983147950,  "lat": 40.7347663,  "lon": -73.9912936},{  "type": "node",  "id": 4983147951,  "lat": 40.7347561,  "lon": -73.9912682},{  "type": "node",  "id": 5148103557,  "lat": 40.7359180,  "lon": -73.9913027},{  "type": "node",  "id": 5148103559,  "lat": 40.7364629,  "lon": -73.9907830},{  "type": "node",  "id": 5148103560,  "lat": 40.7365883,  "lon": -73.9906956},{  "type": "node",  "id": 5148103562,  "lat": 40.7359513,  "lon": -73.9911833},{  "type": "node",  "id": 5148103566,  "lat": 40.7352431,  "lon": -73.9917821},{  "type": "node",  "id": 5148103568,  "lat": 40.7365628,  "lon": -73.9908257},{  "type": "node",  "id": 5155799272,  "lat": 40.7379336,  "lon": -73.9964183},{  "type": "node",  "id": 5243716991,  "lat": 40.7344209,  "lon": -73.9947680},{  "type": "node",  "id": 5243716992,  "lat": 40.7357838,  "lon": -73.9930637},{  "type": "node",  "id": 5243716993,  "lat": 40.7353533,  "lon": -73.9920432},{  "type": "node",  "id": 5244228887,  "lat": 40.7371956,  "lon": -73.9964432},{  "type": "node",  "id": 5350434145,  "lat": 40.7380782,  "lon": -73.9964183},{  "type": "node",  "id": 5416901182,  "lat": 40.7370431,  "lon": -73.9960774},{  "type": "node",  "id": 5416905054,  "lat": 40.7377245,  "lon": -73.9955735},{  "type": "node",  "id": 5481850675,  "lat": 40.7393715,  "lon": -73.9938551},{  "type": "node",  "id": 5481850676,  "lat": 40.7387808,  "lon": -73.9942821},{  "type": "node",  "id": 42430886,  "lat": 40.7307238,  "lon": -73.9890529},{  "type": "node",  "id": 42430898,  "lat": 40.7297790,  "lon": -73.9868120},{  "type": "node",  "id": 42430903,  "lat": 40.7287826,  "lon": -73.9844396},{  "type": "node",  "id": 42431461,  "lat": 40.7333720,  "lon": -73.9852420},{  "type": "node",  "id": 42431464,  "lat": 40.7330330,  "lon": -73.9844410},{  "type": "node",  "id": 42431467,  "lat": 40.7326990,  "lon": -73.9836460},{  "type": "node",  "id": 42434140,  "lat": 40.7295611,  "lon": -73.9898997},{  "type": "node",  "id": 42434142,  "lat": 40.7286043,  "lon": -73.9876651},{  "type": "node",  "id": 42434145,  "lat": 40.7275990,  "lon": -73.9853070},{  "type": "node",  "id": 42434197,  "lat": 40.7287072,  "lon": -73.9897967},{  "type": "node",  "id": 42434201,  "lat": 40.7279920,  "lon": -73.9881146},{  "type": "node",  "id": 42436939,  "lat": 40.7335280,  "lon": -73.9898680},{  "type": "node",  "id": 42436941,  "lat": 40.7326041,  "lon": -73.9876833},{  "type": "node",  "id": 42436942,  "lat": 40.7316580,  "lon": -73.9854440},{  "type": "node",  "id": 42440729,  "lat": 40.7332927,  "lon": -73.9871818},{  "type": "node",  "id": 42440737,  "lat": 40.7323470,  "lon": -73.9849410},{  "type": "node",  "id": 42442851,  "lat": 40.7274076,  "lon": -73.9885349},{  "type": "node",  "id": 42442857,  "lat": 40.7291879,  "lon": -73.9872420},{  "type": "node",  "id": 42442862,  "lat": 40.7303818,  "lon": -73.9863742},{  "type": "node",  "id": 42442870,  "lat": 40.7310190,  "lon": -73.9859090},{  "type": "node",  "id": 42443296,  "lat": 40.7298638,  "lon": -73.9896776},{  "type": "node",  "id": 42443298,  "lat": 40.7299120,  "lon": -73.9889263},{  "type": "node",  "id": 42446528,  "lat": 40.7327560,  "lon": -73.9900270},{  "type": "node",  "id": 42446533,  "lat": 40.7319637,  "lon": -73.9881498},{  "type": "node",  "id": 42446540,  "lat": 40.7316860,  "lon": -73.9874890},{  "type": "node",  "id": 42449014,  "lat": 40.7303763,  "lon": -73.9900724},{  "type": "node",  "id": 42449017,  "lat": 40.7301298,  "lon": -73.9894855},{  "type": "node",  "id": 42449021,  "lat": 40.7281940,  "lon": -73.9848750},{  "type": "node",  "id": 42449886,  "lat": 40.7313257,  "lon": -73.9886146},{  "type": "node",  "id": 42453110,  "lat": 40.7336893,  "lon": -73.9898302},{  "type": "node",  "id": 42453113,  "lat": 40.7338288,  "lon": -73.9898164},{  "type": "node",  "id": 42454673,  "lat": 40.7305969,  "lon": -73.9899114},{  "type": "node",  "id": 42454994,  "lat": 40.7293730,  "lon": -73.9839987},{  "type": "node",  "id": 597937476,  "lat": 40.7332799,  "lon": -73.9871911},{  "type": "node",  "id": 1251612266,  "lat": 40.7336653,  "lon": -73.9839818},{  "type": "node",  "id": 3420766629,  "lat": 40.7274694,  "lon": -73.9886810},{  "type": "node",  "id": 3420766630,  "lat": 40.7274712,  "lon": -73.9884892},{  "type": "node",  "id": 3420766631,  "lat": 40.7279262,  "lon": -73.9881620},{  "type": "node",  "id": 3420766632,  "lat": 40.7280477,  "lon": -73.9880737},{  "type": "node",  "id": 3420766633,  "lat": 40.7285462,  "lon": -73.9877078},{  "type": "node",  "id": 3420766634,  "lat": 40.7286617,  "lon": -73.9876236},{  "type": "node",  "id": 3420766635,  "lat": 40.7291379,  "lon": -73.9872783},{  "type": "node",  "id": 3420766636,  "lat": 40.7292580,  "lon": -73.9871910},{  "type": "node",  "id": 3420766637,  "lat": 40.7297303,  "lon": -73.9868474},{  "type": "node",  "id": 3420766638,  "lat": 40.7298409,  "lon": -73.9867672},{  "type": "node",  "id": 3420766639,  "lat": 40.7304431,  "lon": -73.9863307},{  "type": "node",  "id": 3627218170,  "lat": 40.7312658,  "lon": -73.9884779},{  "type": "node",  "id": 3627218173,  "lat": 40.7326631,  "lon": -73.9878229},{  "type": "node",  "id": 3627218174,  "lat": 40.7325503,  "lon": -73.9875558},{  "type": "node",  "id": 3627218175,  "lat": 40.7313855,  "lon": -73.9887572},{  "type": "node",  "id": 3627218176,  "lat": 40.7325480,  "lon": -73.9877242},{  "type": "node",  "id": 3627218177,  "lat": 40.7327078,  "lon": -73.9876157},{  "type": "node",  "id": 3627218178,  "lat": 40.7312679,  "lon": -73.9886567},{  "type": "node",  "id": 3627218179,  "lat": 40.7319135,  "lon": -73.9881864},{  "type": "node",  "id": 3627218180,  "lat": 40.7313938,  "lon": -73.9885650},{  "type": "node",  "id": 3627218181,  "lat": 40.7320271,  "lon": -73.9881036},{  "type": "node",  "id": 3627218182,  "lat": 40.7320224,  "lon": -73.9882889},{  "type": "node",  "id": 3627218183,  "lat": 40.7319051,  "lon": -73.9880103},{  "type": "node",  "id": 3834459530,  "lat": 40.7288271,  "lon": -73.9845455},{  "type": "node",  "id": 3834459533,  "lat": 40.7294196,  "lon": -73.9841116},{  "type": "node",  "id": 3834459543,  "lat": 40.7300646,  "lon": -73.9836439},{  "type": "node",  "id": 3834459546,  "lat": 40.7276449,  "lon": -73.9854148},{  "type": "node",  "id": 3834459550,  "lat": 40.7282371,  "lon": -73.9849777},{  "type": "node",  "id": 3875091624,  "lat": 40.7276309,  "lon": -73.9854278},{  "type": "node",  "id": 3875091625,  "lat": 40.7282258,  "lon": -73.9849891},{  "type": "node",  "id": 3875091626,  "lat": 40.7288215,  "lon": -73.9845526},{  "type": "node",  "id": 3875091627,  "lat": 40.7294130,  "lon": -73.9841194},{  "type": "node",  "id": 3875091628,  "lat": 40.7300448,  "lon": -73.9836612},{  "type": "node",  "id": 3892037906,  "lat": 40.7300663,  "lon": -73.9875129},{  "type": "node",  "id": 3892037907,  "lat": 40.7300151,  "lon": -73.9876009},{  "type": "node",  "id": 3892039873,  "lat": 40.7300365,  "lon": -73.9875641},{  "type": "node",  "id": 3892040209,  "lat": 40.7299849,  "lon": -73.9876951},{  "type": "node",  "id": 3919350741,  "lat": 40.7295060,  "lon": -73.9899392},{  "type": "node",  "id": 3919350751,  "lat": 40.7296223,  "lon": -73.9900424},{  "type": "node",  "id": 3919359291,  "lat": 40.7296385,  "lon": -73.9898431},{  "type": "node",  "id": 3919359298,  "lat": 40.7295019,  "lon": -73.9897612},{  "type": "node",  "id": 3919359655,  "lat": 40.7300900,  "lon": -73.9893472},{  "type": "node",  "id": 3985008869,  "lat": 40.7332045,  "lon": -73.9872460},{  "type": "node",  "id": 3985008870,  "lat": 40.7332363,  "lon": -73.9870481},{  "type": "node",  "id": 3985008871,  "lat": 40.7333522,  "lon": -73.9873220},{  "type": "node",  "id": 3985008872,  "lat": 40.7333900,  "lon": -73.9871109},{  "type": "node",  "id": 4191922366,  "lat": 40.7334287,  "lon": -73.9852004},{  "type": "node",  "id": 4191922367,  "lat": 40.7330846,  "lon": -73.9845629},{  "type": "node",  "id": 4191922369,  "lat": 40.7327647,  "lon": -73.9835978},{  "type": "node",  "id": 4191922371,  "lat": 40.7330953,  "lon": -73.9843958},{  "type": "node",  "id": 4191926877,  "lat": 40.7329793,  "lon": -73.9843131},{  "type": "node",  "id": 4205565490,  "lat": 40.7312647,  "lon": -73.9857302},{  "type": "node",  "id": 4205565491,  "lat": 40.7318840,  "lon": -73.9852790},{  "type": "node",  "id": 4205565492,  "lat": 40.7326691,  "lon": -73.9847063},{  "type": "node",  "id": 4205565495,  "lat": 40.7333606,  "lon": -73.9842031},{  "type": "node",  "id": 4205565496,  "lat": 40.7339031,  "lon": -73.9838090},{  "type": "node",  "id": 4234578191,  "lat": 40.7319372,  "lon": -73.9900718},{  "type": "node",  "id": 4235733228,  "lat": 40.7298978,  "lon": -73.9890297},{  "type": "node",  "id": 4235733230,  "lat": 40.7298718,  "lon": -73.9895057},{  "type": "node",  "id": 4235733231,  "lat": 40.7298928,  "lon": -73.9891213},{  "type": "node",  "id": 4235733233,  "lat": 40.7300868,  "lon": -73.9895166},{  "type": "node",  "id": 4235733234,  "lat": 40.7301963,  "lon": -73.9894382},{  "type": "node",  "id": 4235733235,  "lat": 40.7301880,  "lon": -73.9896242},{  "type": "node",  "id": 4235734221,  "lat": 40.7299286,  "lon": -73.9888028},{  "type": "node",  "id": 4235734222,  "lat": 40.7298402,  "lon": -73.9886579},{  "type": "node",  "id": 4235734223,  "lat": 40.7297870,  "lon": -73.9886307},{  "type": "node",  "id": 4235734224,  "lat": 40.7298666,  "lon": -73.9886593},{  "type": "node",  "id": 4235734225,  "lat": 40.7299397,  "lon": -73.9885815},{  "type": "node",  "id": 4235734226,  "lat": 40.7298961,  "lon": -73.9886405},{  "type": "node",  "id": 4235738196,  "lat": 40.7292526,  "lon": -73.9873670},{  "type": "node",  "id": 4235738199,  "lat": 40.7285537,  "lon": -73.9875464},{  "type": "node",  "id": 4235738201,  "lat": 40.7286572,  "lon": -73.9877886},{  "type": "node",  "id": 4235738202,  "lat": 40.7291401,  "lon": -73.9871283},{  "type": "node",  "id": 4236104521,  "lat": 40.7298299,  "lon": -73.9869361},{  "type": "node",  "id": 4236104522,  "lat": 40.7304341,  "lon": -73.9864985},{  "type": "node",  "id": 4236104523,  "lat": 40.7303302,  "lon": -73.9862527},{  "type": "node",  "id": 4236104524,  "lat": 40.7297288,  "lon": -73.9866924},{  "type": "node",  "id": 4236104525,  "lat": 40.7303255,  "lon": -73.9864161},{  "type": "node",  "id": 4376122104,  "lat": 40.7273770,  "lon": -73.9884626},{  "type": "node",  "id": 4376122105,  "lat": 40.7279573,  "lon": -73.9880329},{  "type": "node",  "id": 4376122106,  "lat": 40.7285734,  "lon": -73.9875926},{  "type": "node",  "id": 4376122107,  "lat": 40.7291572,  "lon": -73.9871689},{  "type": "node",  "id": 4376122108,  "lat": 40.7297488,  "lon": -73.9867401},{  "type": "node",  "id": 4376122109,  "lat": 40.7303473,  "lon": -73.9862930},{  "type": "node",  "id": 4376122110,  "lat": 40.7309842,  "lon": -73.9858264},{  "type": "node",  "id": 4376122111,  "lat": 40.7316249,  "lon": -73.9853655},{  "type": "node",  "id": 4376122112,  "lat": 40.7323182,  "lon": -73.9848717},{  "type": "node",  "id": 4376122113,  "lat": 40.7330002,  "lon": -73.9843628},{  "type": "node",  "id": 4886319435,  "lat": 40.7329383,  "lon": -73.9884737},{  "type": "node",  "id": 4983147784,  "lat": 40.7338600,  "lon": -73.9898171},{  "type": "node",  "id": 5243723093,  "lat": 40.7321878,  "lon": -73.9845578},{  "type": "node",  "id": 5266421745,  "lat": 40.7334744,  "lon": -73.9876096},{  "type": "node",  "id": 5266421746,  "lat": 40.7330796,  "lon": -73.9866769},{  "type": "node",  "id": 5266421747,  "lat": 40.7329400,  "lon": -73.9874429},{  "type": "node",  "id": 5266421748,  "lat": 40.7336011,  "lon": -73.9869572},{  "type": "node",  "id": 5341393939,  "lat": 40.7296643,  "lon": -73.9837901},{  "type": "node",  "id": 5481911746,  "lat": 40.7337502,  "lon": -73.9868486},{  "type": "node",  "id": 5481912002,  "lat": 40.7313466,  "lon": -73.9847062},{  "type": "node",  "id": 5481937338,  "lat": 40.7285677,  "lon": -73.9894686},{  "type": "node",  "id": 5780009414,  "lat": 40.7299286,  "lon": -73.9887994},{  "type": "node",  "id": 6139477287,  "lat": 40.7318238,  "lon": -73.9853229},{  "type": "node",  "id": 42430914,  "lat": 40.7278460,  "lon": -73.9822280},{  "type": "node",  "id": 42431470,  "lat": 40.7320287,  "lon": -73.9820610},{  "type": "node",  "id": 42436943,  "lat": 40.7306534,  "lon": -73.9830644},{  "type": "node",  "id": 42436944,  "lat": 40.7297250,  "lon": -73.9808580},{  "type": "node",  "id": 42436947,  "lat": 40.7287460,  "lon": -73.9785360},{  "type": "node",  "id": 42437289,  "lat": 40.7338367,  "lon": -73.9807314},{  "type": "node",  "id": 42439006,  "lat": 40.7304280,  "lon": -73.9803529},{  "type": "node",  "id": 42439323,  "lat": 40.7326640,  "lon": -73.9816006},{  "type": "node",  "id": 42440743,  "lat": 40.7313565,  "lon": -73.9825570},{  "type": "node",  "id": 42446547,  "lat": 40.7300191,  "lon": -73.9835360},{  "type": "node",  "id": 42446552,  "lat": 40.7290860,  "lon": -73.9813250},{  "type": "node",  "id": 42446562,  "lat": 40.7281060,  "lon": -73.9789980},{  "type": "node",  "id": 42446889,  "lat": 40.7332587,  "lon": -73.9811531},{  "type": "node",  "id": 42454997,  "lat": 40.7284460,  "lon": -73.9817900},{  "type": "node",  "id": 42455007,  "lat": 40.7274680,  "lon": -73.9794690},{  "type": "node",  "id": 42455452,  "lat": 40.7333400,  "lon": -73.9831760},{  "type": "node",  "id": 685164631,  "lat": 40.7325725,  "lon": -73.9814350},{  "type": "node",  "id": 2384399611,  "lat": 40.7333132,  "lon": -73.9812830},{  "type": "node",  "id": 2562399783,  "lat": 40.7338950,  "lon": -73.9808693},{  "type": "node",  "id": 3834459526,  "lat": 40.7314023,  "lon": -73.9826672},{  "type": "node",  "id": 3834459527,  "lat": 40.7307023,  "lon": -73.9831802},{  "type": "node",  "id": 3834459538,  "lat": 40.7312695,  "lon": -73.9827652},{  "type": "node",  "id": 3834459540,  "lat": 40.7320791,  "lon": -73.9821801},{  "type": "node",  "id": 3834459551,  "lat": 40.7327235,  "lon": -73.9817084},{  "type": "node",  "id": 3875091629,  "lat": 40.7306848,  "lon": -73.9831959},{  "type": "node",  "id": 3875091630,  "lat": 40.7319894,  "lon": -73.9822477},{  "type": "node",  "id": 3875091631,  "lat": 40.7326200,  "lon": -73.9817825},{  "type": "node",  "id": 3875091632,  "lat": 40.7332881,  "lon": -73.9813040},{  "type": "node",  "id": 3875091633,  "lat": 40.7337986,  "lon": -73.9809397},{  "type": "node",  "id": 4161568333,  "lat": 40.7314846,  "lon": -73.9826080},{  "type": "node",  "id": 4161568334,  "lat": 40.7314362,  "lon": -73.9824982},{  "type": "node",  "id": 4161568336,  "lat": 40.7313046,  "lon": -73.9824336},{  "type": "node",  "id": 4161568338,  "lat": 40.7312597,  "lon": -73.9826271},{  "type": "node",  "id": 4161568341,  "lat": 40.7314198,  "lon": -73.9827093},{  "type": "node",  "id": 4191922368,  "lat": 40.7333185,  "lon": -73.9831270},{  "type": "node",  "id": 4191922374,  "lat": 40.7338752,  "lon": -73.9827877},{  "type": "node",  "id": 4205565489,  "lat": 40.7309397,  "lon": -73.9828579},{  "type": "node",  "id": 4207683792,  "lat": 40.7330064,  "lon": -73.9813427},{  "type": "node",  "id": 4216045847,  "lat": 40.7278933,  "lon": -73.9823396},{  "type": "node",  "id": 4298757018,  "lat": 40.7328419,  "lon": -73.9814664},{  "type": "node",  "id": 4298757020,  "lat": 40.7326038,  "lon": -73.9816441},{  "type": "node",  "id": 4298757022,  "lat": 40.7338365,  "lon": -73.9809108},{  "type": "node",  "id": 4298757025,  "lat": 40.7327377,  "lon": -73.9817426},{  "type": "node",  "id": 4298757028,  "lat": 40.7313042,  "lon": -73.9827375},{  "type": "node",  "id": 4298757033,  "lat": 40.7319771,  "lon": -73.9820991},{  "type": "node",  "id": 4298757042,  "lat": 40.7321457,  "lon": -73.9821311},{  "type": "node",  "id": 4298757045,  "lat": 40.7326527,  "lon": -73.9817582},{  "type": "node",  "id": 4298757047,  "lat": 40.7333283,  "lon": -73.9813190},{  "type": "node",  "id": 4298757054,  "lat": 40.7325998,  "lon": -73.9814948},{  "type": "node",  "id": 4298757057,  "lat": 40.7327912,  "lon": -73.9816574},{  "type": "node",  "id": 4298757059,  "lat": 40.7320956,  "lon": -73.9822191},{  "type": "node",  "id": 4298757061,  "lat": 40.7320959,  "lon": -73.9820123},{  "type": "node",  "id": 4298757064,  "lat": 40.7320275,  "lon": -73.9822173},{  "type": "node",  "id": 4298757071,  "lat": 40.7333635,  "lon": -73.9812475},{  "type": "node",  "id": 4298757075,  "lat": 40.7332499,  "lon": -73.9813285},{  "type": "node",  "id": 4298757077,  "lat": 40.7304719,  "lon": -73.9804570},{  "type": "node",  "id": 4298757080,  "lat": 40.7327271,  "lon": -73.9815525},{  "type": "node",  "id": 4298757083,  "lat": 40.7339069,  "lon": -73.9808960},{  "type": "node",  "id": 4502505659,  "lat": 40.7278864,  "lon": -73.9784903},{  "type": "node",  "id": 5147808597,  "lat": 40.7290659,  "lon": -73.9832671},{  "type": "node",  "id": 5147808598,  "lat": 40.7296811,  "lon": -73.9827351},{  "type": "node",  "id": 6133355307,  "lat": 40.7319026,  "lon": -73.9821541},{  "type": "node",  "id": 42428174,  "lat": 40.7378014,  "lon": -73.9901312},{  "type": "node",  "id": 42428179,  "lat": 40.7384888,  "lon": -73.9899514},{  "type": "node",  "id": 42428183,  "lat": 40.7391760,  "lon": -73.9897657},{  "type": "node",  "id": 42428192,  "lat": 40.7398521,  "lon": -73.9895840},{  "type": "node",  "id": 42431452,  "lat": 40.7353127,  "lon": -73.9898332},{  "type": "node",  "id": 42431453,  "lat": 40.7346549,  "lon": -73.9882789},{  "type": "node",  "id": 42431459,  "lat": 40.7339789,  "lon": -73.9866821},{  "type": "node",  "id": 42434352,  "lat": 40.7339693,  "lon": -73.9887747},{  "type": "node",  "id": 42434354,  "lat": 40.7352924,  "lon": -73.9878179},{  "type": "node",  "id": 42434355,  "lat": 40.7358767,  "lon": -73.9873953},{  "type": "node",  "id": 42434357,  "lat": 40.7364685,  "lon": -73.9869673},{  "type": "node",  "id": 42434358,  "lat": 40.7370531,  "lon": -73.9865444},{  "type": "node",  "id": 42434360,  "lat": 40.7376400,  "lon": -73.9861200},{  "type": "node",  "id": 42436582,  "lat": 40.7401675,  "lon": -73.9842643},{  "type": "node",  "id": 42437018,  "lat": 40.7359491,  "lon": -73.9893654},{  "type": "node",  "id": 42437021,  "lat": 40.7365253,  "lon": -73.9889418},{  "type": "node",  "id": 42437280,  "lat": 40.7371225,  "lon": -73.9885174},{  "type": "node",  "id": 42437283,  "lat": 40.7357911,  "lon": -73.9853620},{  "type": "node",  "id": 42437529,  "lat": 40.7373320,  "lon": -73.9853870},{  "type": "node",  "id": 42437530,  "lat": 40.7379130,  "lon": -73.9849697},{  "type": "node",  "id": 42438173,  "lat": 40.7385225,  "lon": -73.9864161},{  "type": "node",  "id": 42438174,  "lat": 40.7382130,  "lon": -73.9856817},{  "type": "node",  "id": 42438849,  "lat": 40.7379410,  "lon": -73.9868350},{  "type": "node",  "id": 42439170,  "lat": 40.7369623,  "lon": -73.9845088},{  "type": "node",  "id": 42439203,  "lat": 40.7382917,  "lon": -73.9876671},{  "type": "node",  "id": 42440323,  "lat": 40.7394943,  "lon": -73.9867925},{  "type": "node",  "id": 42440325,  "lat": 40.7388351,  "lon": -73.9852356},{  "type": "node",  "id": 42440326,  "lat": 40.7381579,  "lon": -73.9836380},{  "type": "node",  "id": 42440721,  "lat": 40.7344472,  "lon": -73.9898997},{  "type": "node",  "id": 42446875,  "lat": 40.7352035,  "lon": -73.9857900},{  "type": "node",  "id": 42446879,  "lat": 40.7345982,  "lon": -73.9843469},{  "type": "node",  "id": 42446925,  "lat": 40.7395037,  "lon": -73.9847586},{  "type": "node",  "id": 42448693,  "lat": 40.7401651,  "lon": -73.9863046},{  "type": "node",  "id": 42449890,  "lat": 40.7346191,  "lon": -73.9862156},{  "type": "node",  "id": 42449893,  "lat": 40.7363755,  "lon": -73.9849362},{  "type": "node",  "id": 42453116,  "lat": 40.7340968,  "lon": -73.9898386},{  "type": "node",  "id": 42453310,  "lat": 40.7388718,  "lon": -73.9872452},{  "type": "node",  "id": 42455465,  "lat": 40.7340120,  "lon": -73.9847730},{  "type": "node",  "id": 42456543,  "lat": 40.7377054,  "lon": -73.9880935},{  "type": "node",  "id": 42459668,  "lat": 40.7374979,  "lon": -73.9875998},{  "type": "node",  "id": 390519635,  "lat": 40.7375427,  "lon": -73.9840861},{  "type": "node",  "id": 595403438,  "lat": 40.7363966,  "lon": -73.9891959},{  "type": "node",  "id": 595403439,  "lat": 40.7351018,  "lon": -73.9899440},{  "type": "node",  "id": 1251830454,  "lat": 40.7382236,  "lon": -73.9866315},{  "type": "node",  "id": 1251830468,  "lat": 40.7376255,  "lon": -73.9851762},{  "type": "node",  "id": 1412706971,  "lat": 40.7366181,  "lon": -73.9892842},{  "type": "node",  "id": 3252590541,  "lat": 40.7364983,  "lon": -73.9888773},{  "type": "node",  "id": 3252590542,  "lat": 40.7364789,  "lon": -73.9889759},{  "type": "node",  "id": 3252590543,  "lat": 40.7366202,  "lon": -73.9891677},{  "type": "node",  "id": 3252590544,  "lat": 40.7365885,  "lon": -73.9888967},{  "type": "node",  "id": 3670742438,  "lat": 40.7365107,  "lon": -73.9892118},{  "type": "node",  "id": 3670742442,  "lat": 40.7352857,  "lon": -73.9897694},{  "type": "node",  "id": 3670742444,  "lat": 40.7359229,  "lon": -73.9893038},{  "type": "node",  "id": 3670742447,  "lat": 40.7358958,  "lon": -73.9894045},{  "type": "node",  "id": 3670742453,  "lat": 40.7353213,  "lon": -73.9900286},{  "type": "node",  "id": 3670742454,  "lat": 40.7343122,  "lon": -73.9898764},{  "type": "node",  "id": 3670742460,  "lat": 40.7343910,  "lon": -73.9897667},{  "type": "node",  "id": 3670742472,  "lat": 40.7353794,  "lon": -73.9897842},{  "type": "node",  "id": 3670742477,  "lat": 40.7345017,  "lon": -73.9900281},{  "type": "node",  "id": 3670742479,  "lat": 40.7360146,  "lon": -73.9893172},{  "type": "node",  "id": 3670742481,  "lat": 40.7345467,  "lon": -73.9899203},{  "type": "node",  "id": 3670742487,  "lat": 40.7365594,  "lon": -73.9892393},{  "type": "node",  "id": 3857400388,  "lat": 40.7399332,  "lon": -73.9895367},{  "type": "node",  "id": 3875000768,  "lat": 40.7377601,  "lon": -73.9900288},{  "type": "node",  "id": 3985008875,  "lat": 40.7339216,  "lon": -73.9867237},{  "type": "node",  "id": 3985008876,  "lat": 40.7339225,  "lon": -73.9865474},{  "type": "node",  "id": 3985008878,  "lat": 40.7340408,  "lon": -73.9868283},{  "type": "node",  "id": 3985008879,  "lat": 40.7340432,  "lon": -73.9866352},{  "type": "node",  "id": 4129804711,  "lat": 40.7343136,  "lon": -73.9836683},{  "type": "node",  "id": 4132605204,  "lat": 40.7369059,  "lon": -73.9843745},{  "type": "node",  "id": 4132605205,  "lat": 40.7369060,  "lon": -73.9845498},{  "type": "node",  "id": 4132605206,  "lat": 40.7370211,  "lon": -73.9844660},{  "type": "node",  "id": 4132605207,  "lat": 40.7370213,  "lon": -73.9846488},{  "type": "node",  "id": 4132605214,  "lat": 40.7374860,  "lon": -73.9839512},{  "type": "node",  "id": 4132605215,  "lat": 40.7374925,  "lon": -73.9841227},{  "type": "node",  "id": 4132605229,  "lat": 40.7376002,  "lon": -73.9842233},{  "type": "node",  "id": 4132605231,  "lat": 40.7376105,  "lon": -73.9840367},{  "type": "node",  "id": 4132605267,  "lat": 40.7381138,  "lon": -73.9836701},{  "type": "node",  "id": 4132605268,  "lat": 40.7382171,  "lon": -73.9837777},{  "type": "node",  "id": 4191917273,  "lat": 40.7339883,  "lon": -73.9847128},{  "type": "node",  "id": 4191917275,  "lat": 40.7339652,  "lon": -73.9848072},{  "type": "node",  "id": 4191917276,  "lat": 40.7340407,  "lon": -73.9848412},{  "type": "node",  "id": 4191922377,  "lat": 40.7345384,  "lon": -73.9843903},{  "type": "node",  "id": 4205565498,  "lat": 40.7349160,  "lon": -73.9899561},{  "type": "node",  "id": 4205565503,  "lat": 40.7374005,  "lon": -73.9883152},{  "type": "node",  "id": 4205565512,  "lat": 40.7403741,  "lon": -73.9867931},{  "type": "node",  "id": 4276517116,  "lat": 40.7398081,  "lon": -73.9894792},{  "type": "node",  "id": 4597668018,  "lat": 40.7352534,  "lon": -73.9898729},{  "type": "node",  "id": 4597668019,  "lat": 40.7363498,  "lon": -73.9890708},{  "type": "node",  "id": 4597668022,  "lat": 40.7359424,  "lon": -73.9895371},{  "type": "node",  "id": 4597668023,  "lat": 40.7360007,  "lon": -73.9894916},{  "type": "node",  "id": 4597668024,  "lat": 40.7360652,  "lon": -73.9894434},{  "type": "node",  "id": 4597668025,  "lat": 40.7365296,  "lon": -73.9890966},{  "type": "node",  "id": 4597668026,  "lat": 40.7371699,  "lon": -73.9886291},{  "type": "node",  "id": 4597668027,  "lat": 40.7366387,  "lon": -73.9890162},{  "type": "node",  "id": 4597668028,  "lat": 40.7383383,  "lon": -73.9877778},{  "type": "node",  "id": 4597668035,  "lat": 40.7353747,  "lon": -73.9899797},{  "type": "node",  "id": 4597668036,  "lat": 40.7365758,  "lon": -73.9890621},{  "type": "node",  "id": 4597668038,  "lat": 40.7377528,  "lon": -73.9882045},{  "type": "node",  "id": 4597668039,  "lat": 40.7389179,  "lon": -73.9873554},{  "type": "node",  "id": 4597668040,  "lat": 40.7395407,  "lon": -73.9869016},{  "type": "node",  "id": 4597668041,  "lat": 40.7402115,  "lon": -73.9864129},{  "type": "node",  "id": 4687262941,  "lat": 40.7398953,  "lon": -73.9896870},{  "type": "node",  "id": 4983147328,  "lat": 40.7349422,  "lon": -73.9899215},{  "type": "node",  "id": 4983147329,  "lat": 40.7349635,  "lon": -73.9899202},{  "type": "node",  "id": 4983147417,  "lat": 40.7353318,  "lon": -73.9899946},{  "type": "node",  "id": 4983147418,  "lat": 40.7353683,  "lon": -73.9899793},{  "type": "node",  "id": 4983147419,  "lat": 40.7350265,  "lon": -73.9900756},{  "type": "node",  "id": 4983147420,  "lat": 40.7351299,  "lon": -73.9900764},{  "type": "node",  "id": 4983147721,  "lat": 40.7347841,  "lon": -73.9900445},{  "type": "node",  "id": 4983147722,  "lat": 40.7348275,  "lon": -73.9900506},{  "type": "node",  "id": 4983147723,  "lat": 40.7346083,  "lon": -73.9900266},{  "type": "node",  "id": 4983147724,  "lat": 40.7346601,  "lon": -73.9900362},{  "type": "node",  "id": 4983147725,  "lat": 40.7344877,  "lon": -73.9899981},{  "type": "node",  "id": 4983147726,  "lat": 40.7345318,  "lon": -73.9900038},{  "type": "node",  "id": 4983147761,  "lat": 40.7343076,  "lon": -73.9898288},{  "type": "node",  "id": 4983147762,  "lat": 40.7343487,  "lon": -73.9898353},{  "type": "node",  "id": 4983147763,  "lat": 40.7345082,  "lon": -73.9898533},{  "type": "node",  "id": 4983147764,  "lat": 40.7345447,  "lon": -73.9898616},{  "type": "node",  "id": 4983147765,  "lat": 40.7345815,  "lon": -73.9898625},{  "type": "node",  "id": 4983147766,  "lat": 40.7346176,  "lon": -73.9898703},{  "type": "node",  "id": 4983147767,  "lat": 40.7347532,  "lon": -73.9898865},{  "type": "node",  "id": 4983147768,  "lat": 40.7347917,  "lon": -73.9898892},{  "type": "node",  "id": 4983147769,  "lat": 40.7348865,  "lon": -73.9899281},{  "type": "node",  "id": 4983147770,  "lat": 40.7349346,  "lon": -73.9899338},{  "type": "node",  "id": 4983147774,  "lat": 40.7349268,  "lon": -73.9899554},{  "type": "node",  "id": 4983147812,  "lat": 40.7351945,  "lon": -73.9899007},{  "type": "node",  "id": 4983147814,  "lat": 40.7352068,  "lon": -73.9898950},{  "type": "node",  "id": 4983147815,  "lat": 40.7342340,  "lon": -73.9896467},{  "type": "node",  "id": 4983147816,  "lat": 40.7342503,  "lon": -73.9896975},{  "type": "node",  "id": 4983147817,  "lat": 40.7341810,  "lon": -73.9895172},{  "type": "node",  "id": 4983147818,  "lat": 40.7342134,  "lon": -73.9896043},{  "type": "node",  "id": 4983147819,  "lat": 40.7350536,  "lon": -73.9897852},{  "type": "node",  "id": 4983147820,  "lat": 40.7351725,  "lon": -73.9898059},{  "type": "node",  "id": 5243716994,  "lat": 40.7340368,  "lon": -73.9889335},{  "type": "node",  "id": 5266429279,  "lat": 40.7396469,  "lon": -73.9850934},{  "type": "node",  "id": 5266429280,  "lat": 40.7399206,  "lon": -73.9857330},{  "type": "node",  "id": 5266429281,  "lat": 40.7404175,  "lon": -73.9861211},{  "type": "node",  "id": 5266429282,  "lat": 40.7370112,  "lon": -73.9885964},{  "type": "node",  "id": 5266429283,  "lat": 40.7357423,  "lon": -73.9895174},{  "type": "node",  "id": 5266429284,  "lat": 40.7358083,  "lon": -73.9894689},{  "type": "node",  "id": 5282642709,  "lat": 40.7355954,  "lon": -73.9848984},{  "type": "node",  "id": 5307048043,  "lat": 40.7391517,  "lon": -73.9870417},{  "type": "node",  "id": 5307048044,  "lat": 40.7383276,  "lon": -73.9876409},{  "type": "node",  "id": 5307048045,  "lat": 40.7354304,  "lon": -73.9899363},{  "type": "node",  "id": 5481897719,  "lat": 40.7394090,  "lon": -73.9885272},{  "type": "node",  "id": 5481911751,  "lat": 40.7341487,  "lon": -73.9870831},{  "type": "node",  "id": 6210354205,  "lat": 40.7398024,  "lon": -73.9895973},{  "type": "node",  "id": 42436586,  "lat": 40.7394963,  "lon": -73.9826645},{  "type": "node",  "id": 42437287,  "lat": 40.7348458,  "lon": -73.9831222},{  "type": "node",  "id": 42439178,  "lat": 40.7360200,  "lon": -73.9822670},{  "type": "node",  "id": 42440330,  "lat": 40.7372097,  "lon": -73.9814011},{  "type": "node",  "id": 42442877,  "lat": 40.7342630,  "lon": -73.9835476},{  "type": "node",  "id": 42442881,  "lat": 40.7354311,  "lon": -73.9826949},{  "type": "node",  "id": 42442889,  "lat": 40.7378801,  "lon": -73.9809087},{  "type": "node",  "id": 42446885,  "lat": 40.7339281,  "lon": -73.9827492},{  "type": "node",  "id": 42448701,  "lat": 40.7388264,  "lon": -73.9831480},{  "type": "node",  "id": 42455887,  "lat": 40.7344252,  "lon": -73.9803078},{  "type": "node",  "id": 387184869,  "lat": 40.7361899,  "lon": -73.9789954},{  "type": "node",  "id": 1692450029,  "lat": 40.7366007,  "lon": -73.9818452},{  "type": "node",  "id": 2384399452,  "lat": 40.7351213,  "lon": -73.9799588},{  "type": "node",  "id": 2384399483,  "lat": 40.7387322,  "lon": -73.9773089},{  "type": "node",  "id": 2384399497,  "lat": 40.7339557,  "lon": -73.9808262},{  "type": "node",  "id": 2384399590,  "lat": 40.7389109,  "lon": -73.9771943},{  "type": "node",  "id": 2384399605,  "lat": 40.7375964,  "lon": -73.9781533},{  "type": "node",  "id": 2562399784,  "lat": 40.7344781,  "lon": -73.9804333},{  "type": "node",  "id": 2562399786,  "lat": 40.7350685,  "lon": -73.9799963},{  "type": "node",  "id": 2562399787,  "lat": 40.7356379,  "lon": -73.9795703},{  "type": "node",  "id": 2562399788,  "lat": 40.7362496,  "lon": -73.9791362},{  "type": "node",  "id": 2562399792,  "lat": 40.7369274,  "lon": -73.9786429},{  "type": "node",  "id": 2562399793,  "lat": 40.7382150,  "lon": -73.9777054},{  "type": "node",  "id": 2704110928,  "lat": 40.7388333,  "lon": -73.9772375},{  "type": "node",  "id": 3601295719,  "lat": 40.7355913,  "lon": -73.9796068},{  "type": "node",  "id": 3601295749,  "lat": 40.7368378,  "lon": -73.9787078},{  "type": "node",  "id": 3601295751,  "lat": 40.7350135,  "lon": -73.9800414},{  "type": "node",  "id": 3601295752,  "lat": 40.7370345,  "lon": -73.9785605},{  "type": "node",  "id": 3686061895,  "lat": 40.7363260,  "lon": -73.9790808},{  "type": "node",  "id": 3686061904,  "lat": 40.7362105,  "lon": -73.9791630},{  "type": "node",  "id": 3686061909,  "lat": 40.7357053,  "lon": -73.9795213},{  "type": "node",  "id": 3686062439,  "lat": 40.7377908,  "lon": -73.9809727},{  "type": "node",  "id": 3817756020,  "lat": 40.7371736,  "lon": -73.9813161},{  "type": "node",  "id": 3875091634,  "lat": 40.7344099,  "lon": -73.9804903},{  "type": "node",  "id": 3875091635,  "lat": 40.7349755,  "lon": -73.9800697},{  "type": "node",  "id": 3875091636,  "lat": 40.7355442,  "lon": -73.9796466},{  "type": "node",  "id": 3875091637,  "lat": 40.7361921,  "lon": -73.9791764},{  "type": "node",  "id": 3875091638,  "lat": 40.7368242,  "lon": -73.9787194},{  "type": "node",  "id": 3875091639,  "lat": 40.7375367,  "lon": -73.9781979},{  "type": "node",  "id": 3875091640,  "lat": 40.7381582,  "lon": -73.9777480},{  "type": "node",  "id": 4115878740,  "lat": 40.7394461,  "lon": -73.9827009},{  "type": "node",  "id": 4115878741,  "lat": 40.7395546,  "lon": -73.9828013},{  "type": "node",  "id": 4129804701,  "lat": 40.7342066,  "lon": -73.9835885},{  "type": "node",  "id": 4129804702,  "lat": 40.7342124,  "lon": -73.9834271},{  "type": "node",  "id": 4129804713,  "lat": 40.7343230,  "lon": -73.9835037},{  "type": "node",  "id": 4129804729,  "lat": 40.7347882,  "lon": -73.9831642},{  "type": "node",  "id": 4129804730,  "lat": 40.7347936,  "lon": -73.9829987},{  "type": "node",  "id": 4129804731,  "lat": 40.7348972,  "lon": -73.9832441},{  "type": "node",  "id": 4129804732,  "lat": 40.7349102,  "lon": -73.9830751},{  "type": "node",  "id": 4132605266,  "lat": 40.7381019,  "lon": -73.9835060},{  "type": "node",  "id": 4132605269,  "lat": 40.7382287,  "lon": -73.9835860},{  "type": "node",  "id": 4132605270,  "lat": 40.7387381,  "lon": -73.9832127},{  "type": "node",  "id": 4132605271,  "lat": 40.7387700,  "lon": -73.9830146},{  "type": "node",  "id": 4132605272,  "lat": 40.7388876,  "lon": -73.9832927},{  "type": "node",  "id": 4132605273,  "lat": 40.7389372,  "lon": -73.9830681},{  "type": "node",  "id": 4132971831,  "lat": 40.7371655,  "lon": -73.9814332},{  "type": "node",  "id": 4132971841,  "lat": 40.7371591,  "lon": -73.9812817},{  "type": "node",  "id": 4132971872,  "lat": 40.7372646,  "lon": -73.9815307},{  "type": "node",  "id": 4132971876,  "lat": 40.7372843,  "lon": -73.9813461},{  "type": "node",  "id": 4205565497,  "lat": 40.7344830,  "lon": -73.9833869},{  "type": "node",  "id": 4205565499,  "lat": 40.7350789,  "lon": -73.9829520},{  "type": "node",  "id": 4205565501,  "lat": 40.7356745,  "lon": -73.9825181},{  "type": "node",  "id": 4205565502,  "lat": 40.7362668,  "lon": -73.9820877},{  "type": "node",  "id": 4244407497,  "lat": 40.7387709,  "lon": -73.9772808},{  "type": "node",  "id": 4296710218,  "lat": 40.7350866,  "lon": -73.9800394},{  "type": "node",  "id": 4296710222,  "lat": 40.7356576,  "lon": -73.9796167},{  "type": "node",  "id": 4296710225,  "lat": 40.7362633,  "lon": -73.9791684},{  "type": "node",  "id": 4298760164,  "lat": 40.7344920,  "lon": -73.9804663},{  "type": "node",  "id": 4298760170,  "lat": 40.7345370,  "lon": -73.9803899},{  "type": "node",  "id": 4298760175,  "lat": 40.7344272,  "lon": -73.9804735},{  "type": "node",  "id": 4376122114,  "lat": 40.7342290,  "lon": -73.9834666},{  "type": "node",  "id": 4376122115,  "lat": 40.7348108,  "lon": -73.9830392},{  "type": "node",  "id": 4376122116,  "lat": 40.7353955,  "lon": -73.9826104},{  "type": "node",  "id": 4376122117,  "lat": 40.7359854,  "lon": -73.9821843},{  "type": "node",  "id": 4376122118,  "lat": 40.7365666,  "lon": -73.9817647},{  "type": "node",  "id": 5266429277,  "lat": 40.7386135,  "lon": -73.9826443},{  "type": "node",  "id": 5266429278,  "lat": 40.7389846,  "lon": -73.9835236},{  "type": "node",  "id": 5690422189,  "lat": 40.7379411,  "lon": -73.9810531},{  "type": "node",  "id": 6139477286,  "lat": 40.7375963,  "lon": -73.9811161},{  "type": "node",  "id": 42427374,  "lat": 40.7422280,  "lon": -73.9970070},{  "type": "node",  "id": 42427381,  "lat": 40.7434266,  "lon": -73.9998517},{  "type": "node",  "id": 42427386,  "lat": 40.7446399,  "lon": -74.0027323},{  "type": "node",  "id": 42427427,  "lat": 40.7459612,  "lon": -73.9980000},{  "type": "node",  "id": 42429661,  "lat": 40.7410560,  "lon": -73.9978620},{  "type": "node",  "id": 42429662,  "lat": 40.7422581,  "lon": -74.0006951},{  "type": "node",  "id": 42434271,  "lat": 40.7446300,  "lon": -73.9989693},{  "type": "node",  "id": 42434279,  "lat": 40.7458555,  "lon": -74.0018592},{  "type": "node",  "id": 42435599,  "lat": 40.7410800,  "lon": -74.0015550},{  "type": "node",  "id": 42435603,  "lat": 40.7416736,  "lon": -74.0011226},{  "type": "node",  "id": 42435610,  "lat": 40.7428467,  "lon": -74.0002679},{  "type": "node",  "id": 42435624,  "lat": 40.7452932,  "lon": -73.9984812},{  "type": "node",  "id": 42437909,  "lat": 40.7465084,  "lon": -74.0013688},{  "type": "node",  "id": 42439275,  "lat": 40.7404700,  "lon": -73.9982870},{  "type": "node",  "id": 42439834,  "lat": 40.7416420,  "lon": -73.9974340},{  "type": "node",  "id": 42443674,  "lat": 40.7452231,  "lon": -74.0023040},{  "type": "node",  "id": 42444916,  "lat": 40.7440556,  "lon": -74.0031573},{  "type": "node",  "id": 2322155478,  "lat": 40.7457939,  "lon": -73.9981219},{  "type": "node",  "id": 4205565513,  "lat": 40.7405031,  "lon": -74.0021253},{  "type": "node",  "id": 4205565514,  "lat": 40.7413349,  "lon": -74.0013703},{  "type": "node",  "id": 4205565517,  "lat": 40.7417223,  "lon": -74.0012340},{  "type": "node",  "id": 4205565518,  "lat": 40.7424920,  "lon": -74.0005289},{  "type": "node",  "id": 4205565522,  "lat": 40.7428883,  "lon": -74.0003768},{  "type": "node",  "id": 4205565523,  "lat": 40.7436764,  "lon": -73.9996691},{  "type": "node",  "id": 4205565533,  "lat": 40.7449235,  "lon": -73.9987549},{  "type": "node",  "id": 4205565536,  "lat": 40.7453419,  "lon": -73.9985970},{  "type": "node",  "id": 4205565537,  "lat": 40.7463505,  "lon": -73.9977166},{  "type": "node",  "id": 4592029848,  "lat": 40.7463568,  "lon": -74.0013139},{  "type": "node",  "id": 4592029849,  "lat": 40.7457169,  "lon": -74.0017812},{  "type": "node",  "id": 4592029850,  "lat": 40.7451048,  "lon": -74.0022241},{  "type": "node",  "id": 4592029851,  "lat": 40.7445183,  "lon": -74.0026461},{  "type": "node",  "id": 4592029852,  "lat": 40.7439428,  "lon": -74.0030703},{  "type": "node",  "id": 4601466497,  "lat": 40.7440008,  "lon": -74.0030275},{  "type": "node",  "id": 4601466498,  "lat": 40.7445838,  "lon": -74.0025990},{  "type": "node",  "id": 4601466499,  "lat": 40.7451697,  "lon": -74.0021772},{  "type": "node",  "id": 4601466500,  "lat": 40.7457974,  "lon": -74.0017224},{  "type": "node",  "id": 4890040965,  "lat": 40.7457738,  "lon": -73.9996226},{  "type": "node",  "id": 5263790013,  "lat": 40.7465268,  "lon": -74.0001664},{  "type": "node",  "id": 5263790014,  "lat": 40.7462411,  "lon": -73.9995468},{  "type": "node",  "id": 5263790015,  "lat": 40.7461548,  "lon": -73.9993433},{  "type": "node",  "id": 5263790016,  "lat": 40.7460573,  "lon": -73.9984436},{  "type": "node",  "id": 5263790017,  "lat": 40.7460777,  "lon": -73.9991253},{  "type": "node",  "id": 5545401589,  "lat": 40.7464555,  "lon": -74.0012446},{  "type": "node",  "id": 5772387834,  "lat": 40.7468125,  "lon": -74.0004320},{  "type": "node",  "id": 5849918502,  "lat": 40.7440100,  "lon": -73.9994252},{  "type": "node",  "id": 5852207686,  "lat": 40.7469578,  "lon": -74.0005401},{  "type": "node",  "id": 5852207688,  "lat": 40.7466899,  "lon": -74.0003429},{  "type": "node",  "id": 5852207689,  "lat": 40.7464213,  "lon": -73.9999755},{  "type": "node",  "id": 5852207690,  "lat": 40.7461304,  "lon": -73.9992807},{  "type": "node",  "id": 5852207691,  "lat": 40.7460465,  "lon": -73.9989145},{  "type": "node",  "id": 5852207692,  "lat": 40.7466502,  "lon": -74.0003088},{  "type": "node",  "id": 5852207693,  "lat": 40.7470018,  "lon": -74.0005886},{  "type": "node",  "id": 5852207694,  "lat": 40.7460171,  "lon": -73.9981513},{  "type": "node",  "id": 5852207695,  "lat": 40.7460531,  "lon": -73.9983762},{  "type": "node",  "id": 5852207696,  "lat": 40.7460399,  "lon": -73.9982761},{  "type": "node",  "id": 5852207697,  "lat": 40.7460423,  "lon": -73.9988150},{  "type": "node",  "id": 5852207698,  "lat": 40.7464920,  "lon": -74.0001165},{  "type": "node",  "id": 5852207700,  "lat": 40.7460592,  "lon": -73.9990191},{  "type": "node",  "id": 5852207701,  "lat": 40.7466108,  "lon": -74.0002702},{  "type": "node",  "id": 5852207702,  "lat": 40.7464481,  "lon": -74.0000357},{  "type": "node",  "id": 5852207703,  "lat": 40.7465684,  "lon": -74.0002220},{  "type": "node",  "id": 5852207704,  "lat": 40.7460602,  "lon": -73.9985226},{  "type": "node",  "id": 5852207705,  "lat": 40.7460986,  "lon": -73.9991910},{  "type": "node",  "id": 5852207834,  "lat": 40.7460099,  "lon": -73.9981055},{  "type": "node",  "id": 6069996234,  "lat": 40.7469273,  "lon": -74.0005174},{  "type": "node",  "id": 6211484420,  "lat": 40.7423051,  "lon": -74.0008012},{  "type": "node",  "id": 6211484426,  "lat": 40.7466321,  "lon": -73.9976552},{  "type": "node",  "id": 6211484431,  "lat": 40.7411287,  "lon": -74.0016663},{  "type": "node",  "id": 6211484434,  "lat": 40.7440557,  "lon": -73.9995337},{  "type": "node",  "id": 6211484438,  "lat": 40.7434744,  "lon": -73.9999570},{  "type": "node",  "id": 6211484439,  "lat": 40.7446765,  "lon": -73.9990816},{  "type": "node",  "id": 6279091712,  "lat": 40.7453213,  "lon": -73.9985604},{  "type": "node",  "id": 6279091713,  "lat": 40.7453253,  "lon": -73.9985576},{  "type": "node",  "id": 6279091714,  "lat": 40.7450522,  "lon": -73.9987497},{  "type": "node",  "id": 6279091715,  "lat": 40.7450001,  "lon": -73.9987695},{  "type": "node",  "id": 42427371,  "lat": 40.7410310,  "lon": -73.9941650},{  "type": "node",  "id": 42427423,  "lat": 40.7435682,  "lon": -73.9923179},{  "type": "node",  "id": 42427426,  "lat": 40.7447650,  "lon": -73.9951595},{  "type": "node",  "id": 42430265,  "lat": 40.7416170,  "lon": -73.9937390},{  "type": "node",  "id": 42430269,  "lat": 40.7422330,  "lon": -73.9932920},{  "type": "node",  "id": 42430271,  "lat": 40.7429010,  "lon": -73.9928050},{  "type": "node",  "id": 42430274,  "lat": 40.7441860,  "lon": -73.9918710},{  "type": "node",  "id": 42430277,  "lat": 40.7448072,  "lon": -73.9914152},{  "type": "node",  "id": 42434270,  "lat": 40.7434310,  "lon": -73.9961290},{  "type": "node",  "id": 42434807,  "lat": 40.7466180,  "lon": -73.9938130},{  "type": "node",  "id": 42437890,  "lat": 40.7440990,  "lon": -73.9956450},{  "type": "node",  "id": 42439836,  "lat": 40.7428110,  "lon": -73.9965800},{  "type": "node",  "id": 42439840,  "lat": 40.7453830,  "lon": -73.9947090},{  "type": "node",  "id": 42439842,  "lat": 40.7460045,  "lon": -73.9942616},{  "type": "node",  "id": 3859048179,  "lat": 40.7423235,  "lon": -73.9914334},{  "type": "node",  "id": 4205565526,  "lat": 40.7439318,  "lon": -73.9952486},{  "type": "node",  "id": 4205565528,  "lat": 40.7440696,  "lon": -73.9955752},{  "type": "node",  "id": 4205565530,  "lat": 40.7442808,  "lon": -73.9960773},{  "type": "node",  "id": 4205565532,  "lat": 40.7444330,  "lon": -73.9954016},{  "type": "node",  "id": 5481850672,  "lat": 40.7445242,  "lon": -73.9945877},{  "type": "node",  "id": 5481850674,  "lat": 40.7445537,  "lon": -73.9946578},{  "type": "node",  "id": 42427477,  "lat": 40.7471747,  "lon": -74.0008813},{  "type": "node",  "id": 42427817,  "lat": 40.7502567,  "lon": -73.9986260},{  "type": "node",  "id": 42434821,  "lat": 40.7502203,  "lon": -74.0023198},{  "type": "node",  "id": 42437688,  "lat": 40.7509001,  "lon": -73.9981551},{  "type": "node",  "id": 42438672,  "lat": 40.7496507,  "lon": -73.9990694},{  "type": "node",  "id": 42444928,  "lat": 40.7477804,  "lon": -74.0004380},{  "type": "node",  "id": 42444933,  "lat": 40.7484089,  "lon": -73.9999781},{  "type": "node",  "id": 42445310,  "lat": 40.7495900,  "lon": -74.0027864},{  "type": "node",  "id": 42453630,  "lat": 40.7489740,  "lon": -74.0032360},{  "type": "node",  "id": 358448394,  "lat": 40.7510669,  "lon": -73.9980331},{  "type": "node",  "id": 3254626250,  "lat": 40.7489515,  "lon": -73.9995811},{  "type": "node",  "id": 3564892574,  "lat": 40.7513184,  "lon": -73.9976920},{  "type": "node",  "id": 4557517549,  "lat": 40.7515002,  "lon": -73.9977160},{  "type": "node",  "id": 4592028452,  "lat": 40.7508028,  "lon": -73.9980734},{  "type": "node",  "id": 4592028453,  "lat": 40.7501805,  "lon": -73.9985250},{  "type": "node",  "id": 4592028455,  "lat": 40.7483005,  "lon": -73.9998977},{  "type": "node",  "id": 4592029846,  "lat": 40.7476657,  "lon": -74.0003591},{  "type": "node",  "id": 4592029847,  "lat": 40.7470591,  "lon": -74.0008005},{  "type": "node",  "id": 4689208568,  "lat": 40.7488993,  "lon": -73.9994606},{  "type": "node",  "id": 4890040964,  "lat": 40.7490875,  "lon": -73.9994815},{  "type": "node",  "id": 5143758605,  "lat": 40.7484761,  "lon": -73.9999289},{  "type": "node",  "id": 5143758606,  "lat": 40.7495495,  "lon": -73.9989860},{  "type": "node",  "id": 5143758608,  "lat": 40.7484409,  "lon": -73.9997952},{  "type": "node",  "id": 5143758609,  "lat": 40.7495996,  "lon": -73.9991068},{  "type": "node",  "id": 5263790012,  "lat": 40.7470537,  "lon": -74.0006563},{  "type": "node",  "id": 5545401590,  "lat": 40.7471151,  "lon": -74.0007565},{  "type": "node",  "id": 5545401591,  "lat": 40.7477300,  "lon": -74.0003155},{  "type": "node",  "id": 5545401592,  "lat": 40.7483585,  "lon": -73.9998582},{  "type": "node",  "id": 5545401593,  "lat": 40.7496032,  "lon": -73.9989469},{  "type": "node",  "id": 5545401594,  "lat": 40.7502065,  "lon": -73.9985060},{  "type": "node",  "id": 5545401596,  "lat": 40.7508507,  "lon": -73.9980384},{  "type": "node",  "id": 5852207699,  "lat": 40.7470873,  "lon": -74.0007090},{  "type": "node",  "id": 5852207706,  "lat": 40.7471184,  "lon": -74.0007574},{  "type": "node",  "id": 6211484423,  "lat": 40.7472518,  "lon": -73.9972139},{  "type": "node",  "id": 42438549,  "lat": 40.7484849,  "lon": -73.9963147},{  "type": "node",  "id": 6211484421,  "lat": 40.7497215,  "lon": -73.9954064},{  "type": "node",  "id": 6211484424,  "lat": 40.7529040,  "lon": -73.9930842},{  "type": "node",  "id": 6211484427,  "lat": 40.7509255,  "lon": -73.9945260},{  "type": "node",  "id": 6211484428,  "lat": 40.7522399,  "lon": -73.9935663},{  "type": "node",  "id": 6211484432,  "lat": 40.7503402,  "lon": -73.9949559},{  "type": "node",  "id": 6211484433,  "lat": 40.7515619,  "lon": -73.9940585},{  "type": "node",  "id": 6211484435,  "lat": 40.7490985,  "lon": -73.9958600},{  "type": "node",  "id": 6211484436,  "lat": 40.7535237,  "lon": -73.9926344},{  "type": "node",  "id": 42428198,  "lat": 40.7405826,  "lon": -73.9893518},{  "type": "node",  "id": 42428201,  "lat": 40.7413644,  "lon": -73.9891371},{  "type": "node",  "id": 42428206,  "lat": 40.7422202,  "lon": -73.9891049},{  "type": "node",  "id": 42428212,  "lat": 40.7430463,  "lon": -73.9891667},{  "type": "node",  "id": 42434268,  "lat": 40.7408800,  "lon": -73.9900700},{  "type": "node",  "id": 42436575,  "lat": 40.7415369,  "lon": -73.9875133},{  "type": "node",  "id": 42436578,  "lat": 40.7408251,  "lon": -73.9858246},{  "type": "node",  "id": 42437881,  "lat": 40.7415497,  "lon": -73.9895958},{  "type": "node",  "id": 42445867,  "lat": 40.7408836,  "lon": -73.9879839},{  "type": "node",  "id": 42445879,  "lat": 40.7421387,  "lon": -73.9870584},{  "type": "node",  "id": 42456555,  "lat": 40.7414386,  "lon": -73.9853785},{  "type": "node",  "id": 274681226,  "lat": 40.7417966,  "lon": -73.9889850},{  "type": "node",  "id": 449581627,  "lat": 40.7423958,  "lon": -73.9889599},{  "type": "node",  "id": 449581964,  "lat": 40.7422455,  "lon": -73.9889504},{  "type": "node",  "id": 449581965,  "lat": 40.7420235,  "lon": -73.9889360},{  "type": "node",  "id": 2302133933,  "lat": 40.7407867,  "lon": -73.9838144},{  "type": "node",  "id": 3009534450,  "lat": 40.7414580,  "lon": -73.9896608},{  "type": "node",  "id": 3009534454,  "lat": 40.7414886,  "lon": -73.9890892},{  "type": "node",  "id": 3009534455,  "lat": 40.7414988,  "lon": -73.9894701},{  "type": "node",  "id": 3009534459,  "lat": 40.7416561,  "lon": -73.9895154},{  "type": "node",  "id": 3009534469,  "lat": 40.7422816,  "lon": -73.9890594},{  "type": "node",  "id": 3019038994,  "lat": 40.7412478,  "lon": -73.9891708},{  "type": "node",  "id": 3019039006,  "lat": 40.7416008,  "lon": -73.9897171},{  "type": "node",  "id": 3019039007,  "lat": 40.7413179,  "lon": -73.9890270},{  "type": "node",  "id": 3019042134,  "lat": 40.7421529,  "lon": -73.9889521},{  "type": "node",  "id": 3857400393,  "lat": 40.7406614,  "lon": -73.9893363},{  "type": "node",  "id": 3857400396,  "lat": 40.7426263,  "lon": -73.9892286},{  "type": "node",  "id": 3857400397,  "lat": 40.7406462,  "lon": -73.9895103},{  "type": "node",  "id": 3857400404,  "lat": 40.7405128,  "lon": -73.9893675},{  "type": "node",  "id": 3857400409,  "lat": 40.7405422,  "lon": -73.9892569},{  "type": "node",  "id": 3884612885,  "lat": 40.7413973,  "lon": -73.9892173},{  "type": "node",  "id": 4205565520,  "lat": 40.7425969,  "lon": -73.9888152},{  "type": "node",  "id": 4276517099,  "lat": 40.7409802,  "lon": -73.9879143},{  "type": "node",  "id": 4276517114,  "lat": 40.7409404,  "lon": -73.9881204},{  "type": "node",  "id": 4276517126,  "lat": 40.7408376,  "lon": -73.9878763},{  "type": "node",  "id": 4597668029,  "lat": 40.7408706,  "lon": -73.9859325},{  "type": "node",  "id": 4597668042,  "lat": 40.7414834,  "lon": -73.9854860},{  "type": "node",  "id": 4687262928,  "lat": 40.7406850,  "lon": -73.9896024},{  "type": "node",  "id": 4687262935,  "lat": 40.7406221,  "lon": -73.9894517},{  "type": "node",  "id": 4763900906,  "lat": 40.7410392,  "lon": -73.9883578},{  "type": "node",  "id": 4763900908,  "lat": 40.7411028,  "lon": -73.9885105},{  "type": "node",  "id": 4763900910,  "lat": 40.7411487,  "lon": -73.9886207},{  "type": "node",  "id": 4763900914,  "lat": 40.7411594,  "lon": -73.9886462},{  "type": "node",  "id": 4763900915,  "lat": 40.7412181,  "lon": -73.9887874},{  "type": "node",  "id": 5147972298,  "lat": 40.7429768,  "lon": -73.9891698},{  "type": "node",  "id": 5147973330,  "lat": 40.7424267,  "lon": -73.9890180},{  "type": "node",  "id": 5147973331,  "lat": 40.7423138,  "lon": -73.9893282},{  "type": "node",  "id": 5147973337,  "lat": 40.7425512,  "lon": -73.9892243},{  "type": "node",  "id": 5147973338,  "lat": 40.7424947,  "lon": -73.9891777},{  "type": "node",  "id": 5254263338,  "lat": 40.7422870,  "lon": -73.9889582},{  "type": "node",  "id": 5256074552,  "lat": 40.7429967,  "lon": -73.9891706},{  "type": "node",  "id": 6210354201,  "lat": 40.7424548,  "lon": -73.9890986},{  "type": "node",  "id": 6243559673,  "lat": 40.7419135,  "lon": -73.9893324},{  "type": "node",  "id": 6243559679,  "lat": 40.7420907,  "lon": -73.9889403},{  "type": "node",  "id": 42429890,  "lat": 40.7414058,  "lon": -73.9833636},{  "type": "node",  "id": 42436751,  "lat": 40.7432609,  "lon": -73.9820120},{  "type": "node",  "id": 42445374,  "lat": 40.7464020,  "lon": -73.9797240},{  "type": "node",  "id": 42445411,  "lat": 40.7438801,  "lon": -73.9815614},{  "type": "node",  "id": 42446270,  "lat": 40.7426439,  "lon": -73.9824628},{  "type": "node",  "id": 42446932,  "lat": 40.7420239,  "lon": -73.9829135},{  "type": "node",  "id": 42446933,  "lat": 40.7444982,  "lon": -73.9811089},{  "type": "node",  "id": 42446934,  "lat": 40.7451327,  "lon": -73.9806540},{  "type": "node",  "id": 42446935,  "lat": 40.7457355,  "lon": -73.9802079},{  "type": "node",  "id": 3997461340,  "lat": 40.7413528,  "lon": -73.9834022},{  "type": "node",  "id": 3997461341,  "lat": 40.7414699,  "lon": -73.9833169},{  "type": "node",  "id": 3997461342,  "lat": 40.7419741,  "lon": -73.9829498},{  "type": "node",  "id": 3997461343,  "lat": 40.7420916,  "lon": -73.9828643},{  "type": "node",  "id": 3997461346,  "lat": 40.7427071,  "lon": -73.9824166},{  "type": "node",  "id": 3997461347,  "lat": 40.7432086,  "lon": -73.9820502},{  "type": "node",  "id": 3997461348,  "lat": 40.7433231,  "lon": -73.9819667},{  "type": "node",  "id": 4234425034,  "lat": 40.7425920,  "lon": -73.9825005},{  "type": "node",  "id": 42434237,  "lat": 40.7230780,  "lon": -73.9764770},{  "type": "node",  "id": 42444835,  "lat": 40.7224980,  "lon": -73.9768930},{  "type": "node",  "id": 42446571,  "lat": 40.7271037,  "lon": -73.9766295},{  "type": "node",  "id": 2704111033,  "lat": 40.7394517,  "lon": -73.9767975},{  "type": "node",  "id": 2704112725,  "lat": 40.7400776,  "lon": -73.9763468},{  "type": "node",  "id": 3875091641,  "lat": 40.7393685,  "lon": -73.9768720},{  "type": "node",  "id": 3875091642,  "lat": 40.7400134,  "lon": -73.9763958},{  "type": "node",  "id": 5007384478,  "lat": 40.7395111,  "lon": -73.9767552},{  "type": "node",  "id": 5007384480,  "lat": 40.7394026,  "lon": -73.9768450},{  "type": "node",  "id": 2384399598,  "lat": 40.7406075,  "lon": -73.9759689},{  "type": "node",  "id": 2704112780,  "lat": 40.7406880,  "lon": -73.9759109},{  "type": "node",  "id": 2704114477,  "lat": 40.7413176,  "lon": -73.9754468},{  "type": "node",  "id": 3833447069,  "lat": 40.7438288,  "lon": -73.9736242},{  "type": "node",  "id": 3833447070,  "lat": 40.7431389,  "lon": -73.9741216},{  "type": "node",  "id": 3834459529,  "lat": 40.7457272,  "lon": -73.9722369},{  "type": "node",  "id": 3834459531,  "lat": 40.7451165,  "lon": -73.9726760},{  "type": "node",  "id": 3834459542,  "lat": 40.7444959,  "lon": -73.9731348},{  "type": "node",  "id": 3839412552,  "lat": 40.7422149,  "lon": -73.9747990},{  "type": "node",  "id": 3871072000,  "lat": 40.7427721,  "lon": -73.9743913},{  "type": "node",  "id": 3875091643,  "lat": 40.7412526,  "lon": -73.9754965},{  "type": "node",  "id": 3875091645,  "lat": 40.7430597,  "lon": -73.9741805},{  "type": "node",  "id": 3875091646,  "lat": 40.7436905,  "lon": -73.9737235},{  "type": "node",  "id": 3875091647,  "lat": 40.7443898,  "lon": -73.9732129},{  "type": "node",  "id": 3875091648,  "lat": 40.7450099,  "lon": -73.9727568},{  "type": "node",  "id": 3875091649,  "lat": 40.7456686,  "lon": -73.9722803},{  "type": "node",  "id": 3932847290,  "lat": 40.7437337,  "lon": -73.9736923},{  "type": "node",  "id": 3932847292,  "lat": 40.7439308,  "lon": -73.9735472},{  "type": "node",  "id": 3946500926,  "lat": 40.7444257,  "lon": -73.9731845},{  "type": "node",  "id": 3946500929,  "lat": 40.7445773,  "lon": -73.9730709},{  "type": "node",  "id": 3946500931,  "lat": 40.7450507,  "lon": -73.9727241},{  "type": "node",  "id": 3946500934,  "lat": 40.7451808,  "lon": -73.9726297},{  "type": "node",  "id": 42435654,  "lat": 40.7553300,  "lon": -73.9911770},{  "type": "node",  "id": 6211484422,  "lat": 40.7552727,  "lon": -73.9913647},{  "type": "node",  "id": 6211484425,  "lat": 40.7541431,  "lon": -73.9922022},{  "type": "node",  "id": 6211484430,  "lat": 40.7547558,  "lon": -73.9917480},{  "type": "way",  "id": 22914179,  "nodes": [    42434285,    42434279,    4601466500,    6211484439,    42434271,    42434270,    42430269,    42434268  ]},{  "type": "way",  "id": 229904064,  "nodes": [    42443513,    4241743151,    3834459541,    3834459525,    3834459528,    3875091620,    3834459537,    3875091621,    3834459532,    3875091622,    3834459544,    3875091623,    3834459535,    3875091624,    3834459546,    3875091625,    3834459550,    3875091626,    3834459530,    3875091627,    3834459533,    3875091628,    3834459543,    3875091629,    3834459527,    3834459538,    4298757028,    3834459526,    4161568333,    3875091630,    4298757064,    3834459540,    4298757042,    3875091631,    4298757045,    3834459551,    4298757057,    4298757075,    3875091632,    2384399611,    4298757071,    3875091633,    4298757022,    2562399783,    2384399497,    3875091634,    4298760175,    2562399784,    4298760170,    3875091635,    3601295751,    2562399786,    2384399452,    3875091636,    3601295719,    2562399787,    3686061909,    3875091637,    3686061904,    2562399788,    3686061895,    3875091638,    3601295749,    2562399792,    3601295752,    3875091639,    2384399605,    3875091640,    2562399793,    2384399483,    4244407497,    2704110928,    2384399590,    3875091641,    5007384480,    2704111033,    5007384478,    3875091642,    2704112725,    2384399598,    2704112780,    3875091643,    2704114477,    3839412552,    3871072000,    3875091645,    3833447070,    3875091646,    3932847290,    3833447069,    3932847292,    3875091647,    3946500926,    3834459542,    3946500929,    3875091648,    3946500931,    3834459531,    3946500934,    3875091649,    3834459529  ]},{  "type": "way",  "id": 396444294,  "nodes": [    42427483,    42427477,    5545401590,    5852207699,    5263790012,    5852207693,    5852207686,    6069996234,    5772387834,    5852207688,    5852207692,    5852207701,    5852207703,    5263790013,    5852207698,    5852207702,    5852207689,    5263790014,    5263790015,    5852207690,    5852207705,    5263790017,    5852207700,    5852207691,    5852207697,    5852207704,    5263790016,    5852207695,    5852207696,    5852207694,    5852207834,    42427427,    42427426,    5481850674,    5481850672,    42427423,    5147973331,    42428206  ]},{  "type": "way",  "id": 663629023,  "nodes": [    6211332483,    6211484437,    6211484429,    6213315378,    4205565508,    4205565513,    6211484431,    4205565517,    6211484420,    4205565522,    6211484438,    6211484434,    6211484439,    4205565536,    5852207834,    6211484426,    6211484423,    42438549,    6211484435,    6211484421,    6211484432,    6211484427,    6211484433,    6211484428,    6211484424,    6211484436,    6211484425,    6211484430,    6211484422,    42435654  ]},{  "type": "way",  "id": 22898269,  "nodes": [    4143851142,    4143859869,    4143859857,    4143859862,    4143859859,    4143851145,    4143851144,    42453777,    42447249,    42447246,    42440825  ]},{  "type": "way",  "id": 5669742,  "nodes": [    42428436,    4754117950,    42428434,    42428433,    42428431,    5481883674,    4754110661,    42428428,    4754110663,    4754110665,    42428425,    42428420,    4754110667,    42428417,    4207150078  ]},{  "type": "way",  "id": 5670193,  "nodes": [    42433581,    1919595926,    5829847195,    42433578,    42433577,    42432148,    42433574,    2151385472,    42433573  ]},{  "type": "way",  "id": 5670418,  "nodes": [    42436014,    42436017,    42436021,    3874650177,    42436024  ]},{  "type": "way",  "id": 5670728,  "nodes": [    42434839,    5804849353,    42438779,    4890553386,    42438781  ]},{  "type": "way",  "id": 5670792,  "nodes": [    42439335,    42439337,    42439340,    42439343,    42430164,    42439345,    42436364,    42436097  ]},{  "type": "way",  "id": 5670814,  "nodes": [    42430143,    4890320316,    4890320305,    42432127,    4890320289,    42439524,    42439527,    42439530,    5829844668,    1919595927,    42439533,    5829827215,    1692433935,    6213204056,    42439537,    3584752262,    42439542  ]},{  "type": "way",  "id": 5671291,  "nodes": [    42432142,    42436770,    4890320281,    42430143  ]},{  "type": "way",  "id": 22927411,  "nodes": [    246890279,    4207143421,    3874650174,    42438060,    42436069,    42436340,    4136500817,    4136500819,    4136500818,    4136500820,    246885162  ]},{  "type": "way",  "id": 22927463,  "nodes": [    42440873,    1701898699,    1701898709,    1701898682,    4142105795,    4142105818,    1701898701,    4142105792,    4872871121,    4142105812,    4892579966,    4142105797,    4207143424  ]},{  "type": "way",  "id": 22927486,  "nodes": [    42445174,    3874650180,    4142105827,    42445171,    42436073,    4142105805,    4142105824,    4142105799,    4142105810,    4892579967,    4142105811  ]},{  "type": "way",  "id": 25199870,  "nodes": [    42436788,    1919595918,    5829847188,    42436784,    42436779,    42432135,    42436770,    42436763  ]},{  "type": "way",  "id": 46139034,  "nodes": [    42448563,    42455200,    42439345,    3252582134,    42445574,    3252582131,    42442286,    4890333656,    4890333634,    42436567,    4890333636,    4890333654,    42424864  ]},{  "type": "way",  "id": 46139036,  "nodes": [    42430126,    4890621095,    2151385487,    42440829,    2151385500,    42452026  ]},{  "type": "way",  "id": 46139046,  "nodes": [    42430126,    42430131,    5155817221,    4890418814,    4143859865  ]},{  "type": "way",  "id": 46171876,  "nodes": [    42436083,    4892579962,    42436088,    42436093,    42436097,    42436100,    42436109,    42436115,    42436119,    42436122,    42436126,    42436129,    42431703,    42436134,    42436135,    42436139,    42436142,    42436146,    42436149  ]},{  "type": "way",  "id": 46332295,  "nodes": [    42452026,    4676469047,    42432142,    42442276,    42454423  ]},{  "type": "way",  "id": 46350690,  "nodes": [    42439335,    2477560535,    2477560540,    42430160  ]},{  "type": "way",  "id": 46350707,  "nodes": [    42442291,    42436088,    42442288,    42436355,    4560056040,    5173374393,    4616536351  ]},{  "type": "way",  "id": 68668546,  "nodes": [    42452026,    42433574,    42452040,    42449570,    42428436  ]},{  "type": "way",  "id": 194923790,  "nodes": [    42448568,    42455207,    42439352,    42442291  ]},{  "type": "way",  "id": 194923810,  "nodes": [    42430154,    4890320302,    5135800965,    42430160,    42430164,    5135896922,    5151708181,    42430167,    5151708184,    4321677346  ]},{  "type": "way",  "id": 348488662,  "nodes": [    42432142,    3546151525,    42432135,    4890320282,    42432127  ]},{  "type": "way",  "id": 412115641,  "nodes": [    42424408,    42440913,    42436064,    42440916,    3874650168,    42440918  ]},{  "type": "way",  "id": 412875705,  "nodes": [    246890098,    246882230,    4143812553,    246891302,    246858445  ]},{  "type": "way",  "id": 420809408,  "nodes": [    4207143427,    4629486881,    42440850  ]},{  "type": "way",  "id": 420809419,  "nodes": [    42436353,    42436355,    42436359,    42436364,    42436369,    42436371  ]},{  "type": "way",  "id": 420809448,  "nodes": [    246857084,    4141593654,    4141593653,    4141593655,    4142073863,    4207143435  ]},{  "type": "way",  "id": 438370108,  "nodes": [    42442294,    4141586737,    4141586733,    4141586735,    4207143436  ]},{  "type": "way",  "id": 439842837,  "nodes": [    42430154,    42442282,    42439527,    42436779,    42442276,    42433577  ]},{  "type": "way",  "id": 467222856,  "nodes": [    42428433,    42440820,    42440823,    42440825,    42433573,    2151385470,    2151385490,    42440829,    2151385497,    4143859873  ]},{  "type": "way",  "id": 607091411,  "nodes": [    4142105812,    5708239345,    42436073,    42436017,    42436069,    42436064,    5366406002  ]},{  "type": "way",  "id": 5671028,  "nodes": [    42442294,    5708245833,    42442291  ]},{  "type": "way",  "id": 22927462,  "nodes": [    42436313,    42440916,    42438060,    42436021,    42445171,    4872871134,    42440873  ]},{  "type": "way",  "id": 22927484,  "nodes": [    4207143432,    1701898730,    4141586734,    4141586731,    4141586736,    4142073862,    42440854  ]},{  "type": "way",  "id": 194923789,  "nodes": [    42442291,    3316533532,    4142105821,    4142105820,    42440850  ]},{  "type": "way",  "id": 313639527,  "nodes": [    3196459700,    3196459701  ]},{  "type": "way",  "id": 313639530,  "nodes": [    3196459695,    3196459708  ]},{  "type": "way",  "id": 313639531,  "nodes": [    246881559,    3196459709  ]},{  "type": "way",  "id": 313639532,  "nodes": [    3196459710,    3196459711  ]},{  "type": "way",  "id": 313639563,  "nodes": [    3196459867,    246882152  ]},{  "type": "way",  "id": 313639564,  "nodes": [    3196459868,    246882215  ]},{  "type": "way",  "id": 313639565,  "nodes": [    3196459869,    3196459870  ]},{  "type": "way",  "id": 371317543,  "nodes": [    3748942455,    3748942453,    3748942450,    3748942445,    3748942443,    3748942441,    3748942439,    3748942436,    3748942435,    3748942433,    3748942434,    3748942437,    3748942438,    3748942440,    3748942442,    3748942444  ]},{  "type": "way",  "id": 413050334,  "nodes": [    42440918,    246882204,    246882134,    4207143421,    42436024,    246882085,    42445174,    246881976,    246881640,    4207143428  ]},{  "type": "way",  "id": 420809410,  "nodes": [    246889572,    4207143428  ]},{  "type": "way",  "id": 420809412,  "nodes": [    42440850,    3748942456,    3748942459,    4207143432  ]},{  "type": "way",  "id": 420809415,  "nodes": [    4207143428,    3874650183,    3316533531,    42440873  ]},{  "type": "way",  "id": 420809416,  "nodes": [    42440854,    1701898705,    1701898669,    1701898687,    1701898668,    4142073864,    4142073861,    246889362  ]},{  "type": "way",  "id": 420809439,  "nodes": [    246889362,    246889572  ]},{  "type": "way",  "id": 420809442,  "nodes": [    4207143422,    246890279  ]},{  "type": "way",  "id": 420809443,  "nodes": [    4207143435,    4142073860,    246889362  ]},{  "type": "way",  "id": 420809446,  "nodes": [    246889572,    246881639,    246881975,    246882084,    4207143422  ]},{  "type": "way",  "id": 438370109,  "nodes": [    3748942454,    3748942448,    3748942447,    3748942446,    3748942449,    3748942451,    3748942452,    3748942457,    3748942460,    3748942461,    3748942462,    3748942463,    3748942464,    3748942465  ]},{  "type": "way",  "id": 496982371,  "nodes": [    4207143428,    42440854  ]},{  "type": "way",  "id": 516664206,  "nodes": [    42436316,    246882231,    42440918  ]},{  "type": "way",  "id": 516664207,  "nodes": [    246890279,    246882133,    246882203,    246890098  ]},{  "type": "way",  "id": 516664208,  "nodes": [    42440918,    246890098  ]},{  "type": "way",  "id": 588157330,  "nodes": [    42440854,    42442294  ]},{  "type": "way",  "id": 5669899,  "nodes": [    42430102,    4890320314,    4890320299,    592484776  ]},{  "type": "way",  "id": 5670462,  "nodes": [    42436567,    4890333643,    4890320315,    592484776  ]},{  "type": "way",  "id": 5670490,  "nodes": [    42436761,    4890633717,    42430136  ]},{  "type": "way",  "id": 5670816,  "nodes": [    42439551,    4890633694,    4890418800,    42439550  ]},{  "type": "way",  "id": 5670932,  "nodes": [    4778174564,    4890419521,    42438781,    4143859866,    2151385505,    4143859865  ]},{  "type": "way",  "id": 5671296,  "nodes": [    42438779,    4890621100,    42430131  ]},{  "type": "way",  "id": 5671327,  "nodes": [    42436093,    42436359  ]},{  "type": "way",  "id": 5672578,  "nodes": [    42442288,    4142105816,    4142105801,    4142105796,    4142105814,    42440847  ]},{  "type": "way",  "id": 22927399,  "nodes": [    246884508,    4136469551,    4136469557,    4136469559,    4136469556,    42424813,    4136469552,    4136469554,    4136469560,    4136469558,    4136469555,    4136469550,    3788379686,    3788379685,    3788379684,    42424408  ]},{  "type": "way",  "id": 22927419,  "nodes": [    246885162,    246886238,    246886237,    246886236,    1701898678,    1701898690,    1701898695,    4136476380,    1701898704,    1701898651,    4136469553,    1701898723,    1701898714,    246884508  ]},{  "type": "way",  "id": 46116390,  "nodes": [    42426060,    4602410644,    4602410642,    4139510041,    4890633719,    4602414021  ]},{  "type": "way",  "id": 46139033,  "nodes": [    42424864,    4890333653,    4890633720,    4616536349,    42423456  ]},{  "type": "way",  "id": 46139040,  "nodes": [    42424408,    42436340,    42436014,    5708229593,    4142105822,    42424619  ]},{  "type": "way",  "id": 46139041,  "nodes": [    42424619,    4602429319,    4602429329,    42432116  ]},{  "type": "way",  "id": 46350693,  "nodes": [    42430160,    3252582132,    42445574,    3252582133,    42436359  ]},{  "type": "way",  "id": 46350710,  "nodes": [    592484776,    4890320320,    4890320290,    42430108  ]},{  "type": "way",  "id": 95181773,  "nodes": [    1104165608,    42436308,    42424408  ]},{  "type": "way",  "id": 256902825,  "nodes": [    42430143,    4890320283,    3788364698,    3788364700,    4890633697,    3788364701,    42423456  ]},{  "type": "way",  "id": 256902826,  "nodes": [    42430143,    4890320280,    5216470312,    42430147,    4890320291,    5135800967,    42430151,    4890320293,    42430154  ]},{  "type": "way",  "id": 342785522,  "nodes": [    4778174564,    42424401,    3567807957  ]},{  "type": "way",  "id": 348488665,  "nodes": [    42432127,    5708230097,    42430147,    4890320319,    42430108,    3546151527,    3546151526,    4890333652,    42424864  ]},{  "type": "way",  "id": 412115639,  "nodes": [    2915066907,    562073774  ]},{  "type": "way",  "id": 412115640,  "nodes": [    2915066906,    562073788  ]},{  "type": "way",  "id": 412117872,  "nodes": [    42436083,    4142105812  ]},{  "type": "way",  "id": 412876925,  "nodes": [    4215665530,    4142105803,    4142105791,    42426089  ]},{  "type": "way",  "id": 420809409,  "nodes": [    4207143426,    4892579968,    42436083  ]},{  "type": "way",  "id": 420809411,  "nodes": [    4207143424,    4142105811,    4602429333,    4142105822  ]},{  "type": "way",  "id": 420809413,  "nodes": [    42436083,    4872871132,    3316533533,    4142105828,    4142105802,    4207143427  ]},{  "type": "way",  "id": 420809414,  "nodes": [    4142105822,    4602429320,    4629486879,    4602429335,    4142105817,    4142105819,    4142105809,    4142105813  ]},{  "type": "way",  "id": 420809417,  "nodes": [    42424619,    4602429322,    4142105806,    4142105823,    42440847,    4142105815,    4142105800,    4207143426  ]},{  "type": "way",  "id": 421832640,  "nodes": [    4142105813,    4215665530  ]},{  "type": "way",  "id": 421832641,  "nodes": [    42426089,    4602429325,    4602429327,    42424619  ]},{  "type": "way",  "id": 421832643,  "nodes": [    4215665529,    4778174561,    42440838  ]},{  "type": "way",  "id": 421832644,  "nodes": [    42423456,    4602414021,    4890633690,    4890633688,    42452015,    4890633695,    4215665529  ]},{  "type": "way",  "id": 438901427,  "nodes": [    3788379733,    2151385506,    42439550,    4778174560,    42440838  ]},{  "type": "way",  "id": 439933949,  "nodes": [    42436763,    4387367047,    2477507992,    42436761  ]},{  "type": "way",  "id": 466956271,  "nodes": [    42445574,    4616536350,    4616536351  ]},{  "type": "way",  "id": 466956272,  "nodes": [    4616536348,    4616536353,    4890333635,    5708233911,    4616536354,    42423456  ]},{  "type": "way",  "id": 466956273,  "nodes": [    4616536348,    4890333647,    4139510037  ]},{  "type": "way",  "id": 466956274,  "nodes": [    4616536352,    4890333638,    42436353  ]},{  "type": "way",  "id": 466956276,  "nodes": [    4616536351,    5173374390,    4890333632,    4616536352,    4890333642,    4890333641,    4616536348  ]},{  "type": "way",  "id": 466956278,  "nodes": [    4616536351,    42442286,    5173374392,    4890320296,    42430102,    4890320304,    42430154  ]},{  "type": "way",  "id": 468414628,  "nodes": [    42436014,    4629486875,    4629486876,    4629486877,    4629486878,    4629486879  ]},{  "type": "way",  "id": 485056260,  "nodes": [    42440838,    4778174559,    4890535524,    4778174564  ]},{  "type": "way",  "id": 485056261,  "nodes": [    4143859865,    3788379733  ]},{  "type": "way",  "id": 485056262,  "nodes": [    4143859865,    4778174563,    4143859873  ]},{  "type": "way",  "id": 485056263,  "nodes": [    4143859873,    2151385504,    3788379733  ]},{  "type": "way",  "id": 497191099,  "nodes": [    42432116,    42436353  ]},{  "type": "way",  "id": 539024459,  "nodes": [    42452015,    4890633707,    42439551,    4890633710,    5216470727,    4890633699,    3788379729,    3788379730,    42436763,    42452026  ]},{  "type": "way",  "id": 539024460,  "nodes": [    3788379733,    3788379732,    2151385507,    4136508184,    4136508181,    42430136,    5135824729,    5216470728,    5216470727,    5216470316,    5135824727,    4890633706,    42430143  ]},{  "type": "way",  "id": 653406301,  "nodes": [    42440838,    4778174562,    4142105813  ]},{  "type": "way",  "id": 5670092,  "nodes": [    1692433911,    5829779506,    42432308,    1918039901,    5829779507,    428628885,    428833476,    428833478,    3463814538,    42432305,    3463814535,    4501924445,    42432302,    42430034,    42432178,    42432282,    42432265,    42432260,    42432255,    42432250,    42432246,    42432244,    42432241,    42432238,    588455744,    1773055865  ]},{  "type": "way",  "id": 5670366,  "nodes": [    42435434,    42435435,    42435440,    42435448,    42435451,    42435456,    42435461,    42435465,    42435468,    42435470,    42432174,    42430038,    5159433792,    3463814539,    42435477,    3463814542,    428833477,    428833475,    42435485,    5829779493,    1918039869,    42435491,    5829779498,    1692433914  ]},{  "type": "way",  "id": 5670773,  "nodes": [    42432171,    5805243783,    42439207  ]},{  "type": "way",  "id": 5670820,  "nodes": [    42427322,    42439580,    42439581,    42428434,    42439583  ]},{  "type": "way",  "id": 68668539,  "nodes": [    42427316,    3316533507,    42427319,    42427322,    42427324  ]},{  "type": "way",  "id": 439842823,  "nodes": [    42427324,    42439207  ]},{  "type": "way",  "id": 505598707,  "nodes": [    591995273,    2052618391,    42428433  ]},{  "type": "way",  "id": 591772210,  "nodes": [    42440810,    42427316,    42440815,    591995262,    591995273  ]},{  "type": "way",  "id": 630828595,  "nodes": [    42439207,    42454175,    42454170,    42445612,    42440810  ]},{  "type": "way",  "id": 5669857,  "nodes": [    486867432,    5296795762,    486867405  ]},{  "type": "way",  "id": 5670079,  "nodes": [    42432085,    1773069829,    42432184  ]},{  "type": "way",  "id": 25198317,  "nodes": [    42432111,    42429644,    4300912711,    42432108  ]},{  "type": "way",  "id": 46139043,  "nodes": [    42429754,    486869282  ]},{  "type": "way",  "id": 165777194,  "nodes": [    1773076514,    1773076511  ]},{  "type": "way",  "id": 165777195,  "nodes": [    1773076513,    42432184,    42437589  ]},{  "type": "way",  "id": 420354140,  "nodes": [    4202930753,    4539951623,    4539951609  ]},{  "type": "way",  "id": 420354141,  "nodes": [    486869282,    5296795600,    4202930754  ]},{  "type": "way",  "id": 420354142,  "nodes": [    1773076511,    42429752  ]},{  "type": "way",  "id": 420354147,  "nodes": [    4202930754,    486869287,    5296795597,    1773076511  ]},{  "type": "way",  "id": 420355077,  "nodes": [    42429752,    4956305806,    1773076513  ]},{  "type": "way",  "id": 421827957,  "nodes": [    4215624277,    5296795599,    42429754  ]},{  "type": "way",  "id": 421827958,  "nodes": [    486867405,    5296795758,    5296795755,    3212472843  ]},{  "type": "way",  "id": 421827959,  "nodes": [    42429754,    4539951608  ]},{  "type": "way",  "id": 421827963,  "nodes": [    1773076511,    5296795598,    486867578,    4215624277  ]},{  "type": "way",  "id": 454031476,  "nodes": [    42452067,    6369613050,    42432085,    5159452349,    42429754  ]},{  "type": "way",  "id": 457937087,  "nodes": [    4539951609,    486869282  ]},{  "type": "way",  "id": 457937088,  "nodes": [    42432085,    4539951607  ]},{  "type": "way",  "id": 457937097,  "nodes": [    3120728297,    4539951620  ]},{  "type": "way",  "id": 485165667,  "nodes": [    4539951608,    4539951622,    5159452348,    6330061743,    486867432  ]},{  "type": "way",  "id": 496989303,  "nodes": [    1773076513,    1773076514  ]},{  "type": "way",  "id": 524395517,  "nodes": [    486868873,    5296795754,    5296795759,    42429645,    5296795761,    42429756,    6330061745,    5159452342,    4202930753  ]},{  "type": "way",  "id": 531510638,  "nodes": [    4539951607,    4539951617,    6330061754,    6330061746,    42432111  ]},{  "type": "way",  "id": 675967321,  "nodes": [    42437589,    6369613035,    42437580,    4202950101  ]},{  "type": "way",  "id": 675967322,  "nodes": [    42437589,    5805243784,    6369613048,    42452067  ]},{  "type": "way",  "id": 5669892,  "nodes": [    42429752,    5296795596,    42430030,    42430034,    42430038,    42430041,    5152337881,    5152337875,    42430044  ]},{  "type": "way",  "id": 5669978,  "nodes": [    42431168,    5824300186,    5481937333,    42431165,    42429676,    5824300190,    42428458  ]},{  "type": "way",  "id": 5670076,  "nodes": [    42432054,    3316523989,    4625620217,    42429647  ]},{  "type": "way",  "id": 5670084,  "nodes": [    42432184,    1773069828,    42432181,    42432178,    42432174,    42432171,    42427327  ]},{  "type": "way",  "id": 5671111,  "nodes": [    4478001045,    4598752850,    5783020900,    42440702,    4491355922,    4241743153,    42443513,    4241743162,    2672569868,    4776041124,    42442842,    42434919,    1773084410,    42437613  ]},{  "type": "way",  "id": 5671339,  "nodes": [    42445603,    6369613028,    5161246305,    42437580,    6369613034,    42445606,    42432250,    42435456,    42445612,    42427319  ]},{  "type": "way",  "id": 5671608,  "nodes": [    42448558,    5829907374,    5829880519,    42436788,    5829880525,    5829880523,    42439533,    5829880527,    5829880524,    42448549  ]},{  "type": "way",  "id": 5672138,  "nodes": [    42448558,    5829827214,    1692433927,    1692433928,    6213204038,    529556119,    42452964,    42440553  ]},{  "type": "way",  "id": 5672184,  "nodes": [    42453166,    3834459532,    5481937330,    3420766627,    4376122103,    42442850,    4577294519,    1773121034  ]},{  "type": "way",  "id": 5672269,  "nodes": [    1773084407,    1773084392,    42442843,    4776041123,    4886404633  ]},{  "type": "way",  "id": 5672871,  "nodes": [    42440823,    42452040,    42432152,    42442273,    42456197,    6024767164,    1919595917,    42448554  ]},{  "type": "way",  "id": 5673093,  "nodes": [    42428460,    42457697,    42445534,    5784518544,    5784518530,    1773084402,    42442848,    4376122102,    3834459537,    42455867  ]},{  "type": "way",  "id": 5673210,  "nodes": [    42439581,    42452048,    42432161,    42442255,    42458267,    42453327,    1918039880  ]},{  "type": "way",  "id": 5673316,  "nodes": [    1773078006,    42437608,    42458313,    3463814536,    42432305,    3463814537,    3463814540,    42435477,    3463814541,    42445498,    42458267,    42428444,    42449576,    42456197,    42433578,    1927715353,    42454423  ]},{  "type": "way",  "id": 25199883,  "nodes": [    42449589,    42433292,    42449582,    4117109189,    42449580,    4745685357,    42440545,    6213204049,    1692433920,    5829827213,    42448552  ]},{  "type": "way",  "id": 40709928,  "nodes": [    42451660,    6369613059,    42432111,    5159452347,    486867432,    42429756,    5159452352,    42437111,    42432054,    3316523990,    5783022067,    1918039896,    42451665  ]},{  "type": "way",  "id": 46139038,  "nodes": [    42430044,    5152337874,    42432165,    42445489,    42439580,    4202984493,    42440815  ]},{  "type": "way",  "id": 118400726,  "nodes": [    42442870,    3420766639,    42442862,    4236104525,    3420766638,    42430898,    3420766637,    3420766636,    42442857,    3420766635,    3420766634,    42434142,    3420766633,    3420766632,    42434201,    3420766631,    3420766630,    42442851,    4275822495,    42442850,    42442848,    42437333,    775996460,    42442843  ]},{  "type": "way",  "id": 125528659,  "nodes": [    42437339,    3834459528,    4376122101,    42437333,    1773084405,    42431168  ]},{  "type": "way",  "id": 127691872,  "nodes": [    42429637,    6369613316,    6369625010,    42429641,    6369625008,    5043087536,    42429643,    5043087532,    42429644,    5296795763,    486867405,    42429645,    5296795760,    42429646,    5277800703,    4690139671,    4591871814,    5277800701,    4690139672,    42429647,    5783022066,    1918039876  ]},{  "type": "way",  "id": 157012386,  "nodes": [    42430187,    5854302558,    3891950553,    2477560563,    5854302553,    1692433938,    1692433935,    1692433932,    1692433928,    1692433924,    3236571730,    1692433920,    5355267632,    1692433919,    1692433918,    1692433916,    1692433914,    5824300192,    1692433911,    42431822  ]},{  "type": "way",  "id": 181546837,  "nodes": [    3891963915,    4321748234,    4679523601,    5799117243,    3891963913,    4679523600,    4321748236,    5829880520,    1919595928,    5829880529,    5481866330,    5829880526,    1919595927,    5829880521,    5829880518,    1919595918,    5829880528,    5829907370,    5799117240,    5829907368,    1919595926,    5829907367,    1919595917,    5829907375,    5243716990,    5829907364,    1919595925  ]},{  "type": "way",  "id": 188559800,  "nodes": [    42438034,    1905965196,    42440804,    42445606,    1991894114,    42452056,    42432181,    42430030,    42442230,    4501924441,    42458313,    5829779496,    1918039898,    42457325,    5829779527,    42431822  ]},{  "type": "way",  "id": 194923786,  "nodes": [    42442247,    5159433792,    42432302,    42442230,    42437605,    1773076509  ]},{  "type": "way",  "id": 221687788,  "nodes": [    42437109,    3314293709,    42429646,    42437111  ]},{  "type": "way",  "id": 221687794,  "nodes": [    42437109,    3314293711,    5296792744,    5854381564,    486868873  ]},{  "type": "way",  "id": 276670360,  "nodes": [    1773121034,    6100764861,    1773121031,    1773084402  ]},{  "type": "way",  "id": 317400962,  "nodes": [    42433581,    5829827195,    4745829026,    5481866326,    5481866323,    1692433924  ]},{  "type": "way",  "id": 351540625,  "nodes": [    42449570,    42439583,    42440820  ]},{  "type": "way",  "id": 420352244,  "nodes": [    4202908171,    5829927767,    42428447  ]},{  "type": "way",  "id": 420352246,  "nodes": [    1919595915,    5829927764,    4755898715,    5829927762,    1918039880  ]},{  "type": "way",  "id": 420355078,  "nodes": [    4202950102,    5296795595,    42429752  ]},{  "type": "way",  "id": 420358890,  "nodes": [    42442247,    42445494  ]},{  "type": "way",  "id": 420358896,  "nodes": [    42431165,    5481937335,    42445534,    4202984497  ]},{  "type": "way",  "id": 421825693,  "nodes": [    4215580881,    5829827207,    42428447  ]},{  "type": "way",  "id": 439842831,  "nodes": [    42439207,    42435470,    42432282,    42452056,    5805243785,    42437589  ]},{  "type": "way",  "id": 439933948,  "nodes": [    1692433938,    5829827216,    42448549,    1919595928,    5829844674,    4387363208,    42450325,    42439335  ]},{  "type": "way",  "id": 452319507,  "nodes": [    5798966629,    4241743163,    5783028387,    42451665,    5783028393,    3927743131,    4776041130  ]},{  "type": "way",  "id": 454963832,  "nodes": [    486869282,    5159452344,    42437113,    42432083,    5991033039,    5783022072,    1918039877  ]},{  "type": "way",  "id": 457937084,  "nodes": [    4539951613,    5798966627,    5783028385,    1918039896,    5783028389,    1918039876,    4202908168  ]},{  "type": "way",  "id": 457937086,  "nodes": [    4539951609,    5159452350,    4539951610,    6336172705,    6336172706,    4539951611,    5991033038,    5783022064,    4539951613  ]},{  "type": "way",  "id": 464522037,  "nodes": [    42428473,    42428471,    4755898680,    42428468,    4755898682,    42428464,    42428460,    42428458,    4755898710,    42428454,    6213204045  ]},{  "type": "way",  "id": 479642039,  "nodes": [    42434205,    3834459535,    4376122105,    42434201,    5481937338,    42434197,    42434196  ]},{  "type": "way",  "id": 479642040,  "nodes": [    4015190382,    42443799,    3420766629,    42442851,    4376122104,    3420766628,    3834459544,    42444827  ]},{  "type": "way",  "id": 481403101,  "nodes": [    1692433932,    5829827193,    42436788  ]},{  "type": "way",  "id": 505772014,  "nodes": [    1773076511,    5296795594,    1773076509  ]},{  "type": "way",  "id": 584850103,  "nodes": [    42427324,    5297484683,    42427327  ]},{  "type": "way",  "id": 584850104,  "nodes": [    42428436,    4755831442,    42452048,    42445489,    42427324  ]},{  "type": "way",  "id": 5671322,  "nodes": [    42442247,    2154628339,    5152337880,    42430044  ]},{  "type": "way",  "id": 5671711,  "nodes": [    42448552,    1919595925  ]},{  "type": "way",  "id": 5672321,  "nodes": [    42427327,    5152337872,    42430041,    5152337873,    2154628332,    42442247  ]},{  "type": "way",  "id": 46348767,  "nodes": [    42432165,    42432161,    4755831438,    42428438,    42432156,    42432152,    42432148,    42432142  ]},{  "type": "way",  "id": 68668516,  "nodes": [    42427327,    42432165  ]},{  "type": "way",  "id": 321993366,  "nodes": [    42454423,    5829844672,    5799117240,    42448558  ]},{  "type": "way",  "id": 321993367,  "nodes": [    42454423,    42436784,    42439530,    42439335  ]},{  "type": "way",  "id": 351540632,  "nodes": [    42449576,    42442269,    42432156,    42449570  ]},{  "type": "way",  "id": 420352247,  "nodes": [    4202908173,    5829907369,    42448558  ]},{  "type": "way",  "id": 420352248,  "nodes": [    4202908172,    5829907371,    42448552  ]},{  "type": "way",  "id": 420352249,  "nodes": [    42428447,    5829927772,    3594715134,    4202908172  ]},{  "type": "way",  "id": 420352250,  "nodes": [    42448552,    5829907372,    5829907363,    5799117244,    42448554,    5829907366,    42433581,    4202908173  ]},{  "type": "way",  "id": 439842835,  "nodes": [    42428441,    3573482109,    42442255,    42442247  ]},{  "type": "way",  "id": 439842836,  "nodes": [    42433577,    42442273,    42442269,    3573482111,    42428441  ]},{  "type": "way",  "id": 496980200,  "nodes": [    1919595925,    5829907365,    1919595919,    5829927763,    5799037570,    1919595915  ]},{  "type": "way",  "id": 496980201,  "nodes": [    1919595925,    5829847185,    42449576  ]},{  "type": "way",  "id": 584850105,  "nodes": [    42428447,    1919595915,    5829844675,    4755898719,    42428444,    4755831435,    3573482110,    42428441,    3573482108,    4755831437,    42428438,    4755831440,    42428436  ]},{  "type": "way",  "id": 5670077,  "nodes": [    42432082,    1773082408  ]},{  "type": "way",  "id": 5670594,  "nodes": [    1773084402,    5286255072,    42431168,    5243701466,    42437618,    1692433907  ]},{  "type": "way",  "id": 5671083,  "nodes": [    42442838,    1918039935,    3927743126,    42443127  ]},{  "type": "way",  "id": 5672214,  "nodes": [    42435485,    42445505,    42453327  ]},{  "type": "way",  "id": 5672625,  "nodes": [    42428454,    42429672,    42445520  ]},{  "type": "way",  "id": 5673047,  "nodes": [    4892845897,    5829779492,    1773082410  ]},{  "type": "way",  "id": 39250165,  "nodes": [    42437111,    6330061741,    6330061770,    6330061766,    4539951610,    42437113  ]},{  "type": "way",  "id": 40709918,  "nodes": [    42437113,    42431712,    1773076778  ]},{  "type": "way",  "id": 40709920,  "nodes": [    42432083,    4539951611,    6330061765,    6330061769,    6330061740,    42432054  ]},{  "type": "way",  "id": 40709921,  "nodes": [    1918039864,    5829779512,    42432082  ]},{  "type": "way",  "id": 40709927,  "nodes": [    42445520,    1773084413,    42437618  ]},{  "type": "way",  "id": 112116234,  "nodes": [    42437114,    42437605,    4202950102  ]},{  "type": "way",  "id": 112116236,  "nodes": [    42437608,    42437114  ]},{  "type": "way",  "id": 118400728,  "nodes": [    42442842,    42442838  ]},{  "type": "way",  "id": 165777255,  "nodes": [    1773076778,    1773078006  ]},{  "type": "way",  "id": 165777635,  "nodes": [    4202950103,    5829779494,    1918039904  ]},{  "type": "way",  "id": 165777951,  "nodes": [    1773084407,    1773084405,    5286255073,    1773084402  ]},{  "type": "way",  "id": 181375404,  "nodes": [    1918039897,    5829779530,    5799037566,    5829779504,    1918039869,    5829779515,    5829779489,    1918039901,    5829779516,    5829779529,    1918039898,    5829779490,    4202908169  ]},{  "type": "way",  "id": 195743157,  "nodes": [    42443127,    1918039877  ]},{  "type": "way",  "id": 195743178,  "nodes": [    1918039877,    4539951613  ]},{  "type": "way",  "id": 195743318,  "nodes": [    42442838,    5829779486,    42442836  ]},{  "type": "way",  "id": 349873701,  "nodes": [    42442843,    4202957549  ]},{  "type": "way",  "id": 349873702,  "nodes": [    1692433907,    42437613,    4202950104  ]},{  "type": "way",  "id": 420352241,  "nodes": [    4202908169,    5829779511,    1918039864  ]},{  "type": "way",  "id": 420352243,  "nodes": [    4202908170,    5829927768,    5829927770,    1918039897  ]},{  "type": "way",  "id": 420352245,  "nodes": [    1918039864,    1918039904  ]},{  "type": "way",  "id": 420355076,  "nodes": [    4202950104,    5829779531,    42437612  ]},{  "type": "way",  "id": 420355079,  "nodes": [    1918039904,    1773082410  ]},{  "type": "way",  "id": 420355080,  "nodes": [    1773078006,    1773082408  ]},{  "type": "way",  "id": 420356798,  "nodes": [    4202957549,    42442842  ]},{  "type": "way",  "id": 420358889,  "nodes": [    4202984494,    42445505  ]},{  "type": "way",  "id": 420358891,  "nodes": [    42445505,    4202984495  ]},{  "type": "way",  "id": 420358892,  "nodes": [    4202984496,    42431165  ]},{  "type": "way",  "id": 420358893,  "nodes": [    42445494,    42445498  ]},{  "type": "way",  "id": 420358894,  "nodes": [    4202984495,    5829753046,    1918039897  ]},{  "type": "way",  "id": 420358895,  "nodes": [    42445498,    4202984494  ]},{  "type": "way",  "id": 420358898,  "nodes": [    1918039897,    42445511,    5829753047,    1692433915,    1692433916,    6213204060,    1692433917,    42421927,    42445520  ]},{  "type": "way",  "id": 439842751,  "nodes": [    1773082408,    4202950103  ]},{  "type": "way",  "id": 439842754,  "nodes": [    1773076509,    5799559461,    1773076778  ]},{  "type": "way",  "id": 439842757,  "nodes": [    1773084391,    5243701467,    1773084407  ]},{  "type": "way",  "id": 439842761,  "nodes": [    1773082410,    5829779518,    1773084410,    1773084391  ]},{  "type": "way",  "id": 439842812,  "nodes": [    42437618,    1773084407  ]},{  "type": "way",  "id": 439842834,  "nodes": [    1773076778,    42437114  ]},{  "type": "way",  "id": 439933947,  "nodes": [    42445520,    4202984496  ]},{  "type": "way",  "id": 457937085,  "nodes": [    4776041130,    4539951613  ]},{  "type": "way",  "id": 497769952,  "nodes": [    42442836,    5798966628,    4892845897  ]},{  "type": "way",  "id": 497769953,  "nodes": [    42443127,    3927743127,    42442836  ]},{  "type": "way",  "id": 524387550,  "nodes": [    1918039904,    5829779519,    494509182,    3927743135,    1918039877  ]},{  "type": "way",  "id": 524387551,  "nodes": [    42457325,    5829779510,    5829779491,    42432308,    5829779501,    5829779509,    42435491,    5829779524,    5829779533,    42445511,    5829927765,    5799037574,    5829927769,    42457319  ]},{  "type": "way",  "id": 524388531,  "nodes": [    42457319,    5829827197,    5799463409,    1692433918  ]},{  "type": "way",  "id": 524388532,  "nodes": [    42457319,    5829927766,    4202908171  ]},{  "type": "way",  "id": 588293370,  "nodes": [    1773082410,    42437612,    5829779485,    5829779497,    42457325  ]},{  "type": "way",  "id": 588293375,  "nodes": [    1918039880,    42457319  ]},{  "type": "way",  "id": 634986990,  "nodes": [    5991033038,    5991033039  ]},{  "type": "way",  "id": 634986991,  "nodes": [    4776041130,    42443127  ]},{  "type": "way",  "id": 653428657,  "nodes": [    1692433919,    4755898713,    4215580881  ]},{  "type": "way",  "id": 663797741,  "nodes": [    42431822,    1692433907,    1773084391  ]},{  "type": "way",  "id": 663862355,  "nodes": [    6213204045,    1692433919  ]},{  "type": "way",  "id": 681061718,  "nodes": [    42437612,    1918039864  ]},{  "type": "way",  "id": 681061719,  "nodes": [    1773082408,    42432083  ]},{  "type": "way",  "id": 681061720,  "nodes": [    42432082,    42437608  ]},{  "type": "way",  "id": 681064391,  "nodes": [    1918039880,    4202908170  ]},{  "type": "way",  "id": 221687789,  "nodes": [    42437108,    2349482103,    42437109  ]},{  "type": "way",  "id": 25163419,  "nodes": [    592509222,    3783073165,    3783073245,    3783071256,    3783073197,    4219836772,    42453610,    3783146167,    42453607,    42453604,    4505948472,    3291987532,    4886404634  ]},{  "type": "way",  "id": 46582968,  "nodes": [    42434237,    42434233,    4216045844,    42434226,    4216045822,    274306324,    4216055441,    42434215,    4216045795,    42434205  ]},{  "type": "way",  "id": 46608205,  "nodes": [    42444827,    5481937325,    6031953403,    6031953405,    3316524005,    42444829,    42444832,    5575804213,    42444833,    42444835  ]},{  "type": "way",  "id": 195743130,  "nodes": [    4557539295,    3212472830,    6139477283,    3314285365,    5854382586,    42432057,    5854381578,    3314293710,    42437109  ]},{  "type": "way",  "id": 221687792,  "nodes": [    42437108,    3314293714,    5854381570,    42432060,    5854381562,    6162064820,    3531834102,    4376122099  ]},{  "type": "way",  "id": 479642033,  "nodes": [    42446547,    5147808598,    42446552,    42446562,    4502505659,    42446571  ]},{  "type": "way",  "id": 479642035,  "nodes": [    42455013,    42455007,    42454997,    5147808597,    42454994  ]},{  "type": "way",  "id": 479642037,  "nodes": [    42449023,    4216045804,    42449021  ]},{  "type": "way",  "id": 479642038,  "nodes": [    42434145,    4216045849,    42434148  ]},{  "type": "way",  "id": 479642042,  "nodes": [    42455867,    6031953401,    3316523995,    42455051,    42455083,    42455175,    42455396  ]},{  "type": "way",  "id": 479642043,  "nodes": [    42437349,    42437346,    42437343,    42437339  ]},{  "type": "way",  "id": 654560917,  "nodes": [    42453166,    42444827,    42434205,    42434145,    42449021,    42430903,    42454994,    5341393939,    42446547,    42436943,    4205565489  ]},{  "type": "way",  "id": 68325065,  "nodes": [    42455039,    4598752856,    5783020909,    4478001045,    5783020907,    4598752834,    42456331,    4598752835,    4202899501  ]},{  "type": "way",  "id": 181546839,  "nodes": [    805771586,    4241743161,    4598752833,    1919595914  ]},{  "type": "way",  "id": 195743177,  "nodes": [    4557539295,    805771586  ]},{  "type": "way",  "id": 226041018,  "nodes": [    42432057,    5854325121,    42432060  ]},{  "type": "way",  "id": 380098432,  "nodes": [    42453601,    4968494267,    6133355298  ]},{  "type": "way",  "id": 420352240,  "nodes": [    4202908168,    4241743156,    4557539295  ]},{  "type": "way",  "id": 420352242,  "nodes": [    4202899501,    4241743152,    5798966629  ]},{  "type": "way",  "id": 439842808,  "nodes": [    3834459525,    42453601  ]},{  "type": "way",  "id": 439842838,  "nodes": [    42429647,    4625620218,    3314293704,    42432057  ]},{  "type": "way",  "id": 439933936,  "nodes": [    4376122099,    3212472829,    805771586  ]},{  "type": "way",  "id": 452319508,  "nodes": [    4598752858,    4598752838,    4491355922  ]},{  "type": "way",  "id": 474054794,  "nodes": [    1919595914,    4598752841,    5783020902,    1919595911,    5783020910,    4598752866,    1919595929  ]},{  "type": "way",  "id": 474054795,  "nodes": [    1919595914,    4332491210,    42456331  ]},{  "type": "way",  "id": 496986996,  "nodes": [    4886266066,    4598752849,    1919595914  ]},{  "type": "way",  "id": 496986997,  "nodes": [    42432063,    3314293705,    4886266066  ]},{  "type": "way",  "id": 496986998,  "nodes": [    42432060,    3314293703,    42432063  ]},{  "type": "way",  "id": 497002542,  "nodes": [    4886404633,    4968494268,    3834459525  ]},{  "type": "way",  "id": 497002543,  "nodes": [    4886404634,    4968494269,    42453601  ]},{  "type": "way",  "id": 497002544,  "nodes": [    42443513,    4241743158,    6133388359,    4968494266,    42453601  ]},{  "type": "way",  "id": 517829857,  "nodes": [    805771586,    5798966629  ]},{  "type": "way",  "id": 620939495,  "nodes": [    5798966629,    4241748518,    4241743160,    42443513  ]},{  "type": "way",  "id": 654560919,  "nodes": [    42437339,    42455867,    5341393940,    6133355297  ]},{  "type": "way",  "id": 654560924,  "nodes": [    6133355298,    5341393941,    42437339  ]},{  "type": "way",  "id": 654560928,  "nodes": [    6133355297,    42453166  ]},{  "type": "way",  "id": 5670896,  "nodes": [    246890579,    42440532,    3874650186  ]},{  "type": "way",  "id": 46431827,  "nodes": [    593816735,    4379778905,    4379778906,    4379778907,    4379778908,    4379778909,    246890579  ]},{  "type": "way",  "id": 46431828,  "nodes": [    591454081,    4629488490  ]},{  "type": "way",  "id": 170113137,  "nodes": [    1812608108,    4629488489,    42448570  ]},{  "type": "way",  "id": 195743557,  "nodes": [    42436903,    4872777176,    42453457  ]},{  "type": "way",  "id": 420809420,  "nodes": [    42448570,    4629486887,    246878876,    42440532,    42453442,    246877762,    591454081,    246876730,    42452008,    5708228060,    42436903  ]},{  "type": "way",  "id": 420809434,  "nodes": [    4207143439,    246890579  ]},{  "type": "way",  "id": 420809438,  "nodes": [    4207143437,    4629486886,    246857084  ]},{  "type": "way",  "id": 420809441,  "nodes": [    4207143436,    42448570  ]},{  "type": "way",  "id": 420809445,  "nodes": [    246890579,    246878875,    4207143437  ]},{  "type": "way",  "id": 682850452,  "nodes": [    42448570,    246857084  ]},{  "type": "way",  "id": 5669993,  "nodes": [    42430796,    42431273,    42431274,    4816151666,    4816151671,    4816151669,    4816151670,    4816151668,    4816151667,    42431275  ]},{  "type": "way",  "id": 5672510,  "nodes": [    42436381,    42454635,    42431528,    42430791  ]},{  "type": "way",  "id": 5672912,  "nodes": [    42456464,    4685597889,    2073622011,    42436377  ]},{  "type": "way",  "id": 5672988,  "nodes": [    42430770,    42456464  ]},{  "type": "way",  "id": 5673342,  "nodes": [    42430761,    5837116452,    42436968,    42448563  ]},{  "type": "way",  "id": 22927287,  "nodes": [    42436371,    42436109,    42448568  ]},{  "type": "way",  "id": 46610726,  "nodes": [    5837088105,    5837101292,    5837101293,    4320035374,    42431531,    4816151662,    4816151665,    4816151664,    4816151663,    42431275,    42452001,    42436384  ]},{  "type": "way",  "id": 46610727,  "nodes": [    5837088105,    5151983467,    42430796,    42430791,    42430782  ]},{  "type": "way",  "id": 68325062,  "nodes": [    42448563,    42436371  ]},{  "type": "way",  "id": 116672642,  "nodes": [    42436374,    2073622000,    5837109293,    1319476957,    1319476955,    42430761  ]},{  "type": "way",  "id": 194923796,  "nodes": [    42448568,    1812608107  ]},{  "type": "way",  "id": 239988110,  "nodes": [    42430782,    42431528,    42431273,    42431531,    42431536,    339044224  ]},{  "type": "way",  "id": 420530113,  "nodes": [    42430782,    42430770,    5837116451,    42430761  ]},{  "type": "way",  "id": 420530114,  "nodes": [    42431515,    42431517,    42431519,    5866337890,    42431524,    42430782  ]},{  "type": "way",  "id": 420895278,  "nodes": [    4207862996,    42436393  ]},{  "type": "way",  "id": 420895282,  "nodes": [    42436377,    42436381,    4207862995  ]},{  "type": "way",  "id": 439842799,  "nodes": [    42436129,    4578212490,    42436894  ]},{  "type": "way",  "id": 439933945,  "nodes": [    42436371,    42436374,    6210448866  ]},{  "type": "way",  "id": 468414634,  "nodes": [    42454642,    42436122,    6210448867,    42436381  ]},{  "type": "way",  "id": 480112482,  "nodes": [    42436393,    6210448840,    5646503062,    4578212489,    42436129  ]},{  "type": "way",  "id": 480112484,  "nodes": [    42436115,    6210448853,    42436374  ]},{  "type": "way",  "id": 492148702,  "nodes": [    4841945993,    4841945994,    4841945995  ]},{  "type": "way",  "id": 492148703,  "nodes": [    42431802,    42431797,    4679523599,    5837088105  ]},{  "type": "way",  "id": 492148704,  "nodes": [    42431802,    4841945995,    339044224,    42436393  ]},{  "type": "way",  "id": 555245141,  "nodes": [    42431706,    5357282419  ]},{  "type": "way",  "id": 663509701,  "nodes": [    6210448866,    6210448861,    6210448864  ]},{  "type": "way",  "id": 663509704,  "nodes": [    6210448866,    42436377  ]},{  "type": "way",  "id": 663509710,  "nodes": [    6210448864,    6210448846  ]},{  "type": "way",  "id": 663509723,  "nodes": [    6210448838,    6210448840  ]},{  "type": "way",  "id": 663509728,  "nodes": [    42436371,    6210448848,    6210448853,    6210448864  ]},{  "type": "way",  "id": 663509729,  "nodes": [    42436384,    42436390,    4207862996  ]},{  "type": "way",  "id": 663509731,  "nodes": [    4207862995,    42436384  ]},{  "type": "way",  "id": 663509733,  "nodes": [    6210448859,    6210448862  ]},{  "type": "way",  "id": 663509737,  "nodes": [    4207862995,    6210448835,    6210448859  ]},{  "type": "way",  "id": 663509739,  "nodes": [    4207862996,    6210448839,    6210448838  ]},{  "type": "way",  "id": 663509740,  "nodes": [    6210448846,    6210448867,    6210448859  ]},{  "type": "way",  "id": 663509742,  "nodes": [    6210448862,    6210448838  ]},{  "type": "way",  "id": 663797739,  "nodes": [    4841945993,    42431802  ]},{  "type": "way",  "id": 5669976,  "nodes": [    42431156,    4872626650,    42431154,    42431151  ]},{  "type": "way",  "id": 5670451,  "nodes": [    42436439,    4634753633,    4685465959,    42429394,    42436431,    4685462237,    42431503  ]},{  "type": "way",  "id": 5670514,  "nodes": [    42430187,    5854302555,    5854302557,    3891950555,    5854302560,    42436963,    5866337889,    42431517,    42436968  ]},{  "type": "way",  "id": 5670941,  "nodes": [    6213204061,    529555931,    42440940,    42433278,    42432205  ]},{  "type": "way",  "id": 5670942,  "nodes": [    42440946,    3874650201,    42440943,    42436146,    6210448845,    6210448843,    42436411  ]},{  "type": "way",  "id": 5671203,  "nodes": [    42436899,    42444267  ]},{  "type": "way",  "id": 5671261,  "nodes": [    42436159,    42431151,    4015418428,    4557517555,    4634753635,    42444897  ]},{  "type": "way",  "id": 5671460,  "nodes": [    42431789,    6213204050,    42446902  ]},{  "type": "way",  "id": 5671603,  "nodes": [    4321748237,    3891963914,    2522434121,    42431515  ]},{  "type": "way",  "id": 5671668,  "nodes": [    42430828,    42433275,    42449076,    6213204071,    42431814,    42436407  ]},{  "type": "way",  "id": 5672073,  "nodes": [    42430813,    42452598,    6213204074,    42431805,    42436400  ]},{  "type": "way",  "id": 5672922,  "nodes": [    42430761,    3546132354,    42431519,    42431773  ]},{  "type": "way",  "id": 5672987,  "nodes": [    42436377,    6210448846,    42436119,    42456986,    4599130009,    4872777160,    42453442  ]},{  "type": "way",  "id": 5673052,  "nodes": [    42436404,    42431812,    6213204034,    42457421,    42433272,    42430824  ]},{  "type": "way",  "id": 5673102,  "nodes": [    42453457,    4872777179,    42444267,    42457735,    42436134,    5357273833,    6210448868,    42436396  ]},{  "type": "way",  "id": 5673605,  "nodes": [    42454325,    42429412,    42431154  ]},{  "type": "way",  "id": 22903923,  "nodes": [    246647922,    373896721,    246647923,    373896722,    246647924,    373896723,    246647925,    373897299,    246647926,    246579724,    593816329,    402475097  ]},{  "type": "way",  "id": 22909247,  "nodes": [    42431156,    373896216,    42440580,    42453517,    42453510,    42453502,    42453489,    42453485,    593816275  ]},{  "type": "way",  "id": 22927373,  "nodes": [    42439099,    3874650207,    42439096,    42436152,    42436423  ]},{  "type": "way",  "id": 24263967,  "nodes": [    4321748238,    5796876484,    42430468,    3891963908,    42448563  ]},{  "type": "way",  "id": 34625443,  "nodes": [    402480678,    246652755,    246579760,    246647952,    246647953,    373897122,    373897123,    246647954,    2376384437,    246647955  ]},{  "type": "way",  "id": 46350694,  "nodes": [    42436100,    42436369,    42455200,    5151708185,    42430167  ]},{  "type": "way",  "id": 46431821,  "nodes": [    593816275,    246649433,    3884622511,    3884622510,    1701920974,    246649429  ]},{  "type": "way",  "id": 46431822,  "nodes": [    42436400,    6210448855,    42436135,    42452610,    3874650195,    42452613  ]},{  "type": "way",  "id": 46431823,  "nodes": [    42453468,    4872761059,    42457426,    42436139,    6210448860,    42436404  ]},{  "type": "way",  "id": 46431825,  "nodes": [    42436894,    42436899,    4872777172,    42436903  ]},{  "type": "way",  "id": 46431829,  "nodes": [    42436384,    6210448862,    42436126,    42452006,    4599130011,    4872777164,    42452008  ]},{  "type": "way",  "id": 46608946,  "nodes": [    4679523598,    4320028826,    4320028840,    42457728,    42452598,    42457421,    42449076,    42440940,    42439078,    42454742  ]},{  "type": "way",  "id": 46610717,  "nodes": [    42430770,    42431524,    42431777  ]},{  "type": "way",  "id": 46610719,  "nodes": [    4557517555,    42429412,    4599130018,    4872626641,    42429431  ]},{  "type": "way",  "id": 46610724,  "nodes": [    5837088104,    5837101294,    5151983466,    4816151672,    4320032276,    42451987  ]},{  "type": "way",  "id": 46610725,  "nodes": [    42436390,    42427898,    42431536,    42431797,    529556011,    4320028830,    42430803  ]},{  "type": "way",  "id": 95181768,  "nodes": [    246648627,    2376384363,    373895306,    2376384353,    246647959,    246649427,    246647962,    246649342,    4599130016,    246649343,    246649344,    4599130014,    246649345,    246874387,    4599130013,    246649346,    246878196,    4599130012,    246878284,    246876728,    4599130010,    246857083,    3874650189,    4207143439  ]},{  "type": "way",  "id": 125479715,  "nodes": [    42436393,    42436396,    4207862997  ]},{  "type": "way",  "id": 170113138,  "nodes": [    1812608107,    1812608108  ]},{  "type": "way",  "id": 194923795,  "nodes": [    42429663,    5545401595,    6211484420,    42429662,    42429661,    42429659,    5481850675,    42429657  ]},{  "type": "way",  "id": 195743558,  "nodes": [    42453457,    246874702,    42452613,    246874388,    42442081,    42453468,    246862852,    42449098  ]},{  "type": "way",  "id": 196116974,  "nodes": [    42459436,    4599130015,    3874650204,    42453472  ]},{  "type": "way",  "id": 216959248,  "nodes": [    42436396,    42431804  ]},{  "type": "way",  "id": 216959250,  "nodes": [    42431804,    6213204051,    42457728,    5244228881,    42430811  ]},{  "type": "way",  "id": 226193933,  "nodes": [    42431508,    42454758,    42439096,    42459436,    4741865080,    42440943,    42449089,    42457426,    42442078,    42452610,    42457735,    42436894,    42452006,    42454642,    42456986,    42440527,    42448568  ]},{  "type": "way",  "id": 277945772,  "nodes": [    2823833421,    2823299575,    4634753632  ]},{  "type": "way",  "id": 277945773,  "nodes": [    2823833422,    2823299578,    2823299577,    4634753632  ]},{  "type": "way",  "id": 420522049,  "nodes": [    42449098,    246862616,    42440946,    42453472,    246882949,    42439099,    593816490,    246647917,    246861652,    246649429  ]},{  "type": "way",  "id": 420522050,  "nodes": [    593816391,    246647956,    246647957,    246647958,    2376384365,    246648627  ]},{  "type": "way",  "id": 420598323,  "nodes": [    42430805,    4320028818,    4320028807,    42430803,    4320028833,    4205565494,    4205565493,    5837088104,    5151983468,    6213204036,    5837088105  ]},{  "type": "way",  "id": 439933934,  "nodes": [    42431497,    6213315380,    42429388  ]},{  "type": "way",  "id": 440268554,  "nodes": [    246652628,    373901771,    246652649,    2208862783,    2376384471,    2376384469,    2376384467,    246647987,    4379966364,    246647988,    246652738,    2376384465,    2376384463,    2376384461,    2034609413,    2208862797,    246652754,    4379966358,    246648824,    2208862779,    2376384447,    246647992,    2376384443,    2376384435,    246647993,    2208862762,    246647994,    593816397  ]},{  "type": "way",  "id": 458106347,  "nodes": [    42435598,    4205565513,    3874589908,    387180916  ]},{  "type": "way",  "id": 458106348,  "nodes": [    42435596,    4205565508,    4634753634,    42436439  ]},{  "type": "way",  "id": 467222812,  "nodes": [    5837088105,    4618691808,    5690453353,    42431789,    42431785,    42429701,    42431777,    42431773  ]},{  "type": "way",  "id": 467222828,  "nodes": [    42430805,    4320028836,    6213204054,    42431802  ]},{  "type": "way",  "id": 468414632,  "nodes": [    3874650186,    42440527,    42436115  ]},{  "type": "way",  "id": 468414633,  "nodes": [    4629488490,    42454642  ]},{  "type": "way",  "id": 474634621,  "nodes": [    42436439,    4685453631,    4685453632,    4685453633,    4685462238  ]},{  "type": "way",  "id": 480112475,  "nodes": [    42432211,    6213315379,    42454742,    42435581  ]},{  "type": "way",  "id": 480112476,  "nodes": [    42435578,    42439078,    5849907289,    42432208  ]},{  "type": "way",  "id": 480112478,  "nodes": [    42436407,    6210448857,    42436142,    42449089,    4872761060,    42449098  ]},{  "type": "way",  "id": 480112483,  "nodes": [    42430791,    42431785  ]},{  "type": "way",  "id": 489375010,  "nodes": [    42431844,    6213315375,    4816137912,    6211332483  ]},{  "type": "way",  "id": 505776502,  "nodes": [    246647920,    373895090,    246647921,    373896719,    246647922  ]},{  "type": "way",  "id": 643444515,  "nodes": [    42431497,    5849926992,    1825740014,    42429378  ]},{  "type": "way",  "id": 663509699,  "nodes": [    6210448840,    6210448868,    6210448865  ]},{  "type": "way",  "id": 663630710,  "nodes": [    6211332483,    6211334191,    42429390  ]},{  "type": "way",  "id": 663797737,  "nodes": [    42431804,    4841945993  ]},{  "type": "way",  "id": 473884145,  "nodes": [    246649429,    373895089,    246647920  ]},{  "type": "way",  "id": 473884146,  "nodes": [    246649429,    3884622509,    3884622508,    3884622507,    246649427  ]},{  "type": "way",  "id": 5670012,  "nodes": [    42431508,    42431505,    42431503  ]},{  "type": "way",  "id": 5670454,  "nodes": [    42436411,    1701921010,    1412724864,    1412724863,    1412724862,    1701920959,    6211332478,    6211334202  ]},{  "type": "way",  "id": 5672535,  "nodes": [    42436427,    42436155,    42454758,    42453477  ]},{  "type": "way",  "id": 5672681,  "nodes": [    42429390,    4685453633,    42436431,    42431149,    42436159,    719516045,    42455357,    1042457827,    1578334770,    4599130017,    1578334788,    593816275  ]},{  "type": "way",  "id": 5673452,  "nodes": [    42436414,    4685465960,    247403144,    247403145,    1825740002,    247402950  ]},{  "type": "way",  "id": 5673622,  "nodes": [    42431500,    42429390  ]},{  "type": "way",  "id": 22909249,  "nodes": [    42453477,    1701920988,    1701920986,    246882782,    1701920956,    1701920975,    246882783,    42431511  ]},{  "type": "way",  "id": 22916321,  "nodes": [    42435583,    6211334204,    42431500  ]},{  "type": "way",  "id": 46610721,  "nodes": [    42431154,    1096111017,    1042457810,    42455357  ]},{  "type": "way",  "id": 46610728,  "nodes": [    42429390,    4685453632  ]},{  "type": "way",  "id": 159657620,  "nodes": [    6211334202,    6211334201,    42435578  ]},{  "type": "way",  "id": 195834894,  "nodes": [    247402950,    1825739998,    1825739995,    1825739992,    5244251222,    42436149  ]},{  "type": "way",  "id": 196117028,  "nodes": [    42436149,    42436152,    42436155,    42431505,    42436159  ]},{  "type": "way",  "id": 196117032,  "nodes": [    42436414,    1825740011,    247403149,    1825740008,    247403147,    6211334192,    6211334195,    6211334202  ]},{  "type": "way",  "id": 277885759,  "nodes": [    2823299564,    2823299565  ]},{  "type": "way",  "id": 277885761,  "nodes": [    2823299566,    2823299567  ]},{  "type": "way",  "id": 277885763,  "nodes": [    2823299568,    1272608956  ]},{  "type": "way",  "id": 277885764,  "nodes": [    2823299569,    2823299571,    2823299564  ]},{  "type": "way",  "id": 277885766,  "nodes": [    4634753632,    2845912197  ]},{  "type": "way",  "id": 280480585,  "nodes": [    2845912197,    2823300805,    2823299573,    2823299572,    2823299570,    2921544680,    2823299569,    2823299558,    2921544667,    2823299559,    2823299560,    2823299561,    2823299562,    5875541578  ]},{  "type": "way",  "id": 420895279,  "nodes": [    4207862997,    42436400  ]},{  "type": "way",  "id": 420895280,  "nodes": [    4207862998,    42436407  ]},{  "type": "way",  "id": 420895283,  "nodes": [    42436400,    42436404,    4207862998  ]},{  "type": "way",  "id": 420895285,  "nodes": [    42436407,    2076950449,    6210448851,    42436411  ]},{  "type": "way",  "id": 438368333,  "nodes": [    246647955,    373896889,    593816391  ]},{  "type": "way",  "id": 439933935,  "nodes": [    42435578,    5331842131,    42435581,    42435583,    4492821528  ]},{  "type": "way",  "id": 452492203,  "nodes": [    4492821528,    6211332476,    42431497  ]},{  "type": "way",  "id": 463532615,  "nodes": [    42431503,    4685453634  ]},{  "type": "way",  "id": 474635896,  "nodes": [    4685462237,    4685462239,    4685462238  ]},{  "type": "way",  "id": 474635897,  "nodes": [    4685453634,    5849948634,    4685453635,    4685462241  ]},{  "type": "way",  "id": 474635898,  "nodes": [    4685453634,    42431500  ]},{  "type": "way",  "id": 474635899,  "nodes": [    42436427,    4685462240,    42436423  ]},{  "type": "way",  "id": 474635900,  "nodes": [    4685462238,    4685453634  ]},{  "type": "way",  "id": 474635901,  "nodes": [    4685462240,    4685462242,    4685462241  ]},{  "type": "way",  "id": 474635902,  "nodes": [    4685462241,    4685453636  ]},{  "type": "way",  "id": 474635903,  "nodes": [    4685453636,    4685453637,    4685453638,    4685453639,    4685453640,    4685453641,    4685453642,    6211334202  ]},{  "type": "way",  "id": 474635904,  "nodes": [    42436423,    42436414  ]},{  "type": "way",  "id": 480112477,  "nodes": [    42436423,    4685453636,    6211334198,    42435578  ]},{  "type": "way",  "id": 489375009,  "nodes": [    4492821528,    4816137911,    6211334203,    42431844  ]},{  "type": "way",  "id": 489375011,  "nodes": [    42431500,    5849926991,    42431844  ]},{  "type": "way",  "id": 496985713,  "nodes": [    42431503,    42436427  ]},{  "type": "way",  "id": 496985714,  "nodes": [    42435581,    6211332477,    4685453635,    42436427  ]},{  "type": "way",  "id": 496989301,  "nodes": [    42431511,    42431508  ]},{  "type": "way",  "id": 496989302,  "nodes": [    42455357,    42431508  ]},{  "type": "way",  "id": 543908289,  "nodes": [    42436149,    42459436  ]},{  "type": "way",  "id": 611258688,  "nodes": [    4685453632,    42429394,    4557517555  ]},{  "type": "way",  "id": 646635888,  "nodes": [    42431844,    42431497  ]},{  "type": "way",  "id": 663509716,  "nodes": [    4207862997,    6210448842,    6210448865  ]},{  "type": "way",  "id": 663509719,  "nodes": [    6210448865,    6210448855  ]},{  "type": "way",  "id": 663509727,  "nodes": [    6210448863,    6210448857  ]},{  "type": "way",  "id": 663509730,  "nodes": [    6210448855,    6210448860,    6210448863  ]},{  "type": "way",  "id": 663509736,  "nodes": [    6210448857,    6210448837,    6210448843,    6210448847,    6210448836,    6210448849,    6210448850,    6211334195,    4685453642  ]},{  "type": "way",  "id": 663509738,  "nodes": [    4207862998,    6210448856,    6210448863  ]},{  "type": "way",  "id": 663612292,  "nodes": [    4685453642,    6211334185,    6211334198,    6211332477,    6211334204,    6211334197,    42431844  ]},{  "type": "way",  "id": 663634122,  "nodes": [    6211334202,    6211527559,    42431831,    1825739972,    42431829,    6213204061,    2076950450,    42431814,    42431812,    42431805,    42431804  ]},{  "type": "way",  "id": 5669643,  "nodes": [    42427390,    2671667955,    429568770,    3874653365,    42427393  ]},{  "type": "way",  "id": 5669835,  "nodes": [    373903786,    42429666  ]},{  "type": "way",  "id": 5671698,  "nodes": [    246580982,    1701920982,    1701920961,    1701921024,    1701921016,    1701920951,    42449345  ]},{  "type": "way",  "id": 5672800,  "nodes": [    42429431,    1578345404,    42431156  ]},{  "type": "way",  "id": 22897720,  "nodes": [    42454328,    3874650227,    4133206257,    4133206253,    42429431  ]},{  "type": "way",  "id": 22909132,  "nodes": [    402480678,    2376384457,    2376384455,    2376384453,    402475097  ]},{  "type": "way",  "id": 22909136,  "nodes": [    4133206254,    42454328  ]},{  "type": "way",  "id": 22909152,  "nodes": [    246648219,    3874650241,    42445766  ]},{  "type": "way",  "id": 22914182,  "nodes": [    4872623863,    4872623868,    4872623855,    1701920994,    1701921000,    42454381  ]},{  "type": "way",  "id": 34625442,  "nodes": [    402475097,    246652740,    42445769  ]},{  "type": "way",  "id": 46334602,  "nodes": [    387180916,    4557517550,    42445766  ]},{  "type": "way",  "id": 46334603,  "nodes": [    4133206254,    2376384451,    246648217,    246648218,    2376384459,    246648219  ]},{  "type": "way",  "id": 46610722,  "nodes": [    4557517550,    42444900,    3564892568,    2985573628,    2985573626,    4557517554  ]},{  "type": "way",  "id": 127693199,  "nodes": [    42429664,    4555500078,    4555542099,    4555542098,    4555500073,    4555542092,    42429663  ]},{  "type": "way",  "id": 127693200,  "nodes": [    42429664,    42454381,    6401966213,    42427390  ]},{  "type": "way",  "id": 127693388,  "nodes": [    42444909,    4557517552,    4557517553,    387180916  ]},{  "type": "way",  "id": 194923791,  "nodes": [    42436439,    4634753636,    42444897  ]},{  "type": "way",  "id": 194923792,  "nodes": [    42445766,    3874650238,    4207862999  ]},{  "type": "way",  "id": 194923793,  "nodes": [    42445769,    1701921009,    1701921001,    1701921023,    1701920922,    246648063  ]},{  "type": "way",  "id": 194923794,  "nodes": [    42429666,    3874653362,    2671667952,    42429664  ]},{  "type": "way",  "id": 194923800,  "nodes": [    4783358909,    42449341,    42439286,    42429664  ]},{  "type": "way",  "id": 194923802,  "nodes": [    42445769,    4379966362,    246652651,    42449345,    246652630,    42439295,    246652326,    42429666  ]},{  "type": "way",  "id": 194923812,  "nodes": [    42429663,    42439280,    4555581031,    4555581029,    3564892571  ]},{  "type": "way",  "id": 196117075,  "nodes": [    4557517554,    5244251221,    42454325,    2671667947,    3874650225,    42454328  ]},{  "type": "way",  "id": 226041041,  "nodes": [    42427390,    42443676  ]},{  "type": "way",  "id": 261569309,  "nodes": [    2823833470,    2671667954  ]},{  "type": "way",  "id": 277885758,  "nodes": [    2823299583,    2823299576,    2823300821,    2823833423  ]},{  "type": "way",  "id": 277885765,  "nodes": [    2823833443,    2823895886,    2823895885,    2823895887  ]},{  "type": "way",  "id": 277885767,  "nodes": [    2823299587,    2823299588  ]},{  "type": "way",  "id": 277885768,  "nodes": [    2823299587,    2823299589  ]},{  "type": "way",  "id": 277885769,  "nodes": [    2823299589,    2823299590  ]},{  "type": "way",  "id": 277885775,  "nodes": [    2823300832,    2845937087,    2823300835,    2823300834,    2823300833,    2823833426  ]},{  "type": "way",  "id": 277885776,  "nodes": [    2823299590,    2823300836,    371247825  ]},{  "type": "way",  "id": 277945774,  "nodes": [    2823833424,    2823299580,    2823299582,    2823299579,    2823300820,    2823833422  ]},{  "type": "way",  "id": 277945775,  "nodes": [    2823833423,    2823299582,    2823300817,    2823833421  ]},{  "type": "way",  "id": 277945776,  "nodes": [    2823833426,    2823299586,    2823833425  ]},{  "type": "way",  "id": 277945777,  "nodes": [    2823833425,    2823299585,    2823300823,    2823299583,    2823299581,    2823300822,    2823833424  ]},{  "type": "way",  "id": 277945783,  "nodes": [    2823833473,    2823833474  ]},{  "type": "way",  "id": 277945785,  "nodes": [    2823833476,    2823833477  ]},{  "type": "way",  "id": 277945788,  "nodes": [    2823833479,    2823833482  ]},{  "type": "way",  "id": 277945790,  "nodes": [    2823833475,    2823833489  ]},{  "type": "way",  "id": 277945792,  "nodes": [    2823833483,    2823833490  ]},{  "type": "way",  "id": 277951748,  "nodes": [    2823895887,    2823895884,    2823833436,    2823833435,    2823299592,    2823299591,    2823300836  ]},{  "type": "way",  "id": 277951749,  "nodes": [    2823895888,    2823833434,    2823833433,    2823833432,    2823833431,    2823833430,    2823833429,    2823833428,    371247825,    2823299584,    2845937086,    2823300833  ]},{  "type": "way",  "id": 277951750,  "nodes": [    2823895889,    2823833443,    2823895888  ]},{  "type": "way",  "id": 308322431,  "nodes": [    246580982,    246579759,    246652650,    4379966363,    246648063,    246648107,    402480678  ]},{  "type": "way",  "id": 348477347,  "nodes": [    42449345,    3874653357,    42449341  ]},{  "type": "way",  "id": 350565058,  "nodes": [    2823833444,    2671667948  ]},{  "type": "way",  "id": 350726031,  "nodes": [    387180916,    3564892567,    3564892566,    3564892565,    4634753637,    42436439  ]},{  "type": "way",  "id": 420895276,  "nodes": [    4207863000,    246580982  ]},{  "type": "way",  "id": 420895277,  "nodes": [    373903786,    246652365,    246580346,    246652629,    4207863000  ]},{  "type": "way",  "id": 420895286,  "nodes": [    4207862999,    4379966360,    42445769  ]},{  "type": "way",  "id": 485556246,  "nodes": [    4783358910,    4783358909  ]},{  "type": "way",  "id": 485556247,  "nodes": [    42445766,    3874650236,    4783358910  ]},{  "type": "way",  "id": 505765239,  "nodes": [    42444897,    4557517554  ]},{  "type": "way",  "id": 505766273,  "nodes": [    4557517552,    4557517551,    4557517550  ]},{  "type": "way",  "id": 541329572,  "nodes": [    42439286,    3874653359,    42439295  ]},{  "type": "way",  "id": 541329577,  "nodes": [    42439295,    246580346  ]},{  "type": "way",  "id": 564392620,  "nodes": [    402475097,    3874650232,    4133206254  ]},{  "type": "way",  "id": 589494594,  "nodes": [    42454328,    246648219  ]},{  "type": "way",  "id": 674686230,  "nodes": [    3564892571,    42444909  ]},{  "type": "way",  "id": 22914180,  "nodes": [    42443677,    4872623873,    42443676  ]},{  "type": "way",  "id": 22914181,  "nodes": [    42427390,    42427386,    4601466498,    6211484438,    42427381  ]},{  "type": "way",  "id": 34625464,  "nodes": [    246579753,    246652085,    373903786  ]},{  "type": "way",  "id": 116672333,  "nodes": [    42430589,    3914863405,    2823833538,    42437914,    42437909  ]},{  "type": "way",  "id": 156994955,  "nodes": [    42443674,    42443676  ]},{  "type": "way",  "id": 156994977,  "nodes": [    42443674,    42427386,    42444916,    42429663  ]},{  "type": "way",  "id": 194923801,  "nodes": [    42429666,    4872623863,    42427393,    246652086,    42443677,    246579597  ]},{  "type": "way",  "id": 226041042,  "nodes": [    42443676,    42434285  ]},{  "type": "way",  "id": 348477348,  "nodes": [    42449341,    2671667948,    3236324368,    3236324362,    42444909,    4601466495,    6211484431,    42435599  ]},{  "type": "way",  "id": 350563861,  "nodes": [    3563148493,    3563148495,    2823833602,    3146796134,    2823833549,    3146796136,    3146796138,    2823833548,    2823833547,    2823833546,    2823833545,    2823833544,    2823833536,    2823833543,    2840192834,    2840192833,    2840192832,    2823833535,    2823833519,    2823833532,    2823833523,    2823833469,    2823833468,    2823833467,    2823833463,    2823833471,    2823833462,    2823833461,    2823833460,    2823833459,    2823833458,    2823833457,    2823833456,    2823833481,    2823833455,    2823833518,    2823833454,    2823833453,    2823833452,    2823833451,    2823833450,    2823833449,    2823833496,    2823833447,    2823833446,    2823833464,    2823833445,    2823833444,    2823833437,    2823895889  ]},{  "type": "way",  "id": 396444293,  "nodes": [    42427483,    42453630,    42445310,    42434821  ]},{  "type": "way",  "id": 458106345,  "nodes": [    42435610,    4205565522,    4601466497,    42444916,    4555500086,    4555501893,    4555501898,    4555542101,    4555501902,    42454381  ]},{  "type": "way",  "id": 458106346,  "nodes": [    42435603,    4205565517,    4601466496,    42439280,    4555581039,    4555581040,    4555581043,    4555581045,    5103334974,    42439286  ]},{  "type": "way",  "id": 674686233,  "nodes": [    4557517549,    3564892574,    5545401596,    4592028452,    5545401594,    4592028453,    5545401593,    5143758606,    4689208568,    5143758608,    5545401592,    4592028455,    5545401591,    4592029846,    5852207706,    5545401590,    4592029847,    5545401589,    4592029848,    4601466500,    4592029849,    4601466499,    4592029850,    4601466498,    4592029851,    4601466497,    4592029852,    5545401595,    4592029853,    4601466496,    4592029854,    3564892573,    3564892572,    3564892571  ]},{  "type": "way",  "id": 5669945,  "nodes": [    42422258,    3914863394,    42430589  ]},{  "type": "way",  "id": 5670247,  "nodes": [    42434285,    4872623861,    42422258  ]},{  "type": "way",  "id": 22903913,  "nodes": [    42437915,    246579582,    246579614,    373910795,    246651361,    42427494  ]},{  "type": "way",  "id": 22904053,  "nodes": [    246579597,    1701844602,    1701844809,    1701844672,    1701844780,    4872623865,    42422258  ]},{  "type": "way",  "id": 22904055,  "nodes": [    246580445,    42427483  ]},{  "type": "way",  "id": 22904149,  "nodes": [    246580466,    3914863402,    4207978494  ]},{  "type": "way",  "id": 34625463,  "nodes": [    246580736,    246579608,    246580301  ]},{  "type": "way",  "id": 46431818,  "nodes": [    4207978495,    246580443,    1701921027,    246580465,    1701920933,    246580466  ]},{  "type": "way",  "id": 46490540,  "nodes": [    42434285,    42437914,    42427483  ]},{  "type": "way",  "id": 194923799,  "nodes": [    246580758,    1701920957,    3914863404,    1701920939,    1701920999,    1701920946,    246651648,    1701844592,    246580736  ]},{  "type": "way",  "id": 194923804,  "nodes": [    246580301,    246579610,    246650838,    42437915  ]},{  "type": "way",  "id": 194923805,  "nodes": [    246580797,    246579746,    246651360,    373910553,    2376384580,    246579748,    246650840  ]},{  "type": "way",  "id": 194923806,  "nodes": [    246650840,    246579749,    246579750,    246579751,    246651647,    246579752,    246579753  ]},{  "type": "way",  "id": 234489459,  "nodes": [    246580736,    246579753  ]},{  "type": "way",  "id": 277945795,  "nodes": [    2823833520,    2823833529,    2823833530,    2823833532  ]},{  "type": "way",  "id": 277945798,  "nodes": [    2823833537,    2845975411,    2823833539  ]},{  "type": "way",  "id": 277945800,  "nodes": [    2823833540,    2823833541  ]},{  "type": "way",  "id": 277945801,  "nodes": [    2823833541,    2823833542,    2823833543  ]},{  "type": "way",  "id": 420908127,  "nodes": [    4207978494,    3914863399,    246580758  ]},{  "type": "way",  "id": 420908128,  "nodes": [    42430589,    3914863396,    4207978495  ]},{  "type": "way",  "id": 464847780,  "nodes": [    246579597,    246580736  ]},{  "type": "way",  "id": 584971507,  "nodes": [    246580758,    42430589  ]},{  "type": "way",  "id": 5669842,  "nodes": [    42429701,    6213204069,    4375703339,    4320105473,    42429700  ]},{  "type": "way",  "id": 5669925,  "nodes": [    4321677346,    4321677347,    42430470,    42430468  ]},{  "type": "way",  "id": 5670170,  "nodes": [    42433260,    4320028823,    42433261  ]},{  "type": "way",  "id": 5670171,  "nodes": [    42433281,    5690447869,    42430235,    42433282,    42433260,    4320028844,    42433286  ]},{  "type": "way",  "id": 5670204,  "nodes": [    42433822,    4320028827,    42433821  ]},{  "type": "way",  "id": 5670488,  "nodes": [    42436793,    6213204062  ]},{  "type": "way",  "id": 5670498,  "nodes": [    42430187,    5854302559,    5854302556,    5854302554,    529556095,    42436844,    42436848  ]},{  "type": "way",  "id": 5671754,  "nodes": [    42433821,    4320032274,    42430233,    4302910548,    1692433942  ]},{  "type": "way",  "id": 5671799,  "nodes": [    42437951,    42450333,    3584752526,    42433281  ]},{  "type": "way",  "id": 5671973,  "nodes": [    42433821,    4320032277,    5690465658,    42451987  ]},{  "type": "way",  "id": 5672159,  "nodes": [    42433261,    42433286,    4320028838,    42433822,    4320028822,    4320028816,    42430803  ]},{  "type": "way",  "id": 5672654,  "nodes": [    42450325,    5151708183,    42430167  ]},{  "type": "way",  "id": 5673309,  "nodes": [    1692433939,    42436848,    4320108384,    42430205  ]},{  "type": "way",  "id": 5673619,  "nodes": [    42430231,    4320108387,    4320108382,    4302910555,    1692433941  ]},{  "type": "way",  "id": 24264070,  "nodes": [    3584752508,    3584752515,    3584752522,    42433281  ]},{  "type": "way",  "id": 25199402,  "nodes": [    42430805,    4320028842,    4320028809,    42433822,    4320028845,    42451987,    4320032286,    42446902,    4320105472,    42429700,    4375703341,    42430231  ]},{  "type": "way",  "id": 46139045,  "nodes": [    4321748238,    4321748237  ]},{  "type": "way",  "id": 46607937,  "nodes": [    42431773,    6213204047  ]},{  "type": "way",  "id": 157012387,  "nodes": [    1692433941,    3584752496,    1692433940,    1692433939,    6213204043  ]},{  "type": "way",  "id": 157012398,  "nodes": [    42433281,    3584752520,    4302910557,    1692433942,    4302910552,    4302903239,    1692433941  ]},{  "type": "way",  "id": 239988103,  "nodes": [    2477560563,    3891950891,    2550012966,    3891950890,    2477560572  ]},{  "type": "way",  "id": 386211349,  "nodes": [    42436796,    3584752228,    42436793  ]},{  "type": "way",  "id": 420352251,  "nodes": [    4202908174,    3573482131,    4321748238  ]},{  "type": "way",  "id": 433124968,  "nodes": [    3891960758,    3891969468,    42430205,    5151719389,    5155816418,    5151711951,    42430217,    5151711953,    4016646252,    4375703337,    42430231,    4375703342,    2580430244,    42430233,    42430235,    5155816417,    4320194279,    42430237  ]},{  "type": "way",  "id": 433124996,  "nodes": [    42431773,    3891969470,    3891960760,    5854302552,    42430187  ]},{  "type": "way",  "id": 433394637,  "nodes": [    4202908174,    5799117241,    4321748235,    4321748237  ]},{  "type": "way",  "id": 433394647,  "nodes": [    3891963915,    5866337888,    4321748238  ]},{  "type": "way",  "id": 433398367,  "nodes": [    42448549,    5829880522,    4202908174  ]},{  "type": "way",  "id": 433442259,  "nodes": [    4321677346,    3891963915  ]},{  "type": "way",  "id": 452339699,  "nodes": [    4321748237,    3891963909,    5155816419,    2477560572,    3891960759,    42430187,    3891960758  ]},{  "type": "way",  "id": 481047237,  "nodes": [    42440553,    3584752204,    42452396  ]},{  "type": "way",  "id": 591756888,  "nodes": [    4679523598,    42430805  ]},{  "type": "way",  "id": 663799680,  "nodes": [    6213204062,    1692433932  ]},{  "type": "way",  "id": 663799681,  "nodes": [    6213204047,    3891969467,    3891969471,    42430205  ]},{  "type": "way",  "id": 663799682,  "nodes": [    6213204043,    1692433938  ]},{  "type": "way",  "id": 5670172,  "nodes": [    42433298,    3584752276,    42433295,    42433292,    42428468  ]},{  "type": "way",  "id": 5670243,  "nodes": [    42434194,    1110619966,    5784518539,    5784518540,    1110619975,    42434196  ]},{  "type": "way",  "id": 5671663,  "nodes": [    42428476,    4762558061,    4500010374,    4500010382,    42449045,    42421877  ]},{  "type": "way",  "id": 5671743,  "nodes": [    42433587,    3584752200,    3584752212,    42449817,    3584752226,    3584752273,    42433298  ]},{  "type": "way",  "id": 5672178,  "nodes": [    42439249,    3627218171,    3919359296,    42445558  ]},{  "type": "way",  "id": 5672873,  "nodes": [    42456205,    42456206,    42433295,    42456211  ]},{  "type": "way",  "id": 5672910,  "nodes": [    42428464,    42449582,    42456206,    3584752215,    42449817  ]},{  "type": "way",  "id": 5673089,  "nodes": [    42437949,    6255601608,    6255601607,    42456303,    42456211,    42449589,    4755898678,    42428473  ]},{  "type": "way",  "id": 5673533,  "nodes": [    42445558,    42445548  ]},{  "type": "way",  "id": 5673614,  "nodes": [    42433587,    3584752195  ]},{  "type": "way",  "id": 46139037,  "nodes": [    42439236,    4234425046,    3919350745,    42445543,    5784518545,    3919359299,    42445548  ]},{  "type": "way",  "id": 68668558,  "nodes": [    42428473,    5784518536,    42445543  ]},{  "type": "way",  "id": 196117006,  "nodes": [    42439249,    3919350757,    595407639  ]},{  "type": "way",  "id": 352705238,  "nodes": [    3584752195,    4745686172,    42456205,    4745686174,    3957463103,    4117109190,    42449580,    42428460  ]},{  "type": "way",  "id": 420358897,  "nodes": [    4202984497,    5784518538,    3919350763,    42439236  ]},{  "type": "way",  "id": 420526338,  "nodes": [    42428471,    3919350758,    42439236  ]},{  "type": "way",  "id": 439933929,  "nodes": [    42437949,    42433354  ]},{  "type": "way",  "id": 439933946,  "nodes": [    4234425009,    595407648  ]},{  "type": "way",  "id": 451904646,  "nodes": [    42445543,    3919350764,    3919350754,    42439249  ]},{  "type": "way",  "id": 453239583,  "nodes": [    4500010367,    4500010369  ]},{  "type": "way",  "id": 467523297,  "nodes": [    595407624,    5784518517,    4621333640,    4621333641,    4621333642,    1773121034  ]},{  "type": "way",  "id": 467523298,  "nodes": [    4015190382,    5784518525,    4621333639,    5784518512,    595407624  ]},{  "type": "way",  "id": 481655770,  "nodes": [    4745685367,    4745685370  ]},{  "type": "way",  "id": 481655774,  "nodes": [    4745685361,    4745685363  ]},{  "type": "way",  "id": 630828592,  "nodes": [    5590991384,    42434196,    5165772929,    5784518520,    4015190382,    5784518526,    1773121034  ]},{  "type": "way",  "id": 653471088,  "nodes": [    42445558,    4234425039,    595407648  ]},{  "type": "way",  "id": 653471090,  "nodes": [    42445548,    4234425037,    42428476  ]},{  "type": "way",  "id": 653471091,  "nodes": [    42445548,    4234425009  ]},{  "type": "way",  "id": 653534688,  "nodes": [    595407624,    42434194,    42439243,    42439249  ]},{  "type": "way",  "id": 5668989,  "nodes": [    42421877,    4841945996,    42421882  ]},{  "type": "way",  "id": 5669806,  "nodes": [    42429373,    42429374,    42429375,    42429377,    1701921025,    42429378  ]},{  "type": "way",  "id": 5669953,  "nodes": [    42430828,    5554351996,    42430824,    42430818,    5244228882,    42430813,    42430811,    5244228880,    4320028815,    4320028826,    42430805  ]},{  "type": "way",  "id": 5669962,  "nodes": [    42430857,    42430861,    42428480,    4500033348,    4500033344,    4234578192,    42430872,    4234578194,    42430886,    3892037906,    4236104521,    42430898,    4376122108,    4236104524,    3834459530,    42430903  ]},{  "type": "way",  "id": 5670087,  "nodes": [    42432191,    5690441975,    4320194276,    42430237  ]},{  "type": "way",  "id": 5670651,  "nodes": [    42430237,    1760570981,    1760570975,    1760570973,    1760570971,    1760570977,    42437951,    42437949  ]},{  "type": "way",  "id": 5670771,  "nodes": [    42427369,    42428183,    4597668028,    42439203,    42438849  ]},{  "type": "way",  "id": 5670782,  "nodes": [    595407639,    3919350751,    42434140  ]},{  "type": "way",  "id": 5670836,  "nodes": [    42437890,    42434270,    42439836,    42427374,    42439834,    42429661,    42439275,    42439830  ]},{  "type": "way",  "id": 5670897,  "nodes": [    42428458,    5824300189,    42440545,    3957463101,    42440548,    1692433936,    1692433937,    42440553,    42436793,    42439537,    1692433940,    4016646249,    4320108989,    42430217  ]},{  "type": "way",  "id": 5671158,  "nodes": [    42434197,    42443799  ]},{  "type": "way",  "id": 5671429,  "nodes": [    42439070,    42446521,    42428489,    5357086126,    42446528,    3627218182,    42446533,    3627218183,    42446540,    42442870,    4376122110,    3834459543,    42446547  ]},{  "type": "way",  "id": 5672001,  "nodes": [    42436856,    4320194273,    1701920971,    42452173,    1701920981,    42433282  ]},{  "type": "way",  "id": 5672040,  "nodes": [    1692433941,    3584752497,    3584752281,    42439542,    3584752265,    3584752242,    42436796,    3584752231,    3584752208,    42452396,    3584752205,    3584752198,    42433587  ]},{  "type": "way",  "id": 5672211,  "nodes": [    42438173,    42453310,    4597668039,    5481897719,    4276517116,    42428192,    4687262941,    42443671  ]},{  "type": "way",  "id": 5672434,  "nodes": [    42440710,    5416901182,    5244228887,    42430253  ]},{  "type": "way",  "id": 5672450,  "nodes": [    42454378,    42430263,    42439834,    42435610  ]},{  "type": "way",  "id": 5672604,  "nodes": [    42454994,    3834459533,    4236104523,    4376122109,    42442862,    4236104522,    3627218170,    42449886,    3627218175,    4234578191,    42453104  ]},{  "type": "way",  "id": 5672607,  "nodes": [    42428483,    42455026,    5357196527,    42449067  ]},{  "type": "way",  "id": 5672896,  "nodes": [    42433298,    3584752292,    42433356,    42456303,    42449045,    42430861,    42455026,    42446521,    42436935  ]},{  "type": "way",  "id": 25162679,  "nodes": [    42436943,    3834459527,    5481912002,    4376122111,    42436942,    3627218174,    42436941,    3627218173,    4886319435,    42436939,    42428491,    42436935,    42429373  ]},{  "type": "way",  "id": 34080062,  "nodes": [    42427374,    42427371,    42427369  ]},{  "type": "way",  "id": 68668541,  "nodes": [    42434268,    42443671,    42427369,    42454378,    42429657,    42439272,    42449333,    42431490,    42440710  ]},{  "type": "way",  "id": 68674966,  "nodes": [    42433354,    3584752516,    3584752508  ]},{  "type": "way",  "id": 127691873,  "nodes": [    42428170,    1412706986,    3884600523,    1412706971,    3670742487  ]},{  "type": "way",  "id": 127693102,  "nodes": [    42430237,    4320194257,    4320194267,    42421889  ]},{  "type": "way",  "id": 164420607,  "nodes": [    42430818,    42433269,    42433264,    4320028821,    42433261  ]},{  "type": "way",  "id": 167923357,  "nodes": [    42428174,    4763882045,    4285055691,    42428170  ]},{  "type": "way",  "id": 195743341,  "nodes": [    42440710,    42429373,    42439070,    5243716991,    4500010387,    42449067,    42430857,    42421877,    42437949  ]},{  "type": "way",  "id": 195743556,  "nodes": [    42432194,    4320194277,    6254325748,    42421901,    42457721,    42430241,    42430857  ]},{  "type": "way",  "id": 196117090,  "nodes": [    42430255,    42430257,    42430259,    42429659,    42430263,    42427371,    42430265,    42430269,    42430271  ]},{  "type": "way",  "id": 197582873,  "nodes": [    1692433936,    3584752194,    42433587  ]},{  "type": "way",  "id": 347306313,  "nodes": [    42430255,    5350434145,    42439826  ]},{  "type": "way",  "id": 352705236,  "nodes": [    42433298,    3584752291,    3584752506,    3584752508  ]},{  "type": "way",  "id": 420522444,  "nodes": [    42449067,    42430247  ]},{  "type": "way",  "id": 420522695,  "nodes": [    42428493,    3670742435,    3670742477,    42440721  ]},{  "type": "way",  "id": 420598345,  "nodes": [    42435598,    42435599  ]},{  "type": "way",  "id": 420598402,  "nodes": [    4597668035,    3670742453,    4205565500  ]},{  "type": "way",  "id": 421825692,  "nodes": [    42428493,    3670742436,    4762558055,    4762558057,    42428491,    42428489,    42428483,    4684550288,    4747401775,    42428480,    4747401774,    42428476,    42428473  ]},{  "type": "way",  "id": 439933928,  "nodes": [    595407648,    4234578193,    42430872,    4234578190,    42453104,    42446528  ]},{  "type": "way",  "id": 464683316,  "nodes": [    42429657,    42428174,    3875000768,    4597668026  ]},{  "type": "way",  "id": 464683323,  "nodes": [    4597668036,    3252590543,    3875000763,    42428170  ]},{  "type": "way",  "id": 481403099,  "nodes": [    42430259,    42439275,    42435603  ]},{  "type": "way",  "id": 492148706,  "nodes": [    42432191,    4320194269,    4320194265,    42436856,    4320194264,    4320194261,    42433261,    5244228879,    4679523598  ]},{  "type": "way",  "id": 496978510,  "nodes": [    42430253,    5413254261,    5155799272,    42430255  ]},{  "type": "way",  "id": 496986855,  "nodes": [    42435599,    42439830  ]},{  "type": "way",  "id": 496993763,  "nodes": [    42439830,    42430257  ]},{  "type": "way",  "id": 505600307,  "nodes": [    42428179,    42454378  ]},{  "type": "way",  "id": 541330444,  "nodes": [    42443671,    42430265  ]},{  "type": "way",  "id": 584971395,  "nodes": [    42434140,    3919350741,    5590991384  ]},{  "type": "way",  "id": 646635889,  "nodes": [    42439073,    42430249,    4500012796,    42439070  ]},{  "type": "way",  "id": 653471089,  "nodes": [    42449014,    5780009413,    4234425018,    42445558  ]},{  "type": "way",  "id": 5669809,  "nodes": [    42429388,    6211332483  ]},{  "type": "way",  "id": 24264114,  "nodes": [    42421882,    42421889,    5690441977,    5690441976,    42432191  ]},{  "type": "way",  "id": 46609036,  "nodes": [    42429378,    6213315371,    6211332474,    42432214  ]},{  "type": "way",  "id": 46610734,  "nodes": [    42433278,    42433275,    42433272  ]},{  "type": "way",  "id": 127693384,  "nodes": [    42429388,    1825740017,    6211334199,    42432214  ]},{  "type": "way",  "id": 194923811,  "nodes": [    42435596,    42435598  ]},{  "type": "way",  "id": 195743555,  "nodes": [    42430828,    42432205,    42432208  ]},{  "type": "way",  "id": 196117073,  "nodes": [    42439823,    5244228885,    5244228886,    42430253  ]},{  "type": "way",  "id": 196117074,  "nodes": [    42439823,    42435596  ]},{  "type": "way",  "id": 196117091,  "nodes": [    42439823,    5244228884,    42429375,    42439073,    5244228883,    42430828  ]},{  "type": "way",  "id": 420522443,  "nodes": [    42432194,    42432199,    42432201  ]},{  "type": "way",  "id": 451905565,  "nodes": [    42421889,    4320194262,    42430241,    5155816415,    42430244,    42430247  ]},{  "type": "way",  "id": 465098295,  "nodes": [    42432214,    42435596  ]},{  "type": "way",  "id": 474635905,  "nodes": [    42432208,    1701920968,    1701920972,    1701921005,    42439073  ]},{  "type": "way",  "id": 480112479,  "nodes": [    42430824,    42432201  ]},{  "type": "way",  "id": 480112480,  "nodes": [    42432199,    42433269,    42430813  ]},{  "type": "way",  "id": 480112481,  "nodes": [    42430811,    42433264,    42432194  ]},{  "type": "way",  "id": 492148705,  "nodes": [    42432194,    1760570979,    1760570978,    1760570982,    1760570980,    1760570976,    4320194260,    42432191  ]},{  "type": "way",  "id": 496979224,  "nodes": [    42429374,    5413254263,    42430253  ]},{  "type": "way",  "id": 496993765,  "nodes": [    42439830,    42439826,    42439823  ]},{  "type": "way",  "id": 496993920,  "nodes": [    42439826,    42435598  ]},{  "type": "way",  "id": 497746842,  "nodes": [    42430247,    42430828  ]},{  "type": "way",  "id": 497746843,  "nodes": [    42430247,    42430249,    5244228888,    42429374  ]},{  "type": "way",  "id": 525033953,  "nodes": [    42432201,    42430828  ]},{  "type": "way",  "id": 611258687,  "nodes": [    42432208,    42432211,    4044849083,    6213315373,    42429378  ]},{  "type": "way",  "id": 663810755,  "nodes": [    42432214,    6213315378  ]},{  "type": "way",  "id": 5670011,  "nodes": [    42431491,    5148103557,    42431490  ]},{  "type": "way",  "id": 5670731,  "nodes": [    42428170,    3875000767,    5148103560,    42438785  ]},{  "type": "way",  "id": 5670785,  "nodes": [    42439272,    5481850676,    42430259  ]},{  "type": "way",  "id": 5670924,  "nodes": [    42440710,    5243716992,    5243716993,    5148103566,    42438784  ]},{  "type": "way",  "id": 5671359,  "nodes": [    42431490,    4500012798,    5416905054,    42430255  ]},{  "type": "way",  "id": 5671459,  "nodes": [    42428170,    4285055690,    42439272  ]},{  "type": "way",  "id": 5672713,  "nodes": [    42449333,    5148103568,    42438785  ]},{  "type": "way",  "id": 22902608,  "nodes": [    4205565500,    2077984663,    2077984658,    42437009,    1251605769,    42428493  ]},{  "type": "way",  "id": 348477349,  "nodes": [    42430257,    42449333  ]},{  "type": "way",  "id": 353655412,  "nodes": [    42431491,    3670742486,    42438784  ]},{  "type": "way",  "id": 509024996,  "nodes": [    4983147327,    4983147326  ]},{  "type": "way",  "id": 509025017,  "nodes": [    4983147823,    4983147822  ]},{  "type": "way",  "id": 509025018,  "nodes": [    4983147825,    4983147824  ]},{  "type": "way",  "id": 509025019,  "nodes": [    4983147827,    4983147826  ]},{  "type": "way",  "id": 509025020,  "nodes": [    4983147829,    4983147828  ]},{  "type": "way",  "id": 509025021,  "nodes": [    4983147831,    4983147830  ]},{  "type": "way",  "id": 509025022,  "nodes": [    4983147833,    4983147832  ]},{  "type": "way",  "id": 509025023,  "nodes": [    4983147835,    4983147834  ]},{  "type": "way",  "id": 509025024,  "nodes": [    4983147837,    4983147836  ]},{  "type": "way",  "id": 509025025,  "nodes": [    4983147839,    4983147838  ]},{  "type": "way",  "id": 509025026,  "nodes": [    4983147841,    4983147840  ]},{  "type": "way",  "id": 509025027,  "nodes": [    4983147843,    4983147842  ]},{  "type": "way",  "id": 509025028,  "nodes": [    4983147845,    4983147844  ]},{  "type": "way",  "id": 509025029,  "nodes": [    4983147849,    4983147848,    4983147847,    4983147846,    4983147849  ]},{  "type": "way",  "id": 509025030,  "nodes": [    4983147853,    4983147852,    4983147851,    4983147850,    4983147853  ]},{  "type": "way",  "id": 509025038,  "nodes": [    4983147898,    4983147897,    4983147896,    4983147895  ]},{  "type": "way",  "id": 509025040,  "nodes": [    4983147902,    4983147901,    4983147900,    4983147899  ]},{  "type": "way",  "id": 509025042,  "nodes": [    4983147904,    4983147903  ]},{  "type": "way",  "id": 509025044,  "nodes": [    4983147906,    4983147905  ]},{  "type": "way",  "id": 509025046,  "nodes": [    4983147908,    4983147907  ]},{  "type": "way",  "id": 509025048,  "nodes": [    4983147910,    4983147909  ]},{  "type": "way",  "id": 509025049,  "nodes": [    4983147913,    4983147912  ]},{  "type": "way",  "id": 509025050,  "nodes": [    4983147915,    4983147914  ]},{  "type": "way",  "id": 509025051,  "nodes": [    4983147917,    4983147916  ]},{  "type": "way",  "id": 509025053,  "nodes": [    4983147925,    4983147924,    4983147854  ]},{  "type": "way",  "id": 509025055,  "nodes": [    4983147928,    4983147927,    4983147926  ]},{  "type": "way",  "id": 509025059,  "nodes": [    4983147934,    4983147933  ]},{  "type": "way",  "id": 509025061,  "nodes": [    4983147941,    4983147940  ]},{  "type": "way",  "id": 509025062,  "nodes": [    4983147943,    4983147942  ]},{  "type": "way",  "id": 509025063,  "nodes": [    4983147947,    4983147946,    4983147945,    4983147944  ]},{  "type": "way",  "id": 509025064,  "nodes": [    4983147951,    4983147950,    4983147949,    4983147948  ]},{  "type": "way",  "id": 558008153,  "nodes": [    42438784,    3670742462,    42436935  ]},{  "type": "way",  "id": 666752576,  "nodes": [    42438784,    3670742441,    3670742474,    42428493  ]},{  "type": "way",  "id": 667344132,  "nodes": [    42438785,    5148103559,    5148103562,    42431491  ]},{  "type": "way",  "id": 5670232,  "nodes": [    42434140,    3919359298,    4235738201,    42434142,    4376122106,    4235738199,    3834459546,    42434145  ]},{  "type": "way",  "id": 5671095,  "nodes": [    42443296,    4235733230,    4235733231,    4235733228,    42443298,    4235734221,    5780009414,    4235734225,    3892040209,    3892037907,    3892039873,    3892037906  ]},{  "type": "way",  "id": 5671659,  "nodes": [    42449021,    3834459550,    4235738202,    4376122107,    42442857,    4235738196,    4235734223,    42443298,    3919359655,    42449017,    4235733235,    42449014  ]},{  "type": "way",  "id": 5672518,  "nodes": [    42449014,    42454673  ]},{  "type": "way",  "id": 420598234,  "nodes": [    4205565496,    1251612266  ]},{  "type": "way",  "id": 420598238,  "nodes": [    1251612266,    4205565495  ]},{  "type": "way",  "id": 420598248,  "nodes": [    42436942,    4205565490  ]},{  "type": "way",  "id": 420598252,  "nodes": [    4205565490,    42442870  ]},{  "type": "way",  "id": 420598256,  "nodes": [    42431464,    4205565492  ]},{  "type": "way",  "id": 420598262,  "nodes": [    42440737,    4205565491  ]},{  "type": "way",  "id": 420598270,  "nodes": [    4205565492,    42440737  ]},{  "type": "way",  "id": 420598286,  "nodes": [    4205565491,    6139477287,    42436942  ]},{  "type": "way",  "id": 420598304,  "nodes": [    4205565495,    4191922371,    42431464  ]},{  "type": "way",  "id": 424094336,  "nodes": [    4235734223,    4235734222,    4235734224,    4235734226,    4235734225  ]},{  "type": "way",  "id": 438389785,  "nodes": [    42434140,    3919359291,    42443296  ]},{  "type": "way",  "id": 438389786,  "nodes": [    42440729,    3985008870,    5266421746,    42440737  ]},{  "type": "way",  "id": 496992369,  "nodes": [    42436941,    3627218177,    5266421747,    3985008869,    597937476,    42440729  ]},{  "type": "way",  "id": 561084838,  "nodes": [    42443296,    4235733233,    42449017,    4235733234,    42430886,    3627218178,    42449886,    3627218180,    3627218179,    42446533,    3627218181,    3627218176,    42436941  ]},{  "type": "way",  "id": 5670009,  "nodes": [    42431470,    3834459540  ]},{  "type": "way",  "id": 5672706,  "nodes": [    42439323,    3834459551,    4298757025,    4191922368,    42455452  ]},{  "type": "way",  "id": 68668526,  "nodes": [    42440743,    4161568336,    4298757077,    42439006  ]},{  "type": "way",  "id": 420598232,  "nodes": [    4205565489,    4161568338,    42440743  ]},{  "type": "way",  "id": 420872180,  "nodes": [    42440743,    4161568334,    6133355307,    4298757033,    42431470,    4298757061,    4298757020,    42439323,    4298757080,    4298757018,    4207683792  ]},{  "type": "way",  "id": 479642034,  "nodes": [    42436947,    42436944,    42436943  ]},{  "type": "way",  "id": 525701044,  "nodes": [    685164631,    4298757054,    42439323  ]},{  "type": "way",  "id": 5671774,  "nodes": [    42440729,    3985008872,    5266421748,    5481911746,    3985008875,    42431459,    3985008879,    42449890,    42446875,    42437283,    42449893,    4132605205,    42439170,    4132605206,    4132605215,    390519635,    4132605231,    4132605267,    42440326,    4132605269,    4132605270,    42448701  ]},{  "type": "way",  "id": 5672176,  "nodes": [    42446528,    42436939,    42453110,    42453113,    4983147784,    42453116  ]},{  "type": "way",  "id": 5672948,  "nodes": [    42455452,    4191922369,    42431467  ]},{  "type": "way",  "id": 22897633,  "nodes": [    42446889,    2384399611,    4298757047,    42446885,    4129804702,    4376122114,    42442877,    4129804711,    42446879,    42446875,    42434355  ]},{  "type": "way",  "id": 22902609,  "nodes": [    42455465,    4191917275,    4191922366,    42431461  ]},{  "type": "way",  "id": 22902610,  "nodes": [    42455452,    4191922374,    42446885  ]},{  "type": "way",  "id": 34079834,  "nodes": [    42439170,    4132605204,    42439178  ]},{  "type": "way",  "id": 34080174,  "nodes": [    42428201,    3019038994,    3857400393,    42428198,    3857400404,    3857400388,    42428192,    6210354205,    42428183,    42428179,    42428174  ]},{  "type": "way",  "id": 46334599,  "nodes": [    42440323,    42440325,    4132605268,    42440326,    4132605266,    4132971872,    42440330,    3817756020,    4132971841,    4296710225,    2562399788,    387184869  ]},{  "type": "way",  "id": 195745914,  "nodes": [    42436582,    4115878741,    42436586  ]},{  "type": "way",  "id": 226040991,  "nodes": [    1692450029,    4132605214,    390519635  ]},{  "type": "way",  "id": 420519634,  "nodes": [    42448701,    4132605272,    5266429278,    42446925  ]},{  "type": "way",  "id": 420519635,  "nodes": [    42448693,    5266429281,    42436578,    42456555  ]},{  "type": "way",  "id": 420598294,  "nodes": [    42442877,    4129804701,    4205565496  ]},{  "type": "way",  "id": 428569356,  "nodes": [    4276517126,    4205565512  ]},{  "type": "way",  "id": 438388090,  "nodes": [    42440737,    4376122112,    5243723093,    4161568341,    3834459526,    42440743  ]},{  "type": "way",  "id": 451808493,  "nodes": [    4376122113,    42431464,    4191922367,    42431461,    3985008876,    42431459  ]},{  "type": "way",  "id": 451808494,  "nodes": [    3834459540,    4298757059,    42431467,    4191926877,    4376122113  ]},{  "type": "way",  "id": 456308943,  "nodes": [    42428198,    3857400409,    4597668040,    42440323  ]},{  "type": "way",  "id": 458166896,  "nodes": [    42445374,    42446935,    42446934,    42446933,    42445411,    3997461348,    42436751,    3997461347,    3997461346,    42446270,    4234425034,    3997461343,    42446932,    3997461342,    3997461341,    42429890,    3997461340,    2302133933,    42436582  ]},{  "type": "way",  "id": 479642036,  "nodes": [    42430903,    4216045847,    42430914  ]},{  "type": "way",  "id": 525768549,  "nodes": [    4597668029,    42436578,    42436582  ]},{  "type": "way",  "id": 525768550,  "nodes": [    4597668029,    4597668041  ]},{  "type": "way",  "id": 543902512,  "nodes": [    42442881,    42449893  ]},{  "type": "way",  "id": 616745175,  "nodes": [    42434357,    42437283,    5282642709,    4129804731,    42437287,    4376122115,    4129804730,    4298757083,    2562399783,    42437289  ]},{  "type": "way",  "id": 630828594,  "nodes": [    42434352,    5266421745,    3985008871,    42440729  ]},{  "type": "way",  "id": 5670254,  "nodes": [    42434352,    42431453,    42434354,    42434355  ]},{  "type": "way",  "id": 5670522,  "nodes": [    4597668019,    3252590542,    42437021  ]},{  "type": "way",  "id": 5670551,  "nodes": [    42437280,    42434357  ]},{  "type": "way",  "id": 5670581,  "nodes": [    42437529,    1251830468,    42437530  ]},{  "type": "way",  "id": 5670673,  "nodes": [    42437530,    42438174,    42438173  ]},{  "type": "way",  "id": 5670736,  "nodes": [    42438849,    42434360,    42437529  ]},{  "type": "way",  "id": 5670770,  "nodes": [    42437529,    4132605207,    42439170  ]},{  "type": "way",  "id": 5670778,  "nodes": [    42438173,    1251830454,    42438849  ]},{  "type": "way",  "id": 5671472,  "nodes": [    42446925,    42440325  ]},{  "type": "way",  "id": 5672710,  "nodes": [    4597668023,    42437018,    3670742444,    42434354,    42449890,    4191917276,    42455465  ]},{  "type": "way",  "id": 5672937,  "nodes": [    42456543,    42439203,    5307048044,    42453310  ]},{  "type": "way",  "id": 5673613,  "nodes": [    42455465,    4191922377,    42446879  ]},{  "type": "way",  "id": 34079830,  "nodes": [    390519635,    4132605229,    42437530  ]},{  "type": "way",  "id": 34080149,  "nodes": [    4205565498,    4983147774,    595403439,    4983147812,    4983147814,    4597668018,    42431452  ]},{  "type": "way",  "id": 68668537,  "nodes": [    42431452,    3670742472,    5266429283,    5266429284,    3670742447,    42437018,    3670742479,    4597668019  ]},{  "type": "way",  "id": 195743164,  "nodes": [    42440721,    3670742460,    5243716994,    42434352  ]},{  "type": "way",  "id": 420520112,  "nodes": [    4597668038,    42428179  ]},{  "type": "way",  "id": 420520113,  "nodes": [    42437021,    3252590544,    5266429282,    42437280  ]},{  "type": "way",  "id": 420598310,  "nodes": [    42440721,    3670742481,    4205565498  ]},{  "type": "way",  "id": 420598373,  "nodes": [    4205565512,    4597668041  ]},{  "type": "way",  "id": 420598391,  "nodes": [    4205565503,    42456543  ]},{  "type": "way",  "id": 438390724,  "nodes": [    42446925,    5266429279,    5266429280,    42448693  ]},{  "type": "way",  "id": 439933950,  "nodes": [    3670742487,    3670742438,    595403438  ]},{  "type": "way",  "id": 451808491,  "nodes": [    42455465,    4191917273  ]},{  "type": "way",  "id": 451808492,  "nodes": [    42431459,    3985008878,    5481911751,    42431453,    3670742442,    42431452  ]},{  "type": "way",  "id": 456308944,  "nodes": [    42453310,    5307048043,    42440323  ]},{  "type": "way",  "id": 456308945,  "nodes": [    42440323,    42448693  ]},{  "type": "way",  "id": 464683313,  "nodes": [    42448693,    4597668041  ]},{  "type": "way",  "id": 464683314,  "nodes": [    42437280,    4205565503  ]},{  "type": "way",  "id": 464683315,  "nodes": [    4597668026,    42437280  ]},{  "type": "way",  "id": 464683317,  "nodes": [    4597668026,    4597668027,    4597668036  ]},{  "type": "way",  "id": 464683326,  "nodes": [    595403438,    4597668024,    4597668023,    4597668022,    5307048045,    4597668035  ]},{  "type": "way",  "id": 464683327,  "nodes": [    4597668036,    4597668025,    595403438  ]},{  "type": "way",  "id": 464683328,  "nodes": [    4597668038,    4597668026  ]},{  "type": "way",  "id": 464683329,  "nodes": [    4597668041,    4597668040,    4597668039,    4597668028,    4597668038  ]},{  "type": "way",  "id": 481403100,  "nodes": [    42431452,    4597668035  ]},{  "type": "way",  "id": 505600313,  "nodes": [    42456543,    4597668038  ]},{  "type": "way",  "id": 509024997,  "nodes": [    4983147329,    4983147328  ]},{  "type": "way",  "id": 509025002,  "nodes": [    4983147418,    4983147417  ]},{  "type": "way",  "id": 509025003,  "nodes": [    4983147420,    4983147419  ]},{  "type": "way",  "id": 509025004,  "nodes": [    4983147722,    4983147721  ]},{  "type": "way",  "id": 509025005,  "nodes": [    4983147724,    4983147723  ]},{  "type": "way",  "id": 509025006,  "nodes": [    4983147726,    4983147725  ]},{  "type": "way",  "id": 509025008,  "nodes": [    4983147762,    4983147761  ]},{  "type": "way",  "id": 509025009,  "nodes": [    4983147764,    4983147763  ]},{  "type": "way",  "id": 509025010,  "nodes": [    4983147766,    4983147765  ]},{  "type": "way",  "id": 509025011,  "nodes": [    4983147768,    4983147767  ]},{  "type": "way",  "id": 509025012,  "nodes": [    4983147770,    4983147769  ]},{  "type": "way",  "id": 509025014,  "nodes": [    4983147816,    4983147815  ]},{  "type": "way",  "id": 509025015,  "nodes": [    4983147818,    4983147817  ]},{  "type": "way",  "id": 509025016,  "nodes": [    4983147820,    4983147819  ]},{  "type": "way",  "id": 543902513,  "nodes": [    42449893,    42434358,    42459668,    42456543  ]},{  "type": "way",  "id": 547089938,  "nodes": [    42453116,    3670742454,    42440721  ]},{  "type": "way",  "id": 584971399,  "nodes": [    42436582,    42446925  ]},{  "type": "way",  "id": 616745173,  "nodes": [    42434355,    42434357  ]},{  "type": "way",  "id": 616745174,  "nodes": [    42434355,    3252590541,    42437021,    4597668036  ]},{  "type": "way",  "id": 616745176,  "nodes": [    42434357,    42434358  ]},{  "type": "way",  "id": 630828554,  "nodes": [    42440325,    42438174  ]},{  "type": "way",  "id": 630828555,  "nodes": [    42434358,    42434360  ]},{  "type": "way",  "id": 5673523,  "nodes": [    42455887,    2562399784,    4298760164,    4376122116,    42442881  ]},{  "type": "way",  "id": 110489638,  "nodes": [    42439178,    4376122117,    4296710218,    2562399786  ]},{  "type": "way",  "id": 157014136,  "nodes": [    42442889,    3686062439,    6139477286,    4132971876,    42440330,    4132971831,    1692450029  ]},{  "type": "way",  "id": 378363223,  "nodes": [    42437287,    4129804729,    4205565497  ]},{  "type": "way",  "id": 420598236,  "nodes": [    42442881,    4205565499  ]},{  "type": "way",  "id": 420598241,  "nodes": [    42439178,    4205565501  ]},{  "type": "way",  "id": 420598266,  "nodes": [    1692450029,    4205565502  ]},{  "type": "way",  "id": 420598275,  "nodes": [    4205565502,    42439178  ]},{  "type": "way",  "id": 420598281,  "nodes": [    4205565501,    42442881  ]},{  "type": "way",  "id": 420598289,  "nodes": [    4205565497,    4129804713,    42442877  ]},{  "type": "way",  "id": 420598299,  "nodes": [    4205565499,    4129804732,    42437287  ]},{  "type": "way",  "id": 489019753,  "nodes": [    42442889,    5690422189,    5266429277,    4132605271,    42448701  ]},{  "type": "way",  "id": 489019754,  "nodes": [    42448701,    4132605273,    4115878740,    42436586  ]},{  "type": "way",  "id": 663499005,  "nodes": [    4296710222,    4376122118,    1692450029  ]},{  "type": "way",  "id": 110489637,  "nodes": [    42427381,    42427374  ]},{  "type": "way",  "id": 156994969,  "nodes": [    5849918502,    42434271  ]},{  "type": "way",  "id": 384178812,  "nodes": [    42437909,    5545401589,    4890040965,    4205565536,    6279091713  ]},{  "type": "way",  "id": 384178813,  "nodes": [    42437909,    42434279,    42443674  ]},{  "type": "way",  "id": 420598333,  "nodes": [    4205565523,    5849918502  ]},{  "type": "way",  "id": 420598335,  "nodes": [    4205565518,    42435610  ]},{  "type": "way",  "id": 420598337,  "nodes": [    4205565514,    42435603  ]},{  "type": "way",  "id": 420598339,  "nodes": [    4205565533,    42435624  ]},{  "type": "way",  "id": 420598349,  "nodes": [    42435610,    42427381  ]},{  "type": "way",  "id": 420598356,  "nodes": [    42435603,    42429662  ]},{  "type": "way",  "id": 420598366,  "nodes": [    42435624,    2322155478,    42427427,    4205565537  ]},{  "type": "way",  "id": 458106344,  "nodes": [    5849918502,    6211484434,    4601466499,    42443674  ]},{  "type": "way",  "id": 465098271,  "nodes": [    42429662,    4205565518  ]},{  "type": "way",  "id": 465098276,  "nodes": [    42435599,    4205565514  ]},{  "type": "way",  "id": 465098285,  "nodes": [    42427381,    4205565523  ]},{  "type": "way",  "id": 465098290,  "nodes": [    42434271,    4205565533  ]},{  "type": "way",  "id": 670519221,  "nodes": [    6279091713,    42435624  ]},{  "type": "way",  "id": 670519226,  "nodes": [    4205565533,    6279091715,    6279091714,    6279091712,    6279091713  ]},{  "type": "way",  "id": 420598327,  "nodes": [    42427426,    4205565532,    42437890  ]},{  "type": "way",  "id": 420598330,  "nodes": [    42434807,    42439842,    42439840,    42427426  ]},{  "type": "way",  "id": 420598415,  "nodes": [    42437890,    4205565528,    4205565526,    42430271  ]},{  "type": "way",  "id": 630828586,  "nodes": [    42430271,    42427423,    42430274,    42430277  ]},{  "type": "way",  "id": 156994961,  "nodes": [    42430265,    42439836,    5849918502  ]},{  "type": "way",  "id": 350726028,  "nodes": [    4557517549,    358448394,    42437688,    42427817,    42438672,    5143758609,    4890040964,    3254626250,    5143758605,    42444933,    42444928,    42427477,    42437909  ]},{  "type": "way",  "id": 420598407,  "nodes": [    42435624,    4205565530,    42437890  ]},{  "type": "way",  "id": 465101198,  "nodes": [    42430271,    3859048179,    3019039006,    42437881  ]},{  "type": "way",  "id": 5670466,  "nodes": [    42436575,    4597668029  ]},{  "type": "way",  "id": 5670876,  "nodes": [    42434268,    4687262928,    3857400397,    4687262935,    42428198  ]},{  "type": "way",  "id": 5671393,  "nodes": [    42445867,    4276517099,    42436575,    42445879  ]},{  "type": "way",  "id": 5671620,  "nodes": [    42437881,    3009534455,    3884612885,    42428201  ]},{  "type": "way",  "id": 5672965,  "nodes": [    449581627,    3009534469,    42428206,    6243559673,    3009534459,    42437881  ]},{  "type": "way",  "id": 22898643,  "nodes": [    449581627,    5254263338  ]},{  "type": "way",  "id": 34080508,  "nodes": [    42437881,    3009534450,    42434268  ]},{  "type": "way",  "id": 420598313,  "nodes": [    4205565520,    449581627  ]},{  "type": "way",  "id": 464683336,  "nodes": [    4597668042,    4597668029  ]},{  "type": "way",  "id": 491162341,  "nodes": [    449581627,    5147973330,    6210354201,    5147973338,    5147973337,    3857400396,    5147972298,    5256074552,    42428212  ]},{  "type": "way",  "id": 526571477,  "nodes": [    42428201,    3019039007,    4763900915,    4763900914,    4763900910,    4763900908,    4763900906,    4276517114,    42445867  ]},{  "type": "way",  "id": 543717203,  "nodes": [    5254263338,    449581964,    3019042134,    6243559679,    449581965,    274681226,    3009534454,    42428201  ]},{  "type": "way",  "id": 587038012,  "nodes": [    42445867,    4276517126  ]}  ]}'
