import functools

import click


def abort_guard(on_abort):
    """
    Wrap a Click command so we can do per-command handling on aborts
    without changing the callback signature.

    - on_abort(ctx, reason) called with reason in {"keyboard", "click"}.
    - Exits with code 1 silently after handler. Swap to `raise` if you want Click's "Aborted!".
    """

    def decorator(fn):
        @functools.wraps(fn)
        def wrapper(*args, **kwargs):
            ctx = click.get_current_context(silent=True)
            try:
                return fn(*args, **kwargs)
            except KeyboardInterrupt:
                if on_abort:
                    on_abort(ctx)
                if ctx:
                    ctx.exit(1)  # silent
                raise SystemExit(1)
            except click.Abort:
                if on_abort:
                    on_abort(ctx)
                if ctx:
                    ctx.exit(1)  # silent
                raise SystemExit(1)

        return wrapper

    return decorator
