#!/usr/bin/env python3
"""
Tests for PandaDock configuration
"""

import pytest
import sys
from pathlib import Path

# Add parent directory to path for imports
sys.path.insert(0, str(Path(__file__).parent.parent))


class TestPandaDockConfig:
    """Test PandaDock configuration functionality"""
    
    def test_config_creation(self):
        """Test basic config creation"""
        try:
            from pandadock.config import PandaDockConfig
            config = PandaDockConfig()
            assert config is not None
        except ImportError as e:
            pytest.skip(f"Config not available: {e}")
    
    def test_algorithm_names(self):
        """Test that new algorithm names are supported"""
        try:
            from pandadock.config import PandaDockConfig
            
            # Test that these are the new algorithm names we expect
            expected_algorithms = ['pandacore', 'pandaml', 'pandaphysics']
            
            # Just verify we can create configs - actual validation depends on implementation
            for algorithm in expected_algorithms:
                config = PandaDockConfig()
                assert config is not None
                
        except ImportError as e:
            pytest.skip(f"Config not available: {e}")
    
    def test_config_attributes(self):
        """Test that config has expected attributes"""
        try:
            from pandadock.config import PandaDockConfig
            config = PandaDockConfig()
            
            # Check for common configuration attributes
            # Note: These may not exist yet, so we test defensively
            expected_attrs = ['docking', 'scoring', 'io']
            
            for attr in expected_attrs:
                # Just check that we can access the attribute without error
                # The actual implementation may vary
                try:
                    getattr(config, attr, None)
                except AttributeError:
                    pass  # Attribute may not be implemented yet
                    
        except ImportError as e:
            pytest.skip(f"Config not available: {e}")


class TestScoringConfig:
    """Test scoring function configuration"""
    
    def test_new_scoring_names(self):
        """Test that new scoring function names are recognized"""
        try:
            # Test the new algorithm names instead of old ones
            new_algorithms = ['pandacore', 'pandaml', 'pandaphysics']
            
            # For now, just test that these are strings (implementation may vary)
            for algorithm in new_algorithms:
                assert isinstance(algorithm, str)
                assert algorithm.startswith('panda')
                
        except Exception as e:
            pytest.skip(f"Scoring config test failed: {e}")


if __name__ == "__main__":
    pytest.main([__file__])